/********************************************************************************
*                                                                               *
*                         T r e e   L i s t   W i d g e t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXTreeList.h,v 1.7 1999/11/09 23:58:38 jeroen Exp $                      *
********************************************************************************/
#ifndef FXTREELIST_H
#define FXTREELIST_H


// Tree List options
enum FXTreeListStyle {
  TREELIST_EXTENDEDSELECT = 0,
  TREELIST_SINGLESELECT   = 0x00100000,       // Single selection mode
  TREELIST_BROWSESELECT   = 0x00200000,       // Browse selection mode
  TREELIST_MULTIPLESELECT = TREELIST_SINGLESELECT|TREELIST_BROWSESELECT,  // Multiple select
  TREELIST_AUTOSELECT     = 0x00400000,       // Automatically select under cursor
  TREELIST_SHOWS_LINES    = 0x00800000,       // Lines shown
  TREELIST_SHOWS_BOXES    = 0x01000000,       // Boxes to expand shown
  TREELIST_ROOT_BOXES     = 0x02000000        // Display root boxes also
  };



// Tree List Item
class FXAPI FXTreeItem : public FXObject {
  FXDECLARE(FXTreeItem)
  friend class FXTreeList;
  friend class FXDirList;
protected:
  FXTreeItem *prev;
  FXTreeItem *next;
  FXTreeItem *parent;
  FXTreeItem *first;
  FXTreeItem *last;
  FXString    label;
  FXIcon     *openIcon;
  FXIcon     *closedIcon;
  void       *data;
  FXuint      state;
  FXint       x,y;
protected:
  FXTreeItem():prev(NULL),next(NULL),parent(NULL),first(NULL),last(NULL),openIcon(NULL),closedIcon(NULL),data(NULL),state(0),x(0),y(0){}
  virtual void draw(const FXTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  void drawFocus(const FXTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual FXint hitItem(const FXTreeList* list,FXint x,FXint y) const;
protected:
  enum{
    SELECTED   = 1,
    MARKED     = 2,
    FOCUS      = 4,
    DISABLED   = 8,
    OPENED     = 16,
    EXPANDED   = 32,
    HASITEMS   = 64,
    DRAGGABLE  = 128
    };
public:
  FXTreeItem(const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL):label(text),openIcon(oi),closedIcon(ci),data(ptr),state(0),x(0),y(0){}
  FXint getNumChildren() const;
  void setText(const FXString& txt){ label=txt; }
  FXString getText() const { return label; }
  void setOpenIcon(FXIcon* icn){ openIcon=icn; }
  FXIcon* getOpenIcon() const { return openIcon; }
  void setClosedIcon(FXIcon* icn){ closedIcon=icn; }
  FXIcon* getClosedIcon() const { return closedIcon; }
  void setData(void* ptr){ data=ptr; }
  void* getData() const { return data; }
  FXbool hasFocus() const { return (state&FOCUS)!=0; }
  FXbool isSelected() const { return (state&SELECTED)!=0; }
  FXbool isOpened() const { return (state&OPENED)!=0; }
  FXbool isExpanded() const { return (state&EXPANDED)!=0; }
  FXbool isEnabled() const { return (state&DISABLED)==0; }
  FXbool isDraggable() const { return (state&DRAGGABLE)!=0; }
  FXTreeItem* getParent() const { return parent; }
  FXTreeItem* getNext() const { return next; }
  FXTreeItem* getPrev() const { return prev; }
  FXTreeItem* getFirst() const { return first; }
  FXTreeItem* getLast() const { return last; }
  FXTreeItem* getBelow() const;
  FXTreeItem* getAbove() const;
  virtual FXint getWidth(const FXTreeList* list) const;
  virtual FXint getHeight(const FXTreeList* list) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);
  virtual ~FXTreeItem(){}
  };



// Item collate function
typedef FXbool (*FXTreeListSortFunc)(const FXTreeItem*,const FXTreeItem*);



// Tree List
class FXAPI FXTreeList : public FXScrollArea {
  FXDECLARE(FXTreeList)
protected:
  FXTreeItem        *firstitem;
  FXTreeItem        *lastitem;
  FXTreeItem        *anchoritem;
  FXTreeItem        *currentitem;
  FXTreeItem        *extentitem;
  FXFont            *font;
  FXTreeListSortFunc sortfunc;
  FXColor            textColor;
  FXColor            selbackColor;
  FXColor            seltextColor;
  FXColor            lineColor;
  FXint              itemWidth;
  FXint              itemHeight;
  FXint              visible;
  FXint              nrows;
  FXint              indent;
  FXTimer           *timer;
  FXString           help;
protected:
  FXTreeList();
  virtual void layout();
  virtual FXTreeItem* createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr);
  void sort(FXTreeItem*& f1,FXTreeItem*& t1,FXTreeItem*& f2,FXTreeItem*& t2,int n);
  void recompute();
  FXbool before(const FXTreeItem* a,const FXTreeItem* b) const;
  FXbool mark(FXTreeItem* beg,FXTreeItem* end,FXuint sel);
  FXbool restore(FXTreeItem* beg,FXTreeItem* end);
private:
  FXTreeList(const FXTreeList&);
  FXTreeList& operator=(const FXTreeList&);
public:
  long onPaint(FXObject*,FXSelector,void*);
  long onEnter(FXObject*,FXSelector,void*);
  long onLeave(FXObject*,FXSelector,void*);
  long onUngrabbed(FXObject*,FXSelector,void*);
  long onMotion(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
  long onKeyRelease(FXObject*,FXSelector,void*);
  long onLeftBtnPress(FXObject*,FXSelector,void*);
  long onLeftBtnRelease(FXObject*,FXSelector,void*);
  long onActivate(FXObject*,FXSelector,void*);
  long onDeactivate(FXObject*,FXSelector,void*);
  long onQueryTip(FXObject*,FXSelector,void*);
  long onQueryHelp(FXObject*,FXSelector,void*);
  long onTipTimer(FXObject*,FXSelector,void*);
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onSelectionLost(FXObject*,FXSelector,void*);
  long onSelectionGained(FXObject*,FXSelector,void*);
  long onAutoScroll(FXObject*,FXSelector,void*);
  long onItemOpened(FXObject*,FXSelector,void*);
  long onItemClosed(FXObject*,FXSelector,void*);
  long onItemExpanded(FXObject*,FXSelector,void*);
  long onItemCollapsed(FXObject*,FXSelector,void*);
  long onChanged(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onDoubleClicked(FXObject*,FXSelector,void*);
  long onTripleClicked(FXObject*,FXSelector,void*);
  long onCommand(FXObject*,FXSelector,void*);
public:
  enum {
    ID_TIPTIMER=FXScrollArea::ID_LAST,
    ID_LAST
    };
public:
  FXTreeList(FXComposite *p,FXint nvis,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0);
  virtual void create();
  virtual void detach();
  virtual FXint getDefaultWidth();
  virtual FXint getDefaultHeight();
  virtual FXint getContentWidth();
  virtual FXint getContentHeight();
  virtual void recalc();
  virtual FXbool canFocus() const;
  FXint getNumItems() const;
  FXint getNumVisible() const { return visible; }
  void setNumVisible(FXint nvis);
  FXTreeItem* getFirstItem() const { return firstitem; }
  FXTreeItem* getLastItem() const { return lastitem; }
  FXTreeItem* addItemFirst(FXTreeItem* p,FXTreeItem* item);
  FXTreeItem* addItemLast(FXTreeItem* p,FXTreeItem* item);
  FXTreeItem* addItemAfter(FXTreeItem* other,FXTreeItem* item);
  FXTreeItem* addItemBefore(FXTreeItem* other,FXTreeItem* item);
  FXTreeItem* addItemFirst(FXTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL);
  FXTreeItem* addItemLast(FXTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL);
  FXTreeItem* addItemAfter(FXTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL);
  FXTreeItem* addItemBefore(FXTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL);
  void removeItem(FXTreeItem* item);
  void removeItems(FXTreeItem* fm,FXTreeItem* to);
  void removeAllItems();
  FXint getItemWidth(const FXTreeItem*) const { return itemWidth; }
  FXint getItemHeight(const FXTreeItem*) const { return itemHeight; }
  virtual FXTreeItem* getItemAt(FXint x,FXint y) const;
  FXTreeItem* findItem(const FXString& text,FXuint len=2147483647) const;
  FXTreeItem* findChildItem(FXTreeItem* parentitem,const FXString& text,FXuint len=2147483647) const;
  void makeItemVisible(FXTreeItem* item);
  void setItemText(FXTreeItem* item,const FXString& text);
  FXString getItemText(const FXTreeItem* item) const;
  void setItemOpenIcon(FXTreeItem* item,FXIcon* icon);
  FXIcon* getItemOpenIcon(const FXTreeItem* item) const;
  void setItemClosedIcon(FXTreeItem* item,FXIcon* icon);
  FXIcon* getItemClosedIcon(const FXTreeItem* item) const;
  void setItemData(FXTreeItem* item,void* ptr) const;
  void* getItemData(const FXTreeItem* item) const;
  FXbool isItemSelected(const FXTreeItem* item) const;
  FXbool isItemCurrent(const FXTreeItem* item) const;
  FXbool isItemVisible(const FXTreeItem* item) const;
  FXbool isItemOpened(const FXTreeItem* item) const;
  FXbool isItemExpanded(const FXTreeItem* item) const;
  FXbool isItemLeaf(const FXTreeItem* item) const;
  FXbool isItemEnabled(const FXTreeItem* item) const;
  FXint hitItem(const FXTreeItem* item,FXint x,FXint y) const;
  void updateItem(FXTreeItem* item);
  FXbool enableItem(FXTreeItem* item);
  FXbool disableItem(FXTreeItem* item);
  FXbool selectItem(FXTreeItem* item);
  FXbool deselectItem(FXTreeItem* item);
  FXbool toggleItem(FXTreeItem* item);
  FXbool openItem(FXTreeItem* item);
  FXbool closeItem(FXTreeItem* item);
  FXbool collapseTree(FXTreeItem* tree);
  FXbool expandTree(FXTreeItem* tree);
  void reparentItem(FXTreeItem* item,FXTreeItem* p);
  void setCurrentItem(FXTreeItem* item);
  FXTreeItem* getCurrentItem() const { return currentitem; }
  void setAnchorItem(FXTreeItem* item);
  FXTreeItem* getAnchorItem() const { return anchoritem; }
  FXbool extendSelection(FXTreeItem* item);
  FXbool killSelection();
  void sortItems();
  void sortChildItems(FXTreeItem* item);
  void setFont(FXFont* fnt);
  FXFont* getFont() const { return font; }
  FXColor getTextColor() const { return textColor; }
  void setTextColor(FXColor clr);
  FXColor getSelBackColor() const { return selbackColor; }
  void setSelBackColor(FXColor clr);
  FXColor getSelTextColor() const { return seltextColor; }
  void setSelTextColor(FXColor clr);
  FXColor getLineColor() const { return lineColor; }
  void setLineColor(FXColor clr);
  void setIndent(FXint in);
  FXint getIndent() const { return indent; }
  void setHelpText(const FXString& text);
  FXString getHelpText() const { return help; }
  FXTreeListSortFunc getSortFunc() const { return sortfunc; }
  void setSortFunc(FXTreeListSortFunc func){ sortfunc=func; }
  FXuint getListStyle() const;
  void setListStyle(FXuint style);
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);
  virtual ~FXTreeList();
  };


#endif
