// arrlfile.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "arrlfile.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>         // For isdigit()
#include <time.h>
#include <string.h>

#define MAX_ARRL_LINE_SIZE    100
#define MAX_ARRL_CALL_SIZE    15

char *RArrlFile::modes[] = {NULL, "CW", "SSB", "", "", "FM", NULL};
char *RArrlFile::bands[] = {NULL, "160", "80", "40", "30", "20", "17",
                         "15", "12", "10", "50", "76", "144", "222", "432",
                         "903", "1.2", "2.3", "3.4", "5.7", "10G", "24G", NULL,
                         NULL, NULL, NULL, NULL, "SAT", "PKT", "NOV", "LHT", NULL};
boolean RArrlFile::contests[] = {FALSE, TRUE, TRUE, FALSE, FALSE, TRUE, TRUE, TRUE, FALSE, TRUE, FALSE,
                              TRUE, FALSE, TRUE, FALSE, FALSE, TRUE, FALSE, FALSE, FALSE, FALSE,
                              TRUE};
#define MAX_ARRL_CONTEST NA_SPRINT_CONTEST                              

RArrlFile::RArrlFile(char *filename)
{
    RArrlFile::open(filename);
}

boolean RArrlFile::open(char *filename, int mode)
{
    if (!RFile::open(filename, mode))
        return FALSE;

    // There is no header to read.  Must make sure to specify the
    //   contest type through a command line option!!!!

    // Set some options to defaults so that required fields are
    // not NULL.
    strcpy(header.callsign, "NOCALL");
    
    location = 0;

    return TRUE;
}

boolean RArrlFile::nextQso(qso_data *qso)
{
    unsigned char count;
    char buffer[MAX_ARRL_LINE_SIZE];
    char *tmpPtr;

    qso->init();

    RFile::getString((char *)buffer, MAX_ARRL_LINE_SIZE);
    if (RFile::eof())
        return FALSE;
    
    // Store band into qso_data
    tmpPtr = strtok(buffer, " ");
    for (count = 1; count <= MAX_BAND; count++)
        if ((bands[count] != NULL) && (strcmp(bands[count], tmpPtr) == 0))
            qso->setBand(count);
            
    // Store mode into qso_data
    tmpPtr = strtok(NULL, " ");
    for (count = 1; modes[count] != NULL; count++)
        if (strcmp(modes[count], tmpPtr) == 0)
            qso->setMode(count);
    // Special case.  TR Log outputs "PH" instead of
    //  SSB as the mode for phone.
    if ((qso->getMode() == 0) && (strcmp(tmpPtr, "PH") == 0))
        qso->setMode(SSB_MODE);

    // Convert the verbose time to time_t format.
    {
        int month, day, year, hours, minutes;

        day = atoi(strtok(NULL, "/"));
        month = atoi(strtok(NULL, "/")) - 1;
        year = atoi(strtok(NULL, " "));
        tmpPtr = strtok(NULL, " ");
        minutes = atoi(&tmpPtr[2]);
        tmpPtr[2] = (char)NULL;
        hours = atoi(tmpPtr);
        qso->setTime(year, month, day, hours, minutes);
    }
        
    // Get QSO number.  We don't need it though.
    tmpPtr = strtok(NULL, " ");

    if (header.contest == ARRL_SS_CONTEST)
    {
        tmpPtr = strtok(NULL, " ");       // Get precedence sent for SS
        tmpPtr = strtok(NULL, " ");       // Get callsign sent
    }
    else
        // Copy callsign of station worked to log_data struct.
        qso->setCallsign(strtok(NULL, " "));

    switch (header.contest)
    {
        case ARRL_SS_CONTEST:
            tmpPtr = strtok(NULL, " ");                // Get check sent
            tmpPtr = strtok(NULL, " ");                // Get section sent
            qso->setSerialNumber(strtok(NULL, " "));
            tmpPtr = strtok(NULL, " ");
            qso->setSSPrecedence(*tmpPtr);             // Get precedence received
            qso->setCallsign(strtok(NULL, " "));
            qso->setSSCheck(strtok(NULL, " "));
            qso->setSection(strtok(NULL, " "));
            break;
        case ARRL_DX_CONTEST_DX:
        case ARRL_10M_CONTEST:
            tmpPtr = strtok(NULL, " ");                 // Get RST sent
            tmpPtr = strtok(NULL, " ");                 // Get info sent
            qso->setReceivedRST(strtok(NULL, " "));
            qso->setInfo(strtok(NULL, " "));
            break;
        case IARU_CONTEST:
            tmpPtr = strtok(NULL, " ");                 // Get RST sent
            tmpPtr = strtok(NULL, " ");                 // Get info sent
            qso->setReceivedRST(strtok(NULL, " "));     // Get RST received
            tmpPtr = strtok(NULL, " ");
            if (isdigit(tmpPtr[0]))
                qso->setItuZone(tmpPtr);
            else
                qso->setInfo(tmpPtr);                   // Save IARU HQ Name
            break;
        case ARRL_DX_CONTEST:
        case ARRL_160M_CONTEST:
            tmpPtr = strtok(NULL, " ");                 // Get RST sent
            tmpPtr = strtok(NULL, " ");                 // Get next field. May be info or RST.
            if (isdigit(*tmpPtr))
                qso->setReceivedRST(tmpPtr);            // Not the expected info, so must be the
                                                        // received RST.  Record it.
            qso->setInfo(strtok(NULL, " "));
            break;
        case FD_CONTEST:
            tmpPtr = strtok(NULL, " ");                 // Skip over sent tx count and cat.
            tmpPtr = strtok(NULL, " ");                 // Skip over sent section
            tmpPtr = strtok(NULL, " ");                 // Get tx count and category
            qso->setFdCategory(tmpPtr[strlen(tmpPtr) - 1]);
            tmpPtr[strlen(tmpPtr) - 1] = (char)NULL;
            qso->setFdTransmitters(tmpPtr);
            qso->setSection(strtok(NULL, " "));       // Get section.
            break;
        case ARRL_VHF_QSO_PARTY:
            buffer[52] = (char)NULL;
            qso->setInfo(&buffer[48]);
            break;
        case NA_QSO_PARTY_CONTEST:
            qso->setName(strtok(NULL, " "));            // Get name received.
            qso->setInfo(strtok(NULL, " "));            // Get state received.
            break;
        case NA_SPRINT_CONTEST:
            qso->setSerialNumber(strtok(NULL, " "));
            qso->setName(strtok(NULL, " "));            // Get name received.
            qso->setInfo(strtok(NULL, " "));            // Get state received.
            break;
        default:
            break;
    };

    tmpPtr = strtok(NULL, " ");
    if ((strchr(tmpPtr, '\r') == NULL)
        && ((strlen(tmpPtr) != 1) || (!isdigit(*tmpPtr))))  // Is this a multiplier?
    {
        qso->setMultCount(1);
        qso->setQsoPoints(strtok(NULL, " "));
    }
    else
        qso->setQsoPoints(tmpPtr);
    
    location++;
    return TRUE;
}

boolean RArrlFile::writeHeader(header_data newheader)
{
    /* Test to see if the contest is supported by this ARRL write code. */
    /*  Return false if it is not supported. */
    if ((newheader.contest == 0)
         || (newheader.contest > MAX_ARRL_CONTEST)
         || (contests[newheader.contest] == FALSE))
        return FALSE;
        
    /* ARRL files do not have headers per se.  Save the info anyway. */
    RQsoFile::storeHeader(newheader);
    return TRUE;
}

boolean RArrlFile::writeQso(qso_data *qso)
{
    char buffer[82];
    char buffer2[80];
    struct tm *dosTime;
    char multChar = '*';
    time_t tempTime;

    location++;

    tempTime = qso->getTime();
    dosTime = gmtime(&tempTime);

    sprintf(buffer, "%-3s %-3s %02d/%02d/%2d %02d%02d ", bands[qso->getBand()],
                modes[qso->getMode()], dosTime->tm_mday,
                dosTime->tm_mon + 1, dosTime->tm_year, dosTime->tm_hour, dosTime->tm_min);
    if (header.contest == ARRL_SS_CONTEST)
        sprintf(buffer2, "%-4d %c ", location, header.ss.precedence);
    else
        sprintf(buffer2, "%-5d ", location);
    strcat(buffer, buffer2);

    if (qso->getMultCount() == 0)
        multChar = ' ';

    switch(header.contest)
    {
        case ARRL_SS_CONTEST:
            sprintf(buffer2, "%-10s %02d %3s ", header.callsign, header.ss.check, header.ss.section);
            strcat(buffer, buffer2);
            sprintf(buffer2, "%-04d %c %-10s %02d %-5s %c %1d", qso->getSerialNumber(),
                    qso->getSSPrecedence(), qso->getCallsign(), qso->getSSCheck(),
                    qso->getSection(), multChar, qso->getQsoPoints());
            break;
        case ARRL_DX_CONTEST:
            sprintf(buffer2, "%-14s %-9d %-3d %-5s %c %1d  0", qso->getCallsign(),
                qso->getSentRST(), qso->getReceivedRST(), qso->getInfo(), multChar,
                qso->getQsoPoints());
            break;
        case ARRL_10M_CONTEST:
            // Need to find a way to include the sent info instead of spaces!
            sprintf(buffer2, " %-13s %-3d %-5s %-3d %-5s %c %1d  0", qso->getCallsign(),
                     qso->getSentRST(), "  ", qso->getReceivedRST(), qso->getInfo(),
                     multChar, qso->getQsoPoints());
            break;
        case ARRL_160M_CONTEST:
            // Need to find a way to include the sent info instead of spaces!
            sprintf(buffer2, " %-12s %-3d %-5s %-3d %-5s %c %1d  0", qso->getCallsign(),
                     qso->getSentRST(), "  ", qso->getReceivedRST(), qso->getInfo(),
                     multChar, qso->getQsoPoints());
            break;
        case IARU_CONTEST:
            if (qso->getItuZone() > 0)
                sprintf(buffer2, "%-13s %-3d %5d %-3d %-02d    %c %1d  0", qso->getCallsign(),
                         qso->getSentRST(), header.zone, qso->getReceivedRST(),
                         qso->getItuZone(), multChar, qso->getQsoPoints());
            else
                sprintf(buffer2, "%-13s %-3d %5d %-3d %-5s %c %1d  0", qso->getCallsign(),
                         qso->getSentRST(), header.zone, qso->getReceivedRST(),
                         qso->getInfo(), multChar, qso->getQsoPoints());
            break;
        case ARRL_DX_CONTEST_DX:
            sprintf(buffer2, "%-14s %-3d      %-3d %-5s %c %1d  0", qso->getCallsign(),
                qso->getSentRST(), qso->getReceivedRST(), qso->getInfo(), multChar,
                qso->getQsoPoints());
            break;
        case FD_CONTEST:
            {
                int transCount = header.fd.transmitters;

                if (transCount == 0)
                    transCount = 1;
                    
                sprintf(buffer2, "%-14s %2d%c %-4s %2d%c %-4s %1d", qso->getCallsign(),
                        transCount, header.fd.category + 'A', header.fd.section,
                        qso->getFdTransmitters(), qso->getFdCategory(),
                        qso->getSection(), qso->getQsoPoints());
            }
            break;
        case ARRL_VHF_QSO_PARTY:
            sprintf(buffer2, "%-14s %4s %4s  %c %1d  0", qso->getCallsign(), header.grid,
                    qso->getInfo(), multChar, qso->getQsoPoints());
            break;
        case NA_QSO_PARTY_CONTEST:
            sprintf(buffer2, "%-14s %-11s %-12s %c        %1d", qso->getCallsign(),
                    qso->getName(), qso->getInfo(), multChar, qso->getQsoPoints());
            break;
        case NA_SPRINT_CONTEST:
            sprintf(buffer2, "%-13s %4d  %-11s %-6s %c        %1d", qso->getCallsign(),
                    qso->getSerialNumber(), qso->getName(), qso->getInfo(), multChar,
                    qso->getQsoPoints());
            break;
        default:
            return FALSE;
    }
    strcat(buffer, buffer2);

    strcat(buffer, "\r\n");
    RFile::write(buffer,strlen(buffer));

    return TRUE;
}

