/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "TFOptions.h"
#include "TFFileRC.h"
#include "Timer.h"
#include "HeightFieldGenRandom.h"
#include "HeightFieldGenSSynth.h"
#include "GlobalTrace.h"
#include "terraform.h"				// defines version number

extern int readhf (HeightField **HF, char *fname, bool haveGUI);

// defined in TFWindow.cc
extern guint TFWhash (gconstpointer key);
extern gint TFWhashcmp (gconstpointer a, gconstpointer b);

// defined in TFFileRC.cc
void hash_key_fetcher (gpointer key, gpointer data, gpointer user_data);
void hash_printer (gpointer key, gpointer data, gpointer user_data);
void hash_key_printer (gpointer key, gpointer data, gpointer user_data);



/*
 *  processArgs: process all args 
 */
void TFOptions::processArgs (HeightField **HF,int argc, char **argv)
{
	processGlobalArgs (argc, argv);
	processAppArgs (HF, argc, argv);
	processRCFile ();
}


/*
 *  processAppArgs: process the application-related command line arguments 
 */
void TFOptions::processAppArgs (HeightField **HF,int argc, char **argv)
{
	int			rc;

	for (int i=1; i<argc; i++)
		{
		if (!strcmp (argv[i], "-b") || !strcmp (argv[i], "--benchmark"))
			{
			Timer			T;
			HeightFieldGenSSynth 	*HFG; 
			int			nHFs=10;
			float			elapsed;
			
			printf ("Benchmarking: ");
			T.start ();
			for (int x=0; x<nHFs; x++)
				{
				printf (".");
				fflush (stdout);
				*HF = new HeightField ();
				HFG = new HeightFieldGenSSynth (*HF);
				HFG->generate (400, 1.5, 0, FALSE);
				delete HFG;
				delete *HF;
				}
			printf ("\n");
			T.stop ();
			elapsed = T.getElapsedSeconds();
			printf (_("Generated %d 400x400 height fields in %f seconds, avg=%f\n"), 
				nHFs, elapsed, elapsed/nHFs);
			exit (0);
			}
		else
		if (!strcmp(argv[i],"-f")||!strcmp(argv[i],"--file")) 
			{
			if (argc >= i+1 && argv[i+1])
				{
				if ((rc=readhf (HF, argv[++i], FALSE)) < 0)
					{
					printf (_("File read failed (rc=%d) for file [%s] ... Exiting\n"), rc, argv[i]);
					exit (-1);
					}
				}
			else
				{
				printf (_("Error: file argument for %s <file> missing ... Exiting\n"), argv[i]);
				exit (-1);
				}
			}
		else
		if (!strcmp(argv[i],"-fm")||!strcmp(argv[i],"--fixed_menus")) 
			{
			TFOptions::s_fixedMenus = TRUE;
			}
		else
		if (!strcmp (argv[i], "-g") || !strcmp (argv[i], "--generate"))
			{
			if (!(*HF))
				{
				*HF = new HeightField ();
				HeightFieldGenRandom::generate (*HF, 400);
				}
			else
				{
				printf ("Error: Height Field is already initialized ... can't generate\n");
				exit (-1);
				}
			}
		else
		if (!strcmp (argv[i], "-gs") || !strcmp (argv[i], "--generate_size"))
			{
			if (argc >= i+1 && argv[i+1])
				{
				if (!(*HF))
					{
					*HF = new HeightField ();
					HeightFieldGenRandom::generate (*HF, atoi(argv[++i]));
					}
				else
					{
					printf ("Error: Height Field is already initialized ... can't %s\n", argv[1]);
					exit (-1);
					}
				}
			else
				{
				printf (_("Error: size argument for %s missing ... Exiting\n"), argv[i]);
				exit (-1);
				}
			}
		else
		if (!strcmp (argv[i], "-l") || !strcmp (argv[i], "--large"))
			{
			TFOptions::s_large = TRUE;
			}
		else
		if (!strcmp (argv[i], "-p30") || !strcmp (argv[i], "--pov30"))
			{
			s_POV30=TRUE;
			}
		else
		if (!strcmp (argv[i], "-pexe") || !strcmp (argv[i], "--pov_exec"))
			{
			if (argc >= i+1 && argv[i+1])
				{
				s_POVexec=argv[++i];
				}
			else
				{
				printf (_("Error: pov executable argument for %s <pov_exec> missing ... Exiting\n"), argv[i]);
				exit (-1);
				}
			}
		else
		if (!strcmp (argv[i], "-s") || !strcmp (argv[i], "--dont_save"))
			{
			s_saveSettings=FALSE;
			}
#ifdef NEVER
		else
		if (!strcmp (argv[i], "-cli"))
			do_gui=FALSE;
#endif
		else 

		// dummy checks we do in process_global_args
		if (!strcmp (argv[i], "-dbg") || !strcmp (argv[i], "--debug"))
			i++;
		else
		if (!strcmp (argv[i], "-v") || !strcmp (argv[i], "--verbose"))
			;
		else
			{
			printf (_("Error: argument [%s] not recognized ... \n"), 
				argv[i]);
			help ();
			exit (0);
			}
		}
} 


/*
 *  processGlobalArgs: process the global command line arguments 
 */
void TFOptions::processGlobalArgs (int argc, char **argv)
{
	// see if we should set a certain trace level 
	for (int i=1; i<argc; i++)
		{
		if (!strcmp (argv[i], "-dbg") || !strcmp (argv[i], "--debug"))
			{
			if (argc >= i+2)	// 1 for count-to-index, 1 for next arg
				GlobalTrace::setTraceLevel (atoi (argv[++i]));
			else
				{
				printf (_("Error: traceLevel argument missing for %s ... Exiting\n"), argv[i]);
				exit (1);
				}
			}
		else
		if (!strcmp (argv[i], "-v") || !strcmp (argv[i], "--verbose"))
			GlobalTrace::setTraceLevel (GlobalTrace::TRACE_VERBOSE);
		else
		if (!strcmp (argv[i], "-version") || !strcmp(argv[i], "--version"))
			{
			printf ("terraform %d.%d.%d\n", TF_MAJOR_VERSION, 
				TF_MINOR_VERSION, TF_MICRO_VERSION);
			exit (0);
			}
		else
		if (!strcmp (argv[i], "-h") || !strcmp (argv[i], "--help") || 
		    !strcmp (argv[i], "-?") || !strcmp (argv[i], "-help") )
			{
			help ();
			exit (0);
			}
		else
		if (!strcmp (argv[i], "--helprc") || !strcmp (argv[i], "-hrc"))
			{
			helpRCFile ();
			exit (0);
			}
		else
		if (!strcmp (argv[i], "--installrc") || !strcmp (argv[i], "-irc"))
			{
			install ();
			}

		}
}

void TFOptions::install ()
{
	TFFileRC	*rcFile = NULL;
	char		*home, 
			buf[80],
			buf2[80];

	home = getenv ("HOME");
	if (SanityCheck::warning((!home), _("can't determine $HOME, cant create RC file"), "TFOptions::install"))
		return;

	bool	overwrite=TRUE;

	sprintf (buf, "%s/%s", home, TF_LOCAL_RC_FILE);
	rcFile = new TFFileRC (buf, NULL);

	// prompt if installing and overwriting with default settings
	if (rcFile->exists() && !s_settingsAreChanged)	
		overwrite = rcFile->promptOverwrite ();

	if (overwrite)
		{
		sprintf (buf2, "Creating %s ... ", buf);
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf2);
		rcFile->open ("w");
		if (rcFile->getFilePtr())
			{
			updateOptionTable ();
			rcFile->writeRCFile (this);
			}
		rcFile->close ();
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");
		}

	delete rcFile;

	exit (0);
}


/*
 *  processRCFile: process the resource file
 */
void TFOptions::processRCFile ()
{
	TFFileRC	*rcFile;
	TFOptionsTable	*optTbl=NULL;
	char		*buf[4], 
			*home, 
			t[128];
	int		lim = 0,
			c = 0;

	sprintf (t, "%s/%s", TF_DATADIR, TF_SYSTEM_RC_FILE);
	buf[lim++] = strdup (t);

	home = getenv ("HOME");
	if (SanityCheck::warning((!home), _("can't determine $HOME, cant find RC file"), "TFOptions::install"))
		return;

	sprintf (t, "%s/%s", home, TF_LOCAL_RC_FILE);
	buf[lim++] = strdup (t);

	while (c < lim)
		{
		rcFile = new TFFileRC (buf[c++], optTbl);
		if (rcFile->exists())
			optTbl = rcFile->parseRCFile ();
		delete rcFile;
		}

	if (optTbl)
		updateTFOptions (optTbl);
	else
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Couldn't find any resource files\n");
		

}


/*
 * updateTFOptions: write the options in optTbl to the actual TFOptions variables
 */
void TFOptions::updateTFOptions (TFOptionsTable *optTbl)
{
	gpointer	t;
	int		changed=0;

	t = optTbl->lookup (TF_OPT_FILL_SEA);
	if (t)
		{
		bool	b = (bool) atoi ((const char*) t);
		TFOptions::s_fillSea = b;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_DRAW_DERIVED_DATA);
	if (t)
		{
		bool	b = (bool) atoi ((const char*) t);
		TFOptions::s_drawDerivedData = b;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_FAST_HALF_RES);
	if (t)
		{
		bool	b = (bool) atoi ((const char*) t);
		TFOptions::s_fastHalfRes = b;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_FAST_HALF_Y_SCALE);
	if (t)
		{
		bool	b = (bool) atoi ((const char*) t);
		TFOptions::s_fastHalfYscale = b;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_FAST_ZROT);
	if (t)
		{
		bool	b = (bool) atoi ((const char*) t);
		TFOptions::s_fastZrot = b;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_SCALE);
	if (t)
		{
		int	i = atoi ((const char*) t);
		TFOptions::s_scale = i;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_SYNC_RATE);
	if (t)
		{
		int	i = atoi ((const char*) t);
		TFOptions::s_syncRate = i;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_RESOLUTION);
	if (t)
		{
		int	i = atoi ((const char*) t);
		TFOptions::s_resolution = i;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_YSCALE);
	if (t)
		{
		float	f = (float) atof ((const char*) t);
		TFOptions::s_yscale = f;
		changed++;
		}

	t = optTbl->lookup (TF_OPT_SEALEVEL);
	if (t)
		{
		float	f = (float) atof ((const char*) t);
		TFOptions::s_sealevel = f;
		changed++;
		}
	t = optTbl->lookup (TF_OPT_POV_FILE);
	if (t)
		{
		char *s = (char *) t;
		strcpy (TFOptions::s_POVfile, s);
		changed++;
		}
	t = optTbl->lookup (TF_OPT_POV_WIDTH);
	if (t)
		{
		int	i = (int) atoi ((const char*) t);
		TFOptions::s_POVwidth = i;
		changed++;
		}
	t = optTbl->lookup (TF_OPT_POV_YSCALE);
	if (t)
		{
		bool 	b = (bool) atoi ((const char*) t);
		TFOptions::s_POVhalfYscale = b;
		changed++;
		}
	t = optTbl->lookup (TF_OPT_POV_KEEP_FILES);
	if (t)
		{
		bool 	b = (bool) atoi ((const char*) t);
		TFOptions::s_POVkeepFiles = b;
		changed++;
		}

	if (changed)
		{
		char 		buf[80];

		s_settingsAreChanged = TRUE;
		sprintf (buf, "Updated %d params from OptionsTable to TFOptions\n", changed);
		GlobalTrace::trace (GlobalTrace::TRACE_FLOW, buf);
		}
}


/*
 * updateOptionTable: update the optionTable from TFOptions
 */
void TFOptions::updateOptionTable ()
{
	char 	buf[80];

	this->insert (const_cast<char*>(TF_OPT_FILL_SEA), 
				  const_cast<char*>(s_fillSea ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_DRAW_DERIVED_DATA), 
				  const_cast<char*>(s_drawDerivedData ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_FAST_HALF_RES), 
				  const_cast<char*>(s_fastHalfRes ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_FAST_HALF_Y_SCALE), 
				  const_cast<char*>(s_fastHalfYscale ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_FAST_ZROT), 
				  const_cast<char*>(s_fastZrot ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_POV_YSCALE), 
				  const_cast<char*>(s_POVhalfYscale ? "1" : "0"));
	this->insert (const_cast<char*>(TF_OPT_POV_KEEP_FILES), 
				  const_cast<char*>(s_POVkeepFiles ? "1" : "0"));

	sprintf (buf, "%d", s_scale);
	this->insert (const_cast<char*>(TF_OPT_SCALE), strdup(buf));
	sprintf (buf, "%d", s_syncRate);
	this->insert (const_cast<char*>(TF_OPT_SYNC_RATE), strdup(buf));
	sprintf (buf, "%d", s_resolution);
	this->insert (const_cast<char*>(TF_OPT_RESOLUTION), strdup(buf));
	sprintf (buf, "%d", s_POVwidth);
	this->insert (const_cast<char*>(TF_OPT_POV_WIDTH), strdup(buf));

	sprintf (buf, "%.2f", s_sealevel);
	this->insert (const_cast<char*>(TF_OPT_SEALEVEL), strdup(buf));
	sprintf (buf, "%.2f", s_yscale);
	this->insert (const_cast<char*>(TF_OPT_YSCALE), strdup(buf));

	sprintf (buf, "%s/%s", TF_DATADIR, TF_DEFAULT_POV_FILE);
	this->insert (const_cast<char*>(TF_OPT_POV_FILE), strdup(buf));
}


/*
 * helpRCFile: display the recognized flags for the RC file
 */
void TFOptions::helpRCFile ()
{
	printf ("terraform %d.%d.%d Resource File Info:\n", TF_MAJOR_VERSION, 
				TF_MINOR_VERSION, TF_MICRO_VERSION);
	printf (_("System data dir: %s, datafile: %s\n"), TF_DATADIR, TF_SYSTEM_RC_FILE);
	printf (_("User resorce file: $HOME/%s\n"), TF_LOCAL_RC_FILE);
	printf (_("\nRecognized terraform resource file tokens are:\n"));

	FlexArray	*flArr = this->collectOptions ();

	for (int i=0; i<flArr->getSize(); i++)
		{
		char *k = (char *)(flArr->El(i));
		hash_key_printer (k, NULL, stdout);
		}
}


/*
 * help: display recognized command line options 
 */
void TFOptions::help ()
{
	printf ("terraform %d.%d.%d\n", TF_MAJOR_VERSION, TF_MINOR_VERSION, 
		TF_MICRO_VERSION);
	printf ("  -b, --benchmark		run a quick spectral synthesis benchmark\n");
	printf ("  -dbg, --debug <level>		set debug to the specified level\n");
	printf ("  -f, --file <file>		read Height Field <file> upon startup\n");
	printf ("  -fm, --fixed_menus		make HF Window menus fixed in a menubar\n");
	printf ("  -h, --help 			print this help text\n");
	printf ("  -hrc, --helprc 		print resource file help\n");
	printf ("  -g, --generate 		generate a 400x400 Height Field upon startup\n");
	printf ("  -gs, --generate_size <size>	generate square HF with specified size\n");
	printf ("  -l, --large 			make slider amounts large where appropriate\n");
	printf ("  -irc, --installrc		install/create user resource file and exit\n");
	printf ("  -pexe, --pov_exec <name>	specify the name of the povray executable\n");
	printf ("  -p30, --pov30			use POVRay 3.0 style calls for render\n");
	printf ("  -s, --dont_save		don't save changed settings upon exit\n");
	printf ("  -version, --version		print version and exit\n");
	printf ("  -v, --verbose			verbose mode\n");
	printf ("  -?, --help			print help and exit\n");
	printf ("\n");
}



void hash_key_fetcher (gpointer key, gpointer data, gpointer user_data)
{
	char		*k = static_cast<char*>(key);
	FlexArray	*f = static_cast<FlexArray*>(user_data);

	f->append (k);
}


void hash_printer (gpointer key, gpointer data, gpointer user_data)
{
	char	*k = static_cast<char*>(key),
		*d = static_cast<char*>(data);
	FILE	*f = static_cast<FILE*>(user_data);

	fprintf (f, "%s=%s\n", k, d);
}


void hash_key_printer (gpointer key, gpointer data, gpointer user_data)
{
	char	*k = static_cast<char*>(key);
	FILE	*f = static_cast<FILE*>(user_data);

	fprintf (f, "\t%s\n", k);
}



