/*
  Module       : main.c
  Purpose      : GDK/Imlib Quick Image Viewer (qiv)
  More         : see qiv README
  Homepage     : http://www.klografx.de/
  Policy       : GNU GPL
*/	

#include <gdk/gdkx.h>
#include <gtk/gtkwidget.h>
#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include "qiv.h"
#include "main.h"

int main(int argc, char **argv)
{
  struct timeval tv;			/* Current time */
  GdkColormap *cmap;
  GdkImlibInitParams params;
  params.flags = PARAMS_DITHER;

  gdk_init(&argc,&argv);		/* Initialize GDK */
  gdk_imlib_init_params(&params);	/* Initialize Imlib */
  
  mod.brightness = mod.contrast = mod.gamma = 256;
  gettimeofday(&tv,NULL);	/* Randomize seed for 'true' random */
  srandom(tv.tv_usec*1000000+tv.tv_sec);
  
  image_idx = options_read(argc, argv);	/* Index of first image */
  images = argc - image_idx;	/* Number of images */
  image_names = &argv[image_idx];
  
  if(filter) /* Filter graphic images */
    filter_images(&images,image_names);
  
  if (!images) { /* No images to display */
    g_print("\nqiv: cannot load any images.\n");
    usage(argv[0],1);
  }

  image_idx = 0;		/* Display first image first */
  if (random_order)
    next_image(0);		/* ...except in random mode */

  if (to_root==1||to_root_t==1||to_root_s==1) {
    params.flags |= PARAMS_VISUALID;
    (GdkVisual*)params.visualid = gdk_window_get_visual(GDK_ROOT_PARENT());
  }

  gdk_imlib_init();
  signal(SIGTERM, finish);
  signal(SIGINT, finish);
  gtk_widget_push_visual(gdk_imlib_get_visual());
  gtk_widget_push_colormap(gdk_imlib_get_colormap());

  cmap = gdk_colormap_get_system();

  /* StatusBar background: ORANGE */
    text_background_color.red   = 65535;
    text_background_color.green = 47534;
    text_background_color.blue  = 0;

  if (!gdk_color_alloc(cmap, &text_background_color))
    fprintf(stderr, "qiv: couldn't allocate color (text_background_color)\n");

  /* error_background: BLUE */
    color_blue.red   = 0;
    color_blue.green = 0;
    color_blue.blue  = 65535;

  if (!gdk_color_alloc(cmap, &color_blue))
  {
    fprintf(stderr, "qiv: couldn't allocate color (color_blue),\nusing black...\n");
  }

  /* background: BLACK */
  if(!bg_set) {
    color_bg.red   = 0;
    color_bg.green = 0;
    color_bg.blue  = 0;
  }

  if (!gdk_color_alloc(cmap, &color_bg)) {
    fprintf(stderr, "qiv: couldn't allocate color (color_bg)");
  }

  screen_x=gdk_screen_width();
  screen_y=gdk_screen_height();
  cursor = gdk_cursor_new(CURSOR);
  
  qiv_load_image();		/* Load & display the first image */

  /* Setup callbacks */
  gdk_event_handler_set((GdkEventFunc)qiv_handle_event,NULL,NULL);
  g_timeout_add_full(G_PRIORITY_LOW,delay,(GSourceFunc)qiv_handle_timer,&slide,NULL);

  MainLoop = g_main_new(TRUE);	/* Allocate qiv main loop */
  g_main_run(MainLoop);		/* Run the loop */

  /*  main loop will never return */

  return(0);
}

void qiv_exit(int code)
{
  g_main_destroy(MainLoop);
  finish(SIGTERM);		/* deprecated, subject to change */
};


/*
 *	Slideshow
 */
 
void qiv_handle_timer(gpointer data)
{
  if (*(char *)data || slide) {
    next_image(0);
    qiv_load_image();
  }
}


/* 
 *	Handle GDK events 
 */

void qiv_handle_event(GdkEvent *ev)
{
  gint temp;
  gboolean exit_slideshow = FALSE; 
  switch(ev->type) {
    case GDK_DELETE:
      qiv_exit(0);
      break;

    case GDK_BUTTON_PRESS:
      jumping=0;		/* abort jump mode if a button is pressed */
      break;

      /* Use release instead of press (Fixes bug with junk being sent
	 to underlying xterm window on exit) */
    case GDK_BUTTON_RELEASE:
      exit_slideshow = TRUE;
      switch (ev->button.button) {
	case 1:		/* 1st or 5th button pressed */
	case 5:		/* scroll wheel down emulated by button 5 */
	  next_image(random_order?0:1);
	  qiv_load_image();
	  break;
	case 2:		/* 2nd button pressed */
	  qiv_exit(0);
	  break;
	case 3:		/* 3rd or 4th button pressed */
	case 4:		/* scroll wheel up emulated by button 4 */
	  next_image(random_order?0:-1);
	  qiv_load_image();
	  break;
      }
      break;

    case GDK_KEY_PRESS:

      exit_slideshow = TRUE;	/* Abort slideshow on any key by default */
   #ifdef DEBUG
      g_print("*** key:\n");	/* display key-codes */
      g_print("\tstring: %s\n",ev->key.string);
      g_print("\tkeyval: %d\n",ev->key.keyval);
   #endif
      if (jumping) {
	if((ev->key.keyval == 65293) ||  /* return key */
	   (ev->key.keyval == 65421) ||  /* return key  from numpad */
	   (jidx == 99)) {                  /* 99 digits already typed */
	  jcmd[jidx] = '\0';
	  jump2image(jcmd);
	  qiv_load_image();
	  jumping=0;
	}
          /* else record keystroke if not null */
	else if(ev->key.string && *(ev->key.string) != '\0')
	  jcmd[jidx++]=*(ev->key.string);
      } else {
	switch (ev->key.keyval) {

	    /* Exit */
	  case 65307: 	        /* ESC */
	  case   'Q':
	  case   'q':
            qiv_exit(0);
	    break;

	    /* Fullscreen mode (on/off) */
	  case 'F':
	  case 'f':
	    exit_slideshow = FALSE;
	    gdk_window_withdraw(win);
	    if(fullscreen) {
	      gdk_gc_destroy(gc);
	    }
	    fullscreen ^= 1;
	    first=1;
	    qiv_load_image();
	    break;

	    /* Maxpect on/off */
	  case 'M':
	  case 'm':
	    maxpect ^= 1;
	    reset_display_settings(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Random on/off */
	  case 'R':
	  case 'r':
	    random_order ^= 1;
	    break;

	    /* Statusbar on/off  */
	  case 'I':
	  case 'i':
	    exit_slideshow = FALSE;
	    statusbar ^= 1;
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Slide show on/off */
	  case 'S':
	  case 's':
	    exit_slideshow = FALSE;
	    slide ^= 1;
	    break;

	    /* Scale_down */
	  case 'T':
	  case 't':
	    scale_down ^= 1;
	    reset_display_settings(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Resize + */
	  case 65451:		/* numpad + */
	  case   '+':
	  case   '=':
	    zoom_in(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Resize - */
	  case 65453:		/* numpad - */
	  case   '-':
	    zoom_out(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Original (best fit) size */
	  case 65421:		/* numpad ENTER */
	  case 65293:		/* ENTER */
	    return_press = 1;
	    reset_display_settings(&win_x,&win_y,&w,&h);
	    mod.brightness=mod.contrast=mod.gamma=256;
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Next picture - or loop to the first */
	  case   ' ':		/* space */
	  case 65366:		/* PgDn */
	  case 65435:		/* numpad PgDn */
	    next_image(random_order?0:1);
	    qiv_load_image();
	    break;

	    /* Previous picture - or loop back to the last */
	  case 65288: 	/* backspace */
	  case 65365: 	/* PgUp */
	  case 65434: 	/* numpad PgUp */
	    next_image(random_order?0:-1);
	    qiv_load_image();
	    break;

	    /* + brightness */
	  case 'B':
	    if(mod.brightness < 512) {
	      mod.brightness += 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* - brightness */
	  case 'b':
	    if(mod.brightness > 0) {
	      mod.brightness -= 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* + contrast */
	  case 'C':
	    if(mod.contrast < 512) {
	      mod.contrast += 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* - contrast */
	  case 'c':
	    if(mod.contrast > 0) {
	      mod.contrast -= 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

             /* + gamma */        
	  case 'G':
	    if(mod.gamma < 512) {
	      mod.gamma += 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* - gamma */
	  case 'g':
	    if(mod.gamma > 0) {
	      mod.gamma -= 8;
	    }
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Delete image */
	  case 'D':
	  case 'd':
	  case 65535:             /* delete key */
	    move2trash(image_names[image_idx]);
	    qiv_load_image();
	    break;

	    /* Jump to image */
	  case 'J':
	  case 'j':
            jumping=1;
	    jidx=0;
	    break;

	    /* Flip horizontal */
	  case 'H':
	  case 'h':
	    gdk_imlib_flip_image_horizontal(im);
	    rotate_press = 1;
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Flip vertical */
	  case 'V':
	  case 'v':
	    gdk_imlib_flip_image_vertical(im);
	    rotate_press = 1;
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Rotate right */
	  case 65363:
	    gdk_imlib_rotate_image(im,1);
	    gdk_imlib_flip_image_horizontal(im);
	    temp = org_w;
	    org_w = org_h;
	    org_h = temp;
	    rotate_press = 1;
	    reset_display_settings(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Rotate left */
	  case 65361:
	    gdk_imlib_rotate_image(im,-1);
	    gdk_imlib_flip_image_vertical(im);
	    temp = org_w;
	    org_w = org_h;
	    org_h = temp;
	    rotate_press = 1;
	    reset_display_settings(&win_x,&win_y,&w,&h);
	    update_image(image_error,win_x,win_y,w,h,mod);
	    break;

	    /* Center image on background */
	  case 'X':
	  case 'x':
	    to_root=1;
	    set_desktop_image();
	    to_root=0;
	    break;

	    /* Tile image on background */
	  case 'Y':
	  case 'y':
	    to_root_t=1;
	    set_desktop_image();
	    to_root_t=0;
	    break;

	  case 'Z':
	  case 'z':
	    to_root_s=1;
	    set_desktop_image();
	    to_root_s=0;
	    break;
	  default:
	    break;
	}
      }
    default:
      break;
  }
  if (exit_slideshow) slide=0;
}

