/***************************************************************************
                          rfonts.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rfonts.cpp 1998/10/09 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
**  Usage:
**  ~~~~~~
**  RFonts* fonts = RFonts::getRFonts();
**    -> returns always the same fonts object
**
*****************************************************************************/

#include "rfonts.h"

#include <qdir.h>

#include "relement.h"
#include "rfile.h"
#include "rconfig.h"
#include "rlog.h"
#include "rprgdef.h"


RFonts* RFonts::theOneAndOnlyRFonts=0;


// Get the one and only RFonts-Object:
//   create a new one on first call
//
RFonts* 
RFonts::getRFonts()
{
  if(theOneAndOnlyRFonts==0) {
    theOneAndOnlyRFonts = new RFonts;
  }
  return theOneAndOnlyRFonts;
}



// Constructor:
//
RFonts::RFonts()
{
  font=0;
  numFonts=0;
  reset();
  init();
}


// Initialize fonts:
//
void
RFonts::init()
{
  // Lookup fonts from directory relative to program path:
  lookupFonts( RCONFIG->getPrgDir().path()+"/fonts");

#ifndef DEF_WINDOWS
  // Lookup from /usr/share/<proj>/fonts:
  lookupFonts( QString("/usr/share/")+DEF_APPNAME_L+"/fonts" );

  // Lookup from /usr/local/<proj>/fonts:
  lookupFonts( QString("/usr/local/")+DEF_APPNAME_L+"/fonts" );

  // Lookup from ~/.<proj>/fonts:
  lookupFonts( RCONFIG->getConfigDir().path()+"/fonts" );
#endif

  readFonts();
}



// Destructor:
//
RFonts::~RFonts()
{
  deleteFonts();
}



// Delete Fonts:
//
void
RFonts::deleteFonts()
{
  if(font && numFonts>0) {
    delete[] font;
  }
}



// Reset all Fonts:
//
void
RFonts::reset()
{
  int fc, lc;
  for(fc=0; fc<numFonts; ++fc) {
    for(lc=0; lc<256; ++lc) {
      font[fc].letter[lc]=0;
      font[fc].elNum[lc]=0;
      font[fc].letWidth[lc]=0.0;
    }
    font[fc].fontName[0]='\0';
    font[fc].letterSpacingFactor=0.2;
    font[fc].wordSpacingFactor=0.6;
    font[fc].lineDistanceFactor=1.4;
  }
}



// Lookup fonts:
//
void
RFonts::lookupFonts( QString _fontPath )
{
  if( !QDir(_fontPath).exists() ) return;

  QDir fontDir( _fontPath, "*.cxf" );
  QStringList lst = fontDir.entryList();
  QStringList::ConstIterator fn;

  // Add new fonts:
  //
  for( fn = lst.begin(); fn!=lst.end(); ++fn ) {
    if( fontPathList.contains( *fn )==0 ) {
      fontPathList.append( _fontPath + "/" + *fn );

      RLOG("\nFont found: ");
      RLOG( (_fontPath + "/" + *fn).latin1() );
    }
  }

}



// Read fonts from files:
//
void
RFonts::readFonts()
{
  QStringList::ConstIterator fn;
  int fi;                                     // Index of font (font[i])

  if(fontPathList.count()>0) {
    QString fnp;   // File name with path
    
    numFonts=fontPathList.count();
    font = new RFont[numFonts];

    for(fn = fontPathList.begin(), fi=0; fn != fontPathList.end(); ++fn, ++fi ) {
      fnp = *fn;
      
      if(fileCheckFormat(fnp.latin1(), "CAM Expert Font")) {
 
        FILE* fp=fopen(fnp.latin1(), "rt");
    
        if(fp) {
          unsigned char c;       // a character of the file
          unsigned char letNum;  // the actual letter number
          char elCode[5];  // code of actual element ('L' 'LT' 'A')
          int  i;          // counter
          float v1,        // values (element coordinates)
                v2,        //
                v3,        //
                v4,        //
                v5;        //
          float ax=0.0,    // actual coordinate (for Lineto)
                ay=0.0;    //
          float elMaxX;    // Max. X-Coordinate of an element (for letWidth)

          // Get Font name:
          //
          fileGetFormatInfo(fnp.latin1(), "Name", font[fi].fontName, 30);
          
          // Get Font dimensions (proportions):
          //
          char dum[128];
          fileGetFormatInfo(fnp.latin1(), "Dimensions", dum, 127);
          sscanf(dum, 
                 "%f %f %f", 
                 &font[fi].letterSpacingFactor,
                 &font[fi].wordSpacingFactor,
                 &font[fi].lineDistanceFactor);
  
          while(!feof(fp)) {
      
            // Read to next letter-head ('[L]')
            //
            c=' ';
            while(!feof(fp) && c!='[') c=fgetc(fp);
      
            if(!feof(fp)) {
      
              // Read the letter number:
              //
              letNum=fgetc(fp);

              // Overread ']':
              //
              fgetc(fp);
      
              // Read number of elements for this letter:
              //
              fscanf(fp, "%d", &font[fi].elNum[letNum]);

              if(font[fi].elNum[letNum]>0) {
      
                // Allocate memory for the letter:
                //
                font[fi].letter[letNum] = new RElement[font[fi].elNum[letNum]];
        
                font[fi].letWidth[letNum] = 0.0;
        
                // Read the elements
                //
                for(i=0; i<font[fi].elNum[letNum]; ++i) {
                  elCode[0]='\0';
                  fscanf(fp, "%2s", elCode);
        
                  // Line:
                  //
                  if(!strcmp(elCode, "L")) {
                    fgetc(fp);
                    fscanf(fp, "%f", &v1);
                    fgetc(fp);
                    fscanf(fp, "%f", &v2);
                    fgetc(fp);
                    fscanf(fp, "%f", &v3);
                    fgetc(fp);
                    fscanf(fp, "%f", &v4);
      
                    font[fi].letter[letNum][i].createLine(v1, v2, v3, v4);
        
                    ax=v3;
                    ay=v4;
                  }
        
                  // Line to:
                  //
                  if(!strcmp(elCode, "LT")) {
                    fgetc(fp);
                    fscanf(fp, "%f", &v1);
                    fgetc(fp);
                    fscanf(fp, "%f", &v2);
        
                    font[fi].letter[letNum][i].createLine(ax, ay, v1, v2);
  
                    ax=v1;
                    ay=v2;
                  }
        
                  // Arc:
                  //
                  if(!strcmp(elCode, "A")) {
                    fgetc(fp);
                    fscanf(fp, "%f", &v1);
                    fgetc(fp);
                    fscanf(fp, "%f", &v2);
                    fgetc(fp);
                    fscanf(fp, "%f", &v3);
                    fgetc(fp);
                    fscanf(fp, "%f", &v4);
                    fgetc(fp);
                    fscanf(fp, "%f", &v5);
        
                    font[fi].letter[letNum][i].createArc(v1, v2, v3, v4, v5, false);
                  }
        
                  // Arc (reversed):
                  //
                  if(!strcmp(elCode, "AR")) {
                    fgetc(fp);
                    fscanf(fp, "%f", &v1);
                    fgetc(fp);
                    fscanf(fp, "%f", &v2);
                    fgetc(fp);
                    fscanf(fp, "%f", &v3);
                    fgetc(fp);
                    fscanf(fp, "%f", &v4);
                    fgetc(fp);
                    fscanf(fp, "%f", &v5);
        
                    font[fi].letter[letNum][i].createArc(v1, v2, v3, v4, v5, true);
                  }
  
                  if(!strcmp(elCode, "L")  ||
                     !strcmp(elCode, "LT") ||
                     !strcmp(elCode, "A")  ||
                     !strcmp(elCode, "AR")    ) {
                     
                    elMaxX = font[fi].letter[letNum][i].right();
  
                    if(elMaxX > font[fi].letWidth[letNum]) {
                      font[fi].letWidth[letNum] = elMaxX;
                    }
                  }
                }
                
              }
            }
          }
          fclose(fp);
          
        }
        
      }
    }
  }
}



// Get Font number from its name:
//
int  
RFonts::getFontNumber(const char* _name)
{
  int i,        // counter
      ret=0;    // returned value

  for(i=0; i<numFonts; ++i) {
    if(!strcmp(font[i].fontName, _name)) {
      ret=i;
      break;
    }
  }

  return ret;
}



// Get Font name from its number:
//
char*
RFonts::getFontName(int _num)
{
  if(isFontValid(_num)) {
    return font[_num].fontName;
  }
  return "";
}



// Get Font letterSpacingFactor from its number:
//
float
RFonts::getFontLetterSpacingFactor(int _num)
{
  if(isFontValid(_num)) {
    return font[_num].letterSpacingFactor;
  }
  return 0.2;
}



// Get Font wordSpacingFactor from its number:
//
float
RFonts::getFontWordSpacingFactor(int _num)
{
  if(isFontValid(_num)) {
    return font[_num].wordSpacingFactor;
  }
  return 0.6;
}



// Get Font lineDistanceFactor from its number:
//
float
RFonts::getFontLineDistanceFactor(int _num)
{
  if(isFontValid(_num)) {
    return font[_num].lineDistanceFactor;
  }
  return 1.4;
}



// Get width of letter _letter from font _font:
//   (for text with height: 9.0mm)
//
float   
RFonts::getLetterWidth(int _font, 
                       unsigned char _letter, 
                       float _height)
{
  float ret=0.0;
  
  if(isLetterValid(_letter)       &&
     isFontValid(_font)           &&
     font[_font].elNum[_letter]>0 &&
     font[_font].letter[_letter]     ) {
     
    float fact=_height/9.0;
    ret=font[_font].letWidth[_letter]*fact;
  }
  
  return ret;
}



// Get width of text _text with font _font:
//
//   _height: letter height
//   _letSpace: letter spacing (gap between two letters of a word)
//   _wordSpace: word spacing (gap between two words)
//
float
RFonts::getTextWidth(int _font, 
                     const unsigned char* _text,
                     float _height, 
                     float _letSpace, 
                     float _wordSpace)
{
  float ret=0.0;
  
  if(_text && isFontValid(_font)) {
    int i;
    float curX=0.0;

    for(i=0; i<(int)strlen((char*)_text); ++i) {
      switch(_text[i]) {
        case 10:
          if(curX>ret) ret=curX;
          curX=0.0;
          break;
        case 13:
          break;
        //case 32:
        //  curX+=_wordSpace;
        //  break;
        default:
          //curX += getLetterWidth(_font, _text[i], _height) 
          //        + _letSpace;
          curX = getTextLineWidth(_font, _text, i, _height, _letSpace, _wordSpace);
          while(_text[i]!=10   && 
                _text[i]!='\0' && 
                i<(int)strlen((char*)_text) ) ++i;
          break;
      }
    }
    if(curX>ret) ret=curX;
  }
  return ret;
}



// Get width of textline _text at index (_index) with font _font:
//
//   _index: index of first letter of line
//   _height: letter height
//   _letSpace: letter spacing (gap between two letters of a word)
//   _wordSpace: word spacing (gap between two words)
//   _fixedWidth: use fixed width for the letters (like 'W')
//
float   
RFonts::getTextLineWidth(int _font, 
                         const unsigned char* _text,
                         int _index,
                         float _height, 
                         float _letSpace, 
                         float _wordSpace,
                         bool _fixedWidth)
{
  float ret=0.0;
  
  if(_text                && 
     isFontValid(_font)   &&
     _index>=0            && 
     _index<(int)strlen((char*)_text)    ) {
     
    int i;
    float curX=0.0;
    bool lastCharWasALetter=false; // Last char was a letter (with a letter Space)
    float letterWidth = getLetterWidth(_font, 'W', _height) + _letSpace;

    for(i=_index; i<(int)strlen((char*)_text); ++i) {
      switch(_text[i]) {
        case 32:
          if(_fixedWidth) {
            curX += letterWidth;
          }
          else {
            curX += _wordSpace;
            if(lastCharWasALetter) curX -= _letSpace;
            lastCharWasALetter=false;
          }
          break;
        default:
          if(_text[i]>=33 &&
             isLetterAvailable(_font, _text[i])) {

            if(_fixedWidth) {
              curX += letterWidth;
            }
            else {
              curX += getLetterWidth(_font, _text[i], _height)
                      + _letSpace;
              lastCharWasALetter=true;
            }
          }
          break;
      }

      if(_text[i]==10 || _text[i]==13) break;
    }
    ret=curX-_letSpace;
    if(ret<0.0) ret=0.0;
  }
  return ret;
}
     



// Get height of text _text with font _font:
//
//   _lineDist: Distance from line to line
//
float
RFonts::getTextHeight(int _font, 
                      const unsigned char* _text,
                      float _height, 
                      float _lineDist)
{
  float ret=0.0;
  
  if(_text && isFontValid(_font)) {
    int i;
    float fact=_height/9.0;

    ret = _height*fact;  // Height of first line

    for(i=0; i<(int)strlen((char*)_text); ++i) {
      if(_text[i]==10) {
        ret+=_lineDist;
      }
    }
  }
  return ret;
}



// Get element _index from _letter of _font:
//
//   _x/_y:  offset of letter
//   _angle: angle of letter
//   _align: alignment (E_LEFT / E_RIGHT / E_CENTER)
//   _index: index of letter in struct RFont (letter)
//   _el:    reserved memory for the element
//
void
RFonts::getLetterElement(int _font, 
                         unsigned char _letter, 
                         float _height,
                         float _x, float _y,
                         float _angle,
                         uint _align,
                         int _index,
                         RElement* _el)
{
  if(_el                    &&
     isFontValid(_font)     &&
     isLetterValid(_letter) &&
     _index>=0              &&
     _index<font[_font].elNum[_letter] ) {

    float fact = _height/9.0;   // factor
    float letOffs=0.0;          // letter offset

    // Calculate the offset of the letter:
    //
    switch(_align) {
      case E_LEFT:         // align left
        letOffs=0.0;
        break;
      case E_RIGHT:        // align right
        letOffs=-font[_font].letWidth[_letter];
        break;
      case E_CENTER:       // align center
        letOffs=-font[_font].letWidth[_letter]/2.0;
        break;
      default:
        break;
    }
    letOffs*=fact;

    _el->getMeasuresFrom(&font[_font].letter[_letter][_index]);

    _el->scale(fact, 0.0, 0.0, false);
    _el->move(letOffs, 0.0, false);
    _el->rotate(_angle, 0.0, 0.0, false);
    _el->move(_x, _y, true);
  }
}



// Get number of elements in letter _letter of font _font:
//
int     
RFonts::getLetterElementNumber(int _font, 
                               unsigned char _letter)
{
  if(isFontValid(_font)     &&
     isLetterValid(_letter)    ) {
    return font[_font].elNum[_letter];
  }
  else {
    return 0;
  }
}



// Check if a letter is available (has at least one element):
//
bool 
RFonts::isLetterAvailable(int _font, 
                          unsigned char _letter)

{ 
  if(isLetterValid(_letter) && 
     isFontValid(_font)     && 
     getLetterElementNumber(_font, _letter)>0) {
    return true;
  }
  else {
    return false; 
  }
}
  
  

// EOF












