/***************************************************************************
                          relement.h  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/****************************************************************************
** relement.h 1998/08/24 A. Mustun RibbonSoft 
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#ifndef RELEMENT_H
#define RELEMENT_H

#include <qcolor.h>
#include <qstring.h>
#include <qnamespace.h>

#include "rflags.h"
#include "rgraphdef.h"
#include "rtypes.h"
#include "rprgdef.h"

#define E_VISIBLE      0x00000001       // Element is visible
#define E_ACTIVE       0x00000002       //            active (not in Undo memory)
#define E_REVERSED     0x00000004       // Arc is reversed (clockwise; G02)
#define E_TOOLOFFL     0x00000008       // Tool offset Left on
#define E_TOOLOFFR     0x00000010       // Tool offset Right on
#define E_TOOLOFFO     0x00000020       // Tool offset Off
#define E_CONV         0x00000040       // Element is already converted
#define E_CONTBEG      0x00000080       // Element is at contour beginning
#define E_CONTEND      0x00000100       // Element is at contour end
#define E_TANGENTIAL   0x00000200       // Startway is tangential
#define E_BROKEN       0x00000400       // Element is highlighted (broken contour)
#define E_BROKEN1      0x00000800       // Elements startpoint is highlighted (broken contour)
#define E_BROKEN2      0x00001000       // Elements endpoint   is highlighted (broken contour)
#define E_STRAIGHT     0x00002000       // Straight Text
#define E_ROUNDIN      0x00004000       // Round Text readable from center / dimension
#define E_ROUNDOUT     0x00008000       // Round Text readable from outside / dimension
#define E_LEFT         0x00010000       // Text alignment: left
#define E_RIGHT        0x00020000       // Text alignment: right
#define E_CENTER       0x00040000       // Text alignment: centered
#define E_TAGGED       0x00080000       // Element tagged
#define E_TAGSTP       0x00100000       // Elements startpoint tagged (stretch)
#define E_TAGEND       0x00200000       // Elements endpoint tagged (stretch)
#define E_UNDO         0x00400000       // Element is in undo memory
#define E_REDO         0x00800000       // Element is in redo memory
#define E_HORIZONTAL   0x01000000       // Horizontal dimension
#define E_VERTICAL     0x02000000       // Vertical dimension
#define E_DIAMETER     0x04000000       // Diameter dimension
#define E_RADIUS       0x08000000       // Radius dimension
#define E_ARROW        0x10000000       // Arrow dimension
#define E_FIXEDWIDTH   0x20000000       // Fixed width for texts
#define E_PROPORTIONAL 0x40000000       // Proportional flag for texts
  

enum elementEnum { // Collectives (Groups):
                   T_NO, T_ALL, T_CIRCULAR, T_STRAIGHT, T_LINE_ARC, T_LONG,
                   
                   // Singles:
                   T_POINT, T_LINE, T_ARC, T_CIRCLE,
                   T_TEXT, T_RECT, T_RAPID,
                   T_STARTWAY, T_ENDWAY,
                   T_DIMENSION, T_HATCHING };

class RGraphic;
class RPainter;

class RElement : public RFlags
{
public:
  RElement(RGraphic* _graphic=0, 
           unsigned int _flags=0);
  ~RElement();

  void  reset();

  elementEnum getElementTyp() const { return elementTyp; }
  void  setElementTyp(elementEnum _typ) { elementTyp=_typ; }

  void  setGraphic(RGraphic* _graphic) { graphic=_graphic; }
  RGraphic* getGraphic() { return graphic; }

  byte  getLayer() { return layer; }
  void  setLayer(byte _lay) { layer=_lay; }
  int   getContour() { return contour; }
  void  setContour(int _cont) { contour=_cont; }
  float getX1() const  { return x1; }
  void  setX1(float _x1)  { x1=_x1; }
  float getY1() const  { return y1; }
  void  setY1(float _y1)  { y1=_y1; }
  float getX2() const  { return x2; }
  void  setX2(float _x2)  { x2=_x2; }
  float getY2() const  { return y2; }
  void  setY2(float _y2)  { y2=_y2; }
  float getCx() const  { return cx; }
  void  setCx(float _cx)  { cx=_cx; }
  float getCy() const  { return cy; }
  void  setCy(float _cy)  { cy=_cy; }
  float getCr() const  { return cr; }
  void  setCr(float _cr)  { cr=_cr; }
  float getA1() const  { return a1; }
  void  setA1(float _a1)  { a1=_a1; }
  float getA2() const  { return a2; }
  void  setA2(float _a2)  { a2=_a2; }
  int   getWidth() { return width; }
  void  setWidth(int _width) { width=_width; }
  QColor getColor() { return color; }
  void  setColor(QColor _color) { color=_color; }
  void  setNamedColor(QString _name) { color.setNamedColor(_name); }
  Qt::PenStyle getStyle() { return style; }
  void  setStyle(Qt::PenStyle _style) { style=_style; }

  void  setDestructionNumber(byte _unumD) { unumD = _unumD; }
  void  setCreationNumber   (byte _unumC) { unumC = _unumC; }
  byte  getDestructionNumber() { return unumD; }
  byte  getCreationNumber()    { return unumC; }

  unsigned char* getText() const { if(!text.isEmpty()) return (unsigned char*)text.data();
                          else                return 0;                             }
  void  setText(const char* _text) { text = _text; }

  uint  getTextMode() const { if     (getFlag(E_ROUNDOUT)) return E_ROUNDOUT;
                              else if(getFlag(E_ROUNDIN))  return E_ROUNDIN;  
                              else                         return E_STRAIGHT; }
  uint  getTextAlign() const { if     (getFlag(E_CENTER))  return E_CENTER;
                               else if(getFlag(E_RIGHT))   return E_RIGHT;
                               else                        return E_LEFT; }

  float getLetterSpacing() const { return x2; }
  void  setLetterSpacing(float _ls) { x2=_ls; }

  float getTextHeight() const { return y2; }
  void  setTextHeight(float _th) { y2=_th; }

  float getWordSpacing() const { return cx; }
  void  setWordSpacing(float _ws) { cx=_ws; }

  float getLineDistance() const { return cy; }
  void  setLineDistance(float _ld) { cy=_ld; }

  int   getFont() const { return (int)a2; }
  void  setFont(int _fnt) { a2=(float)_fnt; }
  
  /*! Get Font flags <br> (E_STRAIGHT, E_ROUNDIN, E_ROUNDOUT, E_LEFT, E_RIGHT, E_CENTER)
  */
  uint getFontFlags() { return getFlags()&(E_STRAIGHT|E_ROUNDIN|E_ROUNDOUT|E_LEFT|E_RIGHT|E_CENTER|E_FIXEDWIDTH|E_PROPORTIONAL); }

  bool  getFixedWidth() { return getFlag(E_FIXEDWIDTH); }

  int   getNumberOfTextLines() const;
  void  getTextLine(int _num, char* _line, int _max) const;

  float getDimDist() const  { return cr; }
  void  setDimDist(float _cr)  { cr=_cr; }

  void  getDimText(char* _text, int _len) const;
  void  setDimText(const char* _text) { text = _text; }

  bool  hasEndpoint() const;
  bool  isCircular() const;
  bool  isInGroup(int _typGroup) const;

  bool  compareWith(const RElement* _element, float _tolerance=DEF_MMTOL, bool _reverse=false);

  void  getDataFrom(const RElement* _element);
  void  getMeasuresFrom(const RElement* _element);
  void  getAttribFrom(const RElement* _element);

  float getAngleAmount() const;
  float getAngleFrom1To2() const;
  float getLength() const;
  
  float getCenterX() const;
  float getCenterY() const;
  float getMiddleX() const;
  float getMiddleY() const;
  float getMiddleDirection() const;
  float getDirection1() const;
  float getDirection2() const;
  void  getDistPoint(float* _resx, float* _resy, float _dist, bool _startPoint) const;
  void  getTouchPoint(float* _resx, float* _resy, float _mx, float _my) const;

  float getDistanceToPoint(float _px, float _py, bool _noLimit=false) const;
  float getDistanceFromPointToPoint(float _px, float _py) const;
  float getDistanceFromLineToPoint(float _px, float _py, bool _noLimit=false) const;
  float getDistanceFromArcToPoint(float _px, float _py, bool _noLimit=false) const;
  float getDistanceFromCircleToPoint(float _px, float _py) const;
  float getDistanceFromTextToPoint(float _px, float _py) const;

  float getRelAngleToPoint(float _px, float _py, bool _startPoint) const;
  
  /*! Get Intersection between this element and an other one:
      \param _el    The other element
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
      \param _onElements true: The intersections must be on both elements
                         false: The intersections can be outside the elements 
  */
  void  getIntersection(const RElement* _el, 
                        bool* _ires1, float* _ix1, float* _iy1, 
                        bool* _ires2, float* _ix2, float* _iy2,
                        bool* _itan,
                        bool _onElements=true ) const;
                        
  /*! Get Intersection between this line and an other one:
      \param _el    The other line
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the intersection is retruned in this variable
  */
  void  getIntersectionLineWithLine(const RElement* _el, 
                                    bool* _ires1, float* _ix1, float* _iy1) const;
                                    
  /*! Get Intersection between this line and an arc:
      \param _el    The arc
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
  */
  void  getIntersectionLineWithArc(const RElement* _el, 
                                   bool* _ires1, float* _ix1, float* _iy1,
                                   bool* _ires2, float* _ix2, float* _iy2,
                                   bool* _itan) const;
                                   
  /*! Get Intersection between this arc and an other one:
      \param _el    The other arc
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
  */
  void  getIntersectionArcWithArc(const RElement* _el, 
                                  bool* _ires1, float* _ix1, float* _iy1,
                                  bool* _ires2, float* _ix2, float* _iy2,
                                  bool* _itan) const;
                                  
  /*! Get Gliding ("The point somewhere between the elements") between this element and an other:
      \param _el   The other element
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGliding(const RElement* _el, 
                   bool* _gres, float* _gx, float* _gy) const;
                   
  /*! Get Gliding between this line and an arc:
      \param _el   The arc
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGlidingLineWithArc(const RElement* _el, 
                              bool* _gres, float* _gx, float* _gy) const;
                              
  /*! Get Gliding between this arc and an other one:
      \param _el   The other arc
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGlidingArcWithArc(const RElement* _el, 
                             bool* _gres, float* _gx, float* _gy) const;
                                  
  /*! Get all intersections (<=8) of all paralles (<=4) of the two elements (this, _el)
      \param _el The other element
      \param _dist Distance to parallels
      \param _ires Pointer to Array of eight bools (for returning if intersection n is valid)
      \param _ipx  Pointer to Array of eight floats (for returning the x-values of the intersection points)
      \param _ipy  Pointer to Array of eight floats (for returning the y-values of the intersection points)
  */
  void  getParallelIntersections(const RElement* _el, 
                                 float _dist,
                                 bool* _ires, float* _ipx, float* _ipy) const;
                                 
                        
  /*! Get the point which we must trim if the user clicked an element for trimming
      \param _mx Mouse coordinate x
      \param _my Mouse coordinate y
      \param _ix Intersection point x
      \param _iy Intersection point y
      \return true: trim point is endpoint
              false: trim point is startpoint
  */
  bool  getTrimPoint(float _mx, float _my, float _ix, float _iy);
  
  /*! Trim an element to a point
      \param _ep Trim endpoint?
      \param _ix1/_iy1 Trim to this point
      \param _ix2/_iy2 Trim circles also to this point
      \param _mang Angle from circle center to mouse click
  */
  void trim(bool _ep,
            float _ix1, float _iy1,
            float _ix2, float _iy2,
            float _mang);

  
  /*! Check if element was trimmed to zero
  */
  bool  isTrimmedToZero(bool _ep, float _ix1, float _iy1);
                                  
  bool  isPointOnElement(float _x, float _y, float _tol=DEF_MMTOL) const;
  bool  isPointInsideSubElements(float _x, float _y) const;
  bool  isParallelWith(const RElement* _el) const;

  bool  getParallel(RElement* _parallel, 
                    float _px, float _py, 
                    float _dist) const;
  void  getParallels(RElement* _parallel1, 
                     RElement* _parallel2, 
                     float _dist) const;

  bool  getBisector(RElement* _bisector, 
                    const RElement* _leg, 
                    float _px, float _py,
                    float _length) const;
  void  getBisectors(RElement* _bisector1, 
                     RElement* _bisector2, 
                     RElement* _bisector3, 
                     RElement* _bisector4, 
                     const RElement* _leg, 
                     float _ix, float _iy,
                     float _length         ) const;

  bool getTangent1(RElement* _tangent,
                   float _x, float _y,
                   float _px, float _py) const;
  bool getTangents1(RElement* _tangent1,
                    RElement* _tangent2,
                    float _x, float _y) const;

  bool getTangent2(RElement* _tangent,
                   const RElement* _secondCircle,
                   float _px, float _py) const;
  bool getTangents2(RElement* _tangent1,
                    RElement* _tangent2,
                    RElement* _tangent3,
                    RElement* _tangent4,
                    const RElement* _secondCircle) const;

  bool  getOrtho(RElement* _ortho,
                 float _px, float _py,
                 float _ang, float _length) const;

  void  forceDirection(bool _pos);

  void  recalculateBorders();
  void  recalculateTextBorders();
  void  recalculateAngles();
  void  recalculateEndpoints();
  float left() const   { return leftBorder; }
  float bottom() const { return bottomBorder; }
  float right() const  { return rightBorder; }
  float top() const    { return topBorder; }
  void  getBorders(float* _left, float* _bottom, float* _right, float* _top) const;

  bool  isOnScreen() const;

  void  createPoint(float _x1, float _y1, byte _layer=0);
  void  createLine(float _x1, float _y1, 
                   float _x2, float _y2, byte _layer=0);
  void  createArc(float _cx, float _cy, 
                  float _cr, 
                  float _a1, float _a2, 
                  bool _reversed, byte _layer=0);
  void  createArcPPP(float _x1, float _y1,
                     float _x2, float _y2,
                     float _x3, float _y3,
                     bool  _reversed,
                     byte _layer=0);
  void  createCircle(float _cx, float _cy, 
                     float _cr, 
                     float _a1, float _a2, 
                     bool _reversed, byte _layer=0);
  void  createPolylineElement(float _x1, float _y1,
                              float _x2, float _y2,
                              float _ab, byte _layer=0);
  void  createRect(float _x1, float _y1, 
                   float _x2, float _y2, byte _layer=0);
  void  createText(float _x, float _y,
                   int   _font,
                   const unsigned char* _text,
                   unsigned int _flags,
                   float _textheight,
                   float _angle=0.0,
                   float _radius=0.0,
                   float _letterspace=DEF_AREAMAX,
                   float _wordspace=DEF_AREAMAX,
                   float _linedistance=DEF_AREAMAX,
                   byte _layer=0);
  void  createRapid(float _x1, float _y1, byte _layer=0);
  void  createRapid(float _x1, float _y1,
                    float _x2, float _y2, 
                    byte _layer=0);
  void  createDimension(float _v1, float _v2, float _v3, float _v4,
                        float _v5, float _v6,
                        float _pos,
                        uint _flags=0,
                        byte _layer=0);
  void  createHatching(float _factor, QString _pattern,
                        byte _layer=0);

  void  turn();
  
  /*! Draw the element
      \param _paint Pointer to RPainter drawing device
      \param _ox    Real offset in X
      \param _oy    Real offset in Y
      \param _prev  Element is in preview (no texts, no changing of color if tagged)
      \param _curr  Explicit use of current settings of painting device (no changing of color or mode)
  */
  void  draw(RPainter* _paint, 
             float _ox=0.0, 
             float _oy=0.0, 
             bool _prev=false,
             bool _curr=false);
  void  fastDraw(RPainter* _paint);
  void  drawSmooth(RPainter* _paint);

  void  drawSmoothLine(RPainter* _paint);
  void  drawSmoothArc(RPainter* _paint);

  void  move(float _x, float _y, bool _updBorder=true);
  void  rotate(float _angle, float _cx, float _cy, bool _updBorder=true);
  void  scale(float _fact, float _cx, float _cy, bool _updBorder=true);
  void  mirror(float _x1, float _y1, float _x2, float _y2, bool _updBorder=true);
  void  moveStartPoint(float _x, float _y, bool _updBorder=true);
  void  moveEndPoint(float _x, float _y, bool _updBorder=true);

  void  tag()   { setFlag(E_TAGGED); }
  void  untag() { delFlag(E_TAGGED); }

  void  createSubElements(int _num);
  void  deleteSubElements();
  void  setSubElement(int _i, RElement* _el);
  RElement* getSubElement(int _i);
  int   getSubElementNum() { return subElementNum; }

  void  outputDebug();
  
protected:
  void  prepareDrawing(RPainter* _paint, int _x, int _y, bool _curr=false);
  void  finishDrawing(RPainter* _paint, int _x, int _y);

    
private:

  elementEnum  elementTyp;        // Element typ:
                                  //   T_POINT, T_LINE, 
                                  //   T_ARC, T_CIRCLE
                                  //   T_TEXT
  RGraphic*    graphic;           // Pointer to graphic
  float        leftBorder,        // borders
               topBorder,         // 
               rightBorder,       //
               bottomBorder;      // 
  byte         layer;             // Layer number (0=no layer)
  int          contour;           // Contour number (-1=no contour)
  byte         unumD,             // UNDO-Number when element was deleted
               unumC;             // UNDO-Number when element was constructed
  float        x1,                // Start point || Reference point for text || Start radius for angle dimension extension line 1
               y1,                //             ||
               x2,                // End point   || Letter space for text || Start radius for angle dimension extension line 2
               y2,                //             || Textheight   for text
               cx,                // Center      || Word space   for text
               cy,                //             || Line dist.   for text
               cr,                // Radius      || Rad. 1st line for text || Pos. for dimension line || Factor for hatching
               a1,                // Start angle || Ref. Angle   for text
               a2;                // End angle   || Font         for text
  byte         width;             // Line width
  QColor       color;             // Line color
  Qt::PenStyle style;             // Line Style (NoPen, SolidLine, DashLine, DotLine, DashDotLine, DashDotDotLine)
  QCString     text;              // text || Alternate text for dimensions || Pattern for hatching
  RElement*    subElement;        // Pointer to array of elements (Border elements for hatchings)
  int          subElementNum;     // Number of sub elements

};


#endif

// EOF





