/***************************************************************************
                          rconfig.cpp  -  description
                             -------------------
    begin                : Sun Sep 12 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rconfig.h"

#include <qapplication.h>
#include <qfile.h>
#include <qregexp.h>
#include <qtextstream.h>

#include "rfile.h"
#include "rfileparser.h"
#include "rlog.h"

/**
 * \class RConfig
 * Class RConfig provides access to global settings.
 */

RConfig* RConfig::theOneAndOnlyConfig=0;


/**
 * Gets the one and only Config-Object.
 *   Creates a new one on first call.
 */
RConfig*
RConfig::getConfig()
{
  if(theOneAndOnlyConfig==0) {
    theOneAndOnlyConfig = new RConfig;
    theOneAndOnlyConfig->ini();
  }
  return theOneAndOnlyConfig;
}



/**
 * Constructor.
 */
RConfig::RConfig()
 : RSettings('=', '"', '"')
{
  cadDimensionTextHeight=5.0;
  cadDimensionTextAutoScale=false;
  cadDimensionTextDistance=2.0;
  cadDimensionOverLength=2.0;
  cadDimensionArrowLength=3.0;

  languageList.setAutoDelete(true);
}



/**
 * Destructor.
 */
RConfig::~RConfig()
{
}



/**
 * Initializes the config object (only called once).
 */
void
RConfig::ini()
{
  addSetting( "Language:Language",  "en" );

  addSetting( "Application:FontSize0", RUNNING_WINDOWS ? "10" : "12" );
  addSetting( "Application:FontSize1", RUNNING_WINDOWS ? "10" : "12" );
  addSetting( "Application:FontSize2", RUNNING_WINDOWS ? "10" : "12" );

  addSetting( "Application:ApplicationWindowX", "10");
  addSetting( "Application:ApplicationWindowY", "10");
  addSetting( "Application:ApplicationWindowWidth", "640");
  addSetting( "Application:ApplicationWindowHeight", "480");
  addSetting( "Application:CamDialogWindowX", "10");
  addSetting( "Application:CamDialogWindowY", "10");
  addSetting( "Application:CamDialogWindowWidth", "655");
  addSetting( "Application:CamDialogWindowHeight", "430");
  addSetting( "Application:FontDialogWindowX", "10");
  addSetting( "Application:FontDialogWindowY", "10");
  addSetting( "Application:FontDialogWindowWidth", "500");
  addSetting( "Application:FontDialogWindowHeight", "420");

  addSetting( "CAD:PreviewElements", "100");
  addSetting( "CAD:SnapRange", "32");
  addSetting( "CAD:MarkSize", "4");
  addSetting( "CAD:GridWidth", "0.01");
  addSetting( "CAD:MetaGrid", "1");
  addSetting( "CAD:DimensionTextHeight", "5");
  addSetting( "CAD:DimensionTextAutoScale", "0");
  addSetting( "CAD:DimensionTextDistance", "2");
  addSetting( "CAD:DimensionOverLength", "2" );
  addSetting( "CAD:DimensionArrowLength", "3" );
  addSetting( "CAD:PrintFactor", "5" );

  addSetting( "Simulation:EndpointSize", "2");
  addSetting( "Simulation:ArrowSize", "3");
  addSetting( "Simulation:BrokenContourTolerance", "0.002");
  addSetting( "Simulation:DelayFactor", "1214");

  addSetting( "DXF:DxfSplineSegments", "16");
  addSetting( "DXF:DxfEllipseSegmentAngle", "5");

  addSetting( "CAM:MachineGenerator", "iso");
  addSetting( "CAM:SupportSubroutines", "0");

  addSetting( "Print:PrintBorderLeft", "25");
  addSetting( "Print:PrintBorderTop", "15");
  addSetting( "Print:PrintBorderRight", "15");
  addSetting( "Print:PrintBorderBottom", "15");
  addSetting( "Print:PrintZeroSize", "5");
  addSetting( "Print:PrintEnhancement", "5");

  addSetting( "Paths:BrowserPath", "netscape");

  verbose=false;
}




/**
 * Creates not existing config files.
 *
 * \param  overwrite Overwrite contents.
 * \return Success.
 */
bool
RConfig::createConfigFile( bool overwrite )
{
  bool ret=true;

  // Config path under win is program path + /.manstyle.
  // Under Linux it's the home path + /.manstyle.
  //
#ifdef DEF_WINDOWS
  setConfigDir(QDir(prgDir.absPath()));
#else
  setConfigDir(QDir(QDir::home().absPath() + "/." + DEF_APPNAME_L));
#endif

  // Create config directory:
  //
  if(!configDir.exists()) {
    if(!configDir.mkdir(configDir.absPath(), true)) {
      // Can't create directory for config file!
      return false;
    }
  }

  // Create ".<app>.conf":
  //
  QFile configFile(configDir.absPath() + "/" + DEF_APPNAME_L + ".conf");
  if(!configFile.exists() || overwrite) {
    if(configFile.open(IO_WriteOnly)) {    // file opened successfully
      QTextStream ts(&configFile);
      RSetting* setting;

      ts <<
      "# This file is automatically generated by " << DEF_APPNAME << ".\n"
      "# Please edit only if " << DEF_APPNAME << " is not running.\n";

      int sep;
      QString section, lastSection="";
      QString variable;

      // Write the sections:
      //
      for(setting=settingList.first(); setting!=0; setting=settingList.next()) {
        sep = setting->getName().find( ':' );
        section = setting->getName().left( sep );
        variable = setting->getName().right( setting->getName().length()-sep-1 );

        if( section!=lastSection ) ts << "\n[" << section << "]\n";

        ts << variable << "=\"" << setting->getValue() << "\"\n";

        lastSection = section.data();
      }

      configFile.close();
    }

    else {
      // Can't create file
      ret=false;
    }
  }

  return ret;
}



/**
 * Reads the config file.
 */
bool
RConfig::readConfigFile()
{
  QString configPath;
  configPath = configDir.absPath() + "/" + DEF_APPNAME_L + ".conf";
  int i=0, j=0, l=0;                  // Index, length of matching string
  QRegExp regSection("\\[[^]]*\\]");  // Reg exp for a section including brackets
  QRegExp regName("[^=[]*");          // Reg exp for a setting name (lvalue)
  QRegExp regValue("\"[^\"]*\"");     // Reg exp for a setting value (rvalue) including quote marks
  QString lSectionName;                // Section name excluding brackets
  QString setName;                    // Setting name
  QString setValue;                   // Setting value

  // Get file contents without comments:
  //
  QString cont = RFileParser::getContents(configPath, false);

  do {
    // Read next section (name/contents):
    //
    i=regSection.match(cont, i, &l);
    if(i==-1) break;
    lSectionName = cont.mid(i+1, l-2);
    i+=l;

    // Read next setting:
    //
    do {
      j=regName.match(cont, i, &l);
      if(j==-1) break;
      setName = cont.mid(j, l);
      if(setName.stripWhiteSpace().isEmpty()) break;
      i=j+l;

      j=regValue.match(cont, i, &l);
      if(j==-1) break;
      setValue = cont.mid(j+1, l-2);
      i=j+l;

      addSetting(lSectionName + ":" + setName.stripWhiteSpace(), setValue);

    } while(true);

  } while(true);


  // Add some values which were not saved in config file:
  //

  // Calculate factor for Speed-Slider (Simulation delay):
  //
  if( getSettingInt("Simulation:DelayFactor")==0 ) {
    QTime startTime = QTime::currentTime();
    for(int t=0; t<30000; ++t) {
      qApp->processEvents(1);
    }
    QTime stopTime = QTime::currentTime();

    uint iStartTime = startTime.hour()*3600000 +
                      startTime.minute()*60000 +
                      startTime.second()* 1000 +
                      startTime.msec();
    uint iStopTime =  stopTime.hour()*3600000 +
                      stopTime.minute()*60000 +
                      stopTime.second()* 1000 +
                      stopTime.msec();

    // Day change:
    //
    if( iStopTime<iStartTime ) iStopTime+=86400000;

    uint mil = iStopTime-iStartTime;
    int delayFactor;
    if(mil>0) delayFactor = (int)((1.0/(float)mil)*810000.0);
    else      delayFactor = 1000;

    addSetting( "Simulation:DelayFactor", delayFactor );

    RLOG("\nTime to count from 0 to 30000: ");
    RLOG((int)mil);
    RLOG("\ndelayFactor: ");
    RLOG(delayFactor);
  }


  // Ini some public variables provided for faster access:
  //
  cadDimensionTextHeight    = getSettingDouble( "CAD:DimensionTextHeight" );
  cadDimensionTextAutoScale = (bool)getSettingInt( "CAD:DimensionTextAutoScale" );
  cadDimensionTextDistance  = getSettingDouble( "CAD:DimensionTextDistance" );
  cadDimensionOverLength    = getSettingDouble( "CAD:DimensionOverLength" );
  cadDimensionArrowLength   = getSettingDouble( "CAD:DimensionArrowLength" );

  return false;
}



// Creates a list of all available languages.
//   The list is stored in "languageList"
//
void
RConfig::createLanguageList()
{
  QString messagePath = fileSearchSubdir( "messages" );

  QDir messageDir( messagePath, "*.cxl" );
  QStringList messageList = messageDir.entryList();
  QStringList::ConstIterator messageIterator;
  QFileInfo messageFileInfo;

  languageList.clear();

  if( messageList.count()>0 ) {
    for( messageIterator=messageList.begin(); messageIterator!=messageList.end(); ++messageIterator ) {
      messageFileInfo.setFile( messagePath+"/"+(*messageIterator) );

      languageStruct* newLanguage = new languageStruct;
      newLanguage->sign = messageFileInfo.baseName();
      newLanguage->name = "?";

      // Find out name of this language:
      //
      QFile f( messageFileInfo.absFilePath() );
      QString s="";                   // Buffer for the first line of the cxl file

      if(f.open(IO_ReadOnly)) {       // file opened successfully
        QTextStream t(&f);            // use a text stream

        s = t.readLine();             // The first line in a string
        f.close();

        newLanguage->name = s.mid( 2, s.find(' ', 2)-2 );
      }

      languageList.append( newLanguage );

      RLOG( "\nLanguage: " );
      RLOG( newLanguage->name );
      RLOG( " / " );
      RLOG( newLanguage->sign );
    }
  }
}



// Read messages from CXL-File:
//
bool
RConfig::readCxlFile()
{
  // Search for language file:
  //
  QString cxlPath;
  cxlPath = searchFile( QString("messages/")+getSetting("Language:Language")+".cxl" );

  if( cxlPath.isEmpty() ) {
    addSetting("Language:Language", "en");
    cxlPath = searchFile( QString("messages/")+getSetting("Language:Language")+".cxl" );
    if( cxlPath.isEmpty() ) return false;
  }

  QFile cxlFile( cxlPath );

  int mesNumber;

  // Read entries from cxl-File:
  //
  if(cxlFile.open(IO_ReadOnly)) {
    QTextStream cxlStream(&cxlFile);
    QString cxlLine;        // a line

    while(!cxlStream.eof()) {
      cxlLine=cxlStream.readLine();
      cxlLine=cxlLine.simplifyWhiteSpace();

      if(!cxlLine.isEmpty() && cxlLine.at(0)!='/' && cxlLine.at(0)!='#') {
        // Parse line which must look like this: 'e135 Text':
        //
        sscanf(cxlLine.latin1(), "%d", &mesNumber);

        if(mesNumber>=0 && mesNumber<=DEF_MESSAGES) {
          mes[mesNumber] = cxlLine.mid(4, 256);
          mes[mesNumber].replace(QRegExp("~"), "\n");
        }
      }
    }
    cxlFile.close();

    /* // Output messages to log file
    char t[16];
    for(int i=0; i<mesNumber; ++i) {
      RLOG("\n");
      sprintf(t, "%03d ", i);
      RLOG( t );
      RLOG( mes[i].replace(QRegExp("\n"), "~").latin1() );
    }
    */

    return true;
  }
  else {
    return false;
  }
}



// Get a message b its number:
//
const char*
RConfig::getMessage(int num)
const
{
  if(num>=0 && num<DEF_MESSAGES) {
    return mes[num].data();
  }
  return "";
}


/**
 * Searchs for a file with relative constant path given
 *   and returns the absolute path of the file.
 */
QString
RConfig::searchFile(const char* _relPath)
const
{
  QString path;

  // Search in program dir (/usr/local/qcad/...):
  path = getPrgDir().path()+"/"+_relPath;

#ifndef DEF_WINDOWS
  // Search in ~/.<proj>/...:
  if(!QFileInfo(path).exists()) {
    path = getConfigDir().path()+"/"+_relPath;
  }

  // Search in /usr/share/<proj>/... dir:
  if(!QFileInfo(path).exists()) {
    path = QString("/usr/share/")+DEF_APPNAME_L+"/"+_relPath;
  }
#endif

  if(!QFileInfo(path).exists()) {
    path="";
  }

  return path;
}


/*! Gets the short two letter sign for a given language.
*/
QString
RConfig::getLanguageSign( QString _languageName )
{
  languageStruct* ls;
  for( ls=languageList.first(); ls!=0; ls=languageList.next() ) {
    if( ls->name.lower()==_languageName.lower() ) {
      return ls->sign;
    }
  }

  /*
  if( _languageName.lower()=="english"  ||
      _languageName.lower()=="englisch" ||
      _languageName.lower()=="anglais"     ) {
    return QString("en");
  }

  else if( _languageName.lower()=="german"  ||
           _languageName.lower()=="deutsch" ||
           _languageName.lower()=="allemand"   ) {
    return QString("de");
  }

  else if( _languageName.lower()=="french"       ||
           _languageName.lower()=="franzsisch"  ||
           _languageName.lower()=="franzoesisch" ||
           _languageName.lower()=="franais"     ||
           _languageName.lower()=="francais"        ) {
    return QString("fr");
  }

  else if( _languageName.lower()=="spanish"  ||
           _languageName.lower()=="spanisch" ||
           _languageName.lower()=="espagnol"    ) {
    return QString("es");
  }

  else if( _languageName.lower()=="italian"  ||
           _languageName.lower()=="italiano" ||
           _languageName.lower()=="italien"    ) {
    return QString("it");
  }
  */

  return QString("en");

}



/*! Gets the full name of a language for a given two letter sign.
*/
QString
RConfig::getLanguageName( QString _languageSign )
{
  languageStruct* ls;
  for( ls=languageList.first(); ls!=0; ls=languageList.next() ) {
    if( ls->sign.lower()==_languageSign.lower() ) {
      return ls->name;
    }
  }

  /*
  if( _languageSign=="en" ) {
    return QString("English");
  }

  else if( _languageSign=="de" ) {
    return QString("German");
  }

  else if( _languageSign=="fr" ) {
    return QString("French");
  }

  else if( _languageSign=="es" ) {
    return QString("Spanish");
  }

  else if( _languageSign=="it" ) {
    return QString("Italian");
  }
  */

  return QString("English");
}



// Gets the number of languages available in languageList.
//
int
RConfig::getLanguageNum()
{
  return languageList.count();
}


QString
RConfig::getLanguage(int _index)
{
  if(_index<(int)languageList.count()) {
    return languageList.at(_index)->name;
  }
  return QString("English");
}


// EOF
















