/*-
# X-BASED ABACUS
#
#  Abacus.c
#
###
#
#  Copyright (c) 1994 - 96	David Albert Bagley, bagleyd@bigfoot.com
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Abacus */

#include <stdlib.h>
#include <stdio.h>
#ifdef VMS
#include <unixlib.h>
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "AbacusP.h"

static void InitializeAbacus(Widget request, Widget new);
static void ExposeAbacus(Widget new, XEvent * event, Region region);
static void ResizeAbacus(AbacusWidget w);
static void DestroyAbacus(Widget old);
static Boolean SetValuesAbacus(Widget current, Widget request, Widget new);

static void QuitAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void SelectAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void ReleaseAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void ClearAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs);

static int  PositionToBead(AbacusWidget w, int x, int y, int *d, int *i, int *j);
static void CheckBeads(AbacusWidget w);
static void ResetBeads(AbacusWidget w);
static void ClearAllBeads(AbacusWidget w);
static void MoveBeads(AbacusWidget w, int d, int i, int j);
static void MoveBeadsUp(AbacusWidget w, int d, int i, int j);
static void MoveBeadsDown(AbacusWidget w, int d, int i, int j);
static void DrawFrame(AbacusWidget w, GC gc);
static void DrawAllBeads(AbacusWidget w, GC beadGC, GC borderGC);
static void DrawBead(AbacusWidget w, GC beadGC, GC borderGC, int d, int i, int j);
static void AddBead(AbacusWidget w, int d, int p);
static void SubBead(AbacusWidget w, int d, int p);
static int  EmptyCounter(AbacusWidget w);
static void SetCounter(AbacusWidget w);

static char defaultTranslationsAbacus[] =
"<KeyPress>q: quit()\n\
   Ctrl<KeyPress>C: quit()\n\
   <Btn1Down>: select()\n\
   <Btn1Up>: release()\n\
   <KeyPress>c: clear()\n\
   <Btn3Down>: clear()";

/* <Btn1Motion>: select()\n\ */

static XtActionsRec actionsListAbacus[] =
{
	{"quit", (XtActionProc) QuitAbacus},
	{"select", (XtActionProc) SelectAbacus},
	{"release", (XtActionProc) ReleaseAbacus},
	{"clear", (XtActionProc) ClearAbacus}
};

static XtResource resourcesAbacus[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(AbacusWidget, core.width), XtRString, "400"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(AbacusWidget, core.height), XtRString, "200"},
	{XtNbars, XtCBars, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.bars), XtRString, "12"},
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.base), XtRString, "10"},
	{XtNspaces, XtCSpaces, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.spaces), XtRString, "2"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
  XtOffset(AbacusWidget, abacus.foreground), XtRString, XtDefaultForeground},
	{XtNbeadColor, XtCForeground, XtRPixel, sizeof (Pixel),
   XtOffset(AbacusWidget, abacus.beadColor), XtRString, XtDefaultForeground},
	{XtNbeadBorder, XtCForeground, XtRPixel, sizeof (Pixel),
 XtOffset(AbacusWidget, abacus.borderColor), XtRString, XtDefaultForeground},
	{XtNtopNumber, XtCTopNumber, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.levels[TOP].number), XtRString, "2"},
	{XtNbottomNumber, XtCBottomNumber, XtRInt, sizeof (int),
       XtOffset(AbacusWidget, abacus.levels[BOTTOM].number), XtRString, "5"},
	{XtNtopFactor, XtCTopFactor, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.levels[TOP].factor), XtRString, "5"},
	{XtNbottomFactor, XtCBottomFactor, XtRInt, sizeof (int),
       XtOffset(AbacusWidget, abacus.levels[BOTTOM].factor), XtRString, "1"},
	{XtNtopOrient, XtCTopOrient, XtRBoolean, sizeof (Boolean),
 XtOffset(AbacusWidget, abacus.levels[TOP].orientation), XtRString, "FALSE"},
	{XtNbottomOrient, XtCBottomOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(AbacusWidget, abacus.levels[BOTTOM].orientation), XtRString, "FALSE"},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.delay), XtRString, "100"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(AbacusWidget, abacus.select), XtRCallback, NULL}
};

AbacusClassRec abacusClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		"Abacus",	/* class name */
		sizeof (AbacusRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeAbacus,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListAbacus,	/* actions */
		XtNumber(actionsListAbacus),	/* num actions */
		resourcesAbacus,	/* resources */
		XtNumber(resourcesAbacus),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyAbacus,	/* destroy */
		(XtWidgetProc) ResizeAbacus,	/* resize */
		(XtExposeProc) ExposeAbacus,	/* expose */
		(XtSetValuesFunc) SetValuesAbacus,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsAbacus,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass abacusWidgetClass = (WidgetClass) & abacusClassRec;

#ifndef HAVE_USLEEP
#if !defined( VMS ) || defined( XVMSUTILS ) ||  ( __VMS_VER >= 70000000 )
#ifdef USE_XVMSUTILS
#include <X11/unix_time.h>
#endif
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#endif
#if defined(SYSV) || defined(SVR4)
#ifdef LESS_THAN_AIX3_2
#include <sys/poll.h>
#else /* !LESS_THAN_AIX3_2 */
#include <poll.h>
#endif /* !LESS_THAN_AIX3_2 */
#endif /* defined(SYSV) || defined(SVR4) */

static int
usleep(long unsigned int usec)
{
#if (defined (SYSV) || defined(SVR4)) && !defined(__hpux)
#if defined(HAVE_NANOSLEEP)
	{
		struct timespec rqt;

		rqt.tv_nsec = 1000 * (usec % (unsigned long) 1000000);
		rqt.tv_sec = usec / (unsigned long) 1000000;
		return nanosleep(&rqt, NULL);
	}
#else
	(void) poll((void *) 0, (int) 0, usec / 1000);	/* ms resolution */
#endif
#else
#ifdef VMS
	long        timadr[2];

	if (usec != 0) {
		timadr[0] = -usec * 10;
		timadr[1] = -1;

		sys$setimr(4, &timadr, 0, 0, 0);
		sys$waitfr(4);
	}
#else
	struct timeval time_out;

#if  0				/* (!defined(AIXV3) && !defined(__hpux)) */
	extern int  select(int, fd_set *, fd_set *, fd_set *, struct timeval *);

#endif

	time_out.tv_usec = usec % (unsigned long) 1000000;
	time_out.tv_sec = usec / (unsigned long) 1000000;
	(void) select(0, (void *) 0, (void *) 0, (void *) 0, &time_out);
#endif
#endif
	return 0;
}
#endif

static void
Sleep(long unsigned int cMilliseconds)
{
	(void) usleep(cMilliseconds * 1000);
}

static void
InitializeAbacus(Widget request, Widget new)
{
	AbacusWidget w = (AbacusWidget) new;
	XGCValues   values;
	XtGCMask    valueMask;
	int         level;

	for (level = 0; level < MAXLEVELS; level++)
		w->abacus.levels[level].position = NULL;
	w->abacus.digits = NULL;
	CheckBeads(w);
	ResetBeads(w);

	valueMask = GCForeground | GCBackground;
	values.background = w->core.background_pixel;

	values.foreground = w->abacus.foreground;
	w->abacus.abacusGC = XtGetGC(new, valueMask, &values);

	values.foreground = w->abacus.beadColor;
	w->abacus.beadGC = XtGetGC(new, valueMask, &values);

	values.foreground = w->abacus.borderColor;
	w->abacus.borderGC = XtGetGC(new, valueMask, &values);

	values.foreground = w->core.background_pixel;
	w->abacus.eraseGC = XtGetGC(new, valueMask, &values);

	ResizeAbacus(w);
}

static void
DestroyAbacus(Widget old)
{
	AbacusWidget w = (AbacusWidget) old;

	XtReleaseGC(old, w->abacus.beadGC);
	XtReleaseGC(old, w->abacus.borderGC);
	XtReleaseGC(old, w->abacus.abacusGC);
	XtReleaseGC(old, w->abacus.eraseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->abacus.select);
}

static void
ResizeAbacus(AbacusWidget w)
{
	int         height;

	w->abacus.delta.x = 8;
	w->abacus.delta.y = 2;
	w->abacus.pos.x = MAX(((int) w->core.width - w->abacus.delta.x) /
			      w->abacus.bars, w->abacus.delta.x);
	w->abacus.pos.y = MAX(((int) w->core.height - 2 * w->abacus.delta.y - 2) /
		(w->abacus.levels[TOP].room + w->abacus.levels[BOTTOM].room),
			      w->abacus.delta.y);
	w->abacus.width = w->abacus.pos.x * w->abacus.bars +
		w->abacus.delta.x + 2;
	w->abacus.levels[TOP].height = w->abacus.pos.y *
		w->abacus.levels[TOP].room + w->abacus.delta.y + 2;
	w->abacus.levels[BOTTOM].height = w->abacus.pos.y *
		w->abacus.levels[BOTTOM].room + w->abacus.delta.y + 2;
	height = w->abacus.levels[TOP].height + w->abacus.levels[BOTTOM].height;
	w->abacus.offset.x = ((int) w->core.width - w->abacus.width + 2) / 2;
	w->abacus.offset.y = ((int) w->core.height - height + 2) / 2;
	w->abacus.beadSize.x = w->abacus.pos.x - w->abacus.delta.x;
	w->abacus.beadSize.y = w->abacus.pos.y - w->abacus.delta.y;
}

static void
ExposeAbacus(Widget new, XEvent * event, Region region)
		 /* Not used */
{
	AbacusWidget w = (AbacusWidget) new;

	if (w->core.visible) {
		DrawFrame(w, w->abacus.abacusGC);
		DrawAllBeads(w, w->abacus.beadGC, w->abacus.borderGC);
	}
}

static      Boolean
SetValuesAbacus(Widget current, Widget request, Widget new)
{
	AbacusWidget c = (AbacusWidget) current, w = (AbacusWidget) new;
	XGCValues   values;
	XtGCMask    valueMask;
	Boolean     redraw = FALSE;
	Boolean     redrawBeads = FALSE;

	CheckBeads(w);
	if (w->abacus.beadColor != c->abacus.beadColor ||
	    w->abacus.borderColor != c->abacus.borderColor ||
	    w->core.background_pixel != c->core.background_pixel) {
		valueMask = GCForeground | GCBackground;
		values.background = w->core.background_pixel;

		values.foreground = w->abacus.beadColor;
		XtReleaseGC(new, w->abacus.beadGC);
		w->abacus.beadGC = XtGetGC(new, valueMask, &values);
		values.foreground = w->abacus.borderColor;
		XtReleaseGC(new, w->abacus.borderGC);
		w->abacus.borderGC = XtGetGC(new, valueMask, &values);

		values.background = w->abacus.beadColor;
		values.foreground = w->core.background_pixel;
		XtReleaseGC(new, w->abacus.eraseGC);
		w->abacus.eraseGC = XtGetGC(new, valueMask, &values);
		redrawBeads = TRUE;
	}
	if (w->abacus.bars != c->abacus.bars ||
	    w->abacus.levels[BOTTOM].orientation !=
	    c->abacus.levels[BOTTOM].orientation ||
	    w->abacus.levels[TOP].orientation !=
	    c->abacus.levels[TOP].orientation ||
	    w->abacus.levels[BOTTOM].number !=
	    c->abacus.levels[BOTTOM].number ||
	    w->abacus.levels[TOP].number != c->abacus.levels[TOP].number ||
	    w->abacus.levels[BOTTOM].factor !=
	    c->abacus.levels[BOTTOM].factor ||
	    w->abacus.levels[TOP].factor != c->abacus.levels[TOP].factor ||
	    w->abacus.spaces != c->abacus.spaces ||
	    w->abacus.base != c->abacus.base) {
		ResetBeads(w);
		redraw = TRUE;
	}
	if (w->abacus.pos.x != c->abacus.pos.x || w->abacus.pos.y != c->abacus.pos.y)
		redrawBeads = TRUE;
	if (redrawBeads && !redraw && XtIsRealized(new) && new->core.visible) {
		DrawFrame(c, c->abacus.eraseGC);
		DrawAllBeads(c, c->abacus.eraseGC, c->abacus.eraseGC);
		DrawFrame(w, w->abacus.abacusGC);
		DrawAllBeads(w, w->abacus.beadGC, w->abacus.borderGC);
	}
	return (redraw);
}

static void
QuitAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}

static void
SelectAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs)
{
	int         d, i, j;

	if (PositionToBead(w, event->xbutton.x, event->xbutton.y, &d, &i, &j)) {
		w->abacus.currentLevel = d;
		w->abacus.currentBar = i;
		w->abacus.currentPosition = j;
		DrawBead(w, w->abacus.borderGC, w->abacus.beadGC, d, i, j);
	} else
		w->abacus.currentLevel = -1;
}

static void
ReleaseAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs)
{
	if (w->abacus.currentLevel < 0)
		return;
	DrawBead(w, w->abacus.eraseGC, w->abacus.eraseGC, w->abacus.currentLevel,
		 w->abacus.currentBar, w->abacus.currentPosition);
	DrawBead(w, w->abacus.beadGC, w->abacus.borderGC, w->abacus.currentLevel,
		 w->abacus.currentBar, w->abacus.currentPosition);
	MoveBeads(w, w->abacus.currentLevel, w->abacus.currentBar,
		  w->abacus.currentPosition);
	w->abacus.currentLevel = -1;
}
static void
ClearAbacus(AbacusWidget w, XEvent * event, char **args, int nArgs)
{
	if (EmptyCounter(w))
		return;
	/* Should check if one really wants to destroy calculations. */
	ClearAllBeads(w);
}

static int
PositionToBead(AbacusWidget w, int x, int y, int *d, int *i, int *j)
{
	x -= w->abacus.offset.x;
	y -= w->abacus.offset.y;
	if (y > w->abacus.levels[TOP].height) {
		y = y - w->abacus.levels[TOP].height;
		*d = BOTTOM;
	} else
		*d = TOP;
	*i = w->abacus.bars - 1 - (x - w->abacus.delta.x / 2) / w->abacus.pos.x;
	*j = (y - w->abacus.delta.y / 2) / w->abacus.pos.y + 1;
	if (*i < 0)
		*i = 0;
	else if (*i >= w->abacus.bars)
		*i = w->abacus.bars - 1;
	if (*j < 1)
		*j = 1;
	else if (*j > w->abacus.levels[*d].room)
		*j = w->abacus.levels[*d].room;
	return ((*j > w->abacus.levels[*d].position[*i] + w->abacus.spaces) ||
		(*j <= w->abacus.levels[*d].position[*i]));
}

static void
CheckBeads(AbacusWidget w)
{
	char        buf[121];

	if (w->abacus.base != 10) {
		XtWarning("Base in this version must be 10");
		w->abacus.base = 10;
	}
	if (w->abacus.bars < 1) {
		(void) sprintf(buf, "Number of Bars out of bounds, use 1..MAXINT");
		XtWarning(buf);
		w->abacus.bars = 12;
	}
	if (w->abacus.levels[TOP].number < 1 ||
	    w->abacus.levels[TOP].number > w->abacus.base) {
		(void) sprintf(buf,
			       "Number of Top Beads out of bounds, use 1..%d", w->abacus.base);
		XtWarning(buf);
		w->abacus.levels[TOP].number = 2;
	}
	if (w->abacus.levels[BOTTOM].number < 1 ||
	    w->abacus.levels[BOTTOM].number > w->abacus.base) {
		(void) sprintf(buf,
			       "Number of Bottom Beads out of bounds, use 1..%d", w->abacus.base);
		XtWarning(buf);
		w->abacus.levels[BOTTOM].number = 5;
	}
	if (w->abacus.spaces < 1) {
		(void) sprintf(buf, "Number of Spaces out of bounds, use 1..MAXINT");
		XtWarning(buf);
		w->abacus.spaces = 2;
	}
	if (w->abacus.levels[TOP].factor < 1 ||
	    w->abacus.levels[TOP].factor > w->abacus.base) {
		(void) sprintf(buf, "Factor of Top Beads out of bounds, use 1..%d",
			       w->abacus.base);
		XtWarning(buf);
		w->abacus.levels[TOP].factor = 5;
	}
	if (w->abacus.levels[BOTTOM].factor < 1 ||
	    w->abacus.levels[BOTTOM].factor > w->abacus.base) {
		(void) sprintf(buf, "Factor of Bottom Beads out of bounds, use 1..%d",
			       w->abacus.base);
		XtWarning(buf);
		w->abacus.levels[BOTTOM].factor = 1;
	}
	if (w->abacus.delay < 0) {
		(void) sprintf(buf, "Delay out of bounds, use 0..MAXINT");
		XtWarning(buf);
		w->abacus.delay = -w->abacus.delay;
	}
}

static void
ResetBeads(AbacusWidget w)
{
	int         i, d;

	w->abacus.currentLevel = -1;
	w->abacus.numDigits = w->abacus.bars + CARRY;
	for (d = BOTTOM; d <= TOP; d++) {
		if (w->abacus.levels[d].position)
			(void) free((void *) w->abacus.levels[d].position);
		if (!(w->abacus.levels[d].position = (int *)
		      malloc(sizeof (int) * w->abacus.bars)))
			            XtError("Not enough memory, exiting.");
	}
	if (w->abacus.digits)
		(void) free((void *) w->abacus.digits);
	if (!(w->abacus.digits = (char *)
	      malloc(sizeof (char) * w->abacus.numDigits)))
		            XtError("Not enough memory, exiting.");

	w->abacus.levels[TOP].room =
		w->abacus.levels[TOP].number + w->abacus.spaces;
	w->abacus.levels[BOTTOM].room =
		w->abacus.levels[BOTTOM].number + w->abacus.spaces;
	for (i = 0; i < w->abacus.bars; i++)
		for (d = BOTTOM; d <= TOP; d++)
			w->abacus.levels[d].position[i] =
				(w->abacus.levels[d].orientation) ? w->abacus.levels[d].number : 0;
	for (i = 0; i < w->abacus.numDigits - 1; i++)
		w->abacus.digits[i] = '0';
	w->abacus.digits[w->abacus.numDigits - 1] = '\0';
}

static void
ClearAllBeads(AbacusWidget w)
{
	int         d, i;

	for (i = 0; i < w->abacus.bars; i++) {
		for (d = DOWN; d <= UP; d++) {
			if (w->abacus.levels[d].orientation)
				MoveBeadsUp(w, d, i, w->abacus.levels[d].room);
			else	/* w->abacus.levels[d].orientation == DOWN */
				MoveBeadsDown(w, d, i, 1);
		}
	}
}

static void
MoveBeads(AbacusWidget w, int d, int i, int j)
{
	if (j <= w->abacus.levels[d].position[i])
		MoveBeadsDown(w, d, i, j);
	else
		MoveBeadsUp(w, d, i, j);
}

static void
MoveBeadsUp(AbacusWidget w, int d, int i, int j)
{
	int         k, l;

	if (j > w->abacus.levels[d].position[i] + w->abacus.spaces) {
		for (l = 0; l < w->abacus.spaces; l++) {
			for (k = w->abacus.levels[d].position[i] + w->abacus.spaces + 1;
			     k <= j; k++) {
				DrawBead(w, w->abacus.eraseGC, w->abacus.eraseGC, d, i, k - l);
				DrawBead(w, w->abacus.beadGC, w->abacus.borderGC,
					 d, i, k - l - 1);
			}
			if (l + 1 != w->abacus.spaces) {
				XFlush(XtDisplay(w));
				Sleep((unsigned long) w->abacus.delay);
			}
		}
		if (w->abacus.levels[d].orientation)
			SubBead(w, w->abacus.levels[d].factor *
				(j - w->abacus.spaces - w->abacus.levels[d].position[i]), i);
		else		/* w->abacus.levels[d].orientation == DOWN */
			AddBead(w, w->abacus.levels[d].factor *
				(j - w->abacus.spaces - w->abacus.levels[d].position[i]), i);
		SetCounter(w);
		w->abacus.levels[d].position[i] = j - w->abacus.spaces;
	}
}

static void
MoveBeadsDown(AbacusWidget w, int d, int i, int j)
{
	int         k, l;

	if (j <= w->abacus.levels[d].position[i]) {
		for (l = 0; l < w->abacus.spaces; l++) {
			for (k = w->abacus.levels[d].position[i]; k >= j; k--) {
				DrawBead(w, w->abacus.eraseGC, w->abacus.eraseGC, d, i, k + l);
				DrawBead(w, w->abacus.beadGC, w->abacus.borderGC,
					 d, i, k + l + 1);
			}
			if (l + 1 != w->abacus.spaces) {
				XFlush(XtDisplay(w));
				Sleep((unsigned int) w->abacus.delay);
			}
		}
		if (w->abacus.levels[d].orientation)
			AddBead(w, w->abacus.levels[d].factor *
				(w->abacus.levels[d].position[i] - j + 1), i);
		else		/* w->abacus.levels[d].orientation == DOWN */
			SubBead(w, w->abacus.levels[d].factor *
				(w->abacus.levels[d].position[i] - j + 1), i);
		SetCounter(w);
		w->abacus.levels[d].position[i] = j - 1;
	}
}

static void
DrawFrame(AbacusWidget w, GC gc)
{
	int         d, i, dx, dy, x, y, yOffset;

	x = w->abacus.bars * w->abacus.pos.x + w->abacus.delta.x - 1;
	for (d = UP; d >= DOWN; d--) {
		dx = w->abacus.beadSize.x / 2 + w->abacus.delta.x + w->abacus.offset.x;
		yOffset = (d == UP) ? 0 : w->abacus.levels[TOP].height;
		y = w->abacus.levels[d].room * w->abacus.pos.y + w->abacus.delta.y - 1;
		dy = w->abacus.delta.y / 2 + yOffset + w->abacus.offset.y;
		XFillRectangle(XtDisplay(w), XtWindow(w), gc,
		     w->abacus.offset.x, yOffset + w->abacus.offset.y, x, 1);
		XFillRectangle(XtDisplay(w), XtWindow(w), gc,
		 w->abacus.offset.x, y + yOffset + w->abacus.offset.y, x, 1);
		XFillRectangle(XtDisplay(w), XtWindow(w), gc,
		     w->abacus.offset.x, yOffset + w->abacus.offset.y, 1, y);
		XFillRectangle(XtDisplay(w), XtWindow(w), gc,
			       x + w->abacus.offset.x, yOffset + w->abacus.offset.y, 1, y + 1);
		for (i = 0; i < w->abacus.bars; i++) {
			XFillRectangle(XtDisplay(w), XtWindow(w), gc, dx, dy, 1, y - 1);
			dx += w->abacus.pos.x;
		}
	}
}

static void
DrawAllBeads(AbacusWidget w, GC beadGC, GC borderGC)
{
	int         d, i, j;

	for (i = 0; i < w->abacus.bars; i++) {
		for (d = DOWN; d <= UP; d++) {
			for (j = 1; j <= w->abacus.levels[d].position[i]; j++)
				DrawBead(w, beadGC, borderGC, d, i, j);
			for (j = w->abacus.spaces + w->abacus.levels[d].position[i] + 1;
			     j <= w->abacus.levels[d].room; j++)
				DrawBead(w, beadGC, borderGC, d, i, j);
		}
	}
	SetCounter(w);
}

static void
XFillCircle(Display * display, Window window, GC gc, int diameter, int ctrX, int ctrY)
{
	if (diameter > 0)
		XFillArc(display, window, gc, ctrX - diameter / 2, ctrY - diameter / 2,
			 diameter, diameter, 0, CIRCLE);
}

static void
XDrawCircle(Display * display, Window window, GC gc, int diameter, int ctrX, int ctrY)
{
	if (diameter > 0)
		XDrawArc(display, window, gc, ctrX - diameter / 2, ctrY - diameter / 2,
			 diameter, diameter, 0, CIRCLE);
}

static void
DrawBead(AbacusWidget w, GC beadGC, GC borderGC, int d, int i, int j)
{
	int         dx, dy, yOffset;

	yOffset = (d == UP) ? 0 : w->abacus.levels[TOP].height;
	dx = (w->abacus.bars - i - 1) * w->abacus.pos.x + w->abacus.delta.x +
		w->abacus.offset.x;
	dy = (j - 1) * w->abacus.pos.y + w->abacus.delta.y + yOffset +
		w->abacus.offset.y;
	if (beadGC != w->abacus.eraseGC || borderGC != w->abacus.eraseGC) {
		if (w->abacus.beadSize.x > w->abacus.beadSize.y) {
			XDrawCircle(XtDisplay(w), XtWindow(w), borderGC, w->abacus.beadSize.y - 1,
				    dx + w->abacus.beadSize.x / 2 -
			   (w->abacus.beadSize.x - w->abacus.beadSize.y) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
			XDrawCircle(XtDisplay(w), XtWindow(w), borderGC, w->abacus.beadSize.y - 1,
				    dx + w->abacus.beadSize.x / 2 +
			   (w->abacus.beadSize.x - w->abacus.beadSize.y) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
			XDrawRectangle(XtDisplay(w), XtWindow(w), borderGC,
				       dx + w->abacus.beadSize.x / 2 -
			(w->abacus.beadSize.x - w->abacus.beadSize.y) / 2, dy,
				       w->abacus.beadSize.x - w->abacus.beadSize.y, w->abacus.beadSize.y);
			XFillCircle(XtDisplay(w), XtWindow(w), beadGC, w->abacus.beadSize.y - 1,
				    dx + w->abacus.beadSize.x / 2 -
			   (w->abacus.beadSize.x - w->abacus.beadSize.y) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
			XFillCircle(XtDisplay(w), XtWindow(w), beadGC, w->abacus.beadSize.y - 1,
				    dx + w->abacus.beadSize.x / 2 +
			   (w->abacus.beadSize.x - w->abacus.beadSize.y) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
			XFillRectangle(XtDisplay(w), XtWindow(w), beadGC,
				       dx + w->abacus.beadSize.x / 2 -
			(w->abacus.beadSize.x - w->abacus.beadSize.y) / 2, dy,
				       w->abacus.beadSize.x - w->abacus.beadSize.y, w->abacus.beadSize.y);
		} else if (w->abacus.beadSize.x < w->abacus.beadSize.y) {
			XDrawCircle(XtDisplay(w), XtWindow(w), borderGC, w->abacus.beadSize.x - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + w->abacus.beadSize.y / 2 -
			  (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2);
			XDrawCircle(XtDisplay(w), XtWindow(w), borderGC, w->abacus.beadSize.x - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + w->abacus.beadSize.y / 2 +
			  (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2);
			XDrawRectangle(XtDisplay(w), XtWindow(w), borderGC,
				       dx, dy + w->abacus.beadSize.y / 2 -
			   (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2,
				       w->abacus.beadSize.x, w->abacus.beadSize.y - w->abacus.beadSize.x);
			XFillCircle(XtDisplay(w), XtWindow(w), beadGC, w->abacus.beadSize.x - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + w->abacus.beadSize.y / 2 -
			  (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2);
			XFillCircle(XtDisplay(w), XtWindow(w), beadGC, w->abacus.beadSize.x - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + w->abacus.beadSize.y / 2 +
			  (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2);
			XFillRectangle(XtDisplay(w), XtWindow(w), beadGC,
				       dx, dy + w->abacus.beadSize.y / 2 -
			   (w->abacus.beadSize.y - w->abacus.beadSize.x) / 2,
				       w->abacus.beadSize.x, w->abacus.beadSize.y - w->abacus.beadSize.x);
		} else {
			XFillCircle(XtDisplay(w), XtWindow(w), beadGC, w->abacus.beadSize.y - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
			XDrawCircle(XtDisplay(w), XtWindow(w), borderGC, w->abacus.beadSize.y - 1,
				    dx + (w->abacus.beadSize.x - 1) / 2,
				    dy + (w->abacus.beadSize.y - 1) / 2);
		}
	} else {
		XFillRectangle(XtDisplay(w), XtWindow(w), beadGC,
		 dx, dy, w->abacus.beadSize.x + 1, w->abacus.beadSize.y + 1);
		XFillRectangle(XtDisplay(w), XtWindow(w), w->abacus.abacusGC,
			       dx + w->abacus.beadSize.x / 2, dy, 1, w->abacus.beadSize.y + 2);
	}
}

static void
AddBead(AbacusWidget w, int d, int p)
{
	int         position = w->abacus.numDigits - 2 - p;
	int         digit = w->abacus.digits[position] - '0' + d;


	w->abacus.digits[position] = (digit % w->abacus.base) + '0';
	if (digit >= w->abacus.base)
		AddBead(w, digit / w->abacus.base, p + 1);
}

static void
SubBead(AbacusWidget w, int d, int p)
{
	int         position = w->abacus.numDigits - 2 - p;
	int         digit = w->abacus.digits[position] - '0' - d;

	w->abacus.digits[position] =
		((digit + w->abacus.base) % w->abacus.base) + '0';
	if (digit < 0)
		SubBead(w, 1 + (-1 - digit) / w->abacus.base, p + 1);
}

static int
EmptyCounter(AbacusWidget w)
{
	int         n = 0;

	while (n < w->abacus.numDigits - 2 && w->abacus.digits[n] == '0')
		n++;
	return (n == w->abacus.numDigits - 2 && w->abacus.digits[n] == '0');
}

static void
SetCounter(AbacusWidget w)
{
	abacusCallbackStruct cb;
	int         n = 0;

	while (n < w->abacus.numDigits - 2 && w->abacus.digits[n] == '0')
		n++;
	if (!(cb.buffer = (char *)
	      malloc(sizeof (char) * (w->abacus.numDigits - n + 1))))
		            XtError("Not enough memory, exiting.");

	(void) strcpy(cb.buffer, &(w->abacus.digits[n]));
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	(void) free((void *) cb.buffer);
}

#ifdef OLDANDINTHEWAY
/* Not used since this restricts the size of abacus to long int. */
static long int
power(int x, int n)
{				/* raise x to the nth power n >= 0 */
	int         i;
	long int    p = 1;

	for (i = 1; i <= n; ++i)
		p = p * x;
	return p;
}

/* This routine will do a XFillArc for a circle */
DiskXI(Display * display, Window window, GC gc,
       int diameter, int ctrX, int ctrY)
{
	int         x, y, p, d;

	x = 0;
	y = diameter / 2;
	p = diameter % 2;
	d = 1 - 2 * y + p;
	while (x < y) {
		DiskPointsXI(display, window, gc, ctrX, ctrY, x, y, p);
		if (d < 0)
			d = d + (4 * x) + 6;
		else {
			d = d + (4 * (x - y)) + 10;
			y--;
		}
		x++;
	}
	if (x == y)
		DiskPointsXI(display, window, gc, ctrX, ctrY, x, y, p);
}				/* DiskXI */

DiskPointsXI(Display * display, Window window, GC gc,
	     int ctrX, int ctrY, int x, int y, int p)
{
	XFillRectangle(display, window, gc,
		       ctrX - x, ctrY - y, 2 * x + p + 1, 1);
	XFillRectangle(display, window, gc,
		       ctrX - x, ctrY + y + p, 2 * x + p + 1, 1);
	XFillRectangle(display, window, gc,
		       ctrX - y, ctrY - x, 2 * y + p + 1, 1);
	XFillRectangle(display, window, gc,
		       ctrX - y, ctrY + x + p, 2 * y + p + 1, 1);
	/*XDrawLine(display, window, gc, ctrX - x, ctrY - y, ctrX + x + p, ctrY - y);
	   XDrawLine(display, window, gc,
	   ctrX - x, ctrY + y + p, ctrX + x + p, ctrY + y + p);
	   XDrawLine(display, window, gc, ctrX - y, ctrY - x, ctrX + y + p, ctrY - x);
	   XDrawLine(display, window, gc,
	   ctrX - y, ctrY + x + p, ctrX + y + p, ctrY + x + p); */
}				/* DiskPointsXI */

#endif
