## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
from mfxutil import destruct, Struct, SubclassResponsibility        #bundle#
from pysoltk import MfxCanvasText                                   #bundle#


# /***********************************************************************
# // a helper class to create common layouts
# ************************************************************************/

# a layout stack
class _LayoutStack:
    def __init__(self, x, y, suit=None):
        self.x = int(round(x))
        self.y = int(round(y))
        self.suit = suit
        self.text_args = {}
        self.text_format = "%d"

    def setText(self, x, y, anchor="center", format=None, **kw):
        self.text_args["x"] = x
        self.text_args["y"] = y
        self.text_args["anchor"] = anchor
        self.text_args.update(kw)
        if format is not None:
            self.text_format = format


class Layout:
    def __init__(self, game, XM=10, YM=10, **kw):
        self.game = game
        self.canvas = self.game.canvas
        self.size = None
        self.s = Struct(
            talon = None,
            waste = None,
            foundations = [],
            rows = [],
            reserves = [],
        )
        self.stackmap = {}
        self.regions = []
        # set visual constants
        images = self.game.app.images
        self.CW = images.CARDW
        self.CH = images.CARDH
        self.XM = XM                    # XMARGIN
        self.YM = YM                    # YMARGIN
        self.XS = self.CW + XM          # XSPACE
        self.YS = self.CH + YM          # YSPACE
        self.XOFFSET = images.CARD_XOFFSET
        self.YOFFSET = images.CARD_YOFFSET
        self.__dict__.update(kw)

    def __createStack(self, x, y, suit=None):
        stack = _LayoutStack(x, y, suit)
        mapkey = (stack.x, stack.y)
        assert not self.stackmap.has_key(mapkey)
        self.stackmap[mapkey] = stack
        return stack

    #
    # public util for use by class Game
    #

    def getTextAttr(self, stack, anchor):
        x, y = 0, 0
        if stack is not None:
            x, y = stack.x, stack.y
        if anchor == "n":
            return (x + self.CW / 2, y - self.YM, "center", "%d")
        if anchor == "nn":
            return (x + self.CW / 2, y - self.YM, "s", "%d")
        if anchor == "s":
            return (x + self.CW / 2, y + self.YS, "center", "%d")
        if anchor == "ss":
            return (x + self.CW / 2, y + self.YS, "n", "%d")
        if anchor == "nw":
            return (x - self.XM, y, "ne", "%d")
        if anchor == "sw":
            return (x - self.XM, y + self.CH, "se", "%d")
        f = "%2d"
        if self.game.game_info.decks > 1:
            f = "%3d"
        if anchor == "ne":
            return (x + self.XS, y, "nw", f)
        if anchor == "se":
            return (x + self.XS, y + self.CH, "sw", f)
        raise Exception, anchor

    def createText(self, stack, anchor, dx=0, dy=0):
        assert stack.texts.ncards is None
        tx, ty, ta, tf = self.getTextAttr(stack, anchor)
        stack.texts.ncards = MfxCanvasText(self.canvas, tx+dx, ty+dy, anchor=ta)
        stack.texts.ncards.text_format = tf

    def setRegion(self, stacks, rects):
        self.regions.append((stacks, rects))


    #
    # util for use by a Game
    #

    def defaultAll(self):
        game = self.game
        # create texts
        if game.s.talon:
            game.s.talon.texts.ncards = self.defaultText(self.s.talon)
        if game.s.waste:
            game.s.waste.texts.ncards = self.defaultText(self.s.waste)
        # define stack-groups
        self.defaultStackGroups()
        # set regions
        self.defaultRegions()

    def defaultText(self, layout_stack):
        ##print layout_stack, layout_stack.text_args
        if layout_stack is None or not layout_stack.text_args:
            return None
        t = apply(MfxCanvasText, (self.game.canvas,), layout_stack.text_args)
        t.text_format = layout_stack.text_format
        return t

    # define stack-groups
    def defaultStackGroups(self):
        game = self.game
        waste = []
        if game.s.waste is not None: waste = [game.s.waste]
        game.sg.talonstacks = [game.s.talon] + waste
        game.sg.dropstacks = game.s.rows + game.s.reserves + waste
        game.sg.openstacks = game.s.foundations + game.s.rows + game.s.reserves
        game.sg.reservestacks = game.s.reserves

    # convert layout-stacks to corresponding stacks in game
    def defaultRegions(self):
        for region in self.regions:
            stacks = []
            for s in region[0]:
                mapkey = (s.x, s.y)
                id = self.game.stackmap[mapkey]
                stacks.append(self.game.allstacks[id])
            ##print stacks, region[1]
            self.game.setRegion(stacks, region[1])

    #
    # Baker's Dozen layout
    #  - left: 2 rows
    #  - right: foundations, talon
    #

    def bakersDozenLayout(self, rows, texts=0, playcards=9):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks
        halfrows = (rows + 1) / 2

        # set size so that at least 9 cards are fully playable
        h = YS + min(2*YS, (playcards-1)*self.YOFFSET)
        h = max(h, 5*YS/2, 3*YS/2+CH)
        h = min(h, 3*YS)

        # create rows
        x, y = XM, YM
        for i in range(halfrows):
            self.s.rows.append(S(x+i*XS, y))
        for i in range(rows-halfrows):
            self.s.rows.append(S(x+i*XS, y+h))

        # create foundations
        x, y = XM + halfrows * XS, YM
        self.setRegion(self.s.rows, (-999, -999, x - CW / 2, 999999))
        for suit in range(4):
            for i in range(decks):
                self.s.foundations.append(S(x+i*XS, y, suit=suit))
            y = y + YS

        # create talon
        h = YM + 2*h
        self.s.talon = s = S(x, h - YS)
        if texts:
            assert 0

        # set window
        self.size = (XM + (halfrows+decks)*XS, h)


    #
    # FreeCell layout
    #  - top: free cells, foundations
    #  - below: rows
    #  - left bottom: talon
    #

    def freeCellLayout(self, rows, reserves, texts=0, playcards=18):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks
        toprows = reserves + 1 + 4*decks
        maxrows = max(rows, toprows)
        w = XM + maxrows*XS

        # set size so that at least 2/3 of a card is visible with 18 cards
        h = CH*2/3 + (playcards-1)*self.YOFFSET
        h = YM + YS + max(h, 3*YS)

        # create reserves & foundations
        x, y = (w - (toprows*XS - XM))/2, YM
        for i in range(reserves):
            self.s.reserves.append(S(x, y))
            x = x + XS
        for suit in range(4):
            for i in range(decks):
                x = x + XS
                self.s.foundations.append(S(x, y, suit=suit))

        # create rows
        x, y = (w - (rows*XS - XM))/2, YM + YS
        for i in range(rows):
            self.s.rows.append(S(x, y))
            x = x + XS
        self.setRegion(self.s.rows, (-999, y - YM / 2, 999999, 999999))

        # create talon
        x, y = XM, h - YS
        self.s.talon = s = S(x, y)
        if texts:
            # place text right of stack
            s.setText(x + XS, y + CH, anchor="sw", format="%3d")

        # set window
        self.size = (XM + (rows+decks)*XS,  h)


    #
    # Gypsy layout
    #  - left: rows
    #  - right: foundations, talon
    #

    def gypsyLayout(self, rows, waste=0, texts=1, playcards=25):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks

        # set size so that at least 2/3 of a card is visible with 25 cards
        h = CH*2/3 + (playcards-1)*self.YOFFSET
        h = YM + max(h, 5*YS)

        # create rows
        x, y = XM, YM
        for i in range(rows):
            self.s.rows.append(S(x, y))
            x = x + XS
        self.setRegion(self.s.rows, (-999, -999, x - CW / 2, 999999))

        # create foundations
        for suit in range(4):
            for i in range(decks):
                self.s.foundations.append(S(x+i*XS, y, suit=suit))
            y = y + YS

        x, y = x + (decks-1)*XS, h - YS
        if texts:
            x = x - XS/2
        self.s.talon = s = S(x, y)
        if texts:
            # place text right of stack
            s.setText(x + XS, y + CH, anchor="sw", format="%3d")
        if waste:
            x = x - XS
            self.s.waste = s = S(x, y)
            if texts:
                # place text left of stack
                s.setText(x - XM, y + CH, anchor="se", format="%3d")

        # set window
        self.size = (XM + (rows+decks)*XS, h)


    #
    # Harp layout
    #  - top: rows
    #  - bottom: foundations, waste, talon
    #

    def harpLayout(self, rows, waste, texts=1, playcards=19):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks
        w = max(rows*XS, (4*decks+waste+1)*XS, (4*decks+1)*XS+2*XM)
        w = XM + w

        # set size so that at least 19 cards are fully playable
        h = YS + (playcards-1)*self.YOFFSET
        h = max(h, 3*YS)

        # top
        x, y = (w - (rows*XS - XM))/2, YM
        for i in range(rows):
            self.s.rows.append(S(x, y))
            x = x + XS

        # bottom
        x, y = XM, YM + h
        self.setRegion(self.s.rows, (-999, -999, 999999, y - YS / 2))
        for suit in range(4):
            for i in range(decks):
                self.s.foundations.append(S(x, y, suit=suit))
                x = x + XS
        if waste:
            x = w - 2*XS
            self.s.waste = s = S(x, y)
            if texts:
                # place text above stack
                s.setText(x + CW / 2, y - YM, anchor="s")
        x = w - XS
        self.s.talon = s = S(x, y)
        if texts:
            # place text above stack
            s.setText(x + CW / 2, y - YM, anchor="s")

        # set window
        self.size = (w, YM + h + YS)


    #
    # Klondike layout
    #  - top: talon, waste, foundations
    #  - bottom: rows
    #

    def klondikeLayout(self, rows, waste, texts=1, playcards=16, center=1):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks
        frows = 4 * decks
        toprows = 1 + waste + frows
        maxrows = max(rows, toprows)
        yextra = 0

        # set size so that at least 2/3 of a card is visible with 16 cards
        h = CH*2/3 + (playcards-1)*self.YOFFSET
        h = max(h, 2*YS)

        # top
        x, y = XM, YM
        self.s.talon = s = S(x, y)
        if texts:
            if waste or not center or maxrows - frows <= 1:
                # place text below stack
                s.setText(x + CW / 2, y + YS, anchor="n")
                yextra = 20
            else:
                # place text right of stack
                s.setText(x + XS, y, anchor="nw", format="%3d")
        if waste:
            x = x + XS
            self.s.waste = s = S(x, y)
            if texts:
                # place text below stack
                s.setText(x + CW / 2, y + YS, anchor="n")
        x = XM + (maxrows - frows) * XS
        if center and frows + 2 * (1 + waste + 1) <= maxrows:
            # center the foundations
            x = XM + (maxrows - frows) * XS / 2

        for suit in range(4):
            for i in range(decks):
                self.s.foundations.append(S(x, y, suit=suit))
                x = x + XS

        # bottom
        x, y = XM, YM + YS + yextra
        self.setRegion(self.s.rows, (-999, y - YM / 2, 999999, 999999))
        for i in range(rows):
            self.s.rows.append(S(x, y))
            x = x + XS

        # set window
        self.size = (XM + maxrows*XS, YM + YS + yextra + h)


    #
    # Yukon layout
    #  - left: rows
    #  - right: foundations
    #  - left bottom: talon
    #

    def yukonLayout(self, rows, texts=0, playcards=20):
        S = self.__createStack
        CW, CH = self.CW, self.CH
        XM, YM = self.XM, self.YM
        XS, YS = self.XS, self.YS

        decks = self.game.game_info.decks

        # set size so that at least 2/3 of a card is visible with 20 cards
        h = CH*2/3 + (playcards-1)*self.YOFFSET
        h = YM + max(h, 4*YS)

        # create rows
        x, y = XM, YM
        for i in range(rows):
            self.s.rows.append(S(x, y))
            x = x + XS
        self.setRegion(self.s.rows, (-999, -999, x - CW / 2, 999999))

        # create foundations
        for suit in range(4):
            for i in range(decks):
                self.s.foundations.append(S(x+i*XS, y, suit=suit))
            y = y + YS

        # create talon
        x, y = XM, h - YS
        self.s.talon = s = S(x, y)
        if texts:
            # place text right of stack
            s.setText(x + XS, y + CH, anchor="sw", format="%3d")

        # set window
        self.size = (XM + (rows+decks)*XS,  h)

