## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#
    from hint import FreeCellType_Hint                              #bundle#
    from pysoltk import MfxCanvasText                               #bundle#


# /***********************************************************************
# // Baker's Game
# ************************************************************************/

# To simplify playing we also consider the number of free rows.
# Note that this only is legal if RowStack.cap.base_rank == ANY_RANK.
class BakersGame_RowStack(SS_RowStack):
    def canMovePile(self, cards):
        max_move = self.game.getNumberOfFreeReserves() + 1
        n = getNumberOfFreeStacks(self.game.s.rows)
        if n > 1:
            max_move = max_move * n
        return SS_RowStack.canMovePile(self, cards) and len(cards) <= max_move


class BakersGame(Game):
    Layout_Method = Layout.freeCellLayout
    RowStack_Class = BakersGame_RowStack
    Hint_Class = FreeCellType_Hint

    #
    # game layout
    #

    def createGame(self, **layout):
        # create layout
        l, s = Layout(self), self.s
        kwdefault(layout, rows=8, reserves=4, texts=0)
        apply(self.Layout_Method, (l,), layout)
        self.setSize(l.size[0], l.size[1])
        # create stacks
        s.talon = InvisibleTalonStack(l.s.talon.x, l.s.talon.y, self)
        for r in l.s.foundations:
            self.s.foundations.append(SS_FoundationStack(r.x, r.y, self, suit=r.suit))
        for r in l.s.rows:
            s.rows.append(self.RowStack_Class(r.x, r.y, self))
        for r in l.s.reserves:
            self.s.reserves.append(ReserveStack(r.x, r.y, self))
        # default
        l.defaultAll()

    #
    # game overrides
    #

    def startGame(self):
        for i in range(5):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        r = self.s.rows
        ##self.s.talon.dealRow(rows=(r[0], r[1], r[6], r[7]))
        self.s.talon.dealRow(rows=r[:4])
        assert len(self.s.talon.cards) == 0

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit == card2.suit and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))


# /***********************************************************************
# //
# ************************************************************************/

class KingOnlyBakersGame(BakersGame):
    RowStack_Class = StackWrapper(FreeCell_SS_RowStack, base_rank=KING)


# /***********************************************************************
# // Eight Off (Baker's Game in a different layout)
# ************************************************************************/

class EightOff(KingOnlyBakersGame):

    #
    # game layout
    #

    def createGame(self, rows=8, reserves=8):
        # create layout
        l, s = Layout(self), self.s

        # set window
        # (piles up to 16 cards are playable without overlap in default window size)
        h = max(2*l.YS, l.YS+(16-1)*l.YOFFSET)
        maxrows = max(rows, reserves)
        self.setSize(l.XM + maxrows*l.XS, l.YM + l.YS + h + l.YS)

        # create stacks
        x, y = l.XM + (maxrows-4)*l.XS/2, l.YM
        for i in range(4):
            s.foundations.append(SS_FoundationStack(x, y, self, i))
            x = x + l.XS
        x, y = l.XM + (maxrows-rows)*l.XS/2, y + l.YS
        for i in range(rows):
            s.rows.append(self.RowStack_Class(x, y, self))
            x = x + l.XS
        x, y = l.XM + (maxrows-reserves)*l.XS/2, self.height - l.YS
        for i in range(reserves):
            s.reserves.append(ReserveStack(x, y, self))
            x = x + l.XS
        self.setRegion(s.reserves, (-999, y - l.CH / 2, 999999, 999999))
        s.talon = InvisibleTalonStack(l.XM, l.YM, self)

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def startGame(self):
        for i in range(5):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        r = self.s.reserves
        self.s.talon.dealRow(rows=[r[0],r[2],r[4],r[6]])
        assert len(self.s.talon.cards) == 0


# /***********************************************************************
# // Seahaven Towers (Baker's Game in a different layout)
# ************************************************************************/

class SeahavenTowers(KingOnlyBakersGame):

    #
    # game layout
    #

    def createGame(self):
        # create layout
        l, s = Layout(self), self.s

        # set window
        # (piles up to 20 cards are playable in default window size)
        h = max(3*l.YS, 20*l.YOFFSET)
        self.setSize(l.XM + 10*l.XS, l.YM + l.YS + h)

        # create stacks
        x, y = l.XM, l.YM
        for i in range(4):
            s.reserves.append(ReserveStack(x + (i+3)*l.XS, y, self))
        for suit in range(4):
            i = (9, 0, 1, 8)[suit]
            s.foundations.append(SS_FoundationStack(x + i*l.XS, y, self, suit))
        x, y = l.XM, l.YM + l.YS
        for i in range(10):
            s.rows.append(self.RowStack_Class(x, y, self))
            x = x + l.XS
        self.setRegion(s.rows, (-999, y - l.YM / 2, 999999, 999999))
        s.talon = InvisibleTalonStack(l.XM, self.height-l.YS, self)

        # define stack-groups
        self.sg.openstacks = s.foundations + s.rows + s.reserves
        self.sg.talonstacks = [s.talon]
        self.sg.dropstacks = s.rows + s.reserves
        self.sg.reservestacks = s.reserves

    #
    # game overrides
    #

    def startGame(self):
        for i in range(4):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        self.s.talon.dealRow(rows=(self.s.reserves[1:3]))
        assert len(self.s.talon.cards) == 0


# /***********************************************************************
# //
# ************************************************************************/

class RelaxedSeahavenTowers(SeahavenTowers):
    RowStack_Class = KingSS_RowStack


# /***********************************************************************
# // Penguin
# ************************************************************************/

class Penguin(KingOnlyBakersGame):

    #
    # game layout
    #

    def createGame(self, rows=7, reserves=7):
        # create layout
        l, s = Layout(self), self.s

        # set window
        # (piles up to 16 cards are playable without overlap in default window size)
        h = max(3*l.YS, l.YS+(16-1)*l.YOFFSET)
        maxrows = max(rows, reserves)
        self.setSize(l.XM + (maxrows+1)*l.XS, l.YM + h + l.YS)

        # extra settings
        self.base_card = None

        # create stacks
        x, y = self.width - l.XS, l.YM
        for i in range(4):
            s.foundations.append(SS_FoundationStack(x, y, self, i, mod=13))
            y = y + l.YS
        x, y = l.XM + (maxrows-rows)*l.XS/2, l.YM
        for i in range(rows):
            s.rows.append(self.RowStack_Class(x, y, self))
            x = x + l.XS
        x, y = l.XM + (maxrows-reserves)*l.XS/2, self.height - l.YS
        for i in range(reserves):
            s.reserves.append(ReserveStack(x, y, self))
            x = x + l.XS
        self.setRegion(s.reserves, (-999, y - l.CH / 2, 999999, 999999))
        s.talon = InvisibleTalonStack(l.XM+1, y, self)

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def _shuffleHook(self, cards):
        # move base cards to top of the Talon (i.e. first cards to be dealt)
        beak = cards[0]
        base_cards = []
        for c in cards[:]:
            if c.rank == beak.rank:
                base_cards.append(c)
                cards.remove(c)
        return cards + base_cards

    def startGame(self):
        self.base_card = self.s.talon.cards[-4]
        self._updateStacks()
        # deal base cards to Foundations
        for i in range(3):
            c = self.s.talon.getCard()
            assert c.rank == self.base_card.rank
            to_stack = self.s.foundations[c.suit * self.game_info.decks]
            self.flipMove(self.s.talon)
            self.moveMove(1, self.s.talon, to_stack, frames=0)
        # deal rows
        for i in range(6):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        assert len(self.s.talon.cards) == 0

    def _restoreGameHook(self, game):
        self.base_card = self.cards[game.loadinfo.base_card_id]
        self._updateStacks()

    def _loadGameHook(self, p):
        self.loadinfo.addattr(base_card_id=None)    # register extra load var.
        self.loadinfo.base_card_id = p.load()

    def _saveGameHook(self, p):
        p.dump(self.base_card.id)

    #
    # game extras
    #

    def _updateStacks(self):
        for s in self.s.foundations:
            s.cap.base_rank = self.base_card.rank
        for s in self.s.rows:
            s.cap.base_rank = (self.base_card.rank - 1) % 13


# register the game
registerGame(GameInfo(45, BakersGame, "Baker's Game",
                      GI.GT_FREECELL, 1, 0))
registerGame(GameInfo(26, KingOnlyBakersGame, "King Only Baker's Game",
                      GI.GT_FREECELL, 1, 0))
registerGame(GameInfo(258, EightOff, "Eight Off",
                      GI.GT_FREECELL, 1, 0))
registerGame(GameInfo(9, SeahavenTowers, "Seahaven Towers",
                      GI.GT_FREECELL, 1, 0))
registerGame(GameInfo(6, RelaxedSeahavenTowers, "Relaxed Seahaven Towers",
                      GI.GT_FREECELL | GI.GT_RELAXED, 1, 0))
registerGame(GameInfo(64, Penguin, "Penguin",
                      GI.GT_FREECELL, 1, 0))

