/*
 * File: taunt.c
 * Author: Brent Hendricks
 * Project: NetSpades
 * Date: 5/25/98
 *
 * This file contains the code for the Taunt Server for NetSpades.
 *
 * Copyright (C) 1998 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <config.h>

#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <socketfunc.h>
#include <engine.h>
#include <taunt.h>

/* Global array of games */
gameInfo_t games[MAX_GAME];
int sock;


void tauntServer( int p[2], int port ) {

  int i, new_sock, gameNum;
  fd_set read_fd_set, active_fd_set;
  struct sockaddr_in clientname;
  size_t size = sizeof (clientname);

  
  /* Signal handler: Need to write specific for taunt */
  signal( SIGINT, tsSigHandler );
  signal( SIGPIPE, tsSigHandler );
  
#ifdef DEBUG_TAUNT
  fprintf( stderr, "TS[%d]: Taunt server invoked\n", getpid() );
#endif
  
  /* Create SERVER socket on port */
  sock = makesocket( port, SERVER, NULL );
  
  /* Check for socket errors */
  if( sock < 0 ) {
    fprintf( stderr, "TS[%d]: %s\n", getpid(), errorMsg[ sock + ERR_OFFSET ] );
    exit(-1);
  }

  /* Start accepting connections */
  if (listen (sock, 4) < 0) {
    fprintf( stderr, "TS[%d]: Error listening to socket\n", getpid() );
    exit(-1);
  }

  /* Initialize the set of active sockets. */
  FD_ZERO( &active_fd_set );
  FD_SET( sock, &active_fd_set );
  FD_SET( p[0], &active_fd_set );

  /* Main port-listener loop */
  while (1) {

    read_fd_set = active_fd_set;

    /* Wait for someone to talk */
    if( select (FD_SETSIZE, &read_fd_set, NULL, NULL, NULL) < 0) {
      fprintf( stderr, "TS[%d]: Error in select (%s)\n", getpid(), 
	      strerror(errno) );
      exit( -1 );
    }
     
    /* Service all the sockets with input pending. */
    for( i = 0; i < FD_SETSIZE; ++i )
      if( FD_ISSET (i, &read_fd_set)) {
	
	/* Data from spades server */
	if( i == p[0] ) {
	  if( (gameNum = GetNewGame( i )) < 0 ) {
	    GameEnded( gameNum, &active_fd_set );
	  }
	}
	
	/* Connection request on original socket. */
	else if( i == sock ) {
	  if ( (new_sock = accept( sock, (struct sockaddr *)&clientname,
				  &size )) < 0 ) {
	    fprintf( stderr, "TS[%d]: Error accepting connection\n", getpid() );
	    close( sock );
	    close( new_sock );
	    exit( -1 );
	  }
	  if( InitPlayer( new_sock ) < 0 ) {
	    close( new_sock );
	  }
	  else {
	    FD_SET( new_sock, &active_fd_set );	  
	  }
	}

	/* Data arriving on an already-connected socket. */
	else {
	  if( ProcessMessage( i ) < 0 ) {
	    close( i );
	    FD_CLR( i, &active_fd_set );
	  }
	}
      } /* if( FD_ISSET ) */
  } /* while( 1 ) */
}


/* 
 * Read in and verify new player data.  Then send message 
 */
int InitPlayer( int msgsock ) {

  int i, gameNum, gamePid, playerNum, clientPid, status=-1;
  char* message;

  /* Check for valid game number */
  if( readint( msgsock, &gameNum ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading game number\n", getpid() );
#endif
  }
  else if( gameNum < 0 || gameNum >= MAX_GAME ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Game num out of bounds\n", getpid() );
#endif
  }

  /* Chack for valid game pid */
  else if( readint( msgsock, &gamePid ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading game PID\n", getpid() );
#endif
  }
  else if( games[gameNum].gamePid != gamePid ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Game PID mismatch\n", getpid() );
#endif
  }

  /* Check for valid player num */
  else if( readint( msgsock, &playerNum ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading player number\n", getpid() );
#endif
  }
  else if( playerNum < 0 || playerNum >= 4 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Player num out of bounds\n", getpid() );
#endif
  }

  /* Check for valid client pid */
  else if( readint( msgsock, &clientPid ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading client PID\n", getpid() );
#endif
  }
  else if( games[gameNum].clientPids[ playerNum ] != clientPid ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Client PID mismatch\n", getpid() );
#endif
  }

  /* Check is this player number is already registered */
  else if ( games[gameNum].playerSock[playerNum] != -1 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Player %d already registered\n", getpid(), 
	    playerNum );
#endif
  }
  
  /* Return 'OK' */
  else if( writeint( msgsock, 1 ) < 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error writing to socket\n", getpid() );
#endif
  }

  /* Send message */
  else if( writeint( msgsock, -1 ) < 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error writing to socket\n", getpid() );
#endif
  }
  else if( writestring( msgsock, "Welcome to TauntServer (tm)") < 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error writing to socket\n", getpid() );
#endif
  }

  /* Everything passed */
  else {
    games[gameNum].playerSock[playerNum] = msgsock;
    status = 0;
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Registered player %d in game %d\n", getpid(),
	    playerNum, gameNum );
#endif
  }
  
  return status;

}


/*
 * Read in a message from a player and send it out players in same game
 */
int ProcessMessage( int msgsock, fd_set* fds ) {

  int i, gameNum, gamePid, playerNum, clientPid, status=-1;
  char* message;

  /* Check for valid game number */
  if( readint( msgsock, &gameNum ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading game number\n", getpid() );
#endif
  }
  else if( gameNum < 0 || gameNum >= MAX_GAME ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Game num out of bounds\n", getpid() );
#endif
  }

  /* Chack for valid game pid */
  else if( readint( msgsock, &gamePid ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading game PID", getpid() );
#endif
  }
  else if( games[gameNum].gamePid != gamePid ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Game PID mismatch\n", getpid() );
#endif 
  }

  /* Check for valid player num */
  else if( readint( msgsock, &playerNum ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading player number", getpid() );
#endif
  }
  else if( playerNum < 0 || playerNum >= 4 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Player num out of bounds\n", getpid() );
#endif
  }

  /* Check for valid client pid */
  else if( readint( msgsock, &clientPid ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading client PID", getpid() );
#endif
  }
  else if( games[gameNum].clientPids[ playerNum ] != clientPid ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Client PID mismatch\n", getpid() );
#endif
  }

  /* Read message */
  else if( readstring( msgsock, &message ) <= 0 ) {
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Error reading message\n", getpid() );
#endif
  }
  else {
    status = 0;
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: \"%s\" from player %d in game %d\n", getpid(),
	    message, playerNum, gameNum );
#endif
    for( i=0; i<4; i++ ) {
      if( games[gameNum].playerSock[i] > 0 ) {

	if( writeint( games[gameNum].playerSock[i], playerNum ) < 0 ) {
	  close( games[gameNum].playerSock[i] );
	  FD_CLR( games[gameNum].playerSock[i], fds );
	  games[gameNum].playerSock[i] = -1;
#ifdef DEBUG_TAUNT
	  fprintf( stderr, "TS[%d]: Error writing to socket\n", getpid() );
#endif
	}
	else if( writestring( games[gameNum].playerSock[i], message ) < 0 ) {
	  close( games[gameNum].playerSock[i] );
	  FD_CLR( games[gameNum].playerSock[i], fds );
	  games[gameNum].playerSock[i] = -1;
#ifdef DEBUG_TAUNT
	  fprintf( stderr, "TS[%d]: Error writing to socket\n", getpid() );
#endif
	}
      }
    }
    free( message );
  }

  return status;

}


int GetNewGame( int pipe ) {

  int i, gameNum, gamePid, status=1;
  
  readint( pipe, &gameNum );
  readint( pipe, &gamePid );

  if( gamePid < 0 ) { 
    /* Game is over so clear gameInfo struct */
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Game Num: %d Pid: %d is over\n", getpid(), 
	    gameNum, games[gameNum].gamePid );
#endif

    /* Encode the ended game number in a return status <0 */
    status = gameNum - MAX_GAME;
    
  }
  else {
    /* Read new game information */
    games[gameNum].gamePid = gamePid;
    for( i=0; i < 4; i++ ) {
      readint( pipe, &(games[gameNum].clientPids[i]) );
      readstring( pipe, &(games[gameNum].players[i]) );
      games[gameNum].playerSock[i] = -1;
    }
    
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: New game registered with TS. Num: %d Pid: %d\n", 
	    getpid(), gameNum, games[gameNum].gamePid );
#endif
  
  }
    return status;
}



void GameEnded( int index, fd_set* fds ) {
  
  /* Deocde the game number from the negative index we receive*/
  int i, gameNum = MAX_GAME+index;
  
  games[gameNum].gamePid = -1;
#ifdef DEBUG_TAUNT
    fprintf( stderr, "TS[%d]: Closing sockets for game %d\n", getpid(), gameNum );
#endif
  for( i=0; i<4; i++) {
    games[gameNum].clientPids[i] = -1;
    if( games[gameNum].playerSock[i] != -1 ) {
      close( games[gameNum].playerSock[i] );
      FD_CLR( games[gameNum].playerSock[i], fds );
      games[gameNum].playerSock[i] = -1;
    }
    free( games[gameNum].players[i] );
  }
}


RETSIGTYPE tsSigHandler(int signum) {
  signal (signum, SIG_DFL);
  /*printf("Oooh boy, I just got signal %d\n",signum);*/

  /* We let our error checker handle broken pipes */
  if( signum != SIGPIPE ) { 
    tsNetClose();
    raise(signum);
  }
}


void tsNetClose( void ) {
  
  int i,j;
  
  close(sock);
  for( i=0; i< MAX_GAME; i++) {
    for( j=0; j<4; j++) {
      if( games[i].playerSock[j] != -1 ) {
	close( games[i].playerSock[j] );
      }
    }
  }
}
	    
	      
