/* -*- mode: fundamental -*- */

%{

#ifndef NDEBUG
#define YYDEBUG 1
#endif

#include "rg.h"

#include "GameItem.h"
#include "GameItems.h"
#include "gameitems.h"
#include "../util/Collection.h"
#include "../util/Object.h"
#include "../util/debug.h"
#include "../util/util.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

extern int rglex();
static void rgerror(char* s);

static int fillStruct(void* data, MemberTable* members, Collection<void>* rubbishMan);

//--------------------------------------------------------------------------------------------------------------------------------

class Variable : public Object
{
private:
 char* pName;

public:
 Variable(const char* _name)
  { pName = newString(_name); }
 Variable(Variable& object)
  { pName = newString(object.pName); }
 virtual ~Variable()
  { delete pName; }
 const char* name()
  { return pName; }
};

//--------------------------------------------------------------------------------------------------------------------------------

class StringVar : public Variable
{
private:
 char* pValue;

public:
 StringVar(const char* _name, const char* _value)
  : Variable(_name) 
  { pValue = newString(_value); }
 StringVar(StringVar& object)
  : Variable(object)
  { pValue = newString(object.pValue); }
 ~StringVar() 
  { delete pValue; }

 StringVar& operator = (const char* _value)
  {
   delete pValue;
   pValue = newString(_value);
   return *this;
  }

 const char* value() { return pValue; }
};
template class Collection<StringVar>;

//--------------------------------------------------------------------------------------------------------------------------------

class IntegerVar : public Variable
{
private:
 int pValue;

public:
 IntegerVar(const char* _name, int _value) 
  : Variable(_name)
  { pValue = _value; }
 IntegerVar(IntegerVar& object)
  : Variable(object)
  { pValue = object.pValue; }

 IntegerVar& operator = (int _value)
  {
   pValue = _value;
   return *this;
  }
 int value() { return pValue; }
};
template class Collection<IntegerVar>;

//--------------------------------------------------------------------------------------------------------------------------------

class VarContext : public Object
{
private:
 VarContext* pNext;
 Collection<IntegerVar>* pIntegerVars;
 Collection<StringVar>* pStringVars;

 VarContext* next(VarContext* _next) { return pNext = _next; }

public:
 VarContext(VarContext* _next)
  {
   // create some new variable collections
   pIntegerVars = new Collection<IntegerVar>(32, 16);
   pStringVars = new Collection<StringVar>(32, 16);

   // if there is a context around us copy it's variables
   if(next(_next))
    {
     foreach(*next()->integerVars(), i) integerVars()->add(new IntegerVar(*next()->integerVars()->at(i)));
     foreach(*next()->stringVars(), i) stringVars()->add(new StringVar(*next()->stringVars()->at(i)));
    }
  }
 ~VarContext()
  {
   delete pIntegerVars;
   delete pStringVars;
  }

 VarContext* next() { return pNext; }
 Collection<IntegerVar>* integerVars() { return pIntegerVars; }
 Collection<StringVar>* stringVars() { return pStringVars; }
};

//--------------------------------------------------------------------------------------------------------------------------------

static VarContext* varContext; // created by rgParse
static GameItems* gameItems; // set by rgParse to it's parameter

//--------------------------------------------------------------------------------------------------------------------------------

%}

%union
{
 char* string;
 int integer;
}

%token <string>  d_string d_symbol
%token <integer> d_number
%token k_hatgame k_level k_cutscene

%type <integer> expr
%left '^' '*' '/' '+' '-'

%start input

%%

input	: k_hatgame d_number {
	    VPRINTF("<rg> hatgame {\n");
            if($2 != 1) { yyerror("Wrong version"); YYABORT; }
          } '{' lines '}' {
	    VPRINTF("<rg> } hatgame\n");
            //if(rg_setGameOptions() != 0) YYABORT;
          }
	;

lines	: /* empty */ {}

	| lines d_symbol '=' d_string
           {
	    // get rid of any old definition
	    foreach(*varContext->stringVars(), i)
	     if(strcmp(varContext->stringVars()->at(i)->name(), $2) == 0)
	      { varContext->stringVars()->del(i); varContext->stringVars()->pack(); break; }

	    // set the new value
	    varContext->stringVars()->add(new StringVar($2, $4));

	    // free the strings we were passed
	    free($2);
	    free($4);
	   }

	| lines d_symbol '=' expr 
           {
	    VPRINTF("<rg> setting int \"%s\" to %d\n", $2, $4);

	    // get rid of any old definition
	    foreach(*varContext->integerVars(), i)
	     if(strcmp(varContext->integerVars()->at(i)->name(), $2) == 0)
	      { varContext->integerVars()->del(i); varContext->integerVars()->pack(); break; }

	    // set the new value
	    varContext->integerVars()->add(new IntegerVar($2, $4));

	    // free the strings we were passed
	    free($2);
	   }

        | lines k_level
           {
	    varContext = new VarContext(varContext);
            VPRINTF("<rg> level {\n");
          } '{' lines '}' {
            LevelOptions* levelOptions = new LevelOptions;
	    memcpy(levelOptions, levelOptionsDefault, sizeof(LevelOptions));
            Collection<void>* rubbishMan = new Collection<void>;
	    fillStruct(levelOptions, levelOptionsMembers, rubbishMan);
	    gameItems->add(new GameItemLevel(levelOptions, rubbishMan));
            VPRINTF("<rg> } level\n");
	  }

	;

expr	: d_number { $$ = $1; }

        | d_symbol
           {
	    foreach(*varContext->integerVars(), i)
	     if(strcmp(varContext->integerVars()->at(i)->name(), $1) == 0)
	      { $$ = varContext->integerVars()->at(i)->value(); goto foundSymbol; }

	    yyerror("undefined symbol");
	    YYABORT;

	   foundSymbol:
	    free($1);
	   }

	| '(' expr ')' { $$ = $2; }
	| expr '+' expr { $$ = $1 + $3; }
	| expr '-' expr { $$ = $1 - $3; }
	| expr '*' expr { $$ = $1 * $3; }
	| expr '/' expr { $$ = $1 / $3; }
	| expr '^' expr { $$ = $1 ^ $3; }
	| '-' expr { $$ = - $2; }

	;

%%

//--------------------------------------------------------------------------------------------------------------------------------

bool rgParse(GameItems* _gameItems)
{
 VPRINTF("<rg> rgParse(...)\n");
 assert(_gameItems != NULL);
 gameItems = _gameItems;
 varContext = new VarContext(NULL);

 int i = yyparse();

 delete varContext;
 return (i==0)? true : false;
}

//--------------------------------------------------------------------------------------------------------------------------------

static int fillStruct(void* data, MemberTable* members, Collection<void>* rubbishMan)
{
 while(members->type != mt_end)
  {
   switch(members->type)
    {
    case mt_str:
     foreach(*varContext->stringVars(), i)
      if(strcmp(varContext->stringVars()->at(i)->name(), members->name) == 0)
       {
	char* s = newString(varContext->stringVars()->at(i)->value());
	rubbishMan->add(s);
	*(char**)((char*)data + (unsigned long)members->offset) = s;
	break;
       }
     break;
     
    case mt_int:
     foreach(*varContext->integerVars(), i)
      if(strcmp(varContext->integerVars()->at(i)->name(), members->name) == 0)
       {
	*(int*)((char*)data + (unsigned long)members->offset) = varContext->integerVars()->at(i)->value();
	break;
       }
     break;
     
    case mt_end:
     fatal("Something has gone horribly, horribly wrong...\n");
     // this can never actually happen, but gcc complains...
     break;
    }
   members++;
  }
 return 0;
}

//--------------------------------------------------------------------------------------------------------------------------------

static void rgerror(char* s)
{
 nonFatal("%s\n", s);
}

//--------------------------------------------------------------------------------------------------------------------------------
