/* The Ace of Penguins - gif2pack.c
   Copyright (C) 1998 DJ Delorie

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>

#define NDEBUG
#include <assert.h>

#include "gd.h"

char cchars[] = "=_0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
#define MAXCOLORS strlen(cchars)

int num_images=0;

char *base = "image_table";
FILE *out;

char **strings=0;
int max_strings=0;
int num_strings=0;

void
add_string(char *s)
{
  if (num_strings >= max_strings)
  {
    max_strings += 100;
    if (strings)
      strings = (char **)realloc(strings, max_strings*sizeof(char *));
    else
      strings = (char **)malloc(max_strings*sizeof(char *));
  }
  strings[num_strings++] = strdup(s);
}

char *data;
int data_size;

void
merge_strings()
{
  int i;
  char *dp;
  data_size = 0;
  for (i=0; i<num_strings; i++)
    data_size += strlen(strings[i]) + 1;
  data = (char *)malloc(data_size+1);
  dp = data;
  for (i=0; i<num_strings; i++)
  {
    strcpy(dp, strings[i]);
    dp += strlen(strings[i]);
    *dp++ = '\n';
  }
  *dp++ = 0;
  /*printf("%s", data);*/
}

#define DUMP 0

int bytes_in_line = 0;

int
put_byte(int b)
{
  if (bytes_in_line == 12)
  {
    fputc('\n', out);
    bytes_in_line = 0;
  }
  fprintf(out, " 0x%02x,", b&0xff);
  bytes_in_line ++;
}

void
compress_data()
{
  int i, j, l;
  int compressed_size = 0;
  int bpv=1, mo=3, cutoff=0x100;

  fprintf(out, "/* This file is generated by gif2pack - DO NOT EDIT */\n");
  fprintf(out, "/* Copyright for this file is copied from the images /*\n");
  fprintf(out, "/* from which it is generated */\n\n");
  fprintf(out, "int %s_num_images = %d;\n", base, num_images);
  fprintf(out, "int %s_data_size = %d;\n", base, data_size);
  fprintf(out, "int %s_num_strings = %d;\n", base, num_strings);
  fprintf(out, "unsigned char %s_compressed[] = {\n", base);

  fprintf(stderr, "%10d     Compressing...\n", data_size);
  for (i=0; i<data_size;)
  {
    int longest_match = 0;
    int longest_match_posn = 0;

    if (i>=cutoff)
    {
      bpv ++;
      mo = 1+2*bpv;
      cutoff <<= 8;
    }

#if DUMP
    printf("checking: ");
    for (l=0; l<data_size-i && l<70; l++)
      if (data[i+l] < ' ')
	putchar('?');
      else
	putchar(data[i+l]);
    putchar('\n');
#endif

    /* first, see if there's a matching string before us */
    for (j=0; j<i; j++)
    {
      if (data[j] != data[i])
	continue;
      for (l=0; l<(i-j) && l<(data_size-i); l++)
	if (data[j+l] != data[i+l])
	  break;
      if (longest_match < l)
      {
	longest_match = l;
	longest_match_posn = j;
      }
    }
#if DUMP
    printf("longest match is %d chars at %d\n",
	   longest_match, longest_match_posn);
#endif

    /* Now see how long the run is */
    for (l=1; l<data_size-i && l<127; l++)
      if (data[i] != data[i+l])
	break;
#if DUMP
    printf("longest run is %d bytes\n", l);
#endif

    /* two bytes for runs, seven for matches, one for uncompressed */
    if (l-1 > longest_match-mo && l>2)
    {
#if DUMP
      printf("run of %d %02x chars\n", l, data[i]);
#endif
      compressed_size += 2;
      assert(l>2 && l<127);

      put_byte(l + 128);
      put_byte(data[i]);
      i += l;
    }
    else if (longest_match > mo)
    {
      int b, t;
#if DUMP
      printf("match %d chars at %d\n", longest_match, longest_match_posn);
#endif
      compressed_size += mo;

      put_byte(128);
      t = longest_match_posn;
      for (b=0; b<bpv; b++)
      {
	put_byte(t);
	t >>= 8;
      }
      t = longest_match;
      for (b=0; b<bpv; b++)
      {
	put_byte(t);
	t >>= 8;
      }
      i += longest_match;
    }
    else
    {
#if DUMP
      printf("uncompressed char: `%c'\n", data[i]<' '?'?':data[i]);
#endif
      compressed_size += 1;
      put_byte(data[i]);
      i++;
    }

    fprintf(stderr, "%10d %10d  %5.1f%%\r",
	    i, compressed_size, compressed_size*100.0/i);
    fflush(stderr);
  }
  fprintf(stderr, "\n");

  fprintf(out, "\n 0 };\n");
  fprintf(out, "int %s_compressed_size = %d;\n", base, compressed_size);

  fprintf(out, "\n/* %d -> %d, %.1f%% of original! */\n",
	  data_size, compressed_size, compressed_size*100.0/data_size);
}

void
convert_gif(char *filename)
{
  FILE *f;
  int used[256];
  int pixel[256];
  int cc[256];
  int ncolors, n;
  int x, y;
  gdImagePtr gd;
  char tmp[80];
  char *line;
  int transparent_color = -1;

  f = fopen(filename, "rb");
  if (!f)
  {
    perror(filename);
    exit(1);
  }
  gd = gdImageCreateFromGif(f);
  fclose(f);

  ncolors = 0;
  memset(used, 0, sizeof(used));
  memset(pixel, 0, sizeof(pixel));
  memset(cc, 0, sizeof(cc));

  transparent_color = gdImageGetTransparent(gd);

  for (x=0; x<gdImageSX(gd); x++)
    for (y=0; y<gdImageSY(gd); y++)
    {
      int c = gdImageGetPixel(gd, x, y);
      if (!used[c])
      {
	used[c] = 1;
	pixel[ncolors] = c;
	cc[c] = cchars[ncolors];
	ncolors++;
      }
    }
  if (ncolors > MAXCOLORS)
  {
    fprintf(stderr, "Error: %s has more than %d colors\n", filename, MAXCOLORS);
    exit(1);
  }

  add_string(filename);
  sprintf(tmp, "%d %d %d 1", gdImageSX(gd), gdImageSY(gd), ncolors);
  add_string(tmp);
  for (n=0; n<ncolors; n++)
  {
    if (pixel[n] == transparent_color)
      sprintf(tmp, "%c c None", cchars[n]);
    else
      sprintf(tmp, "%c c #%02x%02x%02x", cchars[n],
	      gdImageRed(gd, pixel[n]),
	      gdImageGreen(gd, pixel[n]),
	      gdImageBlue(gd, pixel[n]));
    add_string(tmp);
  }
  line = (char *)malloc(gdImageSX(gd)+1);
  line[gdImageSX(gd)] = 0;
  for (y=0; y<gdImageSY(gd); y++)
  {
    for (x=0; x<gdImageSX(gd); x++)
      line[x] = cc[gdImageGetPixel(gd, x, y)];
    add_string(line);
  }

  num_images++;
}

void convert_text(char *filename)
{
  struct stat st;
  char *cp;
  FILE *f = fopen(filename, "r");
  stat(filename, &st);
  cp = (char *)malloc(st.st_size+1);
  fread(cp, 1, st.st_size, f);
  cp[st.st_size] = 0;
  add_string(cp);
  fclose(f);
}

int
main(int argc, char **argv)
{
  int i;
  struct rusage start, end;

  out = stdout;
  while (argc>2 && argv[1][0] == '-')
  {
    if (strcmp(argv[1], "-b") == 0)
      base = argv[2];
    if (strcmp(argv[1], "-o") == 0)
    {
      FILE *tmp = fopen(argv[2], "w");
      if (tmp) out = tmp;
    }
    argc -= 2;
    argv += 2;
  }

  getrusage(RUSAGE_SELF, &start);
  for (i=1; i<argc; i++)
  {
    if (strstr(argv[i], ".gif"))
      convert_gif(argv[i]);
    else
      convert_text(argv[i]);
  }
  merge_strings();
  compress_data();
  if (out != stdout)
    fclose(out);
  getrusage(RUSAGE_SELF, &end);

  i = end.ru_utime.tv_sec - start.ru_utime.tv_sec;
  fprintf(stderr, "CPU Time: %d:%02d\n", i/60, i%60);
  return 0;
}
