//  UMap.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "UMap.h"
#include "UCache.h"
#include "UParser.h"
#include "UCommon.h"

#include <iostream.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef HAVE_MMAP
#include <fcntl.h>
#include <sys/mman.h>
#if !defined (__svr4__) && !defined (__SVR4) && defined (sun)
extern "C" {
extern int munmap(void *, size_t);
}
#endif
#endif

#include <sys/stat.h>
#include <netinet/in.h>





static void l_initUMap();

// I moved this uout of the class definition 
static UCache<UMapCacheItem*>* cache=0;

char*		UMap::DEFAULT_PATH=USE_PATH;
char*		UMap::DEFAULT_NAME="8859_1";

#define UGS_AT(_c, _at) ((_c[_at]<<8)|_c[_at+1])

//
// Destructor
//
UMapCacheItem::~UMapCacheItem()
{
	switch (cacheItem->mapType)
	{
	case UMapDynamic:
		delete cacheItem->name;
		delete cacheItem->decodeMatrix;
		delete cacheItem->encodeMatrix;
		break;
	case UMapMMap:
#ifdef HAVE_MMAP
		munmap ((char*) cacheItem->buf, cacheItem->buflen);
		close (cacheItem->fd);
		delete cacheItem->name;
#endif
		break;
	case UMapStatic:
	case UMapUnicode:
		break;
	}
	delete cacheItem;
}

//
// The items of the structure are copied into a new array if the structure 
// is non-static.
// Static members will not receive the delete. 
//
UMapCacheItem::UMapCacheItem (const UMapCacheItemStruct *orig)
{
	cacheItem = new UMapCacheItemStruct [1];
	CHECKNULL (cacheItem);

	if (orig->mapType==UMapStatic || orig->mapType==UMapMMap ||
		orig->mapType==UMapUnicode)
	{
		memcpy (cacheItem, orig, sizeof (UMapCacheItemStruct));
	}
	else
	{
		int		decodeSize;
		int		encodeSize;

		decodeSize = (orig->decodeLowMax - orig->decodeLowMin +1)
			* (orig->decodeHighMax - orig->decodeHighMin +1);
		encodeSize = (orig->encodeLowMax - orig->encodeLowMin +1)
			* (orig->encodeHighMax - orig->encodeHighMin +1);

		cacheItem->mapType = orig->mapType;

		cacheItem->name = new char [strlen(orig->name) +1];
		CHECKNULL (cacheItem->name);
		strcpy (cacheItem->name, orig->name);

		cacheItem->decodeHighMin = orig->decodeHighMin;
		cacheItem->decodeHighMax = orig->decodeHighMax;
		cacheItem->decodeLowMin = orig->decodeLowMin;
		cacheItem->decodeLowMax = orig->decodeLowMax;
		cacheItem->decodeMatrix = new UCS2[decodeSize];
		CHECKNULL (cacheItem->decodeMatrix);
		memcpy (cacheItem->decodeMatrix,
			orig->decodeMatrix, decodeSize * sizeof (UCS2));

		cacheItem->encodeHighMin = orig->encodeHighMin;
		cacheItem->encodeHighMax = orig->encodeHighMax;
		cacheItem->encodeLowMin = orig->encodeLowMin;
		cacheItem->encodeLowMax = orig->encodeLowMax;
		cacheItem->encodeMatrix = new UCS2[encodeSize];
		CHECKNULL (cacheItem->encodeMatrix);
		memcpy (cacheItem->encodeMatrix,
			orig->encodeMatrix, encodeSize * sizeof (UCS2));
	}
}

//
// UMap
//
UMap::UMap ()
{
	if (cache == 0)
	{
		cache =  new UCache<UMapCacheItem*> (0);
		CHECKNULL (cache);
		l_initUMap ();
	}
	cacheItem = 0;
	(void) rename (DEFAULT_NAME);
}
//
// UMap
//
UMap::UMap (const char *name)
{
	if (cache == 0)
	{
		cache =  new UCache<UMapCacheItem*> (0);
		CHECKNULL (cache);
		l_initUMap ();
	}
	cacheItem = 0;
	if (rename (name) != OK)
	{
		(void) rename (DEFAULT_NAME);
	}
}

//
// UMap
//
UMap::~UMap()
{
	if (cacheItem != 0 && cache != 0) 
	{
		cache->unuseItem (cacheItem->name);
	}
}

//
// later this one should serach the path for files
//
UMap::UStatus
UMap::rename (const char *name)
{
	UMapCacheItem		*item;
	if (cacheItem != 0)
	{
		cache->unuseItem (cacheItem->name);
		cacheItem =0;
	}
	item = cache->getItem (name);
	if (item==0) 
	{
		// Try to load it from file.
		if (load (name) == OK)
		{
			return OK;
		}
		rename (DEFAULT_NAME);
		return ERROR;
	}
	cacheItem = item->cacheItem;
	cache->useItem (cacheItem->name);
	return OK;
}

//
// encode data
//
UCS2
UMap::encode (const UCS2 data)
{
	unsigned char		high = data>>8;
	unsigned char		low = data & 0xff;
	int			arrayIndex;

	if (cacheItem == 0) return 0;
	if (cacheItem->mapType == UMapUnicode) return data;

	if (low < cacheItem->encodeLowMin 
		|| low > cacheItem->encodeLowMax
		|| high < cacheItem->encodeHighMin 
		|| high > cacheItem->encodeHighMax)
	{
		return 0;
	}
	arrayIndex = (high - cacheItem->encodeHighMin) 
		* (cacheItem->encodeLowMax - cacheItem->encodeLowMin+1)
		+ (low - cacheItem->encodeLowMin);
	// big endian
	if (cacheItem->mapType==UMapMMap)
	{
		return (UCS2) ntohs (cacheItem->encodeMatrix[arrayIndex]);
	}
	else
	{
		return cacheItem->encodeMatrix[arrayIndex];
	}
}

//
// decode data
//
UCS2
UMap::decode (const UCS2 data)
{
	unsigned char		high = data>>8;
	unsigned char		low = data & 0xff;
	int			arrayIndex;

	if (cacheItem == 0) return 0;
	if (cacheItem->mapType == UMapUnicode) return data;

	if (low < cacheItem->decodeLowMin 
		|| low > cacheItem->decodeLowMax
		|| high < cacheItem->decodeHighMin 
		|| high > cacheItem->decodeHighMax)
	{
		return 0;
	}
	arrayIndex = (high - cacheItem->decodeHighMin) 
		* (cacheItem->decodeLowMax - cacheItem->decodeLowMin+1)
		+ (low - cacheItem->decodeLowMin);
	// big endian
	if (cacheItem->mapType==UMapMMap)
	{
		return (UCS2) ntohs (cacheItem->decodeMatrix[arrayIndex]);
	}
	else
	{
		return cacheItem->decodeMatrix[arrayIndex];
	}
}

UCS2
UMap::encodeResultMax()
{
	if (cacheItem == 0) return 0;
	return  (cacheItem->decodeHighMax << 8 ) + 0xff;
}

UCS2
UMap::decodeResultMax()
{
	if (cacheItem == 0) return 0;
	return  (cacheItem->encodeHighMax << 8 ) + 0xff;
}

//
// create the stuff from data taht may not change
//
UMap::UMap ( 	char	*name, 
		UMapType	mapType,
		UCS2	decode_high_min, 
		UCS2	decode_high_max, 
		UCS2	decode_low_min, 
		UCS2	decode_low_max, 
		UCS2*	decode_matrix, 

		UCS2	encode_high_min, 
		UCS2	encode_high_max, 
		UCS2	encode_low_min, 
		UCS2	encode_low_max, 
		UCS2*	encode_matrix)
{
	cacheItem = 0;
 	load (name, mapType,
		decode_high_min, 
		decode_high_max, 
		decode_low_min, 
		decode_low_max, 
		decode_matrix, 

		encode_high_min, 
		encode_high_max, 
		encode_low_min, 
		encode_low_max, 
		encode_matrix);
}

#define MAX_LINE 1024

//
// load the stuff from a file
//
UMap::UStatus
UMap::load (const char *nameIn, ifstream* file, const char* filename)
{
	int			sizes[10];
	UParser::UType		types[10];
	void			*values[10];
	char			*names[10];
	UMapCacheItemStruct 	itemToLoad; 
	UParser			*parser;
	int			error;
	int			i;
	UStatus			status;
	unsigned short*		buffer1;
	unsigned short*		buffer2;

	parser = new UParser (file);
	CHECKNULL (parser);

	buffer1 = new unsigned short[65536];
	CHECKNULL (buffer1);
	buffer2 = new unsigned short[65536];
	CHECKNULL (buffer2);

	//
	// Name from file will be safely ignored.
	//
	itemToLoad.name=new char [strlen (nameIn)+1];
	CHECKNULL (itemToLoad.name);
	strcpy (itemToLoad.name, nameIn);

	names[0] = "decode_high_min";
	types[0] = UParser::SHORT;
	sizes[0] = 1;
	values[0] = (void*) &itemToLoad.decodeHighMin;

	names[1] = "decode_high_max";
	types[1] = UParser::SHORT;
	sizes[1] = 1;
	values[1] = (void*) &itemToLoad.decodeHighMax;

	names[2] = "decode_low_min";
	types[2] = UParser::SHORT;
	sizes[2] = 1;
	values[2] = (void*) &itemToLoad.decodeLowMin;

	names[3] = "decode_low_max";
	types[3] = UParser::SHORT;
	sizes[3] = 1;
	values[3] = (void*) &itemToLoad.decodeLowMax;

	itemToLoad.decodeMatrix = buffer1;
	CHECKNULL (itemToLoad.decodeMatrix);

	names[4] = "decode_matrix";
	types[4] = UParser::SHORT_ARRAY;
	sizes[4] = 65536;
	values[4] = (void*) itemToLoad.decodeMatrix;

	names[5] = "encode_high_min";
	types[5] = UParser::SHORT;
	sizes[5] = 1;
	values[5] = (void*) &itemToLoad.encodeHighMin;

	names[6] = "encode_high_max";
	types[6] = UParser::SHORT;
	sizes[6] = 1;
	values[6] = (void*) &itemToLoad.encodeHighMax;

	names[7] = "encode_low_min";
	types[7] = UParser::SHORT;
	sizes[7] = 1;
	values[7] = (void*) &itemToLoad.encodeLowMin;

	names[8] = "encode_low_max";
	types[8] = UParser::SHORT;
	sizes[8] = 1;
	values[8] = (void*) &itemToLoad.encodeLowMax;

	itemToLoad.encodeMatrix = buffer2;
	CHECKNULL (itemToLoad.encodeMatrix);

	names[9] = "encode_matrix";
	types[9] = UParser::SHORT_ARRAY;
	sizes[9] = 65536;
	values[9] = (void*) itemToLoad.encodeMatrix;
	itemToLoad.mapType=UMapDynamic;

	error = parser->parseInput(11, names, types, sizes, values);

	if (error!=0)
	{
		cerr << "error: bad syntax in file '" 
			<< (filename==0?"unkown":filename)
			<< "' at line " << error << "\n";
	}
	delete parser;
	for (i=0; i<10; i++)
	{
		if (sizes[i] == 0) error=1;
	}
	if (error==0 && sizes[4] != (itemToLoad.decodeHighMax - itemToLoad.decodeHighMin +1) 
		* (itemToLoad.decodeLowMax - itemToLoad.decodeLowMin+1))
	{
		cerr << "error: file (decode) '"
			<< (filename==0?"unkown":filename)
			<< "' has corrupted data.\n";
		error=1;
	}
	if (error==0 && sizes[9] != (itemToLoad.encodeHighMax - itemToLoad.encodeHighMin +1) 
		* (itemToLoad.encodeLowMax - itemToLoad.encodeLowMin+1))
	{
		cerr << "error: file (encode) '"
			<< (filename==0?"unkown":filename)
			<< "' has corrupted data.\n";
		error=1;
	}
		
	status=ERROR;
	if (error==0)
	{
		status = load (&itemToLoad);
	}
	delete itemToLoad.name;
	delete buffer1;
	delete buffer2;
	return status;
}

//
// load the stuff from a file
//
UMap::UStatus
UMap::load (const char *name)
{
	char*			pathName;
	char*			fullName;
	ifstream		*file;
	UStatus			status;
	struct stat		statbuf;
	UMapCacheItemStruct	mapStruct;
	unsigned short		start;
	unsigned char*		dstart;
	int			decodeSize;
	int			encodeSize;

	if (strcasecmp (name, "10646")==0)
	{
		// Treated as static
		mapStruct.name = "10646";
		mapStruct.mapType = UMapUnicode;

		mapStruct.decodeHighMin = 0;
		mapStruct.decodeHighMin = 255;
		mapStruct.decodeLowMin = 0;
		mapStruct.decodeLowMax = 255;
		mapStruct.decodeMatrix = 0;

		mapStruct.encodeHighMin = 0;
		mapStruct.encodeHighMin = 255;
		mapStruct.encodeLowMin = 0;
		mapStruct.encodeLowMax = 255;
		mapStruct.encodeMatrix = 0;
		return load (&mapStruct);
	}

	fullName = new char [strlen (name) + strlen (".bumap") +1];
	CHECKNULL (fullName);

	strcpy (fullName, name);
	strcat (fullName, ".bumap");
	pathName = UFindFile (UGetMapPath (), fullName);

#ifdef HAVE_MMAP
	if (pathName !=0 && stat ((const char*) pathName, &statbuf) != -1
		&& (mapStruct.fd=open ((const char*) pathName, O_RDONLY)) != -1
		&& statbuf.st_size > 50)
	{
		mapStruct.buflen = statbuf.st_size;
		mapStruct.buf=(unsigned char*) mmap (0, mapStruct.buflen,  
			PROT_READ, MAP_PRIVATE, mapStruct.fd, 0);

		if (mapStruct.buf!=0 
			&& strncmp ((const char*)mapStruct.buf, 
				"YUDIT-UMAP", strlen ("YUDIT-UMAP"))==0)
		{
			start = UGS_AT (mapStruct.buf, 48);
			if (mapStruct.buflen < start + 16)
			{
				cerr << "error: corrupted umap '" <<
					fullName << "'\n";
				munmap ((char*) mapStruct.buf, 
					mapStruct.buflen);
				close (mapStruct.fd);
				delete pathName;
				delete fullName;
				return ERROR;
			}
			mapStruct.mapType = UMapMMap;
			mapStruct.decodeHighMin = UGS_AT (mapStruct.buf, start);
			mapStruct.decodeHighMax = UGS_AT (mapStruct.buf, start+2);
			mapStruct.decodeLowMin = UGS_AT (mapStruct.buf, start+4);
			mapStruct.decodeLowMax = UGS_AT (mapStruct.buf, start+6);
			mapStruct.encodeHighMin = UGS_AT (mapStruct.buf, start+8);
			mapStruct.encodeHighMax = UGS_AT (mapStruct.buf, start+10);
			mapStruct.encodeLowMin = UGS_AT (mapStruct.buf, start+12);
			mapStruct.encodeLowMax = UGS_AT (mapStruct.buf, start+14);
			dstart = &mapStruct.buf[start+16];
			mapStruct.decodeMatrix = (UCS2*) dstart;
			decodeSize = (mapStruct.decodeLowMax 
				- mapStruct.decodeLowMin +1)
				* (mapStruct.decodeHighMax 
					- mapStruct.decodeHighMin +1);
			encodeSize = (mapStruct.encodeLowMax 
				- mapStruct.encodeLowMin +1)
				* (mapStruct.encodeHighMax 
					- mapStruct.encodeHighMin +1);
			if (mapStruct.buflen < start + 16 + 2 * decodeSize + 2 * encodeSize)
			{
				cerr << "error: corrupted umap '" <<
					fullName << "'\n";
				munmap ((char*) mapStruct.buf, 
					mapStruct.buflen);
				delete pathName;
				delete fullName;
				return ERROR;
			}

			dstart = dstart + decodeSize + decodeSize;
			mapStruct.encodeMatrix = (UCS2*) dstart;
			mapStruct.name = new char [strlen(name) +1];
			CHECKNULL (mapStruct.name);
			strcpy (mapStruct.name, name);
			// The allocated values will be copied. No free needed
			delete pathName;
			delete fullName;
			return load (&mapStruct);
		}
		else
		{
			cerr << "error: could not use binary umap '" <<
					fullName << "'\nTrying ascii umap.\n";
			if (mapStruct.buf!=0)
			{
				munmap ((char*) mapStruct.buf, 
					mapStruct.buflen);
			}
			delete pathName;
			delete fullName;
			close (mapStruct.fd);
		}
	}
#else
	if (pathName!=0)
	{
		cerr << "error: mmap not supported by this binary for umap '" <<
				fullName << "'\nTrying ascii umap.\n";
	}
#endif
	strcpy (fullName, name);
	strcat (fullName, ".umap");
	pathName = UFindFile (UGetMapPath (), fullName);

	if (pathName==0) return ERROR;
	file = new ifstream (pathName);
	CHECKNULL (file);

	if (file->rdbuf() == 0 || !file->rdbuf()->is_open())
	{
		delete pathName;
		delete fullName;
		delete file;
		return ERROR;
	}

	status = load (name, file, pathName);
	delete file;
	delete pathName;
	delete fullName;
	return status;
}

//
// here we are
//
UMap::UStatus
UMap::load (
		char	*nameIn,
		UMapType mapType,
		UCS2	decode_high_min, 
		UCS2	decode_high_max, 
		UCS2	decode_low_min, 
		UCS2	decode_low_max, 
		UCS2*	decode_matrix, 

		UCS2	encode_high_min, 
		UCS2	encode_high_max, 
		UCS2	encode_low_min, 
		UCS2	encode_low_max, 
		UCS2*	encode_matrix)
{
	UMapCacheItemStruct		cacheItemL;

	cacheItemL.name = nameIn;
	cacheItemL.mapType = mapType;
	cacheItemL.decodeHighMin = decode_high_min;
	cacheItemL.decodeHighMax = decode_high_max;
	cacheItemL.decodeLowMin = decode_low_min;
	cacheItemL.decodeLowMax = decode_low_max;
	cacheItemL.decodeMatrix = decode_matrix;

	cacheItemL.encodeHighMin = encode_high_min;
	cacheItemL.encodeHighMax = encode_high_max;
	cacheItemL.encodeLowMin = encode_low_min;
	cacheItemL.encodeLowMax = encode_low_max;
	cacheItemL.encodeMatrix = encode_matrix;
	return load (&cacheItemL);
}

//
// Load from structure. The structure itself will not be used. The
// items inside will be used only if isStatic is set. This is good
// If we want to initialize big structures as default.
//
UMap::UStatus
UMap::load (UMapCacheItemStruct *cacheItemIn)
{
	UMapCacheItem				*item;
	UMapCacheItem				*gotItem;
	UCache<UMapCacheItem*>::UStatus		ret_vle;

	if (cacheItem != 0)
	{
		cache->unuseItem (cacheItem->name);
		cacheItem = 0;
	}
	item = new UMapCacheItem(cacheItemIn);
	CHECKNULL (item);
	ret_vle = cache->addItem (cacheItemIn->name, item);
	gotItem = cache->getItem (cacheItemIn->name);
	if (gotItem != 0)
	{
		cacheItem = gotItem->cacheItem;
	}
	switch (ret_vle)
	{
	case UCache<UMapCacheItem*>::DUPLICATE:
		delete item;
		cache->useItem (cacheItem->name);
	case UCache<UMapCacheItem*>::OK:
		return OK;
	case UCache<UMapCacheItem*>::ERROR:
		rename (DEFAULT_NAME);
		return ERROR;
	}
	return ERROR;
}

//
// Get the name.
//
char *
UMap::getName ()
{
	if (cacheItem == 0) return 0;
	return cacheItem->name;
}

extern UMap*		umapDefault;

void
UTuneUMap (int cacheSize, int batchSize)
{
	if (cache)
	{
		cache->tune (cacheSize, batchSize);
	}
}

void l_initUMap()
{
	UInitUMap ();
	UTuneUMap (0);
}

void UDeinitUMap ()
{
	if (umapDefault != 0) delete (umapDefault);
	umapDefault = 0;
	delete cache;
	cache = 0;
}


