//  UFileDialog.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#define CMARGIN 4
#define MARGIN 10
#define BUTTON_MARGIN 20
#define MAX_DIR_LENGTH          4096

#include "UFileDialog.h"
#include "UTextDialog.h"
#include "UTextButton.h"
#include "UTextLabel.h"
#include "UTextMenu.h"
#include "UTextFrame.h"
#include "UButton.h"
#include "UChoice.h"
#include "UScrolledList.h"
#include "strings.h"
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

static AString initialDir;
static int showHidden = 0;

UFileDialog::UFileDialog (UShell* transient_, 
	UFileDialogType type_, int frameSize_) :
	UDialog (transient_, frameSize_)
{
	type = type_;
	setUp();
}

UFileDialog::UFileDialog (UTop* top_, 
	UFileDialogType type_, int frameSize_) :
	UDialog (top_, frameSize_)
{
	type = type_;
	setUp ();
}

void 
UFileDialog::setUp()
{
	UColor	white (top->display);

	white = "#FFFFFF";

	char 	currDir[MAX_DIR_LENGTH+1];
	if (initialDir.isNull())
	{
		if (getcwd(currDir, MAX_DIR_LENGTH)==0)
		{
			cerr << "warn: can not get current directory.";
			strcpy (currDir, "/");
		}
		currDir[MAX_DIR_LENGTH]=0;
		initialDir = (const char*) currDir;
	}
	workingDir = (const char*) initialDir;

	component[OK] = new UTextButton (this, "OK");
	component[CD] = new UTextButton (this, "CD");
	component[CANCEL] = new UTextButton (this, "CANCEL");
	lastButton = CANCEL;

	component[FILE_LABEL] = new UTextLabel (this, "File:", UTextLine::LEFT);
	component[FILE_TEXT] = new UTextFrame (this);
	((UTextFrame*)component[FILE_TEXT])->uSetBackground (white.getPixel());
	((UTextFrame*)component[FILE_TEXT])->uSetMultiLine (UWidget::SINGLE);

	component[FILE_LIST] = new UScrolledList (this);
	component[DIR_LIST] = new UScrolledList (this);
	component[FILES_LABEL] = new UTextLabel (this, "Files:");
	component[DIRS_LABEL] = new UTextLabel (this, "Directories:");
	component[TOGGLE] = new UButton (this);
	component[TOGGLE_TEXT] = new UTextMenu ((UButton*)component[TOGGLE], 
		"Hidden", "Hidden", UMenu::CHECK);
	component[TOGGLE_TEXT]->select (ExposureMask 
		| ButtonPressMask |ButtonReleaseMask| 
		Button1MotionMask | LeaveWindowMask | EnterWindowMask);
	component[CHOICE] = new UChoice (this);
	toggleMenu = (UMenuItem*) component[TOGGLE_TEXT]->getMenu ();
	if (showHidden) toggleMenu->check();
	((UTextFrame*)component[FILE_TEXT])->uSetConverter ("UTF8");
	rectangle.width = 20;
	rectangle.height = 20;
	existsDialog = new UTextDialog (this, UTextDialog::BOK);
}

UFileDialog::~UFileDialog ()
{
	// autodelete
	delete existsDialog;
}

int
UFileDialog::setEncoding (const char* enco_)
{
	if (((UTextFrame*)component[FILE_TEXT])->uSetConverter (enco_)
		== UString::ERROR)
	{
		return 0;
	}
	return 1;
}

int
UFileDialog::setInput (const char* input_)
{
	if (strcasecmp (input_, "Straight") ==0
		|| strcasecmp (input_, "None")==0)
	{
		((UTextFrame*)component[FILE_TEXT])->uSetInput (0);
		return 1;
	}
	if (((UTextFrame*)component[FILE_TEXT])->uSetInput (input_)
		== UWidget::ERROR)
	{
		return 0;
	}
	return 1;
}

void
UFileDialog::resize (int width, int height)
{
	UBestSize	bs; 
	UPlacement	pl;
	int		diff;

	// make a little correction to make it look cooler
	bs = component[CHOICE]->getBestSize();
	component[CHOICE]->place(width/2 , MARGIN,
		-(bs.width+width/2), -bs.height-MARGIN);
	pl = component[CD]->getPlacement();
	
	diff = -(pl.edges[UPlacement::WEST]+pl.edges[UPlacement::EAST]);
	component[CD]->place (UPlacement::WEST, (width - diff)/2);
	component[CD]->place (UPlacement::EAST, - (width+diff)/2);
	UFrame::resize (width, height);
}

#define MAXVLE(_a, _b) ((_a>_b)?_a:_b)

const UBestSize&
UFileDialog::getBestSize ()
{
	UBestSize	bs[COMP_MAX];
	int		heights[5];
	int		halfWidth;
	int		acctY;
	int		i;

	rescan ();
	for (i=0; i<(int)COMP_MAX; i++)
	{
		// toggle text is handled by toggle
		if (i==TOGGLE_TEXT) continue;
		switch ((USubComponent)i)
		{
		case FILE_LABEL:
		case DIRS_LABEL:
		case FILES_LABEL:
		case TOGGLE:
			component[i]->setMargin (CMARGIN);
			bs[i] = component[i]->getBestSize ();
			break;
		case OK:
		case CANCEL:
		case CD:
			component[i]->setMargin (CMARGIN);
			bs[i] = component[i]->getBestSize ();
			bs[i].width +=  BUTTON_MARGIN;
			break;
		case FILE_TEXT:
			((UTextFrame*)component[i])->setMargin (CMARGIN);
			bs[i] = ((UTextFrame*)component[i])->getBestSize ();
			break;
		default:
			component[i]->setMargin (CMARGIN);
			bs[i] = component[i]->getBestSize ();
			break;
		}
	}

	bs[OK].width = MAXVLE (bs[OK].width, bs[CD].width);
	bs[OK].width = MAXVLE (bs[OK].width, bs[CANCEL].width);
	bs[CD].width = bs[OK].width;
	bs[CANCEL].width = bs[OK].width;
	for (i=0; i<5; i++) heights[i] = 0;
	halfWidth = 0;
	// row 0
	heights[0] = MAXVLE (bs[OK].height, bs[CD].height);
	heights[0] = MAXVLE (heights[0], bs[CANCEL].height);
	halfWidth = MAXVLE (bs[OK].width + bs[CD].width/2,
			bs[CANCEL].width + bs[CD].width/2);
	// row 1
	heights[1] = MAXVLE (bs[FILE_LABEL].height, 
		bs[FILE_TEXT].height);
	halfWidth = MAXVLE (halfWidth, (bs[FILE_TEXT].width
		+ bs[FILE_LABEL].width)/2);

	// row 2
	heights[2] = MAXVLE (bs[FILE_LIST].height, 
			bs[DIR_LIST].height);
	halfWidth = MAXVLE (halfWidth, bs[FILE_LIST].width);
	halfWidth = MAXVLE (halfWidth, bs[DIR_LIST].width);

	// row 3
	heights[3] = MAXVLE (bs[FILES_LABEL].height, 
			bs[DIRS_LABEL].height);
	halfWidth = MAXVLE (halfWidth, bs[FILES_LABEL].width);
	halfWidth = MAXVLE (halfWidth, bs[DIRS_LABEL].width);

	// row 4
	heights[4] = MAXVLE (bs[TOGGLE].height, 
			bs[CHOICE].height);
	halfWidth = MAXVLE (halfWidth, bs[TOGGLE].width);
	halfWidth = MAXVLE (halfWidth, bs[CHOICE].width);

	halfWidth += MARGIN;

	// Place component here.
	// Toggle and choice
	acctY=MARGIN;
	component[TOGGLE]->place(MARGIN, acctY, 
		-bs[TOGGLE].width-MARGIN, -heights[4]-acctY);

	component[CHOICE]->place(halfWidth, acctY, 
		-(bs[CHOICE].width+halfWidth), -heights[4]-acctY);

	acctY += heights[4];
	component[DIRS_LABEL]->place (UPlacement::NORTH, acctY); 
	component[DIRS_LABEL]->place (UPlacement::SOUTH, -heights[3]-acctY); 
	component[FILES_LABEL]->place (UPlacement::NORTH, acctY); 
	component[FILES_LABEL]->place (UPlacement::SOUTH, -heights[3]-acctY); 
	component[DIRS_LABEL]->place (UPlacement::WEST, MARGIN); 
	component[DIRS_LABEL]->position (UPlacement::EAST, 128); 
	component[FILES_LABEL]->position (UPlacement::WEST, 128); 
	component[FILES_LABEL]->place (UPlacement::EAST, MARGIN); 
	acctY += heights[3];
	component[DIR_LIST]->place (UPlacement::NORTH, acctY-CMARGIN); 
	component[FILE_LIST]->place (UPlacement::NORTH, acctY-CMARGIN); 

	component[FILE_LIST]->place (UPlacement::EAST, MARGIN); 
	component[FILE_LIST]->position (UPlacement::WEST, 128); 

	component[DIR_LIST]->position (UPlacement::EAST, 128); 
	component[DIR_LIST]->place (UPlacement::WEST, MARGIN); 

	bestSize.width = halfWidth*2; 
	bestSize.height=2*MARGIN;
	for (i=0; i<5; i++) bestSize.height+=heights[i];
	acctY = MARGIN;
	component[CANCEL]->place (-bs[CANCEL].width-MARGIN, -heights[0]-acctY, 
			MARGIN, acctY);
	component[CD]->place (halfWidth-bs[CD].width/2, 
		-heights[0]-acctY, -halfWidth-bs[CD].width/2, acctY);
	component[OK]->place (MARGIN, -heights[0]-acctY, -bs[OK].width-MARGIN, acctY);
	acctY += heights[0];
	((UTextFrame*)component[FILE_TEXT])->place (bs[FILE_LABEL].width+MARGIN, -heights[1]-acctY, MARGIN, acctY);
	component[FILE_LABEL]->place (MARGIN, -heights[1]-acctY, -bs[FILE_LABEL].width-MARGIN, acctY);
	acctY += heights[1];
	component[DIR_LIST]->place (UPlacement::SOUTH, acctY); 
	component[FILE_LIST]->place (UPlacement::SOUTH, acctY); 
	minimumWidth = bestSize.width;
	minimumHeight = bestSize.height;
	if (rectangle.width > bestSize.width) bestSize.width = rectangle.width;
	if (rectangle.height > bestSize.height) bestSize.height = rectangle.height;

	return bestSize;
}

UDialog::UValue
UFileDialog::getValue()
{
	lastButton = CANCEL;
	((UTextFrame*)component[FILE_TEXT])->setFocus();
	UDialog::getValue();
	if (lastButton==OK) return (UDialog::OK);
	return  (UDialog::CANCEL);
}

void
UFileDialog::setButtonFont (UFont* font_)
{
	int i;
	for (i=0; i<(int)COMP_MAX; i++)
	{
		switch ((USubComponent)i)
		{
		case CANCEL:
		case OK:
		case CD:
		case FILE_LABEL:
		case FILES_LABEL:
		case DIRS_LABEL:
		case TOGGLE_TEXT:
		case CHOICE:
			component[i]->setFont (font_);
			break;
		default:
			break;
		}
	}
	existsDialog->setButtonFont (font_);
}

void
UFileDialog::setTextFont (UFont* font_)
{
	int i;
	for (i=0; i<(int)COMP_MAX; i++)
	{
		switch ((USubComponent)i)
		{
		case DIR_LIST:
		case FILE_LIST:
			component[i]->setFont (font_);
			break;
		case FILE_TEXT:
			component[i]->setFont (font_);
			((UTextFrame*)component[i])->uRefresh();
			break;
		default:
			break;
		}
	}
	existsDialog->setTextFont (font_);
}

void
UFileDialog::setText (USubComponent comp_, const char* utf8_)
{
	switch (comp_)
	{
	case CANCEL:
		((UTextButton*) component[comp_])->setText (utf8_);
		break;
	case OK:
		((UTextButton*) component[comp_])->setText (utf8_);
		break;
	case CD:
		((UTextButton*) component[comp_])->setText (utf8_);
		break;
	case FILE_LABEL:
	case FILES_LABEL:
	case DIRS_LABEL:
	case TOGGLE_TEXT:
		((UTextLabel*) component[comp_])->setText (utf8_);
		break;
	default:
		break;
	}
}

void
UFileDialog::setExists (const char* utf8_)
{
	existsDialog->setText (utf8_);
}

void
UFileDialog::setExistsNo (const char* utf8_)
{
	existsDialog->setButtonText (UTextDialog::BCANCEL, utf8_);
}
void
UFileDialog::setExistsYes (const char* utf8_)
{
	existsDialog->setButtonText (UTextDialog::BOK, utf8_);
}

void
UFileDialog::rescan ()
{
	UPopUp*		pop=0;
	char* 		newDir;
	int		i;
	int		len;
	int		start;

	redoList ();
	// The old one is automatically deleted
	len = strlen ((const char*) workingDir);
	newDir = new char[len+2];
	strcpy (newDir, (const char*) workingDir);
	if (len==0 || newDir[len-1] != '/')
	{
		workingDir += "/";
		newDir[len] = '/';
		newDir[len+1] = 0;
		len++;
	}
	start = 1;
	for (i=len-1; i>=0; i--)
	{
		if (newDir[i]!='/' || i==len-1) continue;
		// sort out double //
		if (newDir[i+1]=='/')
		{
			newDir[i+1]=0;
			continue;
		}
		if (start)
		{
			pop = new UPopUp (((UChoice*)component[CHOICE]),
			 &newDir[i+1], &newDir[i+1], 
			 UMenu::NO_LEFT, UMenu::DOWN);
			start= 0;
		}
		(void) new UTextMenu (pop, &newDir[i+1], &newDir[i+1],
			UMenu::NO_LEFT, UMenu::ARROW_BLANK);
		newDir[i+1]=0;
		
	}
	if (start)
	{
		pop = new UPopUp (((UChoice*)component[CHOICE]),
			  "/", "/", UMenu::NO_LEFT, UMenu::DOWN);
	}
	(void) new UTextMenu (pop, "/", "/", UMenu::NO_LEFT, UMenu::ARROW_BLANK);
	//((UChoice*)component[CHOICE])->getBestSize();
	pop->packItems ();
	pop->pack ();
	
}

void
UFileDialog::redoList ()
{
	const char* 	current;
	DIR*		dir;
	struct dirent*	entry;
	struct stat	buf;
	AString		pathname;

	current = (const char*) workingDir;
	//
	// Delete list here
	//
	fileList.clear();
	dirList.clear();
	dir = 0;
	if (stat (current, &buf) != 0 || (dir=opendir (current))==0)
	{
		cerr << "yudit: Can not read '"
			<< current << "' directory.\n";
		workingDir = (const char*) "/";
		current = (const char*) "/";
		dir=opendir (current);
	}
	if (dir==0) return;
	// read the directory and fill in the list.
	while ((entry=readdir (dir))!=0)
	{
		pathname = (const char*) current;
		if (strcmp ((const char*) current, "/"))
		{
			pathname += "/";
		}
		pathname += entry->d_name;
		if (stat ((const char*) pathname, &buf) != 0
			&& lstat ((const char*) pathname, &buf) != 0)
		{
			cerr << "Yudit: can not find '" << (const char*) pathname << "'\n";
			continue;
		}
		if (!toggleMenu->isChecked() && entry->d_name[0] == '.') continue;
		if (S_ISDIR( buf.st_mode))
		{
			if (strcmp (entry->d_name, "..") == 0) continue;
			if (strcmp (entry->d_name, ".") == 0) continue;
			dirList.addCaseSorted ((const char*) entry->d_name);
		}
		else if (S_ISREG (buf.st_mode) || S_ISLNK (buf.st_mode))
		{
			fileList.addCaseSorted ((const char*) entry->d_name);
		}
	}
	closedir (dir);
	initialDir = (const char*) workingDir;
	pathname = (const char*) current;
	if (strcmp (pathname, "/")!=0)
	{
		dirList.push ("..");
	}
	((UScrolledList*) component[DIR_LIST])->replace (dirList);
	((UScrolledList*) component[FILE_LIST])->replace (fileList);
}

void	
UFileDialog::eventUp (UEvent* event)
{
	UPopUp* 	popUp;
	AString		newDir;
	UTextMenu*	textMenu;
	UBestSize	bs;
	char*		str;
	const char*		cstr;
	int		i;
	int		len;
	const UCS2* const*	ucs2;
	const int*	textSizes;
	UCS2*		ustringVle;
	UString		ustr;
	ifstream*	ift;

	switch (event->type)
	{
	case UEvent::SCROLL_RANGE_HORIZONTAL:
		if (event->client !=0 && event->client==component[FILE_TEXT])
		{
			((UTextFrame*)component[FILE_TEXT])->uHScroll (event->value);
		}
		break;
	case UEvent::ACTIVATED:
		if (event->client !=0 && event->client==component[CANCEL])
		{
			lastButton = CANCEL;
			hide ();
			return;
		}
		if (event->client !=0 && event->client==component[OK])
		{
			selected = workingDir;
			ucs2 = ((UTextFrame*)component[FILE_TEXT])->uGetTextBuffer();
			textSizes = ((UTextFrame*)component[FILE_TEXT])->uGetTextSizes();
			if (ucs2==0 || textSizes[0] ==0) return;
			if (UCS2NullLen(ucs2[0]) != textSizes[0])
			{
				cerr << "text contains null (^@) character.\n";
				return;
			}
			len = strlen ((const char*)selected);
			// chop off '/'
			if (len > 0) selected.truncate (len-1);
			ustr.setConverter("UTF8");
			ustr.putUString (ucs2[0], UCS2NullLen(ucs2[0]));
			str = (char*) ustr.getString(&len);
			if (str[0]!='/') selected += "/";
			selected += str;
			delete str;

			// try open
			ift = new ifstream ((const char*) selected);
			if (type==LOAD)
			if (ift->rdbuf() == 0 || !ift->rdbuf()->is_open())
			{
				cerr << "warn: file '" <<
					(const char*) selected
					<< "' is not readable.\n";
				delete ift;
				return;
			}
			if (type==SAVE && ift->rdbuf() != 0 
					&& ift->rdbuf()->is_open())
			{
				delete ift;
				if (existsDialog->getValue()!=UDialog::OK)
				{
					return;
				}
			}
			else
			{
				delete ift;
			}
			lastButton = OK;
			hide ();
			return;
		}
		if (event->client !=0 && event->client==component[CD])
		{
			ucs2 = ((UTextFrame*)component[FILE_TEXT])->uGetTextBuffer();
			textSizes = ((UTextFrame*)component[FILE_TEXT])->uGetTextSizes();
			if (ucs2==0 || textSizes[0] ==0) return;
			if (UCS2NullLen(ucs2[0]) != textSizes[0])
			{
				cerr << "text contains null (^@) character.\n";
				return;
			}
			
			ustr.setConverter("UTF8");
			ustr.putUString (ucs2[0], UCS2NullLen(ucs2[0]));
			str = (char*) ustr.getString(&len);
			if (cdTo ((const char*)str))
			{
				((UTextFrame*)component[FILE_TEXT])->uClear ();
				((UTextFrame*)component[FILE_TEXT])->uRefresh ();
			}
			delete str;
			return;
		}
		if (event->client !=0 && event->client==component[TOGGLE])
		{
			if (toggleMenu->isChecked())
			{
				toggleMenu->uncheck ();
				redoList ();
			}
			else
			{
				toggleMenu->check ();
				redoList ();
			}
			return;
		}
		break;
	case UEvent::SELECTED:
		if (event->client == component[FILE_LIST])
		{
			((UTextFrame*)component[FILE_TEXT])->uClear ();
			cstr = fileList.at (event->value);
				
			UString str ((const unsigned char*) cstr, "UTF8");
			ustringVle = str.getUString (&len);
			((UTextFrame*)component[FILE_TEXT])->uInsertUCS2(
					ustringVle, len);
			delete ustringVle;
			((UTextFrame*)component[FILE_TEXT])->uRefresh ();
			((UTextFrame*)component[FILE_TEXT])->uClearModified();
		}
		if (event->client == component[DIR_LIST])
		{
			if (event->value==0 && strcmp (workingDir, "/"))
			{
				str = new char[
					strlen ((const char*) workingDir) +1];
				strcpy (str, (const char*) workingDir);
				for (i=strlen(str)-2; i>=0; i--)
				{
					if (str[i]=='/')
					{
						str[i+1]=0;
						break;
					}
				}
				workingDir = str;
				delete str;
				rescan ();
				bs = ((UChoice*)component[CHOICE])->getBestSize();
				component[CHOICE]->resize (bs.width-2*CMARGIN, 
				component[CHOICE]->rectangle.height);
				break;
			}
			else
			{
				cdTo (dirList.at(event->value));
			}
		}
		break;
	case UEvent::MENU_SELECTION:
		if (event->client != component[CHOICE]) break;
		popUp = (UPopUp*) event->popUp;
		for (i=popUp->getSize()-1; i>=0; i--)
		{
			textMenu = (UTextMenu*) popUp->itemAt (i);
			newDir += textMenu->getName();
			if (event->value==i) break;
		}
		workingDir = newDir;
		rescan ();
		//component[CHOICE]->setFont (buttonFont);
		bs = ((UChoice*)component[CHOICE])->getBestSize();
		component[CHOICE]->resize (bs.width-2*CMARGIN, 
			component[CHOICE]->rectangle.height);
		//((UChoice*)component[CHOICE])->pack();
		break;
	default:
		UShell::eventUp (event);
		break;
	}
}

int
UFileDialog::cdTo (const char* dir)
{
	char*		newDir;
	int		i, j, k;
	int		len;
	char 		prev;
	struct stat	buf;
	DIR*		d;
	AString		tryDir;
	UBestSize	bs;
	
	len = strlen(dir);
	newDir = new char [len+3];
	prev='/';
	newDir[0] = '/';
	j=1;
	d = 0;
	for (i=0; i<len; i++)
	{
		if (prev=='/' && dir[i]=='/') continue;
		prev = dir[i];
		newDir[j]=dir[i];
		j++;
	}
	if (prev!='/') newDir[j++] = '/';
	newDir[j] = 0;
	// now we have a slash and duplicate slash free directory


	// absolute
	if (dir[0] == '/')
	{
		if (stat (dir, &buf) != 0 || (d=opendir (dir))==0)
		{
			cerr << "warn: can not cd to '" << dir << "'\n";
			delete newDir;
			return 0;
		}
		closedir (d);
		workingDir = newDir;
		rescan ();
		bs = ((UChoice*)component[CHOICE])->getBestSize();
		component[CHOICE]->resize (bs.width-2*CMARGIN, 
			component[CHOICE]->rectangle.height);
		delete newDir;
		return 1;
	}

	// relative. calculate relative. well.... to an extent...
	j=1;
	len = strlen (newDir);
	tryDir = (const char*) workingDir;
	for (i=1; i<len; i++)
	{
		if (newDir[i]=='/')
		{
			newDir[i]=0;
			if (strcmp (&newDir[j], "..")==0)
			{
				// up
				if (strcmp ((const char*) tryDir, "/")==0)
				{
					cerr << "warn: can not cd to '" << dir << "'\n";
					delete newDir;
					return 0;
				}
				// chop off last part
				for (k=strlen ((const char*) tryDir)-2;
					k>=0; k--)
				{
					if (((const char*) tryDir)[k]=='/')
					{
						tryDir.truncate (k+1);
						break;
					}
				}
			}
			else if (strcmp (&newDir[j], ".")==0)
			{
				// current
			}
			else
			{
				tryDir += &newDir[j];
				tryDir += "/";
			}
			j=i+1;
		}
	}

	if (stat ((const char*) tryDir, &buf) != 0 
		|| (d=opendir ((const char*) tryDir))==0)
	{
		cerr << "warn: can not cd to '" << dir << "'\n";
		delete newDir;
		return 0;
	}
	closedir (d);
	workingDir = (const char*) tryDir;
	rescan ();
	bs = ((UChoice*)component[CHOICE])->getBestSize();
	component[CHOICE]->resize (bs.width-2*CMARGIN, 
		component[CHOICE]->rectangle.height);
	delete newDir;
	return 1;
}

void
UFileDialog::passEvent (UEvent* evnt)
{
	UDialog::passEvent (evnt);
	if (((UTextFrame*)component[FILE_TEXT])->uIsModified())
	{
		((UScrolledList*)component[FILE_LIST])->unselect();
		((UTextFrame*)component[FILE_TEXT])->uClearModified();
	}
}

int
UFileDialog::setXInput (const char* name_, const char* imIn,
	const char* inputStyleIn, const char* textType,
	const char* uStringIn, const char* fontMap, double versionIn)
{
	((UTextFrame*)component[FILE_TEXT])->setXInput (name_, imIn,
		inputStyleIn, textType, uStringIn, fontMap, versionIn);
	return 0;
}

void
UFileDialog::setTitle (const char* title_)
{
	existsDialog->setTitle (title_);
	UShell::setTitle (title_);
}
