/************************************************************************/
/*  Management of X11 fonts: Translation of understandable fonts to	*/
/*  X11 structures.							*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<debugon.h>

#   include	<X11/Xatom.h>

#   include	"appFont.h"
#   include	"appDraw.h"

/************************************************************************/
/*  Strings used for the analysis of X11 font names.			*/
/************************************************************************/
static const char *	EditFontAnyTemplate= "*";

# define	EditFontFamilyTemplate \
			    "-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s"

typedef struct EditFontEncoding
    {
    char *	efeRegistry;
    char *	efeEncoding;
    } EditFontEncoding;

static EditFontEncoding	EditFontXEncodings[]=
    {
	{ "iso8859",	"1"		},	/*  For Text	*/
	{ "iso8859",	"2"		},	/*  For Text	*/
	{ "adobe",	"fontspecific"	},	/*  Symbol.	*/
    };

/************************************************************************/
/*  Make a font query string.						*/
/************************************************************************/
static void appFontQueryString(	char *		target,
				const char *	foundry,
				const char *	family,
				const char *	weight,
				const char *	slant,
				const char *	setWidth,
				const char *	pxlsz,
				const char *	registry,
				const char *	encoding	)
    {
    sprintf( target, EditFontFamilyTemplate,
		foundry,
		family,
		weight,
		slant,
		setWidth,		/* swdth		*/
		EditFontAnyTemplate,	/* adstyl		*/
		pxlsz,
		EditFontAnyTemplate,	/* ptsz			*/
		EditFontAnyTemplate,	/* resx			*/
		EditFontAnyTemplate,	/* resy			*/
		EditFontAnyTemplate,	/* spacing		*/
		EditFontAnyTemplate,	/* average width	*/
		registry,
		encoding );

    return;
    }

/************************************************************************/
/*  Extract properties from an X font selection string.			*/
/************************************************************************/
static int appFontExtractString(	const char *	s,
					const char **	pPast,
					char *		target )
    {
    if  ( *s != '-' )
	{ SDEB(s); return -1;	}
    s++;
    while( *s && *s != '-' )
	{ *(target++)= *(s++); }
    *target= '\0';

    *pPast= s; return 0;
    }

static int appFontSkipString(	const char *	s,
				const char **	pPast )
    {
    if  ( *s != '-' )
	{ SDEB(s); return -1;	}
    s++;
    while( *s && *s != '-' )
	{ s++; }

    *pPast= s; return 0;
    }

static int appFontSplitXFont(	const char *	fontname,
				char *		foundry,
				char *		family,
				char *		weight,
				char *		slant,
				char *		pxlszStr,
				char *		spacing,
				char *		avgwdth,
				char *		registry,
				char *		encoding )
    {
    const char *	s;

    s= fontname;

    if  ( appFontExtractString( s, &s, foundry ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, family ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, weight ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, slant ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontSkipString( s, &s ) ) /* swdth */
	{ SDEB(fontname); return -1;	}

    if  ( appFontSkipString( s, &s ) ) /* adstyl */
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, pxlszStr ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontSkipString( s, &s ) ) /* ptsz */
	{ SDEB(fontname); return -1;	}

    if  ( appFontSkipString( s, &s ) ) /* resx */
	{ SDEB(fontname); return -1;	}

    if  ( appFontSkipString( s, &s ) ) /* resy */
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, spacing ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, avgwdth ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, registry ) )
	{ SDEB(fontname); return -1;	}

    if  ( appFontExtractString( s, &s, encoding ) )
	{ SDEB(fontname); return -1;	}

    if  ( s[0] )
	{ SSDEB(fontname,s); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make a catalog of X11 fonts.					*/
/*									*/
/************************************************************************/
static int appFontXCatalog(	Display *		display,
				const AppFontFamily **	pFamilies,
				int *			pCount		)
    {
    char	familyFormat[120];
	
    char **	fontList;
    int		fontCount;

    int		i;
    int		j;
    unsigned	enc;

    AppFontFamily *		aff;
    AppFontTypeface *		aft;

    static AppFontFamily *	EditFontXFamilies= (AppFontFamily *)0;
    static int			EditFontXFamilyCount= 0;

    if  ( EditFontXFamilyCount > 0 )
	{
	*pFamilies= EditFontXFamilies;
	*pCount= EditFontXFamilyCount;

	return 0;
	}

    for ( enc= 0;
	  enc < sizeof(EditFontXEncodings)/sizeof(EditFontEncoding);
	  enc++ )
	{
	appFontQueryString( familyFormat,
		    EditFontAnyTemplate,	/* foundry,		*/
		    EditFontAnyTemplate,	/* family,		*/
		    EditFontAnyTemplate,	/* weight,		*/
		    EditFontAnyTemplate,	/* slant,		*/
		    EditFontAnyTemplate,	/* setWidth,		*/
		    EditFontAnyTemplate,	/* pxlsz,		*/
		    EditFontXEncodings[enc].efeRegistry,
		    EditFontXEncodings[enc].efeEncoding	);

	fontList= XListFonts( display, familyFormat, 4000, &fontCount );

	if  ( fontCount == 4000 )
	    { LLDEB(fontCount,4000);	}

	for ( i= 0; i < fontCount; i++ )
	    {
	    char			foundry[40];
	    char			family[40];
	    char			familyName[60];

	    char			weight[40];
	    char			slant[40];
	    char			faceName[60];

	    char			ignore[40];
	    char			spacing[40];
	    char			avgwdth[40];

	    char			pxlszStr[40];
	    int *			sizes;
	    int				pxlsz;

	    if  ( appFontSplitXFont(	fontList[i],
					foundry,
					family,
					weight,
					slant,
					pxlszStr,
					spacing,
					avgwdth,
					ignore,		/*  registry  */
					ignore ) )	/*  encoding  */
		{ LSDEB(i,fontList[i]); continue;	}

	    if  ( strcmp( spacing, "m" ) && strcmp( spacing, "p" ) )
		{
		if  ( strcmp( spacing, "c" ) )
		    { SDEB(spacing);	}

		continue;
		}

	    sprintf( familyName, "%s-%s", foundry, family );
	    sprintf( faceName, "%s-%s", weight, slant );

	    aff= EditFontXFamilies;
	    for ( j= 0; j < EditFontXFamilyCount; aff++, j++ )
		{
		if  ( ! strcmp( aff->affFontFamilyName, familyName )	&&
		      aff->affEncoding == enc				)
		    { break;	}
		}

	    if  ( j >= EditFontXFamilyCount )
		{
		aff= (AppFontFamily *)realloc( EditFontXFamilies,
					    (j+1)* sizeof(AppFontFamily) );
		if  ( ! aff )
		    { LXDEB(j,aff); return -1;	}
		EditFontXFamilies= aff;

		aff += j;

		docInitFontFamily( aff );

		aff->affFontFamilyName= strdup( familyName );
		if  ( ! aff->affFontFamilyName )
		    { XDEB(aff->affFontFamilyName); return -1;	}

		aff->affEncoding= enc;

		EditFontXFamilyCount= j+ 1;
		}

	    aft= aff->affFaces;
	    for ( j= 0; j < aff->affFaceCount; aft++, j++ )
		{
		if  ( ! strcmp( aft->aftFaceName, faceName )	)
		    { break;	}
		}

	    if  ( j >= aff->affFaceCount )
		{
		aft= (AppFontTypeface *)realloc( aff->affFaces,
					    (j+1)* sizeof(AppFontTypeface) );
		if  ( ! aft )
		    { LXDEB(j,aft); return -1;	}
		aff->affFaces= aft;

		aft += j;
		aft->aftFaceName= strdup( faceName );
		if  ( ! aft->aftFaceName )
		    { XDEB(aft->aftFaceName); return -1;	}

		aft->aftIsBold= 0;
		if  ( strcmp( weight, "medium" )	&&
		      strcmp( weight, "book" )		&&
		      strcmp( weight, "roman" )		&&
		      strcmp( weight, "thin" )		&&
		      strcmp( weight, "light" )		&&
		      strcmp( weight, "normal" )	&&
		      strcmp( weight, "regular" )	)
		    {
		    if  ( ! strcmp( weight, "bold" )		||
			  ! strcmp( weight, "black" )		||
			  ! strcmp( weight, "ultrablack" )	||
			  ! strcmp( weight, "demi" )		||
			  ! strcmp( weight, "heavy" )		||
			  ! strcmp( weight, "extrabold" )	||
			  ! strcmp( weight, "demibold" )	)
			{ aft->aftIsBold= 1;	}
		    else{ SDEB(weight);		}
		    }

		aft->aftIsSlanted= 0;
		if  ( strcmp( slant, "r" )	)
		    {
		    if  ( ! strcmp( slant, "i" )	||
			  ! strcmp( slant, "o" )	)
			{ aft->aftIsSlanted= 1;	}
		    else{ SDEB(slant);		}
		    }

		aft->aftSizes= (int *)0;
		aft->aftSizeCount= 0;
		aft->aftIsScalable= 0;
		aft->aftIsFixedWidth= ! strcmp( spacing, "m" );

		if  ( aft->aftIsFixedWidth )
		    {
		    if  ( aff->affHasProportionalWidth )
			{ SDEB(fontList[i]);	}
		    aff->affHasFixedWidth= 1;
		    }
		else{
		    if  ( aff->affHasFixedWidth )
			{ SDEB(fontList[i]);	}
		    aff->affHasProportionalWidth= 1;
		    }

		aff->affFaceCount= j+ 1;
		}

	    pxlsz= atoi( pxlszStr );
	    if  ( pxlsz == 0 )
		{ aft->aftIsScalable= 1;	}
	    else{
		if  ( strcmp( avgwdth, "0" ) )
		    {
		    sizes= aft->aftSizes;
		    for ( j= 0; j < aft->aftSizeCount; sizes++, j++ )
			{
			if  ( *sizes == pxlsz )
			    { break;	}
			}

		    if  ( j == aft->aftSizeCount )
			{
			sizes= (int *)realloc(
					aft->aftSizes, (j+1)* sizeof(int) );
			if  ( ! sizes )
			    { LXDEB(j,sizes); return -1;	}
			aft->aftSizes= sizes;

			sizes += j;

			*sizes= pxlsz;

			aft->aftSizeCount= j+ 1;
			}
		    }
		}
	    }

	XFreeFontNames( fontList );
	}

    *pFamilies= EditFontXFamilies;
    *pCount= EditFontXFamilyCount;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Get additional font properties.					*/
/*									*/
/************************************************************************/

static void appFontProperties(	Display *		display,
				AppPhysicalFont *	apf	)
    {
    if  ( ! XGetFontProperty( apf->apfFontStruct,
		XA_UNDERLINE_THICKNESS, &apf->apfUnderlineThicknessPixels ) )
	{
	apf->apfUnderlineThicknessPixels=
	    ( apf->apfFontStruct->ascent+ apf->apfFontStruct->descent )/12;

	if  ( apf->apfUnderlineThicknessPixels == 0 )
	    { apf->apfUnderlineThicknessPixels= 1;	}
	}

    if  ( ! XGetFontProperty( apf->apfFontStruct,
		XA_UNDERLINE_POSITION, &apf->apfUnderLinePositionPixels ) )
	{
	apf->apfUnderLinePositionPixels= ( apf->apfFontStruct->descent/ 2 );
	}

    /*
    {
    Atom		name;
    Atom		value;

    name= XInternAtom( display, "CHARSET_REGISTRY", 1 );
    if  ( name && XGetFontProperty( apf->apfFontStruct, name, &value ) )
	{ SSDEB(XGetAtomName(display,name),XGetAtomName(display,value)); }

    name= XInternAtom( display, "CHARSET_ENCODING", 1 );
    if  ( name && XGetFontProperty( apf->apfFontStruct, name, &value ) )
	{ SSDEB(XGetAtomName(display,name),XGetAtomName(display,value)); }
    }
    */

    apf->apfAscentPixels=  apf->apfFontStruct->ascent;
    apf->apfDescentPixels=  apf->apfFontStruct->descent;
    }

/************************************************************************/
/*									*/
/*  Format the current font name in X11 terms.				*/
/*									*/
/************************************************************************/

static void appFontXFormatCurrent(	char *			target,
					const AppFontFamily *	aff,
					const AppFontTypeface *	aft,
					int			pxlsz )
    {
    char		family[40];
    char		foundry[40];
    char		weight[40];
    char		slant[40];

    const char *	setWidth= EditFontAnyTemplate;

    const char *	registry= EditFontAnyTemplate;
    const char *	encoding= EditFontAnyTemplate;

    char		pxlszStr[40];

    strcpy( family, EditFontAnyTemplate );
    strcpy( foundry, EditFontAnyTemplate );
    strcpy( weight, EditFontAnyTemplate );
    strcpy( slant, EditFontAnyTemplate );
    strcpy( pxlszStr, EditFontAnyTemplate );

    if  ( aff )
	{
	if  ( sscanf( aff->affFontFamilyName, "%[^-]-%[^-]",
						    foundry, family ) != 2 )
	    { SDEB(aff->affFontFamilyName); }

	/*
	registry= aff->effEncoding->efeRegistry;
	encoding= aff->effEncoding->efeEncoding;
	*/
	}

    if  ( aft )
	{
	if  ( sscanf( aft->aftFaceName, "%[^-]-%[^-]", weight, slant ) != 2 )
	    { SDEB(aft->aftFaceName); }
	}

    if  ( pxlsz > 0 )
	{ sprintf( pxlszStr, "%d", pxlsz ); }

    appFontQueryString( target,
	    foundry, family, weight, slant, setWidth,
	    pxlszStr, registry, encoding );

    return;
    }

/************************************************************************/
/*									*/
/*  Find an X11 font with a postscript style font.			*/
/*									*/
/*  1)  Obtain a list of the available X11 fonts.			*/
/*  2)  Try to find an almost exact match on the family name.		*/
/*	Have a preference for fonts with a foundry.			*/
/*  3)  Do not rely too much on scalable fonts. They look ugly and do	*/
/*	not always work transparently with remote displays.		*/
/*									*/
/************************************************************************/

int appFontXFont(	char *			target,
			Display *		display,
			double			magnifiedPixelsPerTwip,
			AppFontFamily *		aff,
			AppFontTypeface *	aft,
			int			twipsSize,
			int			variant )
    {
    const AppFontFamily *	xEff;
    int				xEffCount;

    const AppFontFamily *	xef;
    const AppFontTypeface *	xt;

    int				fam;
    int				face;

    int				pixelSize;
    int				turn;

    if  ( variant == DOCfontSUPERSCRIPT		||
	  variant == DOCfontSUBSCRIPT		)
	{
	pixelSize= TWIPStoPIXELS( magnifiedPixelsPerTwip,
						    ( 6* twipsSize )/ 10 );
	}
    else{
	pixelSize= TWIPStoPIXELS( magnifiedPixelsPerTwip, twipsSize );
	}

    if  ( aft->aftXQueryFormat )
	{
	char *	before= aft->aftXQueryFormat;
	char *	after;
	int	size;
	int	i;

	if  ( *before != '-' )
	    { SDEB(aft->aftXQueryFormat); return -1;	}

	for ( i= 0; i < 6; i++ )
	    {
	    before= strchr( before+ 1, '-' );
	    if  ( ! before )
		{ SDEB(aft->aftXQueryFormat); return -1;	}
	    }

	after= strchr( before+ 1, '-' );
	if  ( ! after )
	    { SDEB(aft->aftXQueryFormat); return -1;	}

	size= before- aft->aftXQueryFormat+ 1;

	memcpy( target, aft->aftXQueryFormat, size );
	sprintf( target+ size, "%d", pixelSize );
	strcpy( target+ strlen( target ), after );

	return 0;
	}

    /*  1  */
    appFontXCatalog( display, &xEff, &xEffCount );

    /*  2  */
    for ( turn= 0; turn < 2; turn++ )
	{
	xef= xEff;
	for ( fam= 0; fam < xEffCount; xef++, fam++ )
	    {
	    char *	xf= strchr( xef->affFontFamilyName, '-' );
	    char *	psf= aff->affFontFamilyName;

	    if  ( turn == 0 && xf == xef->affFontFamilyName )
		{ continue;	}

	    if  ( ! xf )
		{ SDEB(xef->affFontFamilyName); continue;	}
	    xf++;

	    if  ( ! strcmp( xf, aff->affFontFamilyName ) )
		{ break;	}

	    while( ( *xf && *psf )					     &&
		   ( *xf == *psf					||
		     ( isupper( *psf ) && tolower( *psf ) == *xf )	)    )
		{ xf++; psf++;	}

	    if  ( ! *xf && ! * psf )
		{ break;	}
	    }

	if  ( fam < xEffCount )
	    { break;	}
	}
    
    if  ( fam >= xEffCount )
	{ SDEB(aff->affFontFamilyName); return -1;	}

    xt= xef->affFaces;
    for ( face= 0; face < xef->affFaceCount; xt++, face++ )
	{
	if  ( xt->aftIsBold == aft->aftIsBold		&&
	      xt->aftIsSlanted == aft->aftIsSlanted	)
	    { break;	}
	}

    if  ( face >=  xef->affFaceCount )
	{ SDEB(aft->aftFaceName); return -1;	}

    /*  8  */
    if  ( /* ! xt->aftIsScalable && */ xt->aftSizeCount > 0 )
	{
	int		i;

	for ( i= 0; i < xt->aftSizeCount; i++ )
	    {
	    if  ( xt->aftSizes[i] >= pixelSize )
		{ break;	}
	    }

	if  ( i >= xt->aftSizeCount )
	    { i= xt->aftSizeCount-1;	}

	if  ( i > 0 && xt->aftSizes[i] > pixelSize )
	    {
	    if  ( xt->aftSizes[i]- pixelSize	>=
		  pixelSize- xt->aftSizes[i- 1]	)
		{ i--; }
	    }

	if  ( ! xt->aftIsScalable )
	    { pixelSize= xt->aftSizes[i];	}
	else{
	    if  ( xt->aftSizes[i] > pixelSize				&&
		  10* ( xt->aftSizes[i]- pixelSize ) <= pixelSize	)
		{ pixelSize= xt->aftSizes[i];	}
	    else{
		if  ( xt->aftSizes[i] < pixelSize			&&
		      10* ( pixelSize- xt->aftSizes[i] ) <= pixelSize	)
		    { pixelSize= xt->aftSizes[i];	}
		}
	    }
	}

    appFontXFormatCurrent( target, xef, xt, pixelSize );

    return 0;
    }
	    

/************************************************************************/
/*									*/
/*  Get, possibly open a font.						*/
/*									*/
/*  1)  Is it an open font? If so return its number.			*/
/*  2)  Claim memory.							*/
/*  3)  Obtain a list of the available fonts.				*/
/*  4)  Try to find an exact match based on the name.			*/
/*  4a) Try the first word.						*/
/*  5)  Try to map 'fnil' to 'Helvetica'.				*/
/*  6)  Try to map 'fswiss' to 'Helvetica'.				*/
/*  7)  Try to map 'froman' to 'Times'.					*/
/*  8)  Chose the first family that has a fixed width font for 'fmodern'*/
/*  9)  Try to map 'fscript' to 'ITC Zapf Chancery'.			*/
/*									*/
/************************************************************************/

static int appGetPsFamilyByText(	int *			pFamily,
					const char *		name,
					const AppFontFamily *	psf,
					int			psfCount )
    {
    int			fam;

    if  ( ! name )
	{ return -1;	}

    for ( fam= 0; fam < psfCount; psf++, fam++ )
	{
	if  ( psf->affFontFamilyText				&&
	      ! strcmp( name, psf->affFontFamilyText )	)
	    { *pFamily= fam; return 0;	}
	}

    return -1;
    }

static int appGetPsFamilyByName(	int *			pFamily,
					const char *		name,
					const AppFontFamily *	psf,
					int			psfCount )
    {
    int			fam;

    if  ( ! name )
	{ return -1;	}

    for ( fam= 0; fam < psfCount; psf++, fam++ )
	{
	if  ( ! strcmp( name, psf->affFontFamilyName ) )
	    { *pFamily= fam; return 0;	}
	}

    return -1;
    }

static int appGetPsFamilyByStyle(	int *			pFamily,
					const DocumentFont *	df,
					const AppFontFamily *	psf,
					int			psfCount )
    {
    int				fam;
    const AppFontFamily *	aff;

    /*  5  */
    if  ( ! df->dfFamilyStyle			||
	  ! strcmp( df->dfFamilyStyle, "fnil" ) )
	{
	aff= psf;
	for ( fam= 0; fam < psfCount; aff++, fam++ )
	    {
	    if  ( ! strcmp( "Helvetica", aff->affFontFamilyName ) )
		{ *pFamily= fam; return 0;	}
	    }
	}

    /*  6  */
    if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
	{
	aff= psf;
	for ( fam= 0; fam < psfCount; aff++, fam++ )
	    {
	    if  ( ! strcmp( "Helvetica", aff->affFontFamilyName ) )
		{ *pFamily= fam; return 0;	}
	    }
	}

    /*  7  */
    if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
	{
	aff= psf;
	for ( fam= 0; fam < psfCount; aff++, fam++ )
	    {
	    if  ( ! strcmp( "Times", aff->affFontFamilyName ) )
		{ *pFamily= fam; return 0;	}
	    }
	}

    /*  8  */
    if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
	{
	aff= psf;
	for ( fam= 0; fam < psfCount; aff++, fam++ )
	    {
	    if  ( aff->affHasFixedWidth	)
		{ *pFamily= fam; return 0;	}
	    }
	}

    /*  9  */
    if  ( ! strcmp( df->dfFamilyStyle, "fscript" ) )
	{
	aff= psf;
	for ( fam= 0; fam < psfCount; aff++, fam++ )
	    {
	    if  ( ! strcmp( "ITC Zapf Chancery", aff->affFontFamilyName ) )
		{ *pFamily= fam; return 0;	}
	    }
	}

    return -1;
    }

int appGetPsFont(	int *			pFamily,
			int *			pFace,
			AppFontFamily **	pAff,
			AppFontTypeface **	pAft,
			const char *		afmDirectory,
			const DocumentFont *	df,
			TextAttribute		ta )
    {
    int			fam;
    int			face;

    AppFontFamily *	aff;
    int			affCount;

    AppFontFamily *	psf;
    AppFontTypeface *	pst;

    int			attempt;

    /*  3  */
    if  ( psFontCatalog( afmDirectory, &aff, &affCount ) )
	{ SDEB(afmDirectory); return -1;	}

    /*  4  */
    if  ( appGetPsFamilyByText(  &fam, df->dfName, aff, affCount )	&&
	  appGetPsFamilyByName(  &fam, df->dfName, aff, affCount )	&&
	  appGetPsFamilyByStyle( &fam, df, aff, affCount )		)
	{ SSDEB(df->dfName,df->dfFamilyStyle); return -1;	}

    psf= aff+ fam;
    for ( attempt= 0; attempt < 3; attempt++ )
	{
	pst= psf->affFaces;
	for ( face= 0; face < psf->affFaceCount; pst++, face++ )
	    {
	    if  ( ta.taFontIsBold	== pst->aftIsBold	&&
		  ta.taFontIsSlanted	== pst->aftIsSlanted	)
		{ break;	}
	    }

	if  ( face < psf->affFaceCount )
	    { break;	}

	if  ( attempt == 0 )
	    { ta.taFontIsSlanted= 0;	}
	if  ( attempt == 1 )
	    { ta.taFontIsBold= 0;	}
	}

    if  ( face >= psf->affFaceCount )
	{ face= 0;	}

    if  ( face >= psf->affFaceCount )
	{ LLDEB(ta.taFontIsBold,ta.taFontIsSlanted); return -1;	}

    *pFamily= fam;
    *pFace= face;
    *pAff= psf;
    *pAft= pst;

    return 0;
    }

static int appGetPhysicalFont(	AppPhysicalFont **	pAsf,
				AppFontFamily **	pAff,
				AppFontTypeface **	pAft,
				AppPhysicalFontList *	apfl,
				DocumentFont *		df,
				TextAttribute		ta )
    {
    int			fam;
    int			face;
    int			i;

    AppPhysicalFont *	apf;

    AppFontFamily *	psf;
    AppFontTypeface *	pst;

    /*  1  */
    apf= apfl->apflFonts;
    for ( i= 0; i < apfl->apflCount; apf++, i++ )
	{
	TextAttribute	taAsf= apf->apfAttribute;

	if  ( docEqualFont( &taAsf, &ta ) )
	    { *pAsf= apf; return i; }

#	if 0
	if  ( apf->apfDocFamilyNumber	== df->dfDocFamilyNumber	&&
	      taAsf.taFontSizeHalfPoints == ta.taFontSizeHalfPoints	&&
	      taAsf.taFontIsBold	== ta.taFontIsBold		&&
	      taAsf.taFontIsSlanted	== ta.taFontIsSlanted		)
	    { *pAsf= apf; return i; }
#	endif
	}

    /*  2  */
    apf= (AppPhysicalFont *)realloc( apfl->apflFonts,
				(apfl->apflCount+ 1)* sizeof(AppPhysicalFont) );
    if  ( ! apf )
	{ XDEB(apf); return -1;	}
    apfl->apflFonts= apf;
    apf += apfl->apflCount;

    apf->apfAttribute= ta;
    apf->apfDocFamilyNumber= -1;

    apf->apfPsFamilyNumber= -1;
    apf->apfPsFaceNumber= -1;
    apf->apfFontStruct= (XFontStruct *)0;
    apf->apfFontSet= (XFontSet)0;
    apf->apfUnderLinePositionPixels= 0;
    apf->apfUnderlineThicknessPixels= 0;
    apf->apfAscentPixels= 0;
    apf->apfDescentPixels= 0;
    apf->apfFullSizePixels= 0;

    if  ( appGetPsFont( &fam, &face, &psf, &pst,
					    apfl->apflAfmDirectory, df, ta ) )
	{ LDEB(1); return -1;	}

    apf->apfPsFamilyNumber= fam;
    apf->apfPsFaceNumber= face;

    *pAsf= apf;
    *pAff= psf;
    *pAft= pst;

    return 0;
    }

static int appOpenFont(	AppDrawingData *	add,
			DocumentFont *		df,
			DocumentFontInstance *	dfi )
    {
    Display *		display= add->addDisplay;

    AppPhysicalFont *	apf;
    char		scratch[120];

    AppFontFamily *	psf;
    AppFontTypeface *	pst;

    if  ( appGetPhysicalFont( &apf, &psf, &pst, &(add->addPhysicalFontList),
						    df, dfi->dfiAttribute ) )
	{ SLDEB(df->dfName,df->dfDocFamilyNumber); return -1;	}

    apf->apfFullSizePixels= TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip,
				10* dfi->dfiAttribute.taFontSizeHalfPoints );

    if  ( appFontXFont( scratch, display, add->addMagnifiedPixelsPerTwip,
		    psf, pst,
		    10* dfi->dfiAttribute.taFontSizeHalfPoints,
		    dfi->dfiAttribute.taSuperSub ) )
	{ SDEB(df->dfName); return -1;	}

    apf->apfFontStruct= XLoadQueryFont( display, scratch );
    if  ( ! apf->apfFontStruct )
	{ SDEB(scratch); return -1;	}

    /*
    {
    char **		missingCharSets;
    int			missingCount;
    char *		ignoreValue;

    apf->apfFontSet= XCreateFontSet( display, scratch,
			    &missingCharSets, &missingCount, &ignoreValue );
    if  ( ! apf->apfFontSet )
	{ SDEB(scratch); 	}
    else{ LDEB(missingCount);	}
    }
    */

    df->dfPsFamilyNumber= apf->apfPsFamilyNumber;
    appFontProperties( display, apf );

    dfi->dfiPsFaceNumber= apf->apfPsFaceNumber;

    return add->addPhysicalFontList.apflCount++;
    }

/************************************************************************/
/*									*/
/*  Open an instabce of a font for a document.				*/
/*									*/
/*  1)  Was a sensible font set in the document? If not just take the	*/
/*	first one that is available.					*/
/*  2)  To make users of SGML tools not too unhappy, just complain once.*/
/*									*/
/************************************************************************/

static int appGetFontInstance(	DocumentFont **		pDf,
				DocumentFontInstance **	pDfi,
				DocumentFontList *	dfl,
				TextAttribute		ta )
    {
    int				i;
    DocumentFont *		df;
    DocumentFontInstance *	dfi;

    /*  1  */
    if  ( ta.taFontNumber < 0 || ta.taFontNumber >= dfl->dflCount )
	{
	static int	complained;

	/*  2  */
	if  ( ! complained )
	    { LLDEB(ta.taFontNumber,dfl->dflCount); complained= 1;	}

	for ( i= 0; i < dfl->dflCount; i++ )
	    {
	    if  ( dfl->dflFonts[i].dfDocFamilyNumber >= 0 )
		{ ta.taFontNumber= i; break;	}
	    }

	if  ( ta.taFontNumber < 0 || ta.taFontNumber >= dfl->dflCount )
	    { LLDEB(ta.taFontNumber,dfl->dflCount); return -1; }
	}

    df= dfl->dflFonts+ ta.taFontNumber;

    dfi= df->dfInstances;
    for ( i= 0; i < df->dfInstanceCount; dfi++, i++ )
	{
	if  ( docEqualFont( &(dfi->dfiAttribute), &ta ) )
	    { break;	}
	}

    if  ( i == df->dfInstanceCount )
	{
	dfi= (DocumentFontInstance *)realloc( df->dfInstances,
		( df->dfInstanceCount+ 1 )* sizeof(DocumentFontInstance) );
	if  ( ! dfi )
	    { LXDEB(df->dfInstanceCount+ 1,dfi); return -1; }

	df->dfInstances= dfi;

	dfi += df->dfInstanceCount++;

	dfi->dfiAttribute= ta;
	dfi->dfiPrivateFont= -1;
	dfi->dfiPsFaceNumber= -1;
	}

    *pDf= df; *pDfi= dfi; return 0;
    }

int appOpenDocumentFont(	AppDrawingData *	add,
				DocumentFontList *	dfl,
				TextAttribute		ta	)
    {
    DocumentFont *		df;
    DocumentFontInstance *	dfi;

    if  ( appGetFontInstance( &df, &dfi, dfl, ta ) )
	{ LDEB(1); return -1;	}

    if  ( dfi->dfiPrivateFont < 0 )
	{
	dfi->dfiPrivateFont= appOpenFont( add, df, dfi );

	if  ( dfi->dfiPrivateFont < 0 )
	    { SLDEB(df->dfName,dfi->dfiPrivateFont);	}
	}

    return dfi->dfiPrivateFont;
    }

/************************************************************************/
/*  Format the current font name.					*/
/************************************************************************/
void appFontFormatCurrent(	char *			target,
				AppFontFamily *		aff,
				AppFontTypeface *	aft,
				int			size )
    {
    const char *	familyText= "*";
    const char *	faceName= "*";
    char		sizeStr[40];

    sizeStr[0]= '*';
    sizeStr[1]= '\0';

    if  ( aff )
	{
	if  ( aff->affFontFamilyText )
	    { familyText= aff->affFontFamilyText;	}
	else{ familyText= aff->affFontFamilyName;	}
	}

    if  ( aft )
	{ faceName= aft->aftFaceName;	}

    if  ( size > 0 )
	{ sprintf( sizeStr, "%d", size ); }

    sprintf( target, "%s,%s,%s", familyText, faceName, sizeStr );

    return;
    }

/************************************************************************/
/*  Format the current attributes.					*/
/************************************************************************/
void appFontAttributeString(	char *			target,
				const char *		familyName,
				unsigned int		updMask,
				TextAttribute		taNew )
    {
    char		sizeStr[40];

    const char *	star= "*";
    const char *	empty= "";

    const char *	boldString= star;
    const char *	slantString= star;
    const char *	underlineString= star;

    if  ( ! familyName )
	{ familyName= star;	}

    sizeStr[0]= '*';
    sizeStr[1]= '\0';

    if  ( updMask & TAmaskFONTSIZE )
	{ sprintf( sizeStr, "%d", taNew.taFontSizeHalfPoints/ 2 ); }

    if  ( updMask & TAmaskFONTBOLD )
	{
	if  ( taNew.taFontIsBold )
	    { boldString= "Bold";	}
	else{ boldString= empty;	}
	}

    if  ( updMask & TAmaskFONTSLANTED )
	{
	if  ( taNew.taFontIsSlanted )
	    { slantString= "Slanted";	}
	else{ slantString= empty;	}
	}

    if  ( updMask & TAmaskTEXTUNDERLINED )
	{
	if  ( taNew.taIsUnderlined )
	    { underlineString= "Underlined";	}
	else{ underlineString= empty;		}
	}

    sprintf( target, "%s,%s,%s,%s,%s",
	    familyName, boldString, slantString, sizeStr, underlineString );

    return;
    }

/************************************************************************/
/*									*/
/*  Analyse a string with formatted attributes.				*/
/*									*/
/*  NOTE: Scribbles in its input.					*/
/*									*/
/*  1)  Family name.							*/
/*  2)  Bold?								*/
/*  3)  Slanted?							*/
/*  4)  Size.								*/
/*  5)  Underlined?							*/
/*									*/
/*  6)  Find the postscript font that belongs to this font.		*/
/*	As the approximation by style is impossible in this case, an	*/
/*	attempt is made to find the font in the font list of the	*/
/*	document before.						*/
/*									*/
/************************************************************************/
int appFontGetAttributes(	char *			attributeString,
				char **			pFamilyName,
				unsigned int *		pUpdMask,
				TextAttribute *		pTaNew )
    {
    char *		familyName= (char *)0;
    unsigned int	updMask;
    TextAttribute	taNew;

    char *		s= attributeString;

    docInitTextAttribute( &taNew );
    updMask= 0;

    /*  1  */
    if  ( *s == '*' )
	{ s++;	}
    else{
	familyName= s; updMask |= TAmaskFONTFAMILY;

	while( *s && *s != ',' )
	    { s++;	}
	}

    if  ( *s )
	{
	if  ( *s != ',' )
	    { SSDEB(attributeString,s); return -1;	}
	*(s++)= '\0';

	/*  2  */
	if  ( *s == '*' )
	    { s++;	}
	else{
	    if  ( *s == ',' )
		{ taNew.taFontIsBold= 0; updMask |= TAmaskFONTBOLD;	}
	    else{
		if  ( ! strncmp( s, "Bold,", 5 ) )
		    {
		    taNew.taFontIsBold= 1; updMask |= TAmaskFONTBOLD;
		    s += 4;
		    }
		}
	    }
	}

    if  ( *s )
	{
	if  ( *s != ',' )
	    { SSDEB(attributeString,s); return -1;	}
	*(s++)= '\0';

	/*  3  */
	if  ( *s == '*' )
	    { s++;	}
	else{
	    if  ( *s == ',' )
		{ taNew.taFontIsSlanted= 0;  updMask |= TAmaskFONTSLANTED; }
	    else{
		if  ( ! strncmp( s, "Slanted,", 8 ) )
		    {
		    taNew.taFontIsSlanted= 1; updMask |= TAmaskFONTSLANTED;
		    s += 7;
		    }
		}
	    }
	}

    if  ( *s )
	{
	if  ( *s != ',' )
	    { SSDEB(attributeString,s); return -1;	}
	*(s++)= '\0';

	/*  4  */
	if  ( *s == '*' )
	    { s++;	}
	else{
	    char *	past;

	    taNew.taFontSizeHalfPoints= 2* strtod( s, &past )+ 0.4995 ;
	    updMask |= TAmaskFONTSIZE;

	    if  ( past == s )
		{ SSDEB(attributeString,s); return -1;	}
	    s= past;
	    }
	}

    if  ( *s )
	{
	if  ( *s != ',' )
	    { SSDEB(attributeString,s); return -1;	}
	*(s++)= '\0';

	/*  5  */
	if  ( *s == '*' )
	    { s++;	}
	else{
	    if  ( ! *s )
		{ taNew.taIsUnderlined= 0; updMask |= TAmaskTEXTUNDERLINED; }
	    else{
		if  ( ! strncmp( s, "Underlined", 10 ) )
		    {
		    taNew.taIsUnderlined= 1; updMask |= TAmaskTEXTUNDERLINED;
		    s += 10;
		    }
		}
	    }
	}

    if  ( *s )
	{ SSDEB(attributeString,s); return -1;	}

    *pFamilyName= familyName;
    *pTaNew= taNew;
    *pUpdMask= updMask;

    return 0;
    }

void appInitFontList(	AppPhysicalFontList *	apfl )
    {
    apfl->apflAfmDirectory= (char *)0;
    apfl->apflCount= 0;
    apfl->apflFonts= (AppPhysicalFont *)0;

    return;
    }

void appCleanFontList(	Display *		display,
			AppPhysicalFontList *	apfl )
    {
    int		i;

    for ( i= 0; i < apfl->apflCount; i++ )
	{
	if  ( apfl->apflFonts[i].apfFontStruct )
	    { XFreeFont( display, apfl->apflFonts[i].apfFontStruct );	}
	}

    if  ( apfl->apflFonts )
	{ free( apfl->apflFonts );	}

    return;
    }

int appCharExistsInFont(	const XFontStruct *	xfs,
				int			ch )
    {
    if  ( xfs->all_chars_exist )
	{ return 1;	}
    else{
	if  ( ! xfs->per_char )
	    {
	    if  ( ch >= xfs->min_char_or_byte2	&&
		  ch <= xfs->max_char_or_byte2	)
		{ return 1;	}
	    }
	else{
	    if  ( ch >= xfs->min_char_or_byte2			&&
		  ch <= xfs->max_char_or_byte2			&&
		  xfs->per_char[ch- xfs->min_char_or_byte2].width != 0 )
		{ return 1;	}
	    }
	}

    return 0;
    }
