/************************************************************************/
/*									*/
/*  Ted: Management of the selection and the current position.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"

#   include	<X11/Xatom.h>
#   include	<Xm/ScrollBar.h>
#   include	<Xm/ToggleB.h>

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Is the selection a position between two chars?			*/
/*									*/
/************************************************************************/

int tedHasIBarSelection(	const TedDocument *   td )
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi				&&
	  bs->bsEnd.bpBi == bs->bsBegin.bpBi		&&
	  bs->bsEnd.bpStroff == bs->bsBegin.bpStroff	)
	{ return 1;	}

    return 0;
    }

int tedHasSelection(	const TedDocument *   td )
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi )
	{ return 1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Is the selection exactly an object.					*/
/*									*/
/************************************************************************/
int tedGetObjectSelection(	TedDocument *   	td,
				BufferItem **		pBi,
				TextLine **		pTl,
				TextParticule **	pTp,
				InsertedObject **	pIo	)
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi					&&
	  bs->bsEnd.bpBi == bs->bsBegin.bpBi	)
	{
	BufferItem *		bi= bs->bsBegin.bpBi;
	int			part= bs->bsBegin.bpParticule;
	int			line= bs->bsBegin.bpLine;
	TextParticule *		tp= bi->biParaParticules+ part;
	TextLine *		tl= bi->biParaLines+ line;

	if  ( tp->tpKind == DOCkindOBJECT			&&
	      bs->bsBegin.bpStroff == tp->tpStroff		&&
	      bs->bsEnd.bpStroff == tp->tpStroff+ tp->tpStrlen	)
	    {
	    *pBi= bi;
	    *pTl= tl;
	    *pTp= tp;
	    *pIo= bi->biParaObjects+ tp->tpObjectNumber;

	    return 0;
	    }
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Scroll the selection into view.					*/
/*									*/
/*  NOTE: sliderSize is passed to XmScrollBarSetValues() because of a	*/
/*	bug in lesstif Release 0.87.0. (Jan 1999).			*/
/*									*/
/************************************************************************/
void tedScrollToPosition(	EditDocument *		ed,
				const BufferPosition *	bp,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    int				sliderSize;
    int				val;
    int				set;
	
    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    XtVaGetValues( ed->edVerticalScrollbar,
	    XmNvalue,		&val,
	    XmNsliderSize,	&sliderSize,
	    NULL );

    if  ( bp->bpY1 > val+ sliderSize )
	{
	set= bp->bpY1- sliderSize;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( bp->bpY0 < val )
	{
	set= bp->bpY0;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( val+ sliderSize > add->addBackRect.drY1 )
	{
	set= add->addBackRect.drY1- sliderSize+ 1;
	if  ( set < 0 )
	    { set= 0;	}

	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    XtVaGetValues( ed->edHorizontalScrollbar,
	    XmNvalue,		&val,
	    XmNsliderSize,	&sliderSize,
	    NULL );

    if  ( bp->bpX > val+ sliderSize )
	{
	set= bp->bpX- sliderSize;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edHorizontalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( bp->bpX < val )
	{
	set= bp->bpX;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edHorizontalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    *pScrolledX= ed->edVisibleRect.drX0- ox;
    *pScrolledY= ed->edVisibleRect.drY0- oy;

    return;
    }


/************************************************************************/
/*									*/
/*  Extend a selection that begins outside a table cell and ends inside	*/
/*  that cell.								*/
/*									*/
/*  The speccial cases that occur when the selection begins in a table	*/
/*  are covered by tedExtendSelectionFromTable(), that has priority	*/
/*  over tedExtendSelectionIntoTable().					*/
/*									*/
/************************************************************************/

static int tedExtendSelectionIntoTable(	EditDocument *		ed,
					BufferPosition *	bpEnd )
    {
    BufferItem *	cellBi= bpEnd->bpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;

    int			col1= rowBi->biGroupChildCount -1;

    docLastPosition( rowBi->biGroupChildren[col1], bpEnd );
    tedPositionCoordinates( bpEnd, &(ed->edDrawingData) );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Extend a selection that begins inside a table cell and ends outside	*/
/*  that cell.								*/
/*									*/
/*  1)  Not in same section.						*/
/*  2)  Look for selection end in same table.				*/
/*  3)  Found?								*/
/*									*/
/************************************************************************/

static int tedExtendSelectionFromTable(	EditDocument *		ed,
					BufferPosition *	bpBegin,
					BufferPosition *	bpEnd,
					int *			pCol0,
					int *			pCol1 )
    {
    BufferItem *	cellBi= bpBegin->bpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;
    BufferItem *	sectBi= rowBi->biParent;

    BufferItem *	endRowBi= bpEnd->bpBi->biParent->biParent;

    int			row1= rowBi->biNumberInParent;

    int			col0= -1;
    int			col1= -1;

    /*  1  */
    if  ( bpEnd->bpBi->biParent->biParent->biParent != sectBi )
	{
	docFirstPosition( rowBi->biGroupChildren[0], bpBegin );
	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );

	if  ( bpEnd->bpBi->biParaInTable )
	    { tedExtendSelectionIntoTable( ed, bpEnd ); }

	return 0;
	}

    /*  2  */
    while( row1 < sectBi->biGroupChildCount- 1				&&
	   docAlignedColumns( &(rowBi->biRowProperties),
	       &(sectBi->biGroupChildren[row1+1]->biRowProperties) )	)
	{
	if  ( endRowBi == sectBi->biGroupChildren[row1] )
	    { break;	}

	row1++;
	}

    if  ( endRowBi == sectBi->biGroupChildren[row1] )
	{
	if  ( bpEnd->bpBi->biParent->biNumberInParent <
					    cellBi->biNumberInParent )
	    {
	    col0= bpEnd->bpBi->biParent->biNumberInParent;
	    col1= bpBegin->bpBi->biParent->biNumberInParent;

	    docFirstPosition( rowBi->biGroupChildren[col0], bpBegin );
	    rowBi= bpEnd->bpBi->biParent->biParent;
	    docLastPosition( rowBi->biGroupChildren[col1], bpEnd );
	    }
	else{
	    docFirstPosition( bpBegin->bpBi->biParent, bpBegin );
	    docLastPosition( bpEnd->bpBi->biParent, bpEnd );
	    }

	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );
	tedPositionCoordinates( bpEnd, &(ed->edDrawingData) );
	col0= bpBegin->bpBi->biParent->biNumberInParent;
	col1= bpEnd->bpBi->biParent->biNumberInParent;
	}
    else{
	docFirstPosition( rowBi->biGroupChildren[0], bpBegin );
	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );

	if  ( bpEnd->bpBi->biParaInTable )
	    { tedExtendSelectionIntoTable( ed, bpEnd ); }

	return 0;
	}

    *pCol0= col0; *pCol1= col1; return 0;
    }

/************************************************************************/
/*									*/
/*  Extend the selection upon subsequent MotionNotify events.		*/
/*									*/
/************************************************************************/

int tedExtendSelectionToPosition(	EditDocument *		ed,
					const BufferPosition *	bpAnchor,
					const BufferPosition *	bpFound )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    int				col0= -1;
    int				col1= -1;

    BufferSelection		bsExpose;

    BufferPosition		bpFrom;
    BufferPosition		bpTo;

    int				directionToAnchor;
    int				directionBeginAnchor;
    int				directionEndAnchor;
    int				cellChanged= 0;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			mindLine= 1;

    bpFrom= *bpAnchor;
    bpTo= *bpFound;

    directionToAnchor= docComparePositions( &bpTo, bpAnchor, mindLine );
    directionBeginAnchor= docComparePositions(
			    &(td->tdSelection.bsBegin), bpAnchor, mindLine );
    directionEndAnchor= docComparePositions(
			    &(td->tdSelection.bsEnd), bpAnchor, mindLine );

    /********************/
    /*  Before anchor.	*/
    /********************/
    if  ( directionToAnchor < 0 )
	{
	int		directionToBegin;

	if  ( docPositionsInsideCell( &bpTo, &bpFrom ) )
	    {
	    if  ( bpFrom.bpBi->biParaInTable )
		{ col0= col1= bpFrom.bpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( bpTo.bpBi->biParaInTable )
		{
		if  ( tedExtendSelectionFromTable( ed, &bpTo, &bpFrom,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( bpFrom.bpBi->biParaInTable )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &bpFrom ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	directionToBegin= docComparePositions( &bpTo,
				    &(td->tdSelection.bsBegin), mindLine );
	cellChanged= ! docPositionsInsideCell(
					&bpTo, &(td->tdSelection.bsBegin) );

	/****************************************/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionEndAnchor > 0 || cellChanged )
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Left.	*/
	/************************/
	if  ( directionToBegin < 0 )
	    {
	    bsExpose.bsBegin= bpTo;
	    bsExpose.bsEnd= td->tdSelection.bsBegin;
	    bsExpose.bsCol0= bsExpose.bsCol1= -1;

	    tedScrollToPosition( ed, &bpTo, &scrolledX, &scrolledY );

	    if  ( directionEndAnchor > 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
						    scrolledX, scrolledY );
		}

	    tedExposeSelection( ed, &bsExpose, scrolledX, scrolledY );

	    td->tdSelection.bsBegin= bpTo;
	    td->tdSelection.bsEnd= bpFrom;
	    td->tdSelection.bsCol0= col0;
	    td->tdSelection.bsCol1= col1;
	    td->tdSelection.bsDirection= -1;

	    if  ( directionEndAnchor > 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
							scrolledX, scrolledY );
		}

	    return 0;
	    }

	/************************/
	/*  Shrunk Left.	*/
	/************************/
	if  ( directionToBegin > 0 )
	    {
	    bsExpose.bsBegin= td->tdSelection.bsBegin;
	    bsExpose.bsEnd= bpTo;
	    bsExpose.bsCol0= bsExpose.bsCol1= -1;

	    tedScrollToPosition( ed, &bpTo, &scrolledX, &scrolledY );

	    if  ( directionEndAnchor > 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
						    scrolledX, scrolledY );
		}

	    tedExposeSelection( ed, &bsExpose, scrolledX, scrolledY );

	    td->tdSelection.bsBegin= bpTo;
	    td->tdSelection.bsEnd= bpFrom;
	    td->tdSelection.bsCol0= col0;
	    td->tdSelection.bsCol1= col1;
	    td->tdSelection.bsDirection= -1;

	    if  ( directionEndAnchor > 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
						    scrolledX, scrolledY );
		}

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  After anchor.	*/
    /********************/
    if  ( directionToAnchor > 0 )
	{
	int		directionToEnd;

	if  ( docPositionsInsideCell( &bpTo, &bpFrom ) )
	    {
	    if  ( bpFrom.bpBi->biParaInTable )
		{ col0= col1= bpFrom.bpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( bpFrom.bpBi->biParaInTable )
		{
		if  ( tedExtendSelectionFromTable( ed, &bpFrom, &bpTo,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( bpTo.bpBi->biParaInTable )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &bpTo ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	directionToEnd= docComparePositions( &bpTo,
					&(td->tdSelection.bsEnd), mindLine );
	cellChanged= ! docPositionsInsideCell(
					&bpTo, &(td->tdSelection.bsEnd) );

	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0 || cellChanged )
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Right.	*/
	/************************/
	if  ( directionToEnd > 0 )
	    {
	    bsExpose.bsBegin= td->tdSelection.bsEnd;
	    bsExpose.bsEnd= bpTo;
	    bsExpose.bsCol0= bsExpose.bsCol1= -1;

	    tedScrollToPosition( ed, &bpTo, &scrolledX, &scrolledY );

	    if  ( directionBeginAnchor < 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
						    scrolledX, scrolledY );
		}

	    tedExposeSelection( ed, &bsExpose, scrolledX, scrolledY );

	    td->tdSelection.bsBegin= bpFrom;
	    td->tdSelection.bsEnd= bpTo;
	    td->tdSelection.bsCol0= col0;
	    td->tdSelection.bsCol1= col1;
	    td->tdSelection.bsDirection= 1;

	    if  ( directionBeginAnchor < 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
							scrolledX, scrolledY );
		}

	    return 0;
	    }

	/************************/
	/*  Shrunk Right.	*/
	/************************/
	if  ( directionToEnd < 0 )
	    {
	    bsExpose.bsBegin= bpTo;
	    bsExpose.bsEnd= td->tdSelection.bsEnd;
	    bsExpose.bsCol0= bsExpose.bsCol1= -1;

	    tedScrollToPosition( ed, &bpTo, &scrolledX, &scrolledY );

	    if  ( directionBeginAnchor < 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
							scrolledX, scrolledY );
		}

	    tedExposeSelection( ed, &bsExpose, scrolledX, scrolledY );

	    td->tdSelection.bsBegin= bpFrom;
	    td->tdSelection.bsEnd= bpTo;
	    td->tdSelection.bsCol0= col0;
	    td->tdSelection.bsCol1= col1;
	    td->tdSelection.bsDirection= 1;

	    if  ( directionBeginAnchor < 0 || cellChanged )
		{
		tedExposeSelection( ed, &(td->tdSelection),
							scrolledX, scrolledY );
		}

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  At anchor.	*/
    /********************/
	{
	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0 )
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	/****************************************/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionEndAnchor > 0 )
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	td->tdSelection.bsBegin= *bpAnchor;
	td->tdSelection.bsEnd= *bpAnchor;
	td->tdSelection.bsCol0= col0;
	td->tdSelection.bsCol1= col1;
	td->tdSelection.bsDirection= 0;
	}

    return 0;
    }

int tedExtendSelectionToXY(	EditDocument *		ed,
				const BufferPosition *	bpAnchor,
				int			mouseX,
				int			mouseY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    TextLine *			tl;
    TextParticule *		tp;

    BufferPosition		bpTo;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    bpTo= *bpAnchor;

    if  ( tedFindPosition( bd, add, mouseX+ ox, mouseY+ oy,
							&tp, &tl, &bpTo ) )
	{ /* LLDEB(mouseX,mouseY); */ return 0; }

    /*  superfluous: done by tedFindPosition()
    tedPositionCoordinates( &bpTo, &(ed->edDrawingData) );
    */

    return tedExtendSelectionToPosition( ed, bpAnchor, &bpTo );
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/
int tedFirstPosition(	const AppDrawingData *	add,
			BufferDocument *	bd,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;

    if  ( docFirstPosition( &(bd->bdItem), bp ) )
	{ return -1;	}

    tp= bp->bpBi->biParaParticules+ bp->bpParticule;
    tl= bp->bpBi->biParaLines;

    bp->bpX= tedCalculateX( bp->bpBi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedLastPosition(	const AppDrawingData *	add,
			BufferDocument *	bd,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;

    if  ( docLastPosition( &(bd->bdItem), bp ) )
	{ return -1;	}

    tp= bp->bpBi->biParaParticules+ bp->bpParticule;
    tl= bp->bpBi->biParaLines+ bp->bpBi->biParaLineCount- 1;

    bp->bpX= tedCalculateX( bp->bpBi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedNextPosition(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;

    if  ( docNextPosition( bp ) )
	{ return -1;	}

    tp= bp->bpBi->biParaParticules+ bp->bpParticule;
    tl= bp->bpBi->biParaLines+ bp->bpLine;

    bp->bpX= tedCalculateX( bp->bpBi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedPrevPosition(	const AppDrawingData *	add,
			BufferPosition *	bp,
			int			lastOne )
    {
    TextLine *		tl;
    TextParticule *	tp;

    if  ( docPrevPosition( bp, lastOne ) )
	{ return -1;	}

    tp= bp->bpBi->biParaParticules+ bp->bpParticule;
    tl= bp->bpBi->biParaLines+ bp->bpLine;

    bp->bpX= tedCalculateX( bp->bpBi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedBeginOfLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    BufferItem *	bi= bp->bpBi;
    TextLine *		tl= bi->biParaLines+ bp->bpLine;
    int			part= tl->tlFirstParticule;
    TextParticule *	tp= bi->biParaParticules+ part;

    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff;

    bp->bpX= tedCalculateX( bi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedEndOfLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    BufferItem *	bi= bp->bpBi;
    TextLine *		tl= bi->biParaLines+ bp->bpLine;
    int			part= tl->tlFirstParticule+ tl->tlParticuleCount- 1;
    TextParticule *	tp= bi->biParaParticules+ part;

    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff+ tp->tpStrlen;

    bp->bpX= tedCalculateX( bi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedPrevLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    int			part;
    int			off;

    int			x= bp->bpX;

    if  ( docPrevLine( &tp, &tl, bp ) )
	{ return -1;	}

    part= tedFindParticule( tl, bp->bpBi->biParaParticules, x, tl->tlY );
    if  ( part < 0 )
	{ LDEB(part); return -1;	}

    off= tedFindStringOffset( bp->bpBi->biParaParticules+ part,
				bp->bpBi->biParaString, add, &x, x, tl->tlY );
    if  ( off < 0 )
	{ LDEB(off); return -1;	}

    bp->bpStroff= off;
    bp->bpParticule= part;

    bp->bpX= x;
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedNextLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    int			part;
    int			off;

    int			x= bp->bpX;

    if  ( docNextLine( &tp, &tl, bp ) )
	{ return -1;	}

    part= tedFindParticule( tl, bp->bpBi->biParaParticules, x, tl->tlY );
    if  ( part < 0 )
	{ LDEB(part); return -1;	}

    off= tedFindStringOffset( bp->bpBi->biParaParticules+ part,
				bp->bpBi->biParaString, add, &x, x, tl->tlY );
    if  ( off < 0 )
	{ LDEB(off); return -1;	}

    bp->bpStroff= off;
    bp->bpParticule= part;

    bp->bpX= x;
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

    return 0;
    }

int tedPageUp(	BufferPosition *	bp,
		BufferDocument *	bd,
		const AppDrawingData *	add,
		int			pageHeight )
    {
    TextLine *		tl;
    TextParticule *	tp;

    int			x;
    int			y;

    x= bp->bpX;

    tl= bp->bpBi->biParaLines+ bp->bpLine;
    y= tl->tlY- pageHeight;

    if  ( y < 0 )
	{ y= 0;	}

    if  ( tedFindPosition( bd, add, x, y, &tp, &tl, bp ) )
	{ return -1;	}

    return 0;
    }

int tedPageDown(	BufferPosition *	bp,
			BufferDocument *	bd,
			const AppDrawingData *	add,
			int			docHeight,
			int			pageHeight )
    {
    TextLine *		tl;
    TextParticule *	tp;

    int			x;
    int			y;

    x= bp->bpX;

    tl= bp->bpBi->biParaLines+ bp->bpLine;
    y= tl->tlY+ pageHeight;

    if  ( y >= docHeight )
	{ y= docHeight- 1;	}

    if  ( tedFindPosition( bd, add, x, y, &tp, &tl, bp ) )
	{ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Scroll a selection into view.					*/
/*									*/
/*  NOTE: sliderSize is passed to XmScrollBarSetValues() because of a	*/
/*	bug in lesstif Release 0.87.0. (Jan 1999).			*/
/*									*/
/************************************************************************/

void tedScrollToSelection(	EditDocument *		ed,
				const BufferSelection *	bs,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    int				sliderSize;
    int				val;
    int				set;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    XtVaGetValues( ed->edVerticalScrollbar,
	    XmNvalue,		&val,
	    XmNsliderSize,	&sliderSize,
	    NULL );

    if  ( bs->bsEnd.bpY1 > val+ sliderSize )
	{
	set= bs->bsEnd.bpY1- sliderSize;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( bs->bsBegin.bpY0 < val )
	{
	set= bs->bsBegin.bpY0;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( val+ sliderSize > add->addBackRect.drY1 )
	{
	set= add->addBackRect.drY1- sliderSize;
	if  ( set < 0 )
	    { set= 0;	}

	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edVerticalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    XtVaGetValues( ed->edHorizontalScrollbar,
	    XmNvalue,		&val,
	    XmNsliderSize,	&sliderSize,
	    NULL );

    if  ( bs->bsEnd.bpX > val+ sliderSize )
	{
	set= bs->bsEnd.bpX- sliderSize;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edHorizontalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    if  ( bs->bsBegin.bpX < val )
	{
	set= bs->bsBegin.bpX;
	if  ( val != set )
	    {
	    XmScrollBarSetValues( ed->edHorizontalScrollbar, set, sliderSize,
								0,0, True );
	    val= set;
	    }
	}

    *pScrolledX= ed->edVisibleRect.drX0- ox;
    *pScrolledY= ed->edVisibleRect.drY0- oy;

    return;
    }

/************************************************************************/
/*									*/
/*  Start a new selection after a ButtonPress event.			*/
/*									*/
/************************************************************************/

int tedBeginSelection(	EditDocument *		ed,
			TedDocument *		td,
			BufferPosition *	newBegin,
			int			x,
			int			y )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    BufferDocument *		bd= td->tdDocument;
    TextLine *			tl;
    TextParticule *		tp;

    int				scrolledX= 0;
    int				scrolledY= 0;

    *newBegin= td->tdSelection.bsBegin;

    if  ( tedFindPosition( bd, add, x, y, &tp, &tl, newBegin ) )
	{ LLDEB(x,y); return -1;	}

    tedSetSelectedPosition( ed, td, newBegin, &scrolledX, &scrolledY );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember the text attribute of the beginning of the selection.	*/
/*									*/
/*  NOTE that for I-Bar selections, in case of ambiguity, there is a	*/
/*	preference for the attribute of the particule before the	*/
/*	current position.						*/
/*									*/
/************************************************************************/

static void tedSetCurrentTextAttribute(		TedDocument *		td,
						int			IBar,
						const BufferPosition *	bp )
    {
    BufferItem *		bi= bp->bpBi;
    TextParticule *		tp= bi->biParaParticules+ bp->bpParticule;

    if  ( IBar				&&
	  bp->bpParticule > 0		&&
	  bp->bpStroff == tp->tpStroff	&&
	  tp[-1].tpKind == DOCkindTEXT	)
	{ tp--;	}

    td->tdCurrentTextAttribute= tp->tpTextAttribute;
    td->tdCurrentPhysicalFont= tp->tpPhysicalFont;

    return;
    }

/************************************************************************/
/*									*/
/*  Set the selection to a position, and draw an I bar.			*/
/*									*/
/*  0)  If an object (Picture) was selected, remove the special window.	*/
/*  1)  Provoke the old selection to be redrawn. (Before scrolling)	*/
/*  3)  Scroll to the new selection.					*/
/*  4)  Provoke the old selection to be redrawn. (It might have been	*/
/*	scrolled to a different position)				*/
/*  5)  Bookkeeping.							*/
/*  6)  Provoke the new selection to be redrawn.			*/
/*									*/
/*  NOTE:	As scrolling to the new selection can provoke a redraw	*/
/*		the selection needs to be set before the scrollbar is	*/
/*		adapted. Otherwise the selection administration can	*/
/*		still refer to deleted text.				*/
/*									*/
/************************************************************************/

void tedSetSelectedPosition(	EditDocument *		ed,
				TedDocument *		td,
				BufferPosition *	bp,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);
    int				hadSelection;

    /*  0  */
    if  ( td->tdObjectSelected )
	{
	XUnmapWindow( XtDisplay( ed->edDocumentWidget ), td->tdObjectWindow );
	td->tdObjectSelected= 0;
	}

    hadSelection= tedHasSelection( td );

    if  ( hadSelection )
	{
	tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );
	}

    /*  5  */
    tedPositionCoordinates( bp, add );

    td->tdVisibleSelectionCopied= 0;
    td->tdSelection.bsBegin= *bp;
    td->tdSelection.bsEnd= *bp;
    td->tdSelection.bsAnchor= *bp;
    td->tdSelection.bsCol0= -1;
    td->tdSelection.bsCol1= -1;
    td->tdSelection.bsDirection= 0;

    /*  3  */
    tedScrollToPosition( ed, bp, pScrolledX, pScrolledY );

    /*  4  */
    if  ( hadSelection )
	{
	tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );
	}

    /*  6  */
    tedSelectionRectangle( &(td->tdSelectedRectangle),
						add, &(td->tdSelection) );
    tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );

    tedSetCurrentTextAttribute( td, 1, bp );

    return;
    }

/************************************************************************/
/*									*/
/*  A selection was made, do bookkeeping and provoke drawing.		*/
/*									*/
/*  0)  If an object (Picture) was selected, remove the special window.	*/
/*  1)  Provoke the old selection to be redrawn. (Before scrolling)	*/
/*  2)  Stop Cursor blinking.						*/
/*  3)  Scroll to the new selection.					*/
/*  4)  Provoke the old selection to be redrawn. (It might have moved)	*/
/*  5)  Bookkeeping.							*/
/*  6)  Provoke the new selection to be redrawn.			*/
/*									*/
/*  NOTE:	As scrolling to the new selection can provoke a redraw	*/
/*		the selection needs to be set before the scrollbar is	*/
/*		adapted. Otherwise the selection administration can	*/
/*		still refer to deleted text.				*/
/*									*/
/************************************************************************/

void tedSetSelection(	EditDocument *		ed,
			TedDocument *		td,
			BufferSelection *	bs )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);
    EditApplication *		ea= ed->edApplication;
    int				hadSelection;

    int				scrolledX= 0;
    int				scrolledY= 0;
    int *			pScrolledX= &scrolledX;
    int *			pScrolledY= &scrolledY;

    /*  0  */
    if  ( td->tdObjectSelected )
	{
	XUnmapWindow( XtDisplay( ed->edDocumentWidget ), td->tdObjectWindow );
	td->tdObjectSelected= 0;
	}

    hadSelection= tedHasSelection( td );
    if  ( hadSelection )
	{
	/*  1  */
	int	hadIBarSelection= tedHasIBarSelection( td );

	tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );
	/*  2  */
	if  ( hadIBarSelection )
	    { tedStopCursorBlink( ea->eaContext, ed ); }
	}

    /*  5  */
    if  ( bs->bsBegin.bpBi->biParaInTable			&&
	  bs->bsEnd.bpBi->biParaInTable				&&
	  bs->bsBegin.bpBi->biParent->biParent ==
			bs->bsEnd.bpBi->biParent->biParent	)
	{
	bs->bsCol0= bs->bsBegin.bpBi->biParent->biNumberInParent;
	bs->bsCol1= bs->bsEnd.bpBi->biParent->biNumberInParent;
	}

    /*  5  */
    tedSelectionCoordinates( bs, add );
    td->tdVisibleSelectionCopied= 0;
    td->tdSelection= *bs;

    /*  3  */
    tedScrollToSelection( ed, bs, pScrolledX, pScrolledY );

    if  ( hadSelection )
	{
	/*  4  */
	tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );
	}

    /*  6  */
    tedSelectionRectangle( &(td->tdSelectedRectangle),
						add, &(td->tdSelection) );
    tedExposeRectangle( ed, &(td->tdSelectedRectangle),
						*pScrolledX, *pScrolledY );

    tedSetCurrentTextAttribute( td, 0, &(td->tdSelection.bsBegin) );

    return;
    }

/************************************************************************/
/*									*/
/*  'Select All from the 'Edit' menu.					*/
/*									*/
/************************************************************************/
void tedDocSelAll(		EditDocument *	ed,
				XEvent *	event	)
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferSelection		bs;

    docInitSelection( &bs );

    if  ( docFirstPosition( &(bd->bdItem), &bs.bsBegin ) )
	{ LDEB(1); return;	}

    if  ( docLastPosition( &(bd->bdItem), &bs.bsEnd ) )
	{ LDEB(1); return;	}

    bs.bsDirection= 1; bs.bsAnchor= bs.bsBegin;
    tedSetSelection( ed, td, &bs );

    tedAdaptToolsToSelection( ed );

    return;
    }


/************************************************************************/
/*									*/
/*  Change the selection to cover whole paragraphs.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/
int tedSelectWholeParagraph(	EditApplication *	ea,
				int			direction )
    {
    EditDocument *	ed= ea->eaCurrentDocument;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;

    BufferSelection	bs;

    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( direction > 0 )
	{ bi= docNextParagraph( bi );	}
    if  ( direction < 0 )
	{ bi= docPrevParagraph( bi );	}

    if  ( ! bi )
	{ return -1;	}

    if  ( direction == 0 )
	{ direction= 1;	}

    docInitSelection( &bs );

    docSetSelection( &bs, bi, direction, 0, bi->biParaStrlen );

    tedSetSelection( ed, td, &bs );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the rectangular area as it is selected in a table.	*/
/*									*/
/*  This funtion fails if the whole selection is not inside ONE table.	*/
/*									*/
/************************************************************************/

int tedGetTableRectangle(	const BufferSelection *	bs,
				BufferItem **		pSectBi,
				int *			pCol0,
				int *			pCol1,
				int *			pCol11,
				int *			pRow00,
				int *			pRow0,
				int *			pRow1,
				int *			pRow11 )
    {
    BufferItem *	sectBi0;
    BufferItem *	sectBi1;

    int			col0;
    int			col1;
    int			col11;

    int			row0;
    int			row1;
    int			row00;
    int			row01;
    int			row10;
    int			row11;

    if  ( docDelimitTable( bs->bsBegin.bpBi,
				    &sectBi0, &col0, &row00, &row0, &row01 ) )
	{ /* LDEB(1); */ return -1;	}

    if  ( docDelimitTable( bs->bsEnd.bpBi,
				    &sectBi1, &col1, &row10, &row1, &row11 ) )
	{ /* LDEB(1); */ return -1;	}

    if  ( sectBi0 != sectBi1 )
	{ /* XXDEB(sectBi0,sectBi1); */ return -1;	}

    if  ( row00 != row10 || row01 != row11 )
	{ /* LLDEB(row00,row10); LLDEB(row01,row11); */ return -1;	}

    if  ( col0 > col1 )
	{ /* LLDEB(col0,col1); */ return -1;	}

    col11= sectBi0->biGroupChildren[row0]->biGroupChildCount- 1;

    *pSectBi= sectBi0;

    *pCol0= col0;
    *pCol1= col1;
    *pCol11= col11;

    *pRow00= row00;
    *pRow0= row0;
    *pRow1= row1;
    *pRow11= row11;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Table related menu option callbacks.				*/
/*									*/
/************************************************************************/

void tedDocTableSelectTableRectangle(	EditDocument *	ed,
					TedDocument *	td,
					BufferItem *	sectBi,
					int		col0,
					int		col1,
					int		row0,
					int		row1 )
    {
    BufferSelection	bs;
    BufferItem *	cellBi;

    docInitSelection( &bs );

    cellBi= sectBi->biGroupChildren[row0]->biGroupChildren[col0];
    if  ( docFirstPosition( cellBi, &bs.bsBegin ) )
	{ LDEB(0); return;	}

    cellBi= sectBi->biGroupChildren[row1]->biGroupChildren[col1];
    if  ( docLastPosition( cellBi, &bs.bsEnd ) )
	{ LDEB(0); return;	}

    bs.bsCol0= col0; bs.bsCol1= col1;
    bs.bsDirection= 1; bs.bsAnchor= bs.bsBegin;

    if  ( row1 < row0				||
	  ( row1 == row0 && col1 < col0 )	)
	{ bs.bsDirection= -1; bs.bsAnchor= bs.bsEnd; }

    tedSetSelection( ed, td, &bs );
    tedAdaptToolsToSelection( ed );

    return;
    }

void tedDocTableSelectTable(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;

    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;

    int			row0;
    int			row1;
    int			row00;
    int			row11;

    if  ( tedGetTableRectangle( &(td->tdSelection), &sectBi,
					    &col0, &col1, &col11,
					    &row00, &row0, &row1, &row11 ) )
	{ LDEB(1); return;	}

    tedDocTableSelectTableRectangle( ed, td, sectBi,
		    0, sectBi->biGroupChildren[row11]->biGroupChildCount- 1,
		    row00, row11 );
    return;
    }

void tedDocTableSelectRow(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;

    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;

    int			row0;
    int			row1;
    int			row00;
    int			row11;

    if  ( tedGetTableRectangle( &(td->tdSelection), &sectBi,
					    &col0, &col1, &col11,
					    &row00, &row0, &row1, &row11 ) )
	{ LDEB(1); return;	}

    tedDocTableSelectTableRectangle( ed, td, sectBi,
		    0, sectBi->biGroupChildren[row1]->biGroupChildCount- 1,
		    row0, row1 );
    return;
    }

void tedDocTableSelectColumn(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;

    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;

    int			row0;
    int			row1;
    int			row00;
    int			row11;

    if  ( tedGetTableRectangle( &(td->tdSelection), &sectBi,
					    &col0, &col1, &col11,
					    &row00, &row0, &row1, &row11 ) )
	{ LDEB(1); return;	}

    tedDocTableSelectTableRectangle( ed, td, sectBi,
						col0, col1, row00, row11 );
    return;
    }

/************************************************************************/
/*									*/
/*  Is a posintion in a hyperlink,bookmark,field?			*/
/*									*/
/************************************************************************/
static void tedPositionEnvironment(	const BufferPosition *	bp,
					Boolean *		pInHyperlink,
					Boolean *		pInField,
					Boolean *		pInBookmark )
    {
    int				startPart;
    int				endPart;
    const char *		markName;
    int				markSize;

    const char *		fileName;
    int				fileSize;

    *pInHyperlink= False;
    *pInField= False;
    *pInBookmark= False;

    if  ( ! docGetHyperlinkForPosition( bp, &startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize ) )
	{ *pInField= *pInHyperlink= True;	}
    else{
	*pInField= docParticuleInField( bp->bpBi, bp->bpParticule );
	}


    if  ( ! docGetBookmarkForPosition( bp, &startPart, &endPart,
						    &markName, &markSize ) )
	{ *pInBookmark= True;	}

    return;
    }


/************************************************************************/
/*									*/
/*  Adapt tools and rulers to the current position.			*/
/*									*/
/************************************************************************/

void tedAdaptToolsToPosition(	EditDocument *		ed,
				int			exposeRuler )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;
    const DocumentGeometry *	dg= &(bd->bdGeometry);
    const BufferPosition *	bp= &(td->tdSelection.bsBegin);

    Boolean			inHyperlink;
    Boolean			inField;
    Boolean			inBookmark;

    if  ( ! ed->edIsReadonly )
	{ td->tdCanReplaceSelection= 1;	}

    tedPositionEnvironment( bp, &inHyperlink, &inField, &inBookmark );

    tedDocAdaptHorizontalRuler( ed, bp->bpBi, exposeRuler );

    tedAdaptFontIndicatorsToSelection( ea, ed );

    XtSetSensitive( td->tdCopyWidget, False );
    XtSetSensitive( td->tdCutWidget, False );
    XtSetSensitive( td->tdPasteWidget, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdInsPictOption, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdInsFileOption, td->tdCanReplaceSelection );

    XtSetSensitive( td->tdInsSymbolWidget, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdFormatOneParaOption, False );
    XtSetSensitive( td->tdInsHyperlinkWidget, inHyperlink );
    XtSetSensitive( td->tdInsBookmarkWidget, inBookmark || ! inField );
    XtSetSensitive( td->tdInsInsertTableWidget, bp->bpBi->biParaInTable == 0 );

    XtSetSensitive( td->tdTabInsertTableOption, bp->bpBi->biParaInTable == 0 );
    XtSetSensitive( td->tdTabAddRowOption, bp->bpBi->biParaInTable != 0 );
    XtSetSensitive( td->tdTabAddColumnOption, bp->bpBi->biParaInTable != 0 );

    XtSetSensitive( td->tdSelectTableWidget, bp->bpBi->biParaInTable != 0 );
    XtSetSensitive( td->tdSelectRowWidget, bp->bpBi->biParaInTable != 0 );
    XtSetSensitive( td->tdSelectColumnWidget, bp->bpBi->biParaInTable != 0 );

    if  ( TED_FormatTool )
	{
	tedFormatToolAdaptToSelection( TED_FormatTool,
			    &(td->tdSelection), ed->edFileReadOnly, dg );
	}

    return;
    }

void tedAdaptToolsToSelection(	EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;
    const DocumentGeometry *	dg= &(bd->bdGeometry);

    int				exposeRuler= 1;

    Boolean			inHyperlink;
    Boolean			inField;
    Boolean			inBookmark;

    const BufferPosition *	bpBegin= &(td->tdSelection.bsBegin);
    const BufferPosition *	bpEnd= &(td->tdSelection.bsEnd);

    BufferItem *	sectBi;

    int			col0;
    int			col1;
    int			col11;

    int			row0;
    int			row1;
    int			row00;
    int			row11;

    Boolean		tableRectangle= False;

    if  ( tedHasIBarSelection( td ) )
	{
	tedAdaptToolsToPosition( ed, exposeRuler );
	return;
	}

    tedPositionEnvironment( bpBegin, &inHyperlink, &inField, &inBookmark );

    if  ( ! tedGetTableRectangle( &(td->tdSelection), &sectBi,
					    &col0, &col1, &col11,
					    &row00, &row0, &row1, &row11 ) )
	{ tableRectangle= True;	}

    tedDocAdaptHorizontalRuler( ed, bpBegin->bpBi, exposeRuler );

    tedAdaptFontIndicatorsToSelection( ea, ed );

    XtSetSensitive( td->tdFormatOneParaOption,
			    bpEnd->bpBi != bpBegin->bpBi &&
			    docSelectionInsideCell( &(td->tdSelection) ) );

    XtSetSensitive( td->tdCopyWidget, True );

    if  ( ed->edIsReadonly )
	{ td->tdCanReplaceSelection= 0;	}
    else{
	td->tdCanReplaceSelection=
			! tableRectangle || ( col0 == col1 && row0 == row1 );
	}
    XtSetSensitive( td->tdCutWidget, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdPasteWidget, td->tdCanReplaceSelection );

    XtSetSensitive( td->tdInsPictOption, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdInsFileOption, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdInsSymbolWidget, td->tdCanReplaceSelection );
    XtSetSensitive( td->tdInsHyperlinkWidget, bpEnd->bpBi == bpBegin->bpBi );
    XtSetSensitive( td->tdInsBookmarkWidget,
				    bpEnd->bpBi == bpBegin->bpBi && ! inField );
    XtSetSensitive( td->tdInsInsertTableWidget, False );

    XtSetSensitive( td->tdTabInsertTableOption, False );

    XtSetSensitive( td->tdTabAddColumnOption, tableRectangle );
    XtSetSensitive( td->tdTabAddRowOption, tableRectangle );

    XtSetSensitive( td->tdSelectTableWidget, tableRectangle );
    XtSetSensitive( td->tdSelectRowWidget, tableRectangle );
    XtSetSensitive( td->tdSelectColumnWidget, tableRectangle );

    if  ( TED_FormatTool )
	{
	tedFormatToolAdaptToSelection( TED_FormatTool,
			    &(td->tdSelection), ed->edFileReadOnly, dg );
	}
    }
