/************************************************************************/
/*									*/
/*  Ted: Link dialog.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>

#   include	<Xm/DialogS.h>
#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Form.h>
#   include	<Xm/Text.h>
#   include	<Xm/PanedWP.h>
#   include	<Xm/Label.h>

#   include	<Xm/Protocols.h>

#   include	<appUtil.h>

#   include	"tedApp.h"

#   include	<debugon.h>

/************************************************************************/
/*  User data for a link dialog.					*/
/************************************************************************/

typedef struct LinkContext
    {
    Widget		lcFileTextWidget;
    Widget		lcMarkTextWidget;

    Widget		lcSetLinkButton;
    Widget		lcFollowLinkButton;
    Widget		lcRemoveLinkButton;
    Widget		lcCancelButton;

    char *		lcFileText;
    char *		lcMarkText;
    char *		lcSetLinkText;
    char *		lcFollowLinkText;
    char *		lcRemoveLinkText;
    char *		lcCancelText;

    char *		lcSavedFile;
    char *		lcSavedMark;

    int			lcResponse;

    EditDocument *	lcDocument;
    } LinkContext;

/************************************************************************/
/*  Resource table for various texts.					*/
/************************************************************************/
# define xx(x)	x,x

static XtResource TED_LinkResourceTable[]=
    {
	{ xx("hyperlinkDocument"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcFileText),
		    XtRString, "Document" },
	{ xx("hyperlinkBookmark"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcMarkText),
		    XtRString, "Bookmark" },

	{ xx("hyperlinkSetLink"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcSetLinkText),
		    XtRString, "Set Link" },
	{ xx("hyperlinkRemoveLink"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcRemoveLinkText),
		    XtRString, "Remove Link" },
	{ xx("hyperlinkFollowLink"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcFollowLinkText),
		    XtRString, "Follow Link" },
	{ xx("hyperlinkCancel"), XtRString, sizeof(char *),
		    offsetof(LinkContext,lcCancelText),
		    XtRString, "Cancel" },
    };

/************************************************************************/
/*									*/
/*  Generalised button callback.					*/
/*									*/
/*  NOTE that the NO,YES responses are abused for 'Remove,Set Link'.	*/
/*									*/
/************************************************************************/
static void tedLinkDialogClosed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    LinkContext *		lc= (LinkContext *)voidlc;

    lc->lcResponse= AQDrespCANCEL; return;
    }

static void tedLinkButtonPushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    LinkContext *		lc= (LinkContext *)voidlc;

    lc->lcResponse= AQDrespFAILURE;

    if  ( button == lc->lcSetLinkButton )
	{ lc->lcResponse= AQDrespYES; return;	}

    if  ( button == lc->lcFollowLinkButton )
	{ lc->lcResponse= AQDrespOK; return;	}

    if  ( button == lc->lcRemoveLinkButton )
	{ lc->lcResponse= AQDrespNO; return;	}

    if  ( button == lc->lcCancelButton )
	{ lc->lcResponse= AQDrespCANCEL; return;	}

    LDEB(lc->lcResponse); return;
    }

/************************************************************************/
/*									*/
/*  Make the strip with the buttons.					*/
/*									*/
/************************************************************************/

static Widget tedLinkMakeButtonForm(	LinkContext *	lc,
					Widget		parent,
					Widget		dialog )
    {
    Widget		buttonRow;

    buttonRow= appMakeButtonRow( parent, 4 );


    lc->lcSetLinkButton= appMakeRowButton( buttonRow, lc->lcSetLinkText,
				tedLinkButtonPushed, (void *)lc, 0, True );

    lc->lcFollowLinkButton= appMakeRowButton( buttonRow, lc->lcFollowLinkText,
				tedLinkButtonPushed, (void *)lc, 1, False );

    lc->lcRemoveLinkButton= appMakeRowButton( buttonRow, lc->lcRemoveLinkText,
				tedLinkButtonPushed, (void *)lc, 2, False );

    lc->lcCancelButton= appMakeRowButton( buttonRow, lc->lcCancelText,
				tedLinkButtonPushed, (void *)lc, 3, False );

    XtManageChild( lc->lcSetLinkButton );
    XtManageChild( lc->lcFollowLinkButton );
    XtManageChild( lc->lcRemoveLinkButton );
    XtManageChild( lc->lcCancelButton );

    XtVaSetValues( dialog,
			XmNdefaultButton,	lc->lcSetLinkButton,
			NULL );
    XtVaSetValues( dialog,
			XmNcancelButton,	lc->lcCancelButton,
			NULL );

    XtManageChild( buttonRow );

    return buttonRow;
    }

/************************************************************************/
/*  Make the framework of a Question Dialog.				*/
/************************************************************************/
static void tedFreeLinkContext(	Widget			dialog,
				XtPointer		voidlc,
				XtPointer		callData )
    { free( voidlc ); return;	}


static Widget tedMakeLinkDialog(	EditApplication *	ea,
					Widget			relative,
					Widget *		pPaned )
    {
    Widget		shell;
    Widget		dialog;
    Widget		paned;

    const int		withSeparator= 0;

    LinkContext *	lc= (LinkContext *)malloc( sizeof(LinkContext) );

    XtGetApplicationResources( ea->eaTopWidget, (void *)lc,
		TED_LinkResourceTable, XtNumber(TED_LinkResourceTable),
		NULL, 0 );

    lc->lcFileTextWidget= (Widget)0;
    lc->lcMarkTextWidget= (Widget)0;

    lc->lcSetLinkButton= (Widget)0;
    lc->lcFollowLinkButton= (Widget)0;
    lc->lcRemoveLinkButton= (Widget)0;
    lc->lcCancelButton= (Widget)0;

    lc->lcSavedFile= (char *)0;
    lc->lcSavedMark= (char *)0;

    appMakeVerticalDialog( &shell, &dialog,
			    &paned, ea, tedLinkDialogClosed, (void *)lc,
			    (Pixmap)0, withSeparator, "tedHyperlink" );

    XtAddCallback( dialog, XmNdestroyCallback,
				    tedFreeLinkContext, (void *)lc );

    *pPaned= paned;

    return dialog;
    }

/************************************************************************/
/*									*/
/*  Make the 'Destination' half of the link dialog.			*/
/*									*/
/************************************************************************/

static void tedLinkDestinationChanged(	Widget		w,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    LinkContext *		lc= (LinkContext *)voidlc;
    EditDocument *		ed= lc->lcDocument;
    char *			text;

    text= XmTextGetString( w );

    XtSetSensitive( lc->lcSetLinkButton, ! ed->edIsReadonly && text[0] );
    XtSetSensitive( lc->lcRemoveLinkButton, False );

    if  ( text[0] )
	{ XtSetSensitive( lc->lcFollowLinkButton, True );	}
    else{ XtSetSensitive( lc->lcFollowLinkButton, False );	}

    XtFree( text );

    return;
    }

/************************************************************************/
/*									*/
/*  Run the 'Link Dialog' for a certain document.			*/
/*									*/
/************************************************************************/
void tedRunLinkDialog(	EditApplication *	ea,
			EditDocument *		ed,
			Widget			option,
			const char *		fileName,
			int			fileSize,
			const char *		markName,
			int			markSize )
    {
    LinkContext *	lc= (LinkContext *)0;

    Widget		relative= ed->edTopWidget;

    static Widget	linkDialog;

    if  ( ! linkDialog )
	{
	Widget				paned;
	Widget				label;
	Widget				bottomForm;

	linkDialog= tedMakeLinkDialog( ea, relative, &paned );

	XtVaGetValues( linkDialog,
			XmNuserData,	&lc,
			NULL );

	appMakeColumnLabel( &label, paned, lc->lcFileText );
	appMakeColumnText( &(lc->lcFileTextWidget), paned, 0, True );
	appMakeColumnLabel( &label, paned, lc->lcMarkText );
	appMakeColumnText( &(lc->lcMarkTextWidget), paned, 0, True );

	XtAddCallback( lc->lcFileTextWidget, XmNvalueChangedCallback,
				    tedLinkDestinationChanged, (void *)lc );
	XtAddCallback( lc->lcMarkTextWidget, XmNvalueChangedCallback,
				    tedLinkDestinationChanged, (void *)lc );

	bottomForm= tedLinkMakeButtonForm( lc, paned, linkDialog );

	XtManageChild( paned );
	XtManageChild( linkDialog );
	}
    else{
	XtVaGetValues( linkDialog,
			XmNuserData,	&lc,
			NULL );
	}

    /*******************************/

    if  ( relative )
	{
	appDialogRelative( relative, linkDialog );
	appSetRelativeCallback( relative, linkDialog );
	}

    appSetFocusCallback( linkDialog );

    lc->lcDocument= ed;

    if  ( lc->lcSavedFile )
	{ free( lc->lcSavedFile ); lc->lcSavedFile= (char *)0; }
    if  ( lc->lcSavedMark )
	{ free( lc->lcSavedMark ); lc->lcSavedMark= (char *)0; }

    if  ( fileSize+ markSize > 0 )
	{
	char *		saved;

	if  ( fileSize > 0 )
	    {
	    saved= (char *)malloc( fileSize+ 1 );

	    if  ( ! saved )
		{ XDEB(saved);	}
	    else{
		strncpy( saved, fileName, fileSize )[fileSize]= '\0';
		XmTextSetString( lc->lcFileTextWidget, saved );
		lc->lcSavedFile= saved;
		}
	    }
	else{ XmTextSetString( lc->lcFileTextWidget, "" );	}

	if  ( markSize > 0 )
	    {
	    saved= (char *)malloc( markSize+ 1 );

	    if  ( ! saved )
		{ XDEB(saved);	}
	    else{
		strncpy( saved, markName, markSize )[markSize]= '\0';
		XmTextSetString( lc->lcMarkTextWidget, saved );
		lc->lcSavedMark= saved;
		}
	    }
	else{ XmTextSetString( lc->lcMarkTextWidget, "" );	}

	XtSetSensitive( lc->lcRemoveLinkButton, ! ed->edIsReadonly );
	XtSetSensitive( lc->lcFollowLinkButton, True );
	}
    else{
	XtSetSensitive( lc->lcRemoveLinkButton, False );
	XtSetSensitive( lc->lcFollowLinkButton, False );

	XmTextSetString( lc->lcFileTextWidget, "" );
	XmTextSetString( lc->lcMarkTextWidget, "" );
	}

    XtSetSensitive( lc->lcSetLinkButton, False );

    XtManageChild( linkDialog );

    appSetShellTitle( XtParent( linkDialog ), option, ea->eaApplicationName );

    XmProcessTraversal( lc->lcFileTextWidget, XmTRAVERSE_CURRENT );

    lc->lcResponse= AQDrespNONE;
    while( lc->lcResponse == AQDrespNONE )
	{ XtAppProcessEvent( ea->eaContext, XtIMAll ); }

    XtUnmanageChild( linkDialog );

    switch( lc->lcResponse )
	{
	char *		file;
	char *		mark;

	case AQDrespYES:
	    {
	    file= XmTextGetString( lc->lcFileTextWidget );
	    if  ( ! file )
		{ XDEB(file); return;	}
	    mark= XmTextGetString( lc->lcMarkTextWidget );
	    if  ( ! mark )
		{ XDEB(mark); return;	}

	    if  ( tedSetHyperlink( ed, file, mark ) )
		{ SSDEB(file,mark);	}

	    XtFree( file );
	    XtFree( mark );
	    }
	    return;
	case AQDrespOK: /* follow */
	    {
	    file= XmTextGetString( lc->lcFileTextWidget );
	    if  ( ! file )
		{ XDEB(file); return;	}
	    mark= XmTextGetString( lc->lcMarkTextWidget );
	    if  ( ! mark )
		{ XDEB(mark); return;	}

	    fileSize= strlen( file );
	    markSize= strlen( mark );

	    tedFollowLink( relative, option, ed,
				    fileName, fileSize, markName, markSize );

	    XtFree( file );
	    XtFree( mark );
	    }
	    return;
	case AQDrespNO:
	    if  ( tedRemoveHyperlink( ed ) )
		{ LDEB(1);	}
	    return;
	case AQDrespCANCEL:
	    return;
	default:
	    LDEB(lc->lcResponse); return;
	}
    }
