/************************************************************************/
/*  Ted: Interface to the find tool.					*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>

#   include	<Xm/DialogS.h>
#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Form.h>
#   include	<Xm/Text.h>
#   include	<Xm/PanedWP.h>
#   include	<Xm/Label.h>

#   include	<Xm/Protocols.h>

#   include	<appUtil.h>

#   include	"docFind.h"
#   include	"tedApp.h"

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Management of link related stuff.					*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Bring a bookmark in a generally accepted format.			*/
/*									*/
/************************************************************************/

static void tedAdaptBookmarkName(	char *	mark )
    {
    int		len= 0;

    while( *mark )
	{
	if  ( len >= 40 )
	    { *mark= '\0'; break;	}

	if  ( ! isalnum( *mark ) )
	    { *mark= '_';	}

	len++; mark++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert the particule that terminates a link or a bookmark.		*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert end particule.						*/
/*									*/
/************************************************************************/

static int tedLinkSetFieldEnd(	BufferPosition *	bpEnd,
				int *			pEndPart,
				int			particuleKind,
				int			fieldNumber )
    {
    TextParticule *	tp;
    TextAttribute	ta;

    /*  1  */
    docFindLineAndParticule( bpEnd->bpBi, bpEnd->bpStroff, bpEnd, 0 );

    tp= bpEnd->bpBi->biParaParticules+ bpEnd->bpParticule;
    ta= tp->tpTextAttribute;

    if  ( tp->tpStroff != (unsigned)bpEnd->bpStroff )
	{
	int	len= tp->tpStroff+ tp->tpStrlen- bpEnd->bpStroff;

	tp->tpStrlen= bpEnd->bpStroff- tp->tpStroff;

	tp= docInsertTextParticule( bpEnd->bpBi, bpEnd->bpParticule+ 1,
					bpEnd->bpStroff, len, tp->tpKind, ta );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	}

    tp= docInsertTextParticule( bpEnd->bpBi, bpEnd->bpParticule+ 1,
				bpEnd->bpStroff, 0, particuleKind, ta );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= fieldNumber;

    *pEndPart= bpEnd->bpParticule+ 1; return 0;
    }

/************************************************************************/
/*									*/
/*  Insert the particule that starts a link or a bookmark.		*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert start particule.						*/
/*									*/
/************************************************************************/

static int tedLinkSetFieldStart(	BufferPosition *	bpStart,
					int *			pStartPart,
					int *			pEndPart,
					int			particuleKind,
					int			fieldNumber )
    {
    TextParticule *	tp;
    TextAttribute	ta;
    int			endPart= *pEndPart;

    /*  1  */
    docFindLineAndParticule( bpStart->bpBi, bpStart->bpStroff, bpStart, 1 );

    tp= bpStart->bpBi->biParaParticules+ bpStart->bpParticule;
    ta= tp->tpTextAttribute;

    /*  2  */
    if  ( tp->tpStroff != (unsigned)bpStart->bpStroff )
	{
	int		stroff= tp->tpStroff;
	int		len= bpStart->bpStroff- tp->tpStroff;

	tp->tpStrlen= tp->tpStroff+ tp->tpStrlen- bpStart->bpStroff;
	tp->tpStroff= bpStart->bpStroff;

	tp= docInsertTextParticule( bpStart->bpBi, bpStart->bpParticule,
					    stroff, len, tp->tpKind, ta );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}

	if  ( endPart >= bpStart->bpParticule )
	    { endPart++;	}

	bpStart->bpParticule++;

	}

    /*  3  */
    tp= docInsertTextParticule( bpStart->bpBi, bpStart->bpParticule,
				bpStart->bpStroff, 0, particuleKind, ta );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    tp->tpObjectNumber= fieldNumber;

    if  ( endPart >= bpStart->bpParticule )
	{ endPart++;	}

    *pStartPart= bpStart->bpParticule; *pEndPart= endPart; return 0;
    }

/************************************************************************/
/*									*/
/*  Delimit the selection for setting a link or a bookmark.		*/
/*									*/
/************************************************************************/

static int tedLinkDelimitSelection(	BufferSelection *	bsNew,
					const BufferSelection *	bsOrig,
					AppDrawingData *	add )
    {
    int			selectionTruncated= 0;
    int			part;
    TextParticule *	tp;

    *bsNew= *bsOrig;

    if  ( bsNew->bsEnd.bpBi != bsNew->bsBegin.bpBi )
	{
	if  ( docLastPosition( bsNew->bsBegin.bpBi, &(bsNew->bsEnd) ) )
	    { LDEB(1); return -1;	}

	tedPositionCoordinates( &(bsNew->bsEnd), add );
	selectionTruncated= 1;
	}

    part= bsNew->bsBegin.bpParticule;
    tp=  bsNew->bsBegin.bpBi->biParaParticules;

    while( part < bsNew->bsEnd.bpParticule )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART	||
	      tp->tpKind == DOCkindBKMKSTART	)
	    {
	    const int	lastOne= 0;

	    docFindLineAndParticule( bsNew->bsEnd.bpBi, tp->tpStroff,
						&(bsNew->bsEnd), lastOne );

	    tedPositionCoordinates( &(bsNew->bsEnd), add );
	    selectionTruncated= 1;

	    break;
	    }

	part++; tp++;
	}

    if  ( selectionTruncated )
	{
	bsNew->bsCol0= bsNew->bsCol1= -1;

	if  ( bsNew->bsDirection < 0 )
	    { bsNew->bsAnchor= bsNew->bsEnd;		}
	else{ bsNew->bsAnchor= bsNew->bsBegin;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the area covered by a link or a bookmark.			*/
/*									*/
/************************************************************************/
static void tedLinkArea(	DocumentRectangle *	drChanged,
				EditDocument *		ed,
				BufferItem *		bi,
				int			startPart,
				int			endPart )
    {
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferSelection		bsNew;

    docInitSelection( &bsNew );
    bsNew.bsBegin.bpBi= bi;
    bsNew.bsBegin.bpStroff= bi->biParaParticules[startPart].tpStroff;
    bsNew.bsEnd.bpBi= bi;
    bsNew.bsEnd.bpStroff= bi->biParaParticules[endPart].tpStroff;

    docFindLineAndParticule( bi, bsNew.bsBegin.bpStroff, &(bsNew.bsBegin), 1 );
    docFindLineAndParticule( bi, bsNew.bsEnd.bpStroff, &(bsNew.bsEnd), 0 );

    tedSelectionCoordinates( &bsNew, add );

    tedSelectionRectangle( drChanged, add, &bsNew );

    return;
    }

/************************************************************************/
/*									*/
/*  Remove the particules that delimit a link or bookmark and adjust	*/
/*  the selection to changed particule numbers.				*/
/*									*/
/*  1)  Determine the rectangle that is to be redrawn.			*/
/*  2)  Delete the particules from the paragraph.			*/
/*  3)  Adjust the selection.						*/
/*  4)  Redraw.								*/
/*									*/
/************************************************************************/

static void tedRemoveFieldParticules(	EditDocument *		ed,
					BufferItem *		bi,
					int			startPart,
					int			endPart )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    int				y;
    int				lastY;

    DocumentRectangle		drChanged;

    /*  1  */
    tedLinkArea( &drChanged, ed, bi, startPart, endPart );

    /*  2  */
    docDeleteParticules( bi, endPart, 1 );
    docDeleteParticules( bi, startPart, 1 );

    tedLayoutItem( bi, bd, add, bi->biY0, &y, &lastY );

    /*  3  */
    docFindLineAndParticule( td->tdSelection.bsBegin.bpBi,
					td->tdSelection.bsBegin.bpStroff,
					&(td->tdSelection.bsBegin), 1 );
    docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
					td->tdSelection.bsEnd.bpStroff,
					&(td->tdSelection.bsEnd), 0 );

    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, &(td->tdSelection) );

    /*  4  */
    tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

    return;
    }


/************************************************************************/
/*									*/
/*  Run the 'Link Dialog' from the 'Insert Link' menu option.		*/
/*									*/
/************************************************************************/

void tedDocInsertLink(	Widget		option,
			XtPointer	voided,
			XtPointer	voidpbcs	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);

    int				startPart;
    int				endPart;

    const char *		fileName= (const char *)0;
    int				fileSize= 0;
    const char *		markName= (const char *)0;
    int				markSize= 0;

    docGetHyperlinkForPosition( bp, &startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize );

    tedRunLinkDialog( ea, ed, option, fileName, fileSize, markName, markSize );

    tedAdaptToolsToSelection( ed );

    return;
    }

int tedSetHyperlink(	EditDocument *		ed,
			const char *		file,
			const char *		mark )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);
    BufferItem *		bi=bp->bpBi;

    int				startPart;
    int				endPart;

    const char *		fileName= (const char *)0;
    int				fileSize= 0;
    const char *		markName= (const char *)0;
    int				markSize= 0;

    DocumentField *		df;

    char *			adaptedMark= (char *)0;

    if  ( mark )
	{
	adaptedMark= strdup( mark );
	if  ( ! adaptedMark )
	    { XDEB(adaptedMark); return -1;	}

	tedAdaptBookmarkName( adaptedMark );
	}

    if  ( docGetHyperlinkForPosition( bp, &startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize ) )
	{
	BufferSelection		bsNew;

	int			y;

	int			startField;
	int			endField= -1;

	int			lastY= bi->biY0;

	int			part;
	TextParticule *		tp;

	DocumentRectangle	drChanged;

	df= docClaimField( &startField, &(bi->biParaFieldList) );
	if  ( ! df )
	    { XDEB(df); return -1;	}

	df->dfKind= DOCfkHYPERLINK;

	if  ( docSetHyperlinkDestination( df, file, adaptedMark ) )
	    { SSDEB(file,adaptedMark); return -1;	}

	if  ( tedLinkDelimitSelection( &bsNew, &(td->tdSelection), add ) )
	    { LDEB(1); return -1;	}

	/*  find end, split?, insert bkmkend */
	if  ( tedLinkSetFieldEnd( &(bsNew.bsEnd), &endPart,
						DOCkindFIELDEND, endField ) )
	    { LDEB(1); return -1;	}

	/*  find begin, split?, insert bkmkstart */
	if  ( tedLinkSetFieldStart( &(bsNew.bsBegin), &startPart, &endPart,
					    DOCkindFIELDSTART, startField ) )
	    { LDEB(1); return -1;	}

	tp= bsNew.bsBegin.bpBi->biParaParticules+ startPart+ 1;
	for ( part= startPart+ 1; part < endPart; tp++, part++ )
	    { tp->tpTextAttribute.taInField= 1; }

	tedLinkArea( &drChanged, ed, bi, startPart, endPart );

	tedLayoutItem( bi, bd, add, bi->biY0, &y, &lastY );

	docFindLineAndParticule( td->tdSelection.bsBegin.bpBi,
					td->tdSelection.bsBegin.bpStroff,
					&(td->tdSelection.bsBegin), 1 );
	docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
					td->tdSelection.bsEnd.bpStroff,
					&(td->tdSelection.bsEnd), 0 );

	tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

	appDocumentChanged( ea, ed, 1 );

	if  ( adaptedMark )
	    { free( adaptedMark );	}

	return 0;
	}
    else{
	df= bi->biParaFieldList.dflFields;
	df += bi->biParaParticules[startPart].tpObjectNumber;

	if  ( docSetHyperlinkDestination( df, file, adaptedMark ) )
	    { SSDEB(file,adaptedMark); return -1;	}

	appDocumentChanged( ea, ed, 1 );

	if  ( adaptedMark )
	    { free( adaptedMark );	}

	return 0;
	}
    }

int tedRemoveHyperlink(	EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);
    BufferItem *		bi= bp->bpBi;

    int				startPart;
    int				endPart;

    const char *		fileName= (const char *)0;
    int				fileSize= 0;
    const char *		markName= (const char *)0;
    int				markSize= 0;

    DocumentField *		df;

    if  ( docGetHyperlinkForPosition( bp, &startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize ) )
	{ LDEB(1); return -1;	}
    else{
	int			part;
	TextParticule *		tp;

	tp= bi->biParaParticules+ startPart;
	df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	docCleanField( df ); docInitField( df );

	tp= bi->biParaParticules+ startPart+ 1;
	for ( part= startPart+ 1; part < endPart; tp++, part++ )
	    {
	    tp->tpTextAttribute.taInField= 0;
	    tp->tpTextAttribute.taIsUnderlined= 0;
	    }

	tedRemoveFieldParticules( ed, bi, startPart, endPart );

	appDocumentChanged( ea, ed, 1 );

	return 0;
	}
    }

/************************************************************************/
/*									*/
/*  Run the 'Bookmark Dialog' from the 'Insert Bookmark' menu option.	*/
/*									*/
/************************************************************************/

void tedDocInsertBookmark(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcs	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);

    int				startPart;
    int				endPart;

    int				nameProposed= 0;
    char *			savedMark= (char *)0;

    const char *		markName= (const char *)0;
    int				markSize= 0;

    if  ( docGetBookmarkForPosition( bp, &startPart, &endPart,
						    &markName, &markSize ) )
	{
	BufferItem *	bi= td->tdSelection.bsBegin.bpBi;
	int		stroff= td->tdSelection.bsBegin.bpStroff;

	if  ( ! ed->edIsReadonly			&&
	      bi					&&
	      bi == td->tdSelection.bsEnd.bpBi		&&
	      stroff < td->tdSelection.bsEnd.bpStroff	)
	    {
	    unsigned char *	s= bi->biParaString+ stroff;
	    unsigned char *	e;
	    unsigned char *	p;

	    while( ! isalnum( *s )				&&
		   stroff < td->tdSelection.bsEnd.bpStroff	)
		{ s++; stroff++;	}

	    p= e= s;
	    while( stroff < td->tdSelection.bsEnd.bpStroff )
		{
		if  ( isalnum( *e ) )
		    { p= e+ 1;	}

		e++; stroff++;
		}

	    if  ( p- s > 3 )
		{
		savedMark= (char *)malloc( p- s+ 1 );
		if  ( savedMark )
		    {
		    strncpy( savedMark, (char *)s, p- s )[p- s]= '\0';
		    markName= savedMark; markSize= p- s;
		    nameProposed= 1;

		    tedAdaptBookmarkName( savedMark );
		    }
		}
	    }
	}

    tedRunBookmarkDialog( ea, ed, option, nameProposed, markName, markSize );

    if  ( savedMark )
	{ free( savedMark );	}

    tedAdaptToolsToSelection( ed );

    return;
    }

int tedSetBookmark(	EditDocument *		ed,
			const char *		mark )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);
    BufferItem *		bi= bp->bpBi;

    int				startPart;
    int				endPart;

    const char *		markName= (const char *)0;
    int				markSize= 0;

    DocumentField *		df;

    char *			adaptedMark= (char *)0;

    adaptedMark= strdup( mark );
    if  ( ! adaptedMark )
	{ XDEB(adaptedMark); return -1;	}

    tedAdaptBookmarkName( adaptedMark );

    if  ( docGetBookmarkForPosition( bp, &startPart, &endPart,
						    &markName, &markSize ) )
	{
	BufferSelection		bsNew;

	int			y;

	int			lastY= bi->biY0;

	int			startField= -1;
	int			endField= -1;
	int			documentFieldNumber= -1;

	DocumentRectangle	drChanged;

	markSize= strlen( adaptedMark );

	/*  doc  */
	df= docClaimField( &documentFieldNumber, &(bd->bdBookmarks) );
	if  ( ! df )
	    { XDEB(df); return -1;	}
	df->dfKind= DOCfkBKMKSTART;

	if  ( docSetFieldInst( df, (unsigned char *)adaptedMark, markSize ) )
	    { SDEB(adaptedMark); return -1;	}

	/*  start  */
	df= docClaimField( &startField, &(bi->biParaFieldList) );
	if  ( ! df )
	    { XDEB(df); return -1;	}
	df->dfKind= DOCfkBKMKSTART;
	df->dfNumberInDocument= documentFieldNumber;

	if  ( docSetFieldInst( df, (unsigned char *)adaptedMark, markSize ) )
	    { SDEB(adaptedMark); return -1;	}

	/*  end  */
	df= docClaimField( &endField, &(bi->biParaFieldList) );
	if  ( ! df )
	    { XDEB(df); return -1;	}
	df->dfKind= DOCfkBKMKEND;

	if  ( docSetFieldInst( df, (unsigned char *)adaptedMark, markSize ) )
	    { SDEB(adaptedMark); return -1;	}

	if  ( tedLinkDelimitSelection( &bsNew, &(td->tdSelection), add ) )
	    { LDEB(1); return -1;	}

	/*  find end, split?, insert bkmkend */
	if  ( tedLinkSetFieldEnd( &(bsNew.bsEnd), &endPart,
						DOCkindBKMKEND, endField ) )
	    { LDEB(1); return -1;	}

	/*  find begin, split?, insert bkmkstart */
	if  ( tedLinkSetFieldStart( &(bsNew.bsBegin), &startPart, &endPart,
					    DOCkindBKMKSTART, startField ) )
	    { LDEB(1); return -1;	}

	tedLinkArea( &drChanged, ed, bi, startPart, endPart );

	tedLayoutItem( bi, bd, add, bi->biY0, &y, &lastY );

	docFindLineAndParticule( td->tdSelection.bsBegin.bpBi,
					td->tdSelection.bsBegin.bpStroff,
					&(td->tdSelection.bsBegin), 1 );
	docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
					td->tdSelection.bsEnd.bpStroff,
					&(td->tdSelection.bsEnd), 0 );

	tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

	appDocumentChanged( ea, ed, 1 );

	if  ( adaptedMark )
	    { free( adaptedMark );	}

	return 0;
	}
    else{
	int	numberInDocument= -1;

	markSize= strlen( adaptedMark );

	df= bi->biParaFieldList.dflFields;
	df += bi->biParaParticules[startPart].tpObjectNumber;
	if  ( docSetFieldInst( df, (unsigned char *)adaptedMark, markSize ) )
	    { SDEB(adaptedMark); return -1;	}

	numberInDocument= df->dfNumberInDocument;
	if  ( numberInDocument >= 0 )
	    {
	    df= bd->bdBookmarks.dflFields;
	    df += numberInDocument;

	    if  ( docSetFieldInst( df,
				    (unsigned char *)adaptedMark, markSize ) )
		{ SDEB(adaptedMark); return -1;	}
	    }

	df= bi->biParaFieldList.dflFields;
	df += bi->biParaParticules[endPart].tpObjectNumber;
	if  ( docSetFieldInst( df, (unsigned char *)adaptedMark, markSize ) )
	    { SDEB(adaptedMark); return -1;	}

	appDocumentChanged( ea, ed, 1 );

	if  ( adaptedMark )
	    { free( adaptedMark );	}

	return 0;
	}
    }

int tedRemoveBookmark(	EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);
    BufferItem *		bi= bp->bpBi;

    int				startPart;
    int				endPart;

    const char *		markName= (const char *)0;
    int				markSize= 0;

    DocumentField *		df;

    if  ( docGetBookmarkForPosition( bp, &startPart, &endPart,
						    &markName, &markSize ) )
	{ LDEB(1); return -1;	}
    else{
	TextParticule *		tp;

	int			numberInDocument= -1;

	tp= bi->biParaParticules+ startPart;
	df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	numberInDocument= df->dfNumberInDocument;
	docCleanField( df ); docInitField( df );

	tp= bi->biParaParticules+ endPart;
	df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	docCleanField( df ); docInitField( df );

	if  ( numberInDocument >= 0 )
	    {
	    df= bd->bdBookmarks.dflFields+ numberInDocument;
	    docCleanField( df ); docInitField( df );
	    }

	tedRemoveFieldParticules( ed, bi, startPart, endPart );

	appDocumentChanged( ea, ed, 1 );

	return 0;
	}
    }

int tedCopyBookmarkAsLink(	EditDocument *	ed,
				const char *	markName,
				XEvent *	event )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    const BufferPosition *	bp= &(td->tdSelection.bsBegin);
    BufferItem *		bi= bp->bpBi;

    BufferSelection		bs;

    int				markSize= strlen( markName );

    docInitSelection( &bs );

    if  ( docFindBookmark( &bs, td->tdDocument, markName, markSize ) )
	{ SDEB(markName); return -1;	}

    docFindLineAndParticule( bs.bsBegin.bpBi,
				    bs.bsBegin.bpStroff, &(bs.bsBegin), 1 );
    docFindLineAndParticule( bs.bsEnd.bpBi,
				    bs.bsEnd.bpStroff, &(bs.bsEnd), 0 );

    if  ( tedCopyAsLink( ed, bi, &bs, ed->edFilename,
						markName, markSize, event ) )
	{ LDEB(1);	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Select a particular bookmark in a document.				*/
/*									*/
/************************************************************************/

int tedGoToBookmark(	EditDocument *		ed,
			const char *		markName,
			int			markSize )
    {
    BufferSelection		bs;
    TedDocument *		td;

    td= (TedDocument *)ed->edPrivateData;

    docInitSelection( &bs );

    if  ( docFindBookmark( &bs, td->tdDocument, markName, markSize ) )
	{ SDEB(markName); return -1;	}

    if  ( bs.bsBegin.bpBi != bs.bsEnd.bpBi )
	{
	bs.bsBegin.bpBi= bs.bsEnd.bpBi;
	bs.bsBegin.bpStroff= 0;
	}

    docFindLineAndParticule( bs.bsBegin.bpBi,
				bs.bsBegin.bpStroff, &(bs.bsBegin), 1 );
    docFindLineAndParticule( bs.bsEnd.bpBi,
				bs.bsEnd.bpStroff, &(bs.bsEnd), 0 );

    tedSetSelection( ed, td, &bs );

    tedAdaptToolsToSelection( ed );

    return 0;
    }
