/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	"tedApp.h"
#   include	<appImage.h>
#   include	"docPs.h"
#   include	<sioHex.h>
#   include	<appWinMeta.h>

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Translate text attributes to AfmFontInfo.				*/
/*									*/
/************************************************************************/

AfmFontInfo * docPsPrintGetAfi(	const AppPhysicalFontList *	apfl,
				int				physf )
    {
    AppFontFamily *			aff;
    int					affCount;
    AppFontTypeface *			aft;

    AppPhysicalFont *			apf;

    if  ( psFontCatalog( apfl->apflAfmDirectory, &aff, &affCount ) )
	{ SDEB(apfl->apflAfmDirectory); return (AfmFontInfo *)0;	}

    if  ( physf < 0 || physf >= apfl->apflCount )
	{ LLDEB(physf,apfl->apflCount); return (AfmFontInfo *)0;	}

    apf= apfl->apflFonts+ physf;

    if  ( apf->apfPsFamilyNumber < 0		||
	  apf->apfPsFamilyNumber >= affCount	)
	{ LLDEB(apf->apfPsFamilyNumber,affCount); return (AfmFontInfo *)0; }

    aff += apf->apfPsFamilyNumber;
    aft= aff->affFaces;

    if  ( apf->apfPsFaceNumber < 0			||
	  apf->apfPsFaceNumber >=  aff->affFaceCount	)
	{
	LLDEB(apf->apfPsFaceNumber,aff->affFaceCount);
	return (AfmFontInfo *)0;
	}

    aft += apf->apfPsFaceNumber;

    if  ( ! aft->aftPrintingData )
	{ XDEB(aft->aftPrintingData);	}

    return (AfmFontInfo *)aft->aftPrintingData;
    }

/************************************************************************/
/*									*/
/*  Calculate the height of a series of lines in a paragraph.		*/
/*									*/
/************************************************************************/

static void docPsAboveParagraph(	int *				pAbove,
					const BorderProperties **	pBp,
					const BufferItem *		bi,
					const DocumentGeometry *	dg,
					int				atTop )
    {
    int				above= *pAbove;
    const BorderProperties *	bp= (const BorderProperties *)0;

    if  ( bi->biParaInTable || ! atTop )
	{ above += bi->biParaSpaceBeforeTwips;	}

    if  ( bi->biParaTopBorder.bpIsSet )
	{
	above += bi->biParaTopBorder.bpWidthTwips;
	above += bi->biParaTopBorder.bpSpacingTwips;
	bp= &(bi->biParaTopBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	prevBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent > 0 )
		{
		prevBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent- 1];
		}
	    if  ( ! prevBi				||
		  ! prevBi->biParaBoxBorder.bpIsSet	)
		{
		above += bi->biParaBoxBorder.bpWidthTwips;
		above += bi->biParaBoxBorder.bpSpacingTwips;
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    *pAbove= above; *pBp= bp; return;
    }

static void docPsBelowParagraph(	int *				pBelow,
					const BorderProperties **	pBp,
					const BufferItem *		bi,
					const DocumentGeometry *	dg,
					int				ldist )
    {
    int				below= *pBelow;
    const BorderProperties *	bp= (const BorderProperties *)0;

    if  ( bi->biParaBottomBorder.bpIsSet )
	{
	below += ldist;
	below += bi->biParaBottomBorder.bpSpacingTwips;
	below += bi->biParaBottomBorder.bpWidthTwips;
	bp= &(bi->biParaBottomBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	nextBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent <
				bi->biParent->biGroupChildCount- 1 )
		{
		nextBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent+ 1];
		}

	    if  ( ! nextBi				||
		  ! nextBi->biParaBoxBorder.bpIsSet	)
		{
		below += ldist;
		below += bi->biParaBoxBorder.bpSpacingTwips;
		below += bi->biParaBoxBorder.bpWidthTwips;
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    if  ( bi->biParaInTable )
	{ below += bi->biParaSpaceAfterTwips; }

    *pBelow= below; *pBp= bp; return;
    }

/************************************************************************/
/*									*/
/*  Find the position of the next tab.					*/
/*									*/
/************************************************************************/
static int psTabStop(		const BufferItem *		bi,
				int				x0Geometry,
				int				x0TextLines,
				int				tabInterval,
				int *				pTab,
				int *				pX,
				int				xPosition )
    {
    int				tab= -1;
    const ParagraphProperties *	pp= &(bi->biParaProperties);

    if  ( xPosition < x0TextLines )
	{ *pX= x0TextLines; *pTab= tab; return 0; }

    if  ( pp->ppTabCount == 0 )
	{
	if  ( tabInterval == 0 )
	    {
	    LLDEB(pp->ppTabCount,tabInterval);
	    *pX= xPosition; *pTab= tab; return 0;
	    }
	else{
	    *pTab= tab;

	    xPosition -= x0Geometry;
	    *pX= ( xPosition+ 1 )/ tabInterval;
	    *pX *= tabInterval;
	    *pX += tabInterval;
	    *pX += x0Geometry;
	    return 0;
	    }
	}
    else{
	TabStop *	ts= pp->ppTabStops;

	for ( tab= 0; tab < pp->ppTabCount; ts++, tab++ )
	    {
	    if  ( ts->tsTwips > xPosition- x0Geometry )
		{ break;	}
	    }

	if  ( tab >= pp->ppTabCount )
	    {
	    if  ( tabInterval > 0 )
		{
		xPosition -= x0Geometry;
		*pX= ( xPosition+ 1 )/ tabInterval;
		*pX *= tabInterval;
		*pX += tabInterval;
		*pX += x0Geometry;
		return 0;
		}
	    else{ LDEB(tabInterval); return 1;	}
	    }

	*pTab= tab;
	*pX= ts->tsTwips+ x0Geometry;
	return 0;
	}
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of the next stretch of text upto the point	*/
/*  where it can be folded.						*/
/*									*/
/************************************************************************/

static int docPsLayoutWord(	const BufferItem *		bi,
				int *				pX1,
				int *				pVisibleX1,
				int *				pWordAscent,
				int *				pWordDescent,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				x0 )
    {
    int				particuleAscent;
    int				particuleDescent;
    int				fontAscent;
    int				fontDescent;
    int				wordAscent= 0;
    int				wordDescent= 0;

    int				accepted;
    int				width;

    int				len= 0;
    int				xFrom= x0;
    const unsigned char *	from= bi->biParaString+ tp->tpStroff;

    int				x1= x0;
    int				visibleX1= x0;

    int				physicalFont= tp->tpPhysicalFont;

    accepted= 0;
    while( accepted < particuleCount && tp->tpKind == DOCkindTEXT )
	{
	AfmFontInfo *	afi;
	TextAttribute	ta;

	ta= tp->tpTextAttribute;
	afi= docPsPrintGetAfi( apfl, tp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	while( accepted < particuleCount		&&
	       tp->tpPhysicalFont == physicalFont	&&
	       tp->tpKind == DOCkindTEXT		)
	    {
	    AfmBBox		abb;
	    const int		withKerning= 0;
	    int			size;

	    size= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

	    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT ||
		  tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT   )
		{ size= ( 6* size )/ 10; }

	    len += tp->tpStrlen;

	    width= psCalculateStringExtents( &abb, from, len,
						    size, withKerning, afi );

	    x1= xFrom+ width;
	    visibleX1= xFrom+ ( abb.abbRight- abb.abbLeft );

	    pd->pdAfi= afi;
	    pd->pdX0= x0;
	    pd->pdWidth= xFrom+ width- x0;
	    pd->pdVisibleWidth= ( abb.abbRight- abb.abbLeft );
	    pd->pdTabNumber= -1;
	    pd->pdTabKind= -1;
	    pd->pdTabPosition= 0;
	    pd->pdVisiblePixels= 0;
	    pd->pdWhiteUnits= 0;
	    pd->pdCorrectBy= 0;
	    x0= x1;

	    particuleAscent= abb.abbTop;
	    particuleDescent= abb.abbBottom;
	    fontAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	    fontDescent= fontAscent- size;

	    if  ( wordAscent < particuleAscent	&& 
		  wordAscent < fontAscent	)
		{ wordAscent=  fontAscent;	}

	    if  ( wordDescent > particuleDescent	&&
		  wordDescent > fontDescent		)
		{ wordDescent=  fontDescent; }

	    accepted++; tp++; pd++;

	    if  ( ( len > 0 && from[len- 1] == ' ' )	||
		  accepted >= particuleCount		)
		{ break;	}

	    if  ( ! tp->tpTextAttribute.taFontIsSlanted	&&
		  ta.taFontIsSlanted			)
		{
		int		extra;

		if  ( ta.taFontSizeHalfPoints <
				    tp->tpTextAttribute.taFontSizeHalfPoints )
		    { extra= ta.taFontSizeHalfPoints;			}
		else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints;	}

		extra = (int)( -afi->afiTanItalicAngle* 10* extra );

		x0 += extra;
		}
	    }

	if  ( ( len > 0 && from[len- 1] == ' ' )	||
	      accepted >= particuleCount		)
	    { break;	}

	ta= tp->tpTextAttribute;
	physicalFont= tp->tpPhysicalFont;
	from= bi->biParaString+ tp->tpStroff;
	xFrom= x0;
	len= 0;
	}

    *pWordAscent= wordAscent;
    *pWordDescent= wordDescent;
    *pX1= x1;
    *pVisibleX1= visibleX1;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible, but stop on a tab.	*/
/*									*/
/************************************************************************/

#   define	PSfoundNOTHING	0
#   define	PSfoundTAB	1
#   define	PSfoundLINE	2
#   define	PSfoundPARA	3

static int docPsLayoutText(	const BufferItem *		bi,
				int *				pFound,
				int *				pX1,
				int *				pTextAscent,
				int *				pTextDescent,
				int				x1TextLines,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				textAscent= 0;
    int				textDescent= 0;

    int				accepted;
    int				found= PSfoundNOTHING;

    accepted= 0;
    while( accepted < particuleCount && found == PSfoundNOTHING )
	{
	int		done;

	switch( tp->tpKind )
	    {
	    int			x1;
	    int			visibleX1;
	    AfmFontInfo *	afi;

	    int			wordAscent;
	    int			wordDescent;

	    case DOCkindTEXT:
		done= docPsLayoutWord( bi, &x1, &visibleX1,
						&wordAscent, &wordDescent,
						dfl, apfl, tp, pd,
						particuleCount- accepted, x0 );

		if  ( done < 1 )
		    { LDEB(done); return -1;	}
			    
		if  ( visibleX1 >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINE;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;		}
		if  ( textDescent > wordDescent )
		    { textDescent=  wordDescent;	}

		accepted += done; tp += done; pd += done;
		x0= x1;

		if  ( accepted < particuleCount			&&
		      ! tp->tpTextAttribute.taFontIsSlanted	&&
		      tp[-1].tpTextAttribute.taFontIsSlanted	)
		    {
		    int		extra;

		    if  ( tp[-1].tpTextAttribute.taFontSizeHalfPoints	<
			  tp->tpTextAttribute.taFontSizeHalfPoints		)
			{ extra= tp[-1].tpTextAttribute.taFontSizeHalfPoints; }
		    else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints;	}

		    afi= docPsPrintGetAfi( apfl, tp[-1].tpPhysicalFont );
		    if  ( ! afi )
			{ XDEB(afi); return -1;	}

		    extra= (int) ( -afi->afiTanItalicAngle* 10* extra );

		    x0 += extra;
		    }

		break;
	    case DOCkindTAB:
		found= PSfoundTAB;
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
	    case DOCkindBKMKSTART:
	    case DOCkindBKMKEND:
	    case DOCkindXE:
	    case DOCkindTC:
		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= 0;
		pd->pdVisibleWidth= 0;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;

		accepted++; tp++; pd++; break;

	    case DOCkindOBJECT:
		{
		InsertedObject *	io;
		int			width;

		io= bi->biParaObjects+ tp->tpObjectNumber;

		width= ( io->ioScaleX* io->ioTwipsWide )/ 100;

		if  ( x0+ width >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINE;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		wordAscent= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;	}

		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= width;
		pd->pdVisibleWidth= width;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;
		x0 += width;

		accepted++; tp++; pd++; break;
		}

	    default:
		LDEB(tp->tpKind); return -1;
	    }
	}

    if  ( accepted == particuleCount )
	{ found= PSfoundPARA;	}

    *pFound= found;
    *pX1= x0;
    *pTextAscent= textAscent;
    *pTextDescent= textDescent;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible. The caller can	*/
/*  require that a certain number of particules are accepted even if	*/
/*  these do not fit on the line. This is to prevent loops. A formatter	*/
/*  typically wants to be sure that at least one particule is accepted,	*/
/*  such that it can know that it actually advances.			*/
/*									*/
/************************************************************************/

static int docPsLayoutLine(	const BufferItem *		bi,
				int *				pLineAscent,
				int *				pLineDescent,
				int				tabInterval,
				const FormattingFrame *		ff,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				lineAscent= 0;
    int				lineDescent= 0;

    int				accepted;

    int				tabKind= DOCtkLEFT;
    int				tabX0= 0;
    int				tabX1= 0;
    int				tabParticule= -1;

    const TextParticule *	tpp= tp;
    ParticuleData *		pdd= pd;

    accepted= 0;
    while( accepted < particuleCount )
	{
	int		done;
	int		found;

	int		textAscent;
	int		textDescent;

	int		x1;
	AfmFontInfo *	afi;
	int		tab;

	int		visibleSinceTab;
	int		shift= 0;

	if  ( accepted >= acceptAtLeast )
	    { acceptAtLeast= 0;	}

	done= docPsLayoutText( bi, &found, &x1, &textAscent, &textDescent,
				    ff->ffX1TextLines, dfl, apfl, tpp, pdd,
				    particuleCount- accepted,
				    acceptAtLeast, x0 );
	if  ( done < 0 )
	    { LDEB(done); return -1;	}

	if  ( lineAscent < textAscent )
	    { lineAscent=  textAscent;	}
	if  ( lineDescent > textDescent )
	    { lineDescent=  textDescent;	}

	switch( tabKind )
	    {
	    case DOCtkLEFT:
		shift= 0; break;
	    case DOCtkDECIMAL:
		LDEB(tabKind);
		/*FALLTHROUGH*/
	    case DOCtkRIGHT:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1= tabX1;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{ LLDEB(done,tabParticule);	}
		break;
	    case DOCtkCENTRE:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab/ 2- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0; }
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{ LLDEB(done,tabParticule);	}
		break;
	    default:
		LDEB(tabKind);
		shift= 0; break;
	    }

	if  ( shift > 0 )
	    {
	    int		i;

	    for ( i= 0; i < done; i++ )
		{ pdd[i].pdX0 += shift; }
	    }

	accepted += done; tpp += done; pdd += done;
	x0= x1;

	switch( found )
	    {
	    case PSfoundTAB:
		break;

	    case PSfoundLINE:
	    case PSfoundPARA:
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    default:
		LDEB(found); return -1;
	    }

	afi= docPsPrintGetAfi( apfl, tpp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	tab= -1;
	if  ( psTabStop( bi, ff->ffX0Geometry, ff->ffX0TextLines,
						tabInterval, &tab, &x1, x0 ) )
	    {
	    if  ( accepted < 1 )
		{ LLDEB(particuleCount,accepted);	}

	    *pLineAscent= lineAscent; *pLineDescent= lineDescent;
	    return accepted;
	    }

	if  ( tab >= 0 )
	    { tabKind= bi->biParaTabStops[tab].tsKind;	}
	else{ tabKind= DOCtkLEFT;			}
	tabX0= x0;
	tabX1= x1;
	tabParticule= accepted;

	pdd->pdAfi= afi;
	pdd->pdX0= x0;
	pdd->pdWidth= x1- x0;
	pdd->pdVisibleWidth= 0;
	pdd->pdTabNumber= tab;
	pdd->pdTabKind= tabKind;
	pdd->pdTabPosition= x1;
	pdd->pdVisiblePixels= 0;
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( tabKind == DOCtkLEFT )
	    { x0= x1;	}

	accepted++; tpp++; pdd++;
	}

    if  ( accepted < 1 )
	{ LLDEB(particuleCount,accepted);	}

    *pLineAscent= lineAscent; *pLineDescent= lineDescent;
    return accepted;
    }

/************************************************************************/
/*									*/
/*  Justify the particules in a line of text.				*/
/*									*/
/*  1)  Start justification after the last tab of the line. Justifying	*/
/*	after anything else than a left tab is ridiculous. Simply	*/
/*	refuse to.							*/
/*  2)  Ignore organisational particules such as the delimitation of	*/
/*	links and bookmarks at the end of the line.			*/
/*									*/
/************************************************************************/

static void docPsJustifyLine(	const BufferItem *	bi,
				const TextParticule *	tp,
				ParticuleData *		pd,
				int			accepted,
				int			x1TestLines )
    {
    int			i;
    int			extra;
    int			totalWeight;
    int			step;
    int			left;

    ParticuleData *		pdd;
    const TextParticule *	tpp;

    /*  1  */
    left= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( tpp->tpKind == DOCkindTAB )
	    { left= i+ 1;	}
	}

    if  ( left > 0 )
	{
	if  ( pd[left-1].pdTabKind != DOCtkLEFT )
	    { LDEB(pd[left-1].pdTabKind); return;	}

	tp += left; pd += left; accepted -= left;
	}

    /*  2  */
    while( accepted > 0 && tp[accepted- 1].tpStrlen == 0 )
	{ accepted--;	}

    if  ( accepted < 2 )
	{ LDEB(accepted); return;	}

    extra= x1TestLines- pd[accepted- 1].pdX0- pd[accepted- 1].pdVisibleWidth;

    totalWeight= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( bi->biParaString[tpp->tpStroff+tpp->tpStrlen-1] == ' ' )
	    {
	    pdd->pdWhiteUnits=
		sqrt( (double)pdd[0].pdWidth+ (double)pdd[1].pdVisibleWidth );

	    totalWeight += pdd->pdWhiteUnits;
	    }
	}

    left= extra;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= ( extra* pdd->pdWhiteUnits )/ totalWeight;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= 1;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    step= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdX0 += step;
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    pdd->pdWidth += pdd->pdCorrectBy;
	    step += pdd->pdCorrectBy;
	    }
	}

    pdd->pdX0 += step;

    return;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible.			*/
/*									*/
/*  1)  Accept at least one particule.					*/
/*									*/
/************************************************************************/

int docPsLineBox(	LineBox *			lb,
			const BufferItem *		bi,
			int				part,
			int				atTop,
			const int			paraAscent,
			const int			paraDescent,
			const DocumentGeometry *	dg,
			int				tabInterval,
			const DocumentFontList *	dfl,
			AppPhysicalFontList *		apfl,
			const TextParticule *		tp,
			ParticuleData *			pd,
			int				x0,
			const FormattingFrame *		ff )
    {
    int			lineAscent;
    int			lineDescent;
    int			ascent;
    int			descent;
    int			leading;
    int			xShift;

    int			accepted;

    unsigned char *	from;

    if  ( part == 0 )
	{
	docPsAboveParagraph( &(lb->lbSpaceAboveLine),
				    &(lb->lbBorderAboveLine), bi, dg, atTop );
	}

    from= bi->biParaString+ tp->tpStroff;

    /*  1  */
    accepted= docPsLayoutLine( bi, &lineAscent, &lineDescent,
				    tabInterval, ff, dfl, apfl, tp, pd,
				    bi->biParaParticuleCount- part, 1, x0 );

    if  ( accepted <= 0 )
	{ LDEB(accepted); return -1;	}

    if  ( paraAscent > 0 )
	{ leading= ( paraAscent- paraDescent )/ LINEDISTFAC;	}
    else{ leading= 200/ LINEDISTFAC;				}

    ascent= paraAscent;
    descent= -paraDescent;

    if  ( lineAscent > paraAscent )
	{ ascent= lineAscent;	}
    if  ( lineDescent < paraDescent )
	{ descent= -lineDescent; }

    lb->lbLineAscent= ascent;
    lb->lbLineHeight= ascent+ descent;
    lb->lbLeading= leading;

    lb->lbLineDistance= ascent+ descent+ leading;

    if  ( bi->biParaLineSpacingTwips != 0			&&
	  ( part+ accepted < bi->biParaParticuleCount	||
	    bi->biParaLineSpacingIsMultiple		)	)
	{
	if  ( bi->biParaLineSpacingTwips > 0 )
	    {
	    if  ( bi->biParaLineSpacingTwips > lb->lbLineDistance )
		{ lb->lbLineDistance= bi->biParaLineSpacingTwips; }
	    }
	else{ lb->lbLineDistance= -bi->biParaLineSpacingTwips; }
	}

    if  ( part+ accepted == bi->biParaParticuleCount )
	{
	docPsBelowParagraph( &(lb->lbSpaceBelowLine),
		    &(lb->lbBorderBelowLine), bi, dg, lb->lbLineDistance );
	}

    if  ( bi->biParaInTable )
	{
	if  ( part == 0				&&
	      bi->biNumberInParent == 0		)
	    { lb->lbTableMarginAboveLine= ( lb->lbLeading+ 1 )/ 2; }

	if  ( part+ accepted == bi->biParaParticuleCount		&&
	      bi->biNumberInParent == bi->biParent->biGroupChildCount- 1 )
	    { lb->lbTableMarginBelowLine= ( lb->lbLeading )/ 2; }
	}

    lb->lbX0Twips= x0;

    if  ( accepted == 0 )
	{
	lb->lbStringX1Twips= x0;
	lb->lbVisibleX1Twips= x0;
	}
    else{
	ParticuleData *	pdd= pd+ accepted- 1;

	lb->lbStringX1Twips=  pdd->pdX0+ pdd->pdWidth;
	lb->lbVisibleX1Twips= pdd->pdX0+ pdd->pdVisibleWidth;
	}

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    xShift= 0; break;
	case DOCiaRIGHT:
	    xShift= ff->ffX1TextLines- lb->lbVisibleX1Twips;
	    break;
	case DOCiaCENTERED:
	    xShift= ( ff->ffX1TextLines- lb->lbVisibleX1Twips )/ 2;
	    break;
	case DOCiaJUSTIFIED:
	    xShift= 0;
	    if  ( part+ accepted < bi->biParaParticuleCount )
		{ docPsJustifyLine( bi, tp, pd, accepted, ff->ffX1TextLines ); }
	    break;
	default:
	    LDEB(bi->biParaAlignment); xShift= 0;
	    break;
	}

    if  ( xShift != 0 )
	{
	int		i;
	ParticuleData *	pdd= pd;

	for ( i= 0; i < accepted; pdd++, i++ )
	    { pdd->pdX0 += xShift;	}
	}

    return accepted;
    }

int docPsClaimParticuleData(	const BufferItem *	bi,
				ParticuleData **	pParticuleData )
    {
    static ParticuleData *	PSPrintGeometry;
    static int			PSPrintGeometryCount;

    if  ( bi->biParaParticuleCount > PSPrintGeometryCount )
	{
	ParticuleData *	fresh;
	unsigned int	size;

	size= ( bi->biParaParticuleCount+ 1 )* sizeof( ParticuleData );

	fresh= (ParticuleData *)realloc( PSPrintGeometry,
			bi->biParaParticuleCount* sizeof( ParticuleData ) );
	if  ( ! fresh )
	    { LXDEB(bi->biParaParticuleCount,fresh); return -1;	}

	PSPrintGeometry= fresh;
	}

    *pParticuleData= PSPrintGeometry; return 0;
    }


/************************************************************************/
/*									*/
/*  Determine the 'majority' font of a paragraph, and get the font	*/
/*  extents for that font.						*/
/*									*/
/*  1)  Note that subscript/superscript is NOT taken into account.	*/
/*									*/
/************************************************************************/
int docPsParagraphLineExtents(	int *				pParaAscent,
				int *				pParaDescent,
				const AppPhysicalFontList *	apfl,
				const BufferItem *		bi )
    {
    const TextParticule *	tp= bi->biParaParticules;

    int				paraAscent= 0;
    int				paraDescent= 0;
    int				i;

    static int *		counts;
    int *			fresh;

    int				found;
    int				foundCount;

    fresh= (int *)realloc( counts, apfl->apflCount* sizeof(int) );
    if  ( ! fresh )
	{ LXDEB(apfl->apflCount,fresh); return -1;	}
    counts= fresh;

    for ( i= 0; i < apfl->apflCount; i++ )
	{ counts[i]= 0;	}

    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind != DOCkindTEXT		&&
	      tp->tpKind != DOCkindTAB		&&
	      tp->tpKind != DOCkindOBJECT	)
	    { continue;	}

	if  ( tp->tpPhysicalFont < 0			||
	      tp->tpPhysicalFont >= apfl->apflCount	)
	    { LLDEB(tp->tpPhysicalFont,apfl->apflCount); continue;	}

	counts[tp->tpPhysicalFont] += tp->tpStrlen+ 1;
	}

    found= -1;
    foundCount= 0;
    for ( i= 0; i < apfl->apflCount; i++ )
	{
	if  ( counts[i] > foundCount )
	    { found= i; foundCount= counts[i];	}
	}

    if  ( found >= 0 )
	{
	int		size;
	AfmFontInfo *	afi;

	afi= docPsPrintGetAfi( apfl, found );

	size= 10* apfl->apflFonts[found].apfAttribute.taFontSizeHalfPoints;

	paraAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	paraDescent= paraAscent- size;
	}
    else{ /* LDEB(found); */	}

    *pParaAscent= paraAscent; *pParaDescent= paraDescent; return 0;
    }

/************************************************************************/
/*									*/
/*  Initialise Line Geometry.						*/
/*									*/
/************************************************************************/

void docPsInitLineBox(	LineBox *	lb )
    {
    lb->lbSpaceAboveLine= 0;
    lb->lbTableMarginAboveLine= 0;
    lb->lbSpaceBelowLine= 0;
    lb->lbTableMarginBelowLine= 0;
    lb->lbLineAscent= 0;
    lb->lbLineHeight= 0;
    lb->lbLeading= 0;
    lb->lbLineDistance= 0;

    lb->lbBorderAboveLine= (const BorderProperties *)0;
    lb->lbBorderBelowLine= (const BorderProperties *)0;

    lb->lbX0Twips= 0;
    lb->lbStringX1Twips= 0;
    lb->lbVisibleX1Twips= 0;
    }

/************************************************************************/
/*									*/
/*  Find the fonts of an image.						*/
/*									*/
/************************************************************************/

static int docPsListObjectFonts(	InsertedObject *	io,
					DocumentFontList *	dfl,
					const char *		afmDirectory,
					PostScriptFont **	pFontList,
					int *			pCount )
    {
    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisMeta;

    ObjectData *	od;
    MemoryBuffer	mb;

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    od= &(io->ioObjectData); break;
	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:
	    return 0;
	case DOCokOLEOBJECT:
	    od= &(io->ioResultData); break;
	default:
	    LDEB(io->ioKind); return 0;
	}

    mb.mbCapacity= od->odSize;
    mb.mbBytes= od->odBytes;

    sisMem= sioInMemoryOpen( &mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); return -1;	}

    if  ( appMetaListFontsPs( sisMeta, dfl, afmDirectory,
					pFontList, pCount,
					io->ioXExtent, io->ioYExtent,
					io->ioTwipsWide, io->ioTwipsHigh ) )
	{ SDEB(afmDirectory); return -1;	}

    sioInClose( sisMeta );
    sioInClose( sisMem );

    return 0;
    }


/************************************************************************/
/*									*/
/*  Recursively retrieve the list of PostScriptFonts that is used in	*/
/*  a document.								*/
/*									*/
/************************************************************************/

int docPsPrintGetItemFonts(	const BufferItem *		bi,
				DocumentFontList *		dfl,
				const AppPhysicalFontList *	apfl,
				PostScriptFont **		pFontList,
				int *				pCount )
    {
    int			part;
    int			privateFont;
    TextParticule *	tp= bi->biParaParticules+ part;
    AfmFontInfo *	afi;

    InsertedObject *	io;

    int			i;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( docPsPrintGetItemFonts( bi->biGroupChildren[i],
					    dfl, apfl, pFontList, pCount ) )
		    { LDEB(1); return -1;	}
		}
	    break;
	case DOClevPARA:
	    tp= bi->biParaParticules; part= 0;

	    while( part < bi->biParaParticuleCount )
		{
		switch( tp->tpKind )
		    {
		    case DOCkindTEXT:
		    case DOCkindTAB:
			privateFont= tp->tpPhysicalFont;
			afi= docPsPrintGetAfi( apfl, tp->tpPhysicalFont );
			if  ( ! afi )
			    { XDEB(afi); return -1;	}

			if  ( appPsRememberAfi( afi,
					    tp->tpTextAttribute,
					    pFontList, pCount ) )
			    { LDEB(1); return -1;	}

			while( part+ 1 < bi->biParaParticuleCount )
			    {
			    if  ( tp[1].tpKind != DOCkindTEXT	&&
				  tp[1].tpKind != DOCkindTAB	)
				{ break;	}
			    if  ( tp[1].tpPhysicalFont != privateFont	)
				{ break;	}

			    part++; tp++;
			    }
			part++; tp++; break;
		    case DOCkindOBJECT:
			io= bi->biParaObjects+ tp->tpObjectNumber;

			if  ( docPsListObjectFonts( io, dfl,
						    apfl->apflAfmDirectory,
						    pFontList, pCount ) )
			    { LDEB(tp->tpKind); return -1;	}

			part++; tp++; break;
		    case DOCkindFIELDSTART:
		    case DOCkindFIELDEND:
		    case DOCkindBKMKSTART:
		    case DOCkindBKMKEND:
		    case DOCkindXE:
		    case DOCkindTC:
			part++; tp++; break;
		    default:
			LDEB(tp->tpKind); return -1;
		    }
		}
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }
