/************************************************************************/
/*									*/
/*  Text Editor Buffer structure.					*/
/*									*/
/************************************************************************/

#   ifndef	DOC_BUF_H
#   define	DOC_BUF_H

#   include	<bitmap.h>
#   include	<docFont.h>
#   include	"docShape.h"
#   include	<appGeo.h>
#   include	<sioGeneral.h>

/************************************************************************/
/*									*/
/*  An homogenous piece of text. Without hyphenation, it will not be	*/
/*  divided over lines.							*/
/*  It consists of some characters and then some spaces.		*/
/*									*/
/************************************************************************/
#   define	DOCkindUNKNOWN		0
#   define	DOCkindTEXT		1
#   define	DOCkindTAB		2
#   define	DOCkindOBJECT		3
#   define	DOCkindFIELDSTART	4
#   define	DOCkindFIELDEND		5
#   define	DOCkindBKMKSTART	6
#   define	DOCkindBKMKEND		7
#   define	DOCkindBKMKEND		7
#   define	DOCkindTC		8
#   define	DOCkindXE		9

typedef struct TextParticule
    {
    unsigned int	tpStroff;
    unsigned short int	tpStrlen;
    unsigned short int	tpX0;
    unsigned short int	tpPixelsWide;
    short int		tpPhysicalFont;
    short int		tpKind;
    TextAttribute	tpTextAttribute;
    short int		tpObjectNumber;
    } TextParticule;

/************************************************************************/
/*									*/
/*  Used to layout lines of text.					*/
/*									*/
/************************************************************************/
typedef struct FormattingFrame
    {
    int		ffX0Geometry;
    int		ffX0TextLines;
    int		ffX0FirstLine;
    int		ffX1Geometry;
    int		ffX1TextLines;
    int		ffY1;
    } FormattingFrame;

/************************************************************************/
/*									*/
/*  Describes tabs.							*/
/*									*/
/*  !)	RS6000/AIX compler only supports unsigned bitfields.		*/
/*									*/
/************************************************************************/

typedef enum TabKind
    {
    DOCtkLEFT= 0,

    DOCtkRIGHT,
    DOCtkCENTRE,
    DOCtkDECIMAL
    } TabKind;

typedef enum TabLeader
    {
    DOCtlNONE= 0,

    DOCtlDOTS,
    DOCtlHYPH,
    DOCtlUNDERLINE,
    DOCtlTHICK,
    DOCtlEQUAL
    } TabLeader;

typedef struct TabStop
    {
    int			tsTwips;
    int			tsPixels;
    unsigned int	tsKind:2;	/*  (enum)	*/
    unsigned int	tsLeader:3;	/*  (enum)	*/
    } TabStop;

/************************************************************************/
/*									*/
/*  Used to lay out the text on the page.				*/
/*  A text line consists of a series of particules.			*/
/*									*/
/************************************************************************/
typedef struct TextLine
    {
    int			tlStroff;
    int			tlStrlen;
				/****************************************/
				/*  What piece of text.			*/
				/****************************************/
    int			tlFirstParticule;
    int			tlParticuleCount;
				/****************************************/
				/*  What particules in the paragraph.	*/
				/****************************************/
    int			tlY;
    int			tlY0;
    int			tlY1;
				/****************************************/
				/*  Base line.				*/
				/*  Ascender top.			*/
				/*  Descender bottom+ line dist.	*/
				/****************************************/
    } TextLine;

/************************************************************************/
/*									*/
/*  A paragraph in a document.						*/
/*									*/
/*  !)	RS6000/AIX compler only supports unsigned bitfields.		*/
/*									*/
/************************************************************************/

typedef enum BorderStyle
    {
    DOCbsSHADOWED= 0,
    DOCbsDOUBLE,
    DOCbsDOT,
    DOCbsDASH,
    DOCbsHAIR,
    DOCbsDASHSM,
    DOCbsDASHD,
    DOCbsDASHDD,
    DOCbsTRIPLE,
    DOCbsTNTHSG,
    DOCbsTHTNSG,
    DOCbsTNTHTNSG,
    DOCbsTNTHMG,
    DOCbsTHTNMG,
    DOCbsTNTHTNMG,
    DOCbsTNTHLG,
    DOCbsTHTNLG,
    DOCbsTNTHTNLG,
    DOCbsWAVY,
    DOCbsWAVYDB,
    DOCbsDASHDOTSTR,
    DOCbsEMBOSS,
    DOCbsENGRAVE
    } BorderStyle;

typedef struct BorderProperties
    {
    unsigned int	bpColor:8;
    unsigned int	bpSpacingTwips:16;
    unsigned int	bpWidthTwips:8;
    unsigned int	bpStyle:5;		/*  (enum)	*/
    unsigned int	bpThickness:2;		/*  0, 1 or 2	*/
    unsigned int	bpIsSet:1;
    } BorderProperties;

#   define	docCopyBorderProperties( bp1, bp2 ) *(bp1)= *(bp2)

typedef enum ItemAlignment
    {
    DOCiaLEFT= 0,
    DOCiaRIGHT,
    DOCiaCENTERED,
    DOCiaJUSTIFIED
    } ItemAlignment;

typedef enum ObjectKind
    {
    DOCokUNKNOWN= 0,
    DOCokPICTWMETAFILE,
    DOCokPICTPNGBLIP,
    DOCokPICTJPEGBLIP,
    DOCokOLEOBJECT
    } ObjectKind;

typedef enum VerticalTextAlignment
    {
    DOCvtaTOP= 0,
    DOCvtaCENTERED,
    DOCvtaBOTTOM,
    DOCvtaVERT_LEFT,
    DOCvtaVERT_RIGHT
    } VerticalTextAlignment;

typedef enum ShadingPattern
    {
    DOCspSOLID= 0,

    DOCspHORIZONTAL,
    DOCspVERTICAL,
    DOCspDIAGONAL_DOWN,
    DOCspDIAGONAL_UP,
    DOCspCROSSED,
    DOCspDIAGONALCROSSED,
    DOCspDK_HORIZONTAL,
    DOCspDK_VERTICAL,
    DOCspDK_DIAGONAL_DOWN,
    DOCspDK_DIAGONAL_UP,
    DOCspDK_CROSSED,
    DOCspDK_DIAGONALCROSSED
    } ShadingPattern;

typedef struct CellProperties
    {
    int			cpRightBoundaryTwips;
    int			cpRightBoundaryPixels;

    int			cpForegroundColor;
    int			cpBackgroundColor;

    BorderProperties	cpTopBorder;
    BorderProperties	cpBottomBorder;
    BorderProperties	cpLeftBorder;
    BorderProperties	cpRightBorder;

    unsigned int	cpShadingLevel:10;
    unsigned int	cpShadingPattern:4;		/*  (enum)	*/

    unsigned int	cpFirstInMergedRange:1;
    unsigned int	cpMergedWithPrevious:1;

    unsigned int	cpVerticalTextAlignment:3;	/*  (enum)	*/
    } CellProperties;

#   define	docCopyCellProperties( cp1, cp2 ) \
					    ( ( *(cp1)= *(cp2) ), 0 )

#   define	docCleanCellProperties( cp ) /* nothing */

typedef struct RowProperties
    {
    int			rpCellCount;
    CellProperties *	rpCells;

    int			rpHalfGapWidthTwips;
    int			rpHalfGapWidthPixels;

    int			rpHeightTwips;
    int			rpHeightPixels;

    int			rpLeftIndentTwips;
    int			rpLeftIndentPixels;

			/************************************************/
			/*  Row borders are not stored in the word	*/
			/*  binary format, and seem to be irrelevant:	*/
			/************************************************/
    BorderProperties	rpTopBorder;
    BorderProperties	rpBottomBorder;
    BorderProperties	rpLeftBorder;
    BorderProperties	rpRightBorder;
    BorderProperties	rpHorizontalBorder;
    BorderProperties	rpVerticalBorder;

    ItemAlignment	rpAlignment:3;
    unsigned int	rpHasTableParagraphs:1;

    unsigned int	rpHasHorizontalBorders:1;
    unsigned int	rpHasVerticalBorders:1;

    unsigned int	rpIsTableHeader:1;
    unsigned int	rpKeepOnPage:1;
    } RowProperties;

typedef struct SectionProperties
    {
    int			spColumnCount;

    unsigned int	spHasFacingPages:1;
    unsigned int	spHasTitlePage:1;

    struct BufferItem *	spHeader;
    struct BufferItem *	spFirstPageHeader;
    struct BufferItem *	spLeftPageHeader;
    struct BufferItem *	spRightPageHeader;

    struct BufferItem *	spFooter;
    struct BufferItem *	spFirstPageFooter;
    struct BufferItem *	spLeftPageFooter;
    struct BufferItem *	spRightPageFooter;

    } SectionProperties;

typedef struct DocumentProperties
    {
    int			dpContainsTables;
    int			dpTabIntervalTwips;
    int			dpDefaultColor;
    int			dpAnsiCodePage;
    } DocumentProperties;

typedef struct ParagraphProperties
    {
    short int		ppTabCount;
    TabStop *		ppTabStops;

    short int		ppFirstIndentTwips;
    short int		ppFirstIndentPixels;

    short int		ppLeftIndentTwips;
    short int		ppLeftIndentPixels;

    short int		ppRightIndentTwips;
    short int		ppRightIndentPixels;

    short int		ppSpaceBeforeTwips;
    short int		ppSpaceBeforePixels;

    short int		ppSpaceAfterTwips;
    short int		ppSpaceAfterPixels;

    short int		ppLineSpacingTwips;
    short int		ppLineSpacingPixels;

    short int		ppStyle;

    BorderProperties	ppTopBorder;
    BorderProperties	ppBottomBorder;
    BorderProperties	ppLeftBorder;
    BorderProperties	ppRightBorder;
    BorderProperties	ppBetweenBorder;
    BorderProperties	ppBoxBorder;
    BorderProperties	ppBar;

    unsigned int	ppOutlineLevel:4;

    unsigned int	ppShadingLevel:10;
    unsigned int	ppShadingPattern:4;		/*  (enum)	*/

    unsigned int	ppAlignment:3;			/*  (enum)	*/

    unsigned int	ppStartsOnNewPage:1;
    unsigned int	ppInTable:1;
    unsigned int	ppLineSpacingIsMultiple:1;

    unsigned int	ppKeepOnPage:1;
    unsigned int	ppKeepWithNext:1;
    } ParagraphProperties;

/************************************************************************/
/*									*/
/*  Masks for updating paragraph properies.				*/
/*									*/
/************************************************************************/

#   define	PPupdNONE		0x0000

#   define	PPupdLEFT_INDENT	0x0001
#   define	PPupdFIRST_INDENT	0x0002
#   define	PPupdRIGHT_INDENT	0x0004
#   define	PPupdTAB_STOPS		0x0008

#   define	PPupdNEWPAGE		0x0010
#   define	PPupdSPACE_BEFORE	0x0020
#   define	PPupdSPACE_AFTER	0x0040
#   define	PPupdTOP_BORDER		0x0080
#   define	PPupdBOTTOM_BORDER	0x0100
#   define	PPupdALIGNMENT		0x0200
#   define	PPupdLINE_SPACING	0x0400

#   define	PPupdALL		0x7fff

#   define	PPupdCOLUMNS		0x8000		/*  not real	*/

typedef enum ShapeHorizontalAttachment
    {
    SHPshaPAGE= 0,
    SHPshaMARGIN,
    SHPshaCOLUMN
    } ShapeHorizontalAttachment;

typedef enum ShapeVerticalAttachment
    {
    SHPsvaPAGE= 0,
    SHPsvaMARGIN,
    SHPsvaPARAGRAPH
    } ShapeVerticalAttachment;

typedef enum ShapeWrapStyle
    {
    SHPswsTOPBOTTOM= 1,
    SHPswsAROUND,
    SHPswsNONE,
    SHPswsTIGHTLY,
    SHPswsTHROUGH
    } ShapeWrapStyle;

typedef enum ShapeWrapSide
    {
    SHPswsBOTH= 0,
    SHPswsLEFT,
    SHPswsRIGHT,
    SHPswsLARGEST
    } ShapeWrapSide;

typedef struct ShapeProperties
    {
    int			spTwipsLeftOfAnchor;
    int			spTwipsAboveAnchor;
    int			spTwipsBelowAnchor;
    int			spTwipsRightOfAnchor;

    unsigned int	spHorizontalAttachment:2;	/*  (enum)	*/
    unsigned int	spVerticalAttachment:2;		/*  (enum)	*/
    unsigned int	spWrapStyle:3;			/*  (enum)	*/
    unsigned int	spWrapSide:2;			/*  (enum)	*/
    unsigned int	spLockAnchor:1;
    } ShapeProperties;

typedef struct ObjectData
    {
    unsigned char *	odBytes;
    int			odSize;
    } ObjectData;

typedef struct InsertedObject
    {
    int			ioKind;		/*  Kind of object.		*/
    int			ioResultKind;	/*  Kind of object.		*/
    int			ioTwipsWide;	/*  Width of object.		*/
    int			ioTwipsHigh;	/*  Height of object.		*/
    int			ioScaleX;	/*  In %.			*/
    int			ioScaleY;	/*  In %.			*/
    int			ioPixelsWide;	/*  Width of object on screen	*/
    int			ioPixelsHigh;	/*  Height of object on screen	*/
    int			ioXExtent;	/*  Of metafile picture.	*/
    int			ioYExtent;	/*  Of metafile picture.	*/
    int			ioUnitsPerInch;	/*  Only use if > 0		*/

    int			ioDragWide;	/*  PixelsWide during resize.	*/
    int			ioDragHigh;	/*  PixelsHigh during resize.	*/

    ObjectData		ioObjectData;
    ObjectData		ioResultData;

    unsigned char *	ioObjectName;
    unsigned char *	ioObjectClass;
    int			ioBliptag;

    unsigned long	ioPixmap;
    void *		ioPrivate;
    } InsertedObject;

typedef enum FieldKind
    {
    DOCfkFREE= 0,
    DOCfkUNKNOWN,
    DOCfkHYPERLINK,
    DOCfkINDEX_ENTRY,
    DOCfkCONTENTS_ENTRY,
    DOCfkBKMKSTART,
    DOCfkBKMKEND,
    DOCfkPAGEFIELD
    } FieldKind;

typedef struct DocumentField
    {
    FieldKind		dfKind;
    ObjectData		dfData;
    ObjectData		dfInstructions;
    int			dfNumberInDocument;
    } DocumentField;

typedef struct DocumentFieldList
    {
    DocumentField *	dflFields;
    int			dflFieldCount;
    } DocumentFieldList;

typedef struct BufferPara
    {
    unsigned char *	btString;
    unsigned int	btStrlen;

    int			btParticuleCount;
    TextParticule *	btParticules;

    int			btObjectCount;
    InsertedObject *	btObjects;

    int			btLineCount;
    TextLine *		btLines;

    DocumentFieldList	btFieldList;

    int			btShapeCount;
    DrawingShape *	btShapes;

    ParagraphProperties	btProperties;
    } BufferPara;

typedef struct BufferGroup
    {
    struct BufferItem **	bgChildren;
    int				bgChildCount;

    union	BGU
	{
	DocumentProperties	bguDocumentProperties;
	SectionProperties	bguSectionProperties;
	RowProperties		bguRowProperties;
	CellProperties		bguCellProperties;
	} BGU;

    } BufferGroup;

/************************************************************************/
/*  Levels of nesting.							*/
/************************************************************************/
typedef enum ItemLevel
    {
    DOClevANY,
    DOClevOUT,
			/****************************************/
			/*  Ignore, Garbage values.		*/
			/****************************************/
    DOClevDOC,
#	define	biDocumentProperties	BIU.biuGroup.BGU.bguDocumentProperties
#	define	biDocContainsTables	biDocumentProperties.dpContainsTables
#	define	biDocTabIntervalTwips	biDocumentProperties.dpTabIntervalTwips
#	define	biDocDefaultColor	biDocumentProperties.dpDefaultColor
#	define	biDocAnsiCodePage	biDocumentProperties.dpAnsiCodePage
			/****************************************/
			/*					*/
			/*  Document.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
    DOClevSECT,
			/****************************************/
			/*					*/
			/*  Section.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/*					*/
			/****************************************/
#	define	biSectProperties	BIU.biuGroup.BGU.bguSectionProperties
#	define	biSectHeader		biSectProperties.spHeader
#	define	biSectFirstPageHeader	biSectProperties.spFirstPageHeader
#	define	biSectLeftPageHeader	biSectProperties.spLeftPageHeader
#	define	biSectRightPageHeader	biSectProperties.spRightPageHeader

#	define	biSectFooter		biSectProperties.spFooter
#	define	biSectFirstPageFooter	biSectProperties.spFirstPageFooter
#	define	biSectLeftPageFooter	biSectProperties.spLeftPageFooter
#	define	biSectRightPageFooter	biSectProperties.spRightPageFooter

			/****************************************/

    DOClevROW,
			/****************************************/
			/*  Row.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
#	define	biRowProperties		BIU.biuGroup.BGU.bguRowProperties
#	define	biRowHasTableParagraphs	biRowProperties.rpHasTableParagraphs
#	define	biRowHalfGapWidthTwips	biRowProperties.rpHalfGapWidthTwips
#	define	biRowHalfGapWidthPixels \
					biRowProperties.rpHalfGapWidthPixels
#	define	biRowLeftIndentTwips	biRowProperties.rpLeftIndentTwips
#	define	biRowLeftIndentPixels	biRowProperties.rpLeftIndentPixels
#	define	biRowHeightTwips	biRowProperties.rpHeightTwips
#	define	biRowHeightPixels	biRowProperties.rpHeightPixels
#	define	biRowIsTableHeader	biRowProperties.rpIsTableHeader
#	define	biRowKeepOnPage		biRowProperties.rpKeepOnPage
#	define	biRowHasHorizontalBorders \
					biRowProperties.rpHasHorizontalBorders
#	define	biRowHasVerticalBorders \
					biRowProperties.rpHasVerticalBorders
#	define	biRowCells		biRowProperties.rpCells
#	define	biRowCellCount		biRowProperties.rpCellCount
#	define	biRowAlignment		biRowProperties.rpAlignment

#	define	biRowTopBorder		biRowProperties.rpTopBorder
#	define	biRowBottomBorder	biRowProperties.rpBottomBorder
#	define	biRowLeftBorder		biRowProperties.rpLeftBorder
#	define	biRowRightBorder	biRowProperties.rpRightBorder
#	define	biRowVerticalBorder	biRowProperties.rpVerticalBorder
#	define	biRowHorizontalBorder	biRowProperties.rpHorizontalBorder

    DOClevCELL,
			/****************************************/
			/*  Cell.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
#	define	biCellProperties	BIU.biuGroup.BGU.bguCellProperties
#	define	biCellRightBoundaryPixels \
					biCellProperties.cpRightBoundaryPixels
#	define	biCellRightBoundaryTwips \
					biCellProperties.cpRightBoundaryTwips
#	define	biCellTopBorder		biCellProperties.cpTopBorder
#	define	biCellBottomBorder	biCellProperties.cpBottomBorder
#	define	biCellLeftBorder	biCellProperties.cpLeftBorder
#	define	biCellRightBorder	biCellProperties.cpRightBorder

    DOClevPARA,
			/********************************/
			/*  Paragraph.			*/
			/********************************/
#	define	biParaString		BIU.biuPara.btString
#	define	biParaStrlen		BIU.biuPara.btStrlen
#	define	biParaParticuleCount	BIU.biuPara.btParticuleCount
#	define	biParaParticules	BIU.biuPara.btParticules
#	define	biParaObjectCount	BIU.biuPara.btObjectCount
#	define	biParaObjects		BIU.biuPara.btObjects
#	define	biParaLineCount		BIU.biuPara.btLineCount
#	define	biParaLines		BIU.biuPara.btLines
#	define	biParaFieldList		BIU.biuPara.btFieldList
#	define	biParaShapeCount	BIU.biuPara.btShapeCount
#	define	biParaShapes		BIU.biuPara.btShapes
#	define	biParaProperties	BIU.biuPara.btProperties
#	define	biParaInTable		biParaProperties.ppInTable
#	define	biParaLineSpacingIsMultiple \
					biParaProperties.ppLineSpacingIsMultiple
#	define	biParaStartsOnNewPage	biParaProperties.ppStartsOnNewPage
#	define	biParaFirstIndentTwips	biParaProperties.ppFirstIndentTwips
#	define	biParaLeftIndentTwips	biParaProperties.ppLeftIndentTwips
#	define	biParaRightIndentTwips	biParaProperties.ppRightIndentTwips
#	define	biParaSpaceBeforeTwips	biParaProperties.ppSpaceBeforeTwips
#	define	biParaSpaceAfterTwips	biParaProperties.ppSpaceAfterTwips
#	define	biParaLineSpacingTwips	biParaProperties.ppLineSpacingTwips
#	define	biParaAlignment		biParaProperties.ppAlignment
#	define	biParaTabStops		biParaProperties.ppTabStops
#	define	biParaTabCount		biParaProperties.ppTabCount
#	define	biParaFirstIndentPixels \
					biParaProperties.ppFirstIndentPixels
#	define	biParaLeftIndentPixels \
					biParaProperties.ppLeftIndentPixels
#	define	biParaRightIndentPixels \
					biParaProperties.ppRightIndentPixels
#	define	biParaSpaceBeforePixels \
					biParaProperties.ppSpaceBeforePixels
#	define	biParaLineSpacingPixels \
					biParaProperties.ppLineSpacingPixels
#	define	biParaSpaceAfterPixels	biParaProperties.ppSpaceAfterPixels
#	define	biParaKeepOnPage	biParaProperties.ppKeepOnPage
#	define	biParaKeepWithNext	biParaProperties.ppKeepWithNext
#	define	biParaShadingLevel	biParaProperties.ppShadingLevel
#	define	biParaShadingPattern	biParaProperties.ppShadingPattern
#	define	biParaOutlineLevel	biParaProperties.ppOutlineLevel
#	define	biParaTopBorder		biParaProperties.ppTopBorder
#	define	biParaBottomBorder	biParaProperties.ppBottomBorder
#	define	biParaLeftBorder	biParaProperties.ppLeftBorder
#	define	biParaRightBorder	biParaProperties.ppRightBorder
#	define	biParaBoxBorder		biParaProperties.ppBoxBorder
#	define	biParaBetweenBorder	biParaProperties.ppBetweenBorder
#	define	biParaBar		biParaProperties.ppBar
#	define	biParaStyle		biParaProperties.ppStyle

    DOClevTEXT
			/********************************/
			/*  Handeled inside TEXT.	*/
			/********************************/
    } ItemLevel;

#   define	biGroupChildren		BIU.biuGroup.bgChildren
#   define	biGroupChildCount	BIU.biuGroup.bgChildCount

typedef struct BufferItem
    {
    ItemLevel		biLevel;
    struct BufferItem *	biParent;
    int			biNumberInParent;

    int			biY0;
    int			biY1;

    union	BIU
	{
	BufferGroup	biuGroup;
	BufferPara	biuPara;
	} BIU;

    } BufferItem;

/************************************************************************/
/*									*/
/*  A position in a document.						*/
/*									*/
/*  1)  Geometry is derived from the position, NOT the other way round.	*/
/*									*/
/************************************************************************/
typedef struct BufferPosition
    {
    BufferItem *	bpBi;
    int			bpLine;
    int			bpParticule;
    int			bpStroff;

    /*  1  */
    int			bpX;
    int			bpY0;
    int			bpY1;
    } BufferPosition;

typedef struct BufferSelection
    {
    BufferPosition	bsBegin;
    BufferPosition	bsEnd;
    BufferPosition	bsAnchor;
    int			bsCol0;
    int			bsCol1;
    int			bsDirection;
    } BufferSelection;

#   define	docPositionsInsideCell(b,e) \
		((b)->bpBi->biParent == (e)->bpBi->biParent)
#   define	docPositionsInsideRow(b,e) \
		((b)->bpBi->biParent->biParent == (e)->bpBi->biParent->biParent)

#   define	docSelectionInsideCell(bs) \
		docPositionsInsideCell(&(bs)->bsBegin,&(bs)->bsEnd)
#   define	docSelectionInsideRow(bs) \
		docPositionsInsideRow(&(bs)->bsBegin,&(bs)->bsEnd)

/************************************************************************/
/*									*/
/*  Styles in a document.						*/
/*									*/
/*  dsBusy is used to prevent recursion when styles are expanded.	*/
/*									*/
/************************************************************************/

typedef struct DocumentStyle
    {
    int			dsStyleNumber;
    int			dsBasedOn;
    int			dsIsCharacterStyle;
    int			dsBusy;
    ParagraphProperties	dsParagraphProperties;
    TextAttribute	dsTextAttribute;
    char *		dsName;
    } DocumentStyle;

/************************************************************************/
/*									*/
/*  The document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct BufferDocument
    {
    BufferItem		bdItem;
    DocumentGeometry	bdGeometry;
    DocumentProperties	bdProperties;
    DocumentFontList	bdFontList;

    RGB8Color *		bdColors;
    int			bdColorCount;

    DocumentStyle *	bdStyles;
    int			bdStyleCount;

    DocumentFieldList	bdBookmarks;

    unsigned char *	bdTitle;
    unsigned char *	bdSubject;
    unsigned char *	bdKeywords;
    unsigned char *	bdComment;
    unsigned char *	bdAuthor;
    } BufferDocument;

#   define	docHasDocumentInfo( bd )	\
		( (bd)->bdTitle		||	\
		  (bd)->bdSubject	||	\
		  (bd)->bdKeywords	||	\
		  (bd)->bdComment	||	\
		  (bd)->bdAuthor	)

typedef void (*DOC_CLOSE_OBJECT)(	BufferDocument *	bd,
					BufferItem *		bi,
					TextParticule *		tp,
					void *			voiddis );

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

extern BufferDocument *	docPlainReadFile(	SimpleInputStream *	sis );

extern BufferDocument *	docRtfReadFile(		SimpleInputStream *	sis );
extern BufferDocument *	docLeafReadFile(	SimpleInputStream *	sis );
extern BufferDocument *	docWordReadFile(	SimpleInputStream *	sis );

extern BufferDocument * docNewFile(	const char *	fontName,
					int		fontSize	);

extern int docRtfSaveDocument(		SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const BufferSelection *	bs,
					int			saveBookmarks );

extern int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const BufferSelection *	bs,
					const char *		fileName,
					int			fileSize,
					const char *		markName,
					int			markSize );

extern int docHtmlSaveDocument(		SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const char *		filename );

extern int docPlainSaveDocument(	SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const BufferSelection *	bs,
					int			fold,
					int			closed );

extern int docRtfSaveRuler(	SimpleOutputStream *		sos,
				const ParagraphProperties *	pp );

extern int docRtfReadRuler(	SimpleInputStream *	sis,
				ParagraphProperties *	pp );

extern int docInflateTextString(	BufferItem *    bi,
					int		by	);

extern TextLine * docInsertTextLine(	BufferItem *	bi,
					int		line,
					int		stroff,
					int		part );

extern void docCleanItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeDocument( BufferDocument *	bd );

extern void docInitDocument(	BufferDocument *	bd	);

extern void docInitParagraphProperties(	ParagraphProperties *	pp );
extern void docInitDocumentProperties(	DocumentProperties *	dp );
extern void docInitSectionProperties(	SectionProperties *	sp );

extern void docInitTabStop(	TabStop *	ts );

extern void docCleanParagraphProperties(	ParagraphProperties *	pp );
extern void docCleanSectionProperties(		SectionProperties *	sp );

extern int docCopyParagraphProperties(	ParagraphProperties *		to,
					const ParagraphProperties *	from );
extern int docCopySectProperties(	SectionProperties *		to,
					const SectionProperties *	from );
extern int docCopyRowProperties(	RowProperties *			to,
					const RowProperties *		from );

extern int docCopyParagraphRuler(	ParagraphProperties *		to,
					const ParagraphProperties *	from );

extern BufferItem * docInsertItem(	BufferItem *	parent,
					int		n,
					ItemLevel	level	);

extern BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
					BufferItem *		biCellTo,
					int			n,
					BufferItem *		biParaFrom,
					const char *		filename );

extern BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
					BufferItem *		biSectTo,
					int			n,
					BufferItem *		biRowFrom,
					const char *		filename );

extern void docDeleteItems(	BufferDocument *	bd,
				BufferItem *		bi,
				int			first,
				int			count	);

extern TextParticule *	docInsertTextParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind,
						TextAttribute	ta	);

extern TextParticule * docCopyParticule(	BufferItem *		bi,
						int			n,
						int			off,
						int			len,
						int			kind,
						const TextParticule *	from );

void docDeleteParticules	(	BufferItem *	bi,
					int		first,
					int		count	);

extern void docDeleteLines	(	BufferItem *	bi,
					int		first,
					int		count	);

extern int docParaAddTab(	ParagraphProperties *	pp,
				const TabStop *		ts	);

extern void docInitPosition(	BufferPosition *	bp	);
extern void docInitSelection(	BufferSelection *	bs	);

extern int docParaReplaceText(	BufferDocument *	bd,
				BufferItem *		bi,
				int			part,
				unsigned int		stroffBegin,
				int *			pPartShift,
				int *			pStroffShift,
				unsigned int		stroffEnd,
				const unsigned char *	addedText,
				unsigned int		addedLength,
				TextAttribute		addedAttribute,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject );

extern int docSplitParaItem(	BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			stroff );

extern int docSplitGroupItem(	BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			n );

extern BufferItem * docNextParagraph(	BufferItem *	bi	);
extern BufferItem * docPrevParagraph(	BufferItem *	bi	);

extern int docNextPosition(	BufferPosition *	bp	);

extern int docPrevPosition(	BufferPosition *	bp,
				int			lastOne );

extern int docNextLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);

extern int docPrevLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);

extern int docFirstPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern int docLastPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern void docSetSelection(	BufferSelection *	bs,
				BufferItem *		bi,
				int			direction,
				int			start,
				int			length	);

extern int docComparePositions(	const BufferPosition *	bp1,
				const BufferPosition *	bp2,
				int			mindLine );

extern int docCompareItemPositions(	const BufferItem *	bi1,
					const BufferItem *	bi2 );

extern TextParticule * docParaSpecialParticule(	BufferItem *	bi,
					int			kind,
					int			part,
					int			stroff,
					int *			pPartShift,
					int *			pStroffShift );

extern int docFindLineAndParticule(	BufferItem *		bi,
					int			stroff,
					BufferPosition *	bp ,
					int			lastOne );

extern int docFindParticule(		BufferItem *		bi,
					int			stroff,
					int *			pPart,
					int			lastOne );

extern void docDeleteItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docListItem(	int			indent,
				const BufferItem *	bi );

extern void docListParticule(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextParticule *	tp );

extern int docReplaceSelection(	BufferDocument *	bd,
				const BufferSelection *	bs,
				int *			pPartShift,
				int *			pStroffShift,
				const unsigned char *	addedText,
				int			addedLength,
				TextAttribute		addedAttribute,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject );

extern void docInitItem(	BufferItem *	bi,
				BufferItem *	parent,
				int		numberInParent,
				int		level	);

extern char *	docKindStr(		int			kind );
extern char *	docLevelStr(		int			level );
extern char *	docAttributeStr(	TextAttribute		ta );

extern TextParticule *	docInsertObject(	BufferItem *		bi,
						InsertedObject **	pIo,
						int			n,
						int			off,
						TextAttribute		ta );

extern int docSetObjectData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docCopyObjectData(	ObjectData *		odTo,
				const ObjectData *	odFrom );

extern int docSetResultData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docSaveObjectTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern int docSaveResultTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern void docCloseItemObjects(	BufferDocument *	bd,
					BufferItem *		bi,
					void *			voiddisplay,
					DOC_CLOSE_OBJECT	closeObject );

extern void docCleanParticuleObject(	BufferItem *	bi,
					TextParticule *	tp );

extern InsertedObject *	docClaimObject(	int *			pNr,
					BufferItem *		bi );

extern InsertedObject * docClaimObjectCopy( BufferItem *	bi,
					int *			pNr,
					const InsertedObject *	ioFrom );

extern int docSetObjectName(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern int docSetObjectClass(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern DocumentField * docClaimField(	int *			pNr,
					DocumentFieldList *	dfl );

extern int docSetFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern void docInitRowProperties(	RowProperties *	rp );
extern void docCleanRowProperties(	RowProperties *	rp );
extern void docCleanInitRowProperties(	RowProperties *	rp );

extern void docInitCellProperties(	CellProperties *	cp );

extern int docInsertRowColumn(	RowProperties *			rp,
				int				n,
				const CellProperties *	cp );

extern void docInitBorderProperties(	BorderProperties *	bp );

extern int docAlignedColumns(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docEqualRows(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docDelimitTable(	BufferItem *		paraBi,
				BufferItem **		pSectBi,
				int *			pCol,
				int *			pRow0,
				int *			pRow,
				int *			pRow1 );

extern void docParagraphFrame(		FormattingFrame *		ff,
					const DocumentGeometry *	dg,
					int				bottom,
					const BufferItem *		bi );

extern void docInitShapeProperties(	ShapeProperties *		sp );


extern int docGetBookmarkForPosition(	const BufferPosition *	bp,
					int *			pPart,
					int *			pEndPart,
					const char **		pMarkName,
					int *			pMarkSize );

extern DocumentField *	docFindBookmarkField( const DocumentFieldList * dfl,
					const unsigned char *	mark,
					int			markSize );

extern int docGetHyperlink(		DocumentField *		df,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docGetPagefield(		DocumentField *		df );

extern int docGetHyperlinkForPosition(	const BufferPosition *	bp,
					int *			pStartPart,
					int *			pEndPart,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docSetHyperlinkDestination(	DocumentField *		df,
					const char *		file,
					const char *		mark );

extern void docInitField(	DocumentField *	df );
extern void docCleanField(	DocumentField *	df );

extern void docCleanFieldList(	DocumentFieldList *	dfl );
extern void docInitFieldList(	DocumentFieldList *	dfl );

DocumentField *	docClaimFieldCopy(	BufferItem *		bi,
					int *			pNr,
					const DocumentField *	dfFrom );

extern int docCopyParticuleData(	BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom );

extern int docCopyParticules(	BufferItem *		biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int			countFrom,
				int *			pParticulesInserted,
				int *			pCharactersCopied,
				const char *		refFileName );

extern int docEqualBorder(	const BorderProperties *	bp1,
				const BorderProperties *	bp2 );

extern void docInitShape(	DrawingShape *	ds );
extern void docCleanShape(	DrawingShape *	ds );

extern DrawingShape * docClaimShape(	int *			pNr,
					BufferItem *		bi );

extern void docInitStyle(	DocumentStyle *	ds );
extern void docCleanStyle(	DocumentStyle *	ds );

extern DocumentStyle * docInsertStyle(	BufferDocument *	bd,
					int			n,
					const char *		name );

extern int docParticuleInField(		BufferItem *	bi,
					int		part );
extern int docParticuleInBookmark(	BufferItem *	bi,
					int		part );

extern int docSaveParticules(	BufferItem *		bi,
				TextAttribute		ta,
				const unsigned char	inputMapping[256],
				const unsigned char *	text,
				int			len );

extern int docSaveTab(		BufferItem *		bi,
				TextAttribute		ta );

extern unsigned int docParaPropertyDifference(
				const ParagraphProperties *	pp1,
				const ParagraphProperties *	pp2,
				unsigned int			updMask );

extern void docLogRectangle(	const char *			label,
				const DocumentRectangle *	dr );

extern int docSubstitutePageNumber(	BufferItem *		bi,
					int			pageNumber );
#   endif
