/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<debugon.h>

#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Manage inserted objects.						*/
/*									*/
/************************************************************************/

static void docCleanObjectData(	ObjectData *	od )
    {
    if  ( od->odBytes )
	{ free( od->odBytes );	}
    }

static void docInitObjectData(	ObjectData *	od )
    {
    od->odBytes= (unsigned char *)0;
    od->odSize= 0;
    }

static void docCleanObject(	InsertedObject *	io )
    {
    docCleanObjectData( &io->ioObjectData );
    docCleanObjectData( &io->ioResultData );

    if  ( io->ioObjectName )
	{ free( io->ioObjectName );	}
    if  ( io->ioObjectClass )
	{ free( io->ioObjectClass );	}
    
    return;
    }

static void docInitObject(	InsertedObject *	io )
    {
    io->ioKind= DOCokUNKNOWN;
    io->ioResultKind= DOCokUNKNOWN;

    io->ioTwipsWide= 0;
    io->ioTwipsHigh= 0;

    io->ioXExtent= 0;
    io->ioYExtent= 0;

    io->ioPixelsWide= 0;
    io->ioPixelsHigh= 0;
    io->ioScaleX= 100;
    io->ioScaleY= 100;

    io->ioUnitsPerInch= 0;

    io->ioDragWide= 0;
    io->ioDragHigh= 0;

    docInitObjectData( &io->ioObjectData );
    docInitObjectData( &io->ioResultData );
    
    io->ioObjectName= (unsigned char *)0;
    io->ioObjectClass= (unsigned char *)0;
    io->ioBliptag= 0;

    io->ioPixmap= 0L;
    io->ioPrivate= (void *)0;

    return;
    }

static int docObjectSetObjectData(	ObjectData *		od,
					const unsigned char *	bytes,
					int			size )
    {
    unsigned char *	fresh;

    fresh= (unsigned char *)realloc( od->odBytes, size+ 1 );
    if  ( ! fresh )
	{ LXDEB(size,fresh); return -1;	}

    od->odBytes= fresh;
    od->odSize= size;

    memcpy( fresh, bytes, size );
    fresh[size]= '\0';

    return 0;
    }

int docCopyObjectData(		ObjectData *		odTo,
				const ObjectData *	odFrom )
    {
    if  ( docObjectSetObjectData( odTo, odFrom->odBytes, odFrom->odSize ) )
	{ LDEB(odFrom->odSize); return -1;	}

    return 0;
    }

int docSetObjectData(	InsertedObject *	io,
			const unsigned char *	bytes,
			int			size )
    { return docObjectSetObjectData( &io->ioObjectData, bytes, size );	}

int docSetResultData(	InsertedObject *	io,
			const unsigned char *	bytes,
			int			size )
    { return docObjectSetObjectData( &io->ioResultData, bytes, size );	}

int docSetObjectName(	InsertedObject *	io,
			const unsigned char *	name,
			int			len )
    {
    unsigned char *	fresh= (unsigned char *)malloc( len+ 1 );

    if  ( ! fresh )
	{ LXDEB(len,fresh); return -1;	}

    io->ioObjectName= fresh;
    memcpy( fresh, name, len ); fresh[len]= '\0';

    return 0;
    }

int docSetObjectClass(	InsertedObject *	io,
			const unsigned char *	name,
			int			len )
    {
    unsigned char *	fresh= (unsigned char *)malloc( len+ 1 );

    if  ( ! fresh )
	{ LXDEB(len,fresh); return -1;	}

    io->ioObjectClass= fresh;
    memcpy( fresh, name, len ); fresh[len]= '\0';

    return 0;
    }

InsertedObject * docClaimObjectCopy(	BufferItem *		bi,
					int *			pNr,
					const InsertedObject *	ioFrom )
    {
    InsertedObject *	ioTo;
    int			objectNumber;

    ioTo= docClaimObject( &objectNumber, bi );
    if  ( ! ioTo )
	{ XDEB(ioTo); return (InsertedObject *)0;	}

    if  ( ioFrom->ioObjectName						&&
	  docSetObjectName( ioTo, ioFrom->ioObjectName,
			    strlen( (char *)ioFrom->ioObjectName ) )	)
	{ LDEB(1); docCleanObject( ioTo ); return (InsertedObject *)0; }

    if  ( ioFrom->ioObjectClass						&&
	  docSetObjectClass( ioTo, ioFrom->ioObjectClass,
			    strlen( (char *)ioFrom->ioObjectClass ) )	)
	{ LDEB(1); docCleanObject( ioTo ); return (InsertedObject *)0; }

    if  ( docCopyObjectData( &ioTo->ioObjectData, &ioFrom->ioObjectData ) ||
	  docCopyObjectData( &ioTo->ioResultData, &ioFrom->ioResultData ) )
	{
	LDEB(1);
	docCleanObject( ioTo ); docInitObject( ioTo );
	return (InsertedObject *)0;
	}

    ioTo->ioKind= ioFrom->ioKind;
    ioTo->ioResultKind= ioFrom->ioResultKind;
    ioTo->ioTwipsWide= ioFrom->ioTwipsWide;
    ioTo->ioTwipsHigh= ioFrom->ioTwipsHigh;
    ioTo->ioScaleX= ioFrom->ioScaleX;
    ioTo->ioScaleY= ioFrom->ioScaleY;
    ioTo->ioPixelsWide= ioFrom->ioPixelsWide;
    ioTo->ioPixelsHigh= ioFrom->ioPixelsHigh;
    ioTo->ioXExtent= ioFrom->ioXExtent;
    ioTo->ioYExtent= ioFrom->ioYExtent;
    ioTo->ioUnitsPerInch= ioFrom->ioUnitsPerInch;

    ioTo->ioBliptag= ioFrom->ioBliptag; /*?*/

    ioTo->ioDragWide= 0;
    ioTo->ioDragHigh= 0;

    return ioTo;
    }

DocumentField *	docClaimFieldCopy(	BufferItem *		bi,
					int *			pNr,
					const DocumentField *	dfFrom )
    {
    DocumentField *	dfTo;
    int			fieldNumber;

    dfTo= docClaimField( &fieldNumber, &(bi->biParaFieldList) );
    if  ( ! dfTo )
	{ XDEB(dfTo); return (DocumentField *)0;	}

    dfTo->dfKind= dfFrom->dfKind;
    dfTo->dfNumberInDocument= dfFrom->dfNumberInDocument;

    if  ( docCopyObjectData( &dfTo->dfData, &dfFrom->dfData ) 		||
	  docCopyObjectData( &dfTo->dfInstructions, &dfFrom->dfInstructions ) )
	{
	LDEB(1);
	docCleanField( dfTo );
	docInitField( dfTo );
	return (DocumentField *)0;
	}

    *pNr= fieldNumber; return dfTo;
    }

/************************************************************************/
/*									*/
/*  Do cleanup provoked by the deletion of a particule.			*/
/*									*/
/************************************************************************/
void docCleanParticuleObject(	BufferItem *	bi,
				TextParticule *	tp )
    {
    InsertedObject *	io;

    if  ( tp->tpKind != DOCkindOBJECT	||
	  tp->tpPhysicalFont != 0	)
	{ return;	}

    io= bi->biParaObjects+ tp->tpObjectNumber;

    docCleanObject( io );
    docInitObject( io );

    return;
    }

/************************************************************************/
/*									*/
/*  Management of shapes.						*/
/*									*/
/************************************************************************/

void docCleanShape(	DrawingShape *	ds )
    {
    return;
    }

void docInitShape(	DrawingShape *	ds )
    {
    ds->ds_shapeType= SHPtyFREE;

    return;
    }

DrawingShape * docClaimShape(	int *			pNr,
				BufferItem *		bi )
    {
    int			n;
    DrawingShape *	ds;

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return (DrawingShape *)0;	}

    ds= (DrawingShape *)realloc( bi->biParaShapes,
			    (bi->biParaShapeCount+ 1)* sizeof(DrawingShape) );
    if  ( ! ds )
	{ XDEB(ds); return (DrawingShape *)0;	}
    bi->biParaShapes= ds;

    for ( n= 0; n < bi->biParaShapeCount; ds++, n++ )
	{
	if  ( ds->ds_shapeType == SHPtyFREE )
	    { break;	}
	}

    docInitShape( ds );

    if  ( n == bi->biParaShapeCount )
	{ bi->biParaShapeCount++;	}

    *pNr= n; return ds;
    }

/************************************************************************/
/*									*/
/*  Management of links.						*/
/*									*/
/************************************************************************/

void docCleanField(	DocumentField *	df )
    {
    docCleanObjectData( &df->dfData );
    docCleanObjectData( &df->dfInstructions );

    return;
    }

void docInitField(	DocumentField *	df )
    {
    docInitObjectData( &df->dfData );
    docInitObjectData( &df->dfInstructions );

    df->dfKind= DOCfkFREE;

    df->dfNumberInDocument= -1;

    return;
    }

int docSetFieldInst(	DocumentField *		df,
			const unsigned char *	bytes,
			int			size )
    { return docObjectSetObjectData( &df->dfInstructions, bytes, size ); }

DocumentField * docClaimField(	int *			pNr,
				DocumentFieldList *	dfl )
    {
    int			n;
    DocumentField *	df;

    df= (DocumentField *)realloc( dfl->dflFields,
			    (dfl->dflFieldCount+ 1)* sizeof(DocumentField) );
    if  ( ! df )
	{ XDEB(df); return (DocumentField *)0;	}
    dfl->dflFields= df;

    for ( n= 0; n < dfl->dflFieldCount; df++, n++ )
	{
	if  ( df->dfKind == DOCfkFREE )
	    { break;	}
	}

    docInitField( df );

    if  ( n == dfl->dflFieldCount )
	{ dfl->dflFieldCount++;	}

    *pNr= n; return df;
    }

void docInitFieldList(	DocumentFieldList *	dfl )
    {
    dfl->dflFields= (DocumentField *)0;
    dfl->dflFieldCount= 0;

    return;
    }

void docCleanFieldList(	DocumentFieldList *	dfl )
    {
    int			i;
    DocumentField *	df;

    df= dfl->dflFields;
    for ( i= 0; i < dfl->dflFieldCount; df++, i++ )
	{
	if  ( df->dfKind == DOCfkFREE )
	    { continue;	}

	docCleanField( df );
	}

    if  ( dfl->dflFields )
	{ free( dfl->dflFields );	}

    return;
    }

DocumentField *	docFindBookmarkField(	const DocumentFieldList * dfl,
					const unsigned char *	mark,
					int			markSize )
    {
    DocumentField *	df;
    int			i;

    df= dfl->dflFields;
    for ( i= 0; i < dfl->dflFieldCount; df++, i++ )
	{
	if  ( df->dfInstructions.odSize == markSize	&&
	      ! memcmp( df->dfInstructions.odBytes, mark, markSize )	)
	    { return df;	}
	}

    return (DocumentField *)0;
    }
				
/************************************************************************/
/*									*/
/*  Style administration.						*/
/*									*/
/************************************************************************/

void docInitStyle(	DocumentStyle *	ds )
    {
    ds->dsStyleNumber= -1;
    ds->dsBasedOn= -1;
    ds->dsIsCharacterStyle= 0;
    ds->dsBusy= 0;

    docInitParagraphProperties( &(ds->dsParagraphProperties) );

    docInitTextAttribute( &(ds->dsTextAttribute) );

    ds->dsName= (char *)0;
    }

void docCleanStyle(	DocumentStyle *	ds )
    {
    docCleanParagraphProperties( &(ds->dsParagraphProperties) );

    if  ( ds->dsName )
	{ free( ds->dsName );	}
    }

DocumentStyle * docInsertStyle(	BufferDocument *	bd,
				int			n,
				const char *		name )
    {
    DocumentStyle *	ds;
    char *		s= (char *)0;

    if  ( n > bd->bdStyleCount )
	{
	ds= (DocumentStyle *)realloc( bd->bdStyles,
		    ( n + 1 ) * sizeof( DocumentStyle ) );
	}
    else{
	ds= (DocumentStyle *)realloc( bd->bdStyles,
		    ( bd->bdStyleCount + 1 ) * sizeof( DocumentStyle ) );
	}
    if  ( ! ds )
	{ LLDEB(bd->bdStyleCount,ds); return ds; }
    bd->bdStyles= ds;

    if  ( name && name[0] )
	{
	s= strdup( name );
	if  ( ! s )
	    { XDEB(s); return (DocumentStyle *)0;	}
	}

    if  ( n == -1 )
	{
	for ( n= 0; n < bd->bdStyleCount; n++ )
	    {
	    if  (  ds[n].dsStyleNumber < 0 )
		{ break;	}
	    }
	}
    else{
	if  ( n < bd->bdStyleCount && ds[n].dsStyleNumber >= 0 )
	    { LLDEB(n,ds[n].dsStyleNumber);	}
	/* No!!
	for ( i= bd->bdStyleCount; i > n; i-- )
	    { ds[i]= ds[i-1];	}
	*/
	}

    while( bd->bdStyleCount < n )
	{ docInitStyle( ds+ bd->bdStyleCount ); bd->bdStyleCount++;	}

    ds += n;
    docInitStyle( ds );

    ds->dsStyleNumber= n;
    ds->dsName= s;

    if  ( n == bd->bdStyleCount )
	{ bd->bdStyleCount++;	}

    return ds;
    }

/************************************************************************/
/*									*/
/*  Free a BufferItem.							*/
/*									*/
/************************************************************************/

static void docFreeGroupChildren(	BufferDocument *	bd,
					BufferItem *		bi )
    {
    int		i;

    for ( i= 0; i < bi->biGroupChildCount; i++ )
	{ docFreeItem( bd, bi->biGroupChildren[i] ); }
    if  ( bi->biGroupChildren )
	{ free( bi->biGroupChildren );	}
    }

void docCleanItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    int			i;
    DocumentField *	paraField;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	    docFreeGroupChildren( bd, bi );
	    break;

	case DOClevSECT:
	    docCleanSectionProperties( &(bi->biSectProperties) );
	    docFreeGroupChildren( bd, bi );
	    break;

	case DOClevCELL:
	    docFreeGroupChildren( bd, bi );
	    docCleanCellProperties( &(bi->biCellProperties) );
	    break;

	case DOClevROW:
	    docFreeGroupChildren( bd, bi );
	    docCleanRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    if  ( bi->biParaString )
		{ free( bi->biParaString );	}
	    if  ( bi->biParaParticules )
		{ free( bi->biParaParticules );	}

	    for ( i= 0; i < bi->biParaObjectCount; i++ )
		{ docCleanObject( bi->biParaObjects+ i ); }

	    paraField= bi->biParaFieldList.dflFields;
	    for ( i= 0; i < bi->biParaFieldList.dflFieldCount; i++ )
		{
		if  ( paraField->dfKind == DOCfkBKMKSTART	&&
		      paraField->dfNumberInDocument >= 0	)
		    {
		    DocumentField *	docField;

		    docField= bd->bdBookmarks.dflFields+
						paraField->dfNumberInDocument;

		    docCleanField( docField );
		    docInitField( docField );
		    }
		}

	    docCleanFieldList( &(bi->biParaFieldList) );

	    if  ( bi->biParaObjects )
		{ free( bi->biParaObjects );	}

	    if  ( bi->biParaLines )
		{ free( bi->biParaLines );	}

	    if  ( bi->biParaShapes )
		{ free( bi->biParaShapes );	}

	    docCleanParagraphProperties( &(bi->biParaProperties) );

	    break;
	default:
	    /*FALLTHROUGH*/
	case DOClevOUT:
	    LDEB(bi->biLevel);
	    break;
	}

    bi->biLevel= DOClevOUT;
    }

void docFreeItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    docCleanItem( bd, bi );
    free( bi );
    }

void docFreeDocument( BufferDocument *	bd )
    {
    DocumentStyle *	ds;
    int			i;

    docCleanItem( bd, &(bd->bdItem) );

    if  ( bd->bdColors )
	{ free( bd->bdColors );	}

    ds= bd->bdStyles;
    for ( i= 0; i < bd->bdStyleCount; ds++, i++ )
	{
	if  ( ds->dsStyleNumber < 0 )
	    { continue;	}

	docCleanStyle( ds );
	}

    if  ( bd->bdStyles )
	{ free( bd->bdStyles );	}

    docCleanFieldList( &(bd->bdBookmarks) );

    if  ( bd->bdTitle )
	{ free( bd->bdTitle );	}
    if  ( bd->bdSubject )
	{ free( bd->bdSubject );	}
    if  ( bd->bdKeywords )
	{ free( bd->bdKeywords );	}
    if  ( bd->bdComment )
	{ free( bd->bdComment );	}
    if  ( bd->bdAuthor )
	{ free( bd->bdAuthor );	}

    free( bd );
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferItem.						*/
/*									*/
/************************************************************************/
void docInitBorderProperties(	BorderProperties *	bp )
    {
    bp->bpColor= 0;
    bp->bpWidthTwips= 15;
    bp->bpSpacingTwips= 0;
    bp->bpStyle= DOCbsHAIR;
    bp->bpIsSet= 0;
    bp->bpThickness= 0;
    }

void docCleanParagraphProperties(	ParagraphProperties *	pp )
    {
    if  ( pp->ppTabStops )
	{ free( pp->ppTabStops );	}
    }

void docCleanSectionProperties(	SectionProperties *	sp )
    {
    if  ( sp->spHeader )
	{ docFreeItem( (BufferDocument *)0, sp->spHeader );		}
    if  ( sp->spFirstPageHeader )
	{ docFreeItem( (BufferDocument *)0, sp->spFirstPageHeader );	}
    if  ( sp->spLeftPageHeader )
	{ docFreeItem( (BufferDocument *)0, sp->spLeftPageHeader );	}
    if  ( sp->spRightPageHeader )
	{ docFreeItem( (BufferDocument *)0, sp->spRightPageHeader );	}

    if  ( sp->spFooter )
	{ docFreeItem( (BufferDocument *)0, sp->spFooter );		}
    if  ( sp->spFirstPageFooter )
	{ docFreeItem( (BufferDocument *)0, sp->spFirstPageFooter );	}
    if  ( sp->spLeftPageFooter )
	{ docFreeItem( (BufferDocument *)0, sp->spLeftPageFooter );	}
    if  ( sp->spRightPageFooter )
	{ docFreeItem( (BufferDocument *)0, sp->spRightPageFooter );	}

    return;
    }

void docInitParagraphProperties(	ParagraphProperties *	pp )
    {
    pp->ppTabCount= 0;
    pp->ppTabStops= (TabStop *)0;

    pp->ppFirstIndentTwips= 0;
    pp->ppLeftIndentTwips= 0;
    pp->ppRightIndentTwips= 0;
    pp->ppSpaceBeforeTwips= 0;
    pp->ppSpaceAfterTwips= 0;
    pp->ppLineSpacingTwips= 0;

    pp->ppFirstIndentPixels= 0;
    pp->ppLeftIndentPixels= 0;
    pp->ppRightIndentPixels= 0;
    pp->ppSpaceBeforePixels= 0;
    pp->ppSpaceAfterPixels= 0;
    pp->ppLineSpacingPixels= 0;

    pp->ppOutlineLevel= 0;
    pp->ppShadingLevel= 0;

    pp->ppShadingPattern= DOCspSOLID;

    pp->ppAlignment= DOCiaLEFT;

    pp->ppStartsOnNewPage= 0;
    pp->ppInTable= 0;
    pp->ppLineSpacingIsMultiple= 0;

    pp->ppKeepOnPage= 0;
    pp->ppKeepWithNext= 0;

    pp->ppStyle= 0;

    docInitBorderProperties( &(pp->ppTopBorder) );
    docInitBorderProperties( &(pp->ppLeftBorder) );
    docInitBorderProperties( &(pp->ppRightBorder) );
    docInitBorderProperties( &(pp->ppBottomBorder) );
    docInitBorderProperties( &(pp->ppBoxBorder) );
    docInitBorderProperties( &(pp->ppBetweenBorder) );
    docInitBorderProperties( &(pp->ppBar) );

    return;
    }

static int docParaSaveTabs(	ParagraphProperties *		to,
				const ParagraphProperties *	from )
    {
    TabStop *	tabs= (TabStop *)0;

    if  ( from->ppTabCount > 0 )
	{
	int	i;

	tabs= (TabStop *)realloc( to->ppTabStops,
					from->ppTabCount* sizeof(TabStop) );
	if  ( ! tabs )
	    { LXDEB(from->ppTabCount,tabs); return -1;	}

	for ( i= 0; i < from->ppTabCount; i++ )
	    { tabs[i]= from->ppTabStops[i];	}
	}


    to->ppTabStops= tabs;
    to->ppTabCount= from->ppTabCount;

    return 0;
    }

int docCopyParagraphRuler(	ParagraphProperties *		to,
				const ParagraphProperties *	from	)
    {
    if  ( docParaSaveTabs( to, from ) )
	{ LDEB(from->ppTabCount); return -1;	}

    to->ppFirstIndentTwips= from->ppFirstIndentTwips;
    to->ppLeftIndentTwips= from->ppLeftIndentTwips;
    to->ppRightIndentTwips= from->ppRightIndentTwips;

    to->ppFirstIndentPixels= from->ppFirstIndentPixels;
    to->ppLeftIndentPixels= from->ppLeftIndentPixels;
    to->ppRightIndentPixels= from->ppRightIndentPixels;

    return 0;
    }

int docCopyParagraphProperties(	ParagraphProperties *		to,
				const ParagraphProperties *	from	)
    {
    TabStop *	tabs;

    if  ( docParaSaveTabs( to, from ) )
	{ LDEB(from->ppTabCount); return -1;	}

    tabs= to->ppTabStops;
    *to= *from;
    to->ppTabStops= tabs;

    return 0;
    }

void docInitSectionProperties(	SectionProperties *	sp )
    {
    sp->spColumnCount= 1;

    sp->spHasFacingPages= 0;
    sp->spHasTitlePage= 0;

    sp->spHeader= (BufferItem *)0;
    sp->spFirstPageHeader= (BufferItem *)0;
    sp->spLeftPageHeader= (BufferItem *)0;
    sp->spRightPageHeader= (BufferItem *)0;

    sp->spFooter= (BufferItem *)0;
    sp->spFirstPageFooter= (BufferItem *)0;
    sp->spLeftPageFooter= (BufferItem *)0;
    sp->spRightPageFooter= (BufferItem *)0;

    return;
    }

void docInitCellProperties(	CellProperties *	cp )
    {
    cp->cpRightBoundaryTwips= 0;
    cp->cpRightBoundaryPixels= 0;

    cp->cpForegroundColor= -1;
    cp->cpBackgroundColor= -1;

    cp->cpShadingLevel= 0;
    cp->cpShadingPattern= DOCspSOLID;

    cp->cpFirstInMergedRange= 0;
    cp->cpMergedWithPrevious= 0;

    cp->cpVerticalTextAlignment= DOCvtaTOP;

    docInitBorderProperties( &(cp->cpTopBorder) );
    docInitBorderProperties( &(cp->cpLeftBorder) );
    docInitBorderProperties( &(cp->cpRightBorder) );
    docInitBorderProperties( &(cp->cpBottomBorder) );
    }

void docCleanInitRowProperties(	RowProperties *	rp )
    {
    docCleanRowProperties( rp );
    docInitRowProperties( rp );
    }

void docCleanRowProperties(	RowProperties *	rp )
    {
    if  ( rp->rpCells )
	{ free( rp->rpCells );	}
    }

void docInitRowProperties(	RowProperties *	rp )
    {
    rp->rpCellCount= 0;
    rp->rpCells= (CellProperties *)0;

    rp->rpHalfGapWidthTwips= 0;
    rp->rpHalfGapWidthPixels= 0;

    rp->rpHeightTwips= 0;
    rp->rpHeightPixels= 0;

    rp->rpLeftIndentTwips= 0;
    rp->rpLeftIndentPixels= 0;

    rp->rpHasTableParagraphs= 0;

    rp->rpIsTableHeader= 0;
    rp->rpKeepOnPage= 0;

    rp->rpHasHorizontalBorders= 0;
    rp->rpHasVerticalBorders= 0;

    docInitBorderProperties( &rp->rpTopBorder );
    docInitBorderProperties( &rp->rpBottomBorder );
    docInitBorderProperties( &rp->rpLeftBorder );
    docInitBorderProperties( &rp->rpRightBorder );
    docInitBorderProperties( &rp->rpHorizontalBorder );
    docInitBorderProperties( &rp->rpVerticalBorder );
    }

int docInsertRowColumn(	RowProperties *			rp,
			int				n,
			const CellProperties *	cp )
    {
    CellProperties *	fresh;

    fresh= (CellProperties *)realloc( rp->rpCells,
			(rp->rpCellCount+ 1)* sizeof(CellProperties) );
    if  ( ! fresh )
	{ LXDEB(rp->rpCellCount,fresh); return -1;	}
    rp->rpCells= fresh;

    if  ( n < 0 )
	{ n= rp->rpCellCount;	}
    else{
	int		i;

	for ( i= rp->rpCellCount; i > n; i-- )
	    { fresh[i]= fresh[i-1];	}
	}

    fresh[n]= *cp;
    rp->rpCellCount++;

    return 0;
    }

int docCopyRowProperties(	RowProperties *		to,
				const RowProperties *	from	)
    {
    CellProperties *	cp= (CellProperties *)0;

    if  ( from->rpCellCount > 0 )
	{
	int	i;

	cp= (CellProperties *)realloc( to->rpCells,
			    from->rpCellCount* sizeof(CellProperties) );
	if  ( ! cp )
	    { LXDEB(from->rpCellCount,cp); return -1;	}

	for ( i= 0; i < from->rpCellCount; i++ )
	    { cp[i]= from->rpCells[i];	}
	}

    *to= *from; to->rpCells= cp;

    return 0;
    }


int docCopySectProperties(	SectionProperties *		to,
				const SectionProperties *	from	)
    {
    LDEB(1); return -1;
    }

void docInitItem(	BufferItem *	bi,
			BufferItem *	parent,
			int		numberInParent,
			int		level	)
    {
    switch( level )
	{
	case DOClevDOC:
	    bi->biGroupChildren= (BufferItem **)0;
	    bi->biGroupChildCount= 0;
	    docInitDocumentProperties( &(bi->biDocumentProperties) );
	    break;

	case DOClevSECT:
	    bi->biGroupChildren= (BufferItem **)0;
	    bi->biGroupChildCount= 0;
	    docInitSectionProperties( &(bi->biSectProperties) );
	    break;

	case DOClevCELL:
	    bi->biGroupChildren= (BufferItem **)0;
	    bi->biGroupChildCount= 0;
	    docInitCellProperties( &(bi->biCellProperties) );
	    break;

	case DOClevROW:
	    bi->biGroupChildren= (BufferItem **)0;
	    bi->biGroupChildCount= 0;
	    docInitRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    bi->biParaStrlen= 0;
	    bi->biParaString= (unsigned char *)0;

	    bi->biParaParticuleCount= 0;
	    bi->biParaParticules= (TextParticule *)0;

	    bi->biParaLineCount= 0;
	    bi->biParaLines= (TextLine *)0;

	    bi->biParaObjectCount= 0;
	    bi->biParaObjects= (InsertedObject *)0;

	    docInitFieldList( &(bi->biParaFieldList) );

	    bi->biParaShapeCount= 0;
	    bi->biParaShapes= (DrawingShape *)0;

	    docInitParagraphProperties( &(bi->biParaProperties) );

	    break;
	default:
	    bi->biLevel= DOClevOUT;
	    bi->biParent= (BufferItem *)0;
	    LDEB(level); return;
	}

    bi->biLevel= (ItemLevel)level;
    bi->biParent= parent;
    bi->biNumberInParent= numberInParent;

    bi->biY0= bi->biY1= -1;

    return;
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferDocument.					*/
/*									*/
/************************************************************************/

void docInitDocumentProperties(	DocumentProperties *	dp	)
    {
    dp->dpContainsTables= 0;
    dp->dpTabIntervalTwips= 720;
    dp->dpDefaultColor= 0;
    dp->dpAnsiCodePage= -1;
    }

void docInitDocument(	BufferDocument *	bd	)
    {
    docInitItem( &(bd->bdItem), (BufferItem *)0, 0, DOClevDOC );
    appInitDocumentGeometry( &(bd->bdGeometry) );
    docInitDocumentProperties( &(bd->bdProperties) );
    docInitFontList( &(bd->bdFontList) );

    bd->bdColors= (RGB8Color *)0;
    bd->bdColorCount= 0;

    bd->bdStyles= (DocumentStyle *)0;
    bd->bdStyleCount= 0;

    docInitFieldList( &(bd->bdBookmarks) );

    bd->bdTitle= (unsigned char *)0;
    bd->bdSubject= (unsigned char *)0;
    bd->bdKeywords= (unsigned char *)0;
    bd->bdComment= (unsigned char *)0;
    bd->bdAuthor= (unsigned char *)0;
    }


/************************************************************************/
/*									*/
/*  Make a new document consisting of one paragraph with one empty	*/
/*  particule.								*/
/*									*/
/************************************************************************/
BufferDocument * docNewFile(	const char *	fontFamilyName,
				int		fontSize	)
    {
    BufferDocument *	bd;
    TextAttribute	ta;

    BufferItem *	bi;

    const char *	fontFamilyStyle= docFontFamilyStyle( fontFamilyName );

    docInitTextAttribute( &ta );
    ta.taFontNumber= 0;
    ta.taFontSizeHalfPoints= fontSize/ 10;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    if  ( ! docInsertFont( &(bd->bdFontList), 0,
					    fontFamilyStyle, fontFamilyName ) )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0;	}

    bi= docInsertItem( &(bd->bdItem), -1, DOClevSECT );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }

    bi= docInsertItem( bi, -1, DOClevROW );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }

    bi= docInsertItem( bi, -1, DOClevCELL );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }

    bi= docInsertItem( bi, -1, DOClevPARA );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }

    if  ( ! docInsertTextParticule( bi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0;	}

    return bd;
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferPosition/BufferSelection.			*/
/*									*/
/************************************************************************/

void docInitPosition(	BufferPosition *	bp )
    {
    bp->bpBi= (BufferItem *)0;
    bp->bpLine= 0;
    bp->bpParticule= -1;
    bp->bpStroff= -1;

    bp->bpX= -1;
    bp->bpY0= -1;
    bp->bpY1= -1;

    return;
    }

void docInitSelection(	BufferSelection *	bs )
    {
    docInitPosition( &(bs->bsBegin) );
    docInitPosition( &(bs->bsEnd) );
    docInitPosition( &(bs->bsAnchor) );

    bs->bsCol0= -1;
    bs->bsCol1= -1;

    bs->bsDirection= 0;

    return;
    }

/************************************************************************/
/*									*/
/*  Make sure there is enough space in the string of a paragraph.	*/
/*									*/
/************************************************************************/
int docInflateTextString(	BufferItem *    bi,
				int		by	)
    {
    unsigned char *	freshString;
    int			newSize;

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    if  ( bi->biParaStrlen % 50 )
	{ newSize= bi->biParaStrlen+ by+  1; }
    else{ newSize= bi->biParaStrlen+ by+ 51; }

    freshString= (unsigned char *)realloc( bi->biParaString, newSize );
    if  ( ! freshString )
	{ LXDEB(bi->biParaStrlen,freshString); return -1; }

    bi->biParaString= freshString;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Add a Particule to a paragraph.					*/
/*									*/
/************************************************************************/
TextParticule * docCopyParticule(	BufferItem *		bi,
					int			n,
					int			off,
					int			len,
					int			kind,
					const TextParticule *	from	)
    {
    TextParticule *	tp;

    TextParticule	scratch;

    scratch= *from;

    tp= docInsertTextParticule( bi, n, off, len, kind, from->tpTextAttribute );

    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpPhysicalFont= scratch.tpPhysicalFont;

    return tp;
    }

static TextParticule *	docInsertParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind )
    {
    TextParticule *	tp;

    tp= (TextParticule *)realloc( bi->biParaParticules,
		( bi->biParaParticuleCount + 1 ) * sizeof( TextParticule ) );
    if  ( ! tp )
	{ LLDEB(bi->biParaParticuleCount,tp); return tp; }
    bi->biParaParticules= tp;

    if  ( n == -1 )
	{ n= bi->biParaParticuleCount;	}
    else{
	int		i;

	for ( i= bi->biParaParticuleCount; i > n; i-- )
	    { tp[i]= tp[i-1];	}
	}

    tp += n;

    docInitTextAttribute( &tp->tpTextAttribute );

    tp->tpStroff= off;
    tp->tpStrlen= len;
    tp->tpKind= kind;
    tp->tpPhysicalFont= -1;

    tp->tpX0= 0;
    tp->tpPixelsWide= 0;

    bi->biParaParticuleCount++;

    return tp;
    }

TextParticule *	docInsertTextParticule(	BufferItem *	bi,
					int		n,
					int		off,
					int		len,
					int		kind,
					TextAttribute	ta	)
    {
    TextParticule *	tp;

    tp= docInsertParticule( bi, n, off, len, kind );
    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpTextAttribute= ta;

    return tp;
    }

InsertedObject *	docClaimObject(	int *			pNr,
					BufferItem *		bi )
    {
    InsertedObject *	io;

    io= (InsertedObject *)realloc( bi->biParaObjects, ( bi->biParaObjectCount+ 1 )*
						    sizeof( InsertedObject ) );
    if  ( ! io )
	{ LXDEB(bi->biParaObjectCount+1,io); return io; }

    bi->biParaObjects= io;
    io += bi->biParaObjectCount;

    docInitObject( io );

    *pNr= bi->biParaObjectCount; return io;
    }

/************************************************************************/
/*									*/
/*  Insert an object.							*/
/*									*/
/*  NOTE that objects have text attributes. This is for the following	*/
/*  reasons:								*/
/*  *)  The calculation of the descender height below the object.	*/
/*  *)  This makes editing around the object and saving it MUCH easier.	*/
/*									*/
/************************************************************************/

TextParticule *	docInsertObject(	BufferItem *		bi,
					InsertedObject **	pIo,
					int			n,
					int			off,
					TextAttribute		ta )
    {
    TextParticule *	tp;
    InsertedObject *	io;
    int			objectNumber;

    io= docClaimObject( &objectNumber, bi );
    if  ( ! io )
	{ LXDEB(bi->biParaObjectCount+1,io); return (TextParticule *)0; }

    tp= docInsertParticule( bi, n, off, 1, DOCkindOBJECT );
    if  ( ! tp )
	{ return tp;	}

    tp->tpTextAttribute= ta;
    tp->tpObjectNumber= bi->biParaObjectCount++;

    *pIo= io;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Delete a series of particules.					*/
/*									*/
/************************************************************************/
void docDeleteParticules(	BufferItem *	bi,
				int		first,
				int		count	)
    {
    if  ( first > bi->biParaParticuleCount )
	{
	LLDEB(first,bi->biParaParticuleCount);
	first= bi->biParaParticuleCount;
	}

    if  ( first+ count > bi->biParaParticuleCount )
	{
	LLDEB(first+count,bi->biParaParticuleCount);
	count= bi->biParaParticuleCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biParaParticuleCount -= count;

    while( first < bi->biParaParticuleCount )
	{
	bi->biParaParticules[first]= bi->biParaParticules[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Delete a series of items.					*/
/*  2)  Delete an item from its parent.					*/
/*									*/
/************************************************************************/

/*  1  */
void docDeleteItems(	BufferDocument *	bd,
			BufferItem *		bi,
			int			first,
			int			count	)
    {
    int		f;
    int		c;

    if  ( first > bi->biGroupChildCount )
	{
	LLDEB(first,bi->biGroupChildCount);
	first= bi->biGroupChildCount;
	}

    if  ( first+ count > bi->biGroupChildCount )
	{
	LLDEB(first+count,bi->biGroupChildCount);
	count= bi->biGroupChildCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biGroupChildCount -= count;

    f= first; c= count;
    while( c > 0 )
	{ docFreeItem( bd, bi->biGroupChildren[f] ); f++; c--; }

    while( first < bi->biGroupChildCount )
	{
	bi->biGroupChildren[first]= bi->biGroupChildren[first+ count];
	bi->biGroupChildren[first]->biNumberInParent= first;

	first++;
	}

    return;
    }

/*  2  */
void docDeleteItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    if  ( bi->biParent )
	{ docDeleteItems( bd, bi->biParent, bi->biNumberInParent, 1 );	}
    else{ docFreeItem( bd, bi );					}
    }

/************************************************************************/
/*									*/
/*  Delete a series of lines.						*/
/*									*/
/************************************************************************/
void docDeleteLines	(	BufferItem *	bi,
				int		first,
				int		count	)
    {
    if  ( first > bi->biParaLineCount )
	{
	LLDEB(first,bi->biParaLineCount);
	first= bi->biParaLineCount;
	}

    if  ( first+ count > bi->biParaLineCount )
	{
	LLDEB(first+count,bi->biParaLineCount);
	count= bi->biParaLineCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biParaLineCount -= count;

    while( first < bi->biParaLineCount )
	{
	bi->biParaLines[first]= bi->biParaLines[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Remember about a line in a Text Item.				*/
/*									*/
/************************************************************************/
TextLine * docInsertTextLine(	BufferItem *	bi,
				int		line,
				int		stroff,
				int		part )
    {
    TextLine *		tl;
    int			newSize;

    if  ( bi->biParaLineCount % 10 )
	{ newSize= bi->biParaLineCount+  1; }
    else{ newSize= bi->biParaLineCount+ 11; }

    newSize *= sizeof(TextLine);

    tl= (TextLine *)realloc( bi->biParaLines, newSize );
    if  ( ! tl )
	{ LXDEB(bi->biParaLineCount,tl); return (TextLine *)0; }
    bi->biParaLines= tl;

    if  ( line == -1 )
	{ line= bi->biParaLineCount; }
    else{
	int		i;

	for ( i= bi->biParaLineCount; i > line; i-- )
	    { tl[i]= tl[i-1];	}
	}

    tl += line;

    tl->tlStroff= stroff;
    tl->tlStrlen= 0;
    tl->tlFirstParticule= part;
    tl->tlParticuleCount= 0;
    tl->tlY= 0;
    tl->tlY0= 0;
    tl->tlY1= 0;

    bi->biParaLineCount++; return tl;
    }

/************************************************************************/
/*									*/
/*  Add a new child to a parent.					*/
/*									*/
/************************************************************************/
BufferItem * docInsertItem(	BufferItem *	parent,
				int		n,
				ItemLevel	level	)
    {
    int			i;

    int *		countTarget;
    BufferItem ***	childTarget;
    int			newSize;

    BufferItem **	freshChildren;
    BufferItem *	bi;

    if  ( parent->biLevel != level- 1 )
	{
	SSDEB(docLevelStr(parent->biLevel),docLevelStr(level));
	return (BufferItem *)0;
	}

    switch( parent->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    countTarget= &(parent->biGroupChildCount);
	    childTarget= &(parent->biGroupChildren);
	    break;
	default:
	    LDEB(parent->biLevel); return (BufferItem *)0;
	}

    newSize= *countTarget;

    if  ( newSize % 10 )
	{ newSize ++;		}
    else{ newSize += 10;	}

    newSize *= sizeof(BufferItem *);

    freshChildren= (BufferItem **)realloc( *childTarget, newSize );
    if  ( ! freshChildren )
	{ LXDEB(newSize,freshChildren); return (BufferItem *)0; }
    *childTarget= freshChildren;

    bi= (BufferItem *)malloc( sizeof(BufferItem) );
    if  ( ! bi )
	{ XDEB(bi); return bi;	}


    if  ( n == -1 )
	{
	n= *countTarget;

	docInitItem( bi, parent, n, level );
	}
    else{
	docInitItem( bi, parent, n, level );

	for ( i= *countTarget; i > n; i-- )
	    {
	    freshChildren[i]= freshChildren[i-1];

	    freshChildren[i]->biNumberInParent= i;
	    }
	}

    freshChildren[n]= bi;

    (*countTarget)++;

    return bi;
    }

/************************************************************************/
/*									*/
/*  Copy a paragraph to make a new one.					*/
/*									*/
/************************************************************************/

BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
				BufferItem *		biCellTo,
				int			n,
				BufferItem *		biParaFrom,
				const char *		refFileName )
    {
    BufferItem *	insBi;

    int			partTo= 0;
    int			partFrom= 0;
    int			particulesInserted= 0;
    int			charactersCopied= 0;

    insBi= docInsertItem( biCellTo, n, DOClevPARA );
    if  ( ! insBi )
	{ XDEB(insBi); return insBi;	}

    if  ( docCopyParagraphProperties( &insBi->biParaProperties,
					&biParaFrom->biParaProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    if  ( docCopyParticules( insBi, biParaFrom,
			partTo, partFrom, biParaFrom->biParaParticuleCount,
			&particulesInserted, &charactersCopied, refFileName ) )
	{
	LDEB(biParaFrom->biParaParticuleCount);
	docDeleteItem( bdTo, insBi ); return (BufferItem *)0;
	}

    return insBi;
    }

BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
				BufferItem *		biSectTo,
				int			n,
				BufferItem *		biRowFrom,
				const char *		fileName )
    {
    BufferItem *	biRowTo;
    int			col;

    biRowTo= docInsertItem( biSectTo, n, DOClevROW );
    if  ( ! biRowTo )
	{ XDEB(biRowTo); return biRowTo;	}

    if  ( docCopyRowProperties( &biRowTo->biRowProperties,
						&biRowFrom->biRowProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    for ( col= 0; col < biRowFrom->biGroupChildCount; col++ )
	{
	BufferItem *	biCellFrom= biRowFrom->biGroupChildren[col];
	BufferItem *	biCellTo= docInsertItem( biRowTo, col, DOClevCELL );
	int		par;

	if  ( ! biCellTo )
	    { XDEB(biCellTo); return biCellTo;	}

	if  ( docCopyCellProperties( &biCellTo->biCellProperties,
					    &biCellFrom->biCellProperties ) )
	    { LDEB(1); return (BufferItem *)0;	}

	for ( par= 0; par < biCellFrom->biGroupChildCount; par++ )
	    {
	    BufferItem *	biParaFrom= biCellFrom->biGroupChildren[par];

	    if  ( ! docCopyParaItem( bdTo, biCellTo,
						par, biParaFrom, fileName ) )
		{ LDEB(par); return (BufferItem *)0;	}
	    }
	}

    return biRowTo;
    }

/************************************************************************/
/*									*/
/*  Set a tab stop in a paragraph.					*/
/*									*/
/************************************************************************/
int docParaAddTab(		ParagraphProperties *	pp,
				const TabStop *		ts	)
    {
    TabStop *		tabs;
    int			newSize;
    int			i;

    if  ( pp->ppTabCount % 10 )
	{ newSize= pp->ppTabCount+  1; }
    else{ newSize= pp->ppTabCount+ 11; }

    newSize *= sizeof(TabStop);

    tabs= (TabStop *)realloc( pp->ppTabStops, newSize );
    if  ( ! tabs )
	{ LXDEB(newSize,tabs); return -1; }
    pp->ppTabStops= tabs;

    for ( i= pp->ppTabCount; i > 0; i-- )
	{
	if  ( tabs[i-1].tsTwips < ts->tsTwips )
	    { break;	}

	tabs[i]= tabs[i-1];
	}

    tabs[i]= *ts;

    pp->ppTabCount++; return 0;
    }

/************************************************************************/
/*									*/
/*  Find the line and particule number for a certain string position.	*/
/*									*/
/*  NOTE:	This expects the paragraph to be formatted.		*/
/*									*/
/************************************************************************/

int docFindLineAndParticule(	BufferItem *		bi,
				int			stroff,
				BufferPosition *	bp,
				int			lastOne )
    {
    int			line= 0;
    int			part= 0;

    TextLine *		tl= bi->biParaLines;
    TextParticule *	tp;

    while( tl->tlStroff+ tl->tlStrlen < stroff	&&
	   line < bi->biParaLineCount		)
	{ line++; tl++; }

    if  ( line >= bi->biParaLineCount )
	{
	if  ( stroff == bi->biParaStrlen )
	    { line= bi->biParaLineCount- 1;	}
	else{
	    LLDEB(stroff,bi->biParaStrlen);
	    LLDEB(line,bi->biParaLineCount); return -1;
	    }
	}

    while( lastOne				&&
	   line < bi->biParaLineCount -1	&&
	   tl->tlStroff+ tl->tlStrlen == stroff	)
	{ line++; tl++; }

    part= tl->tlFirstParticule; tp= bi->biParaParticules+ part;
    while( tp->tpStroff+ tp->tpStrlen < stroff	&&
	   part < bi->biParaParticuleCount	)
	{ part++; tp++;	}

    if  ( part >= bi->biParaParticuleCount )
	{
	LLDEB(line,bi->biParaLineCount);
	LLDEB(stroff,bi->biParaStrlen);
	LLDEB(part,bi->biParaParticuleCount);
	docListItem( 0, bi );
	return -1;
	}

    while( lastOne				&&
	   part < bi->biParaParticuleCount -1	&&
	   tp->tpStroff+ tp->tpStrlen == stroff	)
	{ part++; tp++;	}

    bp->bpBi= bi;
    bp->bpLine= line;
    bp->bpParticule= part;
    bp->bpStroff= stroff;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the particule number for a certain string position.		*/
/*									*/
/*  NOTE:	This does not expect the paragraph to be formatted.	*/
/*									*/
/************************************************************************/

int docFindParticule(		BufferItem *		bi,
				int			stroff,
				int *			pPart,
				int			lastOne )
    {
    int			part= 0;

    TextParticule *	tp;

    part= 0; tp= bi->biParaParticules+ part;
    while( tp->tpStroff+ tp->tpStrlen < stroff	&&
	   part < bi->biParaParticuleCount	)
	{ part++; tp++;	}

    if  ( part >= bi->biParaParticuleCount )
	{
	LLDEB(stroff,bi->biParaStrlen);
	LLDEB(part,bi->biParaParticuleCount);
	docListItem( 0, bi );
	return -1;
	}

    while( lastOne				&&
	   part < bi->biParaParticuleCount -1	&&
	   tp->tpStroff+ tp->tpStrlen == stroff	)
	{ part++; tp++;	}

    *pPart= part;

    return 0;
    }

/************************************************************************/
/*									*/
/*  1) Are the columns in two RowProperties 'the same' (Do they align?)	*/
/*  2) All column properties identical?					*/
/*									*/
/************************************************************************/

int docEqualBorder(	const BorderProperties *	bp1,
			const BorderProperties *	bp2 )
    {
    if  ( ! bp1->bpIsSet && ! bp2->bpIsSet )
	{ return 1;	}

    if  ( bp1->bpIsSet != bp2->bpIsSet )
	{ return 0;	}

    if  ( bp1->bpStyle != bp2->bpStyle )
	{ return 0;	}

    if  ( bp1->bpThickness != bp2->bpThickness )
	{ return 0;	}

    if  ( bp1->bpSpacingTwips != bp2->bpSpacingTwips )
	{ return 0;	}

    if  ( bp1->bpWidthTwips != bp2->bpWidthTwips )
	{ return 0;	}

    if  ( bp1->bpColor != bp2->bpColor )
	{ return 0;	}

    return 1;
    }

int docAlignedColumns(	const RowProperties *	rp1,
			const RowProperties *	rp2 )
    {
    CellProperties *	cp1;
    CellProperties *	cp2;
    int			i;

    if  ( rp1->rpHasTableParagraphs != rp2->rpHasTableParagraphs )
	{ return 0;	}

    if  ( rp1->rpCellCount != rp2->rpCellCount )
	{ return 0;	}

    if  ( rp1->rpHalfGapWidthTwips != rp2->rpHalfGapWidthTwips )
	{ return 0;	}

    if  ( rp1->rpLeftIndentTwips != rp2->rpLeftIndentTwips )
	{ return 0;	}

    cp1= rp1->rpCells;
    cp2= rp2->rpCells;
    for ( i= 0; i < rp1->rpCellCount; cp2++, cp1++, i++ )
	{
	if  ( cp1->cpRightBoundaryTwips != cp2->cpRightBoundaryTwips )
	    { return 0;	}
	}

    return 1;
    }

int docEqualRows(	const RowProperties *	rp1,
			const RowProperties *	rp2 )
    {
    const CellProperties *	cp1;
    const CellProperties *	cp2;
    int				i;

    if  ( rp1->rpCellCount != rp2->rpCellCount )
	{ return 0;	}

    if  ( rp1->rpHalfGapWidthTwips != rp2->rpHalfGapWidthTwips )
	{ return 0;	}

    if  ( rp1->rpLeftIndentTwips != rp2->rpLeftIndentTwips )
	{ return 0;	}

    if  ( rp1->rpHeightTwips != rp2->rpHeightTwips )
	{ return 0;	}

    if  ( rp1->rpIsTableHeader != rp2->rpIsTableHeader )
	{ return 0;	}

    if  ( rp1->rpHasVerticalBorders != rp2->rpHasVerticalBorders )
	{ return 0;	}
    if  ( rp1->rpHasHorizontalBorders != rp2->rpHasHorizontalBorders )
	{ return 0;	}

    if  ( ! docEqualBorder( &rp1->rpTopBorder, &rp2->rpTopBorder )	)
	{ return 0;	}
    if  ( ! docEqualBorder( &rp1->rpLeftBorder, &rp2->rpLeftBorder )	)
	{ return 0;	}
    if  ( ! docEqualBorder( &rp1->rpRightBorder, &rp2->rpRightBorder )	)
	{ return 0;	}
    if  ( ! docEqualBorder( &rp1->rpBottomBorder, &rp2->rpBottomBorder ) )
	{ return 0;	}
    if  ( rp1->rpHasVerticalBorders					&&
	  ! docEqualBorder( &rp1->rpVerticalBorder, &rp2->rpVerticalBorder ) )
	{ return 0;	}
    if  ( rp1->rpHasHorizontalBorders					&&
	  ! docEqualBorder( &rp1->rpHorizontalBorder, &rp2->rpHorizontalBorder))
	{ return 0;	}

    cp1= rp1->rpCells;
    cp2= rp2->rpCells;
    for ( i= 0; i < rp1->rpCellCount; cp2++, cp1++, i++ )
	{
	if  ( cp1->cpRightBoundaryTwips != cp2->cpRightBoundaryTwips )
	    { return 0;	}
	if  ( cp1->cpFirstInMergedRange != cp2->cpFirstInMergedRange )
	    { return 0;	}
	if  ( cp1->cpMergedWithPrevious != cp2->cpMergedWithPrevious )
	    { return 0;	}

	if  ( ! docEqualBorder( &cp1->cpTopBorder, &cp2->cpTopBorder )	)
	    { return 0;	}
	if  ( ! docEqualBorder( &cp1->cpLeftBorder, &cp2->cpLeftBorder ) )
	    { return 0;	}
	if  ( ! docEqualBorder( &cp1->cpRightBorder, &cp2->cpRightBorder ) )
	    { return 0;	}
	if  ( ! docEqualBorder( &cp1->cpBottomBorder, &cp2->cpBottomBorder ) )
	    { return 0;	}
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Derive the frame for a paragraph from the page rectangle and the	*/
/*  paragraph properties.						*/
/*									*/
/*  For paragraphs inside a table cell, geometry is derived from the	*/
/*  table column.							*/
/*									*/
/************************************************************************/

void docParagraphFrame(		FormattingFrame *		ff,
				const DocumentGeometry *	dg,
				int				bottom,
				const BufferItem *		bi )
    {
    ff->ffX0Geometry= dg->dgLeftMarginTwips;
    ff->ffX1Geometry= dg->dgPaperWideTwips- dg->dgRightMarginTwips;

    if  ( bottom > 0							&&
	  bottom <= dg->dgPaperHighTwips- dg->dgBottomMarginTwips	)
	{ ff->ffY1= bottom;						}
    else{ ff->ffY1= dg->dgPaperHighTwips- dg->dgBottomMarginTwips;	}

    if  ( bi->biParaInTable )
	{
	int			no;
	BufferItem *		cell= bi->biParent;
	BufferItem *		row= cell->biParent;
	CellProperties *	cp;

	cp= &(cell->biCellProperties);
	ff->ffX1Geometry= dg->dgLeftMarginTwips+ cp->cpRightBoundaryTwips;

	no= cell->biNumberInParent;

	if  ( no == 0 )
	    { ff->ffX0Geometry += row->biRowLeftIndentTwips;	}
	else{
	    BufferItem *		prevCell;

	    prevCell= row->biGroupChildren[no- 1];

	    cp= &(prevCell->biCellProperties);
	    ff->ffX0Geometry += cp->cpRightBoundaryTwips;
	    }

	ff->ffX0Geometry += row->biRowHalfGapWidthTwips;
	}

    ff->ffX0TextLines= ff->ffX0Geometry+ bi->biParaLeftIndentTwips;
    ff->ffX1TextLines= ff->ffX1Geometry- bi->biParaRightIndentTwips;

    ff->ffX0FirstLine= ff->ffX0TextLines+ bi->biParaFirstIndentTwips;

    return;
    }

void docInitTabStop(	TabStop *	ts )
    {
    ts->tsTwips= 0;
    ts->tsPixels= 0;
    ts->tsKind= DOCtkLEFT;
    ts->tsLeader= DOCtlNONE;
    }

/************************************************************************/
/*									*/
/*  See whether the field instructions are to an hyperlink.		*/
/*									*/
/*  Ted only supports bookmarks inside paragraphs.			*/
/*									*/
/************************************************************************/

int docGetBookmarkForPosition(		const BufferPosition *	bp,
					int *			pPart,
					int *			pEndPart,
					const char **		pMarkName,
					int *			pMarkSize )
    {
    BufferItem *	bi= bp->bpBi;
    int			beginPart;
    int			endPart;
    TextParticule *	tp;

    DocumentField *	dfBegin= (DocumentField *)0;
    DocumentField *	dfEnd= (DocumentField *)0;

    beginPart= bp->bpParticule;
    tp= bi->biParaParticules+ beginPart;
    while( beginPart >= 0 )
	{
	if  ( tp->tpKind == DOCkindBKMKEND	&&
	      tp->tpStroff != bp->bpStroff	)
	    { return -1;	}

	if  ( tp->tpKind == DOCkindFIELDSTART	||
	      tp->tpKind == DOCkindFIELDEND	)
	    { return -1;	}

	if  ( tp->tpKind == DOCkindBKMKSTART )
	    {
	    dfBegin= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	    break;
	    }

	tp--; beginPart--;
	}

    if  ( beginPart < 0 )
	{ return -1;	}

    endPart= bp->bpParticule;
    tp= bi->biParaParticules+ endPart;
    while( endPart < bi->biParaParticuleCount )
	{
	if  ( tp->tpKind == DOCkindBKMKEND )
	    {
	    dfEnd= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( dfBegin->dfInstructions.odSize ==
				    dfEnd->dfInstructions.odSize	&&
		  ! memcmp( dfBegin->dfInstructions.odBytes,
				    dfEnd->dfInstructions.odBytes,
				    dfEnd->dfInstructions.odSize )	)
		{
		*pPart= beginPart; *pEndPart= endPart;
		*pMarkName= (char *)dfBegin->dfInstructions.odBytes;
		*pMarkSize= dfBegin->dfInstructions.odSize;
		return 0;
		}
	    }

	tp++; endPart++;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  See whether the field instructions are to an hyperlink.		*/
/*									*/
/*  Return the file name and the bookmark.				*/
/*  The strings are NOT '\0' terminated.				*/
/*									*/
/************************************************************************/

int docGetHyperlink(		DocumentField *		df,
				const char **		pFileName,
				int *			pFileSize,
				const char **		pMarkName,
				int *			pMarkSize )
    {
    unsigned char *	bytes= df->dfInstructions.odBytes;
    int			byteCount= df->dfInstructions.odSize;

    char *		fileName= (char *)0;
    int			fileSize= 0;
    char *		markName= (char *)0;
    int			markSize= 0;

    while( byteCount > 0 && *bytes == ' ' )
	{ byteCount--; bytes++;	}

    if  ( byteCount > 10						&&
	  ( ! strncmp( (char *)bytes, "HYPERLINK ", 10 )	||
	    ! strncmp( (char *)bytes, "hyperlink ", 10 )	)	)
	{
	bytes += 10; byteCount -= 10;

	while( byteCount > 0 && *bytes == ' ' )
	    { byteCount--; bytes++;	}

	if  ( byteCount < 2 )
	    {
	    LSDEB(byteCount,(char *)df->dfInstructions.odBytes);
	    return -1;
	    }

	if  ( bytes[0] == '"' )
	    {
	    byteCount--; bytes++; fileName= (char *)bytes;

	    while( byteCount > 0 && *bytes != '"' )
		{ fileSize++; byteCount--; bytes++;	}

	    if  ( byteCount == 0 )
		{ SDEB((char *)df->dfInstructions.odBytes); return -1;	}

	    byteCount--; bytes++;

	    while( byteCount > 0 && *bytes == ' ' )
		{ byteCount--; bytes++;	}
	    }
	else{
	    if  ( isalnum( bytes[0] ) )
		{
		fileName= (char *)bytes;

		while( byteCount > 0 && *bytes && *bytes != ' ' )
		    { fileSize++; byteCount--; bytes++;	}

		if  ( byteCount == 0 )
		    { LDEB(df->dfInstructions.odBytes); return -1;	}

		while( byteCount > 0 && *bytes == ' ' )
		    { byteCount--; bytes++;	}
		}
	    }

	if  ( byteCount > 0 && ! strncmp( (char *)bytes, "\\l ", 3 ) )
	    {
	    bytes += 2; byteCount -= 2;

	    while( byteCount > 0 && *bytes == ' ' )
		{ byteCount--; bytes++;	}

	    if  ( byteCount < 2 )
		{
		LSDEB(byteCount,(char *)df->dfInstructions.odBytes);
		return -1;
		}

	    if  ( bytes[0] == '"' )
		{
		byteCount--; bytes++; markName= (char *)bytes;

		while( byteCount > 0 && *bytes != '"' )
		    { markSize++; byteCount--; bytes++;	}

		if  ( byteCount == 0 )
		    { SDEB((char *)df->dfInstructions.odBytes); return -1; }

		byteCount--; bytes++;

		while( byteCount > 0 && *bytes == ' ' )
		    { byteCount--; bytes++;	}
		}
	    else{
		markName= (char *)bytes;

		while( byteCount > 0 && *bytes && *bytes != ' ' )
		    { markSize++; byteCount--; bytes++;	}

		while( byteCount > 0 && *bytes == ' ' )
		    { byteCount--; bytes++;	}
		}
	    }

	if  ( byteCount > 0 )
	    { SDEB((char *)bytes); }

	*pFileName= fileName;
	*pFileSize= fileSize;
	*pMarkName= markName;
	*pMarkSize= markSize;

	return 0;
	}

    return -1;
    }

int docGetPagefield(		DocumentField *		df )
    {
    unsigned char *	bytes= df->dfInstructions.odBytes;
    int			byteCount= df->dfInstructions.odSize;

    while( byteCount > 0 && *bytes == ' ' )
	{ byteCount--; bytes++;	}

    if  ( byteCount >= 4					&&
	  ( ! strncmp( (char *)bytes, "PAGE ", 4 )	||
	    ! strncmp( (char *)bytes, "page ", 4 )	)	)
	{
	bytes += 4; byteCount -= 4;

	while( byteCount > 0 && *bytes == ' ' )
	    { byteCount--; bytes++;	}

	if  ( byteCount > 0 )
	    { /*pageref*/ return -1;	}

	return 0;
	}

    return -1;
    }

int docGetHyperlinkForPosition(		const BufferPosition *	bp,
					int *			pStartPart,
					int *			pEndPart,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize )
    {
    BufferItem *	bi= bp->bpBi;
    int			startPart= bp->bpParticule;
    TextParticule *	tp= bi->biParaParticules+ startPart;

    while( startPart >= 0 )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( ! docGetHyperlink( df,
				pFileName, pFileSize, pMarkName, pMarkSize ) )
		{
		int	endPart= startPart+ 1;

		tp++;

		while( endPart < bi->biParaParticuleCount )
		    {
		    if  ( tp->tpKind == DOCkindFIELDEND )
			{ break;	}

		    endPart++; tp++;
		    }

		if  ( endPart <  bi->biParaParticuleCount )
		    { *pStartPart= startPart; *pEndPart= endPart; return 0; }
		else{ LLDEB(endPart,bi->biParaParticuleCount); return -1; }
		}
	    }

	if  ( tp->tpKind == DOCkindFIELDEND	||
	       tp->tpKind == DOCkindBKMKEND	)
	    { return -1;	}

	tp--; startPart--;
	}

    return -1;
    }

int docSetHyperlinkDestination(	DocumentField *		df,
				const char *		file,
				const char *		mark )
    {
    int			size;
    char *		fresh;
    char *		to;

    size= 11;
    if  ( file )
	{ size += 1+ strlen( file )+ 2;	}
    if  ( mark )
	{ size += 4+ strlen( mark )+ 2;	}

    fresh= to= (char *)malloc( size+ 1 );
    if  ( ! fresh )
	{ LXDEB(size,fresh); return -1;	}

    strcpy( to, " HYPERLINK " ); to += strlen( to );
    if  ( file && file[0] )
	{
	*(to++)= '"';
	strcpy( to, file ); to += strlen( to );
	strcpy( to, "\" " ); to += strlen( to );
	}

    if  ( mark && mark[0] )
	{
	strcpy( to, "\\l \"" ); to += strlen( to );
	strcpy( to, mark ); to += strlen( to );
	strcpy( to, "\" " ); to += strlen( to );
	}

    size= strlen( fresh );

    if  ( docSetFieldInst( df, (unsigned char *)fresh, size ) )
	{ LDEB(size); free( fresh ); return -1;	}

    free( fresh ); return 0;
    }


/************************************************************************/
/*									*/
/*  Copy a hyperlink to another paragraph.				*/
/*									*/
/*  If the hyperlink is relative to the name of the target document, it */
/*  it is converted to a relative link (the document name is removed)	*/
/*									*/
/************************************************************************/

static int docCopyFieldRelative(	BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom,
					const char *		refFileName,
					int			refFileSize )
    {
    DocumentField *	dfFrom;

    const char *	fileName;
    int			fileSize;
    const char *	markName;
    int			markSize;

    dfFrom= biFrom->biParaFieldList.dflFields+ tpFrom->tpObjectNumber;

    if  ( ! docGetHyperlink( dfFrom, &fileName, &fileSize,
						&markName, &markSize )	&&
	  fileSize == refFileSize 					&&
	  markSize > 0							&&
	  ! strncmp( fileName, refFileName, refFileSize )		)
	{
	DocumentField *	dfTo;
	char *		copy= (char *)malloc( markSize+ 1 );
	int		fieldNumber;

	if  ( ! copy )
	    { XDEB(copy); return -1;	}

	strncpy( copy, markName, markSize )[markSize]= '\0';

	dfTo= docClaimField( &fieldNumber, &(biTo->biParaFieldList) );
	if  ( ! dfTo )
	    { XDEB(dfTo); return -1;	}

	if  ( docSetHyperlinkDestination( dfTo, (char *)0, copy ) )
	    { SDEB(copy); return -1;	}

	dfTo->dfKind= dfFrom->dfKind;
	dfTo->dfNumberInDocument= dfFrom->dfNumberInDocument;
	tpTo->tpObjectNumber= fieldNumber;

	free( copy );

	return 0;
	}

    if  ( docCopyParticuleData( biTo, biFrom, tpTo, tpFrom ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Copy particules plus contents from one paragraph to another.	*/
/*									*/
/*  1)  Claim sufficient memory for the data and for the particules.	*/
/*  2)  Insert the new text into the paragraph.				*/
/*  3)  Insert the new particules into the paragraph.			*/
/*									*/
/*  4)  As this routine is the engine of the 'Paste' mechanism, make	*/
/*	links to the text itself relative.				*/
/*									*/
/************************************************************************/

int docCopyParticules(	BufferItem *		biTo,
			const BufferItem *	biFrom,
			int			partTo,
			int			partFrom,
			int			countFrom,
			int *			pParticulesInserted,
			int *			pCharactersCopied,
			const char *		refFileName )
    {
    TextParticule *		tp;
    TextParticule *		tpTo;
    const TextParticule *	tpFrom;

    int				size;
    int				stroffTo;

    int				i;

    int				replacedEmpty= 0;

    int				refFileSize= 0;

    if  ( refFileName )
	{ refFileSize= strlen( refFileName );	}

    /*  1  */
    size= ( biTo->biParaParticuleCount+ countFrom ) * sizeof( TextParticule );
    tp= (TextParticule *)realloc( biTo->biParaParticules, size );
    if  ( ! tp )
	{ LXDEB(size,tp); return -1;	}
    biTo->biParaParticules= tp;

    /*  2  */
    if  ( biTo->biParaStrlen == 0 && biTo->biParaParticuleCount == 1 )
	{ biTo->biParaParticuleCount--; replacedEmpty= 1;	}

    tpFrom= biFrom->biParaParticules+ partFrom;
    size= tpFrom[countFrom-1].tpStroff+ tpFrom[countFrom-1].tpStrlen-
							    tpFrom->tpStroff;

    if  ( docInflateTextString( biTo, size ) )
	{ LLDEB(biTo->biParaStrlen,size); return -1;	}

    if  ( partTo >= biTo->biParaParticuleCount )
	{
	partTo= biTo->biParaParticuleCount;
	stroffTo= biTo->biParaStrlen;
	}
    else{
	tp= biTo->biParaParticules+ partTo;
	stroffTo= tp->tpStroff;
	}

    if  ( biTo->biParaStrlen > 0 )
	{
	memmove( biTo->biParaString+ stroffTo+ size,
			    biTo->biParaString+ stroffTo,
			    biTo->biParaStrlen- stroffTo );
	}
    memcpy( biTo->biParaString+ stroffTo,
			    biFrom->biParaString+ tpFrom->tpStroff, size );
    biTo->biParaString[biTo->biParaStrlen+ size]= '\0';
			    
    /*  3  */
    tp= biTo->biParaParticules;
    for ( i= biTo->biParaParticuleCount- 1; i >= partTo; i-- )
	{
	tp[i+countFrom]= tp[i];
	tp[i+countFrom].tpStroff += size;
	}

    tpTo= tp+ partTo;
    for ( i= 0; i < countFrom; tpTo++, tpFrom++, i++ )
	{
	*(tpTo)= *(tpFrom);
	tpTo->tpStroff= stroffTo;

	/*  4  */
	if  ( ! refFileName || tpFrom->tpKind != DOCkindFIELDSTART )
	    {
	    if  ( docCopyParticuleData( biTo, biFrom, tpTo, tpFrom ) )
		{ LLDEB(partTo,i); return -1;	}
	    }
	else{
	    if  ( docCopyFieldRelative( biTo, biFrom, tpTo, tpFrom,
						refFileName, refFileSize ) )
		{ LLDEB(partTo,i); return -1;	}
	    }

	stroffTo += tpTo->tpStrlen;
	}

    biTo->biParaParticuleCount += countFrom;
    biTo->biParaStrlen += size;

    *pParticulesInserted += countFrom- replacedEmpty;
    *pCharactersCopied += size;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Copy particule data from one paragraph to another.			*/
/*									*/
/*  9)  Not needed, the document is marked as changed, so this can wait	*/
/*	until it is saved.						*/
/*									*/
/************************************************************************/

int docCopyParticuleData(	BufferItem *		biTo,
				const BufferItem *	biFrom,
				TextParticule *		tpTo,
				const TextParticule *	tpFrom )
    {
    int		nr;

    switch( tpFrom->tpKind )
	{
	case DOCkindTEXT:
	case DOCkindTAB:
	    break;
	case DOCkindOBJECT:
	    {
	    InsertedObject *	io;

	    io= docClaimObjectCopy( biTo, &nr,
			biFrom->biParaObjects+ tpFrom->tpObjectNumber );
	    if  ( ! io )
		{ XDEB(io); return -1;	}

	    tpTo->tpObjectNumber= biTo->biParaObjectCount++;

	    /*  9
	    if  ( io->ioBliptag == 0 )
		{ io->ioBliptag= appGetTimestamp();	}
	    */
	    }
	    break;

	case DOCkindFIELDEND:
	    tpTo->tpObjectNumber= tpFrom->tpObjectNumber;
	    break;

	case DOCkindFIELDSTART:
	case DOCkindBKMKSTART:
	case DOCkindBKMKEND:
	case DOCkindXE:
	case DOCkindTC:
	    {
	    DocumentField *	df;

	    df= docClaimFieldCopy( biTo, &nr,
		    biFrom->biParaFieldList.dflFields+ tpFrom->tpObjectNumber );
	    if  ( ! df )
		{ XDEB(df); return -1;	}

	    tpTo->tpObjectNumber= nr;
	    }
	    break;
	default:
	    LDEB(tpFrom->tpKind); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save paragraph contents for readers.				*/
/*									*/
/************************************************************************/

int docSaveParticules(	BufferItem *		bi,
			TextAttribute		ta,
			const unsigned char	inputMapping[256],
			const unsigned char *	text,
			int			len	)
    {
    while( len > 0 )
	{
	const unsigned char *	particule= text;
	int			particuleLength= 0;
	unsigned char *		to;
	int			i;

	int	off= bi->biParaStrlen;

	while( len > 0 && text[0] != ' ' )
	    { text++; particuleLength++; len--; }
	while( len > 0 && text[0] == ' ' )
	    { text++; particuleLength++; len--; }

	/*
	DEBFUN( "%s(%3d): \"%.*s\"\n",
			    __FILE__, __LINE__, particuleLength, particule );
	*/

	if  ( docInflateTextString( bi, particuleLength ) )
	    { LLDEB(bi->biParaStrlen,particuleLength); return -1; }

	to= bi->biParaString+ off;
	for ( i= 0; i < particuleLength; i++ )
	    { *(to++)= inputMapping[*(particule++)];	}
	*to= '\0';

	bi->biParaStrlen += particuleLength;

	if  ( ! docInsertTextParticule( bi, -1, off, bi->biParaStrlen- off,
							DOCkindTEXT, ta ) )
	    { LDEB(bi->biParaParticuleCount); return -1;	}
	}

    return 0;
    }

int docSaveTab(		BufferItem *		bi,
			TextAttribute		ta )
    {
    if  ( docInflateTextString( bi, 1 ) )
	{ LDEB(bi->biParaStrlen); return -1;	}

    if  ( ! docInsertTextParticule( bi, -1, bi->biParaStrlen, 1,
							    DOCkindTAB, ta ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    bi->biParaString[bi->biParaStrlen++]= ' ';
    bi->biParaString[bi->biParaStrlen  ]= '\0';

    return 0;
    }

/************************************************************************/
/*									*/
/*  Compare paragraph properties.					*/
/*									*/
/************************************************************************/

unsigned int docParaPropertyDifference(	const ParagraphProperties *	pp1,
					const ParagraphProperties *	pp2,
					unsigned int			updMask)
    {
    unsigned int		changedMask= 0;

    if  ( updMask & PPupdFIRST_INDENT )
	{
	if  ( pp1->ppFirstIndentTwips != pp2->ppFirstIndentTwips )
	    { changedMask |= PPupdFIRST_INDENT; }
	}

    if  ( updMask & PPupdLEFT_INDENT )
	{
	if  ( pp1->ppLeftIndentTwips != pp2->ppLeftIndentTwips )
	    { changedMask |= PPupdLEFT_INDENT; }
	}

    if  ( updMask & PPupdRIGHT_INDENT )
	{
	if  ( pp1->ppRightIndentTwips != pp2->ppRightIndentTwips )
	    { changedMask |= PPupdRIGHT_INDENT; }
	}

    if  ( updMask & PPupdTAB_STOPS )
	{
	if  ( pp1->ppTabCount > 0 || pp2->ppTabCount > 0 )
	    {
	    if  ( pp1->ppTabCount != pp2->ppTabCount )
		{ changedMask |= PPupdTAB_STOPS;	}
	    else{
		TabStop *			ts1= pp1->ppTabStops;
		TabStop *			ts2= pp2->ppTabStops;
		int				tab;

		for ( tab= 0; tab < pp2->ppTabCount; tab++ )
		    {
		    if  ( ts1[tab].tsTwips != ts2[tab].tsTwips		||
			  ts1[tab].tsKind != ts2[tab].tsKind		||
			  ts1[tab].tsLeader != ts2[tab].tsLeader	)
			{ changedMask |= PPupdTAB_STOPS; }
		    }
		}
	    }
	}

    if  ( updMask & PPupdALIGNMENT )
	{
	if  ( pp1->ppAlignment != pp2->ppAlignment )
	    { changedMask |= PPupdALIGNMENT; }
	}

    if  ( updMask & PPupdSPACE_BEFORE )
	{
	if  ( pp1->ppSpaceBeforeTwips != pp2->ppSpaceBeforeTwips )
	    { changedMask |= PPupdSPACE_BEFORE; }
	}

    if  ( updMask & PPupdSPACE_AFTER )
	{
	if  ( pp1->ppSpaceAfterTwips != pp2->ppSpaceAfterTwips )
	    { changedMask |= PPupdSPACE_AFTER; }
	}

    if  ( updMask & PPupdNEWPAGE )
	{
	if  ( pp1->ppStartsOnNewPage != pp2->ppStartsOnNewPage )
	    { changedMask |= PPupdNEWPAGE; }
	}

    if  ( updMask & PPupdLINE_SPACING )
	{
	if  ( pp1->ppLineSpacingTwips != pp2->ppLineSpacingTwips )
	    { changedMask |= PPupdLINE_SPACING; }
	if  ( pp1->ppLineSpacingIsMultiple != pp2->ppLineSpacingIsMultiple )
	    { changedMask |= PPupdLINE_SPACING; }
	}

    if  ( updMask & PPupdTOP_BORDER )
	{
	if  ( ! docEqualBorder( &(pp1->ppTopBorder),
						&(pp2->ppTopBorder ) ) )
	    { changedMask |= PPupdTOP_BORDER; }
	}

    if  ( updMask & PPupdBOTTOM_BORDER )
	{
	if  ( ! docEqualBorder( &(pp1->ppBottomBorder),
						&(pp2->ppBottomBorder ) ) )
	    { changedMask |= PPupdBOTTOM_BORDER; }
	}

    return changedMask;
    }
