/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_main.cc
|
| Revision    : 1.0
| Date        : 12/02/96
|
| Object      : None
|
| Description : Main code file
|
| Copyright   : (c) 1996 Richard Kent
|
| EMail       : ceerbk@cee.hw.ac.uk
|
| $Id$
|
|----------------------------------------------------------------------------
|
|    This program is free software; you can redistribute it and/or modify
|    it under the terms of the GNU General Public License as published by
|    the Free Software Foundation; either version 2 of the License, or
|    (at your option) any later version.
|
|    This program is distributed in the hope that it will be useful,
|    but WITHOUT ANY WARRANTY; without even the implied warranty of
|    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|    GNU General Public License for more details.
|
|    You should have received a copy of the GNU General Public License
|    along with this program; if not, write to the Free Software
|    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
****************************************************************************/

static char DPTich_main_cc [] = "$Id$";

#include "DPTich_main.h"
#include "Keyboard.xpm"

FD_mainForm          *mainForm;
FD_playForm          *playForm;
FD_zoomForm          *zoomForm;
FD_displayForm       *displayForm;
FD_rangeForm         *rangeForm;
FD_loopForm          *loopForm;
FD_newForm           *newForm;
FD_recordForm        *recordForm;
FD_convertForm       *convertForm;
FD_mixForm           *mixForm;
FD_channelForm       *channelForm;
FD_duplicateForm     *duplicateForm;
FD_panoramaForm      *panoramaForm;
FD_workspaceForm     *workspaceForm;
FD_miscForm          *miscForm;
FD_rateForm          *rateForm;
FD_ampForm           *ampForm;
FD_balanceForm       *balanceForm;
FD_bounceForm        *bounceForm;
FD_rampForm          *rampForm;
FD_resampleForm      *resampleForm;
FD_pitchForm         *pitchForm;
FD_bufferForm        *bufferForm;
FD_mixplayForm       *mixplayForm;
FD_selectForm        *selectForm;
FD_adjustDCForm      *adjustDCForm;
FD_initialForm       *initialForm;
FD_effectForm        *effectForm;
FD_executeForm       *executeForm;
FD_postForm          *postForm;
FD_aboutForm         *aboutForm;
FD_stretchForm       *stretchForm;
FD_convert4to1Form   *convert4to1Form;
FD_convert4to2Form   *convert4to2Form;
FD_convert2to4Form   *convert2to4Form;
FD_channel4Form      *channel4Form;
FD_bounce4Form       *bounce4Form;
FD_prefsForm         *prefsForm;
FD_mixerForm         *mixerForm;
FD_macroForm         *macroForm;
FD_macrooutputForm   *macrooutputForm;
FD_macroexecForm     *macroexecForm;

Display              *display;
FL_OBJECT            *format;
FL_OBJECT            *compression;

DPSample             *buffer [8];
DPSample             *undo [2];
DPSample             *redo [2];
char                 *undoStr [2];
char                 *redoStr [2];
DPSample             *clip;

int                  current;
double               currentPitch;

ALport               globalInputPort;
ALport               globalOutputPort;
long                 globalChannels;

long                 DSPCancel;
long                 DSPStop;
long                 DSPTime;
double               DSPMix;
double               DSPMaster;
long                 DSPPlayStop;

DSPKitDPReader       *DSPReader_L;
DSPKitDPReader       *DSPReader_R;
DSPKitDPWriter       *DSPWriter;

allEffectsType       *effects;
FL_OBJECT *DSPname   [MAXPARAMS];
FL_OBJECT *DSPchan   [MAXPARAMS];
FL_OBJECT *DSPinput  [MAXPARAMS];
FL_OBJECT *DSPunits  [MAXPARAMS];
FL_OBJECT *DSPslider [MAXPARAMS];
FL_OBJECT *DSPchoice [MAXPARAMS];

double               recordAdjustLeft;
double               recordAdjustRight;
double               recordAdjustThree;
double               recordAdjustFour;
int                  stretchCancel;
int                  first0;
int                  first1;
int                  first2;
int                  initialFormHeight;
int                  recordFormWidth;
int                  preferencesChanged;
int                  macrosChanged;

int                  normalIdle;
int                  heavyIdle;

#ifdef LINUX
FL_OBJECT *mixerSliders [SOUND_MIXER_NRDEVICES * 2];
FL_OBJECT *mixerInButtons [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerOutButtons [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerRecButtons [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerLockButtons [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerMuteButtons [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerLabels [SOUND_MIXER_NRDEVICES];
FL_OBJECT *mixerInGroup;
FL_OBJECT *mixerOutGroup;
#endif

long globalMicrophoneMax;
long globalChannelsMax;


#include "DPTich_effect.h"

/*---------------------------------------------------------------------------
| FUNCTION main
---------------------------------------------------------------------------*/

int main (int argc, char *argv [])
{
  fl_initialize (&argc,argv,"DAP",0,0);
  tooltips_initialize ();
  #ifdef LINUX
  initialiseAudio ();
  #endif

  // Disable error handler
  ALseterrorhandler (0);

  // Check no one else using audio and there are free ports
  long pvBuffer [6];
  pvBuffer [0] = AL_INPUT_COUNT;
  pvBuffer [2] = AL_OUTPUT_COUNT;
  pvBuffer [4] = AL_UNUSED_COUNT;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,6);
  
  if (pvBuffer [5] == 0)
  {
    fl_show_alert ("Error","No audio ports free",
    "Exit other audio applications and try again",1);
    return 0;
  }

  #ifdef TEST
  if (pvBuffer [1] > 0 || pvBuffer [3] > 0)
  {
    if (!fl_show_question_old
      ("Warning","Other audio applications running","Continue ?"))
    return 0;
  }
  #endif

  initialiseForms ();
  #ifdef LINUX
  initialiseLinuxForms ();
  #endif
  fl_set_app_nomainform (TRUE);
  fl_show_form (mainForm->mainForm,
    FL_PLACE_MOUSE | FL_FREE_SIZE,FL_FULLBORDER,TITLE);

  // Set up preferences and macros
  // Note must be done after showing main form
  defaultSamplePrefs ();
  loadPreferences ();
  defaultMacros ();
  loadMacros ();

  // Load in any command line samples
  loadSamples (argc,argv);
  
  fl_show_form (aboutForm->aboutForm,FL_PLACE_MOUSE,FL_FULLBORDER,"About");
  fl_do_forms ();
  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION loadSamples
---------------------------------------------------------------------------*/
void loadSamples (int argc,char *argv [])
{
  // Load in command line samples
  if (argc > 1)
  {
    int i;
    int count=0;
    char *error;
    
    waitCursorOn ();
    for (i=0; i<(argc-1); i++)
    {
      if (count < 8)
      {
        if (error = buffer [count]->loadAIFF (argv [i+1]))
        {
          waitCursorOff ();
          fl_show_alert ("Warning - Could not load file",
            argv [i+1],error,TRUE);
          waitCursorOn ();
        }
        else
        {
          count++;
        }
      }
    }
    waitCursorOff ();
    clearBuffers ();
    fl_set_sample (mainForm->sample,buffer [0]);
    updateSample (buffer [0]);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseForms
---------------------------------------------------------------------------*/

void initialiseForms (void)
{
  FL_OBJECT *obj;
  DPSample  *sample;
  long      i;
  long      min;
  long      max;
  long      pvBuffer [12];
  char      tempString [50];
  
  // Create main sample
  current     = 0;
  buffer [0]  = new DPSample;
  buffer [1]  = new DPSample;
  buffer [2]  = new DPSample;
  buffer [3]  = new DPSample;
  buffer [4]  = new DPSample;
  buffer [5]  = new DPSample;
  buffer [6]  = new DPSample;
  buffer [7]  = new DPSample;
  undo    [0] = 0;
  undoStr [0] = 0;
  undo    [1] = 0;
  undoStr [1] = 0;
  redo    [0] = 0;
  redoStr [0] = 0;
  redo    [1] = 0;
  redoStr [1] = 0;
  clip        = new DPSample;
  sample      = buffer [current];
  
  // Set menu font defaults
  fl_setpup_fontsize (FL_SMALL_SIZE);
  fl_setpup_fontstyle (FL_NORMAL_STYLE);

  // Create the forms
  mainForm        = create_form_mainForm ();
  playForm        = create_form_playForm ();
  zoomForm        = create_form_zoomForm ();
  displayForm     = create_form_displayForm ();
  rangeForm       = create_form_rangeForm ();
  loopForm        = create_form_loopForm ();
  newForm         = create_form_newForm ();
  recordForm      = create_form_recordForm ();
  convertForm     = create_form_convertForm ();
  mixForm         = create_form_mixForm ();
  channelForm     = create_form_channelForm ();
  duplicateForm   = create_form_duplicateForm ();
  panoramaForm    = create_form_panoramaForm ();
  workspaceForm   = create_form_workspaceForm ();
  miscForm        = create_form_miscForm ();
  rateForm        = create_form_rateForm ();
  ampForm         = create_form_ampForm ();
  balanceForm     = create_form_balanceForm ();
  bounceForm      = create_form_bounceForm ();
  rampForm        = create_form_rampForm ();
  resampleForm    = create_form_resampleForm ();
  pitchForm       = create_form_pitchForm ();
  bufferForm      = create_form_bufferForm ();
  mixplayForm     = create_form_mixplayForm ();
  selectForm      = create_form_selectForm ();
  adjustDCForm    = create_form_adjustDCForm ();
  initialForm     = create_form_initialForm ();
  effectForm      = create_form_effectForm ();
  executeForm     = create_form_executeForm ();
  postForm        = create_form_postForm ();
  aboutForm       = create_form_aboutForm ();
  stretchForm     = create_form_stretchForm ();
  convert4to1Form = create_form_convert4to1Form ();
  convert4to2Form = create_form_convert4to2Form ();
  convert2to4Form = create_form_convert2to4Form ();
  channel4Form    = create_form_channel4Form ();
  bounce4Form     = create_form_bounce4Form ();
  prefsForm       = create_form_prefsForm ();
  mixerForm       = create_form_mixerForm ();
  macroForm       = create_form_macroForm ();
  macrooutputForm = create_form_macrooutputForm ();
  macroexecForm   = create_form_macroexecForm ();

  // Initialise effects table
  initialiseEffects ();
  
  // Add main sample object
  int x = mainForm->sample->x;
  int y = mainForm->sample->y;
  int w = mainForm->sample->w;
  int h = mainForm->sample->h;
  
  fl_addto_form (mainForm->mainForm);
    obj = mainForm->sample = fl_add_sample (FL_NORMAL_SAMPLE,x,y,w,h,"");
  fl_end_form ();  
  fl_set_object_resize (obj,FL_RESIZE_ALL);
  fl_set_object_gravity (obj,NorthWestGravity,SouthEastGravity);
  fl_set_object_boxtype (obj,FL_FLAT_BOX);
  fl_set_sample (obj,sample);
  
  // Initialise main form
  fl_set_object_dblbuffer (mainForm->clock,TRUE);
  tooltips_addtip (mainForm->clock,"Tells the time !!");
  fl_set_object_dblbuffer (mainForm->sample,TRUE);
  fl_set_slider_bounds (mainForm->scrollBarSlider,0.0,1.0);
  fl_set_slider_size (mainForm->scrollBarSlider,1.0);
  fl_set_slider_value (mainForm->scrollBarSlider,0.5);
  updateNameBox ();
  fl_addto_choice (mainForm->editMode,"Mono");
  fl_deactivate_object (mainForm->editMode);
  fl_hide_object (mainForm->mixerButton);

  // Initialise minimum form sizes
  fl_set_form_minsize (mainForm->mainForm,620,420);
  fl_set_form_minsize (macrooutputForm->macrooutputForm,260,180);
  fl_set_form_minsize (macroexecForm->macroexecForm,190,250);
  fl_set_form_minsize (mixForm->mixForm,510,200);
  fl_set_form_minsize (rampForm->rampForm,490,195);

  // Initialise file menu
  fl_addto_menu (mainForm->fileMenu,"New|Open|Save|Save As%l");
  fl_addto_menu (mainForm->fileMenu,"Open Clip|Save Clip%l");
  fl_addto_menu (mainForm->fileMenu,"Clear Buffer%l");
  fl_addto_menu (mainForm->fileMenu,"Quit");

  // Initialise edit menu
  fl_addto_menu (mainForm->editMenu,"Undo|Redo%l");
  fl_addto_menu (mainForm->editMenu,"Cut|Copy|Paste%l");
  fl_addto_menu (mainForm->editMenu,"Delete|Remove|Keep|Trim%l");
  fl_addto_menu (mainForm->editMenu,"Overwrite|Mix%l");
  fl_addto_menu (mainForm->editMenu,"Overwrite Range|Mix Range%l");
  fl_addto_menu (mainForm->editMenu,"Add Workspace%l");
  fl_addto_menu (mainForm->editMenu,"Clear Clipboard");
  fl_set_menu_item_mode (mainForm->editMenu,1,FL_PUP_GREY);
  fl_set_menu_item_mode (mainForm->editMenu,2,FL_PUP_GREY);
  
  // Initialise range menu
  fl_addto_menu (mainForm->rangeMenu,"Reverse|Invert|Zero|Channels");
  fl_addto_menu (mainForm->rangeMenu,"Duplicate|Panorama|Adjust DC");
  
  // Initialise misc menu
  fl_addto_menu (mainForm->miscMenu,"Sample Rate");
  fl_addto_menu (mainForm->miscMenu,"Textual Details");
  fl_addto_menu (mainForm->miscMenu,"Clear Undo Buffers");

  // Initialise amp menu
  fl_addto_menu (mainForm->ampMenu,"Amplitude|Balance|Normalise%l");
  fl_addto_menu (mainForm->ampMenu,"Fade In|Fade Out|Bounce%l");
  fl_addto_menu (mainForm->ampMenu,"Ramp%l");
  fl_addto_menu (mainForm->ampMenu,"Wrap");
  fl_set_menu_item_mode (mainForm->ampMenu,8,FL_PUP_BOX);
  
  // Initialise resample menu
  fl_addto_menu (mainForm->resampleMenu,"Non Time|Time");
  
  // Initialise help menu
  fl_addto_menu (mainForm->helpMenu,"About");

  // Initialise global menu
  fl_addto_menu (mainForm->globalMenu,"Edit Preferences%l");
  fl_addto_menu (mainForm->globalMenu,"Load Preferences");
  fl_addto_menu (mainForm->globalMenu,"Save Preferences%l");
  fl_addto_menu (mainForm->globalMenu,"Edit Macros%l");
  fl_addto_menu (mainForm->globalMenu,"Load Macros");
  fl_addto_menu (mainForm->globalMenu,"Save Macros%l");
  fl_addto_menu (mainForm->globalMenu,"Execute Macro");

  // Initially hide all time inputs, show all frame inputs
  fl_hide_object (displayForm->floatInputs);
  fl_show_object (displayForm->intInputs);
  fl_hide_object (loopForm->floatInputs);
  fl_show_object (loopForm->intInputs);
  fl_hide_object (rangeForm->floatInputs);
  fl_show_object (rangeForm->intInputs);

  // Initialise convert menu
  fl_addto_menu (mainForm->convertMenu,"To Mono");
  fl_addto_menu (mainForm->convertMenu,"To Stereo");
  fl_addto_menu (mainForm->convertMenu,"To Quadro%l");
  fl_addto_menu (mainForm->convertMenu,"To 8 Bit");
  fl_addto_menu (mainForm->convertMenu,"To 16 Bit");
  updateConvertMenu ();
  
  // Initialise mix form
  fl_set_slider_bounds (mixForm->ratio,100.0,0.0);
  fl_set_slider_step (mixForm->ratio,1.0);
  fl_addto_choice (mixForm->mode,"Clipboard|Range");
  fl_set_xyplot_return (mixForm->plot,FL_RETURN_ALWAYS);
  fl_set_xyplot_xbounds (mixForm->plot,-4.0,104.0);
  fl_set_xyplot_ybounds (mixForm->plot,-4.0,104.0);
  fl_set_xyplot_xtics (mixForm->plot,-1,-1);
  fl_set_xyplot_ytics (mixForm->plot,-1,-1);
  fl_set_xyplot_symbolsize (mixForm->plot,5);
  fl_hide_object (mixForm->mode);

  // Initialise ramp form
  fl_set_slider_bounds (rampForm->pointSlider,100.0,0.0);
  fl_set_slider_step (rampForm->pointSlider,1.0);
  fl_set_xyplot_return (rampForm->plot,FL_RETURN_ALWAYS);
  fl_set_xyplot_xbounds (rampForm->plot,-4.0,104.0);
  fl_set_xyplot_ybounds (rampForm->plot,-4.0,104.0);
  fl_set_xyplot_xtics (rampForm->plot,-1,-1);
  fl_set_xyplot_ytics (rampForm->plot,-1,-1);
  fl_set_xyplot_symbolsize (rampForm->plot,5);
  fl_addto_choice (rampForm->mode,"Amplitude|Balance");
  fl_addto_choice (rampForm->quadroLeft,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (rampForm->quadroRight,"Chan 1|Chan 2|Chan 3|Chan 4");
  
  // Initialise amp form
  fl_set_slider_bounds (ampForm->slider,200.0,0.0);
  fl_set_slider_step (ampForm->slider,1.0);
  
  // Initialise balance form
  fl_set_slider_bounds (balanceForm->leftSlider,100.0,0.0);
  fl_set_slider_bounds (balanceForm->rightSlider,100.0,0.0);
  fl_set_slider_bounds (balanceForm->slider,0.0,100.0);
  fl_set_slider_step (balanceForm->leftSlider,1.0);
  fl_set_slider_step (balanceForm->rightSlider,1.0);
  fl_set_slider_step (balanceForm->slider,1.0);
  fl_addto_choice (balanceForm->mode,"Balance|Independent");
  fl_addto_choice (balanceForm->quadroLeft,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (balanceForm->quadroRight,"Chan 1|Chan 2|Chan 3|Chan 4");
     
  // Initialise panorama form
  fl_set_slider_bounds (panoramaForm->amount,100.0,-100.0);
  fl_set_slider_step (panoramaForm->amount,1.0);
  fl_set_slider_bounds (panoramaForm->ltorSlider,0.0,1000.0);
  fl_set_slider_step (panoramaForm->ltorSlider,1.0);
  fl_set_slider_bounds (panoramaForm->rtolSlider,0.0,1000.0);
  fl_set_slider_step (panoramaForm->rtolSlider,1.0);
  fl_addto_choice (panoramaForm->quadroLeft,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (panoramaForm->quadroRight,"Chan 1|Chan 2|Chan 3|Chan 4");
  
  // Initialise duplicate form
  fl_set_slider_bounds (duplicateForm->times,1.0,100.0);
  fl_set_slider_step (duplicateForm->times,1.0);
  fl_set_slider_precision (duplicateForm->times,0);
  
  // Initialise rate form
  fl_addto_menu (rateForm->menu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  
  // Initialise play form
  fl_set_slider_bounds (playForm->leftSlider,100.0,0.0);
  fl_set_slider_step (playForm->leftSlider,1.0);
  fl_set_slider_precision (playForm->leftSlider,0);
  fl_set_slider_bounds (playForm->rightSlider,100.0,0.0);
  fl_set_slider_step (playForm->rightSlider,1.0);
  fl_set_slider_precision (playForm->rightSlider,0);
  fl_set_button (playForm->lockButton,TRUE);
  fl_set_button (playForm->scrollButton,FALSE);
  fl_addto_choice (playForm->rateChoice,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz"
    "|Digital|Input Rate");
  fl_deactivate_object (playForm->muteButton);
  fl_hide_object (playForm->playing);
  fl_addto_choice (playForm->modeChoice,"Stereo|Quadro");
  fl_set_choice (playForm->modeChoice,1);
  
  if (ALgetdefault (AL_DEFAULT_DEVICE,AL_CHANNEL_MODE) != -1)
    ALgetminmax (AL_DEFAULT_DEVICE,AL_CHANNEL_MODE,&min,&max);
  else
    max = AL_STEREO;
  globalChannelsMax = max;
  if (max == AL_STEREO)
    fl_deactivate_object (playForm->modeChoice);  
  pvBuffer [0] = AL_OUTPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  sprintf (tempString,"%ld",pvBuffer [1]);
  fl_set_input (playForm->rateInput,tempString);
  fl_hide_object (playForm->channelName);
  fl_hide_object (playForm->linuxGroup);
  fl_addto_menu (playForm->rateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  
  // Initialise record form
  fl_set_slider_bounds (recordForm->leftSlider,100.0,0.0);
  fl_set_slider_step (recordForm->leftSlider,1.0);
  fl_set_slider_precision (recordForm->leftSlider,0);
  fl_set_slider_bounds (recordForm->rightSlider,100.0,0.0);
  fl_set_slider_step (recordForm->rightSlider,1.0);
  fl_set_slider_precision (recordForm->rightSlider,0);
  fl_set_slider_bounds (recordForm->triggerSlider,128.0,0.0);
  fl_set_slider_step (recordForm->leftSlider,1.0);
  fl_set_button (recordForm->lockButton,TRUE);
  fl_set_button (recordForm->scrollButton,FALSE);
  fl_addto_choice (recordForm->rateChoice,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz"
    "|Digital");
  fl_deactivate_object (recordForm->leftLevel);
  fl_deactivate_object (recordForm->rightLevel);
  fl_set_object_dblbuffer (recordForm->leftLevel,TRUE);
  fl_set_object_dblbuffer (recordForm->rightLevel,TRUE);
  fl_set_object_dblbuffer (recordForm->leftOverload,TRUE);
  fl_set_object_dblbuffer (recordForm->rightOverload,TRUE);
  fl_set_object_dblbuffer (recordForm->triggerOverload,TRUE);
  fl_set_slider_bounds (recordForm->leftLevel,128.0,0.0);
  fl_set_slider_bounds (recordForm->rightLevel,128.0,0.0);
  fl_deactivate_object (recordForm->monitorButton);
  if (ALgetdefault (AL_DEFAULT_DEVICE,AL_MIC_MODE) != -1)
    ALgetminmax (AL_DEFAULT_DEVICE,AL_MIC_MODE,&min,&max);
  else
    max = AL_MONO;
  globalMicrophoneMax = max;
  fl_addto_choice (recordForm->sourceChoice,"Line|Digital");
  if (max == AL_MONO)
    fl_addto_choice (recordForm->sourceChoice,"Mic");
  else
    fl_addto_choice (recordForm->sourceChoice,"Mic (mono)|Mic (stereo)");
  fl_hide_object (recordForm->recording);
  recordAdjustLeft  = 0.0;
  recordAdjustRight = 0.0;
  recordAdjustThree = 0.0;
  recordAdjustFour  = 0.0;
  fl_set_slider_bounds (recordForm->threeSlider,100.0,0.0);
  fl_set_slider_step (recordForm->threeSlider,1.0);
  fl_set_slider_precision (recordForm->threeSlider,0);
  fl_set_slider_bounds (recordForm->fourSlider,100.0,0.0);
  fl_set_slider_step (recordForm->fourSlider,1.0);
  fl_set_slider_precision (recordForm->fourSlider,0);
  fl_deactivate_object (recordForm->threeLevel);
  fl_deactivate_object (recordForm->fourLevel);
  fl_set_object_dblbuffer (recordForm->threeLevel,TRUE);
  fl_set_object_dblbuffer (recordForm->fourLevel,TRUE);
  fl_set_object_dblbuffer (recordForm->threeOverload,TRUE);
  fl_set_object_dblbuffer (recordForm->fourOverload,TRUE);
  fl_set_slider_bounds (recordForm->threeLevel,128.0,0.0);
  fl_set_slider_bounds (recordForm->fourLevel,128.0,0.0);
  recordFormWidth = recordForm->recordForm->w;
  fl_set_form_size (
    recordForm->recordForm,
    recordFormWidth - 200,
    recordForm->recordForm->h);
  pvBuffer [0] = AL_INPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  sprintf (tempString,"%ld",pvBuffer [1]);
  fl_set_input (recordForm->rateInput,tempString);
  fl_hide_object (recordForm->channelName);
  fl_hide_object (recordForm->linuxGroup);
  fl_addto_menu (recordForm->rateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  fl_hide_object (recordForm->muteButton);
  
  // Initialise new form
  fl_addto_menu (newForm->rateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  fl_addto_choice (newForm->resolution,"8 bit |16 bit");
  fl_addto_choice (newForm->channels,"Mono|Stereo|Quadro");
  updateNewDetails ();
  
  // Initialise resample form
  fl_set_slider_bounds (resampleForm->rateChangeSlider,1.0,200.0);
  fl_set_slider_step (resampleForm->rateChangeSlider,1.0);
  fl_set_slider_bounds (resampleForm->framesChangeSlider,1.0,200.0);
  fl_set_slider_step (resampleForm->framesChangeSlider,1.0);
  fl_set_slider_bounds (resampleForm->timeChangeSlider,1.0,200.0);
  fl_set_slider_step (resampleForm->timeChangeSlider,1.0);
  fl_set_slider_bounds (resampleForm->pitchChangeSlider,1.0,200.0);
  fl_set_slider_step (resampleForm->pitchChangeSlider,1.0);
  fl_addto_choice (resampleForm->mode,"Non Time|Time");
  fl_addto_menu (resampleForm->newRateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  fl_addto_choice (resampleForm->pitchChangeDir,"+|-");
  resampleForm->ldata = 1;
  
  // Initialise stretch form
  fl_set_slider_bounds (stretchForm->percentCompSlider,0.0,100.0);
  fl_set_slider_step (stretchForm->percentCompSlider,1.0);
  
  // Initialise pitch form
  fl_set_slider_bounds (pitchForm->centSlider,49.0,-50.0);
  x = pitchForm->keyboard->x;
  y = pitchForm->keyboard->y;
  w = pitchForm->keyboard->w;
  h = pitchForm->keyboard->h;  
  fl_addto_form (pitchForm->pitchForm);
    obj = pitchForm->keyboard =
      fl_add_free (FL_NORMAL_FREE,x,y,w,h,"",keyboardHandle);
  fl_end_form ();
  fl_set_object_boxtype (obj,FL_FLAT_BOX);
  fl_set_object_resize (obj,FL_RESIZE_ALL);
  fl_set_object_gravity (obj,NorthWestGravity,SouthEastGravity);
  fl_addto_choice (pitchForm->mode,"Current|New");
  fl_set_pixmap_data (pitchForm->keyboardPixmap,Keyboard);
  fl_hide_object (pitchForm->mode);
  
  // Initialise buffer form
  fl_set_browser_vscrollbar (bufferForm->browser,FL_ON);
  fl_set_browser_dblclick_callback
    (bufferForm->browser,bufferSelectDbl_cb,0);
  fl_hide_object (bufferForm->select);
  fl_hide_object (bufferForm->mode);
  fl_hide_object (bufferForm->copyGroup);
  updateBufferBrowser ();
  fl_addto_choice (bufferForm->mode,"Copy|Move");
  fl_set_browser_vscrollbar (bufferForm->browser2,FL_ON);
  fl_set_browser_dblclick_callback
    (bufferForm->browser2,bufferOKDbl_cb,0);
  fl_addto_menu (bufferForm->mixplay,"Two|Four|Desk");
      
  // Initialise mixplay form
  fl_set_slider_bounds (mixplayForm->leftSlider,100.0,0.0);
  fl_set_slider_step (mixplayForm->leftSlider,1.0);
  fl_set_slider_precision (mixplayForm->leftSlider,0);
  fl_set_slider_bounds (mixplayForm->rightSlider,100.0,0.0);
  fl_set_slider_step (mixplayForm->rightSlider,1.0);
  fl_set_slider_precision (mixplayForm->rightSlider,0);
  fl_set_button (mixplayForm->lockButton,TRUE);
  fl_addto_choice (mixplayForm->rateChoice,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz"
    "|Digital|Input Rate");
  fl_deactivate_object (mixplayForm->muteButton);
  for (i=0; i<8; i++)
  {
    fl_set_slider_bounds (mixplayForm->slider [i],100.0,0.0);
    fl_set_slider_step (mixplayForm->slider [i],1.0);
  }
  fl_set_slider_step (mixplayForm->slider [8],1.0);
  fl_set_slider_bounds (mixplayForm->twoSlider,-100.0,100.0);
  fl_set_slider_step (mixplayForm->twoSlider,1.0);
  fl_set_browser_vscrollbar (mixplayForm->browser,FL_ON);
  fl_set_browser_dblclick_callback
    (mixplayForm->browser,mixplaySelectDbl_cb,0);
  fl_set_positioner_xbounds (mixplayForm->fourPositioner,-100.0,100.0);
  fl_set_positioner_ybounds (mixplayForm->fourPositioner,-100.0,100.0);
  fl_set_positioner_xstep (mixplayForm->fourPositioner,1.0);
  fl_set_positioner_ystep (mixplayForm->fourPositioner,1.0);
  pvBuffer [0] = AL_OUTPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  sprintf (tempString,"%ld",pvBuffer [1]);
  fl_set_input (mixplayForm->rateInput,tempString);
  fl_hide_object (mixplayForm->channelName);
  fl_hide_object (mixplayForm->linuxGroup);
  fl_addto_menu (mixplayForm->rateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");
  
  // Initialise select form
  fl_set_browser_vscrollbar (selectForm->browser,FL_ON);
  fl_set_browser_dblclick_callback (selectForm->browser,selectOKDbl_cb,0);
  
  // Initialise adjust DC form
  fl_set_slider_bounds (adjustDCForm->amount,100.0,-100.0);
  fl_set_slider_step (adjustDCForm->amount,1.0);
  
  #ifdef TEST
  // Set audio defaults
  pvBuffer [0]  = AL_INPUT_SOURCE;
  pvBuffer [1]  = AL_INPUT_LINE;
  pvBuffer [2]  = AL_SPEAKER_MUTE_CTL;
  pvBuffer [3]  = 0;
  pvBuffer [4]  = AL_MIC_MODE;
  pvBuffer [5]  = AL_MONO;
  pvBuffer [6]  = AL_CHANNEL_MODE;
  pvBuffer [7]  = AL_STEREO;
  pvBuffer [8]  = AL_INPUT_RATE;
  pvBuffer [9]  = AL_RATE_44100;
  pvBuffer [10] = AL_OUTPUT_RATE;
  #ifndef LINUX
  pvBuffer [11] = AL_RATE_INPUTRATE;
  #else
  pvBuffer [11] = AL_RATE_44100;
  #endif
  ALsetparams (AL_DEFAULT_DEVICE,pvBuffer,12);
  #endif

  globalChannels = 2;
  
  // Initialise display form
  updateDisplayDetails ();
  
  // Initialise range form
  updateRangeDetails ();
  
  // Initialise loop form
  fl_addto_choice (loopForm->susModeChoice,"Off|Forwards|Forw/Back");
  fl_addto_choice (loopForm->relModeChoice,"Off|Forwards|Forw/Back");
  updateLoopDetails ();
  fl_set_input (loopForm->susBeats,"1");
  fl_set_input (loopForm->relBeats,"1");
  
  // Initialise initial form
  fl_set_slider_bounds (initialForm->susSlider,0.0,50.0);
  fl_set_slider_step (initialForm->susSlider,1.0);
  fl_set_slider_bounds (initialForm->relSlider,0.0,50.0);
  fl_set_slider_step (initialForm->relSlider,1.0);
  fl_addto_choice (initialForm->quadro1Left,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (initialForm->quadro1Right,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (initialForm->quadro2Left,"Chan 1|Chan 2|Chan 3|Chan 4");
  fl_addto_choice (initialForm->quadro2Right,"Chan 1|Chan 2|Chan 3|Chan 4");
  initialFormHeight = initialForm->initialForm->h;
  fl_set_form_size (
    initialForm->initialForm,
    initialForm->initialForm->w,
    initialFormHeight - 80);
  
  // Initialise execute form
  fl_set_slider_bounds (executeForm->percentCompSlider,0.0,100.0);
  fl_set_slider_step (executeForm->percentCompSlider,1.0);
  fl_deactivate_object (executeForm->percentCompSlider);
  fl_set_object_dblbuffer (executeForm->frames,TRUE);
  fl_set_object_dblbuffer (executeForm->time,TRUE);
  fl_set_object_dblbuffer (executeForm->elapsedTime,TRUE);
  fl_set_object_dblbuffer (executeForm->remainTime,TRUE);
  fl_set_object_dblbuffer (executeForm->percentComp,TRUE);
  fl_set_object_dblbuffer (executeForm->percentCompSlider,TRUE);
  fl_hide_object (executeForm->wait);
  
  // Initialise post form
  fl_set_slider_bounds (postForm->leftSlider,100.0,0.0);
  fl_set_slider_step (postForm->leftSlider,1.0);
  fl_set_slider_precision (postForm->leftSlider,0);
  fl_set_slider_bounds (postForm->rightSlider,100.0,0.0);
  fl_set_slider_step (postForm->rightSlider,1.0);
  fl_set_slider_precision (postForm->rightSlider,0);
  fl_set_button (postForm->lockButton,TRUE);
  fl_addto_choice (postForm->rateChoice,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz"
    "|Digital|Input Rate");
  fl_deactivate_object (postForm->muteButton);
  fl_set_slider_bounds (postForm->master,2.0,0.0);
  fl_set_slider_step (postForm->master,0.01);
  fl_set_slider_bounds (postForm->mix,0.0,1.0);
  fl_set_slider_step (postForm->mix,0.01);
  fl_hide_object (postForm->playing);
  pvBuffer [0] = AL_OUTPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  sprintf (tempString,"%ld",pvBuffer [1]);
  fl_set_input (postForm->rateInput,tempString);
  fl_hide_object (postForm->channelName);
  fl_hide_object (postForm->linuxGroup);
  fl_addto_menu (postForm->rateMenu,
    "8 kHz|11.025 kHz|16 kHz|22.05 kHz|32 kHz|44.1 kHz|48 kHz");

  // Initialise macroexec form
  fl_set_browser_dblclick_callback (macroexecForm->browser,
    macroexecBrowserDbl_cb,0);

  // Initialise macro form
  fl_call_object_callback (macroForm->newButton);
  
  // Update audio globals
  updateAudioGlobals (0,0);

  // Set idle and close callbacks
  normalIdle = 500;
  heavyIdle  = 2000;
  fl_set_idle_callback (updateAudioGlobals,(void *) 1);
  fl_set_idle_delta (normalIdle);
  fl_set_atclose (closeAll,0);
  
  // Set interrupt callback
  fl_add_signal_callback (SIGINT,interrupt,0);
  
  // Initialise save file selector (add format and compression)
  fl_use_fselector (1);
  FL_FORM *fileSelect = fl_get_fselector_form ();
  
  fl_addto_form (fileSelect);
    format =
      fl_add_choice (FL_NORMAL_CHOICE,210,130,83,20,"Format");
    compression =
      fl_add_choice (FL_NORMAL_CHOICE,210,170,83,20,"Compression");
  fl_end_form ();
  
  fl_addto_choice (format,"AIFF|AIFFC");
  fl_addto_choice (compression,"None%l|G722|G711 ulaw|G711 alaw%l");
  fl_addto_choice (compression,"MPEG I|MPEG II%l|Multirate|Lossless");
  fl_set_choice (format,1);
  fl_set_choice (compression,1);
  fl_set_object_boxtype (format,FL_UP_BOX);
  fl_set_object_boxtype (compression,FL_UP_BOX);
  fl_set_object_lalign (format,FL_ALIGN_TOP);
  fl_set_object_lalign (compression,FL_ALIGN_TOP);
  fl_set_object_lsize (format,FL_TINY_SIZE);
  fl_set_object_lsize (compression,FL_TINY_SIZE);
  fl_set_object_resize (format,FL_RESIZE_NONE);
  fl_set_object_resize (compression,FL_RESIZE_NONE);
  fl_set_object_gravity (format,FL_NorthEast,FL_NorthEast);
  fl_set_object_gravity (compression,FL_NorthEast,FL_NorthEast);
  fl_set_object_shortcut (format,"#F#f",1);
  fl_set_object_shortcut (compression,"#o#O",1);
  fl_set_choice_fontsize (format,FL_TINY_SIZE);
  fl_set_choice_fontsize (compression,FL_TINY_SIZE);
  fl_set_object_callback (format,saveFileFormat_cb,0);
  fl_set_object_callback (compression,saveFileCompression_cb,0);
  
  // Initial global ports to zero and set current pitch
  globalInputPort  = 0;
  globalOutputPort = 0;
  currentPitch     = 60.0;
  first0           = 1;
  first1           = 1;
  first2           = 1;
  
  // Initialise about form
  i = 0;
  fl_set_object_label (aboutForm->line [i++],
    "Digital Audio Processor");
  fl_set_object_label (aboutForm->line [i++],
    "Copyright (c) 1997 Richard Kent");
  fl_set_object_label (aboutForm->line [i++],
    "Version 1.3 (beta)");
  fl_set_object_label (aboutForm->line [i++],
    "");
  fl_set_object_label (aboutForm->line [i++],
    "DAP comes with ABSOLUTELY NO WARRANTY. This is free");
  fl_set_object_label (aboutForm->line [i++],
    "software, and you are welcome to redistribute it under certain");
  fl_set_object_label (aboutForm->line [i++],
    "conditions. See COPYING for further details");
  fl_set_object_label (aboutForm->line [i++],
    "");
  fl_set_object_label (aboutForm->line [i++],
    "Please EMail me with bug reports, suggestions, comments,");
  fl_set_object_label (aboutForm->line [i++],
    "general banter and anything else you can think of !!!!");
  fl_set_object_label (aboutForm->line [i++],
    "");
  fl_set_object_label (aboutForm->line [i++],
    "Email me at: ceerbk@cee.hw.ac.uk");
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseLinuxForms
---------------------------------------------------------------------------*/

#ifdef LINUX
void initialiseLinuxForms (void)
{
  // Update play, post, mixplay and record forms

  fl_activate_object (playForm->muteButton);
  fl_hide_object (playForm->muteHiddenButton);
  fl_hide_object (playForm->sgGroup);
  fl_show_object (playForm->linuxGroup);
  fl_show_object (playForm->channelName);

  fl_activate_object (postForm->muteButton);
  fl_hide_object (postForm->muteHiddenButton);
  fl_hide_object (postForm->sgGroup);
  fl_show_object (postForm->linuxGroup);
  fl_show_object (postForm->channelName);

  fl_activate_object (mixplayForm->muteButton);
  fl_hide_object (mixplayForm->muteHiddenButton);
  fl_hide_object (mixplayForm->sgGroup);
  fl_show_object (mixplayForm->linuxGroup);
  fl_show_object (mixplayForm->channelName);

  fl_hide_object (recordForm->monitorButton);
  fl_hide_object (recordForm->monitorHiddenButton);
  fl_show_object (recordForm->muteButton);
  fl_hide_object (recordForm->sgGroup);
  fl_show_object (recordForm->linuxGroup);
  fl_show_object (recordForm->channelName);
  
  // Determine if mixer present
  long pvBuffer [2];
  pvBuffer [0] = AL_LINUX_MIXER_PRESENT;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
  if (pvBuffer [1])
  {
    // Create mixer form
    long devMask;
    long recMask;
    long stereoMask;
    long chanCount;
    long i;
    long rows;
    long cols;
    long pvBuffer [6];
    
    pvBuffer [0] = AL_LINUX_MIXER_CHANNELS;
    pvBuffer [2] = AL_LINUX_MIXER_RECORD;
    pvBuffer [4] = AL_LINUX_MIXER_STEREO;
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,6);
    devMask    = pvBuffer [1];
    recMask    = pvBuffer [3];
    stereoMask = pvBuffer [5];

    chanCount = 0;
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
      if (devMask & (1 << i)) chanCount++;

    if (chanCount <= 12)
    {
      rows = 1;
      cols = chanCount;
    }
    else
    {
      rows = 2;
      cols = (chanCount + 1) / 2;
    }

    long row = 0;
    long col = 0;

    fl_addto_form (mixerForm->mixerForm);
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
      if (devMask & (1 << i))
      {
        long stereo = (stereoMask & (1 << i));
        long rec    = (recMask & (1 << i));
        addMixerSlider (mixerForm->mixerForm,i,stereo,rec,row,col);
        col++;
        if (col == cols)
        {
          row++;
          col = 0;
        }
      };
    }
    fl_end_form ();
    fl_set_form_size (mixerForm->mixerForm,
      (2 * MIXER_BORDERX) + (cols * (MIXER_COLSIZE + 
        MIXER_INTERX) - MIXER_INTERX),
      (2 * MIXER_BORDERY) + (rows * (MIXER_ROWSIZE + 
        MIXER_INTERY) - MIXER_INTERY));
            
    pvBuffer [0] = AL_LINUX_CURRENT_INPUT;
    pvBuffer [2] = AL_LINUX_CURRENT_OUTPUT;
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,4);
    long currentInput  = pvBuffer [1];
    long currentOutput = pvBuffer [3];    

    // Update mixer slider positions
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
      if (devMask & (1 << i))
      {
        pvBuffer [0] = AL_LINUX_CHANNEL_BASE + i;
        ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
        fl_set_slider_value
          (mixerSliders [2*i],pvBuffer [1] & 0x000000ff);

        // Update record form as appropriate
        if (i == currentInput)
        {
          fl_set_slider_value
            (recordForm->leftSlider,pvBuffer [1] & 0x000000ff);
          if (!(stereoMask & (1 << i)))
          {
            fl_hide_object (recordForm->rightSlider);
            fl_hide_object (recordForm->lockButton);
            fl_set_object_position (recordForm->leftSlider,
              recordForm->leftSlider->x + 15,
              recordForm->leftSlider->y);
            fl_set_object_label (recordForm->leftSlider,"");
          }
        }

        // Update play, post and mixplay forms as appropriate
        if (i == currentOutput)
        {
          fl_set_slider_value
            (playForm->leftSlider,pvBuffer [1] & 0x000000ff);
          fl_set_slider_value
            (mixplayForm->leftSlider,pvBuffer [1] & 0x000000ff);
          fl_set_slider_value
            (postForm->leftSlider,pvBuffer [1] & 0x000000ff);
          if (!(stereoMask & (1 << i)))
          {
            fl_hide_object (playForm->rightSlider);
            fl_hide_object (playForm->lockButton);
            fl_hide_object (mixplayForm->rightSlider);
            fl_hide_object (mixplayForm->lockButton);
            fl_hide_object (postForm->rightSlider);
            fl_hide_object (postForm->lockButton);
            fl_set_object_position (playForm->leftSlider,
              playForm->leftSlider->x + 15,
              playForm->leftSlider->y);              
            fl_set_object_position (mixplayForm->leftSlider,
              mixplayForm->leftSlider->x + 15,
              mixplayForm->leftSlider->y);              
            fl_set_object_position (mixplayForm->leftSlider,
              mixplayForm->leftSlider->x + 15,
              mixplayForm->leftSlider->y);
            fl_set_object_label (playForm->leftSlider,"");
            fl_set_object_label (mixplayForm->leftSlider,"");
            fl_set_object_label (postForm->leftSlider,"");
          }
        }

        // Now do right channels if stereo
        if (stereoMask & (1 << i))
        {
          fl_set_slider_value
            (mixerSliders [2*i+1],(pvBuffer [1] & 0x0000ff00) >> 8);

          if (i == currentInput)
          {
            fl_set_slider_value
              (recordForm->rightSlider,(pvBuffer [1] & 0x0000ff00) >> 8);
          }

          if (i == currentOutput)
          {
            fl_set_slider_value
              (playForm->rightSlider,(pvBuffer [1] & 0x0000ff00) >> 8);
            fl_set_slider_value
              (mixplayForm->rightSlider,(pvBuffer [1] & 0x0000ff00) >> 8);
            fl_set_slider_value
              (postForm->rightSlider,(pvBuffer [1] & 0x0000ff00) >> 8);
          }
        }
      }
    }

    // Update input,output and record settings
    pvBuffer [0] = AL_LINUX_CURRENT_INPUT;
    pvBuffer [2] = AL_LINUX_CURRENT_OUTPUT;
    pvBuffer [4] = AL_LINUX_CURRENT_REC;
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,6);
    fl_set_button (mixerInButtons [pvBuffer [1]],1);
    fl_set_button (mixerOutButtons [pvBuffer [3]],1);
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
      if (recMask & (1 << i))
      {
        if (pvBuffer [5] & (1 << i))
          fl_set_button (mixerRecButtons [i],1);
        else
          fl_set_button (mixerRecButtons [i],0);
      }
    }

    // Update main form
    fl_set_object_position (mainForm->editMode,
      mainForm->editMode->x,mainForm->editMode->y + 25);
    fl_set_object_position (mainForm->editModeText,
      mainForm->editModeText->x,mainForm->editModeText->y + 25);
    fl_set_object_size (mainForm->bufferBox,
      mainForm->bufferBox->w,mainForm->bufferBox->h + 25);
    fl_show_object (mainForm->mixerButton);

    // Update output channel labels
    pvBuffer [0] = AL_LINUX_CURRENT_OUTPUT;
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
    fl_set_object_label
        (playForm->channelName,ALgetlabel (pvBuffer [1]));
    fl_set_object_label
        (mixplayForm->channelName,ALgetlabel (pvBuffer [1]));
    fl_set_object_label
        (postForm->channelName,ALgetlabel (pvBuffer [1]));
    
    // Check for record channels
    pvBuffer [0] = AL_LINUX_MIXER_RECORD;
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
    if (!(pvBuffer [1]))
    {
      fl_set_slider_value (recordForm->leftSlider,0.0);
      fl_set_slider_value (recordForm->rightSlider,0.0);
      fl_deactivate_object (recordForm->leftSlider);
      fl_deactivate_object (recordForm->rightSlider);
      fl_deactivate_object (recordForm->lockButton);
      fl_deactivate_object (recordForm->muteButton);
      fl_set_object_label (recordForm->channelName,"None");      
    }
    else
    {
      pvBuffer [0] = AL_LINUX_CURRENT_INPUT;
      ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,2);
      fl_set_object_label
        (recordForm->channelName,ALgetlabel (pvBuffer [1]));
    }
  }
  else
  {
    // Deactivate relevant sliders

    fl_set_slider_value (playForm->leftSlider,0.0);
    fl_set_slider_value (playForm->rightSlider,0.0);
    fl_deactivate_object (playForm->leftSlider);
    fl_deactivate_object (playForm->rightSlider);
    fl_deactivate_object (playForm->lockButton);
    fl_deactivate_object (playForm->muteButton);
    fl_set_object_label (playForm->channelName,"None");

    fl_set_slider_value (postForm->leftSlider,0.0);
    fl_set_slider_value (postForm->rightSlider,0.0);
    fl_deactivate_object (postForm->leftSlider);
    fl_deactivate_object (postForm->rightSlider);
    fl_deactivate_object (postForm->lockButton);
    fl_deactivate_object (postForm->muteButton);
    fl_set_object_label (postForm->channelName,"None");

    fl_set_slider_value (mixplayForm->leftSlider,0.0);
    fl_set_slider_value (mixplayForm->rightSlider,0.0);
    fl_deactivate_object (mixplayForm->leftSlider);
    fl_deactivate_object (mixplayForm->rightSlider);
    fl_deactivate_object (mixplayForm->lockButton);
    fl_deactivate_object (mixplayForm->muteButton);
    fl_set_object_label (mixplayForm->channelName,"None");

    fl_set_slider_value (recordForm->leftSlider,0.0);
    fl_set_slider_value (recordForm->rightSlider,0.0);
    fl_deactivate_object (recordForm->leftSlider);
    fl_deactivate_object (recordForm->rightSlider);
    fl_deactivate_object (recordForm->lockButton);
    fl_deactivate_object (recordForm->muteButton);
    fl_set_object_label (recordForm->channelName,"None");
  }  
}

/*---------------------------------------------------------------------------
| FUNCTION addMixerSlider
---------------------------------------------------------------------------*/

void addMixerSlider (
  FL_FORM *form,
  long chan,
  long stereo,
  long rec,
  long row,
  long col)
{
  FL_OBJECT *obj;
  long x;
  long y;
  
  static int firstInGroup  = TRUE;
  static int firstOutGroup = TRUE;
  
  x = MIXER_BORDERX + (col * (MIXER_COLSIZE + MIXER_INTERX));
  y = MIXER_BORDERY + (row * (MIXER_ROWSIZE + MIXER_INTERY));
  
  obj = fl_add_frame (FL_ENGRAVED_FRAME,x-2,y-2,
    MIXER_COLSIZE+4,MIXER_ROWSIZE+4,"");
    fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
  if (stereo)
  {
    obj = mixerSliders [chan*2] = 
      fl_add_slider(FL_VERT_NICE_SLIDER,x+0,y+90,20,90,"L");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_slider_bounds(obj, 100.00, 0.00);
      fl_set_slider_value(obj, 50.00);
      fl_set_object_callback(obj,mixerSlider_cb,chan*2);
    obj = mixerSliders [chan*2+1] = 
      fl_add_slider(FL_VERT_NICE_SLIDER,x+24,y+90,20,90,"R");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_slider_bounds(obj, 100.00, 0.00);
      fl_set_slider_value(obj, 50.00);
      fl_set_object_callback(obj,mixerSlider_cb,chan*2+1);
    obj = mixerLockButtons [chan] = 
      fl_add_lightbutton(FL_PUSH_BUTTON,x+0,y+65,44,20,"Lock");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerLock_cb,chan);
      fl_set_button(obj,1);
  }
  else
  {
    obj = mixerSliders [chan*2] = 
      fl_add_slider(FL_VERT_NICE_SLIDER,x+12,y+90,20,90,"");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_slider_bounds(obj, 100.00, 0.00);
      fl_set_slider_value(obj, 50.00);
      fl_set_object_callback(obj,mixerSlider_cb,chan*2);
  }
  obj = mixerMuteButtons [chan] = 
    fl_add_lightbutton(FL_PUSH_BUTTON,x+0,y+195,44,20,"Mute");
    fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
    fl_set_object_lsize(obj,FL_TINY_SIZE);
    fl_set_object_callback(obj,mixerMute_cb,chan);
  if (rec)
  {
    obj = mixerRecButtons [chan] = 
      fl_add_lightbutton(FL_PUSH_BUTTON,x+0,y+40,44,20,"Rec");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerRec_cb,chan);
  }
  obj = mixerLabels [chan] = 
    fl_add_text(FL_NORMAL_TEXT,x+0,y+220,44,20,globalLabels [chan]);
    fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);

  if (firstInGroup)
  {
    mixerInGroup = fl_bgn_group();
    obj = mixerInButtons [chan] = 
      fl_add_lightbutton(FL_RADIO_BUTTON,x+0,y+0,44,20,"In");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerIn_cb,chan);
    fl_end_group();
    firstInGroup = FALSE;
  }
  else
  {
    fl_addto_group(mixerInGroup);
    obj = mixerInButtons [chan] = 
      fl_add_lightbutton(FL_RADIO_BUTTON,x+0,y+0,44,20,"In");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerIn_cb,chan);
    fl_end_group();
  }

  if (firstOutGroup)
  {
    mixerOutGroup = fl_bgn_group();
    obj = mixerOutButtons [chan] = 
      fl_add_lightbutton(FL_RADIO_BUTTON,x+0,y+20,44,20,"Out");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerOut_cb,chan);
    fl_end_group();
    firstOutGroup = FALSE;
  }
  else
  {
    fl_addto_group(mixerOutGroup);
    obj = mixerOutButtons [chan] = 
      fl_add_lightbutton(FL_RADIO_BUTTON,x+0,y+20,44,20,"Out");
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      fl_set_object_lsize(obj,FL_TINY_SIZE);
      fl_set_object_callback(obj,mixerOut_cb,chan);
    fl_end_group();
  }
}
#endif

/*---------------------------------------------------------------------------
| FUNCTION updateAudioGlobals
---------------------------------------------------------------------------*/

#ifndef LINUX
int updateAudioGlobals (XEvent *xev,void *data)
{
  static signed char tempInputBuf  [4 * METERQUEUESIZE];
  static short       tempOutputBuf [SINEOUTQUEUESIZE];

  long pvBuffer [26];
  long volLeft;
  long volRight;
  long volThree;
  long volFour;

  pvBuffer [0]  = AL_LEFT_SPEAKER_GAIN;
  pvBuffer [2]  = AL_RIGHT_SPEAKER_GAIN;
  pvBuffer [4]  = AL_SPEAKER_MUTE_CTL;
  pvBuffer [6]  = AL_OUTPUT_RATE;
  pvBuffer [8]  = AL_INPUT_SOURCE;
  pvBuffer [10] = AL_LEFT_INPUT_ATTEN;
  pvBuffer [12] = AL_RIGHT_INPUT_ATTEN;
  pvBuffer [14] = AL_INPUT_RATE;
  pvBuffer [16] = AL_MONITOR_CTL;
  
  // Don't look for microphone mode if not necessary
  if (globalMicrophoneMax == AL_MONO)
    pvBuffer [18] = AL_INPUT_RATE; // Doesn't matter (just valid)
  else
    pvBuffer [18] = AL_MIC_MODE;

  pvBuffer [20] = AL_CHANNEL_MODE;
  pvBuffer [22] = AL_LEFT2_INPUT_ATTEN;
  pvBuffer [24] = AL_RIGHT2_INPUT_ATTEN;
  
  // Don't look for quadro stuff if not necessary
  if (globalChannelsMax == AL_STEREO)
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,20);
  else
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,26);

  // Check hardware modes OK
  if (globalMicrophoneMax == AL_MONO)
    pvBuffer [19] = AL_MONO;
  if (globalChannelsMax == AL_STEREO)
    pvBuffer [21] = AL_STEREO;

  int quadro = (pvBuffer [21] == AL_4CHANNEL);
    
  // Hardware Mode
  if (globalChannels == 4 && pvBuffer [21] == AL_STEREO)
  {
    fl_set_choice (playForm->modeChoice,1);
    fl_call_object_callback (playForm->modeChoice);
  }
  else if (globalChannels == 2 && pvBuffer [21] == AL_4CHANNEL)
  {
    fl_set_choice (playForm->modeChoice,2);
    fl_call_object_callback (playForm->modeChoice);
  }

  // Playback parameters
  volLeft  = volToNum (pvBuffer [1]);
  volRight = volToNum (pvBuffer [3]);
  if (fl_get_slider_value (playForm->leftSlider) != volLeft)
    fl_set_slider_value (playForm->leftSlider,volLeft);
  if (fl_get_slider_value (mixplayForm->leftSlider) != volLeft)
    fl_set_slider_value (mixplayForm->leftSlider,volLeft);
  if (fl_get_slider_value (postForm->leftSlider) != volLeft)
    fl_set_slider_value (postForm->leftSlider,volLeft);

  if (fl_get_slider_value (playForm->rightSlider) != volRight)
    fl_set_slider_value (playForm->rightSlider,volRight);
  if (fl_get_slider_value (mixplayForm->rightSlider) != volRight)
    fl_set_slider_value (mixplayForm->rightSlider,volRight);
  if (fl_get_slider_value (postForm->rightSlider) != volRight)
    fl_set_slider_value (postForm->rightSlider,volRight);
    
  if (fl_get_button (playForm->muteButton) != pvBuffer [5])
    fl_set_button (playForm->muteButton,pvBuffer [5]);
  if (fl_get_button (mixplayForm->muteButton) != pvBuffer [5])
    fl_set_button (mixplayForm->muteButton,pvBuffer [5]);
  if (fl_get_button (postForm->muteButton) != pvBuffer [5])
    fl_set_button (postForm->muteButton,pvBuffer [5]);
    
  switch (pvBuffer [7])
  {
    case AL_RATE_8000 :
      if (fl_get_choice (playForm->rateChoice) != 1)
        fl_set_choice (playForm->rateChoice,1);
      if (fl_get_choice (mixplayForm->rateChoice) != 1)
        fl_set_choice (mixplayForm->rateChoice,1);
      if (fl_get_choice (postForm->rateChoice) != 1)
        fl_set_choice (postForm->rateChoice,1);
      break;
    case AL_RATE_11025 :
      if (fl_get_choice (playForm->rateChoice) != 2)
        fl_set_choice (playForm->rateChoice,2);
      if (fl_get_choice (mixplayForm->rateChoice) != 2)
        fl_set_choice (mixplayForm->rateChoice,2);
      if (fl_get_choice (postForm->rateChoice) != 2)
        fl_set_choice (postForm->rateChoice,2);
      break;
    case AL_RATE_16000 :
      if (fl_get_choice (playForm->rateChoice) != 3)
        fl_set_choice (playForm->rateChoice,3);
      if (fl_get_choice (mixplayForm->rateChoice) != 3)
        fl_set_choice (mixplayForm->rateChoice,3);
      if (fl_get_choice (postForm->rateChoice) != 3)
        fl_set_choice (postForm->rateChoice,3);
      break;
    case AL_RATE_22050 :
      if (fl_get_choice (playForm->rateChoice) != 4)
        fl_set_choice (playForm->rateChoice,4);
      if (fl_get_choice (mixplayForm->rateChoice) != 4)
        fl_set_choice (mixplayForm->rateChoice,4);
      if (fl_get_choice (postForm->rateChoice) != 4)
        fl_set_choice (postForm->rateChoice,4);
      break;
    case AL_RATE_32000 :
      if (fl_get_choice (playForm->rateChoice) != 5)
        fl_set_choice (playForm->rateChoice,5);
      if (fl_get_choice (mixplayForm->rateChoice) != 5)
        fl_set_choice (mixplayForm->rateChoice,5);
      if (fl_get_choice (postForm->rateChoice) != 5)
        fl_set_choice (postForm->rateChoice,5);
      break;
    case AL_RATE_44100 :
      if (fl_get_choice (playForm->rateChoice) != 6)
        fl_set_choice (playForm->rateChoice,6);
      if (fl_get_choice (mixplayForm->rateChoice) != 6)
        fl_set_choice (mixplayForm->rateChoice,6);
      if (fl_get_choice (postForm->rateChoice) != 6)
        fl_set_choice (postForm->rateChoice,6);
      break;
    case AL_RATE_48000 :
      if (fl_get_choice (playForm->rateChoice) != 7)
        fl_set_choice (playForm->rateChoice,7);
      if (fl_get_choice (mixplayForm->rateChoice) != 7)
        fl_set_choice (mixplayForm->rateChoice,7);
      if (fl_get_choice (postForm->rateChoice) != 7)
        fl_set_choice (postForm->rateChoice,7);
      break;
    case AL_RATE_AES_1 :
      if (fl_get_choice (playForm->rateChoice) != 8)
        fl_set_choice (playForm->rateChoice,8);
      if (fl_get_choice (mixplayForm->rateChoice) != 8)
        fl_set_choice (mixplayForm->rateChoice,8);
      if (fl_get_choice (postForm->rateChoice) != 8)
        fl_set_choice (postForm->rateChoice,8);
      break;
    case AL_RATE_INPUTRATE :
      if (fl_get_choice (playForm->rateChoice) != 9)
        fl_set_choice (playForm->rateChoice,9);
      if (fl_get_choice (mixplayForm->rateChoice) != 9)
        fl_set_choice (mixplayForm->rateChoice,9);
      if (fl_get_choice (postForm->rateChoice) != 9)
        fl_set_choice (postForm->rateChoice,9);
      break;
    default:
      break; // Error - do not set rate choice
  }

  // Record parameters
  volLeft  = recToNum (pvBuffer [11]);
  volRight = recToNum (pvBuffer [13]);
  if (fl_get_slider_value (recordForm->leftSlider) != volLeft)
    fl_set_slider_value (recordForm->leftSlider,volLeft);
  if (fl_get_slider_value (recordForm->rightSlider) != volRight)
    fl_set_slider_value (recordForm->rightSlider,volRight);
  if (fl_get_button (recordForm->monitorButton) != pvBuffer [17])
    fl_set_button (recordForm->monitorButton,pvBuffer [17]);
  
  if (quadro)
  {
    volThree = recToNum (pvBuffer [23]);
    volFour  = recToNum (pvBuffer [25]);
    if (fl_get_slider_value (recordForm->threeSlider) != volThree)
      fl_set_slider_value (recordForm->threeSlider,volThree);
    if (fl_get_slider_value (recordForm->fourSlider) != volFour)
      fl_set_slider_value (recordForm->fourSlider,volFour);
  }
  
  switch (pvBuffer [15])
  {
    case AL_RATE_8000 :
      if (fl_get_choice (recordForm->rateChoice) != 1)
        fl_set_choice (recordForm->rateChoice,1); break;
    case AL_RATE_11025 :
      if (fl_get_choice (recordForm->rateChoice) != 2)
        fl_set_choice (recordForm->rateChoice,2); break;
    case AL_RATE_16000 :
      if (fl_get_choice (recordForm->rateChoice) != 3)
        fl_set_choice (recordForm->rateChoice,3); break;
    case AL_RATE_22050 :
      if (fl_get_choice (recordForm->rateChoice) != 4)
        fl_set_choice (recordForm->rateChoice,4); break;
    case AL_RATE_32000 :
      if (fl_get_choice (recordForm->rateChoice) != 5)
        fl_set_choice (recordForm->rateChoice,5); break;
    case AL_RATE_44100 :
      if (fl_get_choice (recordForm->rateChoice) != 6)
        fl_set_choice (recordForm->rateChoice,6); break;
    case AL_RATE_48000 :
      if (fl_get_choice (recordForm->rateChoice) != 7)
        fl_set_choice (recordForm->rateChoice,7); break;
    case AL_RATE_AES_1 :
      if (fl_get_choice (recordForm->rateChoice) != 8)
        fl_set_choice (recordForm->rateChoice,8); break;
    default:
      break; // Error - do not set rate choice
  }
  
  switch (pvBuffer [9])
  {
    case AL_INPUT_LINE :
      if (fl_get_choice (recordForm->sourceChoice) != 1)
        fl_set_choice (recordForm->sourceChoice,1); break;
    case AL_INPUT_DIGITAL :
      if (fl_get_choice (recordForm->sourceChoice) != 2)
        fl_set_choice (recordForm->sourceChoice,2); break;
    case AL_INPUT_MIC :
      if (pvBuffer [19] == AL_MONO)
      {
        if (fl_get_choice (recordForm->sourceChoice) != 3)
          fl_set_choice (recordForm->sourceChoice,3); break;
      }
      else
      {
        if (fl_get_choice (recordForm->sourceChoice) != 4)
          fl_set_choice (recordForm->sourceChoice,4); break;
      }
    default:
      break; // Error - do not set rate choice
  }
    
  if (globalInputPort)
  {
    signed char maxL=0;
    signed char maxR=0;
    signed char max3=0;
    signed char max4=0;
    signed char minL=0;
    signed char minR=0;
    signed char min3=0;
    signed char min4=0;
    double realmaxL;
    double realmaxR;
    double realmax3;
    double realmax4;
    double realmax;
    long   i;
    long   j;
    long   k;
    long   l;
    long   length = METERQUEUESIZE;
    long   meterQueueSize_2;

    if (pvBuffer [15] > 0)
      length = pvBuffer [15] * METERQUEUESIZE / 48000;

    if (length > METERQUEUESIZE) length = METERQUEUESIZE;

    if (quadro)
      meterQueueSize_2 = 4 * length;
    else
      meterQueueSize_2 = 2 * length;

    ALreadsamps (globalInputPort,tempInputBuf,meterQueueSize_2);
    
    if (quadro)
    {
      for (i=0,j=1,k=2,l=3; i<meterQueueSize_2; i+=4,j+=4,k+=4,l+=4)
      {
        if (tempInputBuf [i] > maxL) maxL = tempInputBuf [i];
        else if (tempInputBuf [i] < minL) minL = tempInputBuf [i];
        if (tempInputBuf [j] > maxR) maxR = tempInputBuf [j];
        else if (tempInputBuf [j] < minR) minR = tempInputBuf [j];
        if (tempInputBuf [k] > max3) max3 = tempInputBuf [k];
        else if (tempInputBuf [k] < min3) min3 = tempInputBuf [k];
        if (tempInputBuf [l] > max4) max4 = tempInputBuf [l];
        else if (tempInputBuf [l] < min4) min4 = tempInputBuf [l];
      }
    }
    else
    {
      for (i=0,j=1; i<meterQueueSize_2; i+=2,j+=2)
      {
        if (tempInputBuf [i] > maxL) maxL = tempInputBuf [i];
        else if (tempInputBuf [i] < minL) minL = tempInputBuf [i];
        if (tempInputBuf [j] > maxR) maxR = tempInputBuf [j];
        else if (tempInputBuf [j] < minR) minR = tempInputBuf [j];
      }
    }

    if (-((int)minL) > ((int)maxL)) realmaxL = (double) -((int)minL);
    else realmaxL = (double) ((int)maxL);
    if (-((int)minR) > ((int)maxR)) realmaxR = (double) -((int)minR);
    else realmaxR = (double) ((int)maxR);
    
    fl_set_slider_value (recordForm->leftLevel,realmaxL);
    fl_set_slider_value (recordForm->rightLevel,realmaxR);
    
    if (quadro)
    {
      if (-((int)min3) > ((int)max3)) realmax3 = (double) -((int)min3);
      else realmax3 = (double) ((int)max3);
      if (-((int)min4) > ((int)max4)) realmax4 = (double) -((int)min4);
      else realmax4 = (double) ((int)max4);
      
      fl_set_slider_value (recordForm->threeLevel,realmax3);
      fl_set_slider_value (recordForm->fourLevel,realmax4);
    }

    if (realmaxL > 120.0)
      fl_set_object_color (recordForm->leftOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->leftOverload,FL_COL1,FL_COL1);
    
    if (realmaxR > 120.0)
      fl_set_object_color (recordForm->rightOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->rightOverload,FL_COL1,FL_COL1);
    
    if (quadro)
    {
      if (realmax3 > 120.0)
        fl_set_object_color (recordForm->threeOverload,FL_RED,FL_RED);
      else
        fl_set_object_color (recordForm->threeOverload,FL_COL1,FL_COL1);

      if (realmax4 > 120.0)
        fl_set_object_color (recordForm->fourOverload,FL_RED,FL_RED);
      else
        fl_set_object_color (recordForm->fourOverload,FL_COL1,FL_COL1);
    }
    
    realmax = realmaxL;
    if (realmaxR > realmax)
      realmax = realmaxR;
    
    if (quadro)
    {
      if (realmax3 > realmax)
        realmax = realmax3;
      if (realmax4 > realmax)
        realmax = realmax4;
    }
    
    if (realmax >= fl_get_slider_value (recordForm->triggerSlider))
      fl_set_object_color (recordForm->triggerOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->triggerOverload,FL_COL1,FL_COL1);
  }
  
  if (globalOutputPort)
  {
    // Play sine wave at currentPitch (440 hz = A2 = note 57);
    double f;
    long   outputRate = 48000;
    double period;
    long   number;
    long   samples;
    long   length;
    double twopi = 8.0 * atan (1.0);
    double vol;
    static double delta = 0.0;
    
    f = 440.0 * pow (2.0,((currentPitch - 57.0) / 12.0));
    
    if (pvBuffer [7] > 0) outputRate = pvBuffer [7];
    else if (pvBuffer [15] > 0) outputRate = pvBuffer [15];
    
    length = outputRate * SINEOUTQUEUESIZE / 48000;
    if (length > SINEOUTQUEUESIZE) length = SINEOUTQUEUESIZE;

    period  = outputRate / f;
    number  = (long)(length / period);
    samples = (long)(number * period);
    
    vol = 0.2 * 32767.0;
    
    for (long i=0; i<samples; i++)
      tempOutputBuf [i] = (short)(vol * sin ((i + delta) * twopi / period));
    
    ALwritesamps (globalOutputPort,tempOutputBuf,samples);
    
    delta = (number * period) - samples;
  }
  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION updateAudioGlobals (linux)
---------------------------------------------------------------------------*/

#else
int updateAudioGlobals (XEvent *xev,void *data)
{
  static signed char tempInputBuf  [4 * METERQUEUESIZE];
  static short       tempOutputBuf [SINEOUTQUEUESIZE];

  long pvBuffer [16];
  long volLeft;
  long volRight;

  pvBuffer [0]  = AL_OUTPUT_RATE;
  pvBuffer [2]  = AL_INPUT_RATE;
  pvBuffer [4]  = AL_LINUX_DSP_PRESENT;
  pvBuffer [6]  = AL_LINUX_MIXER_PRESENT;
  pvBuffer [8]  = AL_LINUX_CURRENT_INPUT;
  pvBuffer [10] = AL_LINUX_CURRENT_OUTPUT;
  pvBuffer [12] = AL_LINUX_CURRENT_REC;
  pvBuffer [14] = AL_LINUX_MIXER_RECORD;
  ALgetparams (AL_DEFAULT_DEVICE,pvBuffer,16);
  
  if (pvBuffer [7])
  {
    long pvBuffer2 [2];
    
    pvBuffer2 [0] = AL_LINUX_CHANNEL_BASE + pvBuffer [11];
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer2,2);

    // Playback parameters
    volLeft  = pvBuffer2 [1] & 0x000000ff;
    volRight = (pvBuffer2 [1] & 0x0000ff00) >> 8;
  }

  if (pvBuffer [7])
  {
    // Record parameters
    long pvBuffer2 [2];
    
    pvBuffer2 [0] = AL_LINUX_CHANNEL_BASE + pvBuffer [9];
    ALgetparams (AL_DEFAULT_DEVICE,pvBuffer2,2);

    volLeft  = pvBuffer2 [1] & 0x000000ff;
    volRight = (pvBuffer2 [1] & 0x0000ff00) >> 8;
  }
    
  if (globalInputPort)
  {
    signed char maxL=0;
    signed char maxR=0;
    signed char max3=0;
    signed char max4=0;
    signed char minL=0;
    signed char minR=0;
    signed char min3=0;
    signed char min4=0;
    double realmaxL;
    double realmaxR;
    double realmax3;
    double realmax4;
    double realmax;
    long   i;
    long   j;
    long   k;
    long   l;
    long   length = METERQUEUESIZE;
    long   meterQueueSize_2;

    if (pvBuffer [3] > 0)
      length = pvBuffer [3] * METERQUEUESIZE / 48000;

    if (length > METERQUEUESIZE) length = METERQUEUESIZE;

    meterQueueSize_2 = 2 * length;

    ALreadsamps (globalInputPort,tempInputBuf,meterQueueSize_2);
    
    for (i=0,j=1; i<meterQueueSize_2; i+=2,j+=2)
    {
      if (tempInputBuf [i] > maxL) maxL = tempInputBuf [i];
      else if (tempInputBuf [i] < minL) minL = tempInputBuf [i];
      if (tempInputBuf [j] > maxR) maxR = tempInputBuf [j];
      else if (tempInputBuf [j] < minR) minR = tempInputBuf [j];
    }

    if (-((int)minL) > ((int)maxL)) realmaxL = (double) -((int)minL);
    else realmaxL = (double) ((int)maxL);
    if (-((int)minR) > ((int)maxR)) realmaxR = (double) -((int)minR);
    else realmaxR = (double) ((int)maxR);
    
    fl_set_slider_value (recordForm->leftLevel,realmaxL);
    fl_set_slider_value (recordForm->rightLevel,realmaxR);
    
    if (realmaxL > 120.0)
      fl_set_object_color (recordForm->leftOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->leftOverload,FL_COL1,FL_COL1);
    
    if (realmaxR > 120.0)
      fl_set_object_color (recordForm->rightOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->rightOverload,FL_COL1,FL_COL1);
    
    realmax = realmaxL;
    if (realmaxR > realmax)
      realmax = realmaxR;
        
    if (realmax >= fl_get_slider_value (recordForm->triggerSlider))
      fl_set_object_color (recordForm->triggerOverload,FL_RED,FL_RED);
    else
      fl_set_object_color (recordForm->triggerOverload,FL_COL1,FL_COL1);
  }
  
  if (globalOutputPort)
  {
    // Play sine wave at currentPitch (440 hz = A2 = note 57);
    double f;
    long   outputRate = 48000;
    double period;
    long   number;
    long   samples;
    long   length;
    double twopi = 8.0 * atan (1.0);
    double vol;
    static double delta = 0.0;
    
    f = 440.0 * pow (2.0,((currentPitch - 57.0) / 12.0));
    
    if (pvBuffer [1] > 0) outputRate = pvBuffer [1];
    else if (pvBuffer [3] > 0) outputRate = pvBuffer [3];
    
    length = outputRate * SINEOUTQUEUESIZE / 48000;
    if (length > SINEOUTQUEUESIZE) length = SINEOUTQUEUESIZE;

    period  = outputRate / f;
    number  = (long)(length / period);
    samples = (long)(number * period);
    
    vol = 0.2 * 32767.0;
    
    for (long i=0; i<samples; i++)
      tempOutputBuf [i] = (short)(vol * sin ((i + delta) * twopi / period));
    
    ALwritesamps (globalOutputPort,tempOutputBuf,samples);
    
    delta = (number * period) - samples;
  }
  return 0;
}
#endif

/*---------------------------------------------------------------------------
| FUNCTION updateConvertMenu
---------------------------------------------------------------------------*/

void updateConvertMenu (void)
{
  DPSample *sample = fl_get_sample (mainForm->sample);

  if (sample->getChannels () == 1)
  {
    fl_set_menu_item_mode (mainForm->convertMenu,1,FL_PUP_GREY);
    fl_set_menu_item_mode (mainForm->convertMenu,2,FL_PUP_NONE);
    fl_set_menu_item_mode (mainForm->convertMenu,3,FL_PUP_NONE);
  }
  else if (sample->getChannels () == 2)
  {
    fl_set_menu_item_mode (mainForm->convertMenu,1,FL_PUP_NONE);
    fl_set_menu_item_mode (mainForm->convertMenu,2,FL_PUP_GREY);
    fl_set_menu_item_mode (mainForm->convertMenu,3,FL_PUP_NONE);
  }
  else
  {
    fl_set_menu_item_mode (mainForm->convertMenu,1,FL_PUP_NONE);
    fl_set_menu_item_mode (mainForm->convertMenu,2,FL_PUP_NONE);
    fl_set_menu_item_mode (mainForm->convertMenu,3,FL_PUP_GREY);
  }
  
  if (sample->getWidth () == 1)
  {
    fl_set_menu_item_mode (mainForm->convertMenu,4,FL_PUP_GREY);
    fl_set_menu_item_mode (mainForm->convertMenu,5,FL_PUP_NONE);
  }
  else
  {
    fl_set_menu_item_mode (mainForm->convertMenu,4,FL_PUP_NONE);
    fl_set_menu_item_mode (mainForm->convertMenu,5,FL_PUP_GREY);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION updateNameBox
---------------------------------------------------------------------------*/

void updateNameBox (void)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  char titleBar [256];
  
  if (!sample->getValid ())
  {
    strcpy (titleBar,"No sample loaded");
    fl_set_object_label (mainForm->nameBox,titleBar);
    return;
  }
  else if (sample->getFilename () && strlen (sample->getFilename ()) > 0)
  {
    strcpy (titleBar,
      &((sample->getFilename ())
      [stripFilename ((char *) sample->getFilename ())]));
  }
  else
  {
    strcpy (titleBar,"Untitled");
  }
  strcat (titleBar," - ");
  long rate = (long) sample->getRate ();
  sprintf (titleBar + strlen (titleBar),"%d Hz",rate);
  strcat (titleBar," - ");
  int width = sample->getWidth ();
  sprintf (titleBar + strlen (titleBar),"%d bit",width * 8);
  strcat (titleBar," - ");
  int channels = sample->getChannels ();
  if (channels == 1)
    sprintf (titleBar + strlen (titleBar),"Mono");
  else if (channels == 2)
    sprintf (titleBar + strlen (titleBar),"Stereo");
  else if (channels == 4)
    sprintf (titleBar + strlen (titleBar),"Quadro");
  strcat (titleBar," - ");
  long frames = sample->getFrames ();
  sprintf (titleBar + strlen (titleBar),"%d frames",frames);
  strcat (titleBar," - ");
  sprintf (titleBar + strlen (titleBar),"%.3lf secs",(double) frames / rate);
  
  fl_set_object_label (mainForm->nameBox,titleBar);
}

/*---------------------------------------------------------------------------
| FUNCTION updateDisplayDetails
---------------------------------------------------------------------------*/

void updateDisplayDetails (void)
{
  char tempString [50];
  DPSample *sample = fl_get_sample (mainForm->sample);
  long start       = sample->getDisplayStart ();
  long end         = sample->getDisplayEnd ();
  long duration    = end - start;
  double rate      = sample->getRate ();
  
  sprintf (tempString,"%ld",start);
  fl_set_input (displayForm->startText,tempString);
  sprintf (tempString,"%.3lf",start / rate);
  fl_set_input (displayForm->startTimeText,tempString);
  sprintf (tempString,"%ld",end);
  fl_set_input (displayForm->endText,tempString);
  sprintf (tempString,"%.3lf",end / rate);
  fl_set_input (displayForm->endTimeText,tempString);
  sprintf (tempString,"%ld",duration);
  fl_set_object_label (displayForm->durationText,tempString);
  sprintf (tempString,"%.3lf",duration / rate);
  fl_set_object_label (displayForm->durationTimeText,tempString);
  
  fl_set_object_label (displayForm->positionText,"");
  fl_set_object_label (displayForm->timeText,"");
  
  checkManualEdit ();
}

/*---------------------------------------------------------------------------
| FUNCTION checkManualEdit
---------------------------------------------------------------------------*/
void checkManualEdit ()
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  
  if (!sample->getValid ()) return;
  
  if (fl_get_sample_freehand (mainForm->sample) &&
    ((sample->getDisplayEnd () - sample->getDisplayStart ())
    > (mainForm->sample->w) - (2 * BORDERX)))
  {
    fl_set_button (mainForm->manualEdit,0);
    fl_set_sample_freehand (mainForm->sample,0);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION updateRangeDetails
---------------------------------------------------------------------------*/

void updateRangeDetails (void)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  if (!sample->getRangeValid ())
  {
    fl_set_input (rangeForm->startText,"");
    fl_set_input (rangeForm->startTimeText,"");
    fl_set_input (rangeForm->endText,"");
    fl_set_input (rangeForm->endTimeText,"");
    fl_set_object_label (rangeForm->durationText,"");
    fl_set_object_label (rangeForm->durationTimeText,"");
    updateRangeForms ();
    return;
  }
  
  char tempString [50];
  long start    = sample->getRangeStart ();
  long end      = sample->getRangeEnd ();
  long duration = end - start;
  double rate   = sample->getRate ();

  sprintf (tempString,"%ld",start);
  fl_set_input (rangeForm->startText,tempString);
  sprintf (tempString,"%.3lf",start / rate);
  fl_set_input (rangeForm->startTimeText,tempString);
  sprintf (tempString,"%ld",end);
  fl_set_input (rangeForm->endText,tempString);
  sprintf (tempString,"%.3lf",end / rate);
  fl_set_input (rangeForm->endTimeText,tempString);
  sprintf (tempString,"%ld",duration);
  fl_set_object_label (rangeForm->durationText,tempString);
  sprintf (tempString,"%.3lf",duration / rate);
  fl_set_object_label (rangeForm->durationTimeText,tempString);
  updateRangeForms ();
}

/*---------------------------------------------------------------------------
| FUNCTION updateLoopDetails
---------------------------------------------------------------------------*/

void updateLoopDetails (void)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long susLoopMode = sample->getSusLoopMode ();
  long relLoopMode = sample->getRelLoopMode ();
  double rate      = sample->getRate ();

  if (susLoopMode == AF_LOOP_MODE_NOLOOP)
  {
    fl_set_input (loopForm->susStartText,"");
    fl_set_input (loopForm->susStartTimeText,"");
    fl_set_input (loopForm->susEndText,"");
    fl_set_input (loopForm->susEndTimeText,"");
    fl_set_object_label (loopForm->susDurationText,"");
    fl_set_object_label (loopForm->susDurationTimeText,"");
    fl_set_choice (loopForm->susModeChoice,1);
    fl_set_object_label (loopForm->susBPM,"");
    fl_deactivate_object (loopForm->susBeats);
  }
  else
  {
    char tempString [50];
    int  susbeats;
    int  susBPM;
    long start    = sample->getSusLoopStart ();
    long end      = sample->getSusLoopEnd ();
    long duration = end - start;
    
    sprintf (tempString,"%ld",start);
    fl_set_input (loopForm->susStartText,tempString);
    sprintf (tempString,"%.3lf",start / rate);
    fl_set_input (loopForm->susStartTimeText,tempString);
    sprintf (tempString,"%ld",end);
    fl_set_input (loopForm->susEndText,tempString);
    sprintf (tempString,"%.3lf",end / rate);
    fl_set_input (loopForm->susEndTimeText,tempString);
    sprintf (tempString,"%ld",duration);
    fl_set_object_label (loopForm->susDurationText,tempString);
    sprintf (tempString,"%.3lf",duration / rate);
    fl_set_object_label (loopForm->susDurationTimeText,tempString);
    if (susLoopMode == AF_LOOP_MODE_FORW)
      fl_set_choice (loopForm->susModeChoice,2);
    else
      fl_set_choice (loopForm->susModeChoice,3);
    if (!sscanf (fl_get_input (loopForm->susBeats),"%d",&susbeats))
      susbeats = 0;
    if (duration) susBPM = 60 * susbeats * (long) rate / duration;
    else susBPM = 1;
    sprintf (tempString,"%d",susBPM);
    fl_set_object_label (loopForm->susBPM,tempString);
    fl_activate_object (loopForm->susBeats);
  }

  if (relLoopMode == AF_LOOP_MODE_NOLOOP)
  {
    fl_set_input (loopForm->relStartText,"");
    fl_set_input (loopForm->relStartTimeText,"");
    fl_set_input (loopForm->relEndText,"");
    fl_set_input (loopForm->relEndTimeText,"");
    fl_set_object_label (loopForm->relDurationText,"");
    fl_set_object_label (loopForm->relDurationTimeText,"");
    fl_set_choice (loopForm->relModeChoice,1);
    fl_set_object_label (loopForm->relBPM,"");
    fl_deactivate_object (loopForm->relBeats);
  }
  else
  {
    char tempString [50];
    int  relbeats;
    int  relBPM;
    long start    = sample->getRelLoopStart ();
    long end      = sample->getRelLoopEnd ();
    long duration = end - start;
    
    sprintf (tempString,"%ld",start);
    fl_set_input (loopForm->relStartText,tempString);
    sprintf (tempString,"%.3lf",start / rate);
    fl_set_input (loopForm->relStartTimeText,tempString);
    sprintf (tempString,"%ld",end);
    fl_set_input (loopForm->relEndText,tempString);
    sprintf (tempString,"%.3lf",end / rate);
    fl_set_input (loopForm->relEndTimeText,tempString);
    sprintf (tempString,"%ld",duration);
    fl_set_object_label (loopForm->relDurationText,tempString);
    sprintf (tempString,"%.3lf",duration / rate);
    fl_set_object_label (loopForm->relDurationTimeText,tempString);
    if (relLoopMode == AF_LOOP_MODE_FORW)
      fl_set_choice (loopForm->relModeChoice,2);
    else
      fl_set_choice (loopForm->relModeChoice,3);
    if (!sscanf (fl_get_input (loopForm->relBeats),"%d",&relbeats))
      relbeats = 0;
    if (duration) relBPM = 60 * relbeats * (long) rate / duration;
    else relBPM = 1;
    sprintf (tempString,"%d",relBPM);
    fl_set_object_label (loopForm->relBPM,tempString);
    fl_activate_object (loopForm->relBeats);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION updateNewDetails
---------------------------------------------------------------------------*/

void updateNewDetails (void)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  long frames = sample->getFrames ();
  double rate = sample->getRate ();
  char tempString [50];
  sprintf (tempString,"%ld",(long) sample->getRate ());
  fl_set_input (newForm->rateInput,tempString);
  if (sample->getWidth () == 1) fl_set_choice (newForm->resolution,1);
  else fl_set_choice (newForm->resolution,2);
  if (sample->getChannels () == 1) fl_set_choice (newForm->channels,1);
  else fl_set_choice (newForm->channels,2);
//sprintf (tempString,"%ld",frames);
  sprintf (tempString,"%ld",0);
  fl_set_input (newForm->frames,tempString);
//sprintf (tempString,"%.3lf",frames / rate);
  sprintf (tempString,"%.3lf",0.0);
  fl_set_input (newForm->time,tempString);
}

/*---------------------------------------------------------------------------
| FUNCTION interrupt
---------------------------------------------------------------------------*/

void interrupt (int signum,void *data)
{
  if (closeAll (mainForm->mainForm,0) == FL_OK)
    exit (0);
}

/*---------------------------------------------------------------------------
| FUNCTION closeAll
---------------------------------------------------------------------------*/

int closeAll (FL_FORM *form,void *data)
{
  if (form == displayForm->displayForm)
  {
    fl_set_button (mainForm->displayButton,0);
    fl_call_object_callback (mainForm->displayButton);
    fl_redraw_object (mainForm->displayButton);
  }
  else if (form == loopForm->loopForm)
  {
    fl_set_button (mainForm->loopButton,0);
    fl_call_object_callback (mainForm->loopButton);
    fl_redraw_object (mainForm->loopButton);
  }
  else if (form == rangeForm->rangeForm)
  {
    fl_set_button (mainForm->rangeButton,0);
    fl_call_object_callback (mainForm->rangeButton);
    fl_redraw_object (mainForm->rangeButton);
  }
  else if (form == zoomForm->zoomForm)
  {
    fl_set_button (mainForm->zoomButton,0);
    fl_call_object_callback (mainForm->zoomButton);
    fl_redraw_object (mainForm->zoomButton);
  }
  else if (form == playForm->playForm)
  {
    fl_set_button (mainForm->playButton,0);
    fl_call_object_callback (mainForm->playButton);
    fl_redraw_object (mainForm->playButton);
  }
  else if (form == recordForm->recordForm)
  {
    fl_set_button (mainForm->recordButton,0);
    fl_call_object_callback (mainForm->recordButton);
    fl_redraw_object (mainForm->recordButton);
  }
  else if (form == bufferForm->bufferForm)
  {
    fl_set_button (mainForm->bufferButton,0);
    fl_call_object_callback (mainForm->bufferButton);
    fl_redraw_object (mainForm->bufferButton);
  }
  else if (form == mixerForm->mixerForm)
  {
    fl_set_button (mainForm->mixerButton,0);
    fl_call_object_callback (mainForm->mixerButton);
    fl_redraw_object (mainForm->mixerButton);
  }
  else if (form == newForm->newForm)
    fl_call_object_callback (newForm->cancel);
  else if (form == convertForm->convertForm)
    fl_call_object_callback (convertForm->cancel);
  else if (form == mixForm->mixForm)
    fl_call_object_callback (mixForm->cancel);
  else if (form == channelForm->channelForm)
    fl_call_object_callback (channelForm->cancel);
  else if (form == duplicateForm->duplicateForm)
    fl_call_object_callback (duplicateForm->cancel);
  else if (form == panoramaForm->panoramaForm)
    fl_call_object_callback (panoramaForm->cancel);
  else if (form == workspaceForm->workspaceForm)
    fl_call_object_callback (workspaceForm->cancel);
  else if (form == miscForm->miscForm)
    fl_call_object_callback (miscForm->cancel);
  else if (form == rateForm->rateForm)
    fl_call_object_callback (rateForm->cancel);
  else if (form == ampForm->ampForm)
    fl_call_object_callback (ampForm->cancel);
  else if (form == balanceForm->balanceForm)
    fl_call_object_callback (balanceForm->cancel);
  else if (form == bounceForm->bounceForm)
    fl_call_object_callback (bounceForm->cancel);
  else if (form == rampForm->rampForm)
    fl_call_object_callback (rampForm->cancel);
  else if (form == resampleForm->resampleForm)
    fl_call_object_callback (resampleForm->cancel);
  else if (form == pitchForm->pitchForm)
    fl_call_object_callback (pitchForm->cancel);
  else if (form == mixplayForm->mixplayForm)
    fl_call_object_callback (mixplayForm->done);
  else if (form == selectForm->selectForm)
    fl_call_object_callback (selectForm->cancel);
  else if (form == adjustDCForm->adjustDCForm)
    fl_call_object_callback (adjustDCForm->cancel);
  else if (form == initialForm->initialForm)
    fl_call_object_callback (initialForm->cancel);
  else if (form == effectForm->effectForm)
    fl_call_object_callback (effectForm->ok);
  else if (form == executeForm->executeForm)
    fl_call_object_callback (executeForm->cancel);
  else if (form == postForm->postForm)
    fl_call_object_callback (postForm->cancel);
  else if (form == aboutForm->aboutForm)
    fl_call_object_callback (aboutForm->dismiss);
  else if (form == stretchForm->stretchForm)
    fl_call_object_callback (stretchForm->cancel);
  else if (form == convert4to1Form->convert4to1Form)
    fl_call_object_callback (convert4to1Form->cancel);
  else if (form == convert4to2Form->convert4to2Form)
    fl_call_object_callback (convert4to2Form->cancel);
  else if (form == convert2to4Form->convert2to4Form)
    fl_call_object_callback (convert2to4Form->cancel);
  else if (form == channel4Form->channel4Form)
    fl_call_object_callback (channel4Form->cancel);
  else if (form == bounce4Form->bounce4Form)
    fl_call_object_callback (bounce4Form->cancel);
  else if (form == prefsForm->prefsForm)
    fl_call_object_callback (prefsForm->cancel);
  else if (form == macroForm->macroForm)
    fl_call_object_callback (macroForm->OK);
  else if (form == macroexecForm->macroexecForm)
    fl_call_object_callback (macroexecForm->cancel);
  else if (form == macrooutputForm->macrooutputForm)
    fl_call_object_callback (macrooutputForm->OK);
  else if (form == mainForm->mainForm)
  {
    // Check for changed buffers
    
    int i;
    int changed = FALSE;

    for (i=0; i<8; i++)
      if (buffer [i]->getChanged ())
        changed = TRUE;

    if (changed)
      if (!fl_show_question_old
        ("Warning","There are buffers containing unsaved data",
        "Really exit (lose changes) ?"))
        return FL_IGNORE;
    
    if (preferencesChanged)
      if (!fl_show_question_old
        ("Warning","The program preferences have been changed",
        "Really exit (lose changes) ?"))
        return FL_IGNORE;
    
    if (macrosChanged)
      if (!fl_show_question_old
        ("Warning","The program macros have been changed",
        "Really exit (lose changes) ?"))
        return FL_IGNORE;
    
    tooltips_shutdown ();
    fl_finish ();
    delete buffer [0];
    delete buffer [1];
    delete buffer [2];
    delete buffer [3];
    delete buffer [4];
    delete buffer [5];
    delete buffer [6];
    delete buffer [7];
    delete undo [0];
    delete [] undoStr [0];
    delete undo [1];
    delete [] undoStr [1];
    delete redo [0];
    delete [] redoStr [0];
    delete redo [1];
    delete [] redoStr [1];
    delete clip;

    #ifdef TEST
    long pvBuffer [4];
    pvBuffer [0] = AL_INPUT_RATE;
    pvBuffer [1] = 44100;
    pvBuffer [2] = AL_OUTPUT_RATE;
    #ifndef LINUX
    pvBuffer [3] = AL_RATE_INPUTRATE;
    #else
    pvBuffer [3] = 44100;
    #endif
    ALsetparams (AL_DEFAULT_DEVICE,pvBuffer,4);
    #endif

    #ifdef LINUX
    finishAudio ();
    #endif
    
    return FL_OK;
  }
  
  return FL_IGNORE;
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseEffects
---------------------------------------------------------------------------*/

void initialiseEffects (void)
{
  int i;
  int j;
  int k;
  int y;
  FL_OBJECT *obj;
  
  // Initialise effects form
  fl_addto_form (effectForm->effectForm);
  for (i=0,y=50; i<MAXPARAMS; i++,y+=20)
  {
    DSPname[i] = obj = fl_add_text(FL_NORMAL_TEXT,10,y,140,20,"");
      fl_set_object_boxtype(obj,FL_FRAME_BOX);
      fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      
    DSPchan[i] = obj = fl_add_text(FL_NORMAL_TEXT,160,y,30,20,"");
      fl_set_object_boxtype(obj,FL_FRAME_BOX);
      fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      
    DSPinput[i] = obj = fl_add_input(FL_NORMAL_INPUT,200,y,70,20,"");
      fl_set_object_boxtype(obj,FL_FRAME_BOX);
      fl_set_object_callback(obj,effectInput_cb,i);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      
    DSPchoice[i] = obj = fl_add_choice(FL_NORMAL_CHOICE,200,y,70,20,"");
      fl_set_object_boxtype(obj,FL_UP_BOX);
      fl_set_object_callback(obj,effectChoice_cb,i);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);

    DSPunits[i] = obj = fl_add_text(FL_NORMAL_TEXT,280,y,40,20,"");
      fl_set_object_boxtype(obj,FL_FRAME_BOX);
      fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
      
    DSPslider[i] = obj = fl_add_slider(FL_HOR_NICE_SLIDER,330,y,160,20,"");
      fl_set_object_boxtype(obj,FL_UP_BOX);
      fl_set_object_callback(obj,effectSlider_cb,i);
      fl_set_slider_return(obj, FL_RETURN_CHANGED);
      fl_set_object_gravity (obj,NorthWestGravity,NorthWestGravity);
  }
  fl_end_form ();

  // Initialise effects structures
  effects = new allEffectsType;
  effects->effectNo    = 14;
  effects->effect      = new effectType [effects->effectNo];
  effects->effect [0]  = reverberation;
  effects->effect [1]  = gatedReverberation;
  effects->effect [2]  = stereoDelay;
  effects->effect [3]  = modulatedDelay;
  effects->effect [4]  = multiTapDelay;
  effects->effect [5]  = multiDelay;
  effects->effect [6]  = flanger;
  effects->effect [7]  = phaser;
  effects->effect [8]  = distortion;
  effects->effect [9]  = filter;
  effects->effect [10] = autoPan;
  effects->effect [11] = equalizer;
  effects->effect [12] = gate;
  effects->effect [13] = compressorLimiter;
  
  for (i=0; i<effects->effectNo; i++)
  {
    fl_addto_choice (initialForm->type,effects->effect [i].name);
    for (j=0; j<MAXPARAMS; j++)
    {
      if (j < effects->effect [i].paramNo)
      {
        if (effects->effect [i].param [j].type == 0)
        {
          // Real type
          effects->effect [i].param [j].realValue
          = effects->effect [i].param [j].realDefault;
        }
        else
        {
          // Enum type
          effects->effect [i].param [j].enumValue
          = effects->effect [i].param [j].enumDefault;
        }
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION fl_show_question_old
---------------------------------------------------------------------------*/

int fl_show_question_old (const char *s1,const char *s2,const char *s3)
{
  // This is for compatibility with old version of fl_show_question
  int len = 0;
  int showReturn;
  char *finalString;
  
  if (s1)
    len += strlen (s1);
  if (s2)
    len += strlen (s2);
  if (s3)
    len += strlen (s3);
  
  if (!(finalString = new char [len + 3])) return 0;
  strcpy (finalString,"");
  if (s1)
  {
    strcat (finalString,s1);
  }
  if (s2)
  {
    if (s1) strcat (finalString,"\n");
    strcat (finalString,s2);
  }
  if (s3)
  {
    if (s1 || s2) strcat (finalString,"\n");
    strcat (finalString,s3);
  }
  showReturn = fl_show_question (finalString,1);
  delete [] finalString;
  return showReturn;
}

/*---------------------------------------------------------------------------
| FUNCTION waitCursorOn
---------------------------------------------------------------------------*/
void waitCursorOn ()
{
  if (fl_form_is_visible (mainForm->mainForm))
    fl_set_cursor (mainForm->mainForm->window,XC_watch);
  if (fl_form_is_visible (playForm->playForm))
    fl_set_cursor (playForm->playForm->window,XC_watch);
  if (fl_form_is_visible (zoomForm->zoomForm))
    fl_set_cursor (zoomForm->zoomForm->window,XC_watch);
  if (fl_form_is_visible (displayForm->displayForm))
    fl_set_cursor (displayForm->displayForm->window,XC_watch);
  if (fl_form_is_visible (rangeForm->rangeForm))
    fl_set_cursor (rangeForm->rangeForm->window,XC_watch);
  if (fl_form_is_visible (loopForm->loopForm))
    fl_set_cursor (loopForm->loopForm->window,XC_watch);
  if (fl_form_is_visible (newForm->newForm))
    fl_set_cursor (newForm->newForm->window,XC_watch);
  if (fl_form_is_visible (recordForm->recordForm))
    fl_set_cursor (recordForm->recordForm->window,XC_watch);
  if (fl_form_is_visible (convertForm->convertForm))
    fl_set_cursor (convertForm->convertForm->window,XC_watch);
  if (fl_form_is_visible (mixForm->mixForm))
    fl_set_cursor (mixForm->mixForm->window,XC_watch);
  if (fl_form_is_visible (channelForm->channelForm))
    fl_set_cursor (channelForm->channelForm->window,XC_watch);
  if (fl_form_is_visible (duplicateForm->duplicateForm))
    fl_set_cursor (duplicateForm->duplicateForm->window,XC_watch);
  if (fl_form_is_visible (panoramaForm->panoramaForm))
    fl_set_cursor (panoramaForm->panoramaForm->window,XC_watch);
  if (fl_form_is_visible (workspaceForm->workspaceForm))
    fl_set_cursor (workspaceForm->workspaceForm->window,XC_watch);
  if (fl_form_is_visible (miscForm->miscForm))
    fl_set_cursor (miscForm->miscForm->window,XC_watch);
  if (fl_form_is_visible (rateForm->rateForm))
    fl_set_cursor (rateForm->rateForm->window,XC_watch);
  if (fl_form_is_visible (ampForm->ampForm))
    fl_set_cursor (ampForm->ampForm->window,XC_watch);
  if (fl_form_is_visible (balanceForm->balanceForm))
    fl_set_cursor (balanceForm->balanceForm->window,XC_watch);
  if (fl_form_is_visible (bounceForm->bounceForm))
    fl_set_cursor (bounceForm->bounceForm->window,XC_watch);
  if (fl_form_is_visible (rampForm->rampForm))
    fl_set_cursor (rampForm->rampForm->window,XC_watch);
  if (fl_form_is_visible (resampleForm->resampleForm))
    fl_set_cursor (resampleForm->resampleForm->window,XC_watch);
  if (fl_form_is_visible (pitchForm->pitchForm))
    fl_set_cursor (pitchForm->pitchForm->window,XC_watch);
  if (fl_form_is_visible (bufferForm->bufferForm))
    fl_set_cursor (bufferForm->bufferForm->window,XC_watch);
  if (fl_form_is_visible (mixplayForm->mixplayForm))
    fl_set_cursor (mixplayForm->mixplayForm->window,XC_watch);
  if (fl_form_is_visible (selectForm->selectForm))
    fl_set_cursor (selectForm->selectForm->window,XC_watch);
  if (fl_form_is_visible (adjustDCForm->adjustDCForm))
    fl_set_cursor (adjustDCForm->adjustDCForm->window,XC_watch);
  if (fl_form_is_visible (initialForm->initialForm))
    fl_set_cursor (initialForm->initialForm->window,XC_watch);
  if (fl_form_is_visible (effectForm->effectForm))
    fl_set_cursor (effectForm->effectForm->window,XC_watch);
  if (fl_form_is_visible (executeForm->executeForm))
    fl_set_cursor (executeForm->executeForm->window,XC_watch);
  if (fl_form_is_visible (postForm->postForm))
    fl_set_cursor (postForm->postForm->window,XC_watch);
  if (fl_form_is_visible (aboutForm->aboutForm))
    fl_set_cursor (aboutForm->aboutForm->window,XC_watch);
  if (fl_form_is_visible (stretchForm->stretchForm))
    fl_set_cursor (stretchForm->stretchForm->window,XC_watch);
  if (fl_form_is_visible (convert4to1Form->convert4to1Form))
    fl_set_cursor (convert4to1Form->convert4to1Form->window,XC_watch);
  if (fl_form_is_visible (convert4to2Form->convert4to2Form))
    fl_set_cursor (convert4to2Form->convert4to2Form->window,XC_watch);
  if (fl_form_is_visible (convert2to4Form->convert2to4Form))
    fl_set_cursor (convert2to4Form->convert2to4Form->window,XC_watch);
  if (fl_form_is_visible (channel4Form->channel4Form))
    fl_set_cursor (channel4Form->channel4Form->window,XC_watch);
  if (fl_form_is_visible (bounce4Form->bounce4Form))
    fl_set_cursor (bounce4Form->bounce4Form->window,XC_watch);
  if (fl_form_is_visible (prefsForm->prefsForm))
    fl_set_cursor (prefsForm->prefsForm->window,XC_watch);
  if (fl_form_is_visible (mixerForm->mixerForm))
    fl_set_cursor (mixerForm->mixerForm->window,XC_watch);
  if (fl_form_is_visible (macroForm->macroForm))
    fl_set_cursor (macroForm->macroForm->window,XC_watch);
  if (fl_form_is_visible (macrooutputForm->macrooutputForm))
    fl_set_cursor (macrooutputForm->macrooutputForm->window,XC_watch);
  if (fl_form_is_visible (macroexecForm->macroexecForm))
    fl_set_cursor (macroexecForm->macroexecForm->window,XC_watch);
  XFlush (fl_get_display ());
}

/*---------------------------------------------------------------------------
| FUNCTION waitCursorOff
---------------------------------------------------------------------------*/
void waitCursorOff ()
{
  if (fl_form_is_visible (mainForm->mainForm))
    fl_reset_cursor (mainForm->mainForm->window);
  if (fl_form_is_visible (playForm->playForm))
    fl_reset_cursor (playForm->playForm->window);
  if (fl_form_is_visible (zoomForm->zoomForm))
    fl_reset_cursor (zoomForm->zoomForm->window);
  if (fl_form_is_visible (displayForm->displayForm))
    fl_reset_cursor (displayForm->displayForm->window);
  if (fl_form_is_visible (rangeForm->rangeForm))
    fl_reset_cursor (rangeForm->rangeForm->window);
  if (fl_form_is_visible (loopForm->loopForm))
    fl_reset_cursor (loopForm->loopForm->window);
  if (fl_form_is_visible (newForm->newForm))
    fl_reset_cursor (newForm->newForm->window);
  if (fl_form_is_visible (recordForm->recordForm))
    fl_reset_cursor (recordForm->recordForm->window);
  if (fl_form_is_visible (convertForm->convertForm))
    fl_reset_cursor (convertForm->convertForm->window);
  if (fl_form_is_visible (mixForm->mixForm))
    fl_reset_cursor (mixForm->mixForm->window);
  if (fl_form_is_visible (channelForm->channelForm))
    fl_reset_cursor (channelForm->channelForm->window);
  if (fl_form_is_visible (duplicateForm->duplicateForm))
    fl_reset_cursor (duplicateForm->duplicateForm->window);
  if (fl_form_is_visible (panoramaForm->panoramaForm))
    fl_reset_cursor (panoramaForm->panoramaForm->window);
  if (fl_form_is_visible (workspaceForm->workspaceForm))
    fl_reset_cursor (workspaceForm->workspaceForm->window);
  if (fl_form_is_visible (miscForm->miscForm))
    fl_reset_cursor (miscForm->miscForm->window);
  if (fl_form_is_visible (rateForm->rateForm))
    fl_reset_cursor (rateForm->rateForm->window);
  if (fl_form_is_visible (ampForm->ampForm))
    fl_reset_cursor (ampForm->ampForm->window);
  if (fl_form_is_visible (balanceForm->balanceForm))
    fl_reset_cursor (balanceForm->balanceForm->window);
  if (fl_form_is_visible (bounceForm->bounceForm))
    fl_reset_cursor (bounceForm->bounceForm->window);
  if (fl_form_is_visible (rampForm->rampForm))
    fl_reset_cursor (rampForm->rampForm->window);
  if (fl_form_is_visible (resampleForm->resampleForm))
    fl_reset_cursor (resampleForm->resampleForm->window);
  if (fl_form_is_visible (pitchForm->pitchForm))
    fl_reset_cursor (pitchForm->pitchForm->window);
  if (fl_form_is_visible (bufferForm->bufferForm))
    fl_reset_cursor (bufferForm->bufferForm->window);
  if (fl_form_is_visible (mixplayForm->mixplayForm))
    fl_reset_cursor (mixplayForm->mixplayForm->window);
  if (fl_form_is_visible (selectForm->selectForm))
    fl_reset_cursor (selectForm->selectForm->window);
  if (fl_form_is_visible (adjustDCForm->adjustDCForm))
    fl_reset_cursor (adjustDCForm->adjustDCForm->window);
  if (fl_form_is_visible (initialForm->initialForm))
    fl_reset_cursor (initialForm->initialForm->window);
  if (fl_form_is_visible (effectForm->effectForm))
    fl_reset_cursor (effectForm->effectForm->window);
  if (fl_form_is_visible (executeForm->executeForm))
    fl_reset_cursor (executeForm->executeForm->window);
  if (fl_form_is_visible (postForm->postForm))
    fl_reset_cursor (postForm->postForm->window);
  if (fl_form_is_visible (aboutForm->aboutForm))
    fl_reset_cursor (aboutForm->aboutForm->window);
  if (fl_form_is_visible (stretchForm->stretchForm))
    fl_reset_cursor (stretchForm->stretchForm->window);
  if (fl_form_is_visible (convert4to1Form->convert4to1Form))
    fl_reset_cursor (convert4to1Form->convert4to1Form->window);
  if (fl_form_is_visible (convert4to2Form->convert4to2Form))
    fl_reset_cursor (convert4to2Form->convert4to2Form->window);
  if (fl_form_is_visible (convert2to4Form->convert2to4Form))
    fl_reset_cursor (convert2to4Form->convert2to4Form->window);
  if (fl_form_is_visible (channel4Form->channel4Form))
    fl_reset_cursor (channel4Form->channel4Form->window);
  if (fl_form_is_visible (bounce4Form->bounce4Form))
    fl_reset_cursor (bounce4Form->bounce4Form->window);
  if (fl_form_is_visible (prefsForm->prefsForm))
    fl_reset_cursor (prefsForm->prefsForm->window);
  if (fl_form_is_visible (mixerForm->mixerForm))
    fl_reset_cursor (mixerForm->mixerForm->window);
  if (fl_form_is_visible (macroForm->macroForm))
    fl_reset_cursor (macroForm->macroForm->window);
  if (fl_form_is_visible (macrooutputForm->macrooutputForm))
    fl_reset_cursor (macrooutputForm->macrooutputForm->window);
  if (fl_form_is_visible (macroexecForm->macroexecForm))
    fl_reset_cursor (macroexecForm->macroexecForm->window);
  XFlush (fl_get_display ());
}

/*---------------------------------------------------------------------------
| FUNCTION rint (for Solaris)
---------------------------------------------------------------------------*/

#ifdef SUN
double rint (double x)
{
  double temp = floor (x + 0.5);
  if ((x + 0.5) == temp)
    if (temp / 2.0 != floor (temp / 2.0))
      return temp - 1.0;
  return temp;
}
#endif

/*---------------------------------------------------------------------------
| FUNCTION strcasecmp (for Solaris)
---------------------------------------------------------------------------*/

#ifdef SUN
int strcasecmp (const char *s1,const char *s2)
{
  int i;
  
  if (!s1 && !s2) return 0;
  if (!s1) return -1;
  if (!s2) return 1;
  i = 0;
  while (s1 [i] && s2 [i] && (toupper (s1 [i]) == toupper (s2 [i])))
    i++;
  if (!s1 [i] && !s2 [i]) return 0;
  if (!s1 [i]) return -1;
  if (!s2 [i]) return 1;
  if (toupper (s1 [i]) < toupper (s2 [i])) return -1;
  return 1;
}
#endif

/*---------------------------------------------------------------------------
| BUG HACK XpmLibraryVersion
---------------------------------------------------------------------------*/

#ifndef LINUX
int XpmLibraryVersion = 0;
#endif

/***************************************************************************/
