/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xpm.c,v 1.5 1999/11/10 02:18:41 bbarrett Exp $
 *
 *	Function:	- create a pixmap from an XPM pixmap description
 */

#define _NO_PROTO

#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <Xm/Xm.h>

#include "lam.h"

/*
 * global functions
 */
void			xpm_build();

/*
 * external functions
 */
extern int4		stoi();

/*
 * local functions
 */
static int		parse_xpm_desc();
static int		parse_xpm_color();
static int		createximage();
static void		setimagepixels();
static void		setimagepixels8();
int colorhash();

/*
 *	xpm_build
 *
 *	Function:	- build pixmap for arbitrary visual
 */
void
xpm_build(w, xpm, bg, p, wth, hgt)

Widget			w;
char			*xpm[];
Pixel			bg;
Pixmap			*p;
int			*wth;
int			*hgt;

{
	Pixmap		pix;		/* final pixmap */
	XImage		*img;		/* constructed image */
	XColor		col;		/* color specification */
	GC		gc;		/* simply for copying */
	Pixel		col2pix[65537];	/* quick color lookup table */
	Pixel		pixels[65537];	/* allocated pixels */
	int		nalloc;		/* number of pixels allocated */
	int		i, j;		/* favourite indices */
	char		*img_pixels;	/* XImage pixels */
	Pixel		*p_img;		/* ptr in img_pixels */
	char		*p_xpm;		/* ptr in xpm_pixels */
	char            pixel_desc[2];
	int		width;		/* pixmap width */
	int		height;		/* pixmap height */
	int		ncols;		/* num. colors in pixmap */
	char		c[2];		/* character code for color */
	char 		rgb[64];	/* string specifying color value */
	int		depth;		/* depth of display */
	Visual		*visual;	/* visual being used */
	Display		*display;	/* display being used */
	Colormap	colormap;	/* colormap being used */
	int             char_pixel;
	
	*p = XmUNSPECIFIED_PIXMAP;

	display = XtDisplay(w);
	depth = XDefaultDepth(display, DefaultScreen(display));
        visual = DefaultVisual(display, DefaultScreen(display));
	colormap = DefaultColormap(display, DefaultScreen(display));
	
        if (parse_xpm_desc(xpm[0], &width, &height, &ncols, &char_pixel))
	  return;



	if ((ncols == 0) || (width == 0) || (height == 0) || (char_pixel ==0))
	  return;

	/* This will barf badly if char_pixel > 2.  So die now. */
	if (char_pixel > 2)
	  return;

/*
 * Parse and allocate colors.
 */
	nalloc = 0;
	for (i = 1; i <= ncols; ++i) {

		parse_xpm_color(xpm[i], c, rgb, char_pixel);

		if (strcmp("None", rgb)) {
			if (XParseColor(display, colormap, rgb, &col) &&
					XAllocColor(display, colormap, &col)) {
				col2pix[color_hash(c, char_pixel)] = col.pixel;
				pixels[nalloc++] = col.pixel;
			} else {
/*
 * Color allocation failed. Free previously obtained colors and exit.
 */
				if (nalloc) {
					XFreeColors(display, colormap, pixels,
							nalloc, 0);
				}
				return;
			}
		}
		else {
			col2pix[color_hash(c, char_pixel)] = bg;
		}
	}

	img_pixels = malloc((unsigned) (width * height * sizeof(Pixel)));
	if (img_pixels == 0) return;

	p_img = (Pixel *) img_pixels;
/*
 * Parse the image pixel data.
 */
	for (i = ncols + 1; i <= height + ncols; ++i) {
		p_xpm = xpm[i];

		for (j = 0; j < width; ++j) {
		  strncpy(pixel_desc, p_xpm, char_pixel);
		  *p_img++ = col2pix[color_hash(pixel_desc, char_pixel)];
		  p_xpm = p_xpm + char_pixel;
			
		}
	}
/*
 * Create an image from the pixel data and then copy it to a pixmap.
 */
	createximage(display, visual, depth, width, height, &img);

	if (depth == 8)
		setimagepixels8(img, width, height, img_pixels);
	else
		setimagepixels(img, width, height, img_pixels);

	pix = XCreatePixmap(display, RootWindowOfScreen(XtScreen(w)),
				width, height, depth);

	gc = XCreateGC(display, RootWindowOfScreen(XtScreen(w)), 0, (Pixmap) 0);
	XSetFunction(display, gc, GXcopy);

	XPutImage(display, pix, gc, img, 0, 0, 0, 0, width, height);

	XFreeGC(display, gc);
	free(img_pixels);
	XDestroyImage(img);

	*wth = width;
	*hgt = height;
	*p = pix;
}

int
color_hash(char *color, int char_pixel)
{
  int big_number;
  int output;

  if (char_pixel == 1) {
    return ((int) color[0]);
  } else {
    big_number = ((int) color[1]) * 256;
    output = (((int) color[0]) + big_number);
  }
  return output;
}

static int
parse_xpm_desc(desc, width, height, ncols, char_pixel)

char		*desc;
int		*width;
int		*height;
int		*ncols;
int *char_pixel;

{
	char	*tok;
	char    buf[80];

	strcpy(buf, desc);
	
	tok = strtok(buf, " \t");
	*width = (int) stoi((unsigned char *)tok);
	
	tok = strtok(0, " \t");
	*height = (int) stoi((unsigned char *)tok);
	
	tok = strtok(0, " \t");
	*ncols = (int) stoi((unsigned char *)tok);

	tok = strtok(0, " \t");
	*char_pixel = (int) stoi((unsigned char *)tok);

	return(0);
}

static int
parse_xpm_color(desc, c, rgb, char_pixel)

char		*desc;
char		*c;
char		*rgb;
int char_pixel;

{
	char	*tok;
	char    buf[80];

	strcpy(buf, desc + char_pixel);
	strncpy(c, desc, char_pixel);
	
	tok = strtok(buf, " \t");

	while (tok && *tok != 'c') {
		tok = strtok(0, " \t");
		tok = strtok(0, " \t");
	}

	tok = strtok(0, " \t");

	if (! tok) return(LAMERROR);

	strcpy(rgb, tok);

	return(0);
}

/*
 * All the code following this comment is a modified version of code
 * from the XPM library.  The copyright below applies to this code only.
 *
 *
 * Copyright (C) 1989-95 GROUPE BULL
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * GROUPE BULL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of GROUPE BULL shall not be
 * used in advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from GROUPE BULL.
 */

#define zindex(x, y, img) ((y) * img->bytes_per_line) + \
    (((x) * img->bits_per_pixel) >> 3)

#define zindex8(x, y, img) ((y) * img->bytes_per_line) + (x)

static int
createximage(display, visual, depth, width, height, pimg)

Display 		*display;
Visual 			*visual;
unsigned int 		depth;
unsigned int 		width;
unsigned int 		height;
XImage 			**pimg;

{
	int bitmap_pad;

	if (depth > 16) bitmap_pad = 32;
	else if (depth > 8) bitmap_pad = 16;
	else bitmap_pad = 8;
/*
 * Create the XImage with data = 0 and bytes_per_line = 0
 */
	*pimg = XCreateImage(display, visual, depth, ZPixmap, 0, 0,
				width, height, bitmap_pad, 0);
/*
 * Now that bytes_per_line must have been set properly allocate data.
 */
	(*pimg)->data = (char *) malloc((*pimg)->bytes_per_line * height);
	memset((*pimg)->data, 0, (*pimg)->bytes_per_line * height);
	

	if ((*pimg)->data == 0) {
		XDestroyImage(*pimg);
		*pimg = 0;
		return(LAMERROR);
	}

	return(0);
}

/*
 * The functions below are written from X11R5 MIT's code (XImUtil.c)
 *
 * The idea is to have faster functions than the standard XPutPixel function
 * to build the image data. Indeed we can speed up things by suppressing tests
 * performed for each pixel. We do the same tests but at the image level.
 * We also assume that we use only ZPixmap images with null offsets.
 */

static unsigned char reverse_byte[0x100] = {
    0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
    0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
    0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
    0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
    0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
    0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
    0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
    0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
    0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
    0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
    0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
    0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
    0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
    0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
    0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
    0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
    0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
    0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
    0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
    0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
    0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
    0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
    0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
    0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
    0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
    0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
    0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
    0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
    0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
    0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
    0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
    0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff
};

static void
reverse_bytes(pb, n)

unsigned char		*pb;
int			n;

{
	for ( ; n > 0; n--, pb++) *pb = reverse_byte[*pb];
}

static void
znorm(pb, img)

unsigned char		*pb;
XImage 			*img;

{
	unsigned char 	c;

	switch (img->bits_per_pixel) {

	case 2:
		reverse_bytes(pb, 1);
		break;

	case 4:
		*pb = ((*pb >> 4) & 0xF) | ((*pb << 4) & ~0xF);
		break;

	case 16:
		c = *pb;
		*pb = *(pb + 1);
		*(pb + 1) = c;
		break;

	case 24:
		c = *(pb + 2);
		*(pb + 2) = *pb;
		*pb = c;
		break;

	case 32:
		c = *(pb + 3);
		*(pb + 3) = *pb;
		*pb = c;
		c = *(pb + 2);
		*(pb + 2) = *(pb + 1);
		*(pb + 1) = c;
		break;
	}
}

static unsigned char lomask[0x09] = {
0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};
static unsigned char himask[0x09] = {
0xff, 0xfe, 0xfc, 0xf8, 0xf0, 0xe0, 0xc0, 0x80, 0x00};

static void
putbits(src, dstoffset, numbits, dst)

char 			*src;		/* address of source bit string */
int 			dstoffset;	/* bit offset into destination */
int 			numbits;	/* number of bits to copy */
char 			*dst;		/* address of dest. bit string */

{
	unsigned char 	chlo, chhi;
	int 		hibits;

	dst = dst + (dstoffset >> 3);
	dstoffset = dstoffset & 7;
	hibits = 8 - dstoffset;
	chlo = *dst & lomask[dstoffset];

	while (1) {
		chhi = (*src << dstoffset) & himask[dstoffset];

		if (numbits <= hibits) {
			chhi = chhi & lomask[dstoffset + numbits];
			*dst = (*dst & himask[dstoffset + numbits]) | chlo
				| chhi;
			break;
		}

		*dst = chhi | chlo;
		dst++;
		numbits = numbits - hibits;
		chlo = (unsigned char) (*src & himask[hibits]) >> hibits;
		src++;

		if (numbits <= dstoffset) {
			chlo = chlo & lomask[numbits];
			*dst = (*dst & himask[numbits]) | chlo;
			break;
		}

		numbits = numbits - dstoffset;
	}
}

/*
 * Write pixels into an arbitrary depth Z image data structure.
 */
static void
setimagepixels(image, width, height, pixels)

XImage 			*image;
unsigned int 		width;
unsigned int 		height;
Pixel 			*pixels;

{
	char 		*src;
	char 		*dst;
	int 		x, y, i;
	char 		*data;
	Pixel 		pixel, px;
	int 		nbytes, depth, ibpp;

	data = image->data;
	depth = image->depth;
	ibpp = image->bits_per_pixel;
	
	for (y = 0; y < height; y++)
		for (x = 0; x < width; x++, pixels++) {
			pixel = *pixels;

			if (depth == 4) pixel &= 0xf;

			for (i = 0, px = pixel; i < sizeof(unsigned long);
			     			i++, px >>= 8)
				((unsigned char *) &pixel)[i] = px;
			
			src = &data[zindex(x, y, image)];
			dst = (char *) &px;
			px = 0;
			nbytes = (ibpp + 7) >> 3;

			for (i = nbytes; --i >= 0;) *dst++ = *src++;

			if (image->byte_order == MSBFirst)
				znorm((unsigned char *) &px, image);

			putbits((char *) &pixel, (x * ibpp) & 7, ibpp,
					(char *) &px);

			if (image->byte_order == MSBFirst)
				znorm((unsigned char *) &px, image);

			src = (char *) &px;
			dst = &data[zindex(x, y, image)];

			for (i = nbytes; --i >= 0;) *dst++ = *src++;
		}
}

/*
 * Write pixels into a 8-bits Z image data structure.
 */
static void
setimagepixels8(image, width, height, pixels)

XImage 			*image;
unsigned int 		width;
unsigned int 		height;
Pixel 			*pixels;

{
	char 		*data;
	int 		y;
	int 		bpl = image->bytes_per_line;
	char 		*data_ptr, *max_data;

	data = image->data;

	for (y = 0; y < height; y++) {
		data_ptr = data;
		max_data = data_ptr + width;

		while (data_ptr < max_data)
			*(data_ptr++) = (char) *pixels++;

		data += bpl;
	}
}
