/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_pophelp.c,v 1.4 1999/11/11 04:47:33 arodrig6 Exp $
 *
 *	Function:	- popup help for buttons
 *
 *			- When the cursor enters a button with attached help,
 *			  after a short delay a help message will be popped up.
 *			  If the button is pressed or the cursor leaves the
 *			  button before the help message is popped up the
 *			  popup is cancelled.
 *			  Once help for a button is popped up it stays up
 *			  until the cursor leaves the button or the button
 *			  is pressed.
 */

#include <Xm/Frame.h>
#include <Xm/LabelG.h>
#include <string.h>

#include "all_list.h"

#define LABEL_MAX	127			/* max. length of help msg. */
#define POPHELP_DELAY	(1000L)			/* delay popup for 1 sec. */

/*
 * Public functions.
 */
void			xmpi_add_pophelp();

/*
 * Private functions.
 */
static void		pophelp();
static void		popup();
static void		create_popup();
static void		destroy_cb();
static void		arm_cb();
static int		pophelp_cmp();

/*
 * Private variables.
 */
struct pophelp {
	Widget		ph_button;		/* button help attached to */
	Widget		ph_shell;		/* popup help message shell */
	XtIntervalId    ph_timerid;		/* delay timer for popping up */
	char		ph_text[LABEL_MAX + 1];	/* help message text */
	int		ph_up;			/* help currently popped up? */
};

static LIST		*help = 0;		/* list of popup helps */
static Pixel		black;			/* black colour */

/*
 *	xmpi_add_pophelp
 *
 *	Function:	- add popup help for a button
 *			- replaces previous help if any
 *	Accepts:	- button
 *			- help message
 */
void
xmpi_add_pophelp(button, msg)

Widget			button;
char			*msg;

{
	struct pophelp	newpop;
	struct pophelp	*pop;
/*
 * Initialize application help list and colours if necessary.
 */
	memset(&newpop, 0, sizeof(newpop));
	if (help == 0) {
		help = al_init(sizeof(struct pophelp), pophelp_cmp);
		if (help == 0) return;

		black = BlackPixel(XtDisplay(button),
					DefaultScreen(XtDisplay(button)));
	}
/*
 * Check if button already has help.
 */
	newpop.ph_button = button;

	if ((pop = al_find(help, &newpop))) {
/*
 * Is the new help different to the old?
 */
		if (strcmp(pop->ph_text, msg)) {
/*
 * Set new message text and destroy the old help popup if any.  A new
 * popup shell will be created when the help is triggered if ever.  
 */
			strncpy(pop->ph_text, msg, LABEL_MAX);
			if (pop->ph_shell) {
				if (pop->ph_timerid) {
					XtRemoveTimeOut(pop->ph_timerid);
					pop->ph_timerid = 0;
				}
				XtDestroyWidget(pop->ph_shell);
				pop->ph_shell = 0;
				pop->ph_up = 0;
			}
		}
	} else {
/*
 * Add a new entry to the help list.  The popup help shell will be
 * created when the help is triggered if ever.  
 */
		newpop.ph_shell = 0;
		newpop.ph_timerid = 0;
		newpop.ph_up = 0;
		strncpy(newpop.ph_text, msg, LABEL_MAX);

		pop = al_append(help, &newpop);
		if (pop == 0) return;
/*
 * Add callbacks for removing the help when the button is destroyed and
 * for canceling help when it is pressed.
 */
		XtAddCallback(button, XmNdestroyCallback, destroy_cb, NULL);
		XtAddCallback(button, XmNarmCallback, arm_cb, (XtPointer) pop);
/*
 * Set up the button to generate enter/leave events.
 */
		XtAddEventHandler(button,
			EnterWindowMask | LeaveWindowMask,
			False, pophelp, (XtPointer) pop);
	}
}

/*
 *	pophelp
 *
 *	Function:	- handler for button enter and leave events
 *	Accepts:	- button
 *			- handler data
 *			- event
 */
static void
pophelp(button, data, event)

Widget			button;
void			*data;
XEvent			*event;

{
	struct pophelp	*pop = data;

	if (event->type == EnterNotify) {

		if (event->xcrossing.subwindow
				|| event->xcrossing.mode == NotifyUngrab) {
			return;
		}
/*
 * Create the popup if necessary and pop it up.
 */
		if (pop->ph_shell == 0) {
			create_popup(pop);
		}

		pop->ph_timerid = XtAppAddTimeOut(
			XtWidgetToApplicationContext(pop->ph_button), 
			POPHELP_DELAY, popup, (XtPointer) pop);
	}
	else if (event->type == LeaveNotify) {
/*
 * Cancel pending popup if any.
 */
		if (pop->ph_timerid) {
			XtRemoveTimeOut(pop->ph_timerid);
		}

		if (pop->ph_up) {
			XtPopdown(pop->ph_shell);
			pop->ph_up = 0;
		}
	}
}

/*
 *	create_popup
 *
 *	Function:	- create a popup help shell and message
 *	Accepts:	- popup help
 */
static void
create_popup(pop)

struct pophelp		*pop;

{
	Widget		frame;
	XmString	xstr;

	pop->ph_shell = XtVaAppCreateShell("detail_shell", "XMPI",
				applicationShellWidgetClass,
				XtDisplayOfObject(pop->ph_button),
				XmNoverrideRedirect, True, NULL);

	xstr = XmStringCreateSimple(pop->ph_text);
/*
 * We want a simple black line edge around the label.
 */
	frame = XtVaCreateManagedWidget("popup_help",
				xmFrameWidgetClass, pop->ph_shell,
				XmNmarginWidth, 1,
				XmNmarginHeight, 1,
				XmNbottomShadowColor, black,
				XmNtopShadowColor, black,
				XmNshadowType, XmSHADOW_ETCHED_OUT,
				XmNshadowThickness, 1, NULL);

	XtVaCreateManagedWidget("popup_help_label",
				xmLabelGadgetClass, frame,
				XmNlabelString, xstr,
				NULL);

	XmStringFree(xstr);
}

/*
 *	popup
 *
 *	Function:	- pops up help message
 *			- this is only ever invoked by the help timer firing
 *	Accepts:	- popup help
 */
static void
popup(pop)

struct pophelp		*pop;

{
	Position	x, y;			/* button position in root */
	Dimension	border, width, height;	/* button sizes */

	pop->ph_timerid = 0;
/*
 * Find the position of the button in the root window and the button's
 * border size, width and height.  These are recalculated each time
 * because the button may change size or position.  
 */
	XtTranslateCoords(pop->ph_button, (Position) 0, (Position) 0, &x, &y);

	XtVaGetValues(pop->ph_button,
			XmNborderWidth, &border,
			XmNheight, &height,
			XmNwidth, &width, NULL);
/*
 * Set the popup's position to have the top left corner half way along
 * and just below the button.  
 */
        XtVaSetValues(pop->ph_shell,
                        XmNx, x + border + width / 2,
                        XmNy, y + 2 * border + height, NULL);

	pop->ph_up = 1;
	XtPopup(pop->ph_shell, XtGrabNone);
}

/*
 *	pophelp_cmp
 *
 *	Function:	- compare two buttons
 *	Accepts:	- ptr to two entries
 *	Returns:	- 0 if same button else 1
 */
static int
pophelp_cmp(p1, p2)

struct pophelp		*p1, *p2;

{
	return(p1->ph_button != p2->ph_button);
}

/*
 *      destroy_cb
 *
 *      Function:       - button destroy callback
 *			- cleans up any help associated with the button
 *      Accepts:        - button being destroyed
 */
static void
destroy_cb(button)

Widget                  button;

{
	struct pophelp	newpop;
	struct pophelp	*pop;
/*
 * Find the help entry for the button, destroy the popup shell if any
 * and then delete the help entry.  
 */
	newpop.ph_button = button;
	pop = al_find(help, &newpop);

	if (pop->ph_timerid) {
		XtRemoveTimeOut(pop->ph_timerid);
	}

	if (pop->ph_shell) {
		XtDestroyWidget(pop->ph_shell);
	}

	al_delete(help, pop);
}

/*
 *	arm_cb
 *
 *	Function:	- button arm callback
 *			- cancel any pending or popped up help
 *	Accepts:	- button
 *			- popup help
 */
static void
arm_cb(button, pop)

Widget			button;
struct pophelp		*pop;

{
	if (pop->ph_up) {
		XtPopdown(pop->ph_shell);
		pop->ph_up = 0;
	} else if (pop->ph_timerid) {
		XtRemoveTimeOut(pop->ph_timerid);
		pop->ph_timerid = 0;
	}
}
