/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: meta.m,v 1.24 1998/01/05 01:15:43 tiggr Exp $  */

#import "tr.h"
#import "tom/trt.h"

id <TLString> default_asi_value = @"0";

#define META_OUTPUT_TABLE_P  flag0
#define META_OUTPUT_BUCKETS_P  flag1

/* A dictionary from method dispatch bucket (as a vector of strings) to the
   name of an actual bucket, which has already been defined.  */
TLDictionary *mdt_buckets;

/* Dictionary from meta to a vector of method dispatch bucket names.  */
TLDictionary *meta_mdts;

/* The number of unique method dispatch buckets.  */
int bucket_count;

@implementation LTIMeta (tr)

-(TLCons *) addExtensionDescriptions: (TLCons *) l mark: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *se;
  LTTMeta *sm;

  if (mark == k)
    return l;
  mark = k;

  e = [[structure stateSupers] enumerator];
  while ((sm = [e nextObject]))
    l = [[sm semantics] addExtensionDescriptions: l mark: k];

  e = [[structure extensions] enumerator];
  while ((se = [e nextObject]))
    l = [[se semantics] addExtensionDescriptions: l];

  return l;
}

-(TLCons *) addReferenceVariables: (TLCons *) l mark: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *se;
  LTTMeta *sm;

  if (mark == k)
    return l;
  mark = k;

  e = [[structure stateSupers] enumerator];
  while ((sm = [e nextObject]))
    l = [[sm semantics] addReferenceVariables: l mark: k];

  e = [[structure extensions] enumerator];
  while ((se = [e nextObject]))
    l = [[se semantics] addReferenceVariables: l];

  return l;
}

-(TLCons *) compile: (id) s
	       unit: (LTTUnit *) u
      extensionList: (TLCons *) exts
	       mark: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *str;

  if (mark == k)
    return exts;
  mark = k;

  e = [[structure stateSupers] enumerator];
  while ((str = [e nextObject]))
    exts = [[str semantics] compile: s unit: u extensionList: exts mark: k];

  e = [[structure behaviourSupers] enumerator];
  while ((str = [e nextObject]))
    exts = [[str semantics] compile: s unit: u extensionList: exts mark: k];

  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    exts = [[x semantics] compile: s unit: u extensionList: exts];

  return exts;
}

-(id) compileReferenceDeclaration: (id) s
{
  id <TLString> refname;

  if (declared_ref)
    return s;
  declared_ref = 1;

  refname = [structure outputTypeName];
  s = formac (s, @"typedef struct %@ *%@;\n", refname, refname);

  return s;
}

-(void) dontCompileReferenceDeclaration
{
  declared_ref = 1;
}

-(id) compileDeclaration: (id) s
{
  id <TLString> refname;
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *m;

  if (declared_str)
    return s;
  declared_str = 1;

  if (!declared_ref)
    [self compileReferenceDeclaration: s];

  refname = [structure outputTypeName];

  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    s = [[x semantics] compileTypeDeclaration: s];

  e = [[structure stateSupers] enumerator];
  while ((m = [e nextObject]))
    s = [[m semantics] compileDeclaration: s];

  [[[structure posedSelf] semantics] compileDeclaration: s];

  {
    LTTUnit *u = [structure unit];

    if (single_unit && single_unit != u
	&& !(single_tom_unit && u == ltt_builtin_unit))
      return s;
  }

  formac (s, @"struct %@\n{\n", refname);
  [[[structure posedSelf] semantics]
   compileExtensionsDeclaration: s mark: ++search_mark];
  formac (s, @"};\n\n");

  formac (s, @"extern struct %@ *%@;\n", [structure outputTypeName],
	  [structure referenceName]);

  if ([structure classp])
    {
      formac (s, @"extern struct %@ *%@;\n",
	      [ltt_class_state outputTypeName],
	      [structure metaReferenceName]);
      formac (s, @"extern struct %@ %@;\n", [structure outputTypeName],
	      [structure definitionName]);
      formac (s, @"extern struct %@ %@;\n", [ltt_class_state outputTypeName],
	      [structure metaDefinitionName]);
    }

  return s;
}

-(id) compileDefinition: (id) s
{
  LTTUnit *u = [structure unit];
  id <TLEnumerator> e;
  LTTMeta *m;

  if (defined)
    return s;
  defined = YES;

  if ((!(app_units && [app_units memq: u])
       && !(single_unit && single_unit != u))
      || single_tom_unit && u == ltt_builtin_unit)
    {
      e = [[structure stateSupers] enumerator];
      while ((m = [e nextObject]))
	s = [[m semantics] compileDefinition: s];

      s = [[[structure itsClass] semantics] compileDefinition: s];

      /* Have our poser define itself, so we may pinch from its definition.  */
      [[[structure posedSelf] semantics] compileDefinition: s];

      /* There is no need to output a static instance.  */
      if ([structure classp])
	{
	  formac (s, @"struct %@ *%@;\n", [structure outputTypeName],
		  [structure referenceName]);

	  s = formac (s, @"struct %@ %@ =\n{\n", [structure outputTypeName],
		      [structure definitionName]);

	  if ([[[(LTTClass *) structure instance] semantics] hasPointers])
	    default_asi_value = @"TGC_ASI_EXPLICIT_MARK_P";

	  [self compileDefinition: s context: structure
		poser: [structure posedSelf] mark: ++search_mark];

	  default_asi_value = @"0";

	  formac (s, @"\n};\nstruct %@ *%@ = &%@;\n",
		  [structure outputTypeName], [structure referenceName],
		  [structure definitionName]);
	}

      if (ref_supers && !single_unit)
	{
	  LTIMeta *m;

	  e = [ref_supers enumerator];
	  while ((m = [e nextObject]))
	    {
	      formac (s,
		      @"struct _es_c_tom_State *%@ = &%@.c_tom_State;\n",
		      [structure outputSuperReference: [m structure]],
		      [[[m structure] posedSelf] metaDefinitionName]);
	      super_refs = CONS (CONS (self, m), super_refs);
	    }
	}
    }

  return s;
}

-(id) compileDefinition: (id) s
		context: (LTTMeta *) context
		  poser: (LTTMeta *) p
		   mark: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *m;

  if (mark == k)
    return s;
  mark = k;

  [[[structure posedSelf] semantics] compileDefinition: s context: context
				     poser: p mark: mark];

  e = [[structure stateSupers] enumerator];
  while ((m = [e nextObject]))
    s = [[m semantics] compileDefinition: s context: context
		       poser: p mark: mark];

  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    s = [[x semantics] compileDefinition: s context: context
		       poser: p mark: mark];

  return s;
}

-(id) compileExtensionsDeclaration: (id) s mark: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *m;

  if (mark == k)
    return s;
  mark = k;

  e = [[structure stateSupers] enumerator];
  while ((m = [e nextObject]))
    s = [[m semantics] compileExtensionsDeclaration: s mark: mark];

  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    s = [[x semantics] compileDeclaration: s];

  return s;
}

-(id) compileBuckets: (id) s
{
  int num_states, num_behaviours, num_selectors;
  TLVector *conflicts, *buckets;
  TLVector *states, *behaviours;
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *str;
  LTIMeta *m;
  int i, j;

  if (META_OUTPUT_BUCKETS_P)
    return s;

  META_OUTPUT_BUCKETS_P = 1;

  num_selectors = [CO_LTISelector numSelectors];
  buckets = [CO_TLVector vector];
  states = [CO_TLVector vector];
  behaviours = [CO_TLVector vector];

  push_report_context (formac (nil, @"in %@ %@",
			       [structure classp] ? @"class" : @"instance",
			       [[structure lttName] internal]));

  if (!mdt_buckets)
    {
      mdt_buckets = [TLDictionary dictionary];
      [mdt_buckets gcLock];

      meta_mdts = [TLDictionary dictionary];
      [meta_mdts gcLock];
    }

  e = [[structure stateSupers] enumerator];
  while ((str = [e nextObject]))
    {
      m = [str semantics];
      [states addElement: m];
      s = [m compileBuckets: s];
    }
  num_states = [states length];

  e = [[structure behaviourSupers] enumerator];
  while ((str = [e nextObject]))
    {
      m = [str semantics];
      [behaviours addElement: m];
      s = [m compileBuckets: s];
    }
  num_behaviours = [behaviours length];

  deferred_selectors = [TLSet set];

  /* Inherit the methods from our state supers, remembering only the first
     of the encountered conflicts.  */
  imps = [CO_TLVector vectorWith: num_selectors copies: nil];
  conflicts = [CO_TLVector vectorWith: num_selectors copies: nil];
  e = [([structure classp] ? ordered_classes : ordered_instances) enumerator];
  while ([states length] && (m = [e nextObject]))
    if ([states memq: m])
      {
	TLVector *s_imps;
	LTTMeta *p, *str = [m structure];

	[states removeElementIdenticalTo: m];

	[deferred_selectors unionWithSequence: [m deferredSelectors]];

	for (;;)
	  {
	    p = [str poser];
	    if (!p || p == structure)
	      break;
	    str = p;
	    m = [str semantics];
	  }

	s_imps = [m implementations];

	for (i = 0; i < num_selectors; i++)
	  {
	    LTIMethod *imp = [s_imps _elementAtIndex: i];

	    if (imp)
	      {
		LTIMethod *p = [imps _elementAtIndex: i];
		LTIMethod *c = [conflicts _elementAtIndex: i];

		if (p)
		  if (imp == p)
		    continue;
		  else if ([[[p meta] structure]
			    isProperSub: [[imp meta] structure]])
		    p = nil;
		  else if ([[[imp meta] structure]
			    isProperSub: [[p meta] structure]])
		    continue;

		if (!p)
		  [imps _replaceElementAtIndex: i by: imp];
		else if (imp != p)
		  if (!c || [[[imp meta] structure]
			     isProperSub: [[c meta] structure]])
		    [conflicts _replaceElementAtIndex: i by: imp];
	      }
	  }
      }

  if ([states length])
    ABORT ();

  /* Inherit the behavioural methods.  */
  for (j = 0; j < 2; j++)
    {
      e = [(j ? ordered_instances : ordered_classes) enumerator];

      while ([behaviours length] && (m = [e nextObject]))
	if ([behaviours memq: m])
	  {
	    TLVector *s_imps = [m implementations];

	    [behaviours removeElementIdenticalTo: m];

	    [deferred_selectors unionWithSequence: [m deferredSelectors]];

	    for (i = 0; i < num_selectors; i++)
	      {
		id imp = [s_imps _elementAtIndex: i];

		if (imp && ![imps _elementAtIndex: i])
		  [imps _replaceElementAtIndex: i by: imp];
	      }
	  }
    }

  /* Override with our own methods.  */
  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    {
      TLVector *vec = [[x semantics] methods];
      LTIMethod *imp;
      int i, n;

      for (i = 0, n = [vec length]; i < n; i++)
	{
	  LTISelector *sel;
	  int sid;

	  imp = [vec _elementAtIndex: i];
	  sel = [imp selector];
	  sid = [sel identity];

	  if ([imp methodName])
	    {
	      [imps _replaceElementAtIndex: sid by: imp];
	      [conflicts _replaceElementAtIndex: sid by: nil];
	      s = [imp compileDeclaration: s];

	      [deferred_selectors removeElement: [imp selector]];
	    }
	}
    }

  /* Note our own deferred methods.  */
  if (deferred_methods)
    {
      LTIMethod *imp;

      e = [deferred_methods enumerator];
      while ((imp = [e nextObject]))
	{
	  LTISelector *sel = [imp selector];

	  if (![imps _elementAtIndex: [sel identity]])
	    [deferred_selectors addElement: sel];
	}
    }

  /* Unnote deferred selectors implemented through inheritance (it is a
     result of the algorithm that this is possible/necessary).  */
  for (i = 0; i < num_selectors; i++)
    {
      LTIMethod *imp = [imps _elementAtIndex: i];

      if (imp)
	[deferred_selectors removeElement: [imp selector]];
    }

  if (![deferred_selectors length])
    {
      deferred_selectors = nil;
      if (flag_instances_non_deferred && ![self classp])
	instances_non_deferred = CONS (self, instances_non_deferred);
    }
  else
    {
      if (flag_instances_deferred && ![self classp])
	instances_deferred = CONS (self, instances_deferred);

      if (flag_verbose_deferred && (![self classp] || flag_verbose))
	{
	  LTISelector *sel;

	  formac (V_stderr_, @"deferred selectors of %@:\n",
		  ltt_meta_name (structure));

	  e = [deferred_selectors enumerator];
	  while ((sel = [e nextObject]))
	    formac (V_stderr_, @" %@\n", [[[sel structure] lttName] internal]);
	}
    }

  /* Check for conflicts.  */
  for (i = 0; i < num_selectors; i++)
    {
      LTIMethod *imp = [conflicts _elementAtIndex: i];

      if (imp && i != [load_selector identity])
	error (@"conflict between methods %@, %@", [imp methodName],
	       [(LTIMethod *) [imps _elementAtIndex: i] methodName]);
    }

  /* Generate the dispatch table buckets.  */
  for (i = 0; i < num_selectors; i += TRT_BUCKET_SIZE)
    {
      TLVector *v = [CO_TLVector vectorWithCapacity: TRT_BUCKET_SIZE];
      TLString *bucket_name;

      for (j = 0; j < TRT_BUCKET_SIZE; j++)
	{
	  LTIMethod *imp = [imps _elementAtIndex: i + j];

	  if (!imp)
	    {
	      LTISelector *sel = [CO_LTISelector selectorIdentified: i + j];
	      LTISelector *dyn = [dynamic_selectors objectForKey: sel];

	      if (dyn)
		imp = [imps _elementAtIndex: [dyn identity]];
	    }

	  [v addElement: (imp ? [imp methodName] : nil)];
	}

      bucket_name = [mdt_buckets objectForKey: v];

      if (!bucket_name)
	{
	  bucket_name = formac (nil, @"%@bucket_%d",
				TO_MDT_PREFIX, bucket_count++);
	  [mdt_buckets setObject: bucket_name forKey: v];

	  s = formac (s, @"struct trt_method_dispatch_bucket %@ = {{",
		      bucket_name);

	  for (j = 0; j < TRT_BUCKET_SIZE; j++)
	    {
	      id <TLString> imp = [v _elementAtIndex: j];

	      if (j)
		formac (s, @",");
	      else if (!imp)
		formac (s, @"\n");

	      if (imp)
		formac (s, @"\n%@", imp);
	      else
		formac (s, @"_");
	    }

	  formac (s, @"}, 1};\n\n");
	}

      [buckets addElement: bucket_name];
    }

  [meta_mdts setObject: buckets forKey: self];

  pop_report_context ();

  if ([structure poser])
    s = [[[structure poser] semantics] compileBuckets: s];

  return s;
}

-(id) compileTables: (id) s
{
  int num_selectors;
  TLVector *behaviours;
  TLVector *states;
  int num_states, num_behaviours;
  LTTMeta *str;
  TLVector *vec;
  LTIMeta *m;
  int i, n;

  if (META_OUTPUT_TABLE_P)
    return s;

  META_OUTPUT_TABLE_P = 1;

  num_selectors = [CO_LTISelector numSelectors];

  states = [TLVector vector];
  vec = [structure stateSupers];
  for (i = 0, n = [vec length]; i < n; i++)
    {
      str = [vec _elementAtIndex: i];
      m = [str semantics];
      [states addElement: m];
      s = [m compileTables: s];
    }
  num_states = [states length];

  behaviours = [TLVector vector];
  vec = [structure behaviourSupers];
  for (i = 0, n = [vec length]; i < n; i++)
    {
      str = [vec _elementAtIndex: i];
      m = [str semantics];
      [behaviours addElement: m];
      s = [m compileTables: s];
    }
  num_behaviours = [behaviours length];

  if (!app_unit)
    {
      /* Output the method dispatch table.  */
      TLVector *v = [meta_mdts objectForKey: self];

      s = formac (s, @"struct trt_method_dispatch_table %@%@ = { {%d, 0}, \n{",
		  TO_MDT_PREFIX, [structure outputTypeName],
		  (num_selectors + TRT_BUCKET_SIZE - 1) / TRT_BUCKET_SIZE);
      for (i = 0; i < (num_selectors + TRT_BUCKET_SIZE -1) / TRT_BUCKET_SIZE;
	   i++)
	formac (s, @"%@&%@", i ? @",\n" : @"", [v _elementAtIndex: i]);
      formac (s, @"\n}\n};\n");

      /* Output the extension offset table.  */
      {
	LTIExtension *ext;

	formac (s, @"struct trt_extension_offset_table %@%@ = { {%d, 0}, \n{ 0",
		TO_EOT_PREFIX, [structure outputTypeName],
		1 + [CO_LTIExtension numExtensions]);

	vec = [CO_LTIExtension extensions];
	for (i = 0, n = [vec length]; i < n; i++)
	  {
	    ext = [vec _elementAtIndex: i];
	    str = [[ext structure] meta];
	    if ((str == structure || [str isProperStateSub: structure])
		&& [[ext structure] hasVariables]) 
	      formac (s, @",\n(char *) &((struct %@ *) 0)->%@ - (char *) 0",
		      [structure outputTypeName],
		      [[ext structure] outputExtensionFieldName]);
	    else
	      [@", 0" print: s quoted: NO];
	  }
	formac (s, @"\n}};\n");
      }

      /* Output the reference variables offset table.  */
      {
	TLCons *l = [self addReferenceVariables: nil mark: ++search_mark];
	TLVector *ref_vars = [TLVector vectorWithSequence: l];
	int num_ref_vars = [ref_vars length];
	int i;

	formac (s, @"struct trt_refvar_offset_table %@%@ = {{%d, 0}, \n{",
		TO_META_RVO_PREFIX, [structure outputTypeName], num_ref_vars);

	for (i = 0; i < num_ref_vars; i++)
	  {
	    LTTVariable *v = [ref_vars _elementAtIndex: i];

	    formac (s, @"%@((char *) &(((struct %@ *) 0)->%@.%@) - (char *) 0)",
		    i ? @",\n" : @"", [structure outputTypeName],
		    [[v extension] outputExtensionFieldName],
		    [[v lttName] external]);
	  }

	formac (s, @"}};\n");
      }
    }

  /* Output the extensions information descriptions.  */
  {
    TLCons *l = [self addExtensionDescriptions: nil mark: ++search_mark];
    int num_ext = 0, num_own_ext = 0;
    id <TLEnumerator> e;
    LTIExtension *ext;

    if (l)
      {
	e = [l enumerator];

	while ((ext = [e nextObject]))
	  {
	    LTTUnit *u = [[[ext structure] container] unit];

	    if ([[ext structure] meta] == structure)
	      num_own_ext++;
	    num_ext++;
	    if (!app_units || ![app_units memq: u])
	      s = [ext compileExtensionDescription: s];
	  }
      }

    if (!app_unit)
      {
	int num_supers = ([structure numStateSupers]
			  + [structure numBehaviourSupers]);
	int num_subs= [structure numStateSubs] + [structure numBehaviourSubs];
	id <TLEnumerator> e;
	LTTMeta *m;

	if (num_supers)
	  {
	    formac (s, @"struct trt_metas _supers%@ = {{%d, 0}, {",
		    [structure metaDefinitionName], num_supers);
	    e = [[structure stateSupers] enumerator];
	    while ((m = [e nextObject]))
	      formac (s, @"\n(void *) &%@,", [m metaDefinitionName]);
	    e = [[structure behaviourSupers] enumerator];
	    while ((m = [e nextObject]))
	      formac (s, @"\n(void *) &%@,", [m metaDefinitionName]);
	    formac (s, @"\n}};\n");
	  }

	if (num_subs)
	  {
	    formac (s, @"struct trt_metas _subs%@ = {{%d, 0}, {",
		    [structure metaDefinitionName], num_subs);
	    e = [[structure stateSubs] enumerator];
	    while ((m = [e nextObject]))
	      formac (s, @"\n(void *) &%@,", [m metaDefinitionName]);
	    e = [[structure behaviourSubs] enumerator];
	    while ((m = [e nextObject]))
	      formac (s, @"\n(void *) &%@,", [m metaDefinitionName]);
	    formac (s, @"\n}};\n");
	  }

	formac (s, @"struct trt_meta_extensions %@ = {{%d, 0},\n{",
		[structure extensionsDescriptionName], num_own_ext);
	if (l)
	  {
	    e = [l enumerator];

	    num_ext = 0;
	    while ((ext = [e nextObject]))
	      if ([[ext structure] meta] == structure)
		{
		  formac (s, num_ext ? @",\n&%@" : @"\n&%@",
			  [[ext structure] outputExtensionDescriptionName]);
		  num_ext++;
		}
	    if (num_ext != num_own_ext)
	      ABORT ();
	  }

	formac (s, @"\n}};\n");
      }
  }

  return s;
}

-(BOOL) hasPointers
{
  return [self hasPointers: ++search_mark];
}

-(BOOL) hasPointers: (int) k
{
  id <TLEnumerator> e;
  LTTExtension *x;
  LTTMeta *m;

  if (mark == k)
    return NO;
  mark = k;

  e = [[structure stateSupers] enumerator];
  while ((m = [e nextObject]))
    if ([[m semantics] hasPointers: k])
      return YES;

  e = [[structure extensions] enumerator];
  while ((x = [e nextObject]))
    if ([[x semantics] hasPointers])
      return YES;

  return NO;
}

@end

@implementation LTIClass (tr)

-(id) compileDefinition: (id) s
{
  if (!defined)
    {
      LTTUnit *u = [structure unit];
      int emit = ((!(app_units && [app_units memq: u])
		   && !(single_unit && single_unit != u))
		  || single_tom_unit && u == ltt_builtin_unit);

      if (emit)
	{
	  formac (s, @"struct %@ *%@;\n", [ltt_class_state outputTypeName],
		  [structure metaReferenceName]);

	  s = formac (s, @"struct %@ %@;\n", [ltt_class_state outputTypeName],
		      [structure metaDefinitionName]);
	}

      [super compileDefinition: s];

      if (emit)
	{
	  LTTMeta *p = [structure posedSelf];
	  int num_supers = ([structure numStateSupers]
			    + [structure numBehaviourSupers]);
	  int num_subs= ([structure numStateSubs]
			 + [structure numBehaviourSubs]);

	  formac (s, @"struct %@ %@ = {{", [ltt_class_state outputTypeName],
		  [structure metaDefinitionName]);

	  /* isa */
	  formac (s, @"\n&%@", [ltt_class_state metaDefinitionName]);

	  /* asi */
	  formac (s, @",\nTGC_ASI_META_CLASS_P");

	  /* mdt */
	  if (app_unit)
	    formac (s, @",\n{0");
	  else
	    formac (s, @",\n{&%@%@", TO_MDT_PREFIX, [p outputTypeName]);

	  /* eot */
	  if (app_unit)
	    formac (s, @",\n0");
	  else
	    formac (s, @",\n&%@%@", TO_EOT_PREFIX, [p outputTypeName]);

	  /* num_instances */
	  formac (s, @",\n1");

	  /* name */
	  {
	    id <TLString> name = [[structure lttName] internal];
	    formac (s, @",\n{%#, %d}", name, [name length]);
	  }

	  /* rvo */
	  if (app_unit)
	    formac (s, @",\n0");
	  else
	    formac (s, @",\n&%@%@", TO_META_RVO_PREFIX, [p outputTypeName]);

	  /* instance_size */
	  if (app_unit)
	    formac (s, @",\n0");
	  else
	    formac (s, @",\nsizeof (struct %@)", [p outputTypeName]);

	  formac (s, @"\n, 0,\n(struct trt_class **) &%@,\n%@",
		  [structure metaReferenceName],
		  app_unit ? (id) @"0"
		  : formac (nil, @"&%@", [p extensionsDescriptionName]));

	  /* subs */
	  if (num_subs && !app_unit)
	    formac (s, @",\n&_subs%@", [structure metaDefinitionName]);
	  else
	    formac (s, @",\n0");

	  /* supers */
	  if (num_supers && !app_unit)
	    formac (s, @",\n&_supers%@", [structure metaDefinitionName]);
	  else
	    formac (s, @",\n0");

	  /* poser */
	  formac (s, @",\n0");

	  /* state_extensions */
	  formac (s, @",\n0");

	  /* mark */
	  formac (s, @",\n0");

	  /* mark2 */
	  formac (s, @",\n0");

	  formac (s, @"\n}}};\nstruct %@ *%@ = &%@;\n",
		  [ltt_class_state outputTypeName],
		  [structure metaReferenceName],
		  [structure metaDefinitionName]);
	}
    }
  return s;
}

@end
