/* OTM parser -*- objc -*-
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: parse.y,v 1.179 1999/08/18 20:42:54 tiggr Exp $  */

%{
#import <tl/tl.h>
#import <time.h>
#import "global.h"
#import "OTMAlias.h"
#import "OTMAliasAlias.h"
#import "OTMArgument.h"
#import "OTMAssignment.h"
#import "OTMBasic.h"
#import "OTMBind.h"
#import "OTMBreak.h"
#import "OTMBuiltinMethod.h"
#import "OTMCast.h"
#import "OTMCatch.h"
#import "OTMCondExpr.h"
#import "OTMConstant.h"
#import "OTMContinue.h"
#import "OTMClass.h"
#import "OTMCompound.h"
#import "OTMCustomMethod.h"
#import "OTMDynamicType.h"
#import "OTMError.h"
#import "OTMExpr.h"
#import "OTMExtension.h"
#import "OTMIdentifier.h"
#import "OTMInstance.h"
#import "OTMInvocation.h"
#import "OTMITE.h"
#import "OTMLocalVar.h"
#import "OTMLoop.h"
#import "OTMMetaRef.h"
#import "OTMModAssign.h"
#import "OTMNumberCST.h"
#import "OTMObjectVar.h"
#import "OTMOld.h"
#import "OTMReturn.h"
#import "OTMStringCST.h"
#import "OTMTuple.h"
#import "OTMTypeTuple.h"
#import "OTMUnvocation.h"
#import "OTMUnwind.h"
#import "OTMVariable.h"
#import "OTMVarRef.h"

/* Forward declarations.  */
extern int otm_lex ();
void do_postponed_aliases (BOOL must);

#define YYDEBUG 1
#define YYERROR_VERBOSE

int current_line;
id <TLString> current_filename;

/* The current unit.  */
LTTUnit *current_unit;

/* The current output file.  It is NIL when parsing a file which is not to
   be output.  */
id <TLOutputStream> of;

/* The extensions output to the output file.  */
TLSet *extensions_output;

/* The current class or the current instance.
   Only one of these will ever be !NIL.  */
LTTClass *current_class;
LTTInstance *current_instance;

/* This is either the CURRENT_CLASS or the CURRENT_INSTANCE, whichever is
   !nil (they can't be both).  */
LTTMeta *current_either;

/* The current extension.  The main definition of a class or instance is
   also an extension.  It is called the main extension.  */
LTTExtension *current_extension;

/* In a definition, or during the pre and post conditions of a
   declaration, the current method.  */
OTMCustomMethod *current_method;

/* The meta involved in the meta or method or whatever currently being
   output to the output file, be it as a declaration, or as a definition.  */
LTTMeta *output_current_context;

/* Similarly, the method.  */
OTMCustomMethod *output_current_method;

/* The current compound.  */
OTMCompound *current_compound;

/* The type enclosing the current local variable declaration.  */
OTMType *enclosing_type;

/* Semantics to some of the above.  */
#define C_EXT_SEMANTICS		(OTMExtension *) [current_extension semantics]
#define C_CLS_SEMANTICS		(OTMClass *) [current_class semantics]
#define C_INS_SEMANTICS		(OTMInstance *) [current_instance semantics]
#define C_EITHER_SEMANTICS	(OTMMeta *) [current_either semantics]

static enum
{
  GLOBAL_SCOPE,
  INTERFACE_SCOPE,
  IMPLEMENTATION_SCOPE,
} in_what;

/* Lists of still-to-patch supers.  Each entry is a cons cell holding the
   LTTMeta and the super meta.  Semantics of the inheritance depends on
   the list.  The STATE_SUPERS holds instances, but inheritance applies to
   both the instance and its class.  The BEHAVIOUR_SUPERS only applies to
   the actual metas involved.  */
TLCons *postponed_state_supers, *postponed_behaviour_supers;

TLCons *postponed_aliases;

/* The index number of the next argument.  */
int next_arg;

void
yyerror (char *s)
{
  extern char *yytext;
  error (@"parse error at `%s' (%s)", yytext, s);
}

LTTInstance *
instance_in_unit (id <TLString> unit_name, LTTUnit *current,
		  TLVector *instances)
{
  LTTInstance *i = ltt_instance_in_unit (unit_name, current, instances);

  if (!i && instances)
    error (AT"ambiguous reference to %@",
	   [[(LTTInstance *) [instances _elementAtIndex: 0] lttName]
	    internal]);

  if (of && (postponed_state_supers || postponed_behaviour_supers))
    do_postponed_supers (1);

  return i;
}

/* Return a a pair of a variable with the name N and the type T, and the
   DEF expression, or, if T is a type-tuple and N is a tuple of names, and
   they correspond, return a pair of a properly typed tuple of properly
   typed variables, and the default value.

   Thus, this return CONS (arg, value).  */
id
digest_arguments (id n, id t, id def)
{
  BOOL ntup = ![n stringp];
  BOOL ttup = [t isTuple];
  OTMExpr *ret;

  /* A tuple with one element is not a real tuple.  */
  if (ttup)
    {
      TLVector *tv = [t elements];
      if ([tv length] == 1)
	{
	  t = [tv _elementAtIndex: 0];

	  /* XXX What is a tuple consisting of only a tuple?  What if the
             contained tuple also contains only a single element?  */
	  ttup = [t isTuple];
	}
    }

  /* If T is a tuple, it is a Real Tuple (> 1 element).  */
  if (ttup)
    {
      if (!ntup)
	{
	  error (@"attempt to declare `%@' with tuple type", n);
	  return nil;
	}
      else
	{
	  TLVector *tv = [(id) t elements], *nv = [n elements];
	  int i, nn = [nv length], tn = [tv length];
	  TLVector *vars;

	  if (tn != nn)
	    {
	      error (@"number of elements (%d) does not match type (%d)",
		     nn, tn);
	      return nil;
	    }

	  /* Create a vector of typed variables, by zipping the name tuple
             and type tuple.  */
	  vars = [CO_TLVector vectorWithCapacity: tn];
	  for (i = 0; i < nn; i++)
	    [vars addElement: [CO_OTMArgument variableWithName:
					      [nv _elementAtIndex: i]
					      type: [tv _elementAtIndex: i]
					      number: next_arg++]];

	  ret = [CO_OTMTuple tupleWithSequence: vars];
	  [ret setType: t];
	}
    }
  else if (ntup > 1)
    {
      error (@"non-tuple type for tupled argument");
      return nil;
    }
  else
    {
      if (ntup == 1)
	n = [[n elements] _elementAtIndex: 0];
      ret = [CO_OTMArgument variableWithName: n type: t number: next_arg++];
    }

  return CONS (ret, def);
}

/* Load the FILE.  Iff !INTERFACE, the implementation will be loaded.
   Otherwise, the interface will be preferred.  Iff META_NAME, the meta
   with that name must be declared after having read the file.  Iff MUST,
   the file type indicated by the INTERFACE must be found; otherwise an
   error is issued.  */
void
load_file (LTTFile *file, id <TLString> meta_name, BOOL interface, BOOL must)
{
  TLString *full_name = nil;
  FILE *f;
  GCDECL1;

  GCPRO1 (full_name);

  /* Open the file, preferring `.j' above `.t', unless !INTERFACE.  */
  if (!interface)
    f = NULL;
  else
    {
      if ([file loadedInterface] || (!must && [file loadedImplementation]))
	return;

      full_name = [file interfaceFilename];
      f = fopen ([full_name cString], "r");
      if (f)
	[file setLoadedInterface];
      else if (must)
	{
	  error (@"%@: %s", full_name, ERRMSG);
	  return;
	}
      else if (ERRNO != ENOENT)
	warning (@"%@: %s", full_name, ERRMSG);
    }

  if (!f)
    {
      if ([file loadedImplementation] || (!must && [file loadedInterface]))
	return;

      full_name = [file implementationFilename];
      f = fopen ([full_name cString], "r");
      if (f)
	[file setLoadedImplementation];
      else
	error (@"%@: %s", full_name, ERRMSG);
    }

  if (f)
    {
      char buf[4096];
      const char *s;
      TLVector *v;
      int mi, mn;

      s = [full_name cString];
      if (s[0] != '/')
	{
	  /* Make the path absolute.  */
	  if (getcwd (buf, sizeof (buf)))
	    full_name = [CO_TLString stringWithFormat: @"%s/%s", buf, s];
	}

      parse_file (full_name, [file unit], f);
      fclose (f);

      for (mi = 0, v = [file items], mn = [v length]; mi < mn; mi++)
	{
	  LTTExtension *ext = [v _elementAtIndex: mi];
	  OTMExtension *sem = [ext semantics];

	  if (!sem || !([sem definedp] || [sem declaredp]))
	    error (@"%@ fails to define or declare %@",
		   full_name, ltt_ext_name (ext));
	}
    }

  GCUNPRO;
}

/* Load the interface (or the implementation) of the extension EXT.  */
void
load_extension_interface (LTTExtension *ext)
{
  LTTFile *file = [ext container];
  id pof = nil;
  GCDECL1;

  GCPRO1 (pof);

  pof = of;
  of = nil;

  /* Only check the meta in case we're loading the main extension.  */
  load_file (file, [ext isMainExtension] ? [[ext meta] lttName] : nil, 1, 0);

  of = pof;

  GCUNPRO;
}

/* Load the meta M.  This implies that all files which have defined
   extensions or posing classes for M are loaded.  */
void
load_interface (LTTInstance *ins)
{
  LTTExtension *ext;

  /* If the main extension's file has been set loaded, we do not do
     anything since that file will declare this meta.  */
  if (![[ins semantics] fullyLoaded])
    {
      int previous_in_what = in_what;
      TLVector *exts = 0;
      int i, n;
      GCDECL1;

      GCPRO1 (exts);

      [[ins semantics] setFullyLoaded];

      exts = [ins extensions];
      for (i = 0, n = [exts length]; i < n; i++)
	{
	  ext = [exts _elementAtIndex: i];
	  if (![[ext container] loadedInterface]
	      && ![[ext container] loadedImplementation])
	    load_extension_interface (ext);
	}

      GCUNPRO;

      in_what = previous_in_what;
    }

  {
    LTTMeta *p = [ins posedSelf];

    if ((id) ins != p)
      load_interface ([p classp] ? (id) [p instance] : p);
  }
}

/* Add the method declaration of METHOD to the current extension.  */
void
add_method_decl (OTMCustomMethod *method)
{
  BOOL ok = YES, shown = NO;
  OTMMethod *orig;

  if (!current_extension)
    return;

  if (in_what == IMPLEMENTATION_SCOPE && [C_EXT_SEMANTICS declaredp]
      && ![method foreign] && ![method deferredp])
    error (@"method declaration in implementation scope");

  orig = [C_EITHER_SEMANTICS searchSimilarMethod: method];
  if (orig)
    {
      otm_qualifiers q = [method qualifiers];
      otm_qualifiers oq = [orig qualifiers];

      if (![method foreign] && !Q_DEFERRED (oq))
	{
	  if ([method extension] == [orig extension])
	    {
	      error (@"duplicate declaration of identical method");
	      if (!shown)
		cerror_for (orig, @"previous declaration was here");
	      ok = NO;
	    }

	  // XXX This is a hack.  The `searchSimilarMethod:' should return
	  // a list of applicable methods (i.e. all of those in scope,
	  // which can be multiple methods for multiple extensions of this
	  // either)
	  if (([[[orig extension] structure] meta]
	       == [[[method extension] structure] meta])
	      && [[[method extension] structure] isMainExtension])
	    ;
	  else if (!Q_REDO (q))
	    {
	      if (!flag_inhibit_unqualified_redeclare)
		{
		  warning (@"unqualified redeclaration of undeferred method");
		  if (!shown)
		    cerror_for (orig, @"previous declaration was here");
		}
	      [method setRedo: OQ_REDEFINE];
	    }
	}
      if (!Q_PROTECTION (q))
	Q_SET_PROTECTION (q, Q_PROTECTION (oq));
      else if (Q_PROTECTION (q) != Q_PROTECTION (oq))
	{
	  error (@"protection %@ differs from original declaration",
		 qualifier_name (Q_PROTECTION (q)));
	  error_for (orig, @"original protection %@ declared here",
		     qualifier_name (Q_PROTECTION (oq)));
	  [method setProtection: Q_PROTECTION (oq)];
	}

      /* XXX `Inherit' the default arguments from the original?  */
    }

  if (ok)
    {
      /* Make sure the selector of this method is registered.
	 This is needed for foreign methods.  */
      if (of)
	[method selector];

      /* No need to add the declaration of a foreign method twice.  */
      if (in_what == IMPLEMENTATION_SCOPE && [method foreign]
	  && [method extension] == [orig extension])
	return;
      else
	{
	  [C_EXT_SEMANTICS addMethod: method];
	  [method setDeclared: YES];
	}
    }
}

/* Add the method declaration or definition of METHOD with the PRE and
   POST conditions to the current extension.  */
OTMCustomMethod *
add_method_def (OTMCustomMethod *method)
{
  OTMMethod *orig;
  OTMExtension *ext;
  BOOL ok = YES, shown = NO;

  if (!current_extension)
    return nil;

  ext = [method extension];
  orig = [C_EXT_SEMANTICS searchSimilarMethod: method];

  if (orig)
    {
      if (ext == [orig extension])
	{
	  if ([orig definedp])
	    {
	      error (@"redefinition of prototypicially identical method");
	      if (!shown)
		{
		  cerror_for (orig, @"previous definition was here");
		  shown = YES;
		}
	      ok = NO;
	    }
	}
    }
  else
    {
      /* In an implementation, if the interface has been seen, this method
         must have been declared.  */
      if (in_what == IMPLEMENTATION_SCOPE && [ext declaredp])
	error (@"definition of undeclared method %@", method_name (method, 0));
    }

  if (ok && orig
      && ![orig allowedRedeclaration: method
		inSubclass: ([[ext structure] meta]
			     != [[[orig extension] structure] meta])])
    {
      error (@"argument type mismatch with declaration");
      if (!shown)
	{
	  cerror_for (orig, @"previous definition was here");
	  shown = YES;
	}
      ok = NO;
    }

  if (ok)
    {
      if (orig)
	{
	  id f;

	  [method warnDifferingArguments: orig];
	  method = (id) orig;

	  [method refreshLocation];

	  f = [method foreign];
	  if (f)
	    [(OTMCustomMethod *) orig setForeign: f];
	}
      else
	{
	  TLVector *v;

	  v = [C_EXT_SEMANTICS methodsNamed: [method methodName] create: YES];
	  [v addElement: method];
	}

      [method setDefined: YES];

      /* Make sure the selector of this method is registered.
	 This is needed for foreign methods.  */
      if (of && !NUM_ERRORS)
	[method selector];
    }

  return method;
}

/* Add CI to the behavioural super information of SUBI.  */
void
add_super_1 (LTTMeta *subi, LTTMeta *ci)
{
  OTMMeta *cis = [ci semantics];

  if (![cis declaredp] && ![cis definedp])
    postponed_behaviour_supers = CONS (CONS (subi, ci),
				       postponed_behaviour_supers);
  else if ([ci isDirectSub: subi]
	   && (in_what == INTERFACE_SCOPE
	       || (in_what == IMPLEMENTATION_SCOPE
		   && ![[subi semantics] declaredp])))
    warning (@"%@: repeated immediate super %@",
	     ltt_meta_name (subi), ltt_meta_name (ci));
  else
    {
      [subi addBehaviourSuper: ci];
      [[[subi extensionNamed: [[current_extension lttName] internal]] semantics]
	addSuper: cis];
    }
}

/* Handle the normal case of straight inheritance (without shifting meta
   levels).  */
static void
add_super_0 (LTTInstance *subi, LTTInstance *ci)
{
  BOOL already = [ci isDirectSub: subi];
  LTTClass *subc = [subi itsClass];
  LTTClass *cc = [ci itsClass];
  OTMMeta *subis = [subi semantics];
  OTMMeta *cis = [ci semantics];

  if (![cis declaredp] && ![cis definedp])
    postponed_state_supers = CONS (CONS (subi, ci), postponed_state_supers);
  else if (already
	   && (in_what == INTERFACE_SCOPE
	       || (in_what == IMPLEMENTATION_SCOPE && ![subis declaredp])))
    warning (@"%@: repeated immediate super %@",
	     ltt_meta_name (subi), ltt_meta_name (ci));
  else if (!already && in_what == IMPLEMENTATION_SCOPE && [subis declaredp])
    error_for ([subi semantics],
	       @"%@: super indication was absent in interface",
	       ltt_meta_name (ci));
  else
    {
      LTTName *cename = current_extension ? [current_extension lttName] : nil;
      id <TLString> cenamei = cename ? [cename internal] : nil;

      [subi addStateSuper: ci];
      [subc addStateSuper: cc];
      [[[subi extensionNamed: cenamei] semantics] addSuper: cis];
      [[[subc extensionNamed: cenamei] semantics] addSuper: [cc semantics]];
      if ([cis statep])
	[[subi semantics] setStatep];
      if ([[cc semantics] statep])
	[[subc semantics] setStatep];
    }
}

/* Add C to be a super of the current meta.  If !direction, this is state
   inheritance (even if C has no state).  If it is -1, it is inheritance
   of the instance behaviour only.  If it is +1, it indicates inheritance
   of the class behaviour only.  */
static void
add_super (LTTInstance *inst, int direction)
{
  if (!direction)
    {
      if (current_instance)
	add_super_0 (current_instance, inst);
      else if (current_class)
	add_super_0 ([current_class instance], inst);
    }
  else if (direction < 0)
    add_super_1 (current_either, inst);
  else
    add_super_1 (current_either, [inst itsClass]);
}

static void
do_add_super (id what, int posing)
{
  int dir = 0;

  if ([what consp])
    {
      id a, d;

      DECONS (what, a, d);
      if (a)
	what = a, dir = 1;
      else
	what = d, dir = -1;
    }

  add_super (what, dir);

  if (posing)
    {
      /* XXX Check consistency.  */

    }
}

void
do_postponed_aliases (BOOL must)
{
  TLCons *list = postponed_aliases;

  postponed_aliases = nil;

  while (list)
    {
      OTMAliasAlias *a;

      DECONS (list, a, list);
      [[a extension] handlePostponedAlias: a];
    }

  if (must && postponed_aliases)
    internal (@"non-empty postponed aliases");
}

void
do_postponed_supers (BOOL must)
{
  TLCons *states = postponed_state_supers;
  TLCons *behaviours = postponed_behaviour_supers;
  int previous_in_what = in_what;
  id sub, sup;
  TLCons *c;

  postponed_behaviour_supers = postponed_state_supers = nil;
  in_what = GLOBAL_SCOPE;

  while (states)
    {
      DECONS (states, c, states);
      DECONS (c, sub, sup);
      add_super_0 (sub, sup);
    }

  while (behaviours)
    {
      DECONS (behaviours, c, behaviours);
      DECONS (c, sub, sup);
      add_super_1 (sub, sup);
    }

  if (must && (postponed_state_supers || postponed_behaviour_supers))
    internal (@"non-empty postponed supers");

  in_what = previous_in_what;
}

/* Add the variables from the VARS list to the current_either with the
   indicated qualifiers and TYPE.  */
void
object_add_variables (otm_qualifiers q, id type, TLCons *vars, BOOL is_local)
{
  id <TLString> name;
  OTMObjectVar *var;
  
  q = mask_qualifiers (q, ((current_instance ? 0 : OQ_STATIC_MASK)
			   | OQ_PROTECTION_MASK | OQ_MUTABLE_MASK
			   | OQ_REDO_MASK));

  if (Q_REDO (q) && Q_REDO (q) != OQ_REDECLARE)
    q = mask_qualifiers (q, 0);

  if (Q_REDO (q) && is_local)
    {
      error (@"can't redeclare something as local");
      is_local = 0;
    }

  if (!current_either)
    return;

  if (in_what == IMPLEMENTATION_SCOPE && [[current_either semantics] declaredp])
    while (vars)
      {
	OTMCustomMethod *m;
	OTMExpr *e;

	DECONS (vars, name, vars);

	/* XXX Check consistency with declaration.  */
	var = [C_EXT_SEMANTICS variableNamed: name];

	if (!var)
	  {
	    error (@"declaration of %@ not found in interface", name);
	    continue;
	  }

	if ([var type] != type)
	  {
	    error (@"differing declaration of %@", name);
	    error_for (var, @"previous declaration was here");
	  }

	/* Generate the accessor methods.  */
	if (Q_PROTECTION (q) == OQ_PUBLIC)
	  {
	    m = add_method_def ((id) [CO_OTMCustomMethod
				      methodWithExtension: C_EXT_SEMANTICS
				      name: accessor_method_name (name)
				      returnType: type flatp: NO]);
	    current_method = m;
	    current_compound = [CO_OTMCompound compoundWithContainer: (id) m];
	    [current_method setBody: current_compound];
	    e = [CO_OTMAssignment assignmentWithLhs:
			       temp_something_with_type ([var type], 0)
			       rhs: var];
	    e = emit_expr (e);
	    [m setReturnValue: e];
	    current_compound = nil;
	    if (of && !NUM_ERRORS)
	      [current_method compile];
	    current_method = nil;
	  }

	if (Q_MUTABLE (q))
	  {
	    m = add_method_def
	      ((id) [CO_OTMCustomMethod methodWithExtension: C_EXT_SEMANTICS
		     nameTypes: CONS (CONS (modifier_method_name (name),
					    CONS ([CO_OTMVariable
						   variableWithName:
						   @"value" type: type], nil)),
				      nil)
		     returnType: basic_type[BT_VOID] flatp: NO]);

	    current_method = m;
	    current_compound = [CO_OTMCompound compoundWithContainer: (id) m];
	    [current_method setBody: current_compound];
	    e = [CO_OTMAssignment assignmentWithLhs:
		 [C_EITHER_SEMANTICS searchEntityNamed: name supers: YES
		  class: YES variables: YES constants: YES]
		 rhs: [current_compound searchVariableNamed: @"value"]];
	    e = emit_expr (e);
	    current_compound = nil;
	    if (of && !NUM_ERRORS)
	      [current_method compile];
	    current_method = nil;
	  }
      }
  else
    while (vars)
      {
	OTMMethod *m;

	DECONS (vars, name, vars);

	if (Q_REDO (q))
	  var = [C_EXT_SEMANTICS aliasWithName: name
				 type: type qualifiers: q];
	else
	  {
	    if (current_either != ltt_instance_state
		&& current_either != ltt_class_state)
	      {
		var = [C_EITHER_SEMANTICS searchEntityNamed: name
					  supers: YES class: YES
					  variables: YES constants: YES];
		if (var)
		  {
		    warning (@"declaration of %@ hides previous declaration",
			     name);
		    cwarning_for (var, @"%@ previously declared here", name);
		  }
	      }

	    var = [C_EXT_SEMANTICS variableWithName: name
				   type: type qualifiers: q];
	    if (is_local)
	      if (![var staticp] || ![current_either classp])
		error_for (var,
			   @"non-static or non-class variable can't be local");
	      else
		{
		  [var setIsThreadLocal];
		  [C_EITHER_SEMANTICS setThreadLocalVarsP];
		}
	  }

	if (Q_PROTECTION (q) == OQ_PUBLIC)
	  {
	    m = [CO_OTMCustomMethod methodWithExtension: C_EXT_SEMANTICS
				    name: accessor_method_name (name)
				    returnType: type flatp: NO];
	    if (Q_REDO (q) == OQ_REDECLARE)
	      [m setRedo: OQ_REDECLARE];
	    add_method_decl ((id) m);
	  }
	if (Q_MUTABLE (q))
	  {
	    m = [CO_OTMCustomMethod methodWithExtension: C_EXT_SEMANTICS
		     nameTypes: CONS (CONS (modifier_method_name (name),
				  CONS ([CO_OTMVariable variableWithName:
					      @"value" type: type], nil)), nil)
				    returnType: basic_type[BT_VOID] flatp: NO];
	    if (Q_REDO (q) == OQ_REDECLARE)
	      [m setRedo: OQ_REDECLARE];

	    add_method_decl ((id) m);
	  }
      }
}

/* Add the variables from the VARS list to the current_either with the
   indicated qualifiers and TYPE.  */
void
object_add_const (otm_qualifiers q, id <TLString> name, id value)
{
  OTMConstant *c;
  
  q = mask_qualifiers (q, OQ_PROTECTION_MASK);

  if (!current_either)
    return;

  c = [C_EXT_SEMANTICS constantWithName: name qualifiers: q value: value];
}

/* Check the super class declarations of the current meta.  */
static void
check_supers (void)
{
  if ([current_extension isMainExtension])
    {
      /* Make sure Any and State are supers.  */
      if (current_class)
	{
	  /* XXX Should search in the other direction...  */
	  if (current_class != ltt_class_state
	      && ![ltt_class_state isProperStateSub: current_class])
	    [current_class addStateSuper: ltt_class_state];
	  if (![ltt_class_any isProperSub: current_class])
	    [current_class addBehaviourSuper: ltt_class_any];

	  /* Make both the class and instance of this EITHER a superclass
             of All.  */
	  [ltt_instance_all addBehaviourSuper: current_class];
	  [ltt_instance_all addBehaviourSuper: [current_class instance]];
	}
      else if (current_instance
	       && ![ltt_instance_any isProperSub: current_instance])
	[current_instance addBehaviourSuper: ltt_instance_any];
    }
}

/* Setup everything for the interface or implementation (depends on
   DEFINITIONP) of instance or class (depending on IS_CLASS) of the
   extension EXT of the INSTANCE.  If !EXT this is the main extension.  */
static void
set_current_either (LTTInstance *instance, id <TLString> ext, int is_class,
		    id <TLString> foreign, otm_qualifiers q, int definitionp)
{
  OTMMeta *either, *counterpart;
  id <TLString> errmsg;

  current_class = nil;
  current_instance = nil;
  current_either = nil;
  current_extension = nil;

  if (!instance)
    return;

  /* We're loading a named extension.  Make sure the main extension is
     also loaded.  */
  if (ext && ![[[instance extensionNamed: nil] container] loadedInterface])
    load_interface (instance);

  q = mask_qualifiers (q, 0);

  if (is_class)
    current_either = current_class = [instance itsClass];
  else
    current_either = current_instance = instance;

  errmsg = ltt_meta_name (current_either);
  either = C_EITHER_SEMANTICS;
  counterpart = [(is_class ? (LTTMeta *) [(LTTClass *) current_either instance]
		  : (LTTMeta *) [current_either itsClass]) semantics];

  if (ext)
    {
      if (!([either declaredp] || [either definedp]
	    || [[[current_either extensionNamed: nil]
		 container] loadedInterface]))
	internal (@"interface of `%@' not found", errmsg);

      if (current_either == ltt_instance_any || current_either == ltt_class_any
	  || current_either == ltt_instance_all
	  || current_either == ltt_class_all)
	error (@"extensions of %@ not allowed", ltt_meta_name (current_either));
    }
  else
    {
      if (definitionp)
	{
	  if ([either definedp] && ![either foreign])
	    {
	      error (@"redefinition of %@", errmsg);
	      cerror_for (either, @"previous definition of %@ was here",
			  errmsg);
	    }

	  if (Q_DEFERRED (q) && [either declaredp] && ![either deferredp])
	    {
	      error (@"deferred status mismatch with interface of %@", errmsg);
	      cerror_for (either, @"previous declaration was here");
	    }
	}
      else
	{
	  if ([either declaredp])
	    error (@"redeclaration of %@", errmsg);
	  else
	    [either refreshLocation];
	}

      if (!is_class)
	{
	  /* Foreign is only allowed on a class interface.  Instance
	     interface and any implementation are not allowed.  */
	  if (foreign || [[[current_instance itsClass] semantics] foreign])
	    error (@"foreign instances are not possible");

	  /* The declaration of the class must have been seen before the
	     instance is encountered.  */
	  if (![counterpart declaredp] && ![counterpart definedp])
	    error (@"declaration of %@ should preceed %@ of %@",
		   ltt_meta_name ([counterpart structure]),
		   definitionp ? @"definition" : @"declaration",
		   ltt_meta_name (current_instance));
	}
      else
	{
	  if (foreign && ![either declaredp])
	    if (definitionp)
	      error (@"implementation of foreign class meaningless");
	    else
	      {
		[either setForeign: foreign];
		[counterpart setDeclared: YES];
	      }
	}

      if (definitionp)
	[either setDefined: YES];
      else
	[either setDeclared: YES];

      if (Q_DEFERRED (q))
	{
	  /* XXX This kind of information which is applicable to both the
	     instance and the class, should reside in the class; not in
	     both.  */
	  [either setDeferred: YES];
	  [counterpart setDeferred: YES];
	}
    }

  current_extension = [current_either extensionNamed: ext];
  if (!current_extension)
    {
      current_extension = [CO_LTTExtension extensionWithName: ext
					file: ltt_missing_file
					meta: current_either];

      error (@"inventing %@", ltt_ext_name (current_extension));
    }

  if (definitionp)
    {
      [C_EXT_SEMANTICS setDefined: YES];

      if (of)
	{
	  [extensions_output addElement: C_EXT_SEMANTICS];
	  if ([current_extension isMainExtension])
	    if (current_class)
	      {
		[C_EXT_SEMANTICS addSuper: [ltt_class_any semantics]];
		[C_EXT_SEMANTICS addSuper: [ltt_class_state semantics]];
	      }
	    else if (current_instance)
	      [C_EXT_SEMANTICS addSuper: [ltt_instance_any semantics]];
	}
    }
  else
    {
      [C_EXT_SEMANTICS setDeclared: YES];

      if (foreign)
	{
	  /* This is a class.  Set the instance declared.  */
	  [[[[counterpart structure] extensionNamed: ext]
	    semantics] setDeclared: YES];

	  /* Make sure the information on the class object in this
             interface is output.  */
	  if (of)
	    [extensions_output addElement: C_EXT_SEMANTICS];
	}
    }

  if (current_extension)
    push_report_context (formac (nil, @"in %@",
				 ltt_ext_name (current_extension)));
  else if (current_either)
    push_report_context (formac (nil, @"in %@",
				 ltt_meta_name (current_either)));
  else
    push_report_context (formac (nil, @"somewhere in `%@'", current_filename));

  in_what = definitionp ? IMPLEMENTATION_SCOPE : INTERFACE_SCOPE;
}

OTMExpr *
do_return_stmt (int kind, OTMExpr *expr)
{
  OTMExpr *r = nil;

  if (current_method)
    if (expr)
      {
	id rt = [current_method returnType];

	if (rt == basic_type[BT_VOID])
	  if (kind)
	    r = [CO_OTMReturn assignmentWithLhs: void_expr rhs: expr];
	  else
	    {
	      warning (AT"setting void return value has no effect");
	      r = expr;
	    }
	else
	  {
	    OTMExpr *e = [current_method returnValue];

	    if (expr == void_expr)
	      {
		if (!e)
		  {
		    e = temp_something_with_type (rt, 0);
		    emit_assignment (e, nil_something_with_type (rt));
		    [current_method setReturnValue: e];
		  }

		/* Allow `return;' to indicate immediate return without
		   affecting the return value.  */
		expr = e;
	      }
	    else if (!e)
	      {
		e = temp_something_with_type (rt, 0);
		[current_method setReturnValue: e];
	      }

	    r = (kind == 1 ? [CO_OTMReturn assignmentWithLhs: e rhs: expr]
		 : (id) [CO_OTMAssignment assignmentWithLhs: e rhs: expr]);
	  }

	if (kind == 1)
	  [current_method setHaveReturnStatement];
      }

  return r;
}

OTMCompound *
enter_compound (void)
{
  if (current_compound || current_method)
    {
      current_compound = [CO_OTMCompound compoundWithContainer:
					 (current_compound ? current_compound
					  : (id) current_method)];
      if (![current_method body])
	[current_method setBody: current_compound];
    }

  return current_compound;
}

void
exit_compound (void)
{
  if (current_compound)
    {
      current_compound = [current_compound container];
      if (current_compound == (id) current_method)
	current_compound = nil;
    }
}

void
init_local_var (OTMVariable *v, OTMType *t, OTMExpr *expr)
{
  if (current_compound)
    {
      emit_local_var (v);
      emit_assignment (v, [resolve_expr (expr ? expr
					 : nil_something_with_type (t),
					 CONS (t, nil), nil,
					 C_EITHER_SEMANTICS) elaborate]);
      [current_compound releaseTemporaryVariables];
    }
}

OTMLocalVar *
create_local_var (id <TLString> name, OTMType *type, OTMExpr *expr)
{
  OTMLocalVar *r = [CO_OTMLocalVar variableWithName: name type: type];

  init_local_var (r, type, expr);

  return r;
}
%}

%pure_parser

%union {
  int i;
  id v;
}

/* Keywords.  */
%token CLASS END EXTENSION EXTERN IMPLEMENTATION INSTANCE INTERFACE POSING
%token RECEIVER SUPER CONST LOCAL OLD
%token IF ELSE DO FOR WHILE BREAK CONTINUE RETURN PRE POST

/* Qualifiers.  */
%token PUBLIC PRIVATE PROTECTED MUTABLE STATIC DEFERRED REDEFINE REDECLARE

/* Other stuff.  */
%token CATCH BIND UNWIND
%token DYNAMIC
%token EQ GE LE NE AND OR IMPLIES SHL SHR LSR
%token TYPEDEF VOID NIL
%token PLUSPLUS MINMIN
%token <i> ASSIGN
%token <v> STRING_CST IDENTIFIER NUMBER TYPE BASIC_TYPE C_LITERAL

%type <v> actual_top_expression argument_name argument_type
%type <v> array_reference assign_stmt atom bare_method_name_part
%type <v> break_stmt class_name compound .compounded_expr.  continue_stmt
%type <v> do_stmt entity_type .extension. expr expr_sc expr_or_compound
%type <v> .expr. expr_sequence expr_sequence_elts extension extension_name
%type <v> .foreign. for_stmt identifier_list if_stmt .if_else. local_var
%type <v> local_var_decl local_var_decl_list method_decl method_decl_part
%type <v> method_invocation method_invocation_part method_invocation_parts
%type <v> method_name_part method_decl_parts missing_class_name
%type <v> new_class_name object_type resolved_class_name
%type <v> rest_of_method_invocation return_stmt return_type .return_value_name.
%type <v> shared_method_part simple_expr special_form string_cst
%type <v> super_indication top_expr tuple tuple_argument_name
%type <v> tuple_elt_type tuple_field_list tuple_type tuple_type_field_list
%type <v> type_cast type_cast_type while_stmt
%type <v> selector_decl selector_name_parts

%type <i> .qualifiers. return_stmt_kind

%right ';'
%right ','
%right '{'
/* This is for the `break', `return' and `continue' expressions to reach
   as far to the right as possible, and to not reduce with an empty
   argument expression.  */
%right IF ELSE BREAK RETURN CONTINUE CLASS INSTANCE RECEIVER DO FOR WHILE VOID NIL IDENTIFIER TYPE BASIC_TYPE CATCH BIND UNWIND
%right '=' ASSIGN
%right '?' ':'
%left IMPLIES
%left OR
%left AND
%left '<' LE EQ NE GE '>'
%left '^'
%left '|'
%left '&'
%left SHL SHR LSR
%left '+' '-'
%left '*' '/' '%'
%left '~' '!' OLD UNARY_MINUS PLUSPLUS MINMIN
%left '['

%start file

%%
file:
	  /* empty */
	| file file_element
	;

file_element:
	  class_interface semicolon
	| instance_interface semicolon
	| class_implementation semicolon
	| instance_implementation semicolon
	| C_LITERAL
	    {
	      if (of)
		[$1 compileStatement];
	    }
	;

class_interface:
	  INTERFACE .foreign. .qualifiers.
		CLASS new_class_name .extension.
	    {
	      set_current_either ($5, $6, 1, $2, $3, 0);
	    }
	  .colon_super_list.
	    {
	      check_supers ();
	    }
	  .object_variables.
	  decl_list END
	    {
	      pop_report_context ();
	      in_what = GLOBAL_SCOPE;
	    }
	;

class_implementation:
	  IMPLEMENTATION .foreign. .qualifiers.
		CLASS new_class_name .extension.
	    {
	      set_current_either ($5, $6, 1, $2, $3, 1);
	    }
	  .colon_super_list.
	    {
	      check_supers ();
	    }
	  .object_variables.
	    {
	      if (of)
		[C_EXT_SEMANTICS startCompile];
	    }
	  def_list END
	    {
	      [C_EXT_SEMANTICS checkImplementation];
	      if (of)
		[C_EXT_SEMANTICS endCompile];
	      pop_report_context ();
	      in_what = GLOBAL_SCOPE;
	    }
	;

instance_interface:
	  INTERFACE .foreign. .qualifiers.
		INSTANCE new_class_name .extension.
	    {
	      set_current_either ($5, $6, 0, $2, $3, 0);
	    }
	  .colon_super_list.
	    {
	      check_supers ();
	    }
	  .object_variables.
	  decl_list END
	    {
	      pop_report_context ();
	      in_what = GLOBAL_SCOPE;
	    }
	;

instance_implementation:
	  IMPLEMENTATION .foreign. .qualifiers.
		INSTANCE new_class_name .extension.
	    {
	      set_current_either ($5, $6, 0, $2, $3, 1);
	    }
	  .colon_super_list.
	    {
	      check_supers ();
	    }
	  .object_variables.
	    {
	      if (of)
		[C_EXT_SEMANTICS startCompile];
	    }
	  def_list END
	    {
	      [C_EXT_SEMANTICS checkImplementation];
	      if (of)
		[C_EXT_SEMANTICS endCompile];
	      pop_report_context ();
	      in_what = GLOBAL_SCOPE;
	    }
	;

.object_variables.:
	  /* empty */
	| '{'
	    {
	      [C_EITHER_SEMANTICS setStatep];
	    }
	  object_variables '}'
	;

object_variables:
	  /* empty */
	| object_variables object_variable semicolon
	;

object_variable:
	  .qualifiers. entity_type identifier_list
	    {
	      if ($2)
		object_add_variables ($1, $2, $3, 0);
	    }
	| LOCAL .qualifiers. entity_type identifier_list
	    {
	      if ($3)
		object_add_variables ($2, $3, $4, 1);
	    }
	| .qualifiers. CONST IDENTIFIER '=' expr
	    {
	      if (in_what == IMPLEMENTATION_SCOPE)
		{
		  /* XXX Check consistency.  */
		}
	      else
		{
		  object_add_const ($1, $3, $5);
		}
	    }
	;

entity_type:
	  BASIC_TYPE
	| tuple_type
	| object_type
	;

return_type:
	  VOID { $$ = basic_type[BT_VOID]; }
	| DYNAMIC { $$ = the_dynamic_type; }
	| entity_type
	;

argument_type:
	  DYNAMIC { $$ = the_dynamic_type; }
	| entity_type
	;

tuple_elt_type:
	  entity_type
	;

/* The entity with this type is a reference to an object which is an
   instance of the class named CLASS_NAME, unless modified by CLASS or
   INSTANCE, of course.  */
object_type:
	  class_name
	    {
	      $$ = [$1 semantics];
	    }
	| RECEIVER
	    {
	      $$ = basic_type[BT_RECV];
	    }
	| CLASS '(' object_type ')'
	    {
	      $$ = [(OTMMeta *) $3 itsClass];
	    }
	| INSTANCE '(' object_type ')'
	    {
	      $$ = [(OTMMeta *) $3 instance];
	    }
	;

tuple_type:
	  '(' tuple_type_field_list ')'
	    {
	      if ([$2 cdr])
		$$ = [CO_OTMTypeTuple typeTupleWithSequence: $2];
	      else
		$$ = [$2 car];
	    }
	;

tuple_type_field_list:
	  tuple_elt_type
	    {
	      $$ = CONS ($1, nil);
	    }
	| tuple_type_field_list ',' tuple_elt_type
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

.colon_super_list.:
	  /* empty */
	| ':' super_list
	;

super_list:
	  super_indication
	    {
	      do_add_super ($1, 0);
	    }
	| POSING super_indication
	    {
	      do_add_super ($2, 1);
	    }
	| super_list ',' super_indication
	    {
	      do_add_super ($3, 0);
	    }
	| super_list ',' POSING super_indication
	    {
	      do_add_super ($4, 1);
	    }
	;

super_indication:
	  missing_class_name
	| INSTANCE '(' missing_class_name ')'
	    {
	      $$ = CONS (nil, $3);
	    }
	| CLASS '(' missing_class_name ')'
	    {
	      $$ = CONS ($3, nil);
	    }
	;

.extension.:
	  /* empty */ { $$ = NULL; }
	| extension
	;

extension:
	  EXTENSION extension_name { $$ = $2; }
	;

decl_list:
	  /* empty */
	| decl_list annotated_method_decl
	| decl_list typedef
	| decl_list error
	    {
	      error (AT"parse error at `%s'", yytext);
	    }
	;

annotated_method_decl:
	  shared_method_part semicolon
	    {
	      add_method_decl ($1);
	      current_method = nil;
	    }
	;

method_def:
	  shared_method_part
	    {
	      /* Don't set the current method if we're just reading an
                 interface.  */
	      current_method = of ? $1 : nil;
	    }
	  method_body
	    {
	      if (of && !NUM_ERRORS)
		[current_method compile];

	      current_method = nil;
	    }
	;

shared_method_part:
	    {
	      next_tmp = 0;
	      next_arg = 2;
	    }
	  .qualifiers. .foreign. method_decl
	    {
	      $2 = mask_qualifiers ($2, (OQ_DEFERRED_MASK
					 | OQ_REDO_MASK | OQ_PROTECTION_MASK));

	      /* Check foreigness compared to what was declared for this
                 class.  */
	      if (current_either)
		{
		  id f = [C_EITHER_SEMANTICS foreign];

		  if (!$3)
		    {
		      if (f)
			$3 = f;
		    }
		  else if (f == $3)
		    warning (AT"repeated foreign qualifier");
		  else if (f)
		    error (AT"foreign %# differs from class' %#", $3, f);
		}

	      [$4 setQualifiers: $2];
	      if ($3)
		{
		  if (Q_DEFERRED ($2))
		    error (AT"deferred method can't be foreign");
		  else
		    [$4 setForeign: $3];
		}

	      current_method = $4;
	    }
	  .pre. .post.
	    {
	      $$ = current_method;
	    }
	;

.foreign.:
	  /* empty */ { $$ = 0; }
	| EXTERN
	    {
	      $$ = AT"C";
	    }
	;

.qualifiers.:
	  /* empty */ { $$ = 0; }
	| .qualifiers. STATIC
	    {
	      if (current_instance)
		error (AT"static qualifier only allowed in classes");
	      else
		{
		  if (Q_STATIC ($1))
		    error (AT"`static' repeated");
		  $$ = Q_SET_STATIC ($1, OQ_STATIC);
		}
	    }
	| .qualifiers. DEFERRED
	    {
	      if (Q_DEFERRED ($1))
		error (AT"`deferred' repeated");
	      $$ = Q_SET_DEFERRED ($1, OQ_DEFERRED);
	    }
	| .qualifiers. MUTABLE
	    {
	      if (Q_MUTABLE ($1))
		error (AT"`deferred' repeated");
	      $$ = Q_SET_MUTABLE ($1, OQ_MUTABLE);
	    }
	| .qualifiers. REDEFINE
	    {
	      if (Q_REDO ($1))
		error (AT"conflicting qualifiers for `redefine'");
	      $$ = Q_SET_REDO ($1, OQ_REDEFINE);
	    }
	| .qualifiers. REDECLARE
	    {
	      if (Q_REDO ($1))
		error (AT"conflicting qualifiers for `redeclare'");
	      $$ = Q_SET_REDO ($1, OQ_REDECLARE);
	    }
	| .qualifiers. PUBLIC
	    {
	      if (Q_PROTECTION ($1))
		error (AT"conflicting qualifiers for `public'");
	      $$ = Q_SET_PROTECTION ($1, OQ_PUBLIC);
	    }
	| .qualifiers. PROTECTED
	    {
	      if (Q_PROTECTION ($1))
		error (AT"conflicting qualifiers for `protected'");
	      $$ = Q_SET_PROTECTION ($1, OQ_PROTECTED);
	    }
	| .qualifiers. PRIVATE
	    {
	      if (Q_PROTECTION ($1))
		error (AT"conflicting qualifiers for `private'");
	      $$ = Q_SET_PROTECTION ($1, OQ_PRIVATE);
	    }
	;

method_decl:
	  return_type .return_value_name. bare_method_name_part
	    {
	      $$ = $1 ? [CO_OTMCustomMethod methodWithExtension: C_EXT_SEMANTICS
				   name: $3 returnType: $1 flatp: NO] : nil;
	      [$$ setReturnVariables: $2];
	    }
	| return_type .return_value_name. method_decl_parts
	    {
	      $$ = ($1 && $3
		    ? [CO_OTMCustomMethod methodWithExtension: C_EXT_SEMANTICS
				 nameTypes: $3 returnType: $1 flatp: NO] : nil);
	      [$$ setReturnVariables: $2];
	    }
	;

.pre.:
	    /* empty */
	| PRE simple_expr
	    {
	      if (current_method)
		[current_method setPrecondition: $2];
	    }
	;

.post.:
	  /* empty */
	| POST simple_expr
	    {
	      if (current_method)
		[current_method setPostcondition: $2];
	    }
	;

.return_value_name.:
	  /* empty */ { $$ = nil; }
	| tuple_argument_name { $$ = $1; }
	;

method_decl_parts:
	  method_decl_part
	| method_decl_parts method_decl_part
	    {
	      $$ = [$1 nconc: $2];
	    }
	;

method_decl_part:
	  method_name_part argument_type argument_name
	    {
	      $$ = CONS ($1, digest_arguments ($3, $2, 0));
	      $$ = CONS ($$, nil);
	    }
	| method_name_part argument_type argument_name '=' expr
	    {
	      $$ = CONS ($1, digest_arguments ($3, $2, $5));
	      $$ = CONS ($$, nil);
	    }
	;

method_name_part:
	  bare_method_name_part
	| bare_method_name_part ':'
	    {
	      $$ = unique_identifier ([CO_TLString stringWithFormat: AT"%@:",
						$1]);
	    }
	| ':'
	    {
	      $$ = unique_identifier_colon;
	    }
	;

bare_method_name_part:
	  IDENTIFIER
	| BIND { $$ = unique_identifier (AT"bind"); }
	| CLASS { $$ = unique_identifier (AT"class"); }
	| CATCH { $$ = unique_identifier (AT"catch"); }
	| END { $$ = unique_identifier (AT"end"); }
	| EXTENSION { $$ = unique_identifier (AT"extension"); }
	| EXTERN { $$ = unique_identifier (AT"extern"); }
	| IMPLEMENTATION { $$ = unique_identifier (AT"implementation"); }
	| INSTANCE { $$ = unique_identifier (AT"instance"); }
	| INTERFACE { $$ = unique_identifier (AT"interface"); }
	| LOCAL { $$ = unique_identifier (AT"local"); }
	| POSING { $$ = unique_identifier (AT"posing"); }
	| RECEIVER { $$ = unique_identifier (AT"id"); }
	| SUPER { $$ = unique_identifier (AT"super"); }
	| CONST { $$ = unique_identifier (AT"const"); }
	| IF { $$ = unique_identifier (AT"if"); }
	| ELSE { $$ = unique_identifier (AT"else"); }
	| DO { $$ = unique_identifier (AT"do"); }
	| FOR { $$ = unique_identifier (AT"for"); }
	| WHILE { $$ = unique_identifier (AT"while"); }
	| BREAK { $$ = unique_identifier (AT"break"); }
	| CONTINUE { $$ = unique_identifier (AT"continue"); }
	| RETURN { $$ = unique_identifier (AT"return"); }
	| PUBLIC { $$ = unique_identifier (AT"public"); }
	| PRIVATE { $$ = unique_identifier (AT"private"); }
	| PROTECTED { $$ = unique_identifier (AT"protected"); }
	| MUTABLE { $$ = unique_identifier (AT"mutable"); }
	| STATIC { $$ = unique_identifier (AT"static"); }
	| DEFERRED { $$ = unique_identifier (AT"deferred"); }
	| REDEFINE { $$ = unique_identifier (AT"redefine"); }
	| REDECLARE { $$ = unique_identifier (AT"redeclare"); }
	| DYNAMIC { $$ = unique_identifier (AT"dynamic"); }
	| TYPEDEF { $$ = unique_identifier (AT"typedef"); }
	| UNWIND { $$ = unique_identifier (AT"unwind"); }
	| VOID { $$ = unique_identifier (AT"void"); }
	| NIL { $$ = unique_identifier (AT"nil"); }
	| BASIC_TYPE { $$ = unique_identifier ([(OTMMeta *) $1 typeName]); }
	| TYPE
	    {
	      $$ = unique_identifier ([[(LTTMeta *) [$1 _elementAtIndex: 0]
						    lttName] internal]);
	    }
	;

argument_name:
	  IDENTIFIER
	| tuple_argument_name
	;

tuple_argument_name:
	  '(' identifier_list ')'
	    {
	      $$ = [CO_OTMTuple tupleWithSequence: $2];
	    }
	;

def_list:
	  /* empty */
	| typedef
	| def_list method_def
	;

typedef:
	  TYPEDEF entity_type IDENTIFIER
	    {
	      ABORT ();
	    }
	;

method_body:
	  semicolon
	    {
	      if (current_method)
		{
		  otm_qualifiers q = [current_method qualifiers];

		  /* A method declaration in an implementation is only
                     allowed if it is a redeclaration, deferred or
                     foreign.  */
		  if (![current_method definedp] && !Q_REDO (q)
		      && !Q_DEFERRED (q) && ![current_method foreign])
		    error (AT"method definition missing");

		  add_method_def (current_method);
		}
	    }
	|
	    {
	      if (current_method)
		{
		  id f;

		  if ([current_method deferredp])
		    error (AT"definition of method declared deferred");

		  f = [current_method foreign];
		  if (f)
		    error (AT"definition of method declared extern", f);

		  current_method = add_method_def (current_method);
		}
	    }
	  method_body_expression
	;

method_body_expression:
	  '{'
	    {
	      enter_compound ();
	      if (current_method)
		{
		  OTMType *t = [current_method returnType];

		  if (t != basic_type[BT_VOID])
		    {
		      OTMTuple *rvn = [current_method returnVariables];

		      /* Initialize the return value.  */
		      if (rvn)
			[current_method setReturnValue: rvn];
		      else
			do_return_stmt (0, nil_something_with_type (t));
		    }

		  emit_conditions_pre (current_method);
		}
	    }
	  top_expression_list '}'
	    {
	      if ([current_method haveReturnStatement])
		emit_statement ([current_method setHaveReturnStatement]);

	      emit_conditions_post (current_method);
	      exit_compound ();
	    }
	;

compound:
	  '{'
	    {
	      enter_compound ();
	    }
	  top_expression_list '}'
	    {
	      $$ = current_compound;
	      exit_compound ();
	    }
	;

top_expression_list:
	  /* empty */
		{
		  emit_expr (void_expr);
		}
	| ne_top_expression_list
	;

ne_top_expression_list:
	  top_expression
	| ne_top_expression_list top_expression
	;

top_expression:
	  actual_top_expression
	    {
	      if (current_compound && $1)
		{
		  [current_compound setValue: $1];
		  [current_compound releaseTemporaryVariables];
		}
	    }
	;

actual_top_expression:
	  /* A local_var is not an expr (or top_expr) because that is
	     unnecessary.

	     The expression `int a = 1' can be just as well used without
	     the `int a = ' if the A is not used anywhere else, or, it can
	     be braced: `{ int a = 1 }'.  If it _is_ used somewhere else,
	     the declaration can be put in the enclosing compound and
	     then, obviously, the plain assignment `a = 1' can be used as
	     an expression.

	     For the expression `int a = 1, b = a + 1', the expression
	     `{int a = 1; a + 1}' can be used with exactly the same
	     semantics.  Remarks from the previous paragraph are also
	     valid for this case.

	     Also, if `int a = 1, b = 2' were an expression, what would be
	     the semantics of the tuple `(q = 1, 2, int a = 3, b = 4)'?
	     Why?  */
	  local_var semicolon
	| C_LITERAL
	    {
	      $$ = emit_expr ($1);
	    }
	| top_expr
	    {
	      $$ = emit_expr ($1);
	    }
	;

top_expr:
	    {
	      enter_compound ();
	    }
	  '{' top_expression_list '}' %prec '{'
	    {
	      $$ = current_compound;
	      exit_compound ();
	    }
	| expr_sc
	| /* empty */ semicolon
	    {
	      if (!flag_inhibit_empty)
		warning (AT"empty expression");
	      $$ = void_expr;
	    }
	;

local_var:
	  entity_type
	    {
	      enclosing_type = $1;
	    }
	  local_var_decl_list
	    {
	      $$ = $3;
	    }
	;

local_var_decl_list:
	  local_var_decl
	| local_var_decl_list ',' local_var_decl
	    {
	      $$ = $3;
	    }
	;

local_var_decl:
	  IDENTIFIER
	    {
	      $$ = create_local_var ($1, enclosing_type, nil);
	    }
	| IDENTIFIER '=' expr
	    {
	      $$ = create_local_var ($1, enclosing_type, $3);
	    }
	;

if_stmt:
	  IF
	    {
	      $$ = [CO_OTMCondExpr new];
	    }
	  tuple
	    {
	      [$<v>2 setIf: $3];
	    }
	  top_expr
	    {
	      [$<v>2 setThen: $5];
	    }
	  .if_else.
	    {
	      $$ = $<v>2;
	      [$$ setElse: $7];
	    }
	;

.if_else.:
	  /* empty */
	    {
	      $$ = nil;
	    }
	| ELSE top_expr
	    {
	      $$ = $2;
	    }
	;

do_stmt:
	  DO
	    {
	      current_compound = [CO_OTMLoop
				   loopWithContainer: current_compound];
	    }
	  top_expression WHILE tuple semicolon
	    {
	      [(OTMLoop *) current_compound startLoopEnd];
	      [(OTMLoop *) current_compound setCondition: $5 atEnd: 1];

	      $$ = current_compound;
	      current_compound = [current_compound container];
	    }
	;

while_stmt:
	  WHILE
	    {
	      current_compound = [CO_OTMLoop
				   loopWithContainer: current_compound];
	    }
	  tuple
	    {
	      [(OTMLoop *) current_compound setCondition: $3 atEnd: 0];
	    }
	  top_expression
	    {
	      $$ = current_compound;
	      [(OTMLoop *) current_compound startLoopEnd];
	      current_compound = [current_compound container];
	    }
	;

for_stmt:
	  FOR '(' .compounded_expr. semicolon
	    {
	      enter_compound ();
	      emit_statement ($3);
	      current_compound
		= [CO_OTMLoop loopWithContainer: current_compound];
	    }
	  .expr. semicolon
	    {
	      [(OTMLoop *) current_compound setCondition: $6 atEnd: 0];
	    }
	  .compounded_expr. ')' top_expression
	    {
	      OTMCompound *loop = current_compound;

	      [current_compound startLoopEnd];
	      emit_statement ($9);
	      current_compound = [current_compound container];
	      [current_compound setValue: emit_expr (loop)];
	      $$ = current_compound;
	      exit_compound ();
	    }
/*
	| FOR '(' error ')' top_expr
	    {
	      error (AT"error in for loop expression");
	      $$ = [CO_OTMError sharedError];
	    }
 */
	;

return_stmt:
	  return_stmt_kind expr %prec RETURN
	    {
	      $$ = do_return_stmt ($1, $2);
	    }
	| return_stmt_kind %prec RETURN
	    {
	      $$ = do_return_stmt ($1, void_expr);
	    }
	;

return_stmt_kind:
	  '=' { $$ = 0; }
	| RETURN { $$ = 1; }
	;

assign_stmt:
	  atom '=' expr
	    {
	      $$ = [CO_OTMAssignment assignmentWithLhs: $1 rhs: $3];
	    }
	| PLUSPLUS atom
	    {
	      $$ = [CO_OTMModAssign assignmentWithLhs: $2
				 rhs: (id) tll_small_int[1]
				 operator: BO_ADD postp: NO];
	    }
	| atom PLUSPLUS
	    {
	      $$ = [CO_OTMModAssign assignmentWithLhs: $1
				 rhs: (id) tll_small_int[1]
				 operator: BO_ADD postp: YES];
	    }
	| MINMIN atom
	    {
	      $$ = [CO_OTMModAssign assignmentWithLhs: $2
				 rhs: (id) tll_small_int[1]
				 operator: BO_SUB postp: NO];
	    }
	| atom MINMIN
	    {
	      $$ = [CO_OTMModAssign assignmentWithLhs: $1
				 rhs: (id) tll_small_int[1]
				 operator: BO_SUB postp: YES];
	    }
	| atom ASSIGN expr
	    {
	      $$ = [CO_OTMModAssign assignmentWithLhs: $1 rhs: $3 operator: $2
				 postp: YES];
	    }
	| array_reference '=' expr
	    {
	      $$ = build_invocation ([$1 car],
				     CONS (CONS (MNP_SET_AT_1, $3),
					   CONS (CONS (MNP_SET_AT_2, [$1 cdr]),
						 nil)), 0, nil);
	    }
	/* XXX Other operations on array elements...  */
	;

array_reference:
	  expr '[' tuple_field_list ']'
	    {
	      if ([[$3 elements] length] == 1)
		$3 = [[$3 elements] _elementAtIndex: 0];
	      $$ = CONS ($1, $3);
	    }
	| expr '[' error ']'
	    {
	      error (AT"error in index expression");
	      $$ = [CO_OTMError sharedError];
	    }
	;

.expr.:
	  /* empty */
	    {
	      $$ = nil;
	    }
	| expr
	;

.compounded_expr.:
	  /* empty */
	    {
	      $$ = nil;
	    }
	|
	    {
	      enter_compound ();
	    }
	  expr_or_compound
	    {
	      $2 = emit_expr ($2);
	      if ($2)
		[current_compound setValue: $2];
	      $$ = current_compound;
	      exit_compound ();
	    }
	;

expr_sc:
	  if_stmt
	| for_stmt
	| do_stmt
	| while_stmt
	| expr semicolon
	;

expr:
	  simple_expr
	| assign_stmt
	| return_stmt
	| continue_stmt
	| break_stmt
	| special_form
	;

expr_or_compound:
	  expr
	| compound
	;

special_form:
	  CATCH expr_sequence
	    {
	      if ($2 && $2 != an_error)
		{
		  if ([$2 cdr])
		    error (AT"too many elements in catch tag");

		  $2 = [resolve_expr ([$2 car], CONS (the_any_ref_type, nil),
				      NULL, [current_either semantics])
			elaborate];
		}
	      current_compound = [CO_OTMCatch catchWithTag: $2
					   container: current_compound];
	    }
	  expr
	    {
	      [current_compound setValue: emit_expr ($4)];
	      $$ = current_compound;
	      current_compound = [current_compound container];
	    }
	| BIND
	    {
	      enter_compound ();

	      /* This is naughty: a local variable for tom code to
		 reference the argument (with the same name) to the
		 function containing the handlers.  */
	      emit_local_var ([CO_OTMArgument
				variableWithName: AT"condition"
				type: tom_condition_instance]);

	      if (![tom_condition_instance declaredp]
		  && ![tom_condition_instance definedp])
		load_interface ((LTTInstance *)
				[tom_condition_instance structure]);
	    }
	  expr_sequence
	    {
	      current_compound = [CO_OTMBind bindWithContainer:
					     current_compound];
	      [(OTMBind *) current_compound setHandlers: $3];
	    }
	  expr
	    {
	      [current_compound setValue: emit_expr ($5)];
	      $$ = current_compound;
	      current_compound = [current_compound container];
	      [current_compound setValue: emit_expr ($$)];
	      $$ = current_compound;
	      current_compound = [current_compound container];
	    }
	| UNWIND expr_sequence expr
	    {
	      if ($2 && $2 != an_error)
		{
		  if ([$2 cdr])
		    error (AT"too many elements in unwind protection");

		  enter_compound ();
		  [current_compound setValue: emit_expr ([$2 car])];
		  $2 = current_compound;
		  current_compound = [current_compound container];
		}

	      current_compound = [CO_OTMUnwind unwindWithProtection: $2
					    container: current_compound];
	      [current_compound setValue: emit_expr ($3)];
	      $$ = current_compound;
	      current_compound = [current_compound container];
	    }
	;

expr_sequence:
	  '(' expr_sequence_elts ')'
	    {
	      $$ = $2;
	    }
	;

expr_sequence_elts:
	  expr_or_compound
	    {
	      $$ = CONS ($1, nil);
	    }
	| expr_sequence_elts ';' expr_or_compound
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

break_stmt:
	  BREAK
	    {
	      $$ = [CO_OTMBreak jumpStatement];
	    }
	| BREAK expr
	    {
	      $$ = [CO_OTMBreak jumpStatementWithValue: $2];
	    }
	;

continue_stmt:
	  CONTINUE
	    {
	      $$ = [CO_OTMContinue jumpStatement];
	    }
	| CONTINUE expr
	    {
	      $$ = [CO_OTMContinue jumpStatementWithValue: $2];
	    }
	;

simple_expr:
	  atom
	| array_reference
	    {
	      if (!$1 || $1 == an_error)
		$$ = [CO_OTMError sharedError];
	      else
		$$ = invocation ([$1 car], unique_identifier (MNP_AT_1),
				 [$1 cdr]);
	    }
	| '-' expr %prec UNARY_MINUS
	    {
	      $$ = op_invocation1 (builtin_operator_name[BO_NEG], $2);
	    }
	| '~' expr
	    {
	      $$ = op_invocation1 (builtin_operator_name[BO_INV], $2);
	    }
	| '!' expr
	    {
	      $$ = op_invocation1 (builtin_operator_name[BO_NOT], $2);
	    }
	| OLD expr
	    {
	      $$ = [CO_OTMOld with: $2];
	    }
	| expr '*' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_MUL], $1, $3);
	    }
	| expr '/' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_DIV], $1, $3);
	    }
	| expr '%' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_MOD], $1, $3);
	    }
	| expr '+' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_ADD], $1, $3);
	    }
	| expr '-' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_SUB], $1, $3);
	    }
	| expr SHL expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_SHL], $1, $3);
	    }
	| expr SHR expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_SHR], $1, $3);
	    }
	| expr LSR expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_LSR], $1, $3);
	    }
	| expr '&' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_AND], $1, $3);
	    }
	| expr '|' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_OR], $1, $3);
	    }
	| expr '^' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_EOR], $1, $3);
	    }
	| expr '<' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_LT], $1, $3);
	    }
	| expr LE expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_LE], $1, $3);
	    }
	| expr EQ expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_EQ], $1, $3);
	    }
	| expr NE expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_NE], $1, $3);
	    }
	| expr GE expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_GE], $1, $3);
	    }
	| expr '>' expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_GT], $1, $3);
	    }
	| expr AND expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_SC_AND], $1, $3);
	    }
	| expr OR expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_SC_OR], $1, $3);
	    }
	| expr IMPLIES expr
	    {
	      $$ = op_invocation2 (builtin_operator_name[BO_IMPLIES], $1, $3);
	    }
	| expr '?' expr ':' expr
	    {
	      $$ = [CO_OTMCondExpr iteWithCondition: $1 thenPart: $3 elsePart: $5];
	    }
	| expr '?' expr error
	    {
	      error (AT"`:' expected");
	      $$ = [CO_OTMError sharedError];
	    }
	;
atom:
	  NUMBER
	| string_cst
	| IDENTIFIER
	    {
	      $$ = nil;

	      /* Look for the variable in the current compound.  If the
                 compound is not set, we could be in a method condition,
                 and search for a method argument.  */
	      if (current_compound)
		$$ = [current_compound searchVariableNamed: $1];
	      else if (current_method)
		$$ = [current_method searchVariableNamed: $1];

	      if ($$)
		$$ = [OTMVarRef referenceToVariable: $$];
	      else
		{
		  $$ = [C_EITHER_SEMANTICS searchEntityNamed: $1
			supers: YES class: YES variables: YES constants: YES];

		  if (!$$)
		    if (!of)
		      $$ = [[CO_OTMIdentifier gcAlloc] initWithName: $1];
		    else
		      {
			error (AT"undeclared variable `%@'", $1);
			if (current_compound)
			  {
			    $$ = [CO_OTMLocalVar variableWithName: $1
					    type: [ltt_instance_any semantics]];
			    cerror (AT"inventing `%@ %@'",
				    type_name ([$$ type]), $1);
			    emit_local_var ($$);
			  }
		      }
		}
	    }
	| VOID
	    {
	      $$ = void_expr;
	    }
	| NIL
	    {
	      $$ = nil_expr;
	    }
	| tuple
	| method_invocation
	| type_cast
	;

string_cst:
	  STRING_CST
/* XXX
	| string_cst STRING_CST
	    {
	      [$1 append: $2];
	      $$ = $1;
	    }
*/
	;

type_cast:
	  type_cast_type '(' expr ')'
	    {
	      if ($1 && $3)
		{
		  if ($1 == basic_type[BT_SELECTOR]
		      && [$3 isKindOf: [CO_OTMStringCST self]])
		    {
		      $$ = [[CO_LTTSelector selectorWithMangledName:
					      [[(OTMStringCST *) $3 structure]
						string]] semantics];
		      warning (AT"obsolete selector syntax");
		    }
		  else
		    $$ = [CO_OTMCast castWithExpr: $3 type: $1];
		}
	      else
		$$ = $3;
	    }
	| type_cast_type '(' SUPER ')'
	    {
	      error (AT"cast of super not allowed");
	      $$ = [CO_OTMError sharedError];
	    }
	| type_cast_type '(' selector_decl ')'
	    {
	      if ($1 != basic_type[BT_SELECTOR])
		error (AT"invalid cast");
	      else
		$$ = $3;
	    }
	| type_cast_type '(' error ')'
	    {
	      error (AT"parse error at `%s'", yytext);
	      $$ = [CO_OTMError sharedError];
	    }
	;

selector_decl:
	  return_type selector_name_parts
		{
		  TLCons *c = $2;
		  TLString *in_args, *out_args, *name;
		  out_args = [$1 flatFrobnicatedName];
		  in_args = formac (nil, AT"");
		  name = formac (nil, AT"%@", [$1 frobnicatedName]);
		  while (c)
		    {
		      TLCons *th;
		      id <TLString> np;
		      OTMType *tp;

		      DECONS (c, th, c);
		      DECONS (th, np, tp);

		      formac (name, AT"_%@", np);
		      if (tp != nil)
			{
			  formac (name, AT"_%@", [tp frobnicatedName]);
			  formac (in_args, AT"%@", [tp flatFrobnicatedName]);
			}
		    }
		  $$ = [[LTTSelector selectorWithName: name
				     inArgs: in_args outArgs: out_args]
			 semantics];
		}
	;

/* Return a list of (name . type) pairs.  */
selector_name_parts:
	  method_name_part
		{
		  $$ = CONS (CONS ($1, nil), nil);
		}
	| method_name_part argument_type
		{
		  $$ = CONS (CONS ($1, $2), nil);
		}
	| method_name_part argument_type selector_name_parts
		{
		  $$ = CONS (CONS ($1, $2), $3);
		}
	;

type_cast_type:
	  object_type
	| BASIC_TYPE
	;

method_invocation:
	  '[' expr_or_compound rest_of_method_invocation close_bracket
	    {
	      $$ = build_invocation ($2, $3, 0, nil);
	    }
	| '[' class_name rest_of_method_invocation close_bracket
	    {
	      $$ = build_invocation ([[(OTMMeta *) [(LTTMeta *) $2 semantics]
				       itsClass] metaReference], $3, 0, nil);
	    }
	| '[' SUPER rest_of_method_invocation close_bracket
	    {
	      $$ = build_invocation
		([current_method argumentNamed: TO_NAME_SELF], $3, 1, nil);
	    }
	| '[' SUPER '(' object_type close_paren rest_of_method_invocation
	  close_bracket
	    {
	      $$ = build_invocation
		([current_method argumentNamed: TO_NAME_SELF], $6, 1, $4);
	    }
	| IDENTIFIER tuple
	    {
	      OTMExpr *e = [CO_OTMMetaRef metaRefWithMeta:
				       [[current_either itsClass] semantics]];
	      $$ = invocation (e, $1, $2);
	    }
	| '[' error ']'
	    {
	      error (AT"error in invocation");
	      $$ = [CO_OTMError sharedError];
	    }
	;

rest_of_method_invocation:
	  method_name_part
	| method_invocation_parts
	;

method_invocation_parts:
	  method_invocation_part
	| method_invocation_parts method_invocation_part
	    {
	      $$ = [$1 nconc: $2];
	    }
	;

method_invocation_part:
	  method_name_part expr
	    {
	      $$ = CONS (CONS ($1, $2), nil);
	    }
	;

tuple:
	  '(' tuple_field_list ')'
	    {
	      if ([[$2 elements] length] == 1)
		$$ = [[$2 elements] _elementAtIndex: 0];
	      else
		{
		  $$ = $2;
		}
	    }
	| '(' error ')'
	    {
	      error (AT"error in tuple");
	      $$ = [CO_OTMError sharedError];
	    }
	;

tuple_field_list:
	  expr_or_compound
	    {
		$$ = [CO_OTMTuple tuple];
		[$$ addElement: $1];
	    }
	| /* empty */
	    {
		$$ = [CO_OTMTuple tuple];
		[$$ addElement: nil];
	    }
	| tuple_field_list ',' expr_or_compound
	    {
		[$1 addElement: $3];
		$$ = $1;
	    }
	| tuple_field_list ','
	    {
		[$1 addElement: nil];
		$$ = $1;
	    }
	;

identifier_list:
	  IDENTIFIER
	    {
	      $$ = CONS ($1, nil);
	    }
	| identifier_list ',' IDENTIFIER
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

semicolon:
	  ';'
	| error
	    {
	      error (AT"`;' expected");
	    }
	;

close_paren:
	  ')'
	| error
	    {
	      error (AT"`)' expected");
	    }
	;

close_bracket:
	  ']'
	| error
	    {
	      error (AT"`]' expected");
	    }
	;

new_class_name:
	  resolved_class_name
	;

class_name:
	  resolved_class_name
	    {
	      OTMMeta *oins = [$1 semantics];

	      $$ = $1;
	      if ($1 && ![oins fullyLoaded])
		load_interface ($1);
	    }
	;

resolved_class_name:
	  TYPE
	    {
	      $$ = instance_in_unit (nil, current_unit, $1);
	    }
	| TYPE '.' TYPE
	    {
	      $$ = instance_in_unit ([[(LTTMeta *) [$1 _elementAtIndex: 0]
					   lttName] internal],
				     current_unit, $3);
	    }
	| IDENTIFIER '.' TYPE
	    {
	      $$ = instance_in_unit ($1, current_unit, $3);
	    }
	;

missing_class_name:
	  class_name
	| IDENTIFIER
	    {
	      error (AT"inventing class `%@'", $1);

	      $$ = [ltt_missing_unit metaWithName: $1];	      
	    }
	| IDENTIFIER '.' IDENTIFIER
	    {
	      LTTUnit *u = [CO_LTTUnit unitNamed: $1];

	      error (AT"inventing class `%@.%@'", $1, $3);

	      if (!u)
		{
		  error (AT"unknown unit `%@'", $1);
		  u = ltt_missing_unit;
		}

	      $$ = [u metaWithName: $3];
	    }
	;

extension_name:
	  IDENTIFIER
	| TYPE
	    {
	      $$ = [[(LTTMeta *) [$1 _elementAtIndex: 0] lttName] internal];
	    }
	;
%%
