/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMMeta.m,v 1.82 1998/04/28 15:49:06 tiggr Exp $  */

#define OTMMETA_DECLARE_PRIVATE_METHODS
#import "OTMMeta.h"
#import "OTMClass.h"
#import "OTMInstance.h"
#import "OTMExtension.h"
#import "OTMMetaRef.h"
#import "OTMObjectVar.h"
#import "OTMType.h"
#import "global.h"

static id otmmeta_class;

@implementation LTTMeta (Semantics)

+(id) semanticsForInstance: (LTTInstance *) inst
{
  return [(OTMInstance *) [CO_OTMInstance gcAlloc] initWithStructure: inst];
}

+(id) semanticsForClass: (LTTClass *) cls
{
  return [(OTMClass *) [CO_OTMClass gcAlloc] initWithStructure: cls];
}

@end

@implementation OTMMeta

+(void) init
{
  [[ltt_instance_any semantics] setDefined: YES];
  otmmeta_class = [CO_OTMMeta self];
}

+(void) resolveIdentifiers
{
  id <TLEnumerator> e = [CO_LTTMeta metas];
  LTTMeta *str;

  while ((str = [e nextObject]))
    [[str semantics] resolveIdentifiers];
}

-(id) actualSelf: (id) context
{
  return [self posedSelf];
}

-(BOOL) allowedTypeForArgumentRedeclaration: (OTMType *) tt
				 inSubclass: (BOOL) subclass_p
{
  OTMMeta *t = (id) tt;

  return t == self;
}

-(BOOL) classp
{
  return NO;
}

-(TLVector *) collectConditions: (BOOL) pre_not_post
			    for: (OTMCustomMethod *) method
{
  return [self collectConditions: pre_not_post
	       for: method into: nil mark: ++search_mark];
}

-(TLVector *) collectConditions: (BOOL) pre_not_post
			    for: (OTMCustomMethod *) method
			   into: (TLVector *) conditions
			   mark: (int) k
{
  LTTExtension *ext;
  OTMExpr *found;
  TLVector *v;
  LTTMeta *m;
  int i, n;

  if (mark == k)
    return conditions;
  mark = k;

  v = [structure extensions];
  for (i = 0, n = [v length]; i < n; i++)
    {
      ext = [v _elementAtIndex: i];
      found = [(OTMExtension *) [ext semantics]
				collectConditions: pre_not_post for: method];
      if (found)
	{
	  if (!conditions)
	    conditions = [CO_TLVector new];
	  [conditions addElement: found];
	}
    }

  v = [structure stateSupers];
  for (i = 0, n = [v length]; i < n; i++)
    {
      m = [v _elementAtIndex: i];
      conditions = [[m semantics] collectConditions: pre_not_post
				  for: method into: conditions mark: k];
    }

  v = [structure behaviourSupers];
  for (i = 0, n = [v length]; i < n; i++)
    {
      m = [v _elementAtIndex: i];
      conditions = [[m semantics] collectConditions: pre_not_post
				  for: method into: conditions mark: k];
    }

  return conditions;
}

-(void) compileDeclaration
{
  if (!did_output_pointer_decl)
    {
      id refname = [self outputTypeName];

      did_output_pointer_decl = 1;
      formac (of, @"typedef struct %@ *%@;\n", refname, refname);
    }
}

/* Output the extension structure fields making up this meta, after having
   the superclasses do this.  */
-(void) compileObjVarDeclarations: (int) k
{
  LTTExtension *x;
  LTTMeta *m;
  TLVector *v;
  int i, n;

  if (mark == k)
    return;
  mark = k;

  v = [structure stateSupers];
  for (i = 0, n = [v length]; i < n; i++)
    {
      m = [v _elementAtIndex: i];
      [[m semantics] compileObjVarDeclarations: k];
    }

  v = [structure extensions];
  for (i = 0, n = [v length]; i < n; i++)
    {
      OTMExtension *e;

      x = [v _elementAtIndex: i];
      e = [x semantics];
      if ([e hasVariables])
	formac (of, @"struct %@ %@;\n", [x outputExtensionStructName],
		[x outputExtensionFieldName]);
    }
}

-(void) compileObjVarTypeDeclarations: (int) k
{
  TLVector *v;
  int mi, mn;

  if (mark == k || did_output_extension_declarations)
    return;
  mark = k;

  did_output_extension_declarations = 1;

  v = [structure stateSupers];
  for (mi = 0, mn = [v length]; mi < mn; mi++)
    {
      LTTMeta *m = [v _elementAtIndex: mi];
      [[m semantics] compileObjVarTypeDeclarations: k];
    }

  v = [structure extensions];
  for (mi = 0, mn = [v length]; mi < mn; mi++)
    {
      LTTExtension *x = [v _elementAtIndex: mi];
      [[x semantics] compileObjVarTypeDeclarations];
    }
}

-(void) compileSuperReferenceDeclaration: (OTMMeta *) the_super
{
  if (!super_refs)
    super_refs = [TLSet set];

  if (![super_refs memq: the_super])
    {
      [super_refs addElement: the_super];
      formac (of, @"extern struct _es_c_tom_State *%@;\n",
	      [structure outputSuperReference: [the_super structure]]);
    }
}

-(void) description: (id <TLMutableStream>) stream
{
  [super description: stream];
  formac (stream, @" %@", ltt_meta_name (structure));
}

-(void) dumpInfo: (id <TLOutputStream>) s
{
  TLVector *v;
  int mi, mn;

  formac (s, @"\n (supers");

  v = [structure stateSupers];
  for (mi = 0, mn = [v length]; mi < mn; mi++)
    {
      LTTMeta *m = [v _elementAtIndex: mi];
      formac (s, @"\n  (%# %#)", [[[m unit] lttName] internal],
	      [[m lttName] internal]);
    }

  v = [structure behaviourSupers];
  for (mi = 0, mn = [v length]; mi < mn; mi++)
    {
      LTTMeta *m = [v _elementAtIndex: mi];
      formac (s, @"\n  %@", [[m semantics] typeInfo]);
    }

  formac (s, @")");

  if (super_refs)
    {
      id <TLEnumerator> e = [super_refs enumerator];
      OTMMeta *m;

      formac (s, @"\n (super-refs");

      while ((m = [e nextObject]))
	formac (s, @"\n  %@", [m typeInfo]);

      formac (s, @")");
    }
}

-(int) flatElementCount
{
  return 1;
}

-(id <TLString>) foreign
{
  return foreign;
}

-(id <TLString>) flatFrobnicatedName
{
  return @"r";
}

-(id <TLString>) frobnicatedName
{
  return @"r";
}

-(BOOL) fullyLoaded
{
  return fully_loaded;
}

-(void) gcReference
{
  MARK (structure);
  MARK (reference);
  MARK (super_refs);
  MARK (foreign);

  [super gcReference];
}

-initWithStructure: (LTTMeta *) str
{
  if (![super initWithName: [[str lttName] internal]])
    return nil;

  structure = str;

  return self;
}

-(OTMMeta *) instance
{
  return [[structure instance] semantics];
}

-(BOOL) isFullyDefinedType
{
  return YES;
}

-(BOOL) isObjectType
{
  return YES;
}

-(OTMMeta *) itsClass
{
  return [[structure itsClass] semantics];
}

-(BOOL) matches: (OTMType *) t
{
  return [t isObjectType];
}

-(int) matchesConvertibly: (OTMType *) t
{
  return ((self == (id) t
	   || ([t isKindOf: [CO_OTMMeta self]]
	       && [structure isProperSub: [(OTMMeta *) t structure]]))
	  ? 0 : -1);
}

-(OTMType *) matchesExactly: (OTMType *) t
{
  if (self == (void *) t)
    return self;
  return [t matchesExactlyMeta: self];
}

-(OTMType *) matchesExactlyMeta: (OTMMeta *) t
{
  LTTMeta *ts = [t structure];

  if ([structure isProperSub: ts])
    return self;
  if ([ts isProperSub: structure])
    return t;

  return nil;
}

-(OTMMetaRef *) metaReference
{
  if (!reference)
    ASGN_IVAR (reference, [CO_OTMMetaRef metaRefWithMeta: self]);
  return reference;
}

-(TLCons *) methodsNamed: (TLVector *) name_parts
		  sender: (OTMMeta *) sender
		   super: (BOOL) super_p
		confined: (OTMMeta *) confined
{
  return [(super_p ? self : [self posedSelf]) methodsNamed: name_parts
	  sender: sender super: super_p confined: confined
	  list: nil mark: ++search_mark];
}

-(TLCons *) methodsNamed: (TLVector *) name_parts
		  sender: (OTMMeta *) sender
		   super: (BOOL) super_p
		confined: (OTMMeta *) confined
		    list: (TLCons *) l
		    mark: (int) k
{
  if (mark != k)
    {
      BOOL done = !super_p || !confined;
      LTTExtension *ext;
      OTMMeta *sm, *psm;
      TLVector *v;
      LTTMeta *m;
      int i, n;

      mark = k;

      if (!super_p)
	{
	  v = [structure extensions];
	  for (i = 0, n = [v length]; i < n; i++)
	    {
	      ext = [v _elementAtIndex: i];
	      l = [[ext semantics] methodsNamed: name_parts
				   sender: sender list: l];
	    }
	}

      v = [structure stateSupers];
      for (i = 0, n = [v length]; i < n; i++)
	{
	  m = [v _elementAtIndex: i];
	  sm = [m semantics];

	  if (!confined || confined == sm)
	    {
	      psm = [sm posedSelf];
	      l = [psm methodsNamed: name_parts sender: sender
		       super: 0 confined: nil list: l mark: mark];

	      /* This is needed in case SELF is the PSM.  */
	      if (psm != sm)
		l = [sm methodsNamed: name_parts sender: sender
			super: 0 confined: nil list: l mark: mark];

	      done = YES;
	    }
	}

      v = [structure behaviourSupers];
      for (i = 0, n = [v length]; i < n; i++)
	{
	  m = [v _elementAtIndex: i];
	  sm = [m semantics];

	  if (!confined || confined == sm)
	    {
	      psm = [sm posedSelf];
	      l = [psm methodsNamed: name_parts sender: sender
		       super: 0 confined: nil list: l mark: mark];

	      /* This is needed in case SELF is the PSM.  */
	      if (psm != sm)
		l = [sm methodsNamed: name_parts sender: sender
			super: 0 confined: nil list: l mark: mark];

	      done = YES;
	    }
	}

      if (!done)
	error (@"%@ is not a direct super of %@",
	       type_name ((id) confined), type_name ((id) sender));
    }
  return l;
}

-(int) minimumAlignment
{
  return 8 * sizeof (void *);
}

-(id <TLString>) outputCastName
{
  return @"void *";
}

-(id <TLString>) outputTypeEncoding
{
  return @"TRT_TE_REFERENCE";
}

-(id <TLString>) outputFunctionTypeForType
{
  return @"reference_imp";
}

-(id <TLString>) outputTypeName
{
  return [structure outputTypeName];
}

-(OTMMeta *) posedSelf
{
  LTTMeta *poser = [structure posedSelf];

  return structure == poser ? self : [poser semantics];
}

-(id) precompile
{
  if (![self declarationOutputp])
    {
      id previous_context = output_current_context;

      output_current_context = structure;

      [self setDeclarationOutputP: YES];

      [self compileDeclaration];
      [self compileObjVarTypeDeclarations: ++search_mark];

      formac (of, @"struct %@\n{\n", [self outputTypeName]);
      [self compileObjVarDeclarations: ++search_mark];
      formac (of, @"};\n");

      formac (of, @"extern struct trt_class %@;\n",
	      [structure metaDefinitionName]);

      output_current_context = previous_context;
    }

  return self;
}

-(void) resolveIdentifiers
{
  LTTExtension *ext;
  TLVector *v;
  int i, n;

  v = [structure extensions];
  for (i = 0, n = [v length]; i < n; i++)
    {
      ext = [v _elementAtIndex: i];
      [[ext semantics] resolveIdentifiers: self];
    }
}

-(OTMMethod *) searchSimilarMethod: (OTMMethod *) method
{
  return [[self posedSelf] searchSimilarMethod: method mark: ++search_mark];
}

-(OTMMethod *) searchSimilarMethod: (OTMMethod *) method mark: (int) k
{
  LTTExtension *ext;
  OTMMethod *found;
  TLVector *v;
  LTTMeta *m;
  int i, n;

  if (mark == k)
    return nil;
  mark = k;

  v = [structure extensions];
  for (i = 0, n = [v length]; i < n; i++)
    {
      ext = [v _elementAtIndex: i];
      found = [[ext semantics] searchSimilarMethod: method];
      if (found)
	return found;
    }

  v = [structure stateSupers];
  for (i = 0, n = [v length]; i < n; i++)
    {
      m = [v _elementAtIndex: i];
      found = [[m semantics] searchSimilarMethod: method mark: mark];
      if (found)
	return found;
    }

  v = [structure behaviourSupers];
  for (i = 0, n = [v length]; i < n; i++)
    {
      m = [v _elementAtIndex: i];
      found = [[m semantics] searchSimilarMethod: method mark: mark];
      if (found)
	return found;
    }

  return nil;
}

-(void) setForeign: (id <TLString>) f
{
  ASGN_IVAR (foreign, f);
  [self setDefined: YES];
}

-(void) setFullyLoaded
{
  fully_loaded = 1;
}

-(BOOL) statep
{
  return has_state;
}

-(void) setStatep
{
  has_state = YES;
}

-(void) setThreadLocalVarsP
{
  have_local_vars = 1;
}

-(LTTMeta *) structure
{
  return structure;
}

-(OTMObjectVar *) searchEntityNamed: (id <TLString>) n
			     supers: (BOOL) search_supers
			      class: (BOOL) search_class
			  variables: (BOOL) vp
			  constants: (BOOL) cp
{
  OTMObjectVar *v;

  v = [[self posedSelf] searchEntityNamed: n supers: search_supers
			mark: ++search_mark variables: vp constants: cp];

  if (!v && search_class)
    v = [[(OTMMeta *) [[structure itsClass] semantics] posedSelf]
	 searchEntityNamed: n supers: search_supers mark: search_mark
	 variables: vp constants: cp];

  if (v && [v protection] == OQ_PRIVATE)
    {
      /* If we're in a subclass, we're not allowed to reference the
         private entity.  */
      OTMMeta *my_class, *var_class;

      my_class = [self classp] ? self : [[structure itsClass] semantics];
      var_class = [[[[v extension] structure] meta] semantics];
      if (![var_class classp])
	var_class = [var_class meta];

      if (var_class != my_class)
	{
	  error (@"disallowed access to PRIVATE `%@'", n);
	  error_for (v, @"which was declared here");
	}
    }

  return v;
}

-(OTMObjectVar *) searchEntityNamed: (id <TLString>) nm
			     supers: (BOOL) search_supers
			       mark: (int) k
			  variables: (BOOL) vp
			  constants: (BOOL) cp
{
  OTMObjectVar *v;
  TLVector *vec;
  LTTMeta *m;
  int i, n;

  if (mark == k)
    return nil;
  mark = k;

  v = [self entityNamed: nm variables: vp constants: cp];
  if (v)
    return v;

  if (search_supers)
    {
      vec = [structure stateSupers];
      for (i = 0, n = [vec length]; i < n; i++)
	{
	  OTMMeta *sem;

	  m = [vec _elementAtIndex: i];
	  sem = [m semantics];

	  v = [sem searchEntityNamed: nm supers: YES mark: k
		   variables: vp constants: cp];
	  if (v)
	    return v;
	}

      vec = [structure behaviourSupers];
      for (i = 0, n = [vec length]; i < n; i++)
	{
	  OTMMeta *sem;

	  m = [vec _elementAtIndex: i];
	  sem = [m semantics];

	  v = [sem searchEntityNamed: nm supers: YES mark: k
		   variables: NO constants: cp];
	  if (v)
	    return v;
	}
    }

  return nil;
}

-(id <TLString>) typeInfo
{
  return formac (nil, @"(%@ %# %#)",
		 [structure classp] ? @"class" : @"instance",
		 [[[structure unit] lttName] internal],
		 [[structure lttName] internal]);
}

-(BOOL) validCastTo: (OTMType *) type
{
  OTMMeta *m = (id) type;

  return (m == self
	  || ([m isKindOf: otmmeta_class]
	      && ([structure isProperSub: [m structure]]
		  || [[m structure] isProperSub: structure])));
}

-(OTMObjectVar *) entityNamed: (id <TLString>) nm
		    variables: (BOOL) vp
		    constants: (BOOL) cp
{
  TLVector *vec = [structure extensions];
  LTTExtension *ext, *found_ext = nil;
  id v, found_v = nil;
  int i, n;

  for (i = 0, n = [vec length]; i < n; i++)
    {
      BOOL this_found = NO;
      OTMExtension *sem;

      ext = [vec _elementAtIndex: i];
      sem = [ext semantics];

      if (vp)
	{
	  v = [sem variableNamed: nm];
	  if (v)
	    this_found = YES;
	}
      if (cp && !this_found)
	{
	  v = [sem constantNamed: nm];
	  if (v)
	    this_found = YES;
	}

      if (this_found)
	{
	  /* Favour the declaration seen by the current extension.  */
	  if (ext == current_extension)
	    return v;

	  /* Otherwise, see the redeclaration by any (the `first')
             non-main extension.  */
	  if (!found_v || [found_ext isMainExtension])
	    {
	      found_v = v;
	      found_ext = ext;
	    }
	}
    }

  return found_v;
}

/********** OTMExpr protocol **********/

-(id) type
{
  /* A class object is its own type.  */
  return self;
}

@end
