/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMExtension.m,v 1.70 1998/02/02 12:33:31 tiggr Exp $  */

#define OTMEXTENSION_DECLARE_PRIVATE_METHODS
#import "OTMExtension.h"
#import "OTMAlias.h"
#import "OTMAliasAlias.h"
#import "OTMClass.h"
#import "OTMConstant.h"
#import "OTMCustomMethod.h"
#import "OTMMeta.h"
#import "OTMObjectVar.h"
#import "global.h"

@implementation LTTExtension (Semantics)

+(id) semanticsForExtension: (LTTExtension *) ext
{
  return [[CO_OTMExtension gcAlloc] initWithStructure: ext];
}

@end

@implementation OTMExtension

-(void) addMethod: (OTMMethod *) m
{
  TLVector *v = [self methodsNamed: [m firstNamePart] create: YES];

  [v addElement: m];
}

-(void) handlePostponedAlias: (OTMAliasAlias *) alias
{
  id <TLString> name = [alias name];
  OTMObjectVar *var = [[[structure meta] semantics]
			searchEntityNamed: name supers: YES class: NO
			variables: YES constants: NO];

  if (!var)
    postponed_aliases = CONS (alias, postponed_aliases);
  else
    {
      int q = [alias qualifiers];
      OTMType *t = [alias type];

      /* Redo the addition to get the checking done.  */
      [self aliasWithName: name type: t qualifiers: q];
    }
}

-(OTMObjectVar *) aliasWithName: (id <TLString>) n
			   type: (id) t
		     qualifiers: (otm_qualifiers) q
{
  OTMObjectVar *var;
  OTMMeta *meta;

  if (postponed_behaviour_supers || postponed_state_supers)
    var = 0;
  else
    {
      meta = [[structure meta] semantics];
      var = [meta searchEntityNamed: n supers: YES class: NO
		  variables: YES constants: NO];
    }
  if (!var)
    // This drops the qualifiers...
    // Fri Jan 24 12:45:47 1997, tiggr@akebono.ics.ele.tue.nl
    postponed_aliases
      = CONS ([CO_OTMAliasAlias aliasWithExtension: self name: n
			     type: t], postponed_aliases);
  else
    {
      OTMType *ot = [var type];

      if (![t matches: ot])
	error (@"inapplicable alias type: %@ does not match %@",
	       type_name (t), type_name (ot));
      else if (!([ot matchesConvertibly: t] == 0
		 || [t matchesConvertibly: ot] == 0))
	error (@"inapplicable alias type: %@ does not fit %@",
	       type_name (t), type_name (ot));
      else
	{
	  if (!aliases)
	    aliases = [TLDictionary dictionary];
	  var = (id) [CO_OTMAlias aliasWithExtension: self variable: var
			       type: t qualifiers: q];
	  [aliases setObject: var forKey: n];
	}
    }

  return var;
}

-(void) checkImplementation
{
  BOOL is_main_ext = structure == [[structure meta] extensionNamed: nil];
  BOOL is_deferred = [[[structure meta] semantics] deferredp];
  id <TLEnumerator> ev = [methods valueEnumerator];
  BOOL have_moaned = NO;
  TLVector *v;

  while ((v = [ev nextObject]))
    {
      int mi, mn = [v length];

      for (mi = 0; mi < mn; mi++)
	{
	  OTMMethod *m = [v _elementAtIndex: mi];

	  if ([m declaredp] && ![m definedp])
	    if ([m deferredp])
	      {
		if (!is_deferred)
		  error_for (m, @"deferred method `%@' in undeferred class",
			     [m methodName]);
	      }
	    else
	      {
		if (!have_moaned)
		  {
		    if (is_main_ext)
		      error (@"incomplete implementation of %@",
			     ltt_meta_name ([structure meta]));
		    else
		      error (@"incomplete implementation of %@",
			     ltt_ext_name (structure));
		    have_moaned = YES;
		  }
		error_for (m, @"method `%@' declared undeferred "
			   @"but not defined", [m methodName]);
	      }
	}
    }

  /* XXX Do many more checks with respect to super classes.  */
}

-(void) compileDeclareExtensionIdentityName
{
  if (!declared_eid)
    {
      declared_eid = YES;
      formac (of, @"extern %@ %@;\n", TO_EXT_ID_TYPE,
	      [structure outputExtensionIdentityName]);
    }
}

-(void) compileObjVarTypeDeclarations
{
  if (!declared_ivar_struct && (vars_order || vars_static))
    {
      id <TLEnumerator> e;

      declared_ivar_struct = YES;

      if (vars_static)
	{
	  OTMObjectVar *v;

	  e = [vars_static valueEnumerator];
	  while ((v = [e nextObject]))
	    {
	      [[v type] compileDeclaration];
	      [v compileDeclaration];
	    }
	}

      if (vars_order)
	{
	  int mi, mn = [vars_order length];

	  for (mi = 0; mi < mn; mi++)
	    {
	      OTMVariable *v = [vars_order _elementAtIndex: mi];
	      [[v type] compileDeclaration];
	    }

	  formac (of, @"struct %@ %@;\n", [structure outputExtensionStructName],
		  [self outputDeclaration]);
	}
    }
}

-(OTMConstant *) constantNamed: (id <TLString>) n
{
  return constants ? [constants objectForKey: n] : nil;
}

-(OTMConstant *) constantWithName: (id <TLString>) n
		       qualifiers: (otm_qualifiers) q
			    value: (OTMExpr *) val
{
  OTMConstant *cst;

  /* XXX Handle qualifiers.  */

  if (![self cvDeclarationOK: n])
    return nil;

  cst = [CO_OTMConstant constantWithExtension: self name: n value: val];
  if (!constants)
    constants = [TLDictionary dictionary];
  [constants setObject: cst forKey: n];

  return cst;
}

-(BOOL) cvDeclarationOK: (id <TLString>) n
{
  OTMMeta *whole = [[structure meta] semantics];
  OTMObjectVar *v;

  /* Allow a subclass to define a new variable with the same name.  */
  v = [whole entityNamed: n variables: YES constants: YES];
  if (v)
    {
      error (@"duplicate definition of %@", n);
      cerror_for (v, @"%@ previously defined in %@", n,
		  ltt_ext_name ([[v extension] structure]));
      return NO;
    }

  /* Do not check for clashes between the State instance and State class.  */
  if (structure != ltt_ext_i_state)
    {
      v = [[[[whole structure] itsClass] semantics]
		    searchEntityNamed: n supers: YES class: NO
		    variables: YES constants: YES];
      if (v)
	{
	  warning (@"definition of %@ shadows previous definition", n);
	  cwarning_for (v, @"%@ previously defined in %@", n,
			ltt_ext_name ([[v extension] structure]));
	}
    }

  return YES;
}

-(void) description: (id <TLMutableStream>) stream
{
  [super description: stream];

  formac (stream, @" %@ %@",
	  [[structure lttName] internal], ltt_meta_name ([structure meta]));
}

-(void) addSuper: (OTMMeta *) m
{
  if (!supers)
    supers = [TLSet new];
  [supers addElement: m];
}

-(id) compileExtensionDescription: (id) s
{
  LTTName *lname = [structure lttName];
  id <TLString> name = lname ? [lname internal] : nil;
  int num_vars = vars_order ? [vars_order length] : 0;
  int num_statics = 0, num_methods = 0;
  OTMMeta *our_meta = [[structure meta] semantics];
  id static_desc = @"0";
  id <TLEnumerator> e;
  OTMObjectVar *v;
  int i, n, first;
  TLVector *vec;

  /* XXX This isn't really a precompile, but it is where the code is...
     Wed Dec 25 22:12:59 1996, tiggr@tricky.es.ele.tue.nl  */
  [[[structure meta] semantics] precompile];

  /* Output the static variables.  */
  if (vars_static)
    {
      e = [vars_static valueEnumerator];
      while ((v = [e nextObject]))
	if (![v isThreadLocal])
	  {
	    OTMType *t = [[v type] actualSelf: our_meta];

	    [t compileDeclaration];
	    s = formac (s, @"%@ %@;\n", [t outputTypeName], [v outputName]);
	  }
	else
	  s = formac (s, @"int _tlo_%@;\n", [v outputName]);

      /* Output information on the static variables.  */
      first = 1;
      e = [vars_static valueEnumerator];
      while ((v = [e nextObject]))
	{
	  id <TLString> vn = [v variableName];
	  BOOL th_local = [v isThreadLocal];

	  if (first)
	    {
	      first = 0;
	      static_desc = formac (nil, @"_static%@",
				    [structure outputExtensionDescriptionName]);
	      s = formac (s, @"struct trtd_static_var %@[] = {", static_desc);
	    }

	  num_statics++;
	  formac (s, @"\n{{{%#, %d}, %@}, &%@%@, %d},", vn, [vn length],
		  [[v type] outputTypeEncoding], th_local ? @"_tlo_" : @"",
		  [v outputName], th_local);
	}
      if (!first)
	formac (s, @"};\n");
    }

  /* Output information on each selector.  */
  e = [methods valueEnumerator];
  while ((vec = [e nextObject]))
    for (i = 0, n = [vec length]; i < n; i++)
      {
        OTMMethod *m = [vec _elementAtIndex: i];
	[[[m selector] semantics] compileDeclaration];
	if (![m deferredp] && [m redo] != OQ_REDECLARE)
	  num_methods++;
      }

  /* Output information on each method.  */
  if (num_methods)
    {
      first = 1;
      e = [methods valueEnumerator];
      while ((vec = [e nextObject]))
	for (i = 0, n = [vec length]; i < n; i++)
	  {
	    OTMMethod *m = [vec _elementAtIndex: i];

	    if (![m deferredp] && [m redo] != OQ_REDECLARE)
	      {
		if (first)
		  {
		    formac (s, @"\nstatic struct trtd_methods trtd_methods_%@ "
			    @"= {%d, {", [[structure lttName] external],
			    num_methods);
		  }

		formac (s, @"%@{(int_imp) %@, (void *) &%@, &%@}",
			first ? @"\n" : @",\n", [m outputName],
			[[[[m extension] structure] meta] metaDefinitionName],
			[[m selector] outputDefinitionName]);
		first = 0;
	      }
	  }
      formac (s, @"\n}};\n");
    }

  /* Declare our extension identity.  */
  if (num_vars)
    s = formac (s, TO_EXT_ID_TYPE @" %@;\n",
		[structure outputExtensionIdentityName]);

  /* Indicate the supers inherited through us.  */
  if (supers)
    {
      OTMMeta *m;

      e = [supers enumerator];
      while ((m = [e nextObject]))
	[m precompile];

      formac (s, @"static struct trt_metas metas_%@ = {{%d, 0}, {",
	      [[structure lttName] external], [supers length]);

      e = [supers enumerator];
      while ((m = [e nextObject]))
	formac (s, @"\n(void *) &%@,", [[m structure] metaDefinitionName]);
      formac (s, @"\n}};\n");
    }

  /* Describe ourselves.  */
  s = formac (s, @"struct trtd_extension %@ = {\n",
	      [structure outputExtensionDescriptionName]);

  /* meta */
  formac (s, @"(void *) &%@", [[structure meta] metaDefinitionName]);

  /* extension_object */
  formac (s, @",\n0");

  /* eid_in_a_global */
  if (num_vars)
    formac (s, @",\n&%@", [structure outputExtensionIdentityName]);
  else
    formac (s, @",\n0");

  /* name */
  if (name)
    formac (s, @",\n{%#, %d}", name, [name length]);
  else
    formac (s, @",\n{0, 0}");

  /* methods */
  if (num_methods)
    formac (s, @",\n&trtd_methods_%@", [[structure lttName] external]);
  else
    formac (s, @",\n0");

  /* supers */
  if (supers)
    formac (s, @",\n&metas_%@", [[structure lttName] external]);
  else
    formac (s, @",\n0");

  /* state_size */
  if (num_vars)
    formac (s, @",\nsizeof (struct %@)",
	    [structure outputExtensionStructName]);
  else
    formac (s, @",\n0");

  /* state_align */
  formac (s, @",\n%d", vars_align);

  /* statics */
  formac (s, @",\n%@", static_desc);

  /* num_statics */
  formac (s, @",\n%d", num_statics);

  /* num_vars */
  formac (s, @",\n%d", num_vars);

  /* vars  */
  formac (s, @",\n{");
  if (num_vars)
    {
      for (i = 0, first = 1, n = [vars_order length]; i < n; i++)
	{
	  OTMVariable *v = [vars_order _elementAtIndex: i];
	  id <TLString> name = [v variableName];

	  if (first)
	    first = NO;
	  else
	    formac (s, @",");
	  formac (s, @"\n{{{%#, %d}, %@}, "
		  @"(char *) &((struct %@ *) 0)->%@ - (char *) 0}",
		  name, [name length], [[v type] outputTypeEncoding],
		  [structure outputExtensionStructName], [v outputName]);
	}
    }
  formac (s, @"\n}};\n");

  return s;
}

-(void) dumpInfo: (id <TLOutputStream>) s
{
  OTMMeta *basis = [[structure meta] semantics];
  id previous_context = output_current_context;
  id <TLString> name = [[structure lttName] internal];
  id <TLEnumerator> e;
  BOOL printed = NO;
  TLVector *vec;

  output_current_context = [structure meta];

  formac (s, @"\n(extension %# %@ (file %#)",
	  name, [[[structure meta] semantics] typeInfo],
	  [[[structure container] lttName] internal]);

  if (vars_order || vars_static)
    {
      int mi, mn = [vars_order length];
      OTMObjectVar *v;

      formac (s, @"\n (variables %d %#", vars_align,
	      vars_order ? [self outputDeclaration] : nil);

      for (mi = 0; mi < mn; mi++)
	{
	  v = [vars_order _elementAtIndex: mi];
	  [v dumpInfo: s];
	}

      e = [vars_static valueEnumerator];
      while ((v = [e nextObject]))
	[v dumpInfo: s];

      [@")" print: s quoted: NO];
    }

  if (constants)
    {
      OTMConstant *c;

      formac (s, @"\n (constants");

      e = [constants valueEnumerator];
      while ((c = [e nextObject]))
	[c dumpInfo: s];

      [@")" print: s quoted: NO];
    }

  [basis dumpInfo: s];

  e = [methods valueEnumerator];
  while ((vec = [e nextObject]))
    {
      int i, n = [vec length];

      if (!printed)
	{
	  formac (s, @"\n (methods");
	  printed = YES;
	}

      for (i = 0; i < n; i++)
	{
	  OTMMethod *m = [vec _elementAtIndex: i];

	  [m dumpInfo: s];
	}
    }

  if (printed)
    formac (s, @")");

  formac (s, @")");

  output_current_context = previous_context;
}

-(void) endCompile
{
  [self compileExtensionDescription: of];
  formac (of, @"/* End of %@.  */\n", ltt_ext_name (structure));
}

-(TLCons *) methodsNamed: (TLVector *) name_parts
		  sender: (OTMMeta *) sender
		    list: (TLCons *) l
{
  TLVector *v = [self methodsNamed: [name_parts _elementAtIndex: 0] create: NO];
  OTMMeta *sender_other = ([sender classp] ? (id) [(OTMClass *) sender instance]
			   : [sender itsClass]);
  LTTMeta *so_str = [sender_other structure];
  LTTMeta *sender_str = [sender structure];
  LTTMeta *str = [structure meta];  
  OTMMeta *sem = [str semantics];

  if (v)
    {
      int mi, mn = [v length];

      for (mi = 0; mi < mn; mi++)
	{
	  OTMMethod *m = [v _elementAtIndex: mi];

	  if ([m namePartsMatch: name_parts])
	    {
	      if (sender)
		{
		  otm_qualifiers q = [m protection];

		  if (q && !(q == OQ_PUBLIC
			     || (q == OQ_PRIVATE
				 && (sender == sem || sender_other == sem))
			     || (q == OQ_PROTECTED
				 && (sender == sem || sender_other == sem
				     || [str isProperSub: sender_str]
				     || [str isProperSub: so_str]))))
		    continue;
		}

	      /* If L already contains a method which matches this method,
		 and the meta of that method is a subclass of our meta, do
		 not add our method, since that obscures the
		 redeclaration.  */
	      {
		OTMMethod *scm;
		TLCons *e = l;

		while (e)
		  {
		    LTTMeta *sesm;

		    DECONS (e, scm, e);

		    sesm = [[[scm extension] structure] meta];
		    if ([scm identical: m inContext: [sesm semantics]]
			&& [str isProperSub: sesm])
		      break;
		  }

		if (!e)
		  l = CONS (m, l);
	      }
	    }
	}
    }

  return l;
}

-(void) resolveIdentifiers: (OTMMeta *) meta
{
  id <TLEnumerator> e = [methods valueEnumerator];
  TLVector *v;
  int mi, mn;

  while ((v = [e nextObject]))
    for (mi = 0, mn = [v length]; mi < mn; mi++)
      {
	OTMMethod *m = [v _elementAtIndex: mi];
	[m resolveIdentifiers: meta];
      }

  if (constants)
    {
      OTMConstant *c;

      e = [constants valueEnumerator];
      while ((c = [e nextObject]))
	[c resolveInContext: meta];
    }
}

-(OTMMethod *) searchSimilarMethod: (OTMMethod *) method
{
  TLVector *v = [self methodsNamed: [method firstNamePart] create: NO];
  int i, n;

  if (v)
    for (i = 0, n = [v length]; i < n; i++)
      {
	OTMMethod *m = [v _elementAtIndex: i];

	if ([(id) [method methodName] equal: [m methodName]]
	    && [method typesMatch: m])
	  return m;
      }

  return nil;
}

-(OTMExpr *) collectConditions: (BOOL) pre_not_post
			   for: (OTMCustomMethod *) method
{
  OTMCustomMethod *match = (id) [self searchSimilarMethod: method];
  OTMExpr *e;

  if (!match)
    return nil;

  e = pre_not_post ? [match precondition] : [match postcondition];
  return !e ? nil : [e conditionCopyFor: method];
}

-(void) gcReference
{
  MARK (structure);
  MARK (methods);
  MARK (vars_keyed);
  MARK (vars_order);
  MARK (vars_static);
  MARK (aliases);
  MARK (constants);
  MARK (supers);

  [super gcReference];
}

-(BOOL) hasVariables
{
  return !!vars_order;
}

-initWithStructure: (LTTExtension *) str
{
  if (![super init])
    return nil;

  structure = str;
  methods = [TLDictionary dictionary];

  return self;
}

-(TLDictionary *) methods
{
  return methods;
}

-(TLVector *) methodsNamed: (id <TLString>) n create: (BOOL) create
{
  TLVector *v = [methods objectForKey: n];

  if (!v && create)
    {
      v = [CO_TLVector vector];
      [methods setObject: v forKey: n];
    }

  return v;
}

-(id <TLString>) outputDeclaration
{
  TLMutableString *s = nil;

  /* The output declaration can't be cached, because of the BT_RECV types
     and changing values of the OUTPUT_CURRENT_CONTEXT.  */
  if (vars_order)
    {
      int mi, mn;

      s = formac (s, @"{");
      for (mi = 0, mn = [vars_order length]; mi < mn; mi++)
	{
	  OTMObjectVar *v = [vars_order _elementAtIndex: mi];

	  [@" " print: s quoted: NO];
	  [v outputDeclaration: s];
	  [@";" print: s quoted: NO];
	}

      /* Output the class information needed by the runtime.  */
      if ([structure meta] == ltt_class_state
	  && structure == [[structure meta] extensionNamed: nil])
	formac (s, @"%@", TRO_META_COMMON_INFORMATION);

      formac (s, @" }");
    }

  return s;
}

-(id <TLString>) outputName
{
  return [structure outputName];
}

-(id <TLString>) outputOffsetFrom: (LTTMeta *) m
{
  BOOL self_class_p = [[structure meta] classp];
  BOOL current_class_p = [m classp];
  id s = formac (nil, @"(void *) ");

  if (!current_class_p && self_class_p)
    {
      id <TLString> our_isa = formac (nil, @"((tom_object) %@)->%@",
				      TO_NAME_SELF, TO_NAME_ISA);

      if (flag_check_extension_address)
	s = formac (s, @"trt_ext_address ((void *) %@, %@)", our_isa,
		    [structure outputExtensionIdentityName]);
      else
 	s = formac (s, @"((char *) %@ + %@->%@->info.%@->%@[%@])",
		    our_isa, our_isa, TO_NAME_ISA,
		    TO_NAME_EOT, TO_NAME_EOT_OFFSET,
		    [structure outputExtensionIdentityName]);
    }
  else if (flag_check_extension_address)
    s = formac (s, @"trt_ext_address ((void *) %@, %@)", TO_NAME_SELF,
		[structure outputExtensionIdentityName]);
  else
    s = formac (s, @"((char *) %@ + ((tom_object) %@)->%@->info.%@->%@[%@])",
		TO_NAME_SELF, TO_NAME_SELF, TO_NAME_ISA, TO_NAME_EOT,
		TO_NAME_EOT_OFFSET, [structure outputExtensionIdentityName]);
  return s;
}

-(void) startCompile
{
  formac (of, @"\n/* Start of %@.  */\n", ltt_ext_name (structure));
}

-(LTTExtension *) structure
{
  return structure;
}

-(OTMObjectVar *) variableNamed: (id <TLString>) n
{
  OTMObjectVar *v = vars_order ? [vars_keyed objectForKey: n] : nil;

  if (!v && vars_static)
    v = [vars_static objectForKey: n];
  if (!v && aliases)
    v = [aliases objectForKey: n];

  return v;
}

-(OTMObjectVar *) variableWithName: (id <TLString>) n
			      type: (id) t
			qualifiers: (otm_qualifiers) q
{
  OTMObjectVar *v;
  int i;

  /* XXX Handle qualifiers.  */

  if (![self cvDeclarationOK: n])
    return nil;

  i = [t minimumAlignment];
  if (i > vars_align)
    vars_align = i;

  v = [CO_OTMObjectVar variableWithExtension: self name: n type: t
		    qualifiers: q];

  if (Q_STATIC (q))
    {
      if (!vars_static)
	vars_static = [TLDictionary dictionary];

      [vars_static setObject: v forKey: n];
    }
  else
    {
      if (!vars_order)
	{
	  vars_keyed = [TLDictionary dictionary];
	  vars_order = [CO_TLVector vector];
	  [[[structure meta] semantics] setStatep];
	}

      [vars_keyed setObject: v forKey: n];
      [vars_order addElement: v];
    }

  return v;
}

@end
