/* Support different runtimes.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: support.m,v 1.1 1998/01/08 16:12:23 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLPatchedRoots.h"
#import "tl/TLFILEStream.h"
#import "tl/TLMutableString.h"
#import "tl/TLNumber.h"
#import "tl/TLSymbol.h"
#import "tl/TLVector.h"
#import "config.h"
#import <stdlib.h>
#import <ctype.h>
#if HAVE_STRINGS_H
#import <strings.h>
#endif

/* NeXT!  */
#if !HAVE_STRDUP
char *
strdup (const char *s)
{
  return (strcpy (xmalloc (1 + strlen (s)), s));
} /* strdup */
#endif

/* SunOS!  */
#if !HAVE_STRERROR
#import <errno.h>

#if NEED_SYS_ERRLIST
extern char *sys_errlist[];
#endif

char *
strerror (int errno)
{
  return (sys_errlist[errno]);
} /* strerror */
#endif

/******************** memory allocation ********************/

void
xfree (void *p)
{
  if (p)
    {
      QUIT_DISABLE;
      free (p);
      QUIT_ENABLE;
    }
} /* xfree */

void *
xcalloc (unsigned int n, unsigned int m)
{
  unsigned int l = n * m;
  void *p = xmalloc (l);

  if (p)
    bzero ((void *) p, l);

  return (p);
} /* xcalloc */

void *
xmalloc (unsigned int n)
{
  void *p;

  if (n)
    {
      QUIT_DISABLE;
      p = malloc (n);
      QUIT_ENABLE;

      if (!p)
	abort ();
    }
  else
    p = NULL;

  return (p);
} /* xmalloc */

void *
xrealloc (void *p, unsigned int n)
{
  if (n)
    {
      /* Call straight malloc for a NULL P to please braindead
	 implementations (for which malloc (0) != 0).  */
      QUIT_DISABLE;
      p = p ? realloc (p, n) : malloc (n);
      QUIT_ENABLE;
      if (!p)
	abort ();
    }
  else if (p)
    {
      QUIT_DISABLE;
      free (p);
      QUIT_ENABLE;
      p = NULL;
    }
  return (p);
} /* xrealloc */

/******************** selector information ********************/

#if GNU_RUNTIME
#import <objc/encoding.h>

int
sel_num_args (id rcv, SEL sel)
{
  const char *t = sel_get_type (sel);
  int type_count = 0;

  if (!t)
    return (-1);
  while (*t)
    {
      if (*t == '+')
	t++;
      while (*t >= '0' && *t <= '9')
	t++;
      if (!*t)
	break;
      t = objc_skip_typespec (t);
      type_count++;
    }
  return (type_count - 3);
} /* sel_num_args */

BOOL
sel_objects_only (id rcv, SEL sel)
{
  const char *t = sel_get_type (sel);
  int type_count = 0;

  if (!t)
    return (NO);
  while (*t)
    {
      if (*t == '+')
	t++;
      while (*t >= '0' && *t <= '9')
	t++;
      t = objc_skip_type_qualifiers (t);
      if (!*t)
	break;
      if ((type_count == 2 && *t != _C_SEL)
	  || (type_count != 2 && *t != _C_ID))
	return (NO);
      type_count++;
      t = objc_skip_typespec (t);
    }
  return (YES);
} /* sel_objects_only */

#endif

#if NEXT_RUNTIME

int
sel_num_args (id rcv, SEL sel)
{
  struct objc_method *m = class_getInstanceMethod (rcv->isa, sel);
  int type_count = 0;
  const char *t;

  if (!m)
    return (-1);
  t = m->method_types;
  while (*t)
    {
      if (*t == '+')
	t++;
      while (*t >= '0' && *t <= '9')
	t++;
      if (!*t)
	break;
      type_count++;
    }
  return (type_count - 3);
} /* sel_num_args */

BOOL
sel_objects_only (id rcv, SEL sel)
{
  struct objc_method *m = class_getInstanceMethod (rcv->isa, sel);
  int type_count = 0;
  const char *t;

  if (!m)
    return (NO);
  t = m->method_types;
  while (*t)
    {
      if (*t == '+')
	t++;
      while (*t >= '0' && *t <= '9')
	t++;
      if (!*t)
	break;
      if ((type_count == 2 && *t != _C_SEL)
	  || (type_count != 2 && *t != _C_ID))
	return (NO);
      type_count++;
    }
  return (YES);
} /* sel_objects_only */

#endif

/******************** miscellaneous ********************/

id
eval (id a)
{
  return (EVAL (a));
} /* eval */

int
memcasecmp (const char *s1, const char *s2, int n)
{
  while (n)
    {
      char c1 = *s1++;
      char c2 = *s2++;

      n--;

      if (c1 == c2)
	continue;

      if (c1 >= 'a' && c1 <= 'z')
	c1 = c1 - 'a' + 'A';
      if (c2 >= 'a' && c2 <= 'z')
	c2 = c2 - 'a' + 'A';

      if (c1 == c2)
	continue;

      return (s1[-1] > s2[-1] ? 1 : -1);
    }
  return (0);
} /* memcasecmp */

void
print (id object, id <TLOutputStream> stream, int quoted)
{
  if (!object)
    {
      object = @"nil";
      quoted = 0;
    }
  [object print: stream quoted: quoted];
} /* print */

void
print_list_element (id object, id <TLOutputStream> stream, int quoted)
{
  if (!object)
    object = @"nil";
  [object printListElement: stream quoted: quoted];
} /* print_list_element */

void
print_list_elements (TLCons *c, id <TLOutputStream> stream, int quoted)
{
  while (c)
    {
      id o;
      [c car: &o cdr: &c];
      [stream writeByte: ' '];
      print (o, stream, quoted);
    }
} /* print_list_elements */

/* XXX */
#if 0
void
princ (id a)
{
  print (a, NO);
} /* princ */

void
prin1 (id a)
{
  print (a, YES);
} /* prin1 */
#endif

const char *
d (id a)
{
  return description (a);
}

const char *
description (id a)
{
  return ([formac (nil, @"%#", a) cString]);
}

void
range_intersect (int s1, int l1, int *s2, int *l2)
{
  /* The first range must be a defined range.  */
  if (l1 < 0)
    abort ();

  if (*l2 == -1)
    *l2 = s1 + l1 - *s2;
  else if (*l2 < 0)
    /* Negative but not `-1' length is an error.  */
    abort ();

  if (*s2 >= s1 + l1)
    {
      *s2 = s1 + l1;
      *l2 = 0;
    }
  else if (*s2 + *l2 <= s1)
    {
      *s2 = s1;
      *l2 = 0;
    }
  else
    {
      if (*s2 < s1)
	{
	  *l2 += *s2 - s1;
	  *s2 = s1;
	}
      if (*s2 + *l2 > s1 + l1)
	*l2 = s1 + l1 - *s2;
    }
} /* range_intersect */

#define OUTPUT_SOMETHING(TYPE, NAME)  \
static int								\
output_ ## NAME (id <TLOutputStream> stream, char fmt, TYPE v)		\
{									\
  int upc = fmt == 'X';							\
  int base = fmt == 'o' ? 8 : (fmt == 'x' || upc) ? 16 : 10;		\
  char *digits = (upc ? ("FEDCBA9876543210123456789ABCDEF" + 15)	\
		  : ("fedcba9876543210123456789abcdef" + 15));		\
  int neg = v < 0, l;							\
  char buf[30];								\
									\
  for (l = sizeof (buf) - 1; l > 0; l--)				\
    {									\
      buf[l] = digits[v % base];					\
      v /= base;							\
      if (!v)								\
	break;								\
    }									\
  if (!l)								\
    abort ();								\
  if (neg && base == 10)						\
    buf[--l] = '-';							\
  return ([stream writeBytes: sizeof (buf) - l fromBuffer: buf + l]);	\
} /* output_long */

OUTPUT_SOMETHING (long, long)
OUTPUT_SOMETHING (long long, long_long)
/* Expect two warnings about comparison of the unsigned value `<' 0.  */
OUTPUT_SOMETHING (unsigned long, unsigned_long)
OUTPUT_SOMETHING (unsigned long long, unsigned_long_long)

TLMutableString *
llvformac (id <TLStream, TLOutputStream> stream, id <TLString> format,
	   va_list ap, TLVector *av)
{
  int arg_width, arg_dot_width, arg_neg, arg_long, arg_long_long;
  int fmt_len = [format length], j, i, idx = 0;
  int num_args = av ? [av length] : 0;
  const char *fmt = [format cString];
  id o;

  static id tlmutable_string_class;
  if (!tlmutable_string_class)
    tlmutable_string_class = [CO_TLMutableString self];

  if (!stream)
    stream = [tlmutable_string_class mutableString];
  else if (stream == (id) Qt)
    stream = V_stdout_;

  /* Looping with I instead of modifying both FMT and FMT_LEN is less
     error-prone.  */
  for (i = 0; i < fmt_len; i++)
    switch (fmt[i])
      {
      case '%':
	arg_long = arg_long_long = arg_width = arg_neg = 0;
	arg_dot_width = -1;
	if (++i < fmt_len && fmt[i] == '-')
	  arg_neg = i++;
	while (i < fmt_len && fmt[i] >= '0' && fmt[i] <= '9')
	  arg_width = 10 * arg_width + fmt[i++] - '0';
	if (i < fmt_len && fmt[i] == '.')
	  {
	    arg_dot_width = 0;
	    while (++i < fmt_len && fmt[i] >= '0' && fmt[i] <= '9')
	      arg_dot_width = 10 * arg_dot_width + fmt[i] - '0';
	  }
	if (i < fmt_len && fmt[i] == 'l')
	  arg_long = i++;
	else if (fmt[i] == 'q')
	  arg_long_long = i++;
	if (i >= fmt_len)
	  [TLObject error: "trailing `%' in format"];
	if (fmt[i] == '%')
	  {
	    /* XXX Should take the modifiers into account.  */
	    [stream writeByte: '%'];
	    break;
	  }
	if (arg_neg)
	  arg_width = -arg_width;
	switch (!av)
	  {
	  /* TLVector with arguments.  */
	  case 0:
	    if (idx >= num_args)
	      [TLObject error: "not enough arguments to formac (%#)", format];
	    o = [av _elementAtIndex: idx++];
	    switch (fmt[i])
	      {
	      case '#':
		print (o, stream, YES);
		break;

	      case '@':
		print (o, stream, NO);
		break;

	      case 'c':
		[stream writeByte: [o charValue]];
		break;

	      case 'd':
		if (arg_long_long)
		  output_long_long (stream, fmt[i], [o longLongValue]);
		else
		  output_long (stream, fmt[i], (arg_long ? [o longValue]
						: [o intValue]));
		break;

	      case 'o':
	      case 'u':
	      case 'x':
	      case 'X':
		if (arg_long_long)
		  output_unsigned_long_long (stream, fmt[i],
					     [o unsignedLongLongValue]);
		else
		  output_unsigned_long (stream, fmt[i],
					(arg_long ? [o unsignedLongValue]
					 : [o unsignedIntValue]));
		break;

	      default:
		[TLObject error: "bad format specifier: `%c' in format %#",
		 fmt[i], format];
		break;
	      }
	    break;

	  /* Use Plain Old Stdarg.  */
	  default:
	    switch (fmt[i])
	      {
	      case '#':
		o = va_arg (ap, id);
		print (o, stream, YES);
		break;

	      case '@':
		o = va_arg (ap, id);
		print (o, stream, NO);
		break;

	      case 'c':
		[stream writeByte: va_arg (ap, int)];
		break;

	      case 'd':
		if (arg_long_long)
		  output_long_long (stream, fmt[i], va_arg (ap, long long));
		else
		  output_long (stream, fmt[i], (arg_long ? va_arg (ap, long)
						: va_arg (ap, int)));
		break;

	      case 'o':
	      case 'u':
	      case 'x':
	      case 'X':
		if (arg_long_long)
		  output_unsigned_long_long (stream, fmt[i],
					     va_arg (ap, long long));
		else
		  output_unsigned_long (stream, fmt[i],
					(arg_long ? va_arg (ap, long)
					 : va_arg (ap, int)));
		break;

	      case 'e':
	        {
		  char buf[512];

		  sprintf (buf, "%*e", arg_width, va_arg (ap, double));
		  [stream writeBytes: strlen (buf) fromBuffer: buf];
		}
		break;

	      case 'f':
	        {
		  char buf[512];

		  if (arg_dot_width != -1)
		    sprintf (buf, "%*.*f", arg_width, arg_dot_width,
			     va_arg (ap, double));
		  else
		    sprintf (buf, "%*f", arg_width, va_arg (ap, double));
		  [stream writeBytes: strlen (buf) fromBuffer: buf];
		}
		break;

	      case 'g':
	        {
		  char buf[512];

		  sprintf (buf, "%*g", arg_width, va_arg (ap, double));
		  [stream writeBytes: strlen (buf) fromBuffer: buf];
		}
		break;

	      case 'p':
	        {
		  PTR_INT_TYPE v = (long) va_arg (ap, void *);
		  int sh;

		  for (sh = sizeof (v) * 8 - 4; sh >= 0; sh -= 4)
		    [stream writeByte: "0123456789abcdef"[(v >> sh) & 0xf]];
		  break;
		}

	      case 's':
	        {
		  char *s = va_arg (ap, char *);
		  [stream writeBytes: strlen (s) fromBuffer: s];
		  break;
		}

	      default:
		[TLObject error: "bad format specifier: `%c' in format %#",
		 fmt[i], format];
		break;
	      }
	    break;
	  }
	break;

      default:
	/* Skip until the next special character.  */
	for (j = i + 1; j < fmt_len && fmt[j] != '%' && fmt[j] != '\\'; j++);
	[stream writeBytes: j - i fromBuffer: fmt + i];
	i = j - 1;
	break;
      }
  // [stream flushOutput];

  if (av && idx != num_args)
    {
      /* XXX Issue a warning?  */
    }

  return (stream);
}

TLMutableString *
vformac (id stream, id fmt, va_list ap)
{
  return (llvformac (stream, fmt, ap, 0));
}

TLMutableString *
formac (id stream, id fmt, ...)
{
  va_list ap;
  id a;

  va_start (ap, fmt);
  a = llvformac (stream, fmt, ap, 0);
  va_end (ap);

  return (a);
}

/******************** runtime stuff (from gcc) ********************/

#ifdef hpux
#define USG
#endif
#ifdef USG
#undef FLOAT
#include <sys/param.h>
/* This is for hpux.  It is a real screw.  They should change hpux.  */
#undef FLOAT
#include <sys/times.h>
#include <time.h>   /* Correct for hpux at least.  Is it good on other USG?  */
#undef FFS  /* Some systems define this in param.h.  */
#else
#ifndef VMS
#include <sys/time.h>
#include <sys/resource.h>
#endif
#endif

double
get_run_time (void)
{
#ifdef USG
  struct tms tms;
#else
#ifndef VMS
  struct rusage rusage;
#else /* VMS */
  struct
    {
      int proc_user_time;
      int proc_system_time;
      int child_user_time;
      int child_system_time;
    } vms_times;
#endif
#endif

#ifdef USG
  times (&tms);
  return ((double) (tms.tms_utime + tms.tms_stime) / HZ);
#else
#ifndef VMS
  getrusage (0, &rusage);
  return ((double) rusage.ru_utime.tv_sec + rusage.ru_utime.tv_usec / 1e6
	  + (double) rusage.ru_stime.tv_sec + rusage.ru_stime.tv_usec / 1e6);
#else /* VMS */
  times (&vms_times);
  return (vms_times.proc_user_time + vms_times.proc_system_time) / 100.0;
#endif
#endif
} /* get_run_time */
