/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: ltt.m,v 1.46 1998/04/22 12:35:39 tiggr Exp $  */

#import "ltt.h"
#import <sys/types.h>

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen ((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

id CO_LTTClass, CO_LTTEntity, CO_LTTExtension, CO_LTTFile;
id CO_LTTFileItem, CO_LTTInstance, CO_LTTMeta, CO_LTTName;
id CO_LTTSelector, CO_LTTSelArgTypes, CO_LTTStringCST;
id CO_LTTTop, CO_LTTUnit, CO_LTTVariable;
id CO_TLVector, CO_TLString, CO_TLNumber;

/* The name under which this program was invoked.  */
id <TLString> prog;

/* The path to search for files.  */
TLVector *load_path;

/* Global variables pointing to important things.  */
LTTInstance *ltt_instance_any, *ltt_instance_all, *ltt_instance_state;
LTTClass *ltt_class_any, *ltt_class_all, *ltt_class_state;
LTTExtension *ltt_ext_c_state, *ltt_ext_i_state;
LTTUnit *ltt_unit_tom;

/* The units for the builtin information and that of missing information
   (due to errors of the user).  And the file `containing' the builtin
   information.  */
LTTUnit *ltt_builtin_unit, *ltt_missing_unit;
LTTFile *ltt_builtin_file, *ltt_missing_file;

/* The current unit.  Used in selector name mangling.  */
LTTUnit *ltt_current_unit;

/* Value used to mark objects during a search.  This value should be
   modified for each search performed.  */
int search_mark;

/* Return the filename without extension.  */
TLString *
ltt_filename_without_extension (TLString *filename)
{
  TLRange *re = [filename rangeOfString: @"." options: TLSEARCH_BACKWARD];

  return (!re ? filename
	  : [filename stringWithRange: [TLRange rangeWithStart: 0
						length: [re _start]]]);
}

/* This function is only here because GCC 2.7.1 on hppa can't stand the
   expression it contains in the file LTTExtension.m.  Probably because
   the method being invoked was declared for a category of LTTExtension,
   or something weird like that.  */
id
semantics_for_extension (LTTExtension *e)
{
  return [CO_LTTExtension semanticsForExtension: e];
}

char *
get_arg (int *i, int offset, char **argv, int argc)
{
  if (argv[*i][offset])
    return argv[*i] + offset;

  /* An option argument separate from the option should not start with a
     `-' (since that would make it the following option; not an argument).  */
  if (*i + 1 < argc && argv[*i + 1][0] != '-')
    return argv[++(*i)];

  error (@"option needs argument: `%s'", argv[*i]);
  return NULL;
}

TLString *
basename_of_unit_file (id <TLString> dir)
{
  DIR *di = opendir ([dir cString]);
  TLString *found = nil;
  struct dirent *d;
  int wrong = 0;

  if (!di)
    return nil;

  while ((d = readdir (di)))
    {
      int d_namlen = NAMLEN (d);

      if (d_namlen > 2
	  && d->d_name[d_namlen - 1] == 'u'
	  && d->d_name[d_namlen - 2] == '.')
	{
	  if (found)
	    {
	      if (!wrong)
		error (@"multiple unit files in directory `%@':", dir);
	      cerror (@"%@found `%@'", wrong ? @"also " : @"", found);
	      wrong++;
	    }
	  found = [TLString stringWithCString: d->d_name];
	}
    }

  closedir (di);

  if (wrong)
    cerror (@"also found `%@'", found);
  else if (!found)
    error (@"no unit file in directory `%@'", dir);

  return (wrong ? nil
	  : [found stringWithRange:
		   [TLRange rangeWithStart: 0 length: [found length] - 2]]);
}

id <TLString>
locate_file_along_path (id <TLString> name, TLVector *path,
			id <TLString> name_subdir, id <TLString> gen_subdir)
{
  int i, n = [path length];
  id <TLString> res;

  if ([[name string] _fileExistsP])
    return name;

  /* Find it in the exact directories in the search PATH.  */
  for (i = 0; i < n; i++)
    {
      TLString *s = [TLString stringWithFormat: @"%@/%@",
			      [path _elementAtIndex: i], name];
      if ([s _fileExistsP])
	return s;
    }
  
  res = nil;
  if (gen_subdir)
    res = locate_file_along_path ([TLString stringWithFormat: @"%@/%@",
					    gen_subdir, name],
				  path, name_subdir, nil);
  if (!res && name_subdir)
    res = locate_file_along_path ([TLString stringWithFormat: @"%@/%@",
					    name_subdir, name],
				  path, nil, nil);

  return res;
}

void
reset_load_path (void)
{
  [load_path gcUnlock];
  load_path = [TLVector vector];
  [load_path gcLock];
}

void
add_to_load_path (TLVector *vector)
{
  id <TLEnumerator> e = [vector enumerator];
  TLString *s;

  while ((s = [e nextObject]))
    [load_path addElement: s];
}

void
ltt_init (int argc, char **argv)
{
  char *s = strrchr (argv[0], '/');
  static BOOL init = NO;

  if (init)
    return;
  init = YES;

  debug_gc = -1;

  tl_init ();

  CO_LTTClass = [LTTClass self];
  CO_LTTEntity = [LTTEntity self];
  CO_LTTExtension = [LTTExtension self];
  CO_LTTFile = [LTTFile self];
  CO_LTTFileItem = [LTTFileItem self];
  CO_LTTInstance = [LTTInstance self];
  CO_LTTMeta = [LTTMeta self];
  CO_LTTName = [LTTName self];
  CO_LTTSelector = [LTTSelector self];
  CO_LTTSelArgTypes = [LTTSelArgTypes self];
  CO_LTTStringCST = [LTTStringCST self];
  CO_LTTTop = [LTTTop self];
  CO_LTTUnit = [LTTUnit self];
  CO_LTTVariable = [LTTVariable self];
  CO_TLVector = [TLVector self];
  CO_TLString = [TLString self];
  CO_TLNumber = [TLNumber self];

  prog = [TLString stringWithCString: s ? s + 1 : argv[0]];
  [prog gcLock];

  setvbuf (stderr, NULL, _IOLBF, BUFSIZ);

  load_path = [TLVector vectorWithElements: 1, @"."];
  [load_path gcLock];

  ltt_builtin_unit = [CO_LTTUnit unitWithName: (id) TOM_UNIT_NAME_BUILTIN
			      directory: (id) @"<builtin>"];
  ltt_builtin_file = [CO_LTTFile fileWithName: (id) TOM_FILE_NAME_BUILTIN
			      unit: ltt_builtin_unit];
  [ltt_builtin_file setLoadedInfo];
  [ltt_builtin_file setLoadedInterface];
  [ltt_builtin_file setLoadedImplementation];

  ltt_missing_unit = [CO_LTTUnit unitWithName: (id) TOM_UNIT_NAME_MISSING
			      directory: (id) @"<missing>"];
  ltt_missing_file = [CO_LTTFile fileWithName: (id) TOM_FILE_NAME_MISSING
			      unit: ltt_missing_unit];
  [ltt_builtin_file setLoadedInfo];
  [ltt_missing_file setLoadedInterface];
  [ltt_missing_file setLoadedImplementation];

  ltt_instance_any = [CO_LTTMeta instanceAndClassWithName: TOM_CLASS_NAME_ANY
			      unit: ltt_builtin_unit];
  [CO_LTTExtension extensionWithName: nil file: ltt_builtin_file
		meta: ltt_instance_any];
  [ltt_builtin_unit addInstance: ltt_instance_any];
  ltt_class_any = [ltt_instance_any itsClass];
  [CO_LTTExtension extensionWithName: nil file: ltt_builtin_file
		meta: ltt_class_any];

  ltt_instance_all = [CO_LTTMeta instanceAndClassWithName: TOM_CLASS_NAME_ALL
			      unit: ltt_builtin_unit];
  [CO_LTTExtension extensionWithName: nil file: ltt_builtin_file
		meta: ltt_instance_all];
  [ltt_builtin_unit addInstance: ltt_instance_all];
  ltt_class_all = [ltt_instance_all itsClass];
  [CO_LTTExtension extensionWithName: nil file: ltt_builtin_file
		meta: ltt_class_all];
}

void
ltt_finish_init (void)
{
  static BOOL init = NO;

  if (init)
    return;
  init = YES;

  ltt_unit_tom = ltt_find_unit (TOM_UNIT_NAME_TOM);
  if (!ltt_unit_tom)
    error (@"unit `%@' not found", TOM_UNIT_NAME_TOM);
  else
    {
      ltt_instance_state = [ltt_unit_tom instanceNamed:
					 (id) TOM_CLASS_NAME_STATE];
      if (!ltt_instance_state)
	error (@"`%@.%@' not found", TOM_UNIT_NAME_TOM, TOM_CLASS_NAME_STATE);
      else
	{
	  id <TLEnumerator> e_unit = [ltt_units valueEnumerator];
	  LTTUnit *unit;

	  ltt_class_state = [ltt_instance_state itsClass];

	  /* Update all class objects to short-circuit to the state meta
             class behaviour.  */
	  while ((unit = [e_unit nextObject]))
	    {
	      id <TLEnumerator> e_inst = [unit instances];
	      LTTInstance *inst;

	      while ((inst = [e_inst nextObject]))
		{
		  LTTClass *cls = [inst itsClass];

		  if ([cls itsClass])
		    ABORT ();
		  [cls setClass: ltt_class_state];
		}
	    }

	  ltt_ext_c_state = [ltt_class_state extensionNamed: nil];
	  ltt_ext_i_state = [ltt_instance_state extensionNamed: nil];

	  /* All class objects must have the State class as a state super,
             as it is the (main extension of) the State class which outputs
             the information needed by the runtime.  */
	  [ltt_class_any addStateSuper: ltt_class_state];
	  [ltt_class_all addStateSuper: ltt_class_state];
	}
    }
}

id <TLString>
quote (id <TLString> s)
{
  const char *from = [s cString];
  int i, n = [s length];
  char *to = xmalloc (n);
  char c;

  for (i = 0; i < n; i++)
    {
      switch (from[i])
	{
	case '.':
	case '/':
	case '(':
	case ')':
	case '-':
	case ':':
	  c = '_';
	  break;

	default:
	  c = from[i];
	  break;
	}
      to[i] = c;
    }

  return [TLString stringWithCStringNoCopy: to length: n];
}

LTTMeta *
ltt_get_meta (TLString *unit_name, TLString *meta_name, BOOL classp)
{
  LTTUnit *u = [CO_LTTUnit unitNamed: unit_name];
  LTTMeta *m;

  if (!u)
    {
      error (@"no unit `%@'", unit_name);
      return nil;
    }

  m = [u instanceNamed: meta_name];
  if (!m)
    {
      error (@"no class `%@.%@'", unit_name, meta_name);
      return nil;
    }

  return classp ? (id) [m itsClass] : m;
}

LTTInstance *
ltt_instance_in_unit (id <TLString> unit_name, LTTUnit *current,
		      TLVector *instances)
{
  LTTUnit *u = unit_name ? [CO_LTTUnit unitNamed: unit_name] : current;
  LTTInstance *first, *lm = nil;
  id <TLString> s;
  int n;

  if (!u)
    error (@"no known unit: `%@'", unit_name);

  if (!instances)
    return nil;

  n = [instances length];
  if (!n)
    return nil;

  first = [instances _elementAtIndex: 0];
  s = [[first lttName] internal];

  if (n != 1)
    lm = u ? unit_name ? [u instanceNamed: s] : [u findInstanceNamed: s] : nil;
  else if (!unit_name || [first unit] == u)
    lm = first;

  return lm;
}

id <TLString>
ltt_meta_name (LTTMeta *m)
{
  return (id) formac (nil, @"%@ %@.%@", [m classp] ? @"class" : @"instance",
		      [[[m unit] lttName] internal], [[m lttName] internal]);
}

id <TLString>
ltt_ext_name (LTTExtension *e)
{
  id <TLString> name = [[e lttName] internal];
  id <TLString> mn = ltt_meta_name ([e meta]);

  return (id) (name ? formac (nil, @"%@(%@)", mn, name)
	       : formac (nil, @"main extension of %@", mn));
}

id
open_output_file (id <TLString> s)
{
  const char *c = [s cString];
  id stream = nil;

  if (!unlink (c) || ERRNO == ENOENT)
     stream = [TLFILEStream mutableStreamWithFileNamed: s];
  if (stream)
    [(id) stream gcLock];
  else
    error (@"%@: %s", s, ERRMSG);

  return stream;
}

id <TLString>
ltt_lex_string_cst (const char *yytext, int yyleng, char quote, int *nnl)
{
  TLMutableString *str = [TLMutableString mutableString];
  const char *s;
  char c;
  int i;

  *nnl = 0;
  for (s = yytext + 1, i = 1; i < yyleng - 1; s++, i++)
    {
      if (*s != '\\')
	{
	  c = *s;
	  if (c == '\n')
	    (*nnl)++;
	}
      else
	{
	  i++;
	  if (*++s == quote)
	    c = quote;
	  else switch (*s)
	    {
	    case 'f': c = '\f'; break;
	    case 'n': c = '\n'; break;
	    case 'r': c = '\r'; break;
	    case 't': c = '\t'; break;
	    case '\\': c = '\\'; break;
	    case '\n':
	      (*nnl)++;
	      continue;
	    default:
	      warning (@"unknown escape: `%c'", *s);
	      [str appendCChar: '\\'];
	      c = *s;
	      break;
	    }
	}
      [str appendCChar: c];
    }
  return str;
}
