/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: LTTMeta.m,v 1.36 1998/01/22 17:27:11 tiggr Exp $  */

#define LTTMETA_DECLARE_PRIVATE_METHODS
#import "ltt.h"

TLVector *ltt_metas;

@implementation LTTMeta

+initialize
{
  if (!ltt_metas)
    {
      ltt_metas = [TLVector vector];
      [ltt_metas gcLock];
    }

  return self;
}

+(LTTInstance *) instanceAndClassWithName: (id <TLString>) n
				     unit: (LTTUnit *) u
{
  LTTInstance *i = [CO_LTTInstance metaWithName: n unit: u];
  LTTClass *c = [CO_LTTClass metaWithName: n unit: u];

  [c setClass: ltt_class_state];
  [c setInstance: i];
  [i setClass: c];

  [c setSemantics: [self semanticsForClass: c]];
  [i setSemantics: [self semanticsForInstance: i]];

  [ltt_instance_all addBehaviourSuper: c];
  [ltt_instance_all addBehaviourSuper: i];

  return i;
}

+(id <TLEnumerator>) metas
{
  return [ltt_metas enumerator];
}

+(id) metaWithName: (id <TLString>) n unit: (LTTUnit *) u;
{
  LTTMeta *m = [(LTTMeta *) [self gcAlloc] initWithName: n unit: u];

  if (m)
    [ltt_metas addElement: m];

  return m;
}

+(int) numMetas
{
  return [ltt_metas length];
}

-(void) addBehaviourSub: (LTTMeta *) m
{
  [behaviour_subs addElement: m];
}

-(void) addBehaviourSuper: (LTTMeta *) m
{
  if (![m isDirectSub: self])
    {
      [behaviour_supers addElement: m];
      [m addBehaviourSub: self];
    }
}

-(void) addExtension: (LTTExtension *) e
{
  TLString *n = [[e lttName] internal];
  LTTExtension *previous = n ? [extensions objectForKey: n] : main;

  if (previous)
    {
      id <TLString> ext = ltt_ext_name (e);

      error (@"redefinition of %@", ext);
      cerror_for ([previous semantics],
		  @"previous definition of %@ was here", ext);
    }
  else
    {
      if (n)
	{
	  [extensions setObject: e forKey: n];
	  [all_extensions addElement: e];
	}
      else
	{
	  [all_extensions _replaceElementAtIndex: 0 by: e];
	  ASGN_IVAR (main, e);
	}
    }
}

-(void) addStateSub: (LTTMeta *) m
{
  [state_subs addElement: m];
}

-(void) addStateSuper: (LTTMeta *) m
{
  if (![m isDirectStateSub: self])
    {
      [state_supers addElement: m];
      [m addStateSub: self];
    }
}

-(TLVector *) behaviourSubs
{
  return behaviour_subs;
}

-(TLVector *) behaviourSupers
{
  return behaviour_supers;
}

-(BOOL) classp
{
  return NO;
}

-(id <TLString>) definitionName
{
  return (id) formac (nil, @"%@%@", TO_META_DEF_PREFIX,
		      [self qualifiedExternalName]);
}

-(TLCons *) directSuperForProperSuper: (LTTMeta *) m
{
  TLCons *l = nil;
  LTTMeta *s;
  int i, n;

  for (i = 0, n = [state_supers length]; i < n; i++)
    {
      s = [state_supers _elementAtIndex: i];
      if (m == s || [m isProperSub: s])
	l = CONS (s, l);
    }

  for (i = 0, n = [behaviour_supers length]; i < n; i++)
    {
      s = [behaviour_supers _elementAtIndex: i];
      if (m == s || [m isProperSub: s])
	l = CONS (s, l);
    }

  return l;
}

-(LTTExtension *) extensionNamed: (TLString *) n
{
  return n ? [extensions objectForKey: n] : main;
}

-(TLVector *) extensions
{
  return all_extensions;
}

-(id <TLString>) extensionsDescriptionName
{
  return (id) formac (nil, @"%@%@", TO_META_MED_PREFIX,
		      [self qualifiedExternalName]);
}

-(void) gcReference
{
  MARK (unit);
  MARK (class);
  MARK (main);
  MARK (extensions);
  MARK (all_extensions);
  MARK (state_supers);
  MARK (state_subs);
  MARK (behaviour_supers);
  MARK (behaviour_subs);
  MARK (poser);

  [super gcReference];
}

-(id) initWithName: (TLString *) n unit: (LTTUnit *) u
{
  [super initWithName:
	   [CO_LTTName nameWithInternal: n
		    external: formac (nil, @"%@_%@",
				      [[u lttName] external], quote (n))]];

  unit = u;
  extensions = [TLDictionary dictionary];
  all_extensions = [TLVector vector];
  [all_extensions addElement: nil];
  state_supers = [CO_TLVector vector];
  state_subs = [CO_TLVector vector];
  behaviour_supers = [CO_TLVector vector];
  behaviour_subs = [CO_TLVector vector];

  return self;
}

-(LTTInstance *) instance
{
  error (@"%@ does not have instances", ltt_meta_name (self));
  return nil;
}

-(BOOL) isDirectBehaviourSub: (LTTMeta *) m
{
  return !![behaviour_subs memq: m];
}

-(BOOL) isDirectSub: (LTTMeta *) m
{
  return [self isDirectBehaviourSub: m] || [self isDirectStateSub: m];
}

-(BOOL) isDirectStateSub: (LTTMeta *) m
{
  return !![state_subs memq: m];
}

-(BOOL) isProperBehaviourSub: (LTTMeta *) m
{
  return [self isProperBehaviourSub: m mark: ++search_mark];
}

-(BOOL) isProperBehaviourSub: (LTTMeta *) m mark: (int) k
{
  LTTMeta *sub;
  int i, n;

  if (k == mark)
    return NO;
  mark = k;

  if ([self isDirectBehaviourSub: m])
    return YES;

  for (i = 0, n = [behaviour_subs length]; i < n; i++)
    {
      sub = [behaviour_subs _elementAtIndex: i];
      if ([sub isProperBehaviourSub: m mark: mark])
	return YES;
    }

  return NO;
}

-(BOOL) isProperSub: (LTTMeta *) m
{
  return [self isProperSub: m mark: ++search_mark];
}

-(BOOL) isProperSub: (LTTMeta *) m mark: (int) k
{
  LTTMeta *sub;
  int i, n;

  if (k == mark)
    return NO;
  mark = k;

  if ([self isDirectSub: m])
    return YES;

  for (i = 0, n = [state_subs length]; i < n; i++)
    {
      sub = [state_subs _elementAtIndex: i];
      if ([sub isProperSub: m mark: mark])
	return YES;
    }

  for (i = 0, n = [behaviour_subs length]; i < n; i++)
    {
      sub = [behaviour_subs _elementAtIndex: i];
      if ([sub isProperSub: m mark: mark])
	return YES;
    }

  return posed ? [posed isProperSub: m mark: k] : NO;
}

-(BOOL) isProperStateSub: (LTTMeta *) m
{
  return [self isProperStateSub: m mark: ++search_mark];
}

-(BOOL) isProperStateSub: (LTTMeta *) m mark: (int) k
{
  LTTMeta *sub;
  int i, n;

  if (k == mark)
    return NO;
  mark = k;

  if ([self isDirectStateSub: m])
    return YES;

  for (i = 0, n = [state_subs length]; i < n; i++)
    {
      sub = [state_subs _elementAtIndex: i];
      if ([sub isProperStateSub: m mark: mark])
	return YES;
    }

  return posed ? [posed isProperStateSub: m mark: k] : NO;
}

-(LTTClass *) itsClass
{
  return class;
}

-(id <TLString>) metaDefinitionName
{
  return ([self classp]
	  ? formac (nil, @"%@m_%@", TO_META_DEF_PREFIX, [name external])
	  : [class definitionName]);
}

-(id <TLString>) metaReferenceName
{
  return ([self classp]
	  ? formac (nil, @"%@m_%@", TO_META_REF_PREFIX, [name external])
	  : [class referenceName]);
}

-(id <TLString>) metaTypeName
{
  return ([self classp] ? formac (nil, @"m_%@", [name external])
	  : [self outputTypeName]);
}

-(int) numBehaviourSubs
{
  return [behaviour_subs length];
}

-(int) numBehaviourSupers
{
  return [behaviour_supers length];
}

-(int) numStateSubs
{
  return [state_subs length];
}

-(int) numStateSupers
{
  return [state_supers length];
}

-(id <TLString>) outputSuperReference: (LTTMeta *) m
{
  if (ltt_current_unit)
    return formac (nil, @"%@%@_%@_%@", TO_SUPER_REF_PREFIX,
		   [ltt_current_unit outputName],
		   [self qualifiedExternalName], [m qualifiedExternalName]);
  else
    return formac (nil, @"%@%@_%@", TO_SUPER_REF_PREFIX,
		   [self qualifiedExternalName], [m qualifiedExternalName]);
}

-(id <TLString>) outputTypeName
{
  return [self qualifiedExternalName];
}

-(LTTMeta *) posedSelf
{
  return poser ? [poser posedSelf] : self;
}

-(LTTMeta *) posedSelfNot: (LTTMeta *) m
{
  return poser && poser != m ? [poser posedSelfNot: m] : self;
}

-(LTTMeta *) posed
{
  return posed;
}

-(LTTMeta *) poser
{
  return poser;
}

-(void) print: (id) s quoted: (BOOL) qp
{
  formac (s, @"#<%s %lx %@>", class_get_class_name (isa),
	  (long) self, [name internal]);
}

-(id <TLString>) qualifiedExternalName
{
  char c = [self classp] ? 'c' : 'i';

  return formac (nil, @"%c_%@", c, [name external]);
}

-(id <TLString>) referenceName
{
  return (id) formac (nil, @"%@%@", TO_META_REF_PREFIX,
		      [self qualifiedExternalName]);
}

-(void) setClass: (LTTClass *) c
{
  ASGN_IVAR (class, c);
}

-(void) setPosed: (LTTMeta *) p
{
  if (posed)
    ABORT ();
  ASGN_IVAR (posed, p);
}

-(void) setPoser: (LTTMeta *) p
{
  if (poser)
    [poser setPoser: p];
  else
    {
      ASGN_IVAR (poser, p);
      [p setPosed: self];
    }
}

-(TLVector *) stateSubs
{
  return state_subs;
}

-(TLVector *) stateSupers
{
  return state_supers;
}

-(LTTUnit *) unit
{
  return unit;
}

@end
