-- This file is  free  software, which  comes  along  with  SmallEiffel. This
-- software  is  distributed  in the hope that it will be useful, but WITHOUT 
-- ANY  WARRANTY;  without  even  the  implied warranty of MERCHANTABILITY or
-- FITNESS  FOR A PARTICULAR PURPOSE. You can modify it as you want, provided
-- this header is kept unaltered, and a notification of the changes is added.
-- You  are  allowed  to  redistribute  it and sell it, alone or as a part of 
-- another product.
--          Copyright (C) 1994-98 LORIA - UHP - CRIN - INRIA - FRANCE
--            Dominique COLNET and Suzanne COLLIN - colnet@loria.fr 
--                       http://www.loria.fr/SmallEiffel
--
class BASIC_DIRECTORY
   --
   -- Low-level basic tools for file-system directory handling.
   --
   -- Also consider hight level facade class DIRECTORY for a safer usage.
   --

creation make

feature

   path: STRING;
         -- Void or some `path' which may be absolute or not (see `set_path').

   name_list: FIXED_ARRAY[STRING];
         -- Actual list of entries (files or subdirectories)..

feature {NONE} -- Creation :

   make is
         -- Make a new object which is _not_ connected to any 
         -- directory on disk.
         -- Use `set_path' to connect Current to a directory on disk.
      do
         !!name_list.with_capacity(64);
      end;

feature -- Access :

   lower: INTEGER is 0;
         -- Index of the first item.

   upper: INTEGER is
         -- Index of the last item.
      do
         Result := name_list.upper;
      end;

   count: INTEGER is
         -- Number of items (files or directories) in Current.
      do
         Result := name_list.count;
      ensure
         Result >= 0
      end;

   valid_index(index: INTEGER): BOOLEAN is
      do
         Result := index >= 0 and then index <= name_list.upper;
      ensure
         Result = index.in_range(0,upper)
      end;

   is_subdirectory(index: INTEGER): BOOLEAN is
         -- Is the item at `index' a subdirectory ?
      require
         valid_index(index)
      do
         not_yet_implemented;
      end;

   is_file(index: INTEGER): BOOLEAN is
         -- Is the item at `index' a file ?
      require
         valid_index(index)
      do
         Result := not is_subdirectory(index);
      ensure
         Result = not is_subdirectory(index)
      end;

   absolute_path: STRING is
         -- Gives acces to the absolute path of Current.
      do
         not_yet_implemented;
      ensure
         Result /= Void
      end;

   absolute_path_of(index: INTEGER): STRING is
         -- Gives access to absolute path of item at `index'.
      require
         valid_index(index)
      do
         not_yet_implemented;
      ensure
         Result /= Void
      end;

   get_parent_directory: like Current is
         -- Get new object corresponding to parent directory.
         -- Return Void in case of failure.
         -- Also consider `go_parent_directory'.
      do
         not_yet_implemented;
      end;

   get_subdirectory(index: INTEGER): like Current is
         -- Get new directory object corresponding to item at `index'.
         -- Return Void in case of failure.
         -- Also consider `go_subdirectory'.
      require
         is_subdirectory(index)
      do
         not_yet_implemented;
      end;

   has(entry_name: STRING): BOOLEAN is
         -- Does Current contain the `entry_name' (file or subdirectory) ?
      require
         not entry_name.empty
      do
         Result := name_list.has(entry_name);
      end;

   index_of(entry_name: STRING): INTEGER is
         -- Index of `entry_name' (file or subdirectory).
         -- Return `count + 1' if not found.
      do
         Result := name_list.index_of(entry_name);
      ensure
         (Result = count + 1) or valid_index(Result)
      end;

   has_file(filename : STRING): BOOLEAN is
         -- Does Current contain `filename' as an entry for a file ?
      require
         not filename.empty
      local
         i: INTEGER
      do
         i := index_of(filename);
         if valid_index(i) then
            Result := is_file(i);
         end;
      ensure
         Result implies is_file(index_of(filename))
      end;

   has_subdirectory(dirname : STRING): BOOLEAN is
         -- Does Current contain `dirname' as an entry for a subdirectory ?
      require
         not dirname.empty
      local
         i: INTEGER
      do
         i := index_of(dirname);
         if valid_index(i) then
            Result := is_subdirectory(i);
         end;
      ensure
         Result implies is_subdirectory(index_of(dirname))
      end;

   name(index: INTEGER): STRING is
         -- Return the name of entry (file or subdirectory) at `index'.
      require
         valid_index(index)
      do
         Result := name_list.item(index);
      ensure
         has(Result)
      end;

feature -- Modification :

   go_parent_directory: BOOLEAN is
         -- Change Current to its parent directory; do not change 
         -- anything in case of failure.
         -- Return true if succeeded, false otherwise.
         -- Also consider `get_parent_directory'.
      do
         not_yet_implemented;
      end;

   go_subdirectory(index: INTEGER): BOOLEAN is
         -- Change Current to subdirectory at `index'; do not change 
         -- anything in case of failure.
         -- Return true if succeeded, false otherwise.
         -- Also consider `get_subdirectory'.
      require
         is_subdirectory(index)
      do
         not_yet_implemented;
      end;

feature -- Disk access :

   update: BOOLEAN is
         -- Update Current status by reloading information from the 
         -- disk.
         -- Return true if succeeded, false otherwise.
      local
         path_pointer, dirstream_pointer, entry_pointer: POINTER;
         entry: STRING;
      do
         name_list.clear;
         path_pointer := path.to_external;
         dirstream_pointer := se_dir_open(path_pointer);
         if dirstream_pointer.is_not_null then
            from
               Result := true;
            until
               dirstream_pointer.is_null
            loop
               entry_pointer := se_dir_name(dirstream_pointer);
               if entry_pointer.is_not_null then
                  !!entry.from_external_copy(entry_pointer);
                  name_list.add_last(entry);
               end;
               dirstream_pointer := se_dir_next(dirstream_pointer);
            end;
         end;
      end;

feature -- Disk access :

   set_path(p: like path): BOOLEAN is
         -- Connects Current to the existing directory `path' on disk.
         -- Support of absolute and/or relative paths is not guaranteed for 
         -- portability reasons. :-)
         -- Return true if succeeded, false otherwise.
         -- In case of success, `update' has been successfully done.
      require
         not p.empty
      do
         path := p;
         if update then
            Result := true;
         else
            path := Void;
         end;
      ensure
         Result implies path = p
      end;

feature -- current directory handling :

   current_working_directory is
         -- Update Current with the current working directory.
      local
         path_pointer: POINTER;
      do
         path_pointer := se_dir_gcwd(0);
         !!path.from_external_copy(path_pointer);
         if update then 
         else
            path := Void;
         end;
      end;

feature -- Disk modification :

   create_subdirectory(dirname: STRING): BOOLEAN is
         -- Create a new (one-level) subdirectory `dirname' on disk.
         -- Nested creations support and semantics are not guaranteed for 
         -- portability reasons.
         -- Return true if succeeded, false otherwise.
         -- In case of success, `update' has been successfully done.
      require
         not has(dirname)
      do
         not_yet_implemented;
      end;

   delete_subdirectory(dirname: STRING): BOOLEAN is
         -- Delete (one-level) subdirectory `dirname' on disk.
         -- Nested deletions support and semantics are not guaranteed for 
         -- portability reasons.
         -- Return true if succeeded, false otherwise.
         -- In case of success, `update' has been successfully done.
      require
         is_subdirectory(index_of(dirname))
      do
         not_yet_implemented;
      end;

feature {NONE}

   se_dir_open(path_pointer: POINTER): POINTER is
      require
         path_pointer.is_not_null
      external "SmallEiffel"
      end;

   se_dir_name(dirstream_pointer: POINTER): POINTER is
      require
         dirstream_pointer.is_not_null
      external "SmallEiffel"
      end;

   se_dir_next(dirstream_pointer: POINTER): POINTER is
      require
         dirstream_pointer.is_not_null
      external "SmallEiffel"
      end;

   se_dir_gcwd(dummy: INTEGER): POINTER is
      external "SmallEiffel"
      end;

end -- BASIC_DIRECTORY

