(define-class <graphics-device> (<object>) :abstract
  (properties type: <vector> init-value: '#()))

(define-class <x11-device> (<graphics-device>)
  (x-display :sealed)
  (x-window :sealed type: <x-window>)
  (x-gc :sealed type: <x-gcontext>)
  (current-path init-value: #f)
  (current-font init-value: #f)
  (ctm type: <transform>)
  (owner init-value: #f))

(define (open-x11-device (w <x-window>))
  (bind ((scrn (drawable-screen w))
	 (gc (create-gcontext drawable: w
			      foreground: (screen-black-pixel scrn)
			      background: (screen-white-pixel scrn))))
    (make <x11-device>
	  x-window: w
	  x-display: (drawable-display w)
	  ctm: (make-affine-transform)
	  x-gc: gc)))

(define (get-x11-device-color (dev <x11-device>) r g b)
  (let ((key (+ (logical-shift-left
		 (bitwise-and (inexact->exact (* r 512)) 511)
		 18)
		(logical-shift-left
		 (bitwise-and (inexact->exact (* g 512)) 511)
		 9)
		(logical-shift-left
		 (bitwise-and (inexact->exact (* b 512)) 511)
		 0)))
	(tbl (get-property (x-window dev) 'device-color-map #f)))
    (if (not tbl)
	(begin
	  (set! tbl (make-fixnum-table))
	  (set-property! (x-window dev) 'device-color-map tbl)))
    (or (table-lookup tbl key)
	(let ((pix (get-dk-color (drawable-screen (x-window dev))
				 (make-color red: r
					     green: g
					     blue: b))))
	  (table-insert! tbl key pix)
	  pix))))


(define-method device-color ((dev <x11-device>) c)
  (let ((s (drawable-screen (x-window dev))))
    (case c
      ((black) (screen-black-pixel s))
      ((white) (screen-white-pixel s))
      (else
       (case (car c)
	 ((rgb) (get-x11-device-color dev (cadr c) (caddr c) (cadddr c)))
	 ((gray) (get-x11-device-color dev (cadr c) (cadr c) (cadr c))))))))
#|
       (get-dk-color
	s
	(case (car c)
	  ((gray)
	   (make-color red: (cadr c)
		       green: (car c)
		       blue: (cadr c)))
	  ((rgb)
	   (make-color red: (cadr c)
		       green: (caddr c)
		       blue: (cadddr c)))))))))
  |#
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   X11
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (device-point (pt <point>))
  (make-point (inexact->exact (x pt))
	      (inexact->exact (y pt))))

(define-method transform ((self <x11-device>) (p0 <point>))
  (let* ((pt (transform p0 (ctm self)))
	 (p2 (device-point pt)))
    (table-insert! (current-geometry (owner self)) p2 pt)
    (values p2 pt)))

(define-method setcolor ((self <x11-device>) (pixel-value <fixnum>))
  (set-gcontext-foreground! (x-gc self) pixel-value))

(define-method setlinewidth ((self <x11-device>) width)
  (define (sqr a)
    (* a a))
  (let* ((pt (transform (make-size width 0) (ctm self)))
	 (w (inexact->exact 
	     (sqrt (+ (sqr (dx pt)) (sqr (dy pt)))))))
    (set-gcontext-line-width! (x-gc self) w)))

(define-method setfont ((self <x11-device>) (font <text-font>))
  (set-current-font! self font)
  (let* ((d (transform (make-size (font-size font) 0) (ctm self)))
	 (w (inexact->exact (round (sqrt (inner-product d d))))))
    ;(dm "font size ~d ==> SCALE ==> ~d" (font-size font) w)
    (let ((f (if (= w (font-size font))
		 font
		 (make <text-font>
		   font-name: (font-name font)
		   font-style: (font-style font)
		   font-size: w))))
      (set-gcontext-font! (x-gc self) (get-x-font f (x-display self))))))

(define (get-x-font font dpy)
  (get-property dpy font (fill-x-font-cache font dpy)))

(define (fill-x-font-cache font dpy)
  (let* ((fn (get-x-font-name font))
	 (f (open-font dpy fn)))
    (set-property! dpy font f)
    f))

(define-method current-point ((self <x11-device>))
  (let* ((p (current-path self))
	 (k (dequeue-count p)))
    (values (dequeue-ref p (- k 2))
	    (dequeue-ref p (- k 1)))))

(define-method moveto ((self <x11-device>) (pt <point>))
  (set-current-path! self (make-dequeue))
  (let (((pt <point>) (transform self pt)))
    (dequeue-push-back! (current-path self) (x pt))
    (dequeue-push-back! (current-path self) (y pt))
    (values)))

(define-method lineto ((self <x11-device>) (pt <point>))
  (let (((pt <point>) (transform self pt)))
    (dequeue-push-back! (current-path self) (x pt))
    (dequeue-push-back! (current-path self) (y pt))
    (values)))


(define-method curveto ((self <x11-device>) (h1 <point>)
					    (h2 <point>) 
					    (pt <point>))
  (let (((h1 <point>) (transform self h1))
	((h2 <point>) (transform self h2))
	((pt <point>) (transform self pt)))
    (bind ((cx cy (current-point self)))
      (let ((c (curv cx cy (x h1) (y h1) (x h2) (y h2) (x pt) (y pt))))
	(for-each
	 (lambda (t)
	   (let ((p (point-on c t)))
	     (dequeue-push-back! (current-path self) 
				 (inexact->exact (round (x p))))
	     (dequeue-push-back! (current-path self) 
				 (inexact->exact (round (y p))))
	     (values)))
	 '(0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9))
	(dequeue-push-back! (current-path self) (x pt))
	(dequeue-push-back! (current-path self) (y pt))
	(values)))))

(define-method rectstroke ((self <x11-device>) (r <rect>))
  (moveto self (lower-left r))
  (lineto self (lower-right r))
  (lineto self (upper-right r))
  (lineto self (upper-left r))
  (lineto self (lower-left r))
  (stroke self))

(define-method stroke ((self <x11-device>))
  (draw-lines (x-window self) 
	      (x-gc self) 
	      (vector->list (dequeue-state (current-path self)))))

(define-method show ((self <x11-device>) (str <string>))
  (if (> (string-length str) 0)
      (bind ((devx devy (current-point self))
	     (dxs (xshow-x-list (font-afm (current-font self)) 
				(font-size (current-font self))
				str)))
	    (let loop ((devx devx)
		       (devy devy)
		       (dxs dxs)
		       (i 0))
	      (draw-glyphs (x-window self) (x-gc self) 
			   (inexact->exact devx)
			   (inexact->exact devy)
			   (string (string-ref str i)))
	      (if (< (+ i 1) (string-length str))
		  (let ((dp (transform (make-size (car dxs) 0) (ctm self))))
		    ;(dm 141 "for ~s, dp = ~s" (string-ref str i) dp)
		    (loop (+ devx (dx dp))
			  (+ devy (dy dp))
			  (cdr dxs) 
			  (+ i 1))))))))

(define-method fill ((self <x11-device>))
  (draw-lines (x-window self) 
	      (x-gc self) 
	      (vector->list (dequeue-state (current-path self)))
	      fill?: #t))

(define-method closepath ((self <x11-device>))
  ;; what should this do..?
  (values))

(define-method translate ((self <x11-device>) (delta <point>))
  (set-ctm! self (translate (ctm self) delta)))

(define-method concat ((self <x11-device>) tm)
  (set-ctm! self (concatenate-transform (ctm self) tm)))

;;;

(define-method with-gstate-saved ((self <x11-device>) thunk)
  (let ((saved-ctm (ctm self))
	(gc (create-gcontext drawable: (x-window self))))
    (copy-gcontext (x-gc self) gc)
    (thunk)
    (copy-gcontext gc (x-gc self))
    (free-gcontext gc)
    (set-ctm! self saved-ctm)
    (values)))

(define-method with-ctm-saved ((self <x11-device>) thunk)
  (let ((saved-ctm (ctm self)))
    (thunk)
    (set-ctm! self saved-ctm)
    (values)))


(define (show-handle (dev <x11-device>) at)
  (bind ((pt pte (transform dev at)))
    (draw-rectangle (x-window dev) (x-gc dev) 
		    (- (x pt) 1) (- (y pt) 1) 3 3
		    #t)
    pte))
