/* Load needed message catalogs. 
   Copyright (C) 1995, 1996, 1997, 1998 Free Software Foundation, Inc.
   Modified by Enrique Zanardi <ezanard@debian.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <stdlib.h>

#include <unistd.h>
#include <sys/mman.h>

#include "gettext.h"
#include "domain.h"

#ifndef W
# define W(flag, data) ((flag) ? SWAP (data) : (data))
#endif

static inline nls_uint32 SWAP (nls_uint32 i) {
  return (i << 24) | ((i & 0xff00) << 8) | ((i >> 8) & 0xff00) | (i >> 24);
}

/* Load the message catalogs specified by FILENAME.  If it is no valid
   message catalog do nothing.  */
struct loaded_domain *load_domain (char *domain_file) {
  int fd;
  size_t size;
  struct stat st;
  struct mo_file_header *data = (struct mo_file_header *) -1;
  struct loaded_domain *domain;

  /* Try to open the addressed file.  */
  fd = open (domain_file, O_RDONLY);
  if (fd == -1)
    return NULL;

  /* We must know about the size of the file.  */
  if (fstat (fd, &st) != 0
      || (size = (size_t) st.st_size) != st.st_size
      || size < sizeof (struct mo_file_header)) {
    /* Something went wrong.  */
    close (fd);
    return NULL;
  }

  /* Now we are ready to load the file.  If mmap() is available we try
     this first.  If not available or it failed we try to load it.  */
  data = (struct mo_file_header *) mmap (NULL, size, PROT_READ,
					 MAP_PRIVATE, fd, 0);

  if (data != (struct mo_file_header *) -1) {
    /* mmap() call was successful.  */
    close (fd);
  } else {
    return NULL;
  }

  /* Using the magic number we can test whether it really is a message
     catalog file.  */
  if (data->magic != _MAGIC && data->magic != _MAGIC_SWAPPED) {
    /* The magic number is wrong: not a message catalog file.  */
    munmap ((caddr_t) data, size);
    return NULL;
  }

  domain= (struct loaded_domain *) malloc (sizeof (struct loaded_domain));
  if (domain == NULL)
    return NULL;

  domain->data = (char *) data;
  domain->must_swap = data->magic != _MAGIC;

  /* Fill in the information about the available tables.  */
  switch (W (domain->must_swap, data->revision))
    {
    case 0:
      domain->nstrings = W (domain->must_swap, data->nstrings);
      domain->orig_tab = (struct string_desc *)
	((char *) data + W (domain->must_swap, data->orig_tab_offset));
      domain->trans_tab = (struct string_desc *)
	((char *) data + W (domain->must_swap, data->trans_tab_offset));
      break;
    default:
      /* This is an invalid revision.  */
      munmap ((caddr_t) data, size);
      free (domain);
      return NULL;
    }
  return domain;
}
