/*
 * SXPML.C - PML extensions in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define GET_ARRAY_FP_VALUE(type, arr, n)                                      \
    {type *p;                                                                 \
     p = (type *) arr->data;                                                  \
     ret = SS_mk_float((double) p[n]);}

#define GET_ARRAY_FIX_VALUE(type, arr, n)                                     \
    {type *p;                                                                 \
     p = (type *) arr->data;                                                  \
     ret = SS_mk_integer((BIGINT) p[n]);}

#define SET_ARRAY_FP_VALUE(type, arr, n, val)                                 \
    {type *p;                                                                 \
     double v;                                                                \
     p = (type *) arr->data;                                                  \
     SS_args(val,                                                             \
	     SC_DOUBLE_I, &v,                                                 \
	     0);                                                              \
     p[n] = (type) v;}

#define SET_ARRAY_FIX_VALUE(type, arr, n, val)                                \
    {type *p;                                                                 \
     long v;                                                                  \
     p = (type *) arr->data;                                                  \
     SS_args(val,                                                             \
	     SC_LONG_I, &v,                                                   \
	     0);                                                              \
     p[n] = (type) v;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_NUMERIC_ARAYP - function version of SX_NUMERIC_ARRAYP macro */

object *SX_numeric_arrayp(obj)
   object *obj;
   {return(SX_NUMERIC_ARRAYP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAPPINGP - function version of SX_MAPPINGP macro */

object *SX_mappingp(obj)
   object *obj;
   {return(SX_MAPPINGP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SETP - function version of SX_SETP macro */

object *SX_setp(obj)
   object *obj;
   {return(SX_SETP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_ARRAY - allocate and return a C_array
 *             - form: (pm-make-array <type> <size>)
 */

static object *SX_mk_array(argl)
   object *argl;
   {C_array *arr;
    long size, bpi;
    char *type, ltype[MAXLINE];

    type = NULL;
    size = 0L;
    SS_args(argl,
            SC_STRING_I, &type,
            SC_INTEGER_I, &size,
            0);

    if (SX_vif != NULL)
       {defstr *dp;

	dp = PD_inquire_host_type(SX_vif, type);
	if (dp != NULL)
	   type = dp->type;};

    bpi = _PD_lookup_size(type, SX_vif->host_chart);
    arr = FMAKE(C_array, "SX_MK_ARRAY:arr");

    sprintf(ltype, "%s *", type);

    arr->type   = SC_strsavef(ltype, "char*:SX_MK_ARRAY:ltype");
    arr->length = size;
    arr->data   = (byte *) FMAKE_N(char, size*bpi,
                                   "SX_MK_ARRAY:data");

    return(SX_mk_C_array(arr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RESZ_ARRAY - reallocate and return a C_array
 *               - form: (pm-resize-array <array> <size>)
 */

static object *SX_resz_array(argl)
   object *argl;
   {C_array *arr;
    long size, bpi;

    arr  = NULL;
    size = 0L;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            SC_INTEGER_I, &size,
            0);

    if (arr != NULL)
       {bpi = SC_arrlen(arr->data)/arr->length;

        arr->length = size;
        arr->data   = SC_realloc_na(arr->data, size, bpi, FALSE);};

    return(SS_car(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SUB_ARRAY - allocate and return a C_array that is a sub-array of the
 *               - input array.
 *               - form: (pm-sub-array <array> <dimensions> <region>)
 */

static object *SX_sub_array(argl)
   object *argl;
   {C_array *arr, *newarr;
    object *dims, *reg, *obj;
    int i, nd, nr;
    long *idims, *ireg, *pd, *pr, length, rlength, bpi;

    arr  = NULL;
    dims = SS_null;
    reg  = SS_null;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            SS_OBJECT_I, &dims,
            SS_OBJECT_I, &reg,
            0);

    if (arr == NULL)
       SS_error("NO ARRAY SPECIFIED - SX_SUB_ARRAY", argl);

/* extract the array dimensions */
    if (SS_nullobjp(dims))
       SS_error("NO ARRAY DIMENSIONS SPECIFIED - SX_SUB_ARRAY", argl);

    else
       {nd       = _SS_length(dims);
        idims    = FMAKE_N(long, nd + 1, "SX_SUB_ARRAY:idims");
        idims[0] = nd/2;
        for (pd = idims + 1; !SS_nullobjp(dims); pd++)
            SX_GET_INTEGER_FROM_LIST(*pd, dims,
                                     "BAD ARRAY DIMENSIONS - SX_SUB_ARRAY");};

/* extract the region specifications */
    if (SS_nullobjp(reg))
       SS_error("NO REGION SPECIFIED - SX_SUB_ARRAY", argl);

    else
       {nr      = _SS_length(reg);
        ireg    = FMAKE_N(long, nr + 1, "SX_SUB_ARRAY:ireg");
        ireg[0] = nr/2;
        for (pr = ireg + 1; !SS_nullobjp(reg); pr++)
            SX_GET_INTEGER_FROM_LIST(*pr, reg,
                                     "BAD REGION - SX_SUB_ARRAY");};

/* do some error checking */
    for (length = 1, pd = idims + 1, i = 0; i < idims[0]; i++)
        length *= pd[2*i + 1] - pd[2*i] + 1;

    for (rlength = 1, pr = ireg + 1, i = 0; i < ireg[0];  i++)
        rlength *= pr[2*i + 1] - pr[2*i] + 1;

    if (length != arr->length)
       SS_error("BAD DIMENSIONS SPECIFIED FOR ARRAY - SX_SUB_ARRAY", argl);

    if ((rlength > length)  || (rlength < 0))
       SS_error("BAD REGION SPECIFIED FOR ARRAY - SX_SUB_ARRAY", argl);
    
/* allocate the output array */
    bpi = SC_arrlen(arr->data)/arr->length;
    newarr = FMAKE(C_array, "SX_MK_ARRAY:arr");

    newarr->type   = SC_strsavef(arr->type, "char *:SX_SUB_ARRAY:arr->type");
    newarr->length = rlength;
    newarr->data   = (byte *) FMAKE_N(char, rlength*bpi,
                                      "SX_SUB_ARRAY:data"); 

    PM_sub_array(arr->data, newarr->data, idims, ireg, bpi);

    return(SX_mk_C_array(newarr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ARRAY_REF - reference the nth element of a pm-array
 *              - form: (pm-array-ref <array> <n>)
 */

static object *SX_array_ref(argl)
   object *argl;
   {C_array *arr;
    long n;
    object *ret;

    arr = NULL;
    n   = 0L;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            SC_INTEGER_I, &n,
            0);

    ret = SS_null;
    if (arr != NULL)
       {char type[MAXLINE];

        strcpy(type, arr->type);
        if (SX_vif != NULL)
           {defstr *dp;

            dp = PD_inquire_host_type(SX_vif, type);
            if (dp != NULL)
               strcpy(type, dp->type);};

	while (_PD_indirection(type))
	   PD_dereference(type);

        if (strcmp(type, SC_DOUBLE_S) == 0)
	   {GET_ARRAY_FP_VALUE(double, arr, n);}
        else if (strcmp(type, SC_FLOAT_S) == 0)
	   {GET_ARRAY_FP_VALUE(float, arr, n);}
        else if (strcmp(type, SC_REAL_S) == 0)
	   {GET_ARRAY_FP_VALUE(REAL, arr, n);}
        else if (strcmp(type, SC_SHORT_S) == 0)
	   {GET_ARRAY_FIX_VALUE(short, arr, n);}
        else if (strncmp(type, SC_INTEGER_S, 3) == 0)
	   {GET_ARRAY_FIX_VALUE(int, arr, n);}
        else if (strcmp(type, SC_LONG_S) == 0)
	   {GET_ARRAY_FIX_VALUE(long, arr, n);}
        else if (strcmp(type, SC_CHAR_S) == 0)
	   {GET_ARRAY_FIX_VALUE(char, arr, n);};}

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ARRAY_SET - set the nth element of a pm-array
 *              - form: (pm-array-set! <array> <n> <value>)
 */

static object *SX_array_set(argl)
   object *argl;
   {C_array *arr;
    long n;
    object *val;

    arr = NULL;
    n   = 0L;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            SC_INTEGER_I, &n,
            SS_OBJECT_I, &val,
            0);

    if (arr != NULL)
       {char type[MAXLINE];

        strcpy(type, arr->type);
        if (SX_vif != NULL)
           {defstr *dp;

            dp = PD_inquire_host_type(SX_vif, type);
            if (dp != NULL)
               strcpy(type, dp->type);};

	while (_PD_indirection(type))
	   PD_dereference(type);

        if (strcmp(type, SC_DOUBLE_S) == 0)
	   {SET_ARRAY_FP_VALUE(double, arr, n, val);}
        else if (strcmp(type, SC_FLOAT_S) == 0)
	   {SET_ARRAY_FP_VALUE(float, arr, n, val);}
        else if (strcmp(type, SC_REAL_S) == 0)
	   {SET_ARRAY_FP_VALUE(REAL, arr, n, val);}
        else if (strcmp(type, SC_SHORT_S) == 0)
	   {SET_ARRAY_FIX_VALUE(short, arr, n, val);}
        else if (strncmp(type, SC_INTEGER_S, 3) == 0)
	   {SET_ARRAY_FIX_VALUE(int, arr, n, val);}
        else if (strcmp(type, SC_LONG_S) == 0)
	   {SET_ARRAY_FIX_VALUE(long, arr, n, val);}
        else if (strcmp(type, SC_CHAR_S) == 0)
	   {SET_ARRAY_FIX_VALUE(char, arr, n, val);};}

    return(val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_ARRAY - turn a list of numbers into a numeric array */

object *SX_list_array(argl)
   object *argl;
   {object *num, *lst;
    C_array *arr;
    int n;
    double *fp;

    arr = FMAKE(C_array, "SX_LIST_ARRAY:arr");

    n    = 0;
    lst  = argl;
    for (lst = argl; !SS_nullobjp(lst); lst = SS_cdr(lst))
        {num = SS_car(lst);
         if (!SS_numberp(num))
            SS_error("NUMECT NOT A NUMBER - SX_LIST_ARRAY", num);

         n++;};

/* all arrays will be REALS */
    fp          = FMAKE_N(double, n, "SX_LIST_ARRAY:fp");
    arr->type   = SC_strsavef(SC_DOUBLE_S, "char*:SX_LIST_ARRAY:type");
    arr->length = n;
    arr->data   = (byte *) fp;
    for (lst = argl; !SS_nullobjp(lst); lst = SS_cdr(lst), fp++)
        {SS_args(lst,
		 SC_DOUBLE_I, fp,
		 0);};

    return(SX_mk_C_array(arr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ARRAY_LIST - turn a numeric array into a list of numbers */

object *SX_array_list(argl)
   object *argl;
   {object *obj, *lst;
    C_array *arr;
    long i, n;
    char *type, *cp;
    short *sp;
    int *ip;
    long *lp;
    float *fp;
    double *dp;
    byte *data;

    arr = NULL;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            0);

    n    = arr->length;
    type = arr->type;
    data = arr->data;

    lst = SS_null;
    if (strcmp(type, SC_DOUBLE_S) == 0)
       {dp = (double *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_float((double) *dp++);
             lst = SS_mk_cons(obj, lst);};}

    else if (strcmp(type, SC_FLOAT_S) == 0)
       {fp = (float *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_float((double) *fp++);
             lst = SS_mk_cons(obj, lst);};}

    else if (strcmp(type, SC_LONG_S) == 0)
       {lp = (long *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_integer((BIGINT) *lp++);
             lst = SS_mk_cons(obj, lst);};}

    else if ((strcmp(type, SC_INTEGER_S) == 0) ||
	     (strcmp(type, "int") == 0))
       {ip = (int *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_integer((BIGINT) *ip++);
             lst = SS_mk_cons(obj, lst);};}

    else if (strcmp(type, SC_SHORT_S) == 0)
       {sp = (short *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_integer((BIGINT) *sp++);
             lst = SS_mk_cons(obj, lst);};}

    else if (strcmp(type, SC_CHAR_S) == 0)
       {cp = (char *) data;
        for (i = 0L; i < n; i++)
            {obj = SS_mk_integer((BIGINT) *cp++);
             lst = SS_mk_cons(obj, lst);};}

    else
       SS_error("DATA TYPE NOT SUPPORTED - SX_ARRAY_LIST", SS_null);

    if (lst != SS_null)
       lst = SS_reverse(lst);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_NUM_ARR_LEN - return the length of a numeric array */

object *SX_num_arr_len(obj)
   object *obj;
   {long n;

    if (!SX_NUMERIC_ARRAYP(obj))
       SS_error("ARGUMENT NOT NUMERIC ARRAY - SX_NUM_ARR_LEN", obj);

    n = NUMERIC_ARRAY_LENGTH(obj);

    return(SS_mk_integer((BIGINT)n));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_NUM_ARR_EXTR - return the extrema of a numeric array */

object *SX_num_arr_extr(arg)
   object *arg;
   {long n;
    int imin, imax;
    char *type;
    byte *data;
    object *lst, *obj;
    double fmin, fmax;

    if (!SX_NUMERIC_ARRAYP(arg))
       SS_error("ARGUMENT NOT NUMERIC ARRAY - SX_NUM_ARR_LEN", arg);

    n    = NUMERIC_ARRAY_LENGTH(arg);
    type = NUMERIC_ARRAY_TYPE(arg);
    data = NUMERIC_ARRAY_DATA(arg);

    lst = SS_null;
    if (strcmp(type, SC_DOUBLE_S) == 0)
       {PM_minmax(data, (int) n, &fmin, &fmax, &imin, &imax);
	obj = SS_mk_integer((BIGINT)imax);
	lst = SS_mk_cons(obj, lst);
	obj = SS_mk_integer((BIGINT)imin);
	lst = SS_mk_cons(obj, lst);
	obj = SS_mk_float(fmax);
	lst = SS_mk_cons(obj, lst);
	obj = SS_mk_float(fmin);
	lst = SS_mk_cons(obj, lst);}

    else
       SS_error("DATA TYPE NOT SUPPORTED - SX_ARRAY_LIST", SS_null);

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_PDBDATA - given a PM_set object
 *                - allocate and fill a PDB data object with the
 *                - PM_set and return it
 */

static object *SX_set_pdbdata(argl)
   object *argl;
   {PM_set *s;
    g_file *po;
    char *mn, set_name[MAXLINE];
    PDBfile *file;

    s  = NULL;
    po = NULL;
    mn = NULL;
    SS_args(argl,
            G_SET, &s,
            G_FILE, &po,
            SC_STRING_I, &mn,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;

    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);

    else
       SS_error("BAD FILE - SX_SET_PDBDATA", argl);

    if (mn == NULL)
       strcpy(set_name, s->name);
    else
       {strcpy(set_name, mn);
        SFREE(mn);};

    if (s == NULL)
       SS_error("BAD ARGUMENT - SX_SET_PDBDATA", argl);

    return(SX_pdbdata_handler(file, set_name, "PM_set *", &s, TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_SET - read a PM_set out of a PDB file and
 *                - return a PML set object
 *                - FORM:
 *                -    (pdbdata->pm-set <file> <name>)
 */

static object *SX_pdbdata_set(argl)
   object *argl;
   {object *obj;
    char *name;
    PDBfile *file;
    g_file *po;
    syment *ep;
    SC_address data;
    PM_set *s;

    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_PDBDATA_SET", argl);

/* if the first object is a pdbfile, use it, otherwise, use default file */
    argl = SX_get_file(argl, &po);
    file = FILE_FILE(PDBfile, po);

    obj  = SS_car(argl);
    argl = SS_cdr(argl);
    name = SC_strsavef(SS_get_string(obj),
               "char*:SX_PDBDATA_SET:name");

/* check to see whether or not the variable is in the file */
    ep = PD_inquire_entry(file, name, TRUE, NULL);
    if (ep == NULL)
       return(SS_null);

/* read the set */
    if (file == SX_vif)
       {data.diskaddr = PD_entry_address(ep);
	s = *(PM_set **) data.memaddr;}
    else
       {if (!PD_read(file, name, &data.memaddr))
           SS_error(PD_err, obj);
	s = (PM_set *) data.memaddr;};

    if (s->info_type == NULL)
       s->info_type = SC_PCONS_P_S;

    return(SX_mk_set(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_PML_SET - build a PM_set from some numeric arrays at SX level
 *                 -
 *                 - Form:
 *                 -
 *                 - (pm-make-set <name>
 *                 -               <mesh-shape-list>
 *                 -               <element-arrays>)
 *                 -
 *                 - <name>            - name of the set
 *                 - <mesh-shape-list> - List of numbers specifying the mesh
 *                 -                   - shape.  These numbers are the 
 *                 -                   - maximum values of the indexes in 
 *                 -                   - each direction.  The minima are
 *                 -                   - assumed to be zero.
 *                 - <element-arrays>  - List of numeric arrays whose
 *                 -                   - entries specify the elements of
 *                 -                   - the set.
 *                 - Example:
 *                 -
 *                 - (pm-make-set "section" '(10 20) '(x y z))
 *                 -
 *                 - This specifies a two dimensional section of a three
 *                 - dimensional space.  The mesh has 10x20 elements and
 *                 - the numeric arrays x, y, and z are each 200 long.
 *                 -
 *                 - If no element arrays are given a logical set is
 *                 - built from the type and shape specifications.
 *                 -
 *                 - (pm-make-set "section" '(10 20))
 *                 -
 *                 - This specifies a two dimensional set of 2 vectors.
 *                 - The mesh has 10x20 elements which go from (0,0) to (9,19).
 *                 -
 *                 - Note: Logical rectangular meshes are assumed.
 *                 -       This can be generalized later by having the
 *                 -       <mesh-shape-list> specify the mesh connectivity
 *                 -       more generally.
 */

object *SX_make_pml_set(argl)
   object *argl;
   {object *obj, *shape, *components;
    PM_set *set;
    int *maxes, *pm, nd, ne, nde;
    char *name, *type;
    byte **elem, **pe;
    C_array *arr;

    name  = NULL;
    shape = SS_null;
    components = SS_null;
    SS_args(argl,
            SC_STRING_I, &name,
            SS_OBJECT_I, &shape,
            SS_OBJECT_I, &components,
            0);

/* extract the name */
    if (name == NULL)
       SS_error("BAD NAME - SX_MAKE_PML_SET", argl);

/* extract the mesh shape */
    if (SS_nullobjp(shape))
       SS_error("BAD MESH SHAPE - SX_MAKE_PML_SET", argl);
    else
       {nd    = _SS_length(shape);
        maxes = FMAKE_N(int, nd, "SX_MAKE_PML_SET:maxes");
        for (pm = maxes; !SS_nullobjp(shape); )
            SX_GET_INTEGER_FROM_LIST(*pm++, shape,
                                     "BAD MESH INDEX - SX_MAKE_PML_SET");};

/* extract the set elements */
    if (SS_nullobjp(components))
       set = PM_make_lr_domain(name, SC_DOUBLE_S, nd, nd, maxes, NULL);

    else
       {nde  = _SS_length(components);
	elem = FMAKE_N(byte *, nde, "SX_MAKE_PML_SET:elem");

/* get the number of elements */
	obj = SS_car(components);
	if (!SX_NUMERIC_ARRAYP(obj))
	   SS_error("OBJECT NOT NUMERIC ARRAY - SX_MAKE_PML_SET", obj);

	ne      = NUMERIC_ARRAY_LENGTH(obj);
	type    = NUMERIC_ARRAY_TYPE(obj);
	elem[0] = NUMERIC_ARRAY_DATA(obj);

	for (pe = elem; !SS_nullobjp(components); )
	    {obj        = SS_car(components);
	     components = SS_cdr(components);
	     if (SX_NUMERIC_ARRAYP(obj))
	        arr = SS_GET(C_array, obj);
	     else
	        SS_error("BAD ELEMENT ARRAY - SX_MAKE_PML_SET", obj);

	     if (strcmp(SX_promotion_type, "none") != 0)
	        {PM_promote_array(arr, SX_promotion_type, TRUE);
		 type = arr->type;};

	     *pe++ = arr->data;};

	set = _PM_make_set(name, type, FALSE,
			   ne, nd, nde, maxes, elem,
			   NULL, NULL,
			   NULL, NULL, NULL, NULL, NULL, NULL,
			   NULL);};

    return(SX_mk_set(set));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_CP_SET - build and return the PML set object which is the
 *                - cartesian product of the argument sets
 *                -
 *                - FORM:
 *                - (pm-make-cartesian-product-set [<set>]*)
 */

static object *SX_make_cp_set(argl)
   object *argl;
   {int i, n;
    char name[MAXLINE];
    PM_set **sets, *cp;
    object *obj;

    n = _SS_length(argl);
    sets = FMAKE_N(PM_set *, n, "SX_MAKE_CP_SET:sets");

    for (i = 0; i < n; i++)
        {SX_GET_SET_FROM_LIST(sets[i], argl,
			      "ARGUMENT NOT SET - SX_MAKE_CP_SET");};

    sprintf(name, "CP %d", n);

    cp = PM_make_cp_domain(name, SC_REAL_S, n, sets);

    SFREE(sets);

    return(SX_mk_set(cp));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_PML_MAPPING - build a PML mapping object out of a domain, range
 *                     - and attribute set
 *                     -
 *                     - FORM:
 *                     - (pm-make-mapping <domain> <range>
 *                     -                  [<centering> <category> <name>
 *                     -                   <emap> <next>])
 *                     -
 *                     - Centering defaults to zone
 *                     - Existence map thru which to plot defaults to all 1's
 *                     - mapping name
 */

static object *SX_make_pml_mapping(argl)
   object *argl;
   {PM_mapping *f, *next;
    PM_set *domain, *range;
    C_array *arr;
    char label[MAXLINE], *name, *category;
    int centering;

    centering = N_CENT;
    category  = PM_LR_S;
    arr       = NULL;
    name      = NULL;
    next      = NULL;
    SS_args(argl,
            G_SET, &domain,
            G_SET, &range,
            SC_INTEGER_I, &centering,
            SC_STRING_I, &category,
            SC_STRING_I, &name,
	    G_NUM_ARRAY, &arr,
	    G_MAPPING, &next,
            0);

    if (name == NULL)
       sprintf(label, "%s->%s", domain->name, range->name);
    else
       strcpy(label, name);

    f = PM_make_mapping(label, category, domain, range, centering, next);

/* if an existence map was supplied add it to the mapping's attribute list */
    if (arr != NULL)
       {char *emap;
        pcons *inf;

        emap = NULL;
        CONVERT(SC_CHAR_S, &emap, arr->type, arr->data, arr->length);

	inf = (pcons *) f->map;
	inf = SC_add_alist(inf, "EXISTENCE", SC_STRING_S, (byte *) emap);
	f->map = (byte *) inf;};

    return(SX_mk_mapping(f));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GSET - print a g_set */

static void _SX_wr_gset(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<SET|%s,%s>",
                              SET_NAME(obj),
                              SET_ELEMENT_TYPE(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GSET - gc a set */

static void _SX_rl_gset(obj)
   object *obj;
   {

/* you don't know whether a mapping is pointing to this
    PM_set *set;
    set = SS_GET(PM_set, obj);

    PM_rel_set(set, FALSE);
*/

    SS_rl_object(obj);;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_SET - encapsulate a PM_set as an object */

object *SX_mk_set(set)
   PM_set *set;
   {object *op;

    op = SS_mk_object(set, G_SET, SELF_EV, set->name);
    op->print   = _SX_wr_gset;
    op->release = _SX_rl_gset;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_ASCII_SET_NAME - given a file and reference to a set by
 *                           - name or menu number,
 *                           - return the set name as an object
 */

static object *SX_get_ascii_set_name(set)
   object *set;
   {PM_set *s;

    if (SX_SETP(set))
       {s = SS_GET(PM_set, set);
        return(SS_mk_string(s->name));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_ATTR_SET - set an attribute of a PM_set
 *                 - usage: (pm-set-set-attribute! <set>
 *                 -                                <name> <type> <value>)
 */

static object *SX_set_attr_set(argl)
   object *argl;
   {PM_set *s;
    char *name, *type;
    object *val;
    pcons *inf;
    byte *v;
    C_array *arr;

    s    = NULL;
    name = NULL;
    type = NULL;
    val  = SS_null;
    SS_args(argl,
            G_SET, &s,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SS_OBJECT_I, &val,
            0);

    if ((s == NULL) || (name == NULL) || (type == NULL))
       SS_error("INSUFFICIENT ARGUMENTS - SX_SET_ATTR_SET", argl);

/* get the current list */
    if (s->info_type != NULL)
       {if (strcmp(s->info_type, SC_PCONS_P_S) == 0)
	   inf = (pcons *) s->info;
        else
	   inf = NULL;}
    else
       inf = NULL;

    if (strcmp(type, "nil") == 0)
       {if (inf != NULL)
	   s->info = (byte *) SC_rem_alist(inf, name);
	return(SS_t);};

    if (SS_consp(val))
       {object *obj;
	obj = SS_null;
	SS_Assign(obj, SX_list_array(val));
	SS_args(obj,
		G_NUM_ARRAY, &arr,
		0);
	v = arr->data;
	SC_mark(v, 1);
	SS_GC(obj);
	SC_mark(v, -1);}
    else if ((strcmp(type, "int *") == 0) ||
	(strcmp(type, "integer *") == 0))
       {v = SC_alloc_na(1L, sizeof(int), NULL, FALSE);
        SS_args(val,
		SC_INTEGER_I, v,
		0);}
    else if (strcmp(type, SC_DOUBLE_P_S) == 0)
       {v = SC_alloc_na(1L, sizeof(double), NULL, FALSE);
        SS_args(val,
		SC_DOUBLE_I, v,
		0);}
    else if (strcmp(type, SC_STRING_S) == 0)
       {SS_args(val,
		SC_STRING_I, &v,
		0);}
    else
       SS_error("CAN'T HANDLE TYPE - SX_SET_ATTR_SET", argl);

    inf = SC_change_alist(inf, name, type, v);

    s->info_type = SC_PCONS_P_S;
    s->info      = (byte *) inf;

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_MAP_TYPE - set the type of a PM_mapping or the mapping part of
 *                 - a PG_graph object
 */

object *SX_set_map_type(argl)
   object *argl;
   {PM_mapping *f;
    char *name;

    f    = NULL;
    name = NULL;
    SS_args(argl,
            G_MAPPING, &f,
            SC_STRING_I, &name,
            0);

    if ((f == NULL) || (name == NULL))
       SS_error("INSUFFICIENT ARGUMENTS - SX_SET_MAP_TYPE", argl);

    f->map_type = SC_strsavef(name,"char*:SX_SET_MAP_TYPE:type");

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAPPING_PDBDATA - given a PM_mapping object
 *                    - allocate and fill a PDB data object with the
 *                    - PM_mapping and return it
 */

object *SX_mapping_pdbdata(argl)
   object *argl;
   {PM_mapping *f, *pf;
    g_file *po;
    PDBfile *file;
    char *mn, map_name[MAXLINE];
    long i;
    object *ret;

    f  = NULL;
    po = NULL;
    mn = NULL;
    SS_args(argl,
            G_MAPPING, &f,
            G_FILE, &po,
            SC_STRING_I, &mn,
            0);

    if ((po == NULL) || (po == SX_gvif))
       {po   = SX_gvif;
        file = SX_vif;}

    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);

    else
       SS_error("BAD FILE - SX_MAPPING_PDBDATA", argl);

    if (f == NULL)
       SS_error("BAD ARGUMENT - SX_MAPPING_PDBDATA", argl);

    if (mn == NULL)
       {_SX_get_menu(po);
        for (i = 0; TRUE; i++)
            {sprintf(map_name, "Mapping%ld", i);
             if (PD_inquire_entry(file, map_name, TRUE, NULL) == NULL)
                break;};}
    else
       {strcpy(map_name, mn);
        SFREE(mn);};

/* disconnect any function pointers or undefined structs/members */
    for (pf = f; pf != NULL; pf = pf->next)
        {pf->domain->opers = NULL;
         pf->range->opers = NULL;};

/* make sure that the necessary types are known */
    if (PD_inquire_type(file, "PM_mapping") == NULL)
       PD_def_mapping(file);

    ret = SX_pdbdata_handler(file, map_name, "PM_mapping *", &f , TRUE);

/* add to menu */
    _SX_push_menu_item(po, map_name, "PM_mapping *");

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_MAPPING - read a PM_mapping out of a PDB file and
 *                    - return a PML mapping object
 *                    - FORM:
 *                    -    (pdbdata->pm-mapping <file> <name>)
 */

object *SX_pdbdata_mapping(argl)
   object *argl;
   {object *obj;
    int i, ret;
    char *name, dname[MAXLINE];
    PDBfile *file;
    g_file *po;
    syment *ep;
    SC_address data;
    PM_set *domain, *range;
    PM_mapping *pf, *f;
    SX_menu_item *mitems;

    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_PDBDATA_MAPPING", argl);

/* if the first object is a pdbfile, use it, otherwise, use default file */
    argl = SX_get_file(argl, &po);
    file = FILE_FILE(PDBfile, po);

    obj = SS_car(argl);
    if (SS_integerp(obj))
       {mitems = po->menu;
        if (mitems == NULL)
           {_SX_get_menu(po);
            mitems = po->menu;};

        i = SS_INTEGER_VALUE(obj);
        if ((i < 1) || (i > po->n_menu_items))
           return(SS_null);

        name = mitems[i-1].vname;}
    else
       {argl = SS_cdr(argl);
        name = SC_strsavef(SS_get_string(obj),
                "char*:SX_PDBDATA_MAPPING:name");};

/* check to see whether or not the variable is in the file */
    ep = PD_inquire_entry(file, name, TRUE, NULL);
    if (ep == NULL)
       return(SS_null);

/* read the mapping */
    if (file == SX_vif)
       {data.diskaddr = PD_entry_address(ep);
        data.memaddr  = DEREF(data.memaddr);}
    else
       {if (!PD_read(file, name, &data.memaddr))
           SS_error(PD_err, obj);};

/* reconnect any function pointers or undefined structs/members */
    f   = (PM_mapping *) data.memaddr;
    if (f->domain == NULL)
       {strcpy(dname, f->name);
        PD_process_set_name(dname);

        if (!PD_read(file, dname, &data.memaddr))
           SS_error(PD_err, SS_null);
        f->domain = (PM_set *) data.memaddr;};

    ret = TRUE;
    for (pf = f; pf != NULL; pf = pf->next)
        {domain = pf->domain;
         range  = pf->range;
         if (domain != NULL)
            {ret &= PM_set_opers(domain);
	     if (domain->info_type == NULL)
                domain->info_type = SC_PCONS_P_S;};

         if (range != NULL)
            {ret &= PM_set_opers(range);
	     if (range->info_type == NULL)
                range->info_type = SC_PCONS_P_S;};

         if (ret == FALSE)
            SS_error("NO FIELD FOR TYPE - SX_PDBDATA_MAPPING", SS_null);};

    return(SX_mk_mapping(f));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GMAPPING - print a g_mapping */

static void _SX_wr_gmapping(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<MAPPING|%s>", MAPPING_NAME(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GMAPPING - gc a mapping */

static void _SX_rl_gmapping(obj)
   object *obj;
   {PM_mapping *f;

    f = SS_GET(PM_mapping, obj);
    PM_rel_mapping(f, TRUE, TRUE);

    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_MAPPING - encapsulate a PM_mapping as an object */

object *SX_mk_mapping(f)
   PM_mapping *f;
   {object *op;

    op = SS_mk_object(f, G_MAPPING, SELF_EV, f->name);
    op->print   = _SX_wr_gmapping;
    op->release = _SX_rl_gmapping;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ARRAYS_SET - convert the data from a list of C_array objects to
 *               - a set of one higher dimension than the arrays
 *               - FORM:
 *               -    (pm-arrays->set name <array> ...)
 */

static object *SX_arrays_set(argl)
   object *argl;
   {int i, j, n, *maxes, *pm, nd, ne, nde, nep, tflag;
    char *name, *type;
    REAL **elem, *pe;
    byte *data;
    object *obj, *components, *shape, *lst;
    PM_set *set;

    name  = NULL;
    shape = SS_null;
    components = SS_null;
    tflag = FALSE;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_INTEGER_I, &tflag,
            SS_OBJECT_I, &shape,
            0);

    components = SS_cdddr(argl);

/* extract the name */
    if (name == NULL)
       SS_error("BAD NAME - SX_ARRAYS_SET", argl);

/* extract the mesh shape */
    if (SS_nullobjp(shape))
       {nd    = _SS_length(components) + tflag;
        maxes = FMAKE_N(int, nd, "SX_ARRAYS_SET:maxes");
        maxes[nd-1] = _SS_length(SS_car(components));}
    else
       {nd    = _SS_length(shape);
        maxes = FMAKE_N(int, nd, "SX_ARRAYS_SET:maxes");
        for (pm = maxes; !SS_nullobjp(shape); )
            SX_GET_INTEGER_FROM_LIST(*pm++, shape,
                                     "BAD MESH INDEX - SX_ARRAYS_SET");};

/* each component is a list of arrays */
    nde  = _SS_length(components) + tflag;
    elem = FMAKE_N(REAL *, nde, "SX_ARRAYS_SET:elem");
    for (lst = components; !SS_nullobjp(lst); lst = SS_cdr(lst))
        {if (lst == components)
            n = _SS_length(SS_car(lst));
         else if (n != _SS_length(SS_car(lst)))
            SS_error("COMPONENT LISTS NOT SAME LENGTH - SX_ARRAYS_SET",
                     lst);};

/* get the number of elements */
    lst = SS_caar(components);
    if (!SX_NUMERIC_ARRAYP(lst))
       SS_error("OBJECT NOT NUMERIC ARRAY - SX_ARRAYS_SET", lst);

    type = NUMERIC_ARRAY_TYPE(lst);
    data = NUMERIC_ARRAY_DATA(lst);
    nep  = NUMERIC_ARRAY_LENGTH(lst);
    ne   = nep*n;

    for (j = 0; !SS_nullobjp(components); j++, components = SS_cdr(components))
        {pe = FMAKE_N(REAL, ne, "SX_ARRAYS_SET:pe");
         elem[j] = pe;
         lst = SS_car(components);
         for (i = 0; i < n; i++)
             {SX_GET_ARRAY_FROM_LIST(data, lst,
                                     "BAD ELEMENT ARRAY - SX_ARRAYS_SET");

              CONVERT(SC_REAL_S, &pe,
                      type, data,
                      nep, FALSE);

              pe += nep;};};

/* the new component must be made */
    if (tflag)
       {pe = FMAKE_N(REAL, ne, "SX_ARRAYS_SET:pe");
        elem[j] = pe;
        for (i = 0; i < n; i++)
            {for (j = 0; j < nep; j++)
                 *pe++ = i;};};

    set = _PM_make_set(name, SC_REAL_S, FALSE,
                       ne, nd, nde, maxes, elem,
                       NULL, NULL,
                       NULL, NULL, NULL, NULL, NULL, NULL,
                       NULL);

    return(SX_mk_set(set));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LR_AC - convert an LR based mapping into an AC based one
 *          - FORM:
 *          -    (pm-lr->ac <mapping>)
 */

static object *SX_lr_ac(argl)
   object *argl;
   {PM_mapping *f, *g;
    PM_set *odom, *oran, *ndom, *nran;
    PM_mesh_topology *mt;
    int ord, cent, *maxes;
    byte **elements;

    f   = NULL;
    ord = BND_CELL_MAX;
    SS_args(argl,
            G_MAPPING, &f,
            SC_INTEGER_I, &ord,
            0);

    if (f == NULL)
       SS_error("BAD MAPPING - SX_LR_AC", argl);

    odom = f->domain;
    oran = f->range;

/* find the additional mapping information */
    cent = N_CENT;
    PM_mapping_info(f,
                    "CENTERING", &cent,
                    NULL);

    if (odom->dimension != 2)
       SS_error("ONLY 2D MESHES CURRENTLY - SX_LR_AC", SS_null);

    else
       {int kmax, lmax;
        REAL *x, *y, *px, *py;

        maxes = odom->max_index;
        kmax = maxes[0];
        lmax = maxes[1];

        elements = (byte **) odom->elements;
        px = x = (REAL *) elements[0];
        py = y = (REAL *) elements[1];

        mt = PM_lr_ac_mesh_2d(&x, &y, kmax, lmax, 1, kmax, 1, lmax, ord);

/* check the new mesh */
        {int i, nn;
         nn = kmax*lmax;
         for (i = 0; i < nn; i++)
             {if ((x[i] != px[i]) || (y[i] != py[i]))
                 SS_error("BAD CONVERSION - SX_LR_AC", SS_null);};};

        elements = FMAKE_N(byte *, 2, "SX_LR_AC:elements");
        elements[0] = (byte *) x;
        elements[1] = (byte *) y;};

    ndom = _PM_make_set(odom->name, SC_DOUBLE_S, FALSE, odom->n_elements,
			odom->dimension, odom->dimension_elem,
			odom->max_index, elements,
			odom->opers, odom->metric,
			odom->symmetry_type, odom->symmetry,
			PM_MESH_TOPOLOGY_P_S, mt,
                        odom->info_type, odom->info, NULL);

    nran = SX_copy_set(oran);

    g = PM_make_mapping(f->name, PM_AC_S, ndom, nran, cent, NULL);

    return(SX_mk_mapping(g));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_AC_SET - make an Arbitrarily-Connect set from
 *                - lists of cell boundary specs
 *                -    (pm-make-ac-set nodes edges faces zones ... )
 *                - where
 *                -
 *                - nodes is a list of n-tuples (list of number)
 *                -       representing the node position
 *                - edges is a list of 1cells, faces 2cells, etc
 *                - an ncell is a list of numbers describing the
 *                - boundary of the ncell
 *                - see the description in pml.h for details
 */

static object *SX_make_ac_set(argl)
   object *argl;
   {char name[MAXLINE];
    int i, j, k;
    int ne, nd, nde;
    int *nbp, *ncs, nc, ord;
    long **bnd, *pb;
    double **elem;
    object *nodes, *node, *ncells, *ncell;
    PM_set *set;
    PM_mesh_topology *mt;

    nd = _SS_length(argl);
    if (nd < 1)
       SS_error("NO DOMAIN INFO - SX_MAKE_AC_SET", argl);

    nodes = SS_car(argl);
    argl  = SS_cdr(argl);

    ne = _SS_length(nodes);
    if (ne < 1)
       SS_error("BAD NODE LIST - SX_MAKE_AC_SET", nodes);

    nde = _SS_length(SS_car(nodes));

/* construct the element arrays from the lists */
    elem = FMAKE_N(double *, nde, "SX_MAKE_AC_SET:elem");
    for (i = 0; i < nde; i++)
        elem[i] = FMAKE_N(double, ne,
                  "SX_MAKE_AC_SET:elem[]");

    for (i = 0; i < ne; i++, nodes = SS_cdr(nodes))
        {node = SS_car(nodes);
	 for (j = 0; j < nde; j++, node = SS_cdr(node))
             {SS_args(node,
		      SC_DOUBLE_I, elem[j]+i,
		      0);};};

/* construct the cell boundary arrays */
    bnd = FMAKE_N(long *, nd, "SX_MAKE_AC_SET:bnd");
    ncs = FMAKE_N(int, nd, "SX_MAKE_AC_SET:ncs");
    nbp = FMAKE_N(int, nd, "SX_MAKE_AC_SET:nbp");
    for (j = 1; j < nd; j++, argl = SS_cdr(argl))
        {ncells = SS_car(argl);
         ord = _SS_length(SS_car(ncells));
         nc  = _SS_length(ncells);
         pb  = FMAKE_N(long, ord*nc, "SX_MAKE_AC_SET:pb");
         bnd[j] = pb;
         ncs[j] = nc;
         nbp[j] = ord;
         for (i = 0; i < nc; i++, ncells = SS_cdr(ncells))
             {ncell = SS_car(ncells);
	      for (k = 0; k < ord; k++, ncell = SS_cdr(ncell))
                  {SS_args(ncell,
			   SC_LONG_I, pb++,
			   0);};};};

/* reduce the number of dimensions to its proper value now */
    nd--;

/* fill in the 0d part */
    bnd[0] = NULL;
    ncs[0] = ne;
    nbp[0] = 1;

/* put it all together */
    mt = PM_make_topology(nd, nbp, ncs, bnd);

    sprintf(name, "D%d-%d", nd, nde);

    set = _PM_make_set(name, SC_DOUBLE_S, FALSE,
		       ne, nd, nde, NULL, elem,
		       NULL, NULL, NULL, NULL,
		       PM_MESH_TOPOLOGY_P_S, mt,
		       NULL, NULL, NULL);

    return(SX_mk_set(set));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ARRAY_PDBDATA - convert the data from a C_array object to
 *                  - a pdbdata object
 *                  - FORM:
 *                  -    (pm-array->pdbdata <array> <file> <name>)
 */

static object *SX_array_pdbdata(argl)
   object *argl;
   {C_array *arr;
    g_file *po;
    PDBfile *file;
    char *mn, arr_name[MAXLINE];
    static int i = 0;

    arr = NULL;
    po  = NULL;
    mn  = NULL;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
            G_FILE, &po,
            SC_STRING_I, &mn,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;

    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);

    else
       SS_error("BAD FILE - SX_ARRAY_PDBDATA", argl);

    if (arr == NULL)
       SS_error("INVALID ARRAY OBJECT - SX_ARRAY_PDBDATA", argl);

    if (mn == NULL)
       sprintf(arr_name, "Pm-Array%d", i++);
    else
       {strcpy(arr_name, mn);
        SFREE(mn);};

    return(SX_pdbdata_handler(file, arr_name, "C_array", arr, TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_ARRAY - convert the data from a pdbdata object to
 *                  - a C_array object
 *                  - NOTE: if the pdbdata object contains a C_array already
 *                  -       just use it
 *                  - FORM:
 *                  -    (pdbdata->pm_array <pdbdata>)
 */

static object *SX_pdbdata_array(arg)
   object *arg;
   {g_pdbdata *pd;
    syment *ep;
    C_array *arr;

    pd = NULL;
    SS_args(arg,
            G_PDBDATA, &pd,
            0);

    if (pd == NULL)
       SS_error("INVALID PDBDATA OBJECT - SX_PDBDATA_ARRAY", arg);

    ep = pd->ep;

    if (strcmp(PD_entry_type(ep), "C_array") == 0)
       arr = (C_array *) pd->data;
    else
      {arr = FMAKE(C_array, "SX_PDBDATA_ARRAY:arr");
       arr->type   = SC_strsavef(PD_entry_type(ep),
                     "char*:SX_PDBDATA_ARRAY:type");
       arr->length = PD_entry_number(ep);
       arr->data   = pd->data;};

    return(SX_mk_C_array(arr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WR_GNUM_ARRAY - print a g_num_array */

static void _SX_wr_gnum_array(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<ARRAY|%s>", NUMERIC_ARRAY_TYPE(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RL_GNUM_ARRAY - release g_num_array */

static void _SX_rl_gnum_array(obj)
   object *obj;
   {C_array *arr;

    arr = SS_GET(C_array, obj);
/*  GOTCHA - it's currently possible that some PM_set may still be pointing
 *  at type and/or array even though the reference count doesn't reflect it
 *  SFREE(arr->type);
 *  SFREE(arr->data);
 */
    SFREE(arr);
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_C_ARRAY - encapsulate a C_array as an object */

object *SX_mk_C_array(arr)
   C_array *arr;
   {object *op;

    op = SS_mk_object(arr, G_NUM_ARRAY, SELF_EV, arr->type);
    SC_mark(arr, 1);
    SC_mark(arr->type, 1);
    SC_mark(arr->data, 1);

    op->print   = _SX_wr_gnum_array;
    op->release = _SX_rl_gnum_array;;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_REP_TO_AC - map a the given connectivity representation into
 *              - the PACT mesh topology representation
 *              - and return a domain set
 */

PM_set *SX_rep_to_ac(name, rx, ry, n_nodes, n_zones, zones)
   char *name;
   double *rx, *ry;
   int n_nodes, n_zones, *zones;
   {int iz, j, j1, j2, incr;
    int n_sides, *nc, *nbp, *pzone;
    double **elem;
    PM_set *s;
    PM_mesh_topology *mt;
    long **bnd, *ncell, *pcell;

    elem = FMAKE_N(double *, 2, "SX_REP_TO_AC:elem");
    elem[0] = rx;
    elem[1] = ry;

/* allocate the boundary arrays */
    bnd = FMAKE_N(long *, 3, "SX_REP_TO_AC:bnd");
    bnd[2] = FMAKE_N(long, 2*n_zones,
                     "SX_REP_TO_AC:bnd[2]");

/* fill the 2-cells */
    ncell   = bnd[2];
    n_sides = 0;
    pzone   = zones;
    for (iz = 0; iz < n_zones; iz++)
        {incr  = 0;
         while (pzone[incr] != -1)
            incr++;
         pcell = ncell + 2*iz;
	 pcell[BND_CELL_MIN] = n_sides;
	 pcell[BND_CELL_MAX] = n_sides + incr - 1;
	 n_sides += incr;
         pzone   += incr + 1;};

    bnd[1] = FMAKE_N(long, 2*n_sides,
                     "SX_REP_TO_AC:bnd[1]");

/* fill the 1-cells */
    pcell = bnd[1];
    pzone = zones;
    for (iz = 0; iz < n_zones; iz++)
        {incr  = 0;
         while (pzone[incr] != -1)
            incr++;
         for (j = 0; j < incr; j++)
             {j1 = j;
	      j2 = (j + 1) % incr;
	      pcell[BND_CELL_MIN] = pzone[j1];
	      pcell[BND_CELL_MAX] = pzone[j2];
              pcell += 2;};
         pzone   += incr + 1;};

    bnd[0] = NULL;

/* setup the number of cells array */
    nc = FMAKE_N(int, 3, "SX_REP_TO_AC:nc");
    nc[0] = n_nodes;
    nc[1] = n_sides;
    nc[2] = n_zones;

/* setup the number of boundary parameters array */
    nbp = FMAKE_N(int, 3, "SX_REP_TO_AC:nbp");
    nbp[0] = 1;
    nbp[1] = 2;
    nbp[2] = 2;

/* put it all together */
    mt = PM_make_topology(2, nbp, nc, bnd);
    s  = _PM_make_set(name, SC_DOUBLE_S, FALSE, n_nodes, 2, 2,
		      NULL, elem,
		      NULL, NULL, NULL, NULL, 
		      PM_MESH_TOPOLOGY_P_S, mt,
		      NULL, NULL, NULL);

    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_REP_AC_DOMAIN - build an Arbitrarily-Connected domain from a
 *                  - specified representation of the connectivity
 *                  - which is: an array of nodes bounding the zones;
 *                  - an offset; and a maximum number of nodes per zone
 */

static object *SX_rep_ac_domain(argl)
   object *argl;
   {PM_set *s;
    g_file *po;
    PDBfile *file;
    C_array *connct;
    char *xname, *yname, *nzname, *nnname, sname[MAXLINE];
    int incr;
    int n_zones, n_nodes, *zones;
    double *rx, *ry;

    xname  = NULL;
    yname  = NULL;
    connct = NULL;
    nzname = NULL;
    nnname = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &xname,
            SC_STRING_I, &yname,
            SC_STRING_I, &nzname,
            SC_STRING_I, &nnname,
            G_NUM_ARRAY, &connct,
	    LAST);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;

    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);

    else
       SS_error("BAD FILE - SX_REP_AC_DOMAIN", argl);

    if (!PD_read(file, nnname, &n_nodes))
       SS_error("CAN'T READ NUMBER OF NODES - SX_REP_AC_DOMAIN", argl);

    if (!PD_read(file, nzname, &n_zones))
       SS_error("CAN'T READ NUMBER OF ZONES - SX_REP_AC_DOMAIN", argl);

    rx = FMAKE_N(double, n_nodes, "SX_REP_AC_DOMAIN:rx");
    ry = FMAKE_N(double, n_nodes, "SX_REP_AC_DOMAIN:ry");

    incr = PD_read(file, xname, rx);
    if (incr != n_nodes)
       SS_error("READING X VALUES - SX_REP_AC_DOMAIN", argl);

    incr = PD_read(file, yname, ry);
    if (incr != n_nodes)
       SS_error("READING Y VALUES - SX_REP_AC_DOMAIN", argl);

    zones = (int *) connct->data;

    sprintf(sname, "{%s,%s}", xname, yname);

    s = SX_rep_to_ac(sname, rx, ry, n_nodes, n_zones, zones);

    return(SX_mk_set(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_INSTALL_PML_FUNCS - install the PML extensions to Scheme */
 
void SX_install_pml_funcs()
   {
    SS_install("pm-array?",
               "Returns #t if the object is a numeric array, and #f otherwise",
               SS_sargs,
               SX_numeric_arrayp, SS_PR_PROC);

    SS_install("list->pm-array",
               "Returns a numeric array built from a list of numbers",
               SS_nargs,
               SX_list_array, SS_PR_PROC);

    SS_install("pm-array->list",
               "Returns a list of numbers built from a numeric array",
               SS_sargs,
               SX_array_list, SS_PR_PROC);

    SS_install("pm-make-array",
               "Allocate and return a pm-array of the specified type and size",
               SS_nargs,
               SX_mk_array, SS_PR_PROC);

    SS_install("pm-resize-array",
               "Reallocate the given pm-array to the specified size",
               SS_nargs,
               SX_resz_array, SS_PR_PROC);

    SS_install("pm-array-ref",
               "Reference the nth element of a pm-array",
               SS_nargs,
               SX_array_ref, SS_PR_PROC);

    SS_install("pm-array-set!",
               "Set the nth element of a pm-array",
               SS_nargs,
               SX_array_set, SS_PR_PROC);

    SS_install("pm-array-length",
               "Returns the length of the given numeric array",
               SS_sargs,
               SX_num_arr_len, SS_PR_PROC);

    SS_install("pm-array-extrema",
               "Returns the extrema of the given numeric array",
               SS_sargs,
               SX_num_arr_extr, SS_PR_PROC);

    SS_install("pm-sub-array",
               "Return a sub-array of the given numeric array",
               SS_nargs,
               SX_sub_array, SS_PR_PROC);

    SS_install("pm-set?",
               "Returns #t if the object is a PML set, and #f otherwise",
               SS_sargs,
               SX_setp, SS_PR_PROC);

    SS_install("pm-set-mapping-type",
               "Set the type of a mapping object to the given string",
               SS_nargs,
               SX_set_map_type, SS_PR_PROC);

    SS_install("pm-mapping?",
               "Returns #t if the object is a PML mapping, and #f otherwise",
               SS_sargs,
               SX_mappingp, SS_PR_PROC);

    SS_install("pm-grotrian-mapping?",
               "Returns #t if the object is a PML grotrian mapping, and #f otherwise",
               SS_sargs,
               SX_grotrian_mappingp, SS_PR_PROC);

    SS_install("pm-mapping->pdbdata",
               "Write a PML mapping object to a PDB file\nFORM (pm-mapping->pdbdata <mapping> <file> <name>)",
               SS_nargs,
               SX_mapping_pdbdata, SS_PR_PROC);

    SS_install("pdbdata->pm-mapping",
               "Read a PML mapping object from a PDB file\nFORM (pdbdata->pm-mapping <file> <name>)",
               SS_nargs,
               SX_pdbdata_mapping, SS_PR_PROC);

    SS_install("pm-set-name",
               "Return the name of the object iff it is a PM_set and () otherwise",
               SS_sargs,
               SX_get_ascii_set_name, SS_PR_PROC);

    SS_install("pm-set->pdbdata",
               "Write a PML set object to a PDB file\nFORM (pm-set->pdbdata <set> <file> <name>)",
               SS_nargs,
               SX_set_pdbdata, SS_PR_PROC);

    SS_install("pdbdata->pm-set",
               "Read a PML set object from a PDB file\nFORM (pdbdata->pm-set <file> <name>)",
               SS_nargs,
               SX_pdbdata_set, SS_PR_PROC);

    SS_install("pdbdata->pm-array",
               "Convert a PDBDATA object to a numeric array object\nFORM (pdbdata->pm-array <pdbdata>)",
               SS_nargs,
               SX_pdbdata_array, SS_PR_PROC);

    SS_install("pm-array->pdbdata",
               "Convert a numeric array object to a PDBDATA object\nFORM (pm-array->pdbdata <array>)",
               SS_nargs,
               SX_array_pdbdata, SS_PR_PROC);

    SS_install("pm-lr->ac",
               "Convert a logical rectangular mesh based mapping into an arbitrarily connected mesh base one\nFORM (pm-lr->ac <mapping>)",
               SS_nargs,
               SX_lr_ac, SS_PR_PROC);

    SS_install("pm-make-ac-set",
               "Construct an arbitrarily connected set",
               SS_nargs,
               SX_make_ac_set, SS_PR_PROC);

    SS_install("pm-arrays->set",
               "Convert a list of numeric array objects to a set\nFORM (pm-arrays->set (<array> ...) ...)",
               SS_nargs,
               SX_arrays_set, SS_PR_PROC);

    SS_install("pm-mapping-dimension",
               "Returns (d(domain). d(range)) of the given mapping",
               SS_sargs,
               SX_get_dimension, SS_PR_PROC);

    SS_install("pm-mapping-domain",
               "Returns the domain of the given mapping",
               SS_sargs,
               SX_get_domain, SS_PR_PROC);

    SS_install("pm-mapping-range",
               "Returns the range of the given mapping",
               SS_sargs,
               SX_get_range, SS_PR_PROC);

    SS_install("pm-mapping-name",
               "Returns the name of the given mapping",
               SS_sargs,
               SX_get_mapping_name, SS_PR_PROC);

    SS_install("pm-set-volume",
               "Return the product of the extrema of the given set",
               SS_sargs,
               SX_set_volume, SS_PR_PROC);

    SS_install("pm-shift-domain",
               "Add a scalar value to all components of a mapping domain",
               SS_nargs,
               SX_shift_domain, SS_PR_PROC);

    SS_install("pm-shift-range",
               "Add a scalar value to all components of a mapping range",
               SS_nargs,
               SX_shift_range, SS_PR_PROC);

    SS_install("pm-scale-domain",
               "Multiply all components of a mapping domain by a scalar value",
               SS_nargs,
               SX_scale_domain, SS_PR_PROC);

    SS_install("pm-scale-range",
               "Multiply all components of a mapping range by a scalar value",
               SS_nargs,
               SX_scale_range, SS_PR_PROC);

    SS_install("pm-make-set",
               "Returns a PML set\nFORM (pm-make-set <name> <mesh-shape-list> <element-arrays>)",
               SS_nargs,
               SX_make_pml_set, SS_PR_PROC);

    SS_install("pm-make-mapping",
               "Returns a PML mapping\nFORM (pm-make-mapping <domain> <range> [<centering> <category> <name> <emap> <next>])",
               SS_nargs,
               SX_make_pml_mapping, SS_PR_PROC);

    SS_install("pm-make-cartesian-product-set",
               "Returns a newly constructed set\nFORM (pm-make-cartesian-product-set [<set>]*)",
               SS_nargs,
               SX_make_cp_set, SS_PR_PROC);

    SS_install("pm-set-set-attribute!",
               "Set an attribute of the given set",
               SS_nargs,
               SX_set_attr_set, SS_PR_PROC);

    SS_install("pm-connection->ac-domain",
               "Build an Arbitrarily-Connected domain set from the given connectivity representation",
	       SS_nargs,
	       SX_rep_ac_domain, SS_PR_PROC);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SXDTAC - F77 interface to SX_rep_to_ac */

FIXNUM F77_ID(sxdtac_, sxdtac, SXDTAC)(pnc, pname, rx, ry, pnn, pnz, pzones)
   FIXNUM *pnc;
   F77_string pname;
   REAL *rx, *ry;
   FIXNUM *pnn, *pnz, *pzones;
   {int n_nodes, n_zones;
    char name[MAXLINE];
    PM_set *set;

    SC_FORTRAN_STR_C(name, pname, *pnc);

    n_nodes = *pnn;
    n_zones = *pnz;

    set = SX_rep_to_ac(name, rx, ry, n_nodes, n_zones, (int *) pzones);

    return((FIXNUM) SC_ADD_POINTER(set));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
