/*
 * SHPRM2.C - arithmetic functions for Scheme
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "scheme.h"

#define SS_GET_OPERAND(operand, argl)                                        \
    {object *num;                                                            \
     num  = SS_car(argl);                                                    \
     argl = SS_cdr(argl);                                                    \
     switch (SC_arrtype(num, -1))                                            \
        {case SC_INTEGER_I :                                                 \
              operand = SS_INTEGER_VALUE(num);                               \
              break;                                                         \
         case SC_FLOAT_I   :                                                 \
              operand = SS_FLOAT_VALUE(num);                                 \
              type    = SC_FLOAT_I;                                          \
              break;                                                         \
         default           :                                                 \
              SS_error("ARGUMENT MUST BE A NUMBER - SS_GET_OPERAND",         \
                       num);                                                 \
              break;};}

static int
 SC_DECLARE(_SS_zero, (double x1)),
 SC_DECLARE(_SS_neg, (double x1)),
 SC_DECLARE(_SS_pos, (double x1)),
 SC_DECLARE(_SS_even, (double f)),
 SC_DECLARE(_SS_odd, (double f)),
 SC_DECLARE(_SS_equl, (double x1, double x2)),
 SC_DECLARE(_SS_lt, (double x1, double x2)),
 SC_DECLARE(_SS_gt, (double x1, double x2)),
 SC_DECLARE(_SS_le, (double x1, double x2)),
 SC_DECLARE(_SS_ge, (double x1, double x2));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_INSTALL_MATH - install the SCHEME primitives for math */

void SS_install_math()
   {

    SS_install("*",
               "Procedure: Returns product of args or 1 if no args are supplied",
               SS_binary_flt,
               (PFPObject) PM_ftimes, SS_PR_PROC);

    SS_install("+",
               "Procedure: Returns sum of args or 0 if no args are supplied",
               SS_binary_flt,
               (PFPObject) PM_fplus, SS_PR_PROC);

    SS_install("-",
               "Procedure: Returns difference of args",
               SS_binary_flt,
               (PFPObject) PM_fminus, SS_PR_PROC);

    SS_install("/",
               "Procedure: Returns quotient of args (left associative)",
               SS_binary_flt,
               (PFPObject) PM_fdivide, SS_PR_PROC);

    SS_install("<",
               "Procedure: Returns #t iff the first argument is less than the second",
               SS_bin_comp,
               (PFPObject) _SS_lt, SS_PR_PROC);

    SS_install("<=",
               "Procedure: Returns #t iff the first argument is less than or equal to the second",
               SS_bin_comp,
               (PFPObject) _SS_le, SS_PR_PROC);

    SS_install("=",
               "Procedure: Returns #t iff the first argument is equal to the second",
               SS_bin_comp,
               (PFPObject) _SS_equl, SS_PR_PROC);

    SS_install(">",
               "Procedure: Returns #t iff the first argument is greater than the second",
               SS_bin_comp,
               (PFPObject) _SS_gt, SS_PR_PROC);

    SS_install(">=",
               "Procedure: Returns #t iff the first argument is greater than or equal to the second",
               SS_bin_comp,
               (PFPObject) _SS_ge, SS_PR_PROC);

    SS_install("abs",
               "Procedure: Returns the absolute value of a numeric object",
               SS_unary_flt,
               (PFPObject) ABS, SS_PR_PROC);

    SS_install("acos",
               "Procedure: Returns the arc cosine of the argument",
               SS_unary_flt, 
               (PFPObject) acos, SS_PR_PROC);

    SS_install("asin",
               "Procedure: Returns the arc sine of the argument",
               SS_unary_flt, 
               (PFPObject) asin, SS_PR_PROC);

    SS_install("atan",
               "Procedure: Returns the arc tangent of the argument",
               SS_unary_flt, 
               (PFPObject) atan, SS_PR_PROC);

    SS_install("ceiling",
               "Procedure: Returns the smallest integer greater than the argument",
               SS_unary_fix, 
               (PFPObject) ceil, SS_PR_PROC);

    SS_install("cos",
               "Procedure: Returns the cosine of the argument",
               SS_unary_flt, 
               (PFPObject) cos, SS_PR_PROC);

    SS_install("cosh",
               "Procedure: Returns the hyperbolic cosine of the argument",
               SS_unary_flt, 
               (PFPObject) cosh, SS_PR_PROC);

    SS_install("even?",
               "Procedure: Returns #t iff the argument is a number divisible exactly by 2",
               SS_un_comp,
               (PFPObject) _SS_even, SS_PR_PROC);

    SS_install("exp",
               "Procedure: Returns the exponential of the argument",
               SS_unary_flt, 
               (PFPObject) exp, SS_PR_PROC);

    SS_install("expt",
               "Procedure: Returns the first argument raised to the power of the second",
               SS_binary_flt,
               (PFPObject) POW, SS_PR_PROC);

    SS_install("floor",
               "Procedure: Returns the greatest integer less than the argument",
               SS_unary_fix, 
               (PFPObject) floor, SS_PR_PROC);

    SS_install("ln",
               "Procedure: Returns the natural logarithm of the argument",
               SS_unary_flt, 
               (PFPObject) PM_ln, SS_PR_PROC);

    SS_install("log",
               "Procedure: Returns the logarithm base 10 of the argument",
               SS_unary_flt, 
               (PFPObject) PM_log, SS_PR_PROC);

    SS_install("negative?",
               "Procedure: Returns #t iff the argument is a number less than 0",
               SS_un_comp,
               (PFPObject) _SS_neg, SS_PR_PROC);

    SS_install("odd?",
               "Procedure: Returns #t iff the argument is a number that is not even",
               SS_un_comp,
               (PFPObject) _SS_odd, SS_PR_PROC);

    SS_install("positive?",
               "Procedure: Returns #t iff the argument is a number greater than 0",
               SS_un_comp,
               (PFPObject) _SS_pos, SS_PR_PROC);

    SS_install("quotient",
               "Procedure: Returns quotient of two integers",
               SS_binary_fix,
               (PFPObject) PM_ldivide, SS_PR_PROC);

    SS_install("remainder",
               "Procedure: Returns remainder of division of the two arguments",
               SS_binary_fix,
               (PFPObject) PM_lmodulo, SS_PR_PROC);

    SS_install("sin",
               "Procedure: Returns the sine of the argument",
               SS_unary_flt, 
               (PFPObject) sin, SS_PR_PROC);

    SS_install("sinh",
               "Procedure: Returns the hyperbolic sine of the argument",
               SS_unary_flt, 
               (PFPObject) sinh, SS_PR_PROC);

    SS_install("sqrt",
               "Procedure: Returns the principal square root of the argument",
               SS_unary_flt, 
               (PFPObject) sqrt, SS_PR_PROC);

    SS_install("tan",
               "Procedure: Returns the tangent of the argument",
               SS_unary_flt, 
               (PFPObject) tan, SS_PR_PROC);

    SS_install("tanh",
               "Procedure: Returns the hyperbolic tangent of the argument",
               SS_unary_flt, 
               (PFPObject) tanh, SS_PR_PROC);

    SS_install("truncate",
               "Procedure: Returns the integer resulting from the truncation of the argument",
               SS_unary_fix, 
               (PFPObject) PM_fix, SS_PR_PROC);

    SS_install("zero?",
               "Procedure: Returns #t iff the argument is a number equal to 0",
               SS_un_comp,
               (PFPObject) _SS_zero, SS_PR_PROC);

    return;}

/*--------------------------------------------------------------------------*/

/*                          MATHEMATICS HANDLERS                            */

/*--------------------------------------------------------------------------*/

/* SS_UNARY_FLT - the unary operator handler returning floats */

object *SS_unary_flt(proc, argl)
   PFPObject proc;
   object *argl;
   {PFDouble fnc;
    int type;
    double operand, val;

    if (SS_nullobjp(argl))
       SS_error("WRONG NUMBER OF ARGUMENTS - SS_UNARY_FLT", argl);

    fnc = (PFDouble) proc;
    SS_GET_OPERAND(operand, argl);

    val = (*fnc)(operand);

    return(SS_mk_float(val));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_UNARY_FIX - the unary operator handler returning fixed point numbers */

object *SS_unary_fix(proc, argl)
   PFPObject proc;
   object *argl;
   {PFDouble fnc;
    int type;
    double operand;

    if (SS_nullobjp(argl))
       SS_error("WRONG NUMBER OF ARGUMENTS - SS_UNARY_FIX", argl);

    fnc = (PFDouble) proc;
    SS_GET_OPERAND(operand, argl);

    return(SS_mk_integer((BIGINT) ((*fnc)(operand))));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_BINARY_FIX - the binary arithmetic operator handler */

object *SS_binary_fix(proc, argl)
   PFPObject proc;
   object *argl;
   {Register object *x1, *x2;
    PFLong fnc;

    if (_SS_length(argl) != 2)
       SS_error("WRONG NUMBER OF ARGUMENTS - SS_BINARY_FIX", argl);

    fnc = (PFLong) proc;
    x1  = SS_car(argl);
    x2  = SS_cadr(argl);

    if (!SS_integerp(x1) || !SS_integerp(x2))
       SS_error("ARGUMENTS NOT BOTH INTEGERS - SS_BINARY_FIX",
                SS_mk_cons(x1, x2));

    return(SS_mk_integer((BIGINT)((*fnc)(SS_INTEGER_VALUE(x1),
                                SS_INTEGER_VALUE(x2)))));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_BINARY_FLT - the binary arithmetic operator handler */

object *SS_binary_flt(proc, argl)
   PFPObject proc;
   object *argl;
   {int type, ident;
    double acc, operand;
    PFDouble fnc;

    fnc   = (PFDouble) proc;
    ident = 1;
    type  = SC_INTEGER_I;
    if ((fnc == PM_fplus) || (fnc == PM_fminus))
       ident = 0;

    if (fnc == PM_fdivide)
       type = SC_FLOAT_I;

    if (SS_nullobjp(argl))
       return(SS_mk_integer((BIGINT) ident));

    else if (SS_nullobjp(SS_cdr(argl)))
       acc = ident;

    else
       SS_GET_OPERAND(acc, argl);

    while (TRUE)
       {SS_GET_OPERAND(operand, argl);
        acc = (*fnc)(acc, operand);
        if (acc > LONG_MAX)
           type = SC_FLOAT_I;

        if (SS_nullobjp(argl))
           {switch (type)
               {case SC_INTEGER_I :
                     return(SS_mk_integer((BIGINT) acc));
                case SC_FLOAT_I   :
                     return(SS_mk_float(acc));
                default           :
                     SS_error("BAD ARITHMETIC RESULT - SS_BINARY_FLT",
                              SS_null);
                     break;};};};}

/*--------------------------------------------------------------------------*/

/*                      HANDLER AUXILLIARY FUNCTIONS                        */

/*--------------------------------------------------------------------------*/

/*                        UNARAY OPERATIONS                                 */

/*--------------------------------------------------------------------------*/

/*                         PREDICATE HANDLERS                               */

/*--------------------------------------------------------------------------*/

/* SS_UN_COMP - the unary comparison handler */

object *SS_un_comp(proc, argl)
   PFPObject proc;
   object *argl;
   {PFInt fnc;
    int type;
    double operand;

    if (SS_nullobjp(argl))
       SS_error("WRONG NUMBER OF ARGUMENTS - SS_UN_COMP", argl);

    fnc = (PFInt) proc;
    SS_GET_OPERAND(operand, argl);

    return((*fnc)(operand) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_BIN_COMP - the binary comparison handler */

object *SS_bin_comp(proc, argl)
   PFPObject proc;
   object *argl;
   {PFInt fnc;
    int type;
    double c1, c2;

    if (_SS_length(argl) != 2)
       SS_error("WRONG NUMBER OF ARGUMENTS - SS_BIN_COMP", argl);

    fnc = (PFInt) proc;
    SS_GET_OPERAND(c1, argl);
    SS_GET_OPERAND(c2, argl);

    return((*fnc)(c1, c2) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_ZERO - C level 0 test */

static int _SS_zero(x1)
   double x1;
   {return(x1 == 0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_NEG - C level negative test */

static int _SS_neg(x1)
   double x1;
   {return(x1 < 0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_POS - C level positive test */

static int _SS_pos(x1)
   double x1;
   {return(x1 > 0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_EVEN - even predicate in C */

static int _SS_even(f)
   double f;
   {long g;

    g = (((long) PM_fix(f)) >> 1) << 1;

    return(f == (double) g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_ODD - odd predicate in C */

static int _SS_odd(f)
   double f;
   {long g;

    g = (((long) PM_fix(f)) >> 1) << 1;

    return(f != (double) g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_EQUL - = comparison at C level */

static int _SS_equl(x1, x2)
   double x1, x2;
   {return(x1 == x2);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_LT - < comparison at C level */

static int _SS_lt(x1, x2)
   double x1, x2;
   {return(x1 < x2);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_GT - > comparison at C level */

static int _SS_gt(x1, x2)
   double x1, x2;
   {return(x1 > x2);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_LE - <= comparison at C level */

static int _SS_le(x1, x2)
   double x1, x2;
   {return(x1 <= x2);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_GE - >= comparison at C level */

static int _SS_ge(x1, x2)
   double x1, x2;
   {return(x1 >= x2);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
