/*
 * MLCPLX.H - header for complex arithmetic
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#ifndef PCK_COMPLEX

#include "cpyright.h"

#define PCK_COMPLEX

#include "score.h"

/*--------------------------------------------------------------------------*/

/*                               TYPEDEFS                                   */

/*--------------------------------------------------------------------------*/

#ifdef __cplusplus

typedef REAL complex[2];

# define PM_REAL_C(x)       (x)[0]
# define PM_IMAGINARY_C(x)  (x)[1]

#else

struct s_complex
   {REAL real;
    REAL imag;};

typedef struct s_complex complex;

#  define PM_REAL_C(x)       x.real
#  define PM_IMAGINARY_C(x)  x.imag

#endif

/*--------------------------------------------------------------------------*/

/*                       STANDARD PROCEDURAL MACROS                         */

/*--------------------------------------------------------------------------*/

/* PM_COMPLEX - set the parts of a complex number */

#define PM_COMPLEX(b, c)                                                     \
   (PM_REAL_C(cx_reg) = (double) (b),                                        \
    PM_IMAGINARY_C(cx_reg) = (double) (c),                                   \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_COMPLEX_CONJUGATE - return the complex conjugate of c */

#define PM_COMPLEX_CONJUGATE(c)                                              \
   (PM_REAL_C(cx_reg)      =  PM_REAL_C(c),                                  \
    PM_IMAGINARY_C(cx_reg) = -PM_IMAGINARY_C(c),                             \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_PLUS_CC - binary complex addition, b + c */

#define PM_PLUS_CC(b, c)                                                     \
   (PM_REAL_C(cx_reg)      = PM_REAL_C(b) + PM_REAL_C(c),                    \
    PM_IMAGINARY_C(cx_reg) = PM_IMAGINARY_C(b) + PM_IMAGINARY_C(c),          \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_PLUS_RC - b + c where b is real and c is complex */

#define PM_PLUS_RC(b, c)                                                     \
   (PM_REAL_C(cx_reg)      = b + PM_REAL_C(c),                               \
    PM_IMAGINARY_C(cx_reg) = PM_IMAGINARY_C(c),                              \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_MINUS_CC - binary complex subtraction, b - c */

#define PM_MINUS_CC(b, c)                                                    \
   (PM_REAL_C(cx_reg)      = PM_REAL_C(b) - PM_REAL_C(c),                    \
    PM_IMAGINARY_C(cx_reg) = PM_IMAGINARY_C(b) - PM_IMAGINARY_C(c),          \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_TIMES_CC - binary complex multiplication, b * c */

#define PM_TIMES_CC(b, c)                                                    \
   (cx_ttr = (PM_REAL_C(b)*PM_REAL_C(c) -                                    \
              PM_IMAGINARY_C(b)*PM_IMAGINARY_C(c)),                          \
    cx_tti = (PM_REAL_C(b)*PM_IMAGINARY_C(c) +                               \
	      PM_IMAGINARY_C(b)*PM_REAL_C(c)),                               \
    PM_REAL_C(cx_reg)      = cx_ttr,                                         \
    PM_IMAGINARY_C(cx_reg) = cx_tti,                                         \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_TIMES_RC - b * c where b is real and c is complex */

#define PM_TIMES_RC(b, c)                                                    \
   (PM_REAL_C(cx_reg)      = b*PM_REAL_C(c),                                 \
    PM_IMAGINARY_C(cx_reg) = b*PM_IMAGINARY_C(c),                            \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_TIMES_IC - b * c where b is the magnitude of a pure imaginary number
 *             - and c is complex */

#define PM_TIMES_IC(b, c)                                                    \
   (PM_REAL_C(cx_reg)      = -b*PM_IMAGINARY_C(c),                           \
    PM_IMAGINARY_C(cx_reg) =  b*PM_REAL_C(c),                                \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_DIVIDE_CC - binary complex division, b / c */

#define PM_DIVIDE_CC(b, c)                                                   \
   (cx_ttr = PM_REAL_C(c),                                                   \
    cx_tti = PM_IMAGINARY_C(c),                                              \
    cx_ttm = cx_ttr*cx_ttr + cx_tti*cx_tti,                                  \
    cx_ttr = (PM_REAL_C(b)*PM_REAL_C(c) +                                    \
	      PM_IMAGINARY_C(b)*PM_IMAGINARY_C(c)),                          \
    cx_tti = (PM_IMAGINARY_C(b)*PM_REAL_C(c) -                               \
	      PM_REAL_C(b)*PM_IMAGINARY_C(c)),                               \
    PM_REAL_C(cx_reg)      = cx_ttr/cx_ttm,                                  \
    PM_IMAGINARY_C(cx_reg) = cx_tti/cx_ttm,                                  \
    cx_reg)

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_MODULUS_C - compute the complex modulus */

#define PM_MODULUS_C(x)                                                      \
   (cx_ttr = PM_REAL_C(x),                                                   \
    cx_tti = PM_IMAGINARY_C(x),                                              \
    sqrt(cx_ttr*cx_ttr + cx_tti*cx_tti))

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

#define PM_COMPLEX_SWAP(a, b)                                                \
    {complex z;                                                              \
     z   = (a);                                                              \
     (a) = (b);                                                              \
     (b) = z;}

/*--------------------------------------------------------------------------*/

/*                    STANDARD VARIABLE DECLARATIONS                        */

/*--------------------------------------------------------------------------*/

extern double
 cx_ttr,
 cx_tti,
 cx_ttm;

extern complex
 cx_reg;

/*--------------------------------------------------------------------------*/

/*                    STANDARD FUNCTION DECLARATIONS                        */

/*--------------------------------------------------------------------------*/


/* MLFFT.C declarations */

extern int
 SC_DECLARE(PM_fft_sc_real_data,
         (complex **pcy, REAL **px, REAL *ipx, REAL *ipy,
          int n, double xmn, double xmx, int ordr)),
 SC_DECLARE(PM_fft_sc_complex_data,
         (complex **pcy, REAL **px, REAL *ipx,
          complex *ipy, int n, double xmn, double xmx,
          int flag, int ordr)),
 SC_DECLARE(PM_fft_sc_complex, (complex *x, int n, int flag));


/* MLMATH.C declarations */

extern void
 SC_DECLARE(PM_smooth_filter, (complex *z, int n, double pts));

extern int
 SC_DECLARE(PM_smooth_fft,
	 (REAL *x, REAL *y, int n, int pts,
	  DECLFPTR(byte, fnc, (complex *cx, int nt, double tol))));

#endif


