/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/* FLASK */

#ifndef _CONTEXT_H_
#define _CONTEXT_H_

#include "ebitmap.h"


/*
 * The representation of a MLS level used internally
 * by the security policy server.  This representation is
 * private to the security policy server and can be changed
 * without affecting any clients of the security policy server.
 *
 * A MLS level consists of a sensitivity and a category set.
 */
typedef struct mls_level
{
    unsigned int sens;
    ebitmap_t cat;
} mls_level_t;


/*
 * The representation of a MLS range used internally
 * by the security policy server.  This representation is
 * private to the security policy server and can be changed
 * without affecting any clients of the security policy server.
 *
 * A MLS range consists of two MLS levels, high and low.
 * Either high must be equivalent to low or high must dominate low.
 */
typedef struct mls_range
{
    mls_level_t low;
    mls_level_t high;
} mls_range_t;


/*
 * The four MLS relations:  dominates, dominated_by, equivalent, incomparable
 */
#define MLS_RELATION_DOM	1
#define MLS_RELATION_DOMBY	2 
#define MLS_RELATION_EQ		4
#define MLS_RELATION_INCOMP	8


#define mls_level_eq(l1,l2) (((l1).sens == (l2).sens) && ebitmap_cmp(&(l1).cat,&(l2).cat)) 


#define mls_level_relation(l1,l2) (\
(((l1).sens == (l2).sens) && ebitmap_cmp(&(l1).cat,&(l2).cat)) ? MLS_RELATION_EQ : \
(((l1).sens >= (l2).sens) && ebitmap_contains(&(l1).cat, &(l2).cat)) ? MLS_RELATION_DOM : \
(((l2).sens >= (l1).sens) && ebitmap_contains(&(l2).cat, &(l1).cat)) ? MLS_RELATION_DOMBY : \
MLS_RELATION_INCOMP )


#define mls_range_contains(r1,r2) \
((mls_level_relation((r1).low, (r2).low) & \
	  (MLS_RELATION_EQ | MLS_RELATION_DOMBY)) && \
	 (mls_level_relation((r1).high, (r2).high) & \
	  (MLS_RELATION_EQ | MLS_RELATION_DOM))) 


/*
 * The representation of a root security context used internally
 * by the security policy server.  This representation is
 * private to the security policy server and can be changed
 * without affecting any clients of the security policy server.
 *
 * A root security context consists of an authenticated user
 * identity, a role, a type and a MLS range.
 */
typedef struct
{
    unsigned int user;
    unsigned int role;
    unsigned int type;
    mls_range_t range;
} ss_root_context_t;

#define root_context_init(c) \
{ \
    (c)->user = 0;\
    (c)->role = 0; \
    (c)->type = 0; \
    (c)->range.low.sens = 0; \
    ebitmap_init(&(c)->range.low.cat); \
    (c)->range.high.sens = 0; \
    ebitmap_init(&(c)->range.high.cat); \
}

#define root_context_cmp(c1,c2) \
    ( ((c1)->user == (c2)->user)  && \
      ((c1)->role == (c2)->role) && \
      ((c1)->type == (c2)->type) && \
      ((c1)->range.low.sens == (c2)->range.low.sens) && \
      ebitmap_cmp(&(c1)->range.low.cat,&(c2)->range.low.cat) && \
      ((c1)->range.high.sens == (c2)->range.high.sens) && \
      ebitmap_cmp(&(c1)->range.high.cat,&(c2)->range.high.cat) )

extern int root_context_cpy(ss_root_context_t *dst, 
			    ss_root_context_t *src);

#define root_context_destroy(c) \
{ \
    (c)->user = 0; \
    (c)->role = 0; \
    (c)->type = 0; \
    (c)->range.low.sens = 0; \
    ebitmap_destroy(&(c)->range.low.cat); \
    (c)->range.high.sens = 0; \
    ebitmap_destroy(&(c)->range.high.cat); \
}


/*
 * The representation of a child security context used internally
 * by the security policy server.  This representation is
 * private to the security policy server and can be changed
 * without affecting any clients of the security policy server.
 *
 * A child security context consists of a nesting depth, a parent
 * SID and a child value.
 */
typedef struct 
{
    unsigned int depth;
    security_id_t parent;
    unsigned int value;
} ss_child_context_t;

#define child_context_init(c) \
{ \
    (c)->depth = 0; \
    (c)->parent = SECSID_NULL; \
    (c)->value = 0; \
}

#define child_context_cmp(c1,c2) \
    (((c1)->depth == (c2)->depth) && \
     ((c1)->parent == (c2)->parent) && \
     ((c1)->value == (c2)->value))

#define child_context_cpy(dst,src) \
{ \
    (dst)->depth = (src)->depth; \
    (dst)->parent = (src)->parent; \
    (dst)->value = (src)->value; \
}

#define child_context_destroy(c) child_context_init(c)


/*
 * The representation of a security context used internally
 * by the security policy server.  This representation is
 * private to the security policy server and can be changed
 * without affecting any clients of the security policy server.
 *
 * A security context consists of either a root security context
 * or a child security context.
 */
typedef struct 
{
     int isroot;
    union 
    {
	ss_root_context_t root_context;
	ss_child_context_t child_context;
    } context;
#define RCONTEXT(c) ((c)->context.root_context)
#define CCONTEXT(c) ((c)->context.child_context)
} ss_context_t;
    
void context_init(ss_context_t *c, int isroot);
int context_cmp(ss_context_t *c1, ss_context_t *c2);
int context_cpy(ss_context_t *dst, ss_context_t *src);
void context_destroy(ss_context_t *c);

#endif _CONTEXT_H_

/* FLASK */
