/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "RCX_Link.h"
#include "RCX_Cmd.h"

#define VERSION "1.0 b4"

#define kMaxFirmwareSize	65536
#define kMaxLine	256

static char *sTestArgs[]={"rcxfirm", "::unsupported:FIRM0309.LGO"};


class MyLink : public RCX_Link
{
public:
	Bool	DownloadProgress(int soFar, int total);
};

const char *gSerialPort = nil;
MyLink gLink;
const char *gFilename = nil;

int ReadFirmware(FILE *fp, UByte *data);
int GetByte(const char *ptr);
RCX_Result PrintVersion(RCX_Link &link);
void PrintUsage();
int ProcessCommandLine(int argc, char **argv);
void PrintError(RCX_Result error);


int main(int argc, char **argv)
{
	RCX_Result result;
	FILE *fp;
	UByte *data;
	int length;
	
	if (argc==0)
	{
		argv = sTestArgs;
		argc = sizeof(sTestArgs) / sizeof(char *);
	}

	if (!ProcessCommandLine(argc, argv))
	{
		PrintUsage();
		return -1;
	}

	
	if (gFilename)
	{
		fp = fopen(argv[1], "r");
		if (!fp)
		{
			fprintf(stderr, "Error: could not open file \'%s\'\n", argv[1]);
			return -1;
		}
		
		data = new UByte[kMaxFirmwareSize];
		length = ReadFirmware(fp, data);
		fclose(fp);
		
		if (length == 0)
		{
			fprintf(stderr, "Error: \'%s\' is not a legal firmware file\n", argv[1]);
			return -1;
		}
	}
	else
	{
		data = nil;
		length = 0;
	}
	
	result = gLink.Open(gSerialPort);
	if (RCX_ERROR(result))
	{
		PrintError(result);
		return -1;
	}

	result = PrintVersion(gLink);
	if (RCX_ERROR(result)) goto Error;

	if (data)
	{
		printf("downloading.");
		fflush(stdout);
		result = gLink.SendFirmware(data, length);
		printf("\n");
		if (RCX_ERROR(result)) goto Error;
				
		result = PrintVersion(gLink);
		if (RCX_ERROR(result)) goto Error;
	}

	gLink.Close();
	return 0;
	
Error:
	PrintError(result);
	gLink.Close();
	return -1;
}


RCX_Result PrintVersion(RCX_Link &link)
{
	ULong rom, ram;
	RCX_Result result;
	
	result = link.GetVersion(rom, ram);
	if (!RCX_ERROR(result))
		printf("Current Version: %08lx/%08lx\n", rom, ram);

	return result;
}


Bool MyLink::DownloadProgress(int /* soFar */, int /* total */)
{
	putchar('.');
	fflush(stdout);
	return true;
}

int ReadFirmware(FILE *fp, UByte *data)
{
	char line[kMaxLine];
	char *ptr;
	int length = 0;
	int dataLen;
	int byte;
	
	while(fgets(line, kMaxLine, fp))
	{
		ptr = line;
		if (*ptr == 10 || *ptr==13) // hack for DOS line endings
			ptr++;
		
		if (ptr[0] == 0) continue;
		
		if (ptr[0] != 'S')
		{
			printf("%d %d\n", ptr[0], length);
			return 0;
		}

		if (ptr[1] != '1') continue;
		dataLen = GetByte(ptr + 2);
		if (dataLen < 3) return 0;

		ptr += 8;
		for(int i=0; i<dataLen-3; i++)
		{
			byte = GetByte(ptr);
			ptr += 2;
			if (byte == -1) return 0;
			data[length++] = (UByte)byte;
		}
	}

	return length;
}


int GetByte(const char *ptr)
{
	int i;
	int v = 0;
	char c;
	int n;
	
	for(i=0; i<2; i++)
	{
		c = *ptr++;
	
		if (c >= '0' && c <= '9')
			n = c - '0';
		else if (c >= 'A' && c <= 'F')
			n = c - 'A' + 10;
		else if (c >= 'a' && c <= 'f')
			n = c - 'a' + 10;
		else
			return -1;
			
		v = (v << 4) + n;	
	}
	
	return v;
}

void PrintUsage()
{
	fprintf(stderr, "rcxfirm version "VERSION" (built "__DATE__"," __TIME__")\n");
	fprintf(stderr, "     written by Dave Baum\n");
	fprintf(stderr, "Usage: rcxfirm [-s serial_port] [firmware]\n");
}


int ProcessCommandLine(int argc, char **argv)
{
	int v;
	
	for(int i=1; i<argc; i++)
	{
		if (argv[i][0]=='-')
		{
			switch(tolower(argv[i][1]))
			{
				case 's':
					i++;
					if (i==argc || gSerialPort) return 0;
					gSerialPort = argv[i];
					break;
				case 't':
					i++;
					if (i==argc) return 0;
					v = atoi(argv[i]);
					gLink.SetRxTimeout(v);
					break;
				case 'v':
					gLink.SetVerbose(true);
					break;
				default:
					return 0;
			}
		}
		else if (!gFilename)
			gFilename = argv[i];
		else
			return 0;
	}
	
	return 1;
}

void PrintError(RCX_Result error)
{
	if (error >= 0) return;
	
	switch(error)
	{
		case kRCX_OpenSerialError:
			fprintf(stderr, "Could not open serial port\n");
			break;
		case kRCX_IREchoError:
			fprintf(stderr, "Problem talking to IR device\n");
			break;
		case kRCX_ReplyError:
			fprintf(stderr, "No reply from RCX\n");
			break;
		default:
			fprintf(stderr, "Error #%d\n", -error);
			break;
	}
}


