/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "Program.h"
#include "Fragment.h"
#include "RCX_Image.h"
#include "Bytecode.h"
#include "parser.h"
#include "Symbol.h"
#include "BlockStmt.h"
#include "InlineStmt.h"
#include "VarPool.h"
#include "Function.h"
#include "Error.h"

static void EncodeFragment(RCX_Image *image, Fragment *f, VarPool &varPool, bool temps, int index);


Program *gProgram = new Program();

Program::Program() :
	fContext(this)
{
	fCounts[0] = 1;	// leave room for main
	fCounts[1] = 0;
	fMainAdded = false;
	fVarCount = fVarNext = 0;
	fScopes.InsertHead(new Scope(this));
	fInitName = Symbol::Get("_init");
	fInitStmts = new BlockStmt();
}


Program::~Program()
{
	while(Fragment *f=fTasks.RemoveHead())
		delete f;

	while(Fragment *f=fSubs.RemoveHead())
		delete f;
	
	while(Scope *s=fScopes.RemoveHead())
		delete s;

	while(Function *func = fFunctions.RemoveHead())
		delete func;
				
	delete fInitStmts;
}


int Program::AddMainTask(Fragment *f)
{
	CheckName(f->GetName());
	fMainAdded = true;
	fTasks.InsertHead(f);
	return 0;
}


int Program::AddTask(Fragment *f)
{
	CheckName(f->GetName());
	fTasks.InsertTail(f);
	return fCounts[kRCX_TaskFragment]++;
}


int Program::AddSubroutine(Fragment *f)
{
	CheckName(f->GetName());
	fSubs.InsertTail(f);
	return fCounts[kRCX_SubFragment]++;
}


void Program::AddFunction(Function *f)
{
	CheckName(f->GetName());
	fFunctions.InsertTail(f);
}


void Program::AddInitStmt(Stmt *s)
{
	fInitStmts->Add(s);
}


void Program::CheckName(const Symbol *name)
{
	if (Defined(name))
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
}


int Program::CreateVar(Symbol *name)
{
	if (!fContext)
	{
		Error(kErr_NoVarDecl).RaiseLex();
		return kIllegalVar;
	}
	
	Scope *s = fScopes.GetHead();
	
	if (s->Contains(name))
	{
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
		return kIllegalVar;
	}

	int var = fContext->AllocateVar(name);
	s->Define(name, var);

	return var;
}


int Program::AllocateVar(const Symbol *name)
{
	if (fVarNext == kRCX_MaxVars)
	{
		Error(kErr_MaxVars, kRCX_MaxVars).RaiseLex();
		return kIllegalVar;
	}

	int var = fVarNext++;
	if (fVarNext > fVarCount)
		fVarCount = fVarNext;
	
	fVarNames[var] = name;
	
	return var;
}


int Program::GetMark() const
{
	return fVarNext;
}

void Program::SetMark(int mark)
{
	fVarNext = mark;
}


int Program::GetVar(const Symbol *name)
{
	int var = fScopes.GetHead()->Lookup(name);
	if (var == kIllegalVar)
		Error(kErr_Undeclared, name->GetKey());
	
	return var;
}



RCX_Image*	Program::CreateImage(bool targetCM)
{
	RCX_Image *image;
	int index;
	VarPool varPool;
	
	for(int i=0; i<fVarCount; i++)
		varPool.Reserve(i);

	if (!Check(targetCM, varPool)) return nil;
		
	image = new RCX_Image();
	
	image->SetFragmentCount(fCounts[0] + fCounts[1]);
	index = 0;

	// emit tasks
	for(Fragment *task=fTasks.GetHead(); task; task=task->GetNext())
	{
		EncodeFragment(image, task, varPool, true, index++);

		// emit subs that fall within the task's execution space for temp vars
		for(Fragment *sub=fSubs.GetHead(); sub; sub=sub->GetNext())
		{
			if (sub->GetTaskID() == task->GetTaskID())
				EncodeFragment(image, sub, varPool, true, index++);
		}

		varPool.ReserveUsed();
	}

	// emit other subs (multi or no task ID)
	for(Fragment *sub=fSubs.GetHead(); sub; sub=sub->GetNext())
	{
		if (sub->GetTaskID() == Fragment::kNoTaskID || sub->GetTaskID() == Fragment::kMultiTaskID)
			EncodeFragment(image, sub, varPool, false, index++);
	}

	// emit global variable names
	for(int i=0; i<fVarCount; i++)
		image->SetVariable(i, fVarNames[i]->GetKey());

	// emit temp names
	for(int i=fVarCount; i<kRCX_MaxVars; i++)
	{
		if (!varPool.IsFree(i))
		{
			char varName[64];
			sprintf(varName, "__temp_%d", i);
			image->SetVariable(i, varName);
		}
	}
		
	return image;
}


void EncodeFragment(RCX_Image *image, Fragment *f, VarPool &varPool, bool temps, int index)
{
	Bytecode *b = new Bytecode(varPool, temps);

	f->Emit(*b);
	image->SetFragment(index++, f->GetType(), f->GetNumber(),
		b->GetData(), b->GetLength(), f->GetName()->GetKey());

	delete b;
}


bool Program::Check(bool targetCM, VarPool &varPool)
{
	int max;
	bool ok = true;

	// find main
	if (!fMainAdded)
	{
		Error(kErr_UndefinedMain).Raise(0);
		ok = false;
	}
	else
	{
		BlockStmt *mainBlock = fTasks.GetHead()->GetBody();
		
		// insert call for init routine into main task
		if (fInitName)
		{
			Function *f = gProgram->GetFunction(fInitName);
			if (f)
			{
				InlineStmt *s = new InlineStmt();
				s->Add(f->GetBody()->Clone(0));
				mainBlock->Prepend(s);
			}
			else
			{
				Error(kErr_UnknownInit, fInitName->GetKey()).Raise(0);
				ok = false;
			}
		}
		
		// insert init stmts into main task
		mainBlock->Prepend(fInitStmts);
		fInitStmts = 0;
	}
	
	
	max = targetCM ? kRCX_MaxTasks_CM : kRCX_MaxTasks;
	if (fCounts[kRCX_TaskFragment] > max)
	{
		Error(kErr_TooManyTasks, max).Raise(0);
		ok = false;
	}

	max = targetCM ? kRCX_MaxSubs_CM : kRCX_MaxSubs;
	if (fCounts[kRCX_SubFragment] > max)
	{
		Error(kErr_TooManySubs, max).Raise(0);
		ok = false;
	}
	
	Fragment *f;
	for(f=fTasks.GetHead(); f; f=f->GetNext())
	{
		if (!f->Check(&varPool)) ok = false;
		varPool.ReserveUsed();
	}
	
	for(f=fSubs.GetHead(); f; f=f->GetNext())
	{
		if (!f->Check(0)) ok = false;
	}
		
	return ok;
}


Fragment *Program::GetTask(const Symbol *name)
{
	for(Fragment *f = fTasks.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


Fragment *Program::GetSub(const Symbol *name)
{
	for(Fragment *f = fSubs.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


Function *Program::GetFunction(const Symbol *name)
{
	for(Function *f = fFunctions.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


bool Program::Defined(const Symbol *name) const
{
	// this "const" hack is just to avoid writing duplicate
	// versions of GetTask(), GetSub(), and GetFunction() that
	// maintain const-ness
	Program *p = (Program*)(this);

	if (p->GetTask(name)) return true;
	if (p->GetSub(name)) return true;
	if (p->GetFunction(name)) return true;
	
	return fScopes.GetTail()->Contains(name);
}


Scope* Program::PushScope()
{
	Scope *s = new Scope(fContext);
	fScopes.InsertHead(s);
	return s;
}


void Program::PopScope()
{
	delete fScopes.RemoveHead();
}
