/*
 *  C2caml : parses c headers and produces appropriate caml bindings for it
 *  Copyright (C) 1999  Sven LUTHER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* $Id: par.mly,v 1.1 1999/06/19 14:59:20 sven Exp $ */

%{ (* Header *)
  open Var
  open Expr
  let err s = prerr_endline ("> " ^ s ^ ".")
  let xxx_vl = Var.empty ()
  let _ = Var.begin_block Var.vl
%}

/* Declaration */

/* Keywords */
%token Keyword_auto
%token Keyword_register
%token Keyword_static
%token Keyword_extern
%token Keyword_typedef
%token Keyword_void
%token Keyword_char
%token Keyword_short
%token Keyword_int
%token Keyword_long
%token Keyword_float
%token Keyword_double
%token Keyword_signed
%token Keyword_unsigned
%token Keyword_const
%token Keyword_volatile
%token Keyword_struct
%token Keyword_union
%token Keyword_enum
%token Keyword_case
%token Keyword_default
%token Keyword_if
%token Keyword_else
%token Keyword_switch
%token Keyword_while
%token Keyword_do
%token Keyword_for
%token Keyword_goto
%token Keyword_continue
%token Keyword_break
%token Keyword_return
%token Keyword_sizeof
%token Keyword_gnu_attribute

/* Symbols */
%token Symbol_comma
%token Symbol_semicolon
%token Symbol_open_bracket
%token Symbol_close_bracket
%token Symbol_open_paren
%token Symbol_close_paren
%token Symbol_affect
%token Symbol_colon
%token Symbol_dots
%token Symbol_open_brace
%token Symbol_close_brace
%token Symbol_mult_affect
%token Symbol_div_affect
%token Symbol_mod_affect
%token Symbol_add_affect
%token Symbol_sub_affect
%token Symbol_left_affect
%token Symbol_right_affect
%token Symbol_and_affect
%token Symbol_xor_affect
%token Symbol_or_affect
%token Symbol_quest
%token Symbol_logic_or
%token Symbol_logic_and
%token Symbol_or
%token Symbol_xor
%token Symbol_and
%token Symbol_equal
%token Symbol_not_equal
%token Symbol_less
%token Symbol_more
%token Symbol_more_or_equal
%token Symbol_less_or_equal
%token Symbol_left
%token Symbol_right
%token Symbol_add
%token Symbol_sub
%token Symbol_mult
%token Symbol_div
%token Symbol_mod
%token Symbol_add_add
%token Symbol_sub_sub
%token Symbol_neg
%token Symbol_not
%token Symbol_dot
%token Symbol_arrow

/* Others */

%token <string> Ident Typedef_name
%token <string> String
%token <int> Integer_const
%token <char> Char_const
%token <float> Float_const
%token <int> Enum_const
%token Eol

/* Precedences */

%nonassoc if_then_else
%nonassoc if_then

/* Start symbol */

%start translation_unit
%type <unit> translation_unit 

%% /* Rules */

/* pseudo C parser */
translation_unit :
  | external_declaration translation_unit { () }
  | external_declaration { () }
  ;
external_declaration :
  | function_definition { () }
  | declaration { () }
  ;
function_definition :
  | declarator composed_inst { err "f () {}" }
  | declaration_specifier declarator composed_inst { err "type f () {}" }
  | declarator declaration_list composed_inst { err "f () k&r {}" }
  | declaration_specifier declarator declaration_list composed_inst
    { err "type f () k&r {}" }
  ;
declaration :
  | declaration_specifier Symbol_semicolon
    { 
      match get_type $1 with
        | T_enum (i, _) -> 
	(
	  if i <> ""
	  then err ("Naming enum " ^ i);
	    try Var.add_var i Var.vl $1 e_none
	    with Var.Double_defined -> ()
	)
        | T_struct_or_union (i, (b, _)) -> 
	(
	  if i <> ""
	  then err ("Naming " ^ (if b then "struct " else "union ") ^ i);
	    try Var.add_var i Var.vl $1 e_none
	    with Var.Double_defined -> ()
	)
	| _ -> ();
      (*process $1 []*)
    }
  | declaration_specifier declarator_init_list Symbol_semicolon
    {
      (
        match get_type $1 with
        | T_enum (i, _) ->
	(
	  if i <> ""
	  then err ("Naming enum " ^ i );
	    try Var.add_var i Var.vl $1 e_none
	    with Var.Double_defined -> ()
	)
        | T_struct_or_union (i, (b, _)) -> 
	(
	  if i <> ""
	  then err ("Naming " ^ (if b then "struct " else "union ") ^ i);
	    try Var.add_var i Var.vl $1 e_none
	    with Var.Double_defined -> ()
	)
	| _ -> ()
      );
      let rec declare = function
        | [] -> ()
        | (d,e)::q ->
	  if Expr.is_typedef_class $1
	  then
	    try
	      Var.add_var (Expr.get_symbol d) Var.vl $1 (expr E_typedef);
	      declare q
	    with Var.Double_defined -> ()
	  else 
	    err ("Declaring " ^ (Expr.get_symbol d));
	    try Var.add_var (Expr.get_symbol d) Var.vl $1 e; declare q
	    with Var.Double_defined -> ()
      in declare $2
      (*process $1 $2*)
    }
  ;
declaration_list :
  | declaration { [$1] }
  | declaration_list declaration { $2::$1 }
  ;
declaration_specifier :
  | storage_class_specifier type_specifier type_qualifier
    { ntype $1 $3 $2 }
  | storage_class_specifier type_specifier { ntype $1 T_none $2 }
  | storage_class_specifier type_qualifier type_specifier
    { ntype $1 $2 $3 }
  | storage_class_specifier type_qualifier
    { ntype $1 $2 (T_int (true, T_normal)) }
  | storage_class_specifier { ntype $1 T_none (T_int (true, T_normal)) }
  | type_specifier storage_class_specifier type_qualifier
    { ntype $2 $3 $1 }
  | type_specifier storage_class_specifier { ntype $2 T_none $1 }
  | type_specifier type_qualifier storage_class_specifier
    { ntype $3 $2 $1 }
  | type_specifier type_qualifier { ntype T_auto $2 $1 }
  | type_specifier { ntype T_auto T_none $1 }
  | type_qualifier storage_class_specifier type_specifier
    { ntype $2 $1 $3 }
  | type_qualifier storage_class_specifier
    { ntype $2 $1 (T_int (true, T_normal)) }
  | type_qualifier type_specifier storage_class_specifier
    { ntype $3 $1 $2 }
  | type_qualifier type_specifier { ntype T_auto $1 $2 }
  | type_qualifier { ntype T_auto $1 (T_int (true, T_normal)) }
  ;
storage_class_specifier :
  | Keyword_auto { T_auto }
  | Keyword_register { T_register }
  | Keyword_static { T_static }
  | Keyword_extern { T_extern }
  /* Should warrant a special case treatment ... */
  | Keyword_typedef { T_typedef }
  ;
type_specifier :
  | Keyword_void { T_void }
  | sign_specifier size_specifier Keyword_int { T_int ($1, $2) }
  | sign_specifier size_specifier { T_int ($1, $2) }
  | sign_specifier Keyword_int size_specifier { T_int ($1, $3) }
  | sign_specifier Keyword_int { T_int ($1, T_normal) }
  | sign_specifier { T_int (true, T_normal) }
  | size_specifier sign_specifier Keyword_int { T_int ($2, $1) }
  | size_specifier sign_specifier { T_int ($2, $1) }
  | size_specifier Keyword_int sign_specifier { T_int ($3, $1) }
  | size_specifier Keyword_int { T_int (true, $1) }
  | size_specifier { T_int (true, $1) }
  | Keyword_int sign_specifier size_specifier { T_int ($2, $3) }
  | Keyword_int sign_specifier { T_int ($2, T_normal) }
  | Keyword_int size_specifier sign_specifier { T_int ($3, $2) }
  | Keyword_int size_specifier { T_int (true, $2) }
  | Keyword_int { T_int (true, T_normal) }
  | sign_specifier Keyword_char { T_char $1 }
  | Keyword_char sign_specifier { T_char $2 }
  | Keyword_char { T_char true }
  | Keyword_float { T_float T_short }
  | Keyword_double { T_float T_normal }
  | Keyword_long Keyword_double { T_float T_long }
  | Keyword_double Keyword_long { T_float T_long }
  | struct_or_union_specifier { T_struct_or_union (fst $1, snd $1) }
  | enum_specifier { T_enum (fst $1, snd $1) }
  | typedef_name { T_type_name $1 }
  ;
sign_specifier :
  | Keyword_unsigned { false }
  | Keyword_signed { true }
  ;
size_specifier :
  | Keyword_long Keyword_long { T_verylong }
  | Keyword_long { T_long }
  | Keyword_short { T_short }
  ;
type_qualifier :
  | Keyword_const { T_const }
  | Keyword_volatile { T_volatile }
  | Keyword_volatile Keyword_const { T_both }
  | Keyword_const Keyword_volatile { T_both }
  ;
struct_or_union_specifier :
  | struct_or_union bracket_open struct_or_union_declaration_list
    bracket_close { "", ($1, ()) }
  | struct_or_union Ident bracket_open struct_or_union_declaration_list
    bracket_close { $2, ($1, ()) }
  | struct_or_union Ident { $2, ($1, ()) }
  ;
struct_or_union :
  | Keyword_struct { true }
  | Keyword_union { false }
  ; 
struct_or_union_declaration_list :
  | struct_or_union_declaration { () }
  | struct_or_union_declaration_list struct_or_union_declaration { () }
  ;
declarator_init_list :
  | declarator_init { [$1] }
  | declarator_init_list Symbol_comma declarator_init { $3::$1 }
  ;
declarator_init :
  | declarator { $1, e_none }
  | declarator Symbol_affect initialisator { $1, $3 }
  ;
struct_or_union_declaration :
  | specifier_qualifier_list struct_or_union_declarator_list Symbol_semicolon
    { () }
  ;
specifier_qualifier_list :
  | type_specifier { () }
  | type_specifier specifier_qualifier_list { () }
  | type_qualifier { () }
  | type_qualifier specifier_qualifier_list { () }
  ; 
struct_or_union_declarator_list :
  | struct_or_union_declarator { () }
  | struct_or_union_declarator_list Symbol_comma struct_or_union_declarator
    { () }
  ;
struct_or_union_declarator :
  | declarator { $1 }
  | Symbol_colon constant_expr { D_none }
  | declarator Symbol_colon constant_expr { $1 }
  ; 
enum_specifier :
  | Keyword_enum bracket_open enumerator_list
    bracket_close { "", $3 }
  | Keyword_enum Ident bracket_open enumerator_list
    bracket_close { $2, $4 }
  | Keyword_enum Ident { "", [] }
  ;
enumerator_list :
  | enumerator { [$1] }
  | enumerator_list Symbol_comma enumerator { $3::$1 }
  ;
enumerator :
  | Ident
    {
      err ("Declaring enum ident " ^ $1);
      (
        try Var.add_var $1 Var.vl (ntype T_auto T_none (T_int (true, T_normal)))
          (expr E_enum_ident)
        with Var.Double_defined -> ()
      );
      ($1, None)
    }
  | Ident Symbol_affect constant_expr
    {
      err ("Declaring enum ident " ^ $1);
      (
        try Var.add_var $1 Var.vl (ntype T_auto T_none (T_int (true, T_normal)))
          (expr E_enum_ident)
        with Var.Double_defined -> ()
      );
      $1, Some 0
    }
  ;
declarator :
  | absolute_declarator { $1 }
  | pointer absolute_declarator { D_pointer $2 }
  ;
absolute_declarator :
  | Ident { D_symbol $1 }
  | Symbol_open_paren declarator Symbol_close_paren { $2 }
  | absolute_declarator Symbol_open_brace Symbol_close_brace
    { D_table ($1, expr E_none) }
  | absolute_declarator Symbol_open_brace constant_expr Symbol_close_brace
    { D_table ($1, $3) }
  | absolute_declarator Symbol_open_paren parameter_type_list Symbol_close_paren
    gnu_function_attribute
    { err "function 1 : f (type_a a, type_b b, ...)"; D_function $1 }
  | absolute_declarator Symbol_open_paren Symbol_close_paren
    gnu_function_attribute
    { err "function 2 : f ()"; D_function $1 }
  | absolute_declarator Symbol_open_paren identifier_list Symbol_close_paren
    gnu_function_attribute
    { err "function 3 : f (a, b, c, ...)"; D_function $1 }
  ;
gnu_function_attribute :
  | { () }
  | Keyword_gnu_attribute Symbol_open_paren Symbol_open_paren
    gnu_function_attribute_expr Symbol_close_paren Symbol_close_paren { () }
  ;
gnu_function_attribute_expr :
  | Ident { () }
  | Ident Symbol_open_paren expr Symbol_close_paren { () }
  | Ident Symbol_open_paren expr Symbol_comma expr Symbol_comma expr
    Symbol_close_paren { () }
  ;
pointer :
  | Symbol_mult { () }
  | Symbol_mult type_qualifier_list { () }
  | Symbol_mult pointer { () }
  | Symbol_mult type_qualifier_list pointer { () }
  ;
type_qualifier_list :
  | type_qualifier { () }
  | type_qualifier_list type_qualifier { () }
  ;
parameter_type_list :
  | parameter_list { () }
  | parameter_list Symbol_comma Symbol_dots { () }
  ;
parameter_list :
  | parameter_declaration { () }
  | parameter_list Symbol_comma parameter_declaration { () }
  ;
parameter_declaration :
  | declaration_specifier declarator { () }
  | declaration_specifier { () }
  | declaration_specifier abstract_declarator { () }
  ;
identifier_list :
  | Ident { [$1] }
  | identifier_list Symbol_comma Ident { $3::$1 }
  ;
initialisator :
  | affectation_expr { $1 }
  | bracket_open initialisator_list bracket_close
    { expr (E_liste $2) }
  | bracket_open initialisator_list Symbol_comma bracket_close
    { expr (E_liste $2) }
  ;
initialisator_list :
  | initialisator { [$1] }
  | initialisator_list Symbol_comma initialisator { $3::$1 }
  ;
type_name :
  | specifier_qualifier_list { () }
  | specifier_qualifier_list abstract_declarator { () }
  ;
abstract_declarator :
  | pointer { () }
  | absolute_abstract_declarator { () }
  | pointer absolute_abstract_declarator { () }
  ;
absolute_abstract_declarator :
  | Symbol_open_paren abstract_declarator Symbol_close_paren { () }
  | Symbol_open_brace Symbol_close_brace { () }
  | Symbol_open_brace constant_expr Symbol_close_brace { () }
  | absolute_abstract_declarator Symbol_open_brace Symbol_close_brace { () }
  | absolute_abstract_declarator Symbol_open_brace constant_expr
    Symbol_close_brace { () }
  | Symbol_open_paren Symbol_close_paren { () }
  | Symbol_open_paren parameter_type_list Symbol_close_paren { () }
  | absolute_abstract_declarator Symbol_open_paren Symbol_close_paren { () }
  | absolute_abstract_declarator Symbol_open_paren parameter_type_list
    Symbol_close_paren { () }
  ;
typedef_name :
  | Typedef_name { $1 }
  ; 
inst :
  | labeled_inst { expr E_none }
  | expr_inst { $1 }
  | composed_inst { expr E_none }
  | selection_inst { expr E_none }
  | iteration_inst { expr E_none }
  | jump_inst { expr E_none }
  ;
labeled_inst :
  | Ident Symbol_colon inst { () }
  | Keyword_case constant_expr Symbol_colon inst { () }
  | Keyword_default Symbol_colon inst { () }
  ;
expr_inst :
  | Symbol_semicolon { expr E_none }
  | expr Symbol_semicolon { $1 }
  ;
composed_inst :
  | bracket_open bracket_close { () }
  | bracket_open instruction_list bracket_close { () }
  | bracket_open declaration_list bracket_close { () }
  | bracket_open declaration_list instruction_list bracket_close
    { () }
  ;
instruction_list :
  | inst { () }
  | instruction_list inst { () }
  ;
selection_inst :
  | Keyword_if Symbol_open_paren expr Symbol_close_paren inst 
    %prec if_then { () }
  | Keyword_if Symbol_open_paren expr Symbol_close_paren inst Keyword_else
    inst %prec if_then_else { () }
  | Keyword_switch bracket_open expr bracket_close inst { () }
  ;
iteration_inst :
  | Keyword_while Symbol_open_paren expr Symbol_close_paren inst { () }
  | Keyword_do inst Keyword_while Symbol_open_paren expr Symbol_close_paren
    Symbol_semicolon { () }
  | Keyword_for Symbol_open_paren Symbol_semicolon Symbol_semicolon
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren Symbol_semicolon Symbol_semicolon expr
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren Symbol_semicolon expr Symbol_semicolon 
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren Symbol_semicolon expr Symbol_semicolon expr
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren expr Symbol_semicolon Symbol_semicolon
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren expr Symbol_semicolon Symbol_semicolon
    expr Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren expr Symbol_semicolon expr Symbol_semicolon
    Symbol_close_paren inst { () }
  | Keyword_for Symbol_open_paren expr Symbol_semicolon expr Symbol_semicolon
    expr Symbol_close_paren inst { () }
  ;
jump_inst :
  | Keyword_goto Ident Symbol_semicolon { I_goto $2 }
  | Keyword_continue Symbol_semicolon { I_continue }
  | Keyword_break Symbol_semicolon { I_break }
  | Keyword_return Symbol_semicolon { I_return (expr (E_int 0)) }
  | Keyword_return expr Symbol_semicolon { I_return $2 }
  ;
expr :
  | affectation_expr { $1 }
  | expr Symbol_comma affectation_expr { flip $3 }
  ;
affectation_expr :
  | conditional_expr { $1 }
  | unar_expr affectation_op affectation_expr { flip $3 }
  ;
affectation_op :
  | Symbol_affect { Op_affect }
  | Symbol_mult_affect { Op_mult_affect }
  | Symbol_div_affect { Op_div_affect }
  | Symbol_mod_affect { Op_mod_affect }
  | Symbol_add_affect { Op_add_affect }
  | Symbol_sub_affect { Op_sub_affect }
  | Symbol_left_affect { Op_left_affect }
  | Symbol_right_affect { Op_right_affect }
  | Symbol_and_affect { Op_and_affect }
  | Symbol_xor_affect { Op_xor_affect }
  | Symbol_or_affect { Op_or_affect }
  ;
conditional_expr :
  | or_logic_expr { $1 }
  | or_logic_expr Symbol_quest expr Symbol_colon conditional_expr { $3 }
  ;
constant_expr :
  | conditional_expr { $1 }
  ;
or_logic_expr :
  | and_logic_expr { $1 }
  | or_logic_expr Symbol_logic_and and_logic_expr { $3 }
  ;
and_logic_expr :
  | or_expr { $1 }
  | and_logic_expr Symbol_logic_or or_expr { $3 }
  ;
or_expr :
  | xor_expr { $1 }
  | or_expr Symbol_or xor_expr { $3 }
  ;
xor_expr :
  | and_expr { $1 }
  | xor_expr Symbol_xor and_expr { $3 }
  ;
and_expr :
  | equal_expr { $1 }
  | and_expr Symbol_and equal_expr { $3 }
  ;
equal_expr :
  | relational_expr { $1 }
  | equal_expr Symbol_equal relational_expr { $3 }
  | equal_expr Symbol_not_equal relational_expr { $3 }
  ;
relational_expr :
  | shift_expr { $1 }
  | relational_expr Symbol_less shift_expr { $3 }
  | relational_expr Symbol_more shift_expr { $3 }
  | relational_expr Symbol_more_or_equal shift_expr
	{ $3 }
  | relational_expr Symbol_less_or_equal shift_expr
    { $3 }
  ;
shift_expr :
  | add_expr { $1 }
  | shift_expr Symbol_left add_expr { $3 }
  | shift_expr Symbol_right add_expr { $3 }
  ;
add_expr :
  | mult_expr { $1 }
  | add_expr Symbol_add mult_expr { $3 }
  | add_expr Symbol_sub mult_expr { $3 }
  ;
mult_expr :
  | conv_expr { $1 }
  | mult_expr Symbol_mult conv_expr { $3 }
  | mult_expr Symbol_div conv_expr { $3 }
  | mult_expr Symbol_mod conv_expr { $3 }
  ;
conv_expr :
  | unar_expr { $1 }
  | Symbol_open_paren type_name Symbol_close_paren conv_expr { $4 }
  ;
unar_expr :
  | postfixed_expr { $1 }
  | Symbol_add_add unar_expr { flip $2 }
  | Symbol_sub_sub unar_expr { flip $2 }
  | unar_op conv_expr { $2 }
  | Keyword_sizeof unar_expr { $2 }
  | Keyword_sizeof Symbol_open_paren type_name Symbol_close_paren
    { expr (E_int 0) }
  ;
unar_op :
  | Symbol_and { Op_and }
  | Symbol_mult { Op_mult }
  | Symbol_add { Op_add }
  | Symbol_sub { Op_sub }
  | Symbol_neg { Op_neg }
  | Symbol_not { Op_not }
  ;
postfixed_expr :
  | primar_expr { $1 }
  | postfixed_expr Symbol_open_brace expr Symbol_close_brace { $1 }
  | postfixed_expr Symbol_open_paren Symbol_close_paren { $1 }
  | postfixed_expr Symbol_open_paren arg_expr_list Symbol_close_paren { $1 }
  | postfixed_expr Symbol_dot Ident { $1 }
  | postfixed_expr Symbol_arrow Ident { $1 }
  | postfixed_expr Symbol_add_add { flip $1 }
  | postfixed_expr Symbol_sub_sub { flip $1 }
  ;
primar_expr :
  | Ident { expr (E_ident $1) }
  | constant { $1 }
  | String { expr (E_string $1) }
  | Symbol_open_paren expr Symbol_close_paren { $2 }
  ;
arg_expr_list :
  | affectation_expr { [$1] }
  | arg_expr_list Symbol_comma affectation_expr { $3::$1 }
  ;
constant : 
  | Integer_const { expr (E_int $1) }
  | Char_const { expr (E_char $1) }
  | Float_const { expr (E_float $1) }
  | Enum_const { expr (E_enum $1) }
  ;
bracket_open :
  | Symbol_open_bracket { err "Open block"; Var.begin_block Var.vl }
  ;
bracket_close :
  | Symbol_close_bracket { err "Close block"; Var.end_block Var.vl }
  ;
%% (* Trailer *)
