/*
 *  COSS Naming Service administration tool for MICO
 *  Copyright (C) 1997 Kai-Uwe Sattler
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 mico@informatik.uni-frankfurt.de
 */

#include <CORBA.h>
#include <mico/impl.h>
#include "NamingClient.h"

#ifdef HAVE_EXCEPTIONS
#define CATCH(__exp, __name) 					\
  try {								\
    __exp;							\
  }								\
  catch (CosNaming::NamingContext::NotFound_catch &exc) {	\
    cerr << "NotFound: " << __name << endl;			\
  }								\
  catch (CosNaming::NamingContext::CannotProceed_catch &exc) {	\
    cerr << "CannotProceed: " << __name << endl;	       	\
  }								\
  catch (CosNaming::NamingContext::InvalidName_catch &exc) {	\
    cerr << "InvalidName: " << __name << endl;			\
  }								\
  catch (CosNaming::NamingContext::AlreadyBound_catch &exc) {	\
    cerr << "AlreadyBound: " << __name << endl;			\
  }
#else
#define CATCH(__exp, __name)			\
  {  						\
    __exp;					\
  }
#endif


NamingClient::NamingClient (CosNaming::NamingContext_ptr nc)
{
  root_nc = CosNaming::NamingContext::_duplicate (nc);
  current_nc = CosNaming::NamingContext::_duplicate (nc);
  current_path.length (0);
}

NamingClient::~NamingClient ()
{
}

void NamingClient::change_context (const char* s)
{
  CosNaming::Name name;
  CosNaming::NamingContext_var new_nctx;
  CORBA::Object_var obj = resolve_name (s);
#ifdef HAVE_EXCEPTIONS
  try {
#endif
    new_nctx = CosNaming::NamingContext::_narrow (obj);
#ifdef HAVE_EXCEPTIONS
  }
  catch (CORBA::SystemException_catch & sx) {
    new_nctx = CosNaming::NamingContext::_nil ();
  }
#endif
  if (! CORBA::is_nil (new_nctx)) {
    current_nc = CosNaming::NamingContext::_duplicate (new_nctx);

    create_name_from_string (s, name);
    current_path = name;
  }
}

void NamingClient::list_context (const char* s)
{
  if (s == NULL) {
    assert (! CORBA::is_nil (current_nc));
    list_context (current_nc);
  }
  else {
    CosNaming::NamingContext_var nctx;
    CORBA::Object_var obj = resolve_name (s);
#ifdef HAVE_EXCEPTIONS
    try {
#endif
      nctx = CosNaming::NamingContext::_narrow (obj);
#ifdef HAVE_EXCEPTIONS
    }
    catch (CORBA::SystemException_catch & sx) {
      nctx = CosNaming::NamingContext::_nil ();
    }
#endif
    if (!CORBA::is_nil (nctx))
      list_context (nctx);
    else {
      CORBA::ORB_var orb = CORBA::ORB_instance("mico-local-orb");
      cout << orb->object_to_string (obj) << endl;
    }
  }
}

void NamingClient::create_context (const char* s)
{
  CosNaming::NamingContext_var nc;
  CosNaming::Name name;

  create_name_from_string (s, name);
  CATCH (nc = root_nc->bind_new_context (name), s);
}

void NamingClient::print_current_context ()
{
  string path ("/");
  for (CORBA::ULong i = 0; i < current_path.length (); i++) {
    path += current_path[i].id;
    if (i < current_path.length () - 1)
      path += "/";
  }
  cout << path.c_str () << endl;
}

void NamingClient::remove_name (const char* s)
{
  CosNaming::Name name;
  create_name_from_string (s, name);
  CATCH (root_nc->unbind (name), s);
}

void NamingClient::bind (const char* s, CORBA::Object_ptr obj)
{
  CosNaming::Name name;
  create_name_from_string (s, name);
  CATCH (root_nc->bind (name, obj), s);
}

void NamingClient::create_name_from_string (const char* s, 
					    CosNaming::Name& name)
{
  string str (s);
  size_t pos = 0, p = 0;
  CORBA::ULong num = 0;

  if (str.length() >= 2 && str[0] == '\'') {
    str = str.substr (1);
    if (str[str.length()-1] == '\'')
      str = str.substr (0, str.length()-1);
    name = current_path;
    num = name.length();
    name.length (num+1);
    name[num].id = CORBA::string_dup (str.c_str());
    name[num].kind = CORBA::string_dup ("");
    return;
  }
  if (str == "..") {
    if (current_path.length () <= 1) {
      name.length (0);
      return;
    }
    name.length (current_path.length () - 1);
    for (CORBA::ULong i = 0; i < current_path.length () - 1; i++)
      name[i] = current_path[i];
  }
  else {
    if (str[0] == '/') {
      pos = 1;
    } else {
      name = current_path;
      num = name.length();
    }
    do {
      p = str.find ('/', pos);
      num++;

      string sub;
      if ((CORBA::Long)p < 0)
        sub = str.substr (pos);
      else
        sub = str.substr (pos, p - pos);
      pos = p + 1;
      name.length (num);
      name[num - 1].id = CORBA::string_dup (sub.c_str ());
      name[num - 1].kind = CORBA::string_dup ("");
    } while ((CORBA::Long)p >= 0);
  }
}

void NamingClient::list_context (CosNaming::NamingContext_ptr nc)
{
  CosNaming::BindingList_var blist;
  CosNaming::BindingIterator_var biter;
  
  nc->list (100, blist, biter);
  for (CORBA::ULong i = 0; i < blist->length (); i++) {
    cout << "\t";
    CosNaming::Binding b (blist[i]);
    for (CORBA::ULong k = 0; k < b.binding_name.length (); k++)
      cout << b.binding_name[k].id.in() << " ";
    cout << endl;
  }
}

CORBA::Object_ptr NamingClient::resolve_name (const char* s)
{
  CosNaming::Name name;
  CORBA::Object_ptr result = CORBA::Object::_nil();

  create_name_from_string (s, name);
  if (name.length() == 0)
    return CosNaming::NamingContext::_duplicate (root_nc);

  CATCH (result = root_nc->resolve (name), s);
  return result;
}

void
NamingClient::show_url (const char * s)
{
  CosNaming::NamingContextExt_var nce =
    CosNaming::NamingContextExt::_narrow (root_nc);

  if (CORBA::is_nil (nce)) {
    cerr << "we aren't talking to an Interoperable Naming Service" << endl;
    return;
  }

  /*
   * Determine address of name service
   */

  const CORBA::Address * addr =
    nce->_ior()->addr (CORBA::IORProfile::TAG_INTERNET_IOP);

  if (!addr) {
    cerr << "we aren't talking to the INS using IIOP" << endl;
    return;
  }

  char tmp[64];
  const MICO::InetAddress * ina = (const MICO::InetAddress *) addr;
  sprintf (tmp, "%lu", (unsigned long) ina->port());

  string straddr = ina->host();
  straddr += ':';
  straddr += tmp;

  CosNaming::Name name;
  create_name_from_string (s, name);
#ifdef HAVE_EXCEPTIONS
  try {
#endif
    CORBA::String_var r1, r2;
    r1 = nce->to_string (name);
    r2 = nce->to_url (straddr.c_str(), r1.in());
    cout << r2.in() << endl;
#ifdef HAVE_EXCEPTIONS
  }
  catch (CosNaming::NamingContext::NotFound_catch &exc) {
    cerr << "NotFound: " << s << endl;
  }
  catch (CosNaming::NamingContext::CannotProceed_catch &exc) {
    cerr << "CannotProceed: " << s << endl;
  }
  catch (CosNaming::NamingContext::InvalidName_catch &exc) {
    cerr << "InvalidName: " << s << endl;
  }
  catch (CosNaming::NamingContext::AlreadyBound_catch &exc) {
    cerr << "AlreadyBound: " << s << endl;
  }
#endif
}
