// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	__APE_SOCKET_H__
#define	__APE_SOCKET_H__

#ifndef	__APE_THREAD_H__
#include <APE/thread.h>
#endif

#include <winsock.h>

class InetHostAddress;
class InetMaskAddress;

class __EXPORT InetAddress 
{
protected:
	struct in_addr ipaddr;
	static MutexCounter counter;

public:
	InetAddress();
	InetAddress(struct in_addr addr);
	InetAddress(const char *address);

	char	*getHostname(void);

	inline struct in_addr getAddress(void)
		{return ipaddr;};
};	

class __EXPORT InetMaskAddress : public InetAddress
{
public:
	InetMaskAddress(const char *mask);

	friend InetHostAddress operator&(InetHostAddress &addr, InetMaskAddress &mask);
};

class __EXPORT InetHostAddress : public InetAddress
{
public:
	InetHostAddress(const char *host = NULL);
	InetHostAddress(struct in_addr addr);

	InetHostAddress &operator=(struct in_addr);
	bool operator==(InetHostAddress &a);
	bool operator!=(InetHostAddress &a);
	InetHostAddress &operator&=(InetMaskAddress &mask);

	friend class InetMaskAddress;
	friend InetHostAddress operator&(InetHostAddress &addr, InetMaskAddress &mask);
};

class __EXPORT BroadcastAddress : public InetAddress
{
public:
	BroadcastAddress(const char *net = "255.255.255.255");
};

class __EXPORT Socket
{
protected:
	SOCKET so;

	enum
	{
		SOCKET_INITIAL,
		SOCKET_AVAILABLE,
		SOCKET_BOUND,
		SOCKET_CONNECTED
	} state;

public:
	virtual ~Socket();

	Socket(int domain, int type, int protocol);
	Socket(SOCKET fd);
	Socket(const Socket &source);

	Socket &operator=(const Socket &s);
	InetHostAddress Peek(short *port = NULL);	

	inline virtual int Read(void *addr, size_t len)
		{return recv(so, (char *)addr, len, 0);};

	inline virtual int Write(void *addr, size_t len)
		{return send(so, (char *)addr, len, 0);};

	inline virtual int Peek(void *addr, size_t len)
		{return recv(so, (char *)addr, len, MSG_PEEK);};

	friend inline int read(Socket &s, void *addr, size_t len)
		{return s.Read(addr, len);};

	friend inline int write(Socket &s, void *addr, size_t len)
		{return s.Write(addr, len);};

	friend inline int peek(Socket &s, void *addr, size_t len)
		{return s.Peek(addr, len);};
	
	friend inline InetHostAddress peek(Socket &s, short *port = NULL)
		{return s.Peek(port);};
};

class __EXPORT UDPSocket : public Socket
{
public:
	UDPSocket(void);
	UDPSocket(InetAddress &bind, short port);
	UDPSocket(InetHostAddress &host, short port);
	UDPSocket(BroadcastAddress &cast, short port);

	int Write(InetHostAddress &host, short port, void *addr, size_t len);

	friend inline InetHostAddress peek(UDPSocket s)
		{return s.Peek();};	

	friend inline int write(UDPSocket &s, InetHostAddress &host, short port, void *addr, size_t len)
		{return s.Write(host, port, addr, len);};	
};

class __EXPORT UDPSimplex : public Socket
{
private:
	short peer;

public:
	UDPSimplex(InetAddress &bind, short port, short peer);

	int Broadcast(BroadcastAddress &subnet);	
	int Connect(InetHostAddress &host);
	int Disconnect(void);
	int Loopback(void);
};

class __EXPORT UDPDuplex
{
private:
	UDPSimplex *sender;
	UDPSimplex *receiver;

public:
	UDPDuplex(InetAddress &bind, short send, short recv);
	~UDPDuplex();

	int Broadcast(BroadcastAddress &subnet);
	int Connect(InetHostAddress &host);
	int Disconnect(void);
	int Loopback(void);

	inline int Peek(void *addr, size_t len)
		{return receiver->Peek(addr, len);};

	inline int Read(void *addr, size_t len)
		{return receiver->Read(addr, len);};

	inline int Write(void *addr, size_t len)
		{return sender->Write(addr, len);};
};

class __EXPORT TCPSocket : public Socket
{
public:
	TCPSocket(InetAddress &bind, short port, int backlog);
	TCPSocket(InetHostAddress &host, short port);
	TCPSocket(TCPSocket &server);
	TCPSocket(TCPSocket *server);

	inline int Write(char *buf)
		{return send(so, buf, strlen(buf), 0);};

	virtual bool OnAccept(InetHostAddress ia, short port)
		{return true;};

	int Readline(char *buf, size_t max);
	InetHostAddress getHost(short *port = NULL);
	InetHostAddress getPeer(short *port = NULL);

	friend inline int readline(TCPSocket &s, char *buf, size_t max)
		{return s.Readline(buf, max);};

	friend inline int write(TCPSocket &s, char *buf)
		{return send(s.so, buf, strlen(buf), 0);};
};

class TCPSession : public TCPSocket, public Thread
{
protected:
	void Final(void)
		{delete this;};
public:
	TCPSession(Semaphore *start, TCPSocket &server, int pri = 0, int stack = 0);
	TCPSession(TCPSocket *server, Semaphore *start = NULL, int pri = 0, int stack = 0);
};

inline struct in_addr getaddress(InetAddress &ia)
	{return ia.getAddress();};
#endif
