%
% Copyright (C) 1997 Thomas Nordin and Alastair Reid
%

\begin{code}

module Casm
	( Casm(..)
	, Kind(..)
	, ppKind
	, lookupKind    -- :: String -> Maybe Kind
        , kindToCType   -- :: Kind   -> String
	, Param(..)
	, ppCasm
	, Target(..)
	) where

import Pretty
import PrettyUtils  -- extensions

import Maybe  ( fromMaybe )
import Target ( Target(..) )
import Char   ( isAlpha )
\end{code}

%************************************************************************
%*									*
\subsection{Data structures}
%*									*
%************************************************************************

\begin{code}

data Casm 
  = Casm
      String   -- a unique label - in case we need to generate decls for this casm.
      String   -- name of external DLL/.so where the function lives. (FFI b.end only)
      String   -- Calling convention (FFI b.end only.)
      Bool     -- Are we going to be GC safe?
      Doc      -- initialisation code
      Doc      -- C code
      Doc      -- cleanup code
      [Param]  -- arguments
      [Param]  -- results

\end{code}

The @Kind@ type represent the repertoire of primitive Haskell types 
the FFI of a target supports.

\begin{code}

data Kind
  = Int
  | Word
  | Addr
  | Float
  | Double
  | PackedString  -- NHC extension?
  | Char
  | StablePtr
  | Foreign String -- freeing function
    -- GHC extensions
  | ByteArr
  | MutByteArr
  deriving (Show)

lookupKind :: String -> Maybe Kind
lookupKind s = lookup s' kindNames
 where
  s' = takeWhile isAlpha (dropWhile (not.isAlpha) s)

kindNames :: [(String, Kind)]
kindNames =
  [ ("Char",   Char)
  , ("Int",    Int)
  , ("Word", Word)
  , ("Addr",   Addr)
  , ("Float",  Float)
  , ("Double", Double)
  , ("PackedString", PackedString)
  , ("StablePtr", StablePtr)
  , ("ByteArray", ByteArr)
  , ("MutableByteArray", MutByteArr)
  , ("ForeignObj", Foreign "")
  ]

ppKind :: Kind -> Doc
ppKind k = text ("%%" ++ show k)

kindToCType :: Target -> Kind -> String
kindToCType GHC_ccall k  = kindToStgType k
kindToCType _         k  =
 case k of
  Int          -> "int"
  Word         -> "unsigned int"
  Addr         -> "void *"
  Float        -> "float"
  Double       -> "double"
  Char         -> "char"
  PackedString -> "char *"
  StablePtr    -> "long"
  (Foreign _)  -> "void *"
  ByteArr      -> "void *"
  MutByteArr   -> "void *"

kindToStgType :: Kind -> String
kindToStgType k =
 "Stg" ++
 case k of 
   Int        -> "Int"
   Word       -> "Word"
   Addr       -> "Addr"
   Float      -> "Float"
   Double     -> "Double"
   Char       -> "Char"
   StablePtr  -> "StablePtr"
   ByteArr    -> "ByteArray"
   MutByteArr -> "ByteArray"
   Foreign _  -> "ForeignObj"

-- NHC specific
kindToCall :: Kind -> String
kindToCall Int          = "GET_INT_VALUE(nodeptr)"
kindToCall Char         = "GET_CHAR_VALUE(nodeptr)"
--kindToCall Bool         = "GET_BOOL_VALUE(nodeptr)"
kindToCall Float        = "GET_FLOAT_VALUE(nodeptr)"
kindToCall Double       = "GET_DOUBLE_VALUE(nodeptr)"
kindToCall PackedString = "getPackedString(nodeptr)"
kindToCall Word         = "GET_INT_VALUE(nodeptr)"
kindToCall Addr         = "getVoidStar(nodeptr)"
kindToCall StablePtr    = "stableInsert(nodeptr)"
kindToCall (Foreign _)  = "(cdataArg((CData*)GET_INT_VALUE(nodeptr)))->cval"

kindToRtn :: Kind -> String -> String
kindToRtn Int          v = "mkInt("++v++")"
kindToRtn Char         v = "mkChar("++v++")"
--kindToRtn Bool         v = "mkBool("++v++")"
kindToRtn Float        v = "mkFloat("++v++")"
kindToRtn Double       v = "mkDouble("++v++")"
kindToRtn PackedString v = "mkString("++v++")"
kindToRtn Word         v = "mkInt("++v++")"
kindToRtn Addr         v = "mkAddr("++v++")"
kindToRtn StablePtr    v = "stableRef("++v++")"
kindToRtn (Foreign f)  v = "mkForeign((void*)"++v++",(gccval)"++f++")"

\end{code}

The @Param@ type encodes a value that is either an argument
or a result. 

\begin{code}
data Param = 
  Param {
	-- the name to use on the Haskell side to bind
	-- the parameter value to.
    haskellName :: String,  -- the name to use name to be used where?
        -- the name that the parameter is bound to inside C chunks.
    cName	:: String,
	-- the expression the parameter is initially bound to.
    cExpr	:: String,
	-- the C type of the parameter.
    cType	:: String,
        -- the basic Haskell type of the parameter.
    paramKind	:: Kind,
	-- for parameters that are cast to another type before being
	-- used inside a C chunk (via a %declare DIS), the cast needs
	-- to be performed at some stage. By setting the 'needsDecl'
	-- flag, code to initialise/cast a 'declare'd variable will
	-- be emitted by the code backend.
	-- 
	-- [ This flag is only ever looked at by the backends that generate
	--   a separate C function stub to hold the body of a procedure
	--   specification (i.e., they cannot express this inline as with
	--   _casm_s and the like.) 'Normal' parameters are then bound
	--   to C function parameter names, but ones that are declared
	--   needs to be handled specially, hence the need for this extra
	--   field in the Param type.
	-- ]
    needsDecl   :: Bool
  }

getKind :: Param -> Kind
getKind p = paramKind p
\end{code}


%************************************************************************
%*									*
\subsection{The main entry points}
%*									*
%************************************************************************

ToDo:

o If the arg names overlap with the result names, we should either:
    - not redeclare the result holder (assuming the same name is used)
     or
    - report an error

    - ignore it

  since it is almost certainly not what the user wanted and they
  can easily rename to avoid the warning.

@ppCasm@ produces C/Haskell code, returning a four tuple containing:
 
  - Haskell code to invoke Haskell/C code that interfaces to
    the external function we're interfacing to. GHC backends also
    bind the Haskell results (if any) to variable names. This
    is done to try to generate code that doesn't use `return'
    needlessly.
  - optional Haskell declaration for Haskell stub.
  - optional C prototypes to stub functions that performs the
    actual call (and unmarshals the result(s))
  - optional C code that massages arguments/results around the
    call to the actual function we're interfacing to.
  - entry in primitive table (Hugs only.)

GHC uses only the first, inlining the function/code block we want
to call upon from Haskell.

\begin{code}
ppCasm :: Target 
       -> Bool 
       -> Casm 
       -> ( Doc
	  , Doc
	  , Doc
	  , Doc
	  , Doc
	  )
ppCasm Hugs debugStubs (Casm name _ _ gcsafe start ccode end args results) =
  ( text name' <+> textline (map haskellName args)
  , text "primitive" <+> text name' <+> text "::" <+> typ
  , empty
  , ppPrimDecl Hugs name'
      (  withSemi start
      $$ vcatMap (pprParamToC Hugs) args
      $$ braces
           (  ccode
           $$ vcatMap pprParamFromC (reverse results)
           $$ ppSimpleApply "hugs_returnIO" (text $ show $ length results) <> semi
           $$ end
           )
      )
  , ppStruct [doubleQuotes $ text name', text (show arity), text name']
  )
 where
  typ   = ppType Hugs (map getKind args) (map getKind results)

  -- The IO monad takes 2 extra arguments
  arity = length args + 2
  name' = addPrimPrefix name

ppCasm FFI debugStubs (Casm name ext_name cconv gcsafe start ccode end args results) =
 ( (text name') <+> hsep (map (text.haskellName) args) $$ pop
 , text "foreign import " <+> text cconv <+> ext_loc <+>
   safecall <+> text name' <+> text "::" <+> prim_type $$
   access_decls
 , proto_ghc $$ access_proto_ghc -- haskell decl.
 , ppPrimStgDecl c_type
    	(  declareResult
        $$ start
	$$ stubDebugCode debugStubs name
    	$$ hsep (map initArg args)
	$$ ppBlock (ccode $$ withSemi push)
        ) $$ access_stubs
 , proto $$ access_proto
 )
 where
    ext_loc =
      (if null ext_name then empty else text (show ext_name)) <+>
      (text (show name'))

    safecall
     | gcsafe    = empty
     | otherwise = text "unsafe"

    name'      = addPrimPrefix name
    proto      = text "extern" <+> c_type     <> semi
    proto_ghc  = text "extern" <+> c_type_stg <> semi
    prim_type  = ppType FFI (map getKind args) prim_result_ty
    prim_result_ty 
      | length results > 1 = [Addr]
      | otherwise	   = map getKind results

    c_type_stg = ret_type_stg   <+> 
                 ppSimpleApply name' ((map (arg_ghc) args) `sepdBy` comma)
    c_type = ret_type   <+> 
             ppSimpleApply name' ((map (arg) args) `sepdBy` comma)

    arg_ghc p = text (kindToStgType (paramKind p)) <+> text (haskellName p)

    arg p = text (cType p) <+> text (cName p)
    {-
     C stub return either void, a prim type or a pointer to
     a structure holding multiple results.
    -}
    (ret_type,ret_type_stg) =
      case results of
	 []  -> let v = text "void" in (v,v)
	 [p] -> (text (cType p), text (kindToStgType (paramKind p)))
	 _   -> (text "void*", text "StgAddr")

    (declareResult, push, pop, access_stubs, access_proto_ghc, access_proto, access_decls) = 
                                 resultPassing FFI (cconv, ext_name) name' end results

      -- for the FFI stub generated from an Casm, initialise any parameters that
      -- are passed in at type different to which it is used inside the stub body.
      -- (i.e., for any parameter that subsequently falls under the spell of a 
      --  %declare DIS.)
    initArg p | needsDecl p = ppAssign (haskellName p) (parens (text (cType p)) <> text (cName p))
    initArg _		    = empty

ppCasm GHC_ccall debugStubs (Casm name _ _ gcsafe start ccode end args results) =
 ( (ccall gcsafe) (text name') (map (text.haskellName) args) $$
   pop
 , empty
 , proto_ghc $$ access_proto_ghc -- haskell decl.
 , ppPrimStgDecl c_type
    	(  declareResult
        $$ start
	$$ stubDebugCode debugStubs name
	$$ ppBlock (ccode $$ withSemi push)
        ) $$ access_stubs
 , proto $$ access_proto
 )
 where
    name'      = addPrimPrefix name
    proto      = text "extern" <+> c_type     <> semi
    proto_ghc  = text "extern" <+> c_type_stg <> semi
    c_type_stg = ret_type_stg   <+> 
                 ppSimpleApply name' ((map (arg_ghc) args) `sepdBy` comma)
    c_type = ret_type   <+> 
             ppSimpleApply name' ((map (arg) args) `sepdBy` comma)

    arg_ghc p =
        (text (kindToStgType (paramKind p))) <+> text (haskellName p)

    arg p = text (cType p) <+> text (haskellName p)
    {-
     C stub return either void, a prim type or a pointer to
     a structure holding multiple results.
    -}
    (ret_type,ret_type_stg) =
      case results of
	 []  -> let v = text "void" in (v,v)
	 [p] -> (text (cType p), text (kindToStgType (paramKind p)))
	 _  -> (text "void*", text "StgAddr")

    (declareResult, push, pop, access_stubs, access_proto_ghc, access_proto, _) = 
                                 resultPassing GHC_ccall ("","") name' end results

ppCasm GHC_casm debugStubs (Casm name _ _ gcsafe start ccode end args results) =
  ( casm gcsafe
    	(  declareResult
        $$ start
    	$$ hsep (zipWith initArg args [0..])		
        $$ stubDebugCode debugStubs name
	$$ ppBlock (ccode $$ push )
	) (map (text.haskellName) args)
      $$ pop
  , empty
  , empty
  , empty
  , empty
  )
  where
    (declareResult, push, pop, access_stubs, access_proto_ghc, access_proto, _) = 
           resultPassing GHC_casm ("","") name end results

    initArg p n = ppAssign (cExpr p) (parens (text (cType p)) <> text ("%" ++ show n))

ppCasm NHC debugStubs (Casm name _ _ gcsafe init ccode end args results) =
  ( text "(IO ( _ -> let res = " <>
    text name' <+> textline (map haskellName args) <+>
    text "in seq res (Right res)))"
  , text name' <+> text "primitive" <+> text arity <+> text "::" <+> typ
  , empty
  , ppPrimDecl NHC name'
      (  init
      $$ vcatMap (pprParamToC NHC) args
      $$ ppBlock
           (  ccode
           $$ ppAssign "nodeptr" pprResult
           $$ end
	   $$ ptext "C_RETURN(nodeptr);"
           )
      )
  , ppStruct [doubleQuotes $ text name', text (show arity), text name']
  )
 where
  name' = addPrimPrefix name
  typ   = ppType NHC (map getKind args) (map getKind results)
  arity = show (length args)
  res   = results

  pprResult = 
   case res of
     [] -> text "mkUnit()"
     _  -> 
       let
        rs = map (\ p -> text (kindToRtn (paramKind p) (haskellName p))) res
	n  = length rs
       in
       text "mkTuple" <> 
       text (show n)  <> 
       parens (commaList rs)

\end{code}

%************************************************************************
%*									*
\subsection{Hugs code for constructing a casm}
%*									*
%************************************************************************

The only complication in printing the type is that we replace Stable
Pointers with type variables.  Fortunately, any type variables will do
(since the type var doesn't affect how we pack or unpack) so we just
use @a1@ ... @am@ for the arguments and @r1@ ... @rn@ for the results.

\begin{code}

ppType :: Target -> [Kind] -> [Kind] -> Doc
ppType target args res 
  = (zipWith (kindToType target) atvs args ++ [r]) `sepdBy` text " -> "
   where
    atvs = [ 'a':show i | i <- [1..] ]
    rtvs = [ 'r':show i | i <- [1..] ]
    r    = 
      case target of { Hugs -> (text "IO" <+>) ; FFI -> (text "IO" <+>) ; _ -> id} $
      ppTuple (zipWith (kindToType target) rtvs res)

kindToType :: Target -> String -> Kind -> Doc
kindToType tgt tv k@StablePtr = text (show k) <+> text tv
kindToType tgt tv Word        = text "Word32"
kindToType tgt tv (Foreign _) = text "ForeignObj"
kindToType tgt tv k           = text (show k)

\end{code}

Hugs specific functions for wrapping and unwrapping
arguments and results.

\begin{code}

pprParamToC :: Target -> Param -> Doc
pprParamToC Hugs param =
 case param of
  Param nm _ _ cty (Foreign free) _ ->
    ppAssign nm (ppCast (text cty) (text "hugs->getForeign()"))
  Param nm _ _ _ StablePtr _ ->
    ppAssign nm (text "hugs->getInt()")  -- now marshalled in Haskell
  Param _ _ expr cty kind _ ->
    ppAssign expr src
   where
    src = parens (text cty) <> parens (text ("hugs->get" ++ show kind ++ "()"))

pprParamToC NHC param =
  case param of
    Param _ _ expr cty kind _ -> 
         ppAssign "nodeptr" get_arg   
      $$ text "IND_REMOVE(nodeptr);" 
      $$ ppAssign expr (parens (text cty) <> text (kindToCall kind))
      $$ text ""
     where
      nodeptr = text "nodeptr"
      get_arg = text "C_GETARG1(arg_count++)"

pprParamFromC :: Param -> Doc
pprParamFromC (Param _ _ expr _ (Foreign free) _) =
     text "hugs->putForeign" <> ppTuple [text expr, text free] <> semi
pprParamFromC (Param _ _ expr _ StablePtr _) =
     ppSimpleApply "hugs->putInt" (text expr) <> semi
pprParamFromC (Param _ _ expr _ kind _) =
     ppSimpleApply ("hugs->put" ++ show kind) (text expr) <> semi

\end{code}

\begin{code}

ppPrimDecl :: Target -> String -> Doc -> Doc
ppPrimDecl Hugs name body 
    =  ppSimpleApply "PROTO_PRIM" (text name) <> semi
    $$ ppSimpleApply "primFun" (text name)
    $$ char '{'
    $$ indent body
    $$ char '}'
ppPrimDecl NHC name body 
    =  (text "C_HEADER" <> parens (text name))
    $$ char '{'
    $$ indent body
    $$ char '}'

ppPrimStgDecl :: Doc -> Doc -> Doc
ppPrimStgDecl c_type body
    =  c_type
    $$ char '{'
    $$ indent body
    $$ char '}'
\end{code}

%************************************************************************
%*									*
\subsection{GHC code for returning multiple results from a casm}
%*									*
%************************************************************************

resultPassing decides which policy to use for returning results from C.
There are three possibilities:

\begin{enumerate}
\item Returning nothing is easy:

  casm ``...'' >> return ()

\item Returning one thing is easy too:

  casm ``...''

\item Returning many things is tricky.  We pack up all the bits into a
  struct and return a pointer to the struct.  We then read each returned
  value out of the struct.  Blech!

  casm ``static struct { int res1; float res2; } gc_result;
         ...;
         gc_result.res1 = res1;
         gc_result.res2 = res2;
         %r = &gc_result;'' >>= \ gc_result ->
  casm ``%r = (struct { int res1; float res2; }*)%0 -> x;'' (gc_result::Addr) >>= \ x ->
  casm ``%r = (struct { int res1; float res2; }*)%0 -> y;'' (gc_result::Addr) >>= \ y ->


\end{itemize}

\begin{code}

resultPassing :: Target	  -- Currently just GHC_{ccall,casm}
	      -> (String,String)
	      -> String	  -- function name
	      -> Doc	  -- cleanup code
	      -> [Param] 
	      -> ( Doc	  -- How to declare the var
                 , Doc	  -- How to save the vars in C
		 , Doc    -- C function stubs to fetch pieces of result.
		 , Doc    -- Prototypes for C function stubs 
		          --   (STG signatures.)
		 , Doc    -- Prototypes for C function stubs 
		          --   (`native' signatures.)
                 , Doc    -- How to get them back in Haskell
		 , Doc    -- Possible foreign decls for the access stubs.
		 )     
-- No vars
resultPassing target _ nm end [] =
  ( empty
  , end
  , empty
  , empty
  , empty
  , empty
  , empty
  )

-- One var
resultPassing target _ nm end [x@(Param n _ e _ _ _)] =
 case target of
  GHC_ccall ->
    ( empty
    , end $$ ppSimpleApply "return" (text e)
    , textline [">>= \\ ", n, " ->"]
    , empty
    , empty
    , empty
    , empty
    )
  FFI ->
    ( empty
    , end $$ ppSimpleApply "return" (text e)
    , textline [">>= \\ ", n, " ->"]
    , empty
    , empty
    , empty
    , empty
    )
  GHC_casm ->
   ( empty
   , end $$ ppAssign "%r" (text e)
   , textline [">>= \\ ", n, " ->"]
   , empty
   , empty
   , empty
   , empty
   )
-- Many vars
resultPassing target (cconv,ext_nm) nm end xs =
 case target of 
  GHC_ccall ->
    ( text "static" <+> structTy <+> text "gc_result;"
    , vcatMap copyIn xs $$ end $$ return_result
    , text ">>= \\ gc_result ->" $$ vcatMap copyOut xs
    , vcatMap mkCopyOutStub xs
    , vcat protos_ghc
    , vcat protos
    , empty
    )
  FFI ->
    ( text "static" <+> structTy <+> text "gc_result;"
    , vcatMap copyIn xs $$ end $$ return_result
    , text ">>= \\ gc_result ->" $$ vcatMap copyOut xs
    , vcatMap mkCopyOutStub xs
    , vcat protos_ghc
    , vcat protos
    , vcat ffi_decls
    )
  GHC_casm ->
    ( text "static" <+> structTy <+> text "gc_result;"
    , vcatMap copyIn xs $$ end $$ return_result
    , text ">>= \\ gc_result ->" $$ vcatMap copyOut xs
    , empty
    , empty
    , empty
    , empty
    )
 where
  (protos_ghc, protos, ffi_decls) = unzip3 $ map mkCopyOutProtos xs

  structTy = text "struct" <+> braces (semiList (map decl xs) <> semi)
   where
    decl :: Param -> Doc
    decl p = text (kindToCType target (paramKind p)) <+> text (haskellName p)

  return_result =
    case target of
      GHC_casm  -> text "%r = &gc_result;"
      GHC_ccall -> ppSimpleApply "return" (text "&gc_result")
      FFI       -> ppSimpleApply "return" (text "&gc_result")

  -- Copy a return value into the result structure.
  copyIn :: Param -> Doc
  copyIn p = ppAssign ("gc_result." ++ haskellName p) (text (cExpr p))

  -- Copy a return value out of the result structure.
  copyOut :: Param -> Doc
  copyOut p =
     let n = haskellName p in
     case target of 
      FFI ->
        text "access_" <> text nm <> text "_" <> text n <+>
	hsep [(text "(gc_result :: Addr)")] <+> textline [">>= \\", n, "->"]
      GHC_ccall ->
        (ccall False{-not safe-}) (text "access_" <> text nm <> text "_" <> text n)
	[(text "(gc_result :: Addr)")] <+> textline [">>= \\", n, "->"]
      GHC_casm ->
	(casm False{-not safe-} (ppAssign "%r" ("%0" `deRef` n)) 
	         [text "(gc_result :: Addr)"])   <+> 
	textline [">>= \\", n, "->"]

  {-
    C_ret_type access_funNm_field(void *ptr) 
    { return ((struct _type)ptr)->field; }
  -}
  mkCopyOutStub p =
    let n = haskellName p in
    text (cType p) <+> text (access_nm n) <> 
    text "(void *ptr){ return(" <> ("ptr" `deRef` n) <> text ");}"

  mkCopyOutProtos (Param n _ _ ct k _) =
   ( mkProto (kindToStgType k) (kindToStgType Addr)
   , mkProto ct "void*"
   , text "foreign import" <+> text cconv <+> (ext_loc n) <+> 
     text "unsafe" <+> text (access_nm n) <+> text "::" <+>
     ppType FFI [Addr] [k]
   )
    where
     mkProto ret_ty arg_ty  =
      text "extern " <>  (text ret_ty) <+>
      text (access_nm n) <> text ('(':arg_ty++");")

  ext_loc n =
      (if null ext_nm then empty else text (show ext_nm)) <+>
      (text (show (access_nm n)))

  access_nm n = "access_" ++ nm ++ '_':n

  deRef arg n = ppCast (structTy <> char '*') (text arg) <> 
                text "->" <> 
		text n

\end{code}

%************************************************************************
%*									*
\subsection{GHC Specific Utilities}
%*									*
%************************************************************************

\begin{code}

casm :: Bool -> Doc -> [Doc] -> Doc
casm safe d args =
  ((if safe then ("_casm_GC_ ``", "''") else ("_casm_ ``", "''"))
       `around`
   (ppBlock (indent d))) <+> hsep args

ccall :: Bool -> Doc -> [Doc] -> Doc
ccall safe d args = (text call <+> d) <+> hsep args
  where
   call | safe      = "_ccall_GC_"
        | otherwise = "_ccall_"

stubDebugCode :: Bool -> String -> Doc
stubDebugCode True nm = text ("__current_fun__ = " ++ show nm) <> semi
stubDebugCode _    _  = empty

addPrimPrefix :: String -> String
addPrimPrefix nm = "prim_" ++ nm
\end{code}

%************************************************************************
%*									*
\subsection{Tests}
%*									*
%************************************************************************

COMMENTED OUT.

\begin{code}
{-
tst t = putStrLn $ render (call $$ decl $$ entry $$ c)
 where
  (call, decl, entry, c) = ppCasm Hugs False t

t1 = Casm 
       "foo"
       False
       (text "int x; void* q; char p; void* q")
       (text "p = foo(&q,x,y);")
       (text "free q")
       [ Param "x" "x" "int" Int    
       , Param "q" "q" "int *"  Addr] 
       [ Param "p" "'p'" "char" Char
       , Param "q" "q" "char *" Addr] 
-}
\end{code}

