/****************************************************************************
**
** output.c
**
** Michael S. Borella 
**
** Make genparse output, a .h file, a .c file, and a callback file
**
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <malloc.h>
#include "parse_cl.h"
#include "genparse.h"
#include "error.h"
#include "mytime.h"
#include "getuserinfo.h"

extern clarg_t arg_array[];
extern char version[];
extern FILE *yyin;
extern char global_callback[];
extern struct arg_t *my_args;
extern int include_ptr;
extern char *includes[];
extern char *mandatory;

/*---------------------------------------------------------------------------
 * 
 * create_comments()
 *
 *-------------------------------------------------------------------------*/

void create_comments(FILE *fp, char *filename, char *purpose)
{

  char *user;

  /*
   * Get user information
   */

  user = getuserinfo();


  /*
   * Tell user what we're doing
   */

  printf("creating %s...", filename);


  /*
   * Print some nice, friendly comments and some #include's
   */

  fprintf(fp, "/*****************************************************************************\n");
  fprintf(fp, "**\n");
  fprintf(fp, "** %s\n", filename);
  fprintf(fp, "**\n");

  /*
   * Get current time
   */
  
  mytime_gettime();

  fprintf(fp, "** %s", mytime_ctime());
  fprintf(fp, "** %s\n", user);
  fprintf(fp, "**\n");
  fprintf(fp, "** %s\n", purpose);
  fprintf(fp, "**\n");
  fprintf(fp, "** Automatically created by genparse v%s\n", version);
  fprintf(fp, "**\n");
  fprintf(fp, "** See http://www.borella.net/mike/Software/ for details\n");
  fprintf(fp, "**\n");
  fprintf(fp, "*****************************************************************************/\n");
  fprintf(fp, "\n");
  
}


/*---------------------------------------------------------------------------
 * 
 * create_header_file()
 *
 *-------------------------------------------------------------------------*/

char *create_header_file(char *outfile, clarg_t *arg_array)
{
  int i, n;
  FILE *fp;
  char *fn;
  char *ptr;
  char temp[TOKEN_SIZE];
  char temp2[TOKEN_SIZE];
  char *type;
  void ftlerr(char *,...);
  int get_current_flag(void);

  /*
   * Form file name - replace .c with .h
   */

  if (outfile==NULL)
    ftlerr("output file not specified");

  fn = strdup(outfile);
  ptr = strrchr(fn, '.');
  if (ptr)
    {
      ptr++;
      *ptr = 'h';
    }
  else
    ftlerr("File name does not end with ',c'");


  /*
   * Open output file
   */

  fp = fopen(fn, "w");
  if (!fp) 
    ftlerr("can't open %s for writing", fn);


  /*
   * The the 'generic' part of the comments, then the specific stuff
   */

  create_comments(fp, fn, "Header file for command line parsing routines");
  fprintf(fp, "#include <stdio.h>\n");
  fprintf(fp, "\n");


  /*
   * Print the return type of the parsing routine
   * Insert help param first
   */

  fprintf(fp, "struct arg_t\n");
  fprintf(fp, "{\n");
  fprintf(fp, "  int h;\n");
  n = get_current_flag();
  for (i=0; i<=n; i++)
    {
      switch(arg_array[i].type)
	{
	case STRING_T:
	  fprintf(fp, "  char *%c;\n", 
		  arg_array[i].short_param);
	  break;
	case FLAG_T:
	case INT_T:
	  fprintf(fp, "  int %c;\n", 
		  arg_array[i].short_param);
	  break;
	case FLOAT_T:
	  fprintf(fp, "  float %c;\n", 
		  arg_array[i].short_param);
	  break;
	case CHAR_T:
	  fprintf(fp, "  char %c;\n", 
		  arg_array[i].short_param);
	  break;
	}
    }
  fprintf(fp, "  int optind;\n");
  fprintf(fp, "};\n");
  fprintf(fp, "\n");

  /*
   * Get rid of that nasty .h
   */

  strncpy(temp, outfile, strlen(outfile));
  temp[strlen(outfile)-2] = EOS;

  /*
   * Some useful function prototypes
   */

  fprintf(fp, "struct arg_t *%s(int, char **);\n", temp);
  fprintf(fp, "void usage(char *);\n");
  fprintf(fp, "void free_args(struct arg_t *);\n");

  /*
   * More prototypes - callback functions this time
   */

  if (global_callback[0])
    {
      fprintf(fp, "int %s(struct arg_t *);\n", global_callback);
    }
  for (i=0; i<=n; i++)
    {
      if (arg_array[i].callback[0])
	{

	  switch(arg_array[i].type)
	    {
	    case INT_T:
	    case FLAG_T:
	      type = strdup("int");
	      break;
	    case STRING_T:
	      type = strdup("char *");
	      break;
	    case FLOAT_T:
	      type = strdup("float");
	      break;
	    case CHAR_T:
	      type = strdup("char");
	      break;
	    default: /* This gets rid of a compiler warning */
	      type = strdup("");
	      ftlerr("bad parameter type");
	    }

	  sprintf(temp2, "int param_%c_callback(%s);\n", 
		  arg_array[i].short_param, type);
	  fprintf(fp, temp2);
	}
    }

  fclose(fp);
  return(fn);

}

/*---------------------------------------------------------------------------
 * 
 * print_header()
 *
 *-------------------------------------------------------------------------*/

void print_header(FILE *fp, char *outfile, char *h_file)
{
  int i;

  /*
   * Print generic part of comments
   */

  create_comments(fp, outfile, "Main command line parsing routines");
  fprintf(fp, "\n");
  fprintf(fp, "#include <stdio.h>\n");
  fprintf(fp, "#include <unistd.h>\n");
  fprintf(fp, "#include <stdlib.h>\n");
  fprintf(fp, "#include <string.h>\n");
  fprintf(fp, "#include <getopt.h>\n");

  /*
   * Mandatory parameter, if any;
   */

  if (mandatory)
    fprintf(fp, "\nchar mandatory[] = \"%s\";\n", mandatory);


  /*
   * Include files from genparse file
   */

  i = 0;
  while(i < include_ptr)
    fprintf(fp, "#include <%s>\n", includes[i++]);

  /*
   * Our own header file 
   */

  fprintf(fp, "#include \"%s\"\n", h_file);

  

  fprintf(fp, "\n");
}


/*---------------------------------------------------------------------------
 * 
 * print_funcbegin()
 *
 *-------------------------------------------------------------------------*/

void print_funcbegin(FILE *fp, clarg_t *a, char *outfile)
{
  char temp[TOKEN_SIZE];
  int i,n;
  int get_current_flag(void);


  /*
   * Get rid of that nasty .c
   */

  strncpy(temp, outfile, strlen(outfile));
  temp[strlen(outfile)-2] = EOS;

  /*
   * Print the function declaration and local vars
   */

  fprintf(fp, "/*----------------------------------------------------------------------------\n");
  fprintf(fp, " *\n");
  fprintf(fp, " * %s()\n", temp);
  fprintf(fp, " *\n");
  fprintf(fp, " *----------------------------------------------------------------------------\n");
  fprintf(fp, " */\n\n");
  fprintf(fp, "struct arg_t *%s(int argc, char *argv[])\n", temp);
  fprintf(fp, "{\n");
  fprintf(fp, "  extern char *optarg;\n");
  fprintf(fp, "  extern int optind;\n");
  fprintf(fp, "  int option_index = 0;\n");
  fprintf(fp, "  char c;\n");
  fprintf(fp, "  struct arg_t *my_arg;\n");
  fprintf(fp, "  int errflg = 0;\n\n");  


  /*
   * Determine number of parameters
   */

  n = get_current_flag();


  /*
   * Loop through all args, building struct options for the long args
   * Do help param first.
   */
  
  fprintf(fp, "  static struct option long_options[] =\n");
  fprintf(fp, "  {\n");
  fprintf(fp, "    {\"help\", 0, 0, 'h'},\n"); /* 1st 0 = no params */
  for (i=0; i<=n; i++)
    {
      fprintf(fp, "    {\"%s\", 1, 0, '%c'},\n", a[i].long_param, 
              a[i].short_param);
    }
  fprintf(fp, "    {0, 0, 0, 0}\n");
  fprintf(fp, "  };\n");
  fprintf(fp, "\n\n");


  /*
   * Make malloc for argument structure
   */

  fprintf(fp, "  my_arg = (struct arg_t *) malloc (sizeof(struct arg_t));\n\n");


  /*
   * Write default values if any.  Do help first
   */
  
  fprintf(fp, "  my_arg->h = 0;\n");
  for (i=0; i<=n; i++)
    {
      if (a[i].type == FLAG_T)
	{
	  fprintf(fp, "  my_arg->%c = 0;\n", a[i].short_param);
	}
      else
	{
	  if (a[i].type == STRING_T)
	    {
	      if (a[i].default_val[0])
		fprintf(fp, "  my_arg->%c = strdup(\"%s\");\n", 
			a[i].short_param, a[i].default_val);
	      else
		fprintf(fp, "  my_arg->%c = NULL;\n", a[i].short_param);
	    }
	  else 
	    if (a[i].default_val[0])
	      fprintf(fp, "  my_arg->%c = %s;\n", a[i].short_param, 
		      a[i].default_val);
	}
    }
  fprintf(fp, "\n");
  
  
  /*
   * Make the getopt_long() call
   */

  fprintf(fp, "  while ((c = getopt_long(argc, argv, \"");
  i = 0;
  fprintf(fp, "h");
  while (i <= n)
    {
      if (a[i].type == FLAG_T) 
	fprintf(fp, "%c", a[i].short_param);
      else
	fprintf(fp, "%c:", a[i].short_param);
      i++;
    }
  fprintf(fp, "\", long_options, &option_index)) != EOF)\n");


  /*
   * Print the switch statement
   */

  fprintf(fp, "    {\n");
  fprintf(fp, "      switch(c)\n");
  fprintf(fp, "        {\n");
}

 
/*---------------------------------------------------------------------------
 * 
 * print_flags()
 *
 *-------------------------------------------------------------------------*/

void print_flags(FILE *fp, clarg_t *a)
{
  int i, n;
  int get_current_flag(void);

  /*
   * Determine number of parameters
   */

  n = get_current_flag();

  /*
   * Iterate through the params, doing help first
   */

  fprintf(fp, "        case 'h':\n");
  fprintf(fp, "          my_arg->h ++;\n");
  fprintf(fp, "          break;\n\n");	      
  
  for (i=0; i<=n; i++)
    {
      fprintf(fp, "        case '%c':\n", a[i].short_param);
      
      /*
       * Store in argument structure of proper type
       */

      switch(a[i].type)
	{
	case STRING_T:
	  fprintf(fp, "          my_arg->%c = strdup(optarg);\n", 
		  a[i].short_param);
	  break;
	case INT_T:
	  fprintf(fp, "          my_arg->%c = atoi(optarg);\n", 
		  a[i].short_param);
	  break;
	case FLOAT_T:
	  fprintf(fp, "          my_arg->%c = atof(optarg);\n", 
		  a[i].short_param);
	  break;
	case CHAR_T:
	  fprintf(fp, "          my_arg->%c = *optarg;\n", 
		  a[i].short_param);
	  break;
	case FLAG_T:
	  fprintf(fp, "          my_arg->%c ++;\n", a[i].short_param);
	  break;
	}

      /*
       * Do range checking
       */
      
      if (a[i].low_range[0])
	{
	  fprintf(fp, "          if (my_arg->%c < %s)\n", a[i].short_param,
		  a[i].low_range);
	  fprintf(fp, "            {\n");
	  fprintf(fp, "              fprintf(stderr, \"parameter range error: %c must be >= %s\\n\");\n",
		  a[i].short_param, a[i].low_range);
	  fprintf(fp, "              errflg ++;\n");
	  fprintf(fp, "            }\n");
	}
      if (a[i].high_range[0])
	{
	  fprintf(fp, "          if (my_arg->%c > %s)\n", a[i].short_param,
		  a[i].high_range);
	  fprintf(fp, "            {\n");
	  fprintf(fp, "              fprintf(stderr, \"parameter range error: %c must be <= %s\\n\");\n",
		  a[i].short_param, a[i].high_range);
	  fprintf(fp, "              errflg ++;\n");
	  fprintf(fp, "            }\n");
	}

      /*
       * Do callbacks
       */

      if (arg_array[i].callback[0])
	{
	  fprintf(fp, "          if (!param_%c_callback(my_arg->%c))\n", 
		  arg_array[i].short_param, arg_array[i].short_param);
	  fprintf(fp, "            usage(argv[0]);\n");
	}
      fprintf(fp, "          break;\n\n");	      
    }

  

  /*
   * Print default action for unknown parameters
   */

  fprintf(fp, "        default:\n");
  fprintf(fp, "          usage(argv[0]);\n");
  fprintf(fp, "\n");

}


/*---------------------------------------------------------------------------
 * 
 * print_funcend()
 *
 *-------------------------------------------------------------------------*/

void print_funcend(FILE *fp)
{
  fprintf(fp, "        }\n");
  fprintf(fp, "    } /* while */\n");
  fprintf(fp, "\n");
  fprintf(fp, "    if (errflg)\n");
  fprintf(fp, "      usage(argv[0]);\n");
  fprintf(fp, "\n");
  fprintf(fp, "    if (!%s(my_arg))\n", global_callback);
  fprintf(fp, "      usage(argv[0]);\n");
  fprintf(fp, "\n");
  fprintf(fp, "  my_arg->optind = optind;\n");
  fprintf(fp, "  return my_arg;\n");
  fprintf(fp, "}\n");
}


/*---------------------------------------------------------------------------
 * 
 * print_usage()
 *
 *-------------------------------------------------------------------------*/

void print_usage(FILE *fp, clarg_t *a)
{
  int i, n;
  int get_current_flag(void);

  /*
   * Determine number of parameters
   */

  n = get_current_flag();

  /*
   * Print the function declaration
   */
  
  fprintf(fp, "/*----------------------------------------------------------------------------\n");
  fprintf(fp, " *\n");
  fprintf(fp, " * usage()\n");
  fprintf(fp, " *\n");
  fprintf(fp, " *----------------------------------------------------------------------------\n");
  fprintf(fp, " */\n\n");
  fprintf(fp, "void usage(char *executable)\n");
  fprintf(fp, "{\n");
  

  fprintf(fp, "  printf(\"usage: ");
  fprintf(fp, "%s", "%s ");
  fprintf(fp, "[ -");
  for (i = 0; i<=n; i++)
    fprintf(fp, "%c", a[i].short_param);
  if (mandatory)
    fprintf(fp, " ] %s\\n\", executable", "%s ");
  else
    fprintf(fp, " ] \\n\", executable");
  if (mandatory)
    fprintf(fp, ", mandatory);\n");
  else
    fprintf(fp,");\n");

  /*
   * Do help then iterate through params
   */

  fprintf(fp, "  printf(\"  [ -h ] [ --help ] Print help screen \\n\");\n");

  for (i=0; i<=n; i++)
    {
      fprintf(fp, "  printf(\"  [ -%c ] \");\n", a[i].short_param);
      if (a[i].long_param[0]) 
	fprintf(fp, "  printf(\"[ --%s ] \");\n", a[i].long_param);
      if (a[i].desc[0]) 
	fprintf(fp, "  printf(\"%s\");\n", a[i].desc);
      if (a[i].default_val[0])
	fprintf(fp, "  printf(\" (default = %s)\");\n", a[i].default_val);
      fprintf(fp, "  printf(\"\\n\");\n");
    }

  fprintf(fp, "\n  exit(1);\n");
  fprintf(fp, "}\n");
  fprintf(fp, "\n");
  
}


/*---------------------------------------------------------------------------
 * 
 * create_callbacks()
 *
 *-------------------------------------------------------------------------*/

void create_callbacks(char *outfile, clarg_t *a)
{
  int i, n;
  char *ptr;
  char *type;
  char filename[TOKEN_SIZE];
  char str[TOKEN_SIZE];
  FILE *fp;
  int cb_flag = 0;
  int get_current_flag(void);

  /*
   * Determine number of parameters
   */

  n = get_current_flag();


  /*
   * First make sure that we have to do anything at all
   */

  if (global_callback[0])
    cb_flag = 1;
  else
    {
      for (i=0; i<=n; i++)
	{
	  if (a[i].callback[0])
	    cb_flag = 1;
	}
    }
  if (!cb_flag)
    return;


  /*
   * Form file name - replace .c with _cb.c
   */
  
  strncpy(filename, outfile, TOKEN_SIZE);
  ptr = strrchr(filename, '.');
  if (ptr)
    { /* this is lame - i'll fix it later */
      *ptr = '_';
      ptr ++;
      *ptr = 'c';
      ptr ++;
      *ptr = 'b';
      ptr ++;
      *ptr = '.';
      ptr++;
      *ptr = 'c';
      ptr++;
      *ptr = '\0';
    }
  else
    ftlerr("File name does not have a '.'");


  /*
   * Open output file if it does not already exist
   */

  fp = fopen(filename, "r");
  if (fp != NULL)
    {
      printf("%s already exists...backing up...", filename);
      sprintf(str, "cp %s %s.bak", filename, filename);
      if (system(str) < 0)
	ftlerr("Can't copy %s to backup file", filename);
    }

  fp = fopen(filename, "w");
  if (!fp) 
    ftlerr("can't open %s for writing", filename);
  

  /*
   * The the 'generic' part of the comments, then the specific stuff
   */

  create_comments(fp, filename, "Callback functions for command line parsing routines");
  fprintf(fp, "#include <stdio.h>\n");

  /*
   * Now we have to change filename again...sigh.
   */

  ptr = strrchr(filename, '_');
  if (ptr)
    {
      *ptr = '.';
      ptr++;
      *ptr = 'h';
      ptr++;
      *ptr = '\0';
    }
  else
    ftlerr("File name does not end with '_cb.c'");

  fprintf(fp, "#include \"%s\"\n", filename);
  fprintf(fp, "\n");


  /*
   * Just leave the functions blank for now -- the user will fill them in
   */

  if (global_callback[0])
    {
      fprintf(fp, "int %s(struct arg_t *a)\n", global_callback);
      fprintf(fp, "{\n");
      fprintf(fp, "  return 1;\n");
      fprintf(fp, "}\n\n");
    }

  for (i=0; i<=n; i++)
    {
      if (arg_array[i].callback[0])
	{
	  switch(arg_array[i].type)
	    {
	    case INT_T:
	    case FLAG_T:
	      type = strdup("int");
	      break;
	    case STRING_T:
	      type = strdup("char *");
	      break;
	    case FLOAT_T:
	      type = strdup("float");
	      break;
	    case CHAR_T:
	      type = strdup("char");
	      break;
	    default: /* This gets rid of a compiler warning */
	      type = strdup("");
	      ftlerr("bad parameter type");
	    }

	  fprintf(fp, "int param_%c_callback(%s var)\n", 
		  arg_array[i].short_param, type);
	  fprintf(fp, "{\n");
	  fprintf(fp, "  return 1;\n");
	  fprintf(fp, "}\n\n");
	}
    }
  

}


/*---------------------------------------------------------------------------
 * 
 * print_free_args
 *
 *-------------------------------------------------------------------------*/

void print_free_args (FILE *fp, clarg_t *a)
{
  int i,n;
  int get_current_flag(void);

  n = get_current_flag();

  /*
   * Print the function declaration
   */
  
  fprintf(fp, "/*----------------------------------------------------------------------------\n");
  fprintf(fp, " *\n");
  fprintf(fp, " * free_args()\n");
  fprintf(fp, " *\n");
  fprintf(fp, " *----------------------------------------------------------------------------\n");
  fprintf(fp, " */\n\n");
  fprintf(fp, "void free_args(struct arg_t *my_arg)\n");
  fprintf(fp, "{\n");
  
  for (i = 0; i<=n; i++)
    if (a[i].type == STRING_T)
      fprintf(fp, "  if (my_arg->%c != NULL) free(my_arg->%c);\n", 
	      a[i].short_param, a[i].short_param);

  fprintf(fp, "  free(my_arg);\n");
  fprintf(fp, "}\n");
  fprintf(fp, "\n");
}
