/*
 *	fhist - file history and comparison tools
 *	Copyright (C) 1991, 1992, 1993, 1994, 1995, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	Derived from a work
 *	Copyright (C) 1990 David I. Bell.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: operating system entry point, and parse arguments
 */

#include <ac/stdio.h>
#include <errno.h>
#include <ac/stdlib.h>
#include <ac/string.h>

#include <arglex.h>
#include <cmalloc.h>
#include <compare.h>
#include <error_intl.h>
#include <fileio.h>
#include <help.h>
#include <progname.h>
#include <version.h>
#include <work.h>

short outputflag;
short conflictflag;

enum
{
	arglex_token_conflicts,
	arglex_token_debug,
	arglex_token_failure,
	arglex_token_ignore,
	arglex_token_output,
	arglex_token_unchanged,
	arglex_token_verbose,
};

static arglex_table_ty argtab[] =
{
	{ "-Conflicts",	arglex_token_conflicts,	},
	{ "-Debug",	arglex_token_debug,	},
	{ "-Failure",	arglex_token_failure,	},
	{ "-Ignore",	arglex_token_ignore,	},
	{ "-Output",	arglex_token_output,	},
	{ "-Unchanged",	arglex_token_unchanged,	},
	{ "-Verbose",	arglex_token_verbose,	},
	{ 0, 0, }, /* end marker */
};


static void usage _((void));

static void
usage()
{
	char		*progname;

	progname = progname_get();
	fprintf(stderr, "usage: %s [ <option>... ] <basefile> <file1> <file2>\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERSion\n", progname);
	exit(1);
}


static void main_help _((void));

static void
main_help()
{
	help(NULL, usage);
}


/*
 * Expand a global variable name.
 * Returns original argument if string is not a global variable,
 * or returns copied value of global variable.
 */

static char *expand _((char *));

static char *
expand(cp)
	char		*cp;	/* string to be checked */
{
	char		*var;	/* variable value */

	if (!cp || (*cp != '.'))
		return cp;
	var = getenv(cp);
	if (!var || !*var)
		return cp;
	cp = r_alloc_and_check(strlen(var) + 1);
	strcpy(cp, var);
	return cp;
}


/*
 * This is the main program.
 */

int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	char		*cp;		/* current option character */
	char		*basename;	/* file which is the base file */
	char		*nameA;
	char		*nameB;		/* the two divergent files */
	char		*ofile;		/* output merged filename */
	char		*cfile;		/* output conflict filename */
	int		filecount;	/* number of files specified */
	sub_context_ty	*scp;

	arglex_init(argc, argv, argtab);
	switch (arglex())
	{
	case arglex_token_help:
		main_help();
		exit(0);

	case arglex_token_version:
		version();
		exit(0);

	default:
		break;
	}
	basename = NULL;
	nameA = NULL;
	nameB = NULL;
	ofile = NULL;
	cfile = NULL;
	fc.verbosity = VERBOSE_DEFAULT;
	fc.debugflag = 0;
	outputflag = 0;
	conflictflag = 0;
	ignoreflag = 0;
	filecount = 0;
	failcount = INFINITY;
	unchangecount = INFINITY;
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			bad_argument(usage);
			/* NOTREACHED */

		case arglex_token_string:
			/* a file name */
			cp = expand(arglex_value.alv_string);
			filecount++;
			if (!basename)
				basename = cp;
			else if (!nameA)
				nameA = cp;
			else if (!nameB)
				nameB = cp;
			else
			{
				fatal_intl
				(
					0,
					i18n("only three input files allowed")
				);
			}
			break;

		case arglex_token_debug:
			/* type debugging information */
			fc.debugflag = 1;
			break;

		case arglex_token_ignore:
			/* ignore conflicts */
			ignoreflag = 1;
			break;

		case arglex_token_verbose:
			/* be verbose */
			if (arglex() != arglex_token_number)
			{
				fc.verbosity = VERBOSE_FULL;
				continue;
			}
			fc.verbosity = arglex_value.alv_number;
			if (fc.verbosity < 0)
			{
				scp = sub_context_new();
				sub_var_set(scp, "Number", "%d", fc.verbosity);
				fatal_intl
				(
					scp,
					i18n("verbosity $number out of range")
				);
				/* NOTREACHED */
				sub_context_delete(scp);
			}
			break;

		case arglex_token_output:
			/* specify output file */
			if (arglex() != arglex_token_string)
				usage();
			outputflag = 1;
			ofile = expand(arglex_value.alv_string);
			break;

		case arglex_token_conflicts:
			/* specify conflict file */
			conflictflag = 1;
			if (arglex() != arglex_token_string)
				continue;
			cfile = expand(arglex_value.alv_string);
			break;

		case arglex_token_failure:
			/* fail if too many conflicts */
			if (arglex() != arglex_token_number)
			{
				fatal_intl
				(
					0,
				     i18n("please specify maximum change count")
				);
			}
			failcount = arglex_value.alv_number;
			if (failcount < 0)
			{
				scp = sub_context_new();
				sub_var_set(scp, "Number", "%d", failcount);
				fatal_intl
				(
					scp,
				i18n("failure count of $number is out of range")
				);
				/* NOTREACHED */
				sub_context_delete(scp);
			}
			break;

		case arglex_token_unchanged:
			/* limit number of unchanged lines */
			if (arglex() != arglex_token_number)
				fatal_intl(0, i18n("please specify maximum unchanged count"));
			unchangecount = arglex_value.alv_number;
			if (unchangecount < 0)
			{
				scp = sub_context_new();
				sub_var_set(scp, "Number", "%d", unchangecount);
				fatal_intl
				(
					scp,
				     i18n("unchange count $number out of range")
				);
				/* NOTREACHED */
				sub_context_delete(scp);
			}
			break;
		}
		arglex();
	}
	if (filecount == 1)
	{
		/* converting conflict file */
		if (conflictflag)
			fatal_intl(0, i18n("conflicting options for conflict file conversion"));
		convertconflicts(basename, ofile);
		exit(0);
	}
	if (filecount != 3)
		fatal_intl(0, i18n("must have three input files for merge"));
	if (!outputflag && !conflictflag)
		outputflag = 1;
	fmerge(basename, nameA, nameB);
	if (outputflag)
		dumpmergedfile(ofile);
	if (conflictflag)
		dumpconflicts(cfile);
	if (conflicts && !conflictflag)
	{
		scp = sub_context_new();
		sub_var_set(scp, "Number", "%ld", (long)conflicts);
		sub_var_optional(scp, "Number");
		fatal_intl(scp, i18n("conflict detected"));
		/* NOTREACHED */
		sub_context_delete(scp);
	}
	if (fc.verbosity && conflicts)
	{
		scp = sub_context_new();
		sub_var_set(scp, "Number", "%ld", conflicts);
		sub_var_optional(scp, "Number");
		error_intl
		(
			scp,
			i18n("warning: conflict detected")
		);
		sub_context_delete(scp);
	}
	exit(0);
	return 0;
}
