{
  (* This script converts the Hint commands of Coq 6.2.2 to Hint commands
    for Coq 6.2.3 *)

  let stop mesg = 
    begin
      output_string stderr ("h_convert : "^mesg^"\n");
      flush stderr;
      exit 1
    end
 
  let usage () =
    begin
      List.iter (fun x ->  output_string stderr (x^"\n"))
      	["";
	 "This script converts the Hint/Auto syntax from Coq 6.2.2 to 6.2.3.";
	 "";
	 "Usage : h_convert file.v <hint list> <auto list>";
	 "Example : h_convert Le.v \"v62 arith\" \"arith logic\"";
	 " (be careful with the parentheses) ";
	 "";
	 "The first argument is the file to convert.";
	 "It is saved as file.v.OLD before doing the conversion";
	 "";
	 "The second argument is a list of names to convert the Hint commands.";
	 "In our example, it will replace :";
	 "  \"Hint foo bar\"         with \"Hints Resolve foo bar : v62 arith\"";
	 "  \"Hint Unfold foo bar\"  with \"Hints Unfold foo bar : v62 arith\"";
	 "  \"Immediate foo bar\"    with \"Hints Immediate foo bar : v62 arith\"";
	 "";
	 "The third argument is a list of names to convert the Auto and Trivial tactics.";
	 "In our example, it will replace :";
	 "  \"Auto\"                 with \"Auto with arith logic\"";
	 "  \"Trivial\"              with \"Trivial with arith logic\"";
	 "  \"EAuto\"                with \"EAuto with arith logic\"";
	 "";
	];
      flush stderr;
      exit 2
    end

  let _ = 
    if Array.length Sys.argv <> 4 then usage ()
  let file_name = Sys.argv.(1)
  let ofile_name = file_name^".OLD"
  let dbnames = match Sys.argv.(2) with
      "" -> ""
    | s  -> " : "^s
  let auto_arg = match Sys.argv.(3) with
      "" -> ""
    | s -> " with "^s
  let _ = 
    let com = "cp "^file_name^" "^ofile_name in
      if Sys.command com <> 0 
      then stop ("Can't create file"^ofile_name^". Abort")
  let inch = open_in ofile_name 
  let outch = open_out file_name
}

let identchar = 
  ['A'-'Z''a'-'z''_''\'''0'-'9']
let space = [' ''\t''\n']

rule conv = parse
  | "Hint Unfold"(space+identchar+)+space*'.'
      { let s = Lexing.lexeme lexbuf in
       	let s' = String.sub s 11 (String.length s - 11) in
	let s' = String.sub s' 0 (String.length s' - 1) in
	let s'' = "Hints Unfold "^s'^dbnames^"." in
      	  begin 
	    output_string outch s'';
	    conv lexbuf
	  end
      }
  |    "Hint"(space+identchar+)+space*'.' 
      { let s = Lexing.lexeme lexbuf in
      	let s' = String.sub s 4 (String.length s - 4) in
	let s' = String.sub s' 0 (String.length s' - 1) in
	let s'' = "Hints Resolve"^s'^dbnames^"." in
      	  begin 
	    output_string outch s'';
	    conv lexbuf
	  end
      }
  |  "Immediate"(space+identchar+)+space*'.'
      { let s = Lexing.lexeme lexbuf in
       	let s' = String.sub s 9 (String.length s - 9) in
	let s' = String.sub s' 0 (String.length s' - 1) in
	let s'' = "Hints Immediate"^s'^dbnames^"." in
          begin 
	    output_string outch s'';
	    conv lexbuf
	  end
      }
  | 'E'?"Auto"(space+['0'-'9']+)? 
    {begin 
       output_string outch ((Lexing.lexeme lexbuf)^auto_arg);
       conv lexbuf
     end}
 
 | "Trivial"
    {begin 
       output_string outch ((Lexing.lexeme lexbuf)^auto_arg);
       conv lexbuf
     end}

      (* Cette regle est ncessaire pour ne pas attraper les identificateurs 
	 qui contiendraient la sous-chane "Auto" *)
  | identchar+ 
    {begin 
       output_string outch (Lexing.lexeme lexbuf);
       conv lexbuf
     end}
  | _  
      {begin 
	 output_string outch (Lexing.lexeme lexbuf);
	 conv lexbuf
       end} 
  | eof {
      print_string ("Conversion done. Old file saved as "^ofile_name);
      print_newline ();
      close_in inch;
      flush outch;
      close_out outch;
      exit 0}

{
  let _ = conv (Lexing.from_channel inch) 
}
