(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                   tcc.ml                                 *)
(****************************************************************************)

(* JCF -- 6 janvier 1998  EXPERIMENTAL *)

(*
 *  L'ide est, en quelque sorte, d'avoir de "vraies" mtavariables
 *  dans Coq, c'est--dire de donner des preuves incompltes -- mais
 *  o les trous sont typs -- et que les sous-buts correspondants
 *  soient engendrs pour finir la preuve.
 *
 *  Exemple : 
 *    J'ai le but
 *        (x:nat) { y:nat | (minus y x) = x }
 *    et je donne la preuve incomplte
 *        [x:nat](exist nat [y:nat]((minus y x)=x) (plus x x) ?)
 *    ce qui engendre le but
 *        (minus (plus x x) x)=x
 *)

(*  Pour cela, on procde de la manire suivante :
 *
 *  1. Un terme de preuve incomplet est un terme contenant des variables
 *     existentielles (XTRA "ISEVAR") i.e. "?" en syntaxe concrte.
 *     La rsolution de ces variables n'est plus ncessairement totale
 *     (ise_resolve called with fail_evar=false) et les variables
 *     existentielles restantes sont remplaces par des mta-variables
 *     castes par leur types (celui est connu : soit donn, soit trouv
 *     pendant la phase de rsolution).
 *
 *  2. On met ensuite le terme " plat" i.e. on n'autorise des MV qu'au
 *     permier niveau et pour chacune d'elles, si ncessaire, on donne
 *      son tour un terme de preuve incomplet pour la rsoudre.
 *     Exemple: le terme (f a ? [x:nat](e ?)) donne
 *         (f a ?1 ?2) avec ?2 => [x:nat]?3 et ?3 => (e ?4)
 *         ?1 et ?4 donneront des buts
 *
 *  3. On crit ensuite une tactique tcc qui engendre les sous-buts
 *      partir d'une preuve incomplte.
 *)

open Pp
open Std
open More_util
open Names
open Generic
open Term
open Termenv
open Tacmach
open Himsg
open Reduction
open Trad
open Proof_trees
open Tactics
open Tacticals


type metamap = (int * constr) list

type term_with_holes = TH of constr * metamap * sg_proofs
and  sg_proofs       = (term_with_holes option) list

(* pour debugger *)

let rec pp_th (TH(c,mm,sg)) =
  [< 'sTR"TH=[ "; hOV 0 [< pTERM c; 'fNL;
			   (* pp_mm mm; 'fNL; *)
			   pp_sg sg >] ; 'sTR "]" >]
and pp_mm l =
  hOV 0 (prlist_with_sep (fun _ -> [< 'fNL >]) 
	   (fun (n,c) -> [< 'iNT n; 'sTR" --> "; pTERM c >]) l)
and pp_sg sg =
  hOV 0 (prlist_with_sep (fun _ -> [< 'fNL >])
	   (function None -> [< 'sTR"None" >]
	      | Some th -> [< pp_th th >]) sg)
     


(*  compute_metamap : constr -> term_with_holes
 *  ralise le 2. ci-dessus
 *
 *  Pour cela, on renvoie une metamap qui indique pour chaque meta-variable
 *  si elle correspond  un but (None) ou si elle rduite  son tour
 *  par un terme de preuve incomplet (Some c).
 *
 *  On a donc l'INVARIANT suivant : le terme c rendu est "de niveau 1"
 *  -- i.e.  plat -- et la metamap contient autant d'lments qu'il y 
 *  a de meta-variables dans c. On suppose de plus que l'ordre dans la
 *  metamap correspond  celui des buts qui seront engendrs par le refine.
 *)

let replace_by_meta sign = function
  | TH (DOP0(Meta _) | DOP2(Cast,DOP0(Meta _),_) as m, mm, sgp) -> m,mm,sgp
  | (TH (c,mm,_)) as th ->
      let n = newMETA() in
      let m = DOP0(Meta n) in
      (* quand on introduit une mv on calcule son type *)
      let ty = match c with
	| DOP2(Lambda,c1,DLAM(Name id,DOP2(Cast,_,ty))) ->
	    mkNamedProd id c1 ty
	| DOP2(Lambda,c1,DLAM(Anonymous,DOP2(Cast,_,ty))) ->
	    mkArrow c1 ty
	| DOPN((AppL|MutCase _),_) ->
	    let j = ise_resolve true empty_evd mm (gLOB sign) c in
	      j._TYPE
	| DOPN(Fix (_,j),v) ->
	    v.(j) (* en pleine confiance ! *)
	| _ -> invalid_arg "Tcc.replace_by_meta (TO DO)" 
      in
      DOP2(Cast,m,ty),[n,ty],[Some th]

exception NoMeta

let replace_in_array sign a =
  let l = Array.to_list a in
    if List.for_all (function (TH (_,_,[])) -> true | _ -> false) l then
      raise NoMeta
    else
      let a' = Array.map (function
			      (TH (c,mm,[])) -> c,mm,[]
			    | th -> replace_by_meta sign th)
		 a in
      let v' = Array.map (fun (x,_,_) -> x) a' in
      let mm = Array.fold_left (@) [] (Array.map (fun (_,x,_) -> x) a') in
      let sgp = Array.fold_left (@) [] (Array.map (fun (_,_,x) -> x) a') in
 	v',mm,sgp

let fresh sign n =
  let id = match n with Name x -> x | _ -> id_of_string "_" in
  next_global_ident_away id (ids_of_sign sign)

let rec compute_metamap sign = function

  (* le terme est directement une preuve *)
    DOP0(Sort _)
  | DOPN((Const _ | Abst _ | MutInd _ | MutConstruct _),_)
  | VAR _ | Rel _ as c -> TH (c,[],[])

  (* le terme est une mv => un but *)
  | DOP0(Meta n) as c ->
      Pp.warning (Printf.sprintf ("compute_metamap: MV(%d) sans type !\n") n);
      TH (c,[],[None])
  | DOP2(Cast,DOP0(Meta n),ty) as c -> TH (c,[n,ty],[None])

  (* abstraction => il faut dcomposer si le terme dessous n'est pas pur
   *    attention : dans ce cas il faut remplacer (Rel 1) par (VAR x)
   *    o x est une variable FRAICHE *)
  | DOP2(Lambda,c1,DLAM(name,c2)) as c ->
      let v = fresh sign name in
      let tj = ise_resolve_type true empty_evd [] (gLOB sign) c1 in
      let sign' = add_sign (v,tj) sign in
      begin match compute_metamap sign' (subst1 (VAR v) c2) with
	  (* terme de preuve complet *)
	  TH (_,_,[]) -> TH (c,[],[])

	  (* terme de preuve incomplet *)    
	| th ->
	    let m,mm,sgp = replace_by_meta sign' th in
	    TH (DOP2(Lambda,c1,DLAM(Name v,m)), mm, sgp)
      end

  (* 4. Application *)
  | DOPN((AppL|MutCase _) as op,v) as c ->
      let a = Array.map (compute_metamap sign) v in
	begin
	  try
	    let v',mm,sgp = replace_in_array sign a in TH (DOPN(op,v'),mm,sgp)
	  with NoMeta ->
	    TH (c,[],[])
	end

  (* 5. Fix. *)
  | DOPN(Fix _,_) as c ->
      let (ni,i,ai,fi,v) = destFix c in
      let vi = List.rev (List.map (fresh sign) fi) in
      let sign' =
	List.fold_left
	  (fun sign (v,ar) -> add_sign (v,ar) sign)
	  sign
	  (List.combine vi (Array.to_list ai)) in
      let a = Array.map
		(compute_metamap sign')
		(Array.map (substl (List.map (fun x -> VAR x) vi)) v) in
	begin
	  try
	    let v',mm,sgp = replace_in_array sign' a in
	    let fi' = List.rev (List.map (fun id -> Name id) vi) in
	    let fix = mkFix ni i ai fi' v' in
	      TH (fix,mm,sgp)
	  with NoMeta ->
	    TH (c,[],[])
	end
	      
  (* Cast. Est-ce bien exact ? *)
  | DOP2(Cast,c,t) -> compute_metamap sign c
      (*let TH (c',mm,sgp) = compute_metamap sign c in
	TH (DOP2(Cast,c',t),mm,sgp) *)
      
  (* Produit. Est-ce bien exact ? *)
  | DOP2(Prod,_,_) as c ->
      if occur_meta c then
	error "Refine: proof term contains metas in a product"
      else
      	TH (c,[],[])
 
  (* Autres cas. *)
  | DOP0 Implicit ->
      error "Implicits not allowed in Refine"

  | _ ->
      invalid_arg "Tcc.compute_metamap"


(*  tcc_aux : term_with_holes -> tactic
 * 
 *  Ralise le 3. ci-dessus
 *)

let rec tcc_aux (TH (c,mm,sgp) as th) gl =

  match (c,sgp) with
    (* mv => sous-but : on ne fait rien *)
      (DOP0(Meta _) | DOP2(Cast,DOP0(Meta _),_)) , _ ->
	tclIDTAC gl
	  
    (* terme pur => refine *)
    | _,[] ->
	refine c gl
	
    (* abstraction => intro *)
    | DOP2(Lambda,_,
	   DLAM(Name id,(DOP0(Meta _)|DOP2(Cast,DOP0(Meta _),_) as m))) ,_ ->
	begin match sgp with
	    [None] -> introduction id gl
	  | [Some th] ->
	      tclTHEN (introduction id) (tcc_aux th) gl
	  | _ -> invalid_arg "Tcc.tcc_aux (bad length)"
	end

    | DOP2(Lambda,_,_),_ ->
	error "invalid abstraction passed to function tcc_aux !"
      
    (* fix => tactique Fix *)
    | DOPN(Fix _,_) , _ ->
      	let (ni,_,ai,fi,_) = destFix c in
	let ids =
	  List.map (function Name id -> id | _ ->
		      error "recursive functions must have names !") fi in
	  tclTHENS
	    (mutual_fix ids (List.map succ (Array.to_list ni))
                    (List.map body_of_type (List.tl (Array.to_list ai))))
	    (List.map (function
			   None -> tclIDTAC 
			 | Some th -> tcc_aux th) sgp)
	    gl

    (* sinon on fait refine du terme puis appels rec. sur les sous-buts.
     * c'est le cas pour AppL et MutCase. *)
    | _ ->
	tclTHENS
	  (refine c)
	  (List.map (function None -> tclIDTAC | Some th -> tcc_aux th) sgp)
	  gl

(* Et finalement la tactique refine elle-mme : *)

let refine c gl =
  let sign = pf_hyps gl in
  let th = compute_metamap sign c in
    tcc_aux th gl

let refine_tac = Tacmach.hide_constr_tactic "Refine" refine

let my_constr_of_com_casted sigma sign com typ = 
  let env = gLOB sign in
  let c = Astterm.raw_constr_of_com sigma env com in
  let cc = mkCast (nf_ise1 sigma c) (nf_ise1 sigma typ) in
    try (ise_resolve false sigma [] env cc)._VAL
    with e -> Stdpp.raise_with_loc (Ast.loc com) e
;;

let dyn_tcc args gl = 
  match args with 
      [(COMMAND com)]  -> 
	let sign = pf_hyps gl in
        refine
	  (my_constr_of_com_casted (project gl) sign com (pf_concl gl)) gl
    | [(CONSTR c)] -> refine c gl
    | _ -> assert false
;;

let tcc_tac = hide_tactic "Tcc" dyn_tcc;;

