(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               progmach.ml                                *)
(****************************************************************************)

open Std;;
open Pp;;
open More_util;;
open Vectops;;
open Names;;
open Generic;;
open Impuniv;;
open Term;;
open Initial;;
open Printer;;
open Himsg;;
open Reduction;;
open Typing;;
open Machops;;
open Termenv;;
open Environ;;
open Termenv;;
open Astterm;;
open Constrtypes;;
open Tradevar;;


let traceme (s:string) (j:judgement) = ();;

let j_val = Machops.j_val;;
let assumption_of_judgement = Machops.assumption_of_judgement ;;

let vect_lift = map_i_vect lift 0;;

let vect_lift_type = map_i_vect (fun i -> type_app (lift i)) 0;;

let make_Kn n = 
 let rec makerec = function
    0 -> (Rel (n+1))
  | n -> mkLambda Anonymous mkImplicit (makerec (n-1))
 in makerec n
    
;;

let is_case_rec isrectok =
  match isrectok with
    CoqAst.Str(_,"REC") -> true
  | _ -> false

let mt_evd = (Evd.mt_evd() : unit Evd.evar_map);;

(* Reduction ... *)

let red_mind sigma ((nparams,sp,ar,lc),la) =
  let num_to_eat = min nparams (List.length la) in
  let eaten,rest = chop_list num_to_eat la in
  let x_arg_eater = make_Kn num_to_eat in
  let rest_nparams = nparams - num_to_eat in
  ((rest_nparams,sp,
    hnf_prod_applist sigma "red_ind" ar eaten,
    Array.map
      (fun c ->
        let c' = nf_beta (subst1 x_arg_eater c) in
 	hnf_prod_applist sigma "red_mind" c' (List.map (lift 1) eaten)) lc),
   rest)
;;

let conv1_minductive sigma convfun (mind1,l1) (mind2,l2) =
    let (DOPN(MutInd(sp1,i1),cl1)) = mind1 and
        (DOPN(MutInd(sp2,i2),cl2)) = mind2  in
    let ar1 = mind_arity mind1 and
        ar2 = mind_arity mind2 and
        mis1 = mind_specif_of_mind mind1 and
        mis2 = mind_specif_of_mind mind2 in
    let nparams1 = mis_nparams mis1 and
        nparams2 = mis_nparams mis2 in
    let mib1 = mis1.mib and mib2 = mis2.mib in
    let n1 = mib1.mINDNTYPES and n2 = mib2.mINDNTYPES in
    let mpack1 = mib1.mINDPACKETS and mpack2 = mib2.mINDPACKETS in
    let conv_pack mip1 mip2 = 
        let (_,lc1) = decomp_all_DLAMV_name (mip1.mINDLC) and
            (_,lc2) = decomp_all_DLAMV_name (mip2.mINDLC) in
        let ((nparams1,_,ar1,lc1),l1) =
              red_mind sigma ((nparams1,sp1,ar1,lc1),l1) and
            ((nparams2,_,ar2,lc2),l2) =
              red_mind sigma ((nparams2,sp2,ar2,lc2),l2) in
         nparams1=nparams2
         & convfun ar1 ar2
         & for_all2eq_vect convfun lc1 lc2
         & for_all2eq convfun l1 l2
    in  n1=n2
      & i1=i2
      & for_all2eq_vect conv_pack mpack1 mpack2
;;

let conv1_mconstructor sigma convfun
  (DOPN(MutConstruct((x1,y1),i1),cl1),l1)
  (DOPN(MutConstruct((x2,y2),i2),cl2),l2) =
    i1 = i2
    & ( x1 = x2 & y1 = y2
        or conv1_minductive sigma convfun (DOPN(MutInd (x1,y1),cl1),l1)
                                          (DOPN(MutInd (x2,y2),cl2),l2))
;;


(* equality of terms without universe adjustment for synthesis *)
let rec conv1_x sigma term1 term2 =
  term1 = term2 or eqappr_x sigma (hnf sigma term1,hnf sigma term2)

and eqappr_x sigma = function
       ((Rel(n),l1),(Rel(m),l2)) ->
           (n=m) & (List.length(l1) = List.length(l2))
         & (for_all2 (conv1_x sigma) l1 l2)
     | ((DOP2(Cast,c,_),l),appr2) -> eqappr_x sigma ((c,l),appr2)
     | (appr1,(DOP2(Cast,c,_),l)) -> eqappr_x sigma (appr1,(c,l))
     | ((VAR id1,l1),(VAR id2,l2)) ->
         (id1=id2) & (List.length l1 = List.length l2)
         & (for_all2 (conv1_x sigma) l1 l2)
     | ((DOP0(Meta(n)),l1),(DOP0(Meta(m)),l2)) ->
         ((n=m) & (List.length(l1) = List.length(l2))
          & (for_all2 (conv1_x sigma) l1 l2))
     | ((DOP0(Sort s1),[]),(DOP0(Sort s2),[])) -> sort_cmp CONV_X s1 s2
     | ((DOP2(Lambda,c1,DLAM(_,c2)),[]),(DOP2(Lambda,c'1,DLAM(_,c'2)),[])) -> 
         (conv1_x sigma) c1 c'1 & (conv1_x sigma) c2 c'2
     | ((DOP2(Prod,c1,DLAM(_,c2)),[]),(DOP2(Prod,c'1,DLAM(_,c'2)),[])) -> 
         (conv1_x sigma) c1 c'1 & (conv1_x sigma) c2 c'2

     | ((DOPN(Const sp1,al1),l1),(DOPN(Const sp2,al2),l2)) ->
         ((sp1=sp2) & (al1=al2) & (List.length(l1) = List.length(l2))
          & (for_all2 (conv1_x sigma) l1 l2))
         or ((evaluable_const sigma (DOPN(Const sp2,al2)))
             & eqappr_x sigma ((DOPN(Const sp1,al1),l1),
                               apprec sigma
                                 (const_value sigma (DOPN(Const sp2,al2))) l2))
        or ((not (evaluable_const sigma (DOPN(Const sp2,al2))))
            & (evaluable_const sigma (DOPN(Const sp1,al1)))
            & eqappr_x sigma  (apprec sigma
                                 (const_value sigma (DOPN(Const sp1,al1))) l1,
                               (DOPN(Const sp2,al2),l2)))
     | ((DOPN(Const sp1,al1),l1),p2) ->  
         (evaluable_const sigma (DOPN(Const sp1,al1)))
         & eqappr_x sigma (apprec sigma
                             (const_value sigma (DOPN(Const sp1,al1))) l1, p2)
     | (p1,(DOPN(Const sp2,al2),l2))  -> 
         (evaluable_const sigma (DOPN(Const sp2,al2)))
         & eqappr_x sigma (p1, apprec sigma
                             (const_value sigma (DOPN(Const sp2,al2))) l2)

     | ((DOPN(Abst sp1,al1),l1),(DOPN(Abst sp2,al2),l2)) ->
         ((sp1=sp2) & (al1=al2) & (List.length(l1) = List.length(l2))
          & (for_all2 (conv1_x sigma) l1 l2))
        or ((evaluable_abst (DOPN(Abst sp2,al2)))
            & eqappr_x sigma ((DOPN(Abst sp1,al1),l1),
                              apprec sigma
                                (abst_value (DOPN(Abst sp2,al2))) l2))
        or ((not (evaluable_abst (DOPN(Abst sp2,al2))))
            & (evaluable_abst (DOPN(Abst sp1,al1)))
            & eqappr_x sigma (apprec sigma
                                (abst_value (DOPN(Abst sp1,al1))) l1,
                                (DOPN(Abst sp2,al2),l2)))
     | ((DOPN(Abst sp1,al1),l1),p2) ->  
         (evaluable_abst (DOPN(Abst sp1,al1)))
         & eqappr_x sigma (apprec sigma
                             (abst_value (DOPN(Abst sp1,al1))) l1, p2)
     | (p1,(DOPN(Abst sp2,al2),l2))  -> 
         (evaluable_abst (DOPN(Abst sp2,al2)))
         & eqappr_x sigma (p1, apprec sigma
                             (abst_value (DOPN(Abst sp2,al2))) l2)
     | ((DOPN(MutInd (x_0,x_1),cl1) as mind1,l'1),
        (DOPN(MutInd (x_2,x_3),cl2) as mind2,l'2)) ->
       (((x_0,x_1) = (x_2,x_3) & for_all2eq_vect (conv1_x sigma) cl1 cl2)
        & for_all2eq (conv1_x sigma) l'1 l'2)
        or conv1_minductive sigma (conv1_x sigma) (mind1,l'1) (mind2,l'2)
         
     | ((DOPN(MutConstruct (x_0,x_1),cl1) as mconstr1,l1),
        (DOPN(MutConstruct (x_2,x_3),cl2) as mconstr2,l2)) ->
        ((((x_0,x_1)=(x_2,x_3)) & for_all2eq_vect (conv1_x sigma) cl1 cl2)
         & for_all2eq (conv1_x sigma) l1 l2)
         or conv1_mconstructor sigma (conv1_x sigma) (mconstr1,l1)
          (mconstr2,l2)

     | ((DOPN(MutCase _,_) as constr1,l'1),
        (DOPN(MutCase _,_) as constr2,l'2)) -> 
       let (_,p1,c1,cl1) = destCase constr1 in
       let (_,p2,c2,cl2) = destCase constr2 in
       (conv1_x sigma) p1 p2 & (conv1_x sigma) c1 c2 &
       (for_all2eq_vect (conv1_x sigma) cl1 cl2) &
       (for_all2eq (conv1_x sigma) l'1 l'2)

     | ((DOPN(Fix (x_0,x_1),cl1),l'1),(DOPN(Fix (x_2,x_3),cl2),l'2)) -> 
       (x_0,x_1) = (x_2,x_3) &
       (for_all2eq_vect (conv1_x sigma) cl1 cl2) &
       (for_all2eq (conv1_x sigma) l'1 l'2)

     | ((DOPN(CoFix i1,cl1),l'1),(DOPN(CoFix i2,cl2),l'2)) -> 
       i1 = i2 &
       (for_all2eq_vect (conv1_x sigma) cl1 cl2) &
       (for_all2eq (conv1_x sigma) l'1 l'2)

     | (DOP0(Implicit),[]),(DOP0(Implicit),[]) -> true
     | (DLAM(_,c1),[]),(DLAM(_,c2),[]) -> conv1_x sigma c1 c2
     | (DLAMV(_,vc1),[]),(DLAMV(_,vc2),[]) ->
         for_all2eq_vect (conv1_x sigma) vc1 vc2
     | _ -> false;;


(* Atomic typing rules *)
let prog_relative sigma n env = 
  let j = Machops.relative sigma n env  in
  if strip_outer_cast j._TYPE = mkImplicit
  then error "Cannot use a logical assumption in a program";
  j
;;

(* Clone of Machops... *)

let apply_rel_list_prog sigma env nocheck argjl funj =
let rec apply_rec typ = function
    [] -> {_VAL=applist(funj._VAL,List.map j_val argjl);
           _TYPE=typ;
           _KIND = funj._KIND}
  | hj::restjl ->
  match whd_betadeltaiota sigma typ with
     DOP2(Prod,c1,DLAM(_,c2)) ->
     if nocheck or conv1_x sigma hj._TYPE c1 then
         apply_rec (subst1 hj._VAL c2) restjl
        else (match strip_outer_cast hj._VAL with
                DOP1(XTRA("COMMENT",[]),c) -> apply_rec typ restjl
              | _ -> error_cant_apply "Type Error" FW env funj argjl)
   | _ -> error_cant_apply "Non-functional construction" FW env funj argjl
in apply_rec funj._TYPE argjl;;

let cast_rel_prog sigma env cj tj =
    if conv1_x sigma cj._TYPE tj._VAL then
        {_VAL=j_val cj;
         _TYPE=tj._VAL;
         _KIND = whd_betadeltaiota sigma tj._TYPE}
    else error_actual_type FW env cj tj
;;


let assumption_of_judgement_prog sigma j =
let typ = whd_betadeltaiota sigma j._TYPE in 
match typ with DOP0(Sort s) -> {body=j._VAL; typ=s} 
             | _ -> {body=j._VAL;typ=implicit_sort}
;;


(* Calling Trad... *)
(* These functions are sometimes called with terms containing XTRA.
   Mach fails on these terms. *)
(* TODO: introduire un infexecute_rec dans Trad, ce qui permettrait de virer
 * les Meta et isevars de Mach, et Mach serait vraiment le noyau!
 * idee: en principe, Program (qui est une tactique) ne devrait jamais
 * appeler directement Mach. On regroupe ici tous les appels a Mach et Trad,
 * qui devront un jour devenir des appels a Trad.
 *)
let gENV sigma sign = (gLOB sign, gLOB(Mach.fsign_of_sign sigma sign));;

let infexemeta sigma metamap (ccienv,fwenv) c =
  Mach.infexemeta sigma metamap (ccienv,fwenv) c

let infexecute sigma sign c = infexemeta sigma [] (gENV sigma sign) c;;

let execute_meta_constr sigma metamap env c =
  Trad.ise_resolve true sigma metamap env c;;
let execute_constr_type sigma env c =
  Trad.ise_resolve_type true sigma [] env c;;
let execute_constr_rec sigma env c = execute_meta_constr sigma [] env c;;
let execute_constr sigma sign c = execute_meta_constr sigma [] (gLOB sign) c;;

let execute_nocheck sigma metamap sign c =
  Trad.ise_resolve_nocheck sigma metamap (gLOB sign) c
;;


let infmachine sigma (ccienv,fwenv) c =
  let j = execute_meta_constr sigma [] ccienv c in
  infexemeta sigma [] (ccienv,fwenv) j._VAL
;;

(* OLD version of propify
let whd_propify = function
  DOPN(Const sp,cl) -> DOPN(Const (ccisp_of sp),cl)
| DOPN(MutInd(sp,i),cl) -> DOPN(MutInd (ccisp_of sp,i),cl)
| DOPN(MutConstruct((sp,j),i),cl) -> DOPN(MutConstruct((ccisp_of sp,j),i),cl)
| x -> x
*)
(* A better propify ? requires the sign.
 * At least, no problem with constant environments (variables cl below).
 *)
let whd_propify env c =
  try
    match c with
      DOPN(Const sp,cl) ->
      	let ccsp = ccisp_of sp in
      	global_reference env ccsp (id_of_global (Const ccsp))
    | DOPN(MutInd(sp,i),cl) ->
      	let ccsp = ccisp_of sp in
      	global_reference env ccsp (id_of_global (MutInd(ccsp,i)))
    | DOPN(MutConstruct((sp,j),i),cl) ->
      	let ccsp = ccisp_of sp in
      	global_reference env ccsp (id_of_global (MutConstruct((ccsp,j),i)))
    | x -> x
  with Anomaly _ -> c
;;

let propify env = strong (whd_propify env)


let execute_meta_pure sigma metamap env pg = 
  Trad.ise_resolve true sigma metamap env (propify env pg) ;;
let execute_pure_rec sigma env pg = execute_meta_pure sigma [] env pg;;
let execute_pure sigma sign pg = execute_meta_pure sigma [] (gLOB sign) pg;;
let execute_pure_type sigma sign pg =
  let env = gLOB sign in
  execute_constr_type sigma env (propify env pg);;

(* This code should be merged with Trad's... *)

(* The signature of existential variables of progmach *)
let isevars = ref mt_evd;;

let set_ise evc = isevars := evc;;
let get_ise () = !isevars;;

(*-----*)
let evar_hnf c =
    match hnf_constr (get_ise()) c with
    DOPN(AppL,cl) -> (hd_vect cl,list_of_tl_vect cl)
  | c -> (c,[])
;;

let evar_apprec l c = evar_hnf (applistc c l);;

let try_whd_ise c =
  try whd_ise1 (get_ise()) c
  with (Failure _ | UserError _) -> c ;;

let try_nf_ise = strong try_whd_ise;;



(* Error messages *)
let error_annot_not_inf comenv t =
  errorlabstrm "Annotation non informative" 
    [< term0 comenv t; 'sPC;
       'sTR"is not allowed as an annotation because it is non informative" >]
;;

let error_comment_inf comenv t =
  errorlabstrm "Informative comment"
    [< term0 comenv t; 'sPC;
       'sTR"is an informative assumptions which is not allowed in comments">]
;;

let error_annot_wrong_type (comenv,env) c t =
  errorlabstrm "Bad annotation"
  [< 'sTR"The program"; 'bRK(1,1); fterm0 env c; 'sPC;
     'sTR"cannot be annotated by"; 'bRK(1,1); term0 comenv t >]
;;

let wrong_number_of_cases_message env (c,ct) expn = 
  let c = try_nf_ise c and ct = try_nf_ise ct in
    error_number_branches FW env c ct expn
;;



(* Conversion with existential variables, similar to Trad.
 * TODO: find out how to factorize with Trad's code.
 * the only difference seems to be conversion modulo isomorphism of
 * inductive types. Add another boolean in Trad's conversion ?
 *)
let conversion_problems = ref ([] : (conv_pb * constr * constr) list);;

let add_conv_pb pb = (conversion_problems := pb::!conversion_problems);;

let get_changed_pb lsp =
  let (pbs,pbs1) = List.fold_left
      (fun (pbs,pbs1) pb ->
    	if status_changed lsp pb then (pb::pbs,pbs1)
        else (pbs,pb::pbs1))
      ([],[])
      !conversion_problems in
  conversion_problems := pbs1;
  pbs
;;


(* Precondition: one of the terms of the pb is an uninstanciated evar,
 * possibly applied to arguments.
 *)
let rec solve_pb1 pb =
  match solve_simple_eqn evar_conv1_x isevars pb with
    Some lsp ->
     let pbs = get_changed_pb lsp in
      List.for_all (fun (_,t1,t2) -> evar_conv1_x t1 t2) pbs
  | None -> (add_conv_pb pb; true)

and evar_conv1_x term1 term2 =
  let term1 = whd_ise1 (get_ise()) term1
  and term2 = whd_ise1 (get_ise()) term2 in 
   if eq_constr term1 term2 then true
   else if (not(has_undefined_isevars isevars term1)) &
            not(has_undefined_isevars isevars term2)
   then conv1_x (get_ise()) term1 term2
   else if ise_undefined isevars term1 or ise_undefined isevars term2
   then solve_pb1(CONV_X,term1,term2)
   else 
     let (t1,l1) = evar_hnf term1
     and (t2,l2) = evar_hnf term2 in
     if (head_is_embedded_exist isevars t1 & not (is_eliminator t2)) or
        (head_is_embedded_exist isevars t2 & not (is_eliminator t1))
     then (add_conv_pb (CONV_X,applist(t1,l1),applist(t2,l2)); true)
     else evar_eqappr1_x ((t1,l1),(t2,l2))

and evar_eqappr1_x = function
    ((DOPN(Const sp1,al1),l1),(DOPN(Const sp2,al2),l2)) ->
      solve_pb1(CONV_X,DOPN(Const sp1,al1),DOPN(Const sp2,al2))
      & for_all2eq evar_conv1_x l1 l2

  | ((DOPN(Const sp1,al1),l1),(t2,l2)) ->  
      solve_pb1(CONV_X,DOPN(Const sp1,al1),t2)
      & for_all2eq evar_conv1_x l1 l2

  | ((t1,l1),(DOPN(Const sp2,al2),l2))  -> 
      solve_pb1(CONV_X,t1,DOPN(Const sp2,al2))
      & for_all2eq evar_conv1_x l1 l2

  | ((DOPN(Abst sp1,al1),l1),(DOPN(Abst sp2,al2),l2)) ->
      ((sp1=sp2) & (al1=al2) & (List.length(l1) = List.length(l2)) 
       & (for_all2 (evar_conv1_x) l1 l2))
      or ((evaluable_abst (DOPN(Abst sp2,al2)))
          & evar_eqappr1_x
            ((DOPN(Abst sp1,al1),l1),
             evar_apprec l2 (abst_value (DOPN(Abst sp2,al2)))))
      or ((not (evaluable_abst (DOPN(Abst sp2,al2))))
          & (evaluable_abst (DOPN(Abst sp1,al1)))
          & evar_eqappr1_x 
            (evar_apprec l1 (abst_value (DOPN(Abst sp1,al1))),
             (DOPN(Abst sp2,al2),l2)))

     | ((DOPN(Abst sp1,al1),l1),p2) ->  
         (evaluable_abst (DOPN(Abst sp1,al1)))
         & evar_eqappr1_x (evar_apprec l1 (abst_value (DOPN(Abst sp1,al1))),p2)

     | (p1,(DOPN(Abst sp2,al2),l2))  -> 
         (evaluable_abst (DOPN(Abst sp2,al2)))
         & evar_eqappr1_x (p1,evar_apprec l2 (abst_value (DOPN(Abst sp2,al2))))

     | ((Rel(n),l1),(Rel(m),l2)) ->
         (n=m) & (List.length(l1) = List.length(l2))
         & (for_all2 evar_conv1_x l1 l2)
     | ((DOP2(Cast,c,_),l),appr2) -> evar_eqappr1_x ((c,l),appr2)
     | (appr1,(DOP2(Cast,c,_),l)) -> evar_eqappr1_x (appr1,(c,l))
     | ((VAR id1,l1),(VAR id2,l2)) ->
         (id1=id2) & (List.length l1 = List.length l2)
         & (for_all2 evar_conv1_x l1 l2)
     | ((DOP0(Meta(n)),l1),(DOP0(Meta(m)),l2)) ->
           (n=m) & (List.length(l1) = List.length(l2))
         & (for_all2 evar_conv1_x l1 l2)
     | ((DOP0(Sort s1),[]),(DOP0(Sort s2),[])) -> sort_cmp CONV_X s1 s2
     | ((DOP2(Lambda,c1,DLAM(_,c2)),[]),(DOP2(Lambda,c'1,DLAM(_,c'2)),[])) -> 
         evar_conv1_x c1 c'1 & evar_conv1_x c2 c'2
     | ((DOP2(Prod,c1,DLAM(_,c2)),[]),(DOP2(Prod,c'1,DLAM(_,c'2)),[])) -> 
         evar_conv1_x c1 c'1 & evar_conv1_x c2 c'2
     | ((DOPN(MutInd _ as o1,cl1) as ind1,l'1),
        (DOPN(MutInd _ as o2,cl2) as ind2,l'2)) ->
         (ise_try isevars
            [(fun () -> (o1 = o2)
               & for_all2eq_vect evar_conv1_x cl1 cl2
               & for_all2eq evar_conv1_x l'1 l'2);
             (fun () -> conv1_minductive (get_ise())
                 evar_conv1_x (ind1,l'1) (ind2,l'2))])

     | ((DOPN(MutConstruct _ as o1,cl1) as constr1,l1),
        (DOPN(MutConstruct _ as o2,cl2) as constr2,l2)) ->
         (ise_try isevars
	    [(fun () -> (o1=o2)
                    & (for_all2eq_vect evar_conv1_x cl1 cl2)
                    & (for_all2eq evar_conv1_x l1 l2));
             (fun () -> conv1_mconstructor (get_ise()) evar_conv1_x
                 (constr1,l1) (constr2,l2))])

     | ((DOPN(MutCase _,_) as constr1,l'1),
        (DOPN(MutCase _,_) as constr2,l'2)) -> 
       let (_,p1,c1,cl1) = destCase constr1 in
       let (_,p2,c2,cl2) = destCase constr2 in
       evar_conv1_x p1 p2 & evar_conv1_x c1 c2
       & (for_all2eq_vect evar_conv1_x cl1 cl2)
       & (for_all2eq evar_conv1_x l'1 l'2)

     | ((DOPN(Fix _ as op1,cl1),l1),(DOPN(Fix _ as op2,cl2),l2))   ->
       op1 = op2 & 
       (for_all2eq_vect evar_conv1_x cl1 cl2) &
       (for_all2eq evar_conv1_x l1 l2)

     | ((DOPN(CoFix(i1),cl1),l1),(DOPN(CoFix(i2),cl2),l2))   ->
       i1=i2 & 
       (for_all2eq_vect evar_conv1_x cl1 cl2) &
       (for_all2eq evar_conv1_x l1 l2)

     | (DOP0(Implicit),[]),(DOP0(Implicit),[]) -> true
(* added to compare easily the specification of fixed points *)
     | (DLAM(_,c1),[]),(DLAM(_,c2),[]) -> evar_conv1_x c1 c2
     | (DLAMV(_,vc1),[]),(DLAMV(_,vc2),[]) ->
         for_all2eq_vect evar_conv1_x vc1 vc2
     | _ -> false;;

let the_conv1_x t1 t2 = conv1_x (get_ise()) t1 t2 or evar_conv1_x t1 t2;;

let evar_type_fixpoint fx vdeft =
  let (j,vargs) =
    match fx with
        DOPN(Fix(_,j),vargs) -> (j,vargs)
      | DOPN(CoFix j,vargs) -> (j,vargs) in
  let lt = Array.length vdeft in 
  let var = Array.sub vargs 0 lt in 
    if for_all2eq_vect (fun ari ti ->  the_conv1_x (lift lt ari) ti) var vdeft
    then var.(j) 
    else error "Implicit Syntax: Ill-typed case recursive call (branches are malformed)";;

let type_case_elim_pg env sigma ct pt lft p c =
  let (_,bty,rslty) = type_case_branches env sigma ct pt p c in
  let n = Array.length lft and expn = Array.length bty in
    if n<>expn then error_number_branches FW env c ct expn 
    else
let rec check_conv i = 
    if i = n then () else
    if not (conv1_x sigma lft.(i) (bty.(i)))
    then error_ill_formed_branch FW env c i lft.(i) bty.(i) 
    else check_conv (i+1) 
in check_conv 0; rslty;;

let the_type_case_elim_pg env sigma ct pt lft p c =
  let (_,bty,rslty) = type_case_branches env sigma ct pt p c in
  let n = Array.length lft and expn = Array.length bty in
    if n<>expn then error_number_branches FW env c ct expn 
    else
let rec check_conv i = 
    if i = n then () else
    if not (the_conv1_x lft.(i) (bty.(i)))
    then error_ill_formed_branch FW env c i lft.(i) bty.(i) 
    else check_conv (i+1) 
in check_conv 0; rslty;;


let attempt_coercion env typ =
 let rec apprec argj =
    match whd_betadeltaiota (get_ise()) argj._TYPE with
    DOP2(Prod,argc1,argB) ->
    let k = new_isevar isevars env argc1 FW in
    let argj = {_VAL = applist(argj._VAL,[k]);
                _TYPE = sAPP argB k;
                _KIND = argj._KIND}
    in if ise_try isevars [fun () -> the_conv1_x (sAPP argB k) typ] then
        argj
       else apprec argj

  | _ -> failwith "attempt_coercion"
 in apprec
    
;;

let the_apply_rel_list_prog env argjl funj vtcon =
  let rec apply_rec acc typ = function
      [] ->
 	(match vtcon with
        | (_,(_,Some typ')) -> let _ = the_conv1_x typ typ' in ()
    	| _ -> ());
        {_VAL=applist(j_val funj,List.map j_val (List.rev acc));
          _TYPE=typ;
          _KIND = funj._KIND}
    | hj::restjl ->
  match whd_betadeltaiota (get_ise()) typ with
     DOP2(Prod,c1,DLAM(_,c2)) ->
     if the_conv1_x hj._TYPE c1 then
         apply_rec (hj::acc) (subst1 hj._VAL c2) restjl
        else (match strip_outer_cast hj._VAL with
                DOP1(XTRA("COMMENT",[]),c) -> apply_rec (hj::acc) typ restjl
              | _ -> 
                (try let hj = attempt_coercion env c1 hj
                     in apply_rec (hj::acc) (subst1 hj._VAL c2) restjl
                 with UserError _ | Failure _ ->
                   error_cant_apply "Type error" FW env funj argjl))
   | _ -> error_cant_apply "Non-functional construction" FW env funj argjl
in apply_rec [] funj._TYPE argjl;;

let the_cast_rel_prog env cj tj =
    if the_conv1_x cj._TYPE tj._VAL then
        {_VAL=j_val cj;
         _TYPE=tj._VAL;
         _KIND = whd_betadeltaiota (get_ise()) tj._TYPE}
    else (try attempt_coercion env tj._VAL cj
         with UserError _ | Failure _ -> error_actual_type FW env cj tj)
;;

let find_sorted_assoc p = 
 let rec findrec = function 
   ((a,ta)::l) -> if a < p then findrec l 
             else if a = p then ta else raise Not_found
  | _     -> raise Not_found
 in findrec ;;

let case_branches v = Array.sub v 2 (Array.length v-2);;

let map_lift_fst_n m = List.map (function (n,t)->(n+m,t));;
let map_lift_fst = map_lift_fst_n 1;;

let rec instantiate_recarg sp lrc = 
  function  
      Mrec(j)        -> Imbr(sp,j,lrc)
    | Imbr(sp1,k,l)  -> Imbr(sp1,k, List.map (instantiate_recarg sp lrc) l)
    | Norec          -> Norec
    | Param(k)       -> List.nth lrc k
;;

(* mind_recarg is a vector giving the list of recargs for each type in
the mutually inductive definition *)

(* propagate checking for F,incorporating recursive arguments *)
let check_term mind_recvec f = 
 let rec crec n l (lrec,c) = 
   match (lrec,strip_outer_cast c) with
       (Param(_)::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l 
         in  crec (n+1) l' (lr,b)
     | (Norec::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l 
         in  crec (n+1) l' (lr,b)
     | (Mrec(i)::lr,DOP2(Lambda,_,DLAM(_,b)))  -> 
         let l' = map_lift_fst l 
         in  crec (n+1) ((1,mind_recvec.(i))::l') (lr,b)
     | (Imbr(sp,i,lrc)::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l in
         let sprecargs = mind_recargs (mkMutInd sp i [||]) in
         let lc = (Array.map 
                       (List.map (instantiate_recarg sp lrc))
                    sprecargs.(i))
         in  crec (n+1) ((1,lc)::l') (lr,b)
     | _,f_0 -> f n l f_0
 in crec ;;

let is_inst_var sigma k c = 
  match whd_betadeltaiota_stack sigma c [] with 
      (Rel n,_) -> n=k
    | _         -> false;;


let is_subterm_specif sigma lcx mind_recvec = 
  let rec crec n lst c = 
    match whd_betadeltaiota_stack sigma c [] with 
        ((Rel k),_)         -> find_sorted_assoc k lst
      |  (DOPN(MutCase _,_) as x,_) ->
           let ( _,_,c,br) = destCase x in
             if Array.length br = 0 
             then [||] 
             else
               let lcv = (try if is_inst_var sigma n c then lcx 
                              else (crec n lst c) 
                          with Not_found -> (Array.create (Array.length br) []))
               in if Array.length br <> Array.length lcv 
	         then assert false
                 else let stl = map2_vect 
                                  (fun lc a -> 
                                     check_term mind_recvec 
                                       crec  n lst (lc,a)) lcv br 
                 in stl.(0)


	 | (DOPN(Fix(_),la) as mc,l) ->
             let (recindxs,i,typarray,funnames,bodies) = destUntypedFix mc in
             let nbfix   = List.length funnames in
             let decrArg = recindxs.(i) in 
             let theBody = bodies.(i)   in
             let (gamma,strippedBody) = decompose_lam_n (decrArg+1) theBody in
             let absTypes = List.map snd gamma in 
             let nbOfAbst = nbfix+decrArg+1 in
             let newlst = 
               if (List.length l < (decrArg+1)) 
               then ((nbOfAbst,lcx) ::
                     (map_lift_fst_n nbOfAbst lst))
               else 
                 let theDecrArg  = List.nth l decrArg in
                 let recArgsDecrArg = 
                   try (crec n lst theDecrArg)
	           with Not_found -> Array.create 0 [] 
                 in if (Array.length recArgsDecrArg)=0
                   then ((nbOfAbst,lcx) ::
                         (map_lift_fst_n nbOfAbst lst))
                   else ((1,recArgsDecrArg)::
                         (nbOfAbst,lcx) ::
                         (map_lift_fst_n nbOfAbst lst))                     
             in  (crec (n+nbOfAbst) newlst strippedBody)

      |  (DOP2(Lambda,_,DLAM(_,b)),[]) -> 
           let lst' = map_lift_fst lst 
           in crec  (n+1) lst' b

(***** Experimental change *************************)
|  (DOP0(Meta _),_)             -> [||]
(***************************************************)
|  _                        -> raise Not_found
  in crec;;

let is_subterm sigma lcx mind_recvec n lst c = 
try is_subterm_specif sigma lcx mind_recvec n lst c; true 
with Not_found -> false;;

  let noccur_with_meta n m term = let rec occur_rec n = function
    Rel(p)        -> if n<=p & p<n+m then raise Occur
  | VAR _         -> ()
  | DOPN(op,cl)   -> (match op with
                         AppL ->( match strip_outer_cast (cl.(0)) with
                                    DOP0 (Meta _) -> ()
                                   | _             -> Array.iter (occur_rec n) cl)
                        | _    -> Array.iter (occur_rec n) cl)
  | DOPL(_,cl)    -> List.iter (occur_rec n) cl
  | DOP0(_)       -> ()
  | DOP1(_,c)     -> occur_rec n c
  | DOP2(_,c1,c2) -> occur_rec n c1; occur_rec n c2
  | DLAM(_,c)     -> occur_rec (n+1) c
  | DLAMV(_,v)    -> Array.iter (occur_rec (n+1)) v
(* this case is unused:  | _             -> () *)
  in try (occur_rec n term; true) with Occur -> false;;

let check_subterm_rec_meta_prog sigma vectn k def = 
if k < 0 then true else
let nfi = Array.length vectn in 
 (* check fi does not appear in the k+1 first abstractions, 
      gives the type of the k+1-eme abstraction  *)
let rec check_occur n def = match strip_outer_cast def with
      DOP2(Lambda,a,DLAM(_,b)) -> if noccur_with_meta n nfi a then
                      if n = k+1 then (a,b) else check_occur (n+1) b
                      else error "Bad occurrence of recursive call"
    | DOP2(XTRA("ANNOT",[]),c,_) -> check_occur n c
    | _ -> error "Not enough abstractions in the definition"
in let (c,d) = check_occur 1 def 
in let (DOPN(MutInd(sp,tyi),_) as mI,largs) = 
  try find_minductype sigma c 
  with Induc -> error "Recursive definition on a non inductive type" in

let mind_recvec = mind_recargs mI in 
let lcx = mind_recvec.(tyi) in

   let rec check_rec_call n lst t = 
            (* n gives the index of the recursive variable *)
     (noccur_with_meta (n+k+1) nfi t) or 
(* no recursive call in the term *)
      (match whd_betadeltaiota_stack sigma t [] with 
	   (Rel p,l) -> 
	     if n+k+1 <= p & p < n+k+nfi+1 (* recursive call *)
	     then let glob = nfi+n+k-p in  (* the index of the recursive call *) 
		  let np = vectn.(glob) in (* the decreasing arg of the rec call *)
		    if List.length l > np then 
		      (match chop_list np l with
			   (la,(z::lrest)) -> 
	                     if (is_subterm sigma lcx mind_recvec n lst z) 
                  	     then List.for_all (check_rec_call n lst) (la@lrest)
                  	     else error "Recursive call applied to an illegal term"
		      	 | _ -> assert false)
		    else error  "Not enough arguments for the recursive call"
	     else List.for_all (check_rec_call n lst) l        
	 | (DOPN(MutCase _,_) as mc,l) ->
             let (ci,p,c_0,lrest) = destCase mc in
             let lc = (try if is_inst_var sigma n c_0 
                           then lcx 
		           else is_subterm_specif sigma lcx mind_recvec n lst c_0
		       with Not_found -> Array.create (Array.length lrest) []) in
               (for_all2eq_vect
		  (fun c_0 a -> check_term mind_recvec (check_rec_call) n lst (c_0,a))
		  lc lrest) & (List.for_all (check_rec_call n lst) (c_0::p::l)) 


	 | (DOPN(Fix(_),la) as mc,l) ->
             (List.for_all (check_rec_call n lst) l) &
             let (recindxs,i,typarray,funnames,bodies) = destUntypedFix mc in
             let nbfix       = List.length funnames in
             let decrArg     = recindxs.(i) 
             in if (List.length l < (decrArg+1)) 
                then (for_all_vect (check_rec_call n lst) la)
                else 
                  let theDecrArg  = List.nth l decrArg in
                  let recArgsDecrArg = 
                    try (is_subterm_specif sigma lcx mind_recvec n lst theDecrArg)
	            with Not_found -> Array.create 0 [] 
                  in if (Array.length recArgsDecrArg)=0
                    then (for_all_vect (check_rec_call n lst) la)
                    else 
                 let theBody = bodies.(i)   in
                 let (gamma,strippedBody) = decompose_lam_n (decrArg+1) theBody in
                 let absTypes = List.map snd gamma in 
                 let nbOfAbst = nbfix+decrArg+1 in
                 let newlst = ((1,recArgsDecrArg)::(map_lift_fst_n nbOfAbst lst))
                 in  ((for_all_vect 
			 (fun t -> check_rec_call n lst t)
			 typarray) &
                      (for_all_i (fun n -> check_rec_call n lst) n absTypes) &
                      (check_rec_call (n+nbOfAbst) newlst strippedBody))


	 | (DOP2(_,a,b),l) -> (check_rec_call n lst a)
                               & (check_rec_call n lst b)
                               & (List.for_all (check_rec_call n lst) l)
	 | (DOPN(_,la),l) -> (for_all_vect (check_rec_call n lst) la)
                              & (List.for_all (check_rec_call n lst) l)
	 | (DOP0 (Meta _),l) -> true
	 | (DLAM(_,t),l)  -> (check_rec_call (n+1) (map_lift_fst lst) t)
                              & (List.for_all (check_rec_call n lst) l)
	 | (DLAMV(_,vt),l)  -> 
	     (for_all_vect (check_rec_call (n+1) (map_lift_fst lst)) vt)
             & (List.for_all (check_rec_call n lst) l)
	 | (_,l)    ->   List.for_all (check_rec_call n lst) l
      ) 

in check_rec_call 1 [] d;;    
 

let check_fix_prog sigma (DOPN(Fix(nvect,j),vargs)) = 
 let nbfix = let nv = Array.length vargs in 
            if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let varit = Array.sub vargs 0 nbfix in
  let ldef = last_vect vargs in
  let ln = Array.length nvect and la = Array.length varit in
  if ln <> la then error "Ill-formed fix term"
  else let (lna,vdefs) = decomp_DLAMV_name ln ldef in 
       let vlna = Array.of_list lna in
       let check_type i = 
         try check_subterm_rec_meta_prog sigma nvect nvect.(i) vdefs.(i) 
         with UserError (s,str) -> error_ill_formed_rec_body str FW lna i vdefs
       in for i = 0 to ln-1 do check_type i done;;

(* Remove implicit syntax without modification of comments *)

let unsafe_fmachine_prog_noannot metamap sign constr =
    
 let rec exemeta_rec vtcon env cstr = match cstr with
  DOP0(XTRA("ISEVAR",[])) ->
    (match vtcon with
      (is_ass,(Some valc, tyc)) ->
      	exemeta_rec (is_ass,(None,tyc)) env valc
    | (_,(None,Some ty)) ->
      	let evarty = Machops.j_val_cast (exemeta_rec def_vty_con env ty) in
      	let k = new_isevar isevars env evarty FW in
      	exemeta_rec vtcon env k
    | (true,(None,None)) ->
      	let k = new_isevar isevars env (mkCast dummy_sort dummy_sort) FW in
      	exemeta_rec vtcon env k
    | (false,(None,None)) ->
 	error "There is an unknown subterm I cannot solve")

| DOP0(Meta n) ->
    let metaty =
      try List.assoc n metamap
      with Not_found -> error "Metavariable wasn't in the metamap" 
    in
      (match metaty with
           DOP2(Cast,typ,kind) -> {_VAL=cstr; _TYPE=typ; _KIND=kind}
         | typ ->
             {_VAL=cstr;
              _TYPE=typ;
              _KIND=whd_betadeltaiota (get_ise())
                     (exemeta_rec mt_tycon env typ)._TYPE})

|  Rel(n) -> prog_relative (get_ise()) n env

| DOP2(Lambda,DOP1(XTRA("COMMENT",[]),c),DLAM(na,b)) ->
    (* Introduce a dummy variable for this logical variable *)
    let dummy = {body=mkImplicit;typ=prop} in 
    let env' = (add_rel (na,dummy) env) in
    let bj = exemeta_rec mt_tycon env' b
    in {_VAL = (DOP2(Lambda,
                    DOP1(XTRA("COMMENT",[]),c),
                    DLAM(na,j_val bj)));
        _TYPE = pop bj._TYPE;
        _KIND = bj._KIND}

| DOP2(XTRA("ANNOT",[]),c,t) ->
  let cj = exemeta_rec vtcon env c in
           {_VAL=DOP2(XTRA("ANNOT",[]),(Machops.cast_of_judgement cj),t);
            _TYPE = cj._TYPE;
            _KIND = cj._KIND}

| DOP1(XTRA("COMMENT",[]),c) ->  (* Should not occur ?? *)
       {_VAL=cstr; _TYPE = mkProp; _KIND = DOP0(Sort types)}

| DOPN(XTRA("REC",[]),cl) ->  
  let p = cl.(0)
  and c = cl.(1)
  and lf = Array.sub cl 2 ((Array.length cl) - 2) in
  let cj = exemeta_rec mt_tycon env  c
  and pj = exemeta_rec mt_tycon env  p in
  let p = pj._VAL and
      c = cj._VAL in
  let ct = nf_ise1 !isevars cj._TYPE
  and pt = nf_ise1 !isevars pj._TYPE in
  let (_,bty,rsty) =
    Indrec.type_rec_branches true (get_ise()) env ct pt p c
  in if Array.length bty <> Array.length lf then 
    wrong_number_of_cases_message env (c,ct) (Array.length bty)
   else 
  let lfj = map2_vect (fun tyc f -> exemeta_rec (mk_tycon tyc) env f) bty lf in
  let lft = (Array.map (fun j -> j._TYPE) lfj) in
  let rEC = Array.append [|p; c|] (Array.map j_val lfj) in
      {_VAL=Indrec.transform_rec env (get_ise()) rEC (ct,pt);
       _TYPE = rsty;
       _KIND = sort_of_arity (get_ise()) pt}

| DOPN(XTRA("MLCASE",[isrectok]),cl)  ->
    let isrec = is_case_rec isrectok in
    let c = cl.(0) and  lf = tl_vect cl in
      (try match vtcon with
           (_,(_,Some pred)) -> 
             let predj =  exemeta_rec mt_tycon env pred in 
               exemeta_rec vtcon env  
                 (Indrec.make_case_ml isrec predj._VAL c None lf)  
         | _ -> error "notype"
       with UserError _ -> 
         let cj = exemeta_rec mt_tycon env  c in
         let rec findtype i =
           if i > Array.length lf then 
             errorlabstrm "Progmach.execute"
               [< 'sTR"Not enough information in branches to infer type"; 'sPC;
                  'sTR"in ML case on term:"; 'wS 1; pFTERMINENV(env,cj._VAL) >]
           else
             try
               let expti = Indrec.branch_scheme (get_ise()) isrec i cj._TYPE in
               let fj = exemeta_rec (mk_tycon expti) env  lf.(i-1) in 
               let efjt = try_nf_ise fj._TYPE in 
               let pred = 
                 Indrec.pred_case_ml
                   env (get_ise()) isrec (cj._VAL,cj._TYPE) lf (i,efjt)
               in if has_ise pred then error"isevar" else pred
             with UserError _ -> findtype (i+1) in
         let pred =
           (try findtype 1
            with Induc -> error "ML Case on a non inductive type")
         in exemeta_rec vtcon env 
              (Indrec.make_case_ml isrec pred cj._VAL None lf))

| DOPN(XTRA("MULTCASE",l),cl) ->
    Multcase.compile_multcase
      (exemeta_rec,
       inh_ass_of_j,
       get_ise)
      vtcon env cstr

(* TODO: ratrapper les Not_found *)
| VAR id ->
    let {body=typ;typ=kind} = snd(lookup_glob id env)
    in {_VAL=cstr; _TYPE=typ; _KIND=mkSort kind}

| DOPN(Abst _,_) ->
    if evaluable_abst cstr
    then exemeta_rec mt_tycon env (abst_value cstr)
    else error "Cannot typecheck an unevaluable abstraction"

| DOPN(Const sp,_) ->
    let typ = type_of_const2 (get_ise()) env cstr in
      (match typ with
           DOP2(Cast,typ,kind) -> {_VAL=cstr;_TYPE=typ;_KIND = kind}
         | _ -> {_VAL=cstr;_TYPE=typ;
                 _KIND= whd_betadeltaiota (get_ise())
                         (exemeta_rec def_vty_con env typ)._TYPE})

| DOPN(MutInd _,_) ->
  let {body=typ;typ=kind} = type_of_mind (get_ise()) env cstr
  in {_VAL=cstr; _TYPE=typ; _KIND=mkSort kind}
 
| DOPN(MutConstruct _,_) -> 
  let (typ,kind) = destCast (type_of_mconstr (get_ise()) env cstr)
  in {_VAL=cstr; _TYPE=typ; _KIND=kind}

| DOP0(Sort(Prop(c))) -> fcn_proposition c
| DOP0(Sort(Type(u))) -> fcn_type_with_universe (u)
| DOPN(AppL,tl) -> 
  let j = exemeta_rec mt_tycon env (hd_vect tl) in
  let apply_one_arg (tycon,jl) c = 
    let cj = exemeta_rec (app_dom_tycon isevars tycon) env c in
    let rtc = app_rng_tycon isevars cj._VAL tycon in
    (rtc,cj::jl) in
  let jl = List.rev(snd(List.fold_left apply_one_arg
			  (mk_tycon j._TYPE,[]) (list_of_tl_vect tl))) in
  the_apply_rel_list_prog env jl j vtcon

| DOP2(Lambda,c1,DLAM(name,c2))      -> 
  let jfw = exemeta_rec (abs_dom_valcon isevars vtcon) env c1 in
  let assumfw = assumption_of_judgement (get_ise()) env jfw in
  let varfw = (name,assumfw) in
  let j' = exemeta_rec (abs_rng_tycon isevars vtcon)
      (add_rel varfw env) c2 in 
  abs_rel (get_ise()) name assumfw j'

| DOP2(Prod,c1,DLAM(name,c2))        ->
  let jfw = exemeta_rec def_vty_con env  c1 in
  let assumfw = assumption_of_judgement (get_ise()) env jfw in
  let varfw = (name,assumfw) in
  let j' = exemeta_rec def_vty_con (add_rel varfw env) c2 in
  gen_rel (get_ise()) FW env name assumfw j'

| DOPN(MutCase _,_) ->
  let (_,p,c,lf) = destCase cstr in
  (let {_TYPE=ct} as cj = exemeta_rec mt_tycon env  c in
   let {_TYPE=pt;_VAL=p} as pj = exemeta_rec mt_tycon env  p in
   let (mind,bty,rsty) = type_case_branches env (get_ise()) ct pt p c in
    if Array.length bty <> Array.length lf then 
    wrong_number_of_cases_message env (cj._VAL,ct) (Array.length bty)
    else
   let lfj =
     map2_vect (fun tyc f -> exemeta_rec (mk_tycon tyc) env  f) bty lf in
   let lft = (Array.map (fun j -> j._TYPE) lfj) in
     {_VAL=mkMutCaseA (ci_of_mind mind) (j_val pj) (j_val cj)
            (Array.map j_val lfj);
      _TYPE=the_type_case_elim_pg env (get_ise()) ct pj._TYPE lft p c;
      _KIND = sort_of_arity (get_ise()) pj._TYPE})

| DOPN(Fix(vn,i),cl) ->
  if  exists_vect (fun n -> n < 0) vn then 
      error "General Fixpoints not allowed";
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let larj = Array.map (exemeta_rec def_vty_con env) lar in
  let lara = Array.map (assumption_of_judgement (get_ise()) env) larj in 
  let newenv =
    it_vect2 (fun env name arfw -> add_rel (name,arfw) env)
      env (Array.of_list (List.rev lfi)) (vect_lift_type lara) in
  let vdefj = map_i_vect 
     (fun i def ->
       exemeta_rec (mk_tycon (lift nbfix larj.(i)._VAL)) newenv def) 0 vdef in
  let vdefv = put_DLAMSV lfi (Array.map (fun j -> j._VAL) vdefj) in
  let larv = Array.map Machops.j_val_cast larj in
  let fix = DOPN(Fix(vn,i),Array.append larv [|vdefv|])
  in check_fix_prog (get_ise()) fix;
  {_VAL= fix;
   _TYPE=evar_type_fixpoint fix (Array.map (fun j -> j._TYPE) vdefj);
   _KIND=larj.(i)._TYPE}

| DOPN(CoFix i,cl) ->
    let nbfix =
      let nv = Array.length cl in 
      if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
    let lar = Array.sub cl 0 nbfix in
    let ldef = last_vect cl in
    let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
    let larj = Array.map (exemeta_rec def_vty_con env) lar in
    let lara = Array.map (assumption_of_judgement (get_ise()) env) larj in 
    let newenv =
      it_vect2 (fun env name ar -> add_rel (name,ar) env)
        env (Array.of_list (List.rev lfi)) (vect_lift_type lara) in
    let vdefj =
      map_i_vect
 	(fun i def ->
	  exemeta_rec (mk_tycon (lift nbfix (larj.(i)._VAL))) newenv def)
 	0 vdef in
    let vdefv = put_DLAMSV lfi (Array.map (fun j -> j._VAL) vdefj) in
    let larv  = Array.map Machops.j_val_cast larj in
    let cofix = DOPN(CoFix i,Array.append larv [|vdefv|]) in
      check_cofix (get_ise()) cofix;
      {_VAL= cofix;
      	_TYPE=evar_type_fixpoint cofix (Array.map (fun j -> j._TYPE) vdefj);
      	_KIND=larj.(i)._TYPE}

| DOP2(Cast,c,t) ->
   let tj = exemeta_rec def_vty_con env  t in
   let cj =
     exemeta_rec
       (mk_tycon2 vtcon 
	  (body_of_type (assumption_of_judgement (get_ise()) env tj))) env  c
   in the_cast_rel_prog env cj tj

| _ -> error_cant_execute FW env cstr
 
and 

(* Coercions are not implemented for programs *)

inh_ass_of_j env j =
   let typ = whd_betadeltaiota (get_ise()) j._TYPE in
     match typ with
         DOP0(Sort s) -> {body=j._VAL;typ=s}
       | _ -> error_assumption FW env j._VAL

in exemeta_rec mt_tycon sign constr
;;

(* Use the usual execution functions on CCI terms to apply the comments *)

let force_cast env c = match c with DOP2(Cast,_,_) -> c 
   | _ -> let j = unsafe_fmachine_prog_noannot [] env c in DOP2(Cast,c,j._TYPE);;
       
let force_cast_type env c =
  match c with
      DOP2(Cast,c,DOP0(Sort s)) -> {body=c;typ=s}
    | _ ->
	match (unsafe_fmachine_prog_noannot [] env c)._TYPE with
	    DOP0(Sort s) -> {body=c;typ=s}
	  | _ -> anomaly "force_cast_type";;
       
let execute_annot =

 let rec exemeta_rec  env comenv =  function

 (DOP2(Lambda,DOP1(XTRA("COMMENT",[]),c),DLAM(na,b))) ->
  (* [x:{C}]B --> Btype
     when C is well-typed in comenv and is purely logical

     [x:{C}]B --> Btype
     when C is wel-typed in comenv and is informative, and irrelevant
     because of levels

     [x:{C}]B --> (x:extC)Btype
     when C is well-typed in comenv and is informative and relevant
     the extraction of C is extC

     B is well-typed in the environments:
     -- add the extraction of C to env (Implicit if C is logical)
     -- add C to comenv
   *)
  let (cj,cinf) = infmachine (get_ise()) (comenv,env) c in
  let relevant =
    match cinf with
        Logic -> false
      | Inf cinfj -> error_comment_inf comenv cj._VAL in
  let env' = add_rel (na,assumption_of_judgement (get_ise()) comenv cj) env in
  let comenv' =
    add_rel (na,assumption_of_judgement (get_ise()) comenv cj) comenv in
  let bj = exemeta_rec env' comenv' b in
    (DOP2(Lambda, DOP1(XTRA("COMMENT",[]),j_val cj), DLAM(na,bj)))

| (DOP2(XTRA("ANNOT",[]),c,t)) -> (* c is supposed to be a Cast *)
  (* c::{T} --> ctype
     when T is well-typed in comenv and extraction_T is convertible
     with ctype
   *)
  let (tj,tinf) = infmachine (get_ise()) (comenv,env) t in
      (match tinf with
           Logic -> error_annot_not_inf comenv t
         | Inf tinfj ->
             traceme "Tinfj" tinfj;
             if the_conv1_x tinfj._VAL (cast_type c) then
               DOP2(XTRA("ANNOT",[]), c,tj._VAL)
             else error_annot_wrong_type (comenv,env) c t)

| (DOP1(XTRA("COMMENT",[]),c)) ->
    let (cj,cinf) = infmachine (get_ise()) (comenv,env) c in 
    let relevant =
      match cinf with
          Logic -> false
        | Inf cinfj -> error_comment_inf comenv cj._VAL
    in (DOP1(XTRA("COMMENT",[]),cj._VAL))

| DOPN(AppL,tl)      -> DOPN(AppL,Array.map (exemeta_rec env comenv) tl)
| (DOPN(MutCase ci,cl)) -> DOPN(MutCase ci,Array.map (exemeta_rec env comenv) cl)

| (DOP2(lamorprod,c1,DLAM(name,c2)) as x)  -> (* c1 is supposed to be a Cast *)
    (match lamorprod with
         Prod | Lambda ->
           let newc1 = exemeta_rec env comenv c1 in 
           (* Forces a Cast  *)
           let newc1 = force_cast_type env newc1 in
           let varfw = (name,newc1) in
           let assumcci =
             (try assumption_of_judgement_prog (get_ise())
                (execute_pure_rec (get_ise()) comenv c1) 
              with UserError _ -> newc1) in
           let varcci = (name,assumcci) in
           let newc2 =
             exemeta_rec (add_rel varfw env) (add_rel varcci comenv) c2
           in DOP2(lamorprod, incast_type newc1, DLAM(name,newc2))
       | _ -> x)

| (DOPN(Fix(vn,i),cl) as val_0) -> (* arities are supposed to be casted *)
  if  exists_vect (fun n -> n < 0) vn then 
      error "General Fixpoints not allowed";
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let newlar = Array.map 
              (fun a -> force_cast env (exemeta_rec env comenv a)) lar in
  let newenv,newcomenv = it_vect2 
     (fun (env,comenv) name newar -> 
        let assumcci =  (try assumption_of_judgement_prog (get_ise()) 
          (execute_pure_rec (get_ise()) comenv newar)
                         with UserError _ -> outcast_type newar)
        in (add_rel (name,outcast_type newar) env),(add_rel (name,assumcci) comenv))
     (env,comenv) (Array.of_list (List.rev lfi)) (vect_lift newlar) in
  let vdefj = Array.map (fun def -> exemeta_rec newenv newcomenv def) vdef in
  let vdefv = put_DLAMSV lfi vdefj in
  let larv = newlar in
  DOPN(Fix(vn,i),Array.append larv [|vdefv|])

| (DOPN(CoFix i,cl) as val_0) ->
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let newlar = Array.map (exemeta_rec env comenv) lar in
  let newenv,newcomenv = it_vect2 
     (fun (env,comenv) name newar -> 
        let assumcci =  (try assumption_of_judgement_prog (get_ise()) 
          (execute_pure_rec (get_ise()) comenv newar) 
                         with UserError _ -> outcast_type newar)
        in (add_rel (name,outcast_type newar) env),(add_rel (name,assumcci) comenv))
     (env,comenv) (Array.of_list (List.rev lfi)) (vect_lift newlar) in
  let vdefj = Array.map (fun def -> exemeta_rec newenv newcomenv def) vdef in
  let vdefv = put_DLAMSV lfi vdefj in
  let larv = newlar in
  DOPN(CoFix i,Array.append larv [|vdefv|])

| DOP2(Cast,c,t) -> (* Explicit Casts are removed in the noannot phase so 
                       only casts generated by the system are there and
                       they do not contain annotations. Also they do not 
                       satisfy the property of having casted lambdas *)
   let newc = exemeta_rec env comenv c 
   in DOP2(Cast,newc,t) 

| x -> x
in exemeta_rec;; 

let rec has_annot = function 
    DOP1(XTRA("COMMENT",[]),c) -> true 
  | DOP2(XTRA("ANNOT",[]),c,t) -> true
  | DOP1(_,c) -> has_annot c
  | DOP2(_,c1,c2) -> has_annot c1 or has_annot c2
  | DOPN(_,cl) -> exists_vect has_annot cl
  | DOPL(_,cl) -> List.exists has_annot cl
  | DLAM(_,c) -> has_annot c
  | DLAMV(_,cl) -> exists_vect has_annot cl
  | _ -> false
;;

let unsafe_fmachine_prog metamap sign comsign constr = 
    let j = unsafe_fmachine_prog_noannot metamap sign constr in 
    let newval = 
    if has_annot j._VAL then execute_annot sign comsign j._VAL else j._VAL
    and newtyp = if has_annot j._TYPE then execute_annot sign comsign j._TYPE 
        else j._TYPE
    in {_VAL=newval;_TYPE=newtyp;_KIND=j._KIND};;


(******* An old version of unsafe_fmachine_prog used to be here.
         If needed, use CVS! (revision 1.17)
********)

let is_fmachine_prog metamap (comsign,sign) c =
  conversion_problems := [];
  set_ise mt_evd;
  let j = unsafe_fmachine_prog metamap sign comsign c
  in strong (whd_ise (get_ise())) j._VAL;;


(* sigma is the clenv of the current goal. *)
let unsafe_machine_prog (nocheck,noverify) (sigma,metamap) env comenv constr =
    
 let rec exemeta_rec env comenv cstr = match cstr with
  DOP0(Meta n) ->
    let metaty =
      try List.assoc n metamap
      with Not_found -> error "Metavariable wasn't in the metamap" 
    in
      (match metaty with
           DOP2(Cast,typ,kind) -> {_VAL=cstr; _TYPE=typ; _KIND=kind}
         | _ ->
             {_VAL=cstr;
              _TYPE=metaty;
              _KIND=whd_betadeltaiota sigma
                     (exemeta_rec (gLOB(get_globals env))
                        (gLOB(get_globals comenv)) metaty)._TYPE})
| Rel(n) -> prog_relative sigma n env

| DOP2(Lambda,DOP1(XTRA("COMMENT",[]),c),DLAM(na,b)) ->
  (* [x:{C}]B --> Btype
     when C is well-typed in comenv and is purely logical

     [x:{C}]B --> Btype
     when C is wel-typed in comenv and is informative, and irrelevant
     because of levels

     [x:{C}]B --> (x:extC)Btype
     when C is well-typed in comenv and is informative and relevant
     the extraction of C is extC

     B is well-typed in the environments:
     -- add the extraction of C to env (Implicit if C is logical)
     -- add C to comenv
   *)

  let (cj,cinf) = infmachine sigma (comenv,env) c in
  let relevant = (match cinf with
                  Logic -> false
                | Inf cinfj ->
                  anomaly "Unimplemented(progmach)") in
    if not relevant then
      let env' = (add_rel (na,assumption_of_judgement sigma comenv cj) env)
      and comenv' =
 	(add_rel (na,assumption_of_judgement sigma comenv cj) comenv) in
      let bj = exemeta_rec env' comenv' b
      in {_VAL = (DOP2(Lambda,
                          DOP1(XTRA("COMMENT",[]),j_val cj),
                          DLAM(na,j_val bj)));
              _TYPE = pop bj._TYPE;
              _KIND = bj._KIND}
      else anomaly "Unimplemented(progmach)"

| DOP2(XTRA("ANNOT",[]),c,t) ->
  (* c::{T} --> ctype
     when T is well-typed in comenv and extraction_T is convertible
     with ctype
   *)
  let cj = exemeta_rec env comenv c and
      (tj,tinf) = infmachine sigma (comenv,env) t in
      (match tinf with
       Logic -> error "Annotation must be of computational nature"
     | Inf tinfj ->
       traceme "cj" cj;
       traceme "Tinfj" tinfj;
       if conv1_x sigma tinfj._VAL cj._TYPE then
           {_VAL=DOP2(XTRA("ANNOT",[]),j_val cj,j_val tj);
            _TYPE = cj._TYPE;
            _KIND = cj._KIND}
       else error "Mistyped annotation in program")

| DOP1(XTRA("COMMENT",[]),c) ->
  (* c::{T} --> ctype
     when T is well-typed in comenv and extraction_T is convertible
     with ctype
   *)
  let (tj,tinf) = infmachine sigma (comenv,env) c in
      (match tinf with
       Inf tinfj -> error "must be a logical argument"
     | Logic ->
           {_VAL=(DOP1(XTRA("COMMENT",[]),j_val tj));
            _TYPE = tj._TYPE;
            _KIND = tj._KIND})

| DOPN(XTRA("REC",[]),cl) ->
  let p = cl.(0)
  and c = cl.(1)
  and lf = Array.sub cl 2 ((Array.length cl) - 2) in
  let {_TYPE=ct} as cj = exemeta_rec env comenv c
  and {_TYPE=pt} as pj = exemeta_rec env comenv p in
        exemeta_rec env comenv (Indrec.transform_rec env sigma cl (ct,pt)) 

| DOPN(XTRA("MLCASE",[isrectok]),cl) -> 
  let isrec = is_case_rec isrectok in
  let c = cl.(0) and  lf = tl_vect cl in
  let cj = exemeta_rec env comenv c
  in if Array.length lf = 0 then error "ML case expects at least one branch"
     else let fj = exemeta_rec env comenv lf.(0) in
     let pred = 
       Indrec.pred_case_ml env sigma isrec (cj._VAL,cj._TYPE) lf (1,fj._TYPE)
     in exemeta_rec env comenv
        (Indrec.make_case_ml isrec pred cj._VAL None lf)

| VAR id ->
    let {body=typ;typ=kind} = snd(lookup_glob id env)
    in {_VAL=cstr; _TYPE=typ; _KIND=mkSort kind}

| DOPN(Abst _,_) ->
    if evaluable_abst cstr
    then exemeta_rec env comenv (abst_value cstr)
    else error "Cannot typecheck an unevaluable abstraction"

| DOPN(Const _,_) ->
    let (typ,kind) = destCast(type_of_const2 sigma env cstr)
    in {_VAL=cstr; _TYPE=typ; _KIND = kind}

| DOPN(MutInd _,_) ->
    let {body=typ;typ=kind} = type_of_mind sigma env cstr
    in {_VAL=cstr; _TYPE=typ; _KIND=mkSort kind}
 
| DOPN(MutConstruct _,_) -> 
    let (typ,kind) = destCast(type_of_mconstr sigma env cstr)
    in {_VAL=cstr; _TYPE=typ; _KIND=kind}

| DOP0(Sort(Prop(c))) -> fcn_proposition c
| DOP0(Sort(Type(u))) -> fcn_type_with_universe (u)
| DOPN(AppL,tl)           -> 
  let j = exemeta_rec env comenv (hd_vect tl) in
  let jl = map_vect_list (exemeta_rec env comenv) (tl_vect tl) in
      apply_rel_list_prog sigma env nocheck jl j

| DOP2(Lambda,c1,DLAM(name,c2))      -> 
  let jfw = exemeta_rec env comenv c1 in
  let jcci = try execute_pure_rec sigma comenv jfw._VAL 
             with UserError _ -> jfw in
  let assumfw = assumption_of_judgement sigma comenv jfw in
  let varfw = (name,assumfw) in
  let assumcci = assumption_of_judgement_prog sigma jcci in
  let varcci = (name,assumcci) in
  let j' = exemeta_rec (add_rel varfw env) (add_rel varcci comenv) c2 in 
      abs_rel sigma name assumfw j'

| DOP2(Prod,c1,DLAM(name,c2))        ->
  let jfw = exemeta_rec env comenv c1 in
 let jcci = try execute_pure_rec sigma comenv jfw._VAL 
             with UserError _ -> jfw in
  let assumfw = assumption_of_judgement sigma comenv jfw in
  let varfw = (name,assumfw) in
  let assumcci = assumption_of_judgement_prog sigma jcci in
  let varcci = (name,assumcci) in
  let j' = exemeta_rec (add_rel varfw env) (add_rel varcci comenv) c2 in
      gen_rel sigma FW env name assumfw j'

| DOPN(MutCase _,_) ->
  let (_,p,c,lf) = destCase cstr in
  (let {_TYPE=ct} as cj = exemeta_rec env comenv c in
   let pj = exemeta_rec env comenv p in
   let lfj = Array.map (exemeta_rec env comenv) lf in
   let lft = (Array.map (fun j -> j._TYPE) lfj) in
   let (mind,_) = find_mrectype sigma cj._TYPE
   in  {_VAL=mkMutCaseA (ci_of_mind mind) (j_val pj) (j_val cj) (Array.map j_val lfj);
        _TYPE=type_case_elim_pg env sigma ct pj._TYPE lft p c;
        _KIND = sort_of_arity sigma pj._TYPE})

| (DOPN(Fix(vn,i),cl) as cstr) ->
  if  exists_vect (fun n -> n < 0) vn then 
      error "General Fixpoints not allowed";
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let newenv,newcomenv = it_vect2 
     (fun (env,comenv) name ar -> 
        let arfw = exemeta_rec env comenv ar in
        let assumfw = assumption_of_judgement sigma comenv arfw in
        let arcci = try execute_pure_rec sigma comenv arfw._VAL 
                    with UserError _ -> arfw in
        let assumcci = assumption_of_judgement_prog sigma arcci
        in (add_rel (name,assumfw) env),(add_rel (name,assumcci) comenv))
     (env,comenv) (Array.of_list (List.rev lfi)) (vect_lift lar) in
  let larj = Array.map (exemeta_rec env comenv) lar in
  let lara = Array.map (assumption_of_judgement sigma env) larj in
  let vdefj = Array.map (exemeta_rec newenv newcomenv) vdef in
  let fix =
    mkFix vn i lara (List.rev lfi) (Array.map (fun j -> j._VAL) vdefj) in
  type_fixpoint sigma lfi lara vdefj;
  check_fix_prog sigma fix;
  make_judge fix lara.(i)

| DOPN(CoFix i,cl) ->
(*  if nofix  
  then error "General Fixpoints not allowed"
  else *)
    let nbfix =
      let nv = Array.length cl in 
        if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
    let lar = Array.sub cl 0 nbfix in
    let ldef = last_vect cl in
    let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
    let newenv,newcomenv =
      it_vect2 
        (fun (env,comenv) name ar -> 
           let arfw = exemeta_rec env comenv ar in
           let assumfw = assumption_of_judgement sigma comenv arfw in
           let arcci =
             try execute_pure_rec sigma comenv arfw._VAL
             with UserError _ -> arfw in
           let assumcci = assumption_of_judgement_prog sigma arcci in
             (add_rel (name,assumfw) env),(add_rel (name,assumcci) comenv))
        (env,comenv) (Array.of_list (List.rev lfi)) (vect_lift lar) in
    let larj  = Array.map (exemeta_rec env comenv) lar in
    let lara = Array.map (assumption_of_judgement sigma env) larj in
    let vdefj = Array.map (exemeta_rec newenv newcomenv) vdef in
    let cofix =
      mkCoFix i lara (List.rev lfi) (Array.map (fun j -> j._VAL) vdefj) in
    type_fixpoint sigma lfi lara vdefj;
    check_cofix sigma cofix;
    make_judge cofix lara.(i)

| DOP2(Cast,c,t) ->
  let cj = exemeta_rec env comenv c in
  let tj = exemeta_rec env comenv t
  in cast_rel_prog sigma env cj tj

|  DOP0(XTRA("ISEVAR",[])) -> errorlabstrm "unsafe_machine_prog"
            [< 'sTR"No implicit syntax; all type arguments are needed" >]

| _ -> error_cant_execute FW env cstr
 in exemeta_rec env comenv constr
;;

(* WITHOUT INFORMATION *)
let execute_meta_prog sigma sign metamap constr =
  let (ccienv,fwenv) = gENV sigma sign in 
  let j =
    unsafe_machine_prog (false,false) (sigma,metamap) fwenv ccienv constr in
  {_VAL=j._VAL;_TYPE=strong strip_outer_cast j._TYPE;_KIND=j._KIND};;

let type_meta_of_prog sigma sign metamap c =
  (execute_meta_prog sigma sign metamap c)._TYPE;;
let type_of_prog sigma sign c =
  (execute_meta_prog sigma sign [] c)._TYPE;;



(* Use only the global environment for variables *)
let absolutize_prog sigma env = 
 let genv = snd (gENV sigma (get_globals env)) in

 let rec absrec env = function
    (VAR _) as x -> absolutize_fw sigma genv x
  | Rel _ as t -> t
  | DOP2(XTRA("ANNOT",[]),c,t) ->
    (DOP2(XTRA("ANNOT",[]),(absrec env) c,
          absolutize_cci sigma env t))

  | DOP1(XTRA("COMMENT",[]),c) ->
    (DOP1(XTRA("COMMENT",[]),absolutize_cci sigma env c))

  | DOP1(XTRA("LAMEQN",_),_) as eqn ->
      Multcase_astterm.absolutize_eqn (absrec env) FW genv eqn

  | DOP0 _ as t -> t
  | DOP1(oper,c) -> DOP1(oper,absrec env c)
  | DOP2(oper,c1,c2) -> DOP2(oper,absrec env c1,absrec env c2)
  | DOPN(oper,cl) -> DOPN(oper,Array.map (absrec env) cl)
  | DOPL(oper,cl) -> DOPL(oper,List.map (absrec env) cl)
  | DLAM(na,c) -> DLAM(na,absrec (add_rel (na,()) env)  c)
  | DLAMV(na,cl) -> DLAMV(na,Array.map (absrec (add_rel (na,()) env)) cl)
 in absrec env
;;

let dbize_prog sigma env com =
  let t =
    try dbize env com
    with e ->
      wrap_error
 	(Ast.loc com, "Progmach.dbize_prog",
         [< 'sTR"During conversion from explicit-names to" ; 'sPC ;
           'sTR"debruijn-indices" >], e,
         [< 'sTR"Perhaps the input is malformed" >]) in
  let c =
    try absolutize_prog sigma env t
    with e -> wrap_error
      	(Ast.loc com, "Progmach.dbize_prog",
	 [< 'sTR"During the relocation of global references," >], e,
	 [< 'sTR"Perhaps the input is malformed" >])
  in c
;;


let raw_prog_of_progcom sigma sign c =
  is_fmachine_prog [] (gENV sigma sign)
    (Sosub.soexecute(dbize_prog sigma (gLOB sign) c)) ;;

(* $Id: progmach.ml,v 1.43 1999/11/07 05:17:11 barras Exp $ *)
